;; SKK tutorial (version 3.8 for SKK version 8.6 and later versions.)
;; Copyright (C) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995
;; Masahiko Sato (masahiko@sato.riec.tohoku.ac.jp)

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either versions 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SKK, see the file COPYING.  If not, write to the Free
;; Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Following people contributed modifications to skk-tut.el:
;;	Wataru Matsui (matsui@gf.hm.rd.sanyo.co.jp)
;;	Shuhei KOBAYASHI (tba4kac@aix370.ecip.tohoku.ac.jp)
;;	ͦ (jshen@cas.org)
;;	Haru'yasu Ueda (hal@sics.se)
;;	Toyonobu Yoshida (toyono-y@is.aist-nara.ac.jp)
;;	IIDA Yosiaki (iida@sayla.secom-sis.co.jp)
;;	紴 (PBC01764@niftyserve.or.jp / minakaji@mix.or.jp)

;; version 3.8 released 1995.5.13
;; version 3.7 released 1993.5.20
;; version 3.6 released 1992.9.19
;; version 3.5 released 1992.5.31
;; version 3.4 released 1992.4.12
;; version 3.3 released 1991.4.20
;; version 3.2 released 1990.4.15
;; version 2.2 released 1989.4.15

;; Introduced by W. Matsui (matsui@atr-rd.atr.co.jp), 1991.8.14.
;; modified by Masahiko Sato, 1992.5.30.
(defvar skk-tut-file (concat exec-directory "SKK.tut")
  "Full path name of the SKK tutorial file.")
(defvar skk-file "skk"
  "This will be used as the argument of load.  You can specify the locally
installed SKK if necessary.")

(defvar skk-japanese-tut nil)
(defvar skk-tut-jisyo "~/SKK-TUT-JISYO")

(defconst skk-problem-nos 37)

(setq j-mode-line-skk nil)

(defun skk-tutorial ()
  (interactive)
  (let ((inhibit-quit t) v)
    ;; we load skk-file even if it has already been loaded.  this is to
    ;; reset constants
    (load skk-file)
    (setq v (skk-version))
    (if (< (string-to-int (char-to-string (aref v 0))) 6)
	;; the version of skk just loaded is an old one
	(error
	   "Your skk%s is too old to run the tutorial.  It must be at least version 6." v))
    (setq skk-problem-count 0
	  skk-tutorial-end nil
	  skk-jisyo-save nil)
    (if (not (fboundp 'j-save-other-window))
	(fset 'j-save-other-window (symbol-function 'other-window)))
    (fset 'other-window 'j-other-window)
    (save-excursion
      (set-buffer (get-buffer-create " *skk-tut-jisyo*"))
      (setq buffer-file-name (expand-file-name skk-tut-jisyo)))
    (set-buffer (get-buffer-create " *skk-tutorial*"))
    (erase-buffer)
    ;; Modified by W. Matsui, 1991.8.14.
    (insert-file-contents skk-tut-file)
    (goto-char (point-min))
    ;; check if the text is in Japanese or in English
    (setq skk-japanese-tut (looking-at ";; SKK Japanese"))
    ;; if the editor is Mule, modify the text a little
    (if (boundp 'MULE)
	;; modified by Shuhei KOBAYASHI, 1994.6.29.
	;; 95.4.9 Mikio Nakajima
	(progn
	  (save-excursion (replace-string "Nemacs" "Mule" nil))
	  (save-excursion
	    (replace-string "Σ" "ͣ" nil) )))
    (switch-to-buffer (get-buffer-create "**"))
    (erase-buffer)
    (setq skk-tutorial-map (make-keymap))
    ;; modified by ͦ, 1994.9.8.
    ;; modified by Haru'yasu Ueda, 1994.9.21.
    ;; modified by Toyonobu Yoshida, 1994.10.3.
    ;; modified by IIDA Yosiaki, 1994.10.6.
    (fillarray (if (arrayp skk-tutorial-map)
		   skk-tutorial-map
		 (car (cdr skk-tutorial-map))) 'skk-error)
    (use-local-map skk-tutorial-map)
    (erase-buffer)
    (skk-get-new-page)
    (delete-other-windows)
    (split-window-vertically nil)
    (other-window 1)
    (enlarge-window (- (window-height (selected-window)) 20))
    (switch-to-buffer (get-buffer-create "**"))
    ;; modified by Shuhei KOBAYASHI, 1994.6.29.
    (make-local-variable 'skk-search-prog-list)
    (setq skk-search-prog-list '((j-search-jisyo-file skk-tut-jisyo 0 t)))
    (local-set-key "\C-xn" 'skk-next-window)
    (local-set-key "\C-xq" 'skk-quit-tutorial)
    (local-set-key "\C-xs" 'skk-skip-problem)
    (local-set-key "\C-x\C-j" 'j-skk-mode)
    (local-set-key "\C-h" 'skk-nil) ;; disable help
    (switch-to-buffer-other-window "**")
    (goto-char (point-max))
    (beginning-of-line)
    (skk-answer-window)
    (message "")))

(defun j-skk-mode ()
  (interactive)
  (or (featurep 'skk) (provide 'skk))
  (if skk-mode
      (progn
	(j-change-mode-line "--")
	(setq j-mode t)
	(use-local-map skk-map))
    (or j-mode-line-skk
	(setq mode-line-format
	      (cons "" (cons 'j-mode-line-skk
			     (cdr (default-value 'mode-line-format))))))
    (j-change-mode-line "--")
    (setq skk-mode t
	  j-mode t
	  j-katakana nil)
    (local-set-key "\C-j" 'j-kakutei)
    (setq j-emacs-local-map
	  (j-convert-to-vector (copy-keymap (current-local-map))))
    (define-key minibuffer-local-map "\C-j" 'skk-mode)
    (define-key minibuffer-local-map "\C-m" 'j-newline)
    (j-setup-skk-map)
    (j-setup-skk-zenkaku-map)
    (j-setup-skk-abbrev-map)
    (use-local-map skk-map)
    (if (= skk-problem-count 2) (skk-disable))))

(defun skk-error ()
  (interactive)
  (switch-to-buffer-other-window "**"))

(defun skk-nil ()
  (interactive)
  (error ""))

(defun skk-get-new-page ()
  (save-excursion
    (set-buffer " *skk-tutorial*")
    (goto-char (point-min))
    ;; modified by Toyonobu Yoshida, 1994.10.3.
    (search-forward "--\n" nil t)
    (delete-region (point-min) (point))
    (if (looking-at ";")
	(let (p)
	  (forward-char 3)
	  (setq p (point))
	  (end-of-line)
	  (save-excursion (eval-region p (point) nil))
	  (forward-char 1)))
    (or skk-tutorial-end
	(let ((p (point)))
	  (search-forward "\n>>")
	  (end-of-line)
	  (copy-to-buffer "**" p (point)))))
  (setq skk-problem-count (1+ skk-problem-count))
  (setq mode-line-buffer-identification
	(concat "ӣˣ˥塼ȥꥢ:  "
		(int-to-string skk-problem-count)
		" ʻĤ "
		(int-to-string (- skk-problem-nos skk-problem-count))
		""))
  (set-buffer-modified-p t)
  (sit-for 0))

(defun skk-quit-tutorial (&optional now)
  (interactive)
  (if (or now (yes-or-no-p "˥塼ȥꥢޤ? "))
      (let ((inhibit-quit t))
	(delete-other-windows)
	(kill-buffer " *skk-tutorial*")
	(kill-buffer "**")
	(kill-buffer "**")
	(let ((buff (get-file-buffer skk-tut-jisyo)))
	  (if buff
	      (progn
		(set-buffer buff)
		(set-buffer-modified-p nil)
		(kill-buffer buff))))
	(if (fboundp 'j-save-set-henkan-point)
	    (fset
	     'j-set-henkan-point (symbol-function 'j-save-set-henkan-point)))
	(if (fboundp 'j-save-abbrev-input)
	    (fset 'j-abbrev-input (symbol-function 'j-save-abbrev-input)))
	(if (fboundp 'j-save-other-window)
	    (fset 'other-window (symbol-function 'j-save-other-window)))
	(fmakunbound 'j-save-set-henkan-point)
	(fmakunbound 'j-save-abbrev-input)
	(fmakunbound 'j-save-other-window)
	(save-excursion
	  (set-buffer "*scratch*")
	  (setq j-skk-mode-invoked nil)
	  (skk-mode t)
	  (skk-mode nil)))))

(defun skk-answer-window ()
  (interactive)
  ;(switch-to-buffer "**")
  (goto-char (point-max))
  (search-backward "\n>>")
  (forward-char 1)
  (let ((p (point)))
    (setq skk-ans
	  (buffer-substring (+ 3 (point)) (progn (end-of-line) (point))))
    (goto-char p))
  (switch-to-buffer-other-window "**")
  ;(goto-char (point-min))
  (skk-insert-answer-form))

(defun skk-next-window ()
  (interactive)
  (let (user-ans)
    (goto-char (point-min))
    (end-of-line)         
    (skip-chars-backward " \t")
    (setq user-ans (buffer-substring (+ 3 (point-min)) (point)))
    (if (string= skk-ans user-ans)
	(progn
	  (message "")
	  (erase-buffer)
	  (other-window 1)
	  (skk-get-new-page)
	  (if skk-tutorial-end (skk-quit-tutorial t) (skk-answer-window)))
      (message "㤤ޤ⤦٤äƤߤƲ")
      (ding)
      (save-excursion
	(set-buffer " *skk-tutorial*")
	(goto-char (point-min))
	(if (looking-at ";")
	    (let (p)
	      (end-of-line)
	      (setq p (point))
	      (eval-last-sexp t)
	      (delete-region p (point)))))
      (erase-buffer)
      (skk-insert-answer-form)
      (goto-char (+ 3 (point-min)))
      (insert user-ans))))

(defun skk-skip-problem ()
  (interactive)
  (erase-buffer)
  (if (= skk-problem-count 2) (j-skk-mode))
  (other-window 1)
  (skk-get-new-page)
  (if skk-tutorial-end (skk-quit-tutorial t) (skk-answer-window)))

(defun skk-insert-answer-form ()
  (insert ">> \n\n"
	  "* ǤC-x n; ǤˤϡC-x q; "
	  "åפˤϡC-x s *")
  (goto-char (+ (point-min) 3)))

(defun skk-disable ()
  (if (not (fboundp 'j-save-set-henkan-point))
      (progn
	(fset 'j-save-set-henkan-point (symbol-function 'j-set-henkan-point))
	(fset 'j-set-henkan-point (symbol-function 'j-set-henkan-point-tmp))))
  (if (not (fboundp 'j-save-abbrev-input))
      (progn
	(fset 'j-save-abbrev-input (symbol-function 'j-abbrev-input))
	(fset 'j-abbrev-input (symbol-function 'j-abbrev-input-tmp))))
  (skk-use-jisyo ""))

(defun j-set-henkan-point-tmp ()
  (interactive)
  (if j-mode 
      (error "/ʥ⡼ɤǤϡʸϤޤȤޤ")
    (insert (if j-zenkaku (concat (char-to-string 163)
				  (char-to-string (+ last-command-char 128)))
	      last-command-char))))

(defun j-abbrev-input-tmp ()
  (interactive)
  (if j-mode
      (error "ΥϤޤȤޤ")
    (insert last-command-char)))

;; functions below are called from the tutorial text

(defun skk-enable ()
  (fset 'j-set-henkan-point (symbol-function 'j-save-set-henkan-point))
  (fset 'j-abbrev-input (symbol-function 'j-save-abbrev-input))
  ;; save jisyo buffer, and modify jisyo buffer for this tutorial
  (skk-use-jisyo " ///ƻ/\n"))

;; The following two functions are tricky, since they are executed by
;; "eval-region".

(defun skk-test-name ()
  (let ((p (point-min)) (q (point-max)))
    (widen)
    (search-forward "\n>> ")
    (insert "My name is " (user-full-name) ".")
    (narrow-to-region p q)
    (goto-char q)))

(defun skk-test-today ()
  (let ((p (point-min)) (q (point-max)))
    (widen)
    (search-forward "\n>> ")
    (insert "֤礦ϡ" (j-date) "Ǥ")
    (narrow-to-region p q)
    (goto-char q)))

(defun skk-use-jisyo (str)
  (save-excursion
    (set-buffer (get-file-buffer skk-tut-jisyo))
    (erase-buffer)
    (insert ";; okuri-ari-entries.\n;; okuri-nasi entries.\n" str)
    (goto-char (point-min))
    (search-forward ";; okuri-ari entries.\n" nil t)
    (setq j-okuri-ari-min (point))
    (search-forward "\n;; okuri-nasi entries." nil t)
    (beginning-of-line)
    (setq j-okuri-ari-max (point))
    (end-of-line)
    (setq j-okuri-nasi-min (1+ (point)))))

(defun skk-use-okuri-jisyo (str1 &optional str2)
  (save-excursion
    (set-buffer (get-file-buffer skk-tut-jisyo))
    (erase-buffer)
    (insert ";; okuri-ari entries.\n" str1 ";; okuri-nasi entries.\n")
    (if str2 (insert str2))
    (goto-char (point-min))
    (search-forward ";; okuri-ari entries.\n" nil t)
    (setq j-okuri-ari-min (point))
    (search-forward "\n;; okuri-nasi entries." nil t)
    (beginning-of-line)
    (setq j-okuri-ari-max (point))
    (end-of-line)
    (setq j-okuri-nasi-min (1+ (point)))))

(defun skk-end-of-tutorial ()
  (message "")
  (switch-to-buffer " **")
  (erase-buffer)
  (goto-char (point-min))
  (if skk-japanese-tut
      (progn
	(insert "SKK 塼ȥꥢϤǽǤ\n\n"
		"SKK ˴ؤ䡢ȡbug report \n\n"
		"\tskk@sato.riec.tohoku.ac.jp\n\n"
		"겼ʤΥɥ쥹 SKK ᥤ󥰥ꥹȤΥɥ쥹Ǥ\n"
		"̾盧Υɥ쥹ФƤʤΤǡСǤʤϤλݤ\n"
		"ƥ᡼꤯ SKK ᥤ󥰥ꥹȤػô˾ξ\n\n"
		"\tskk-join@sato.riec.tohoku.ac.jp\n\n"
		"إ᡼꤯\n\n"
		"!! Ǹ <return> 򲡤Ƥ"))
    (insert "Now we end the SKK tutorial.\n\n"
	    "Please send comments, questions and bug reports on SKK to:\n\n"
	    "\tskk@sato.riec.tohoku.ac.jp\n\n"
	    "This is the address of the SKK mailing list, and normally the responces\n"
	    "will be sent only to the ML members.  So, if you are not a ML member,\n"
	    "please say so in your mail.  If you are interested in joining the SKK ML,\n"
	    "send a mail to:\n\n"
	    "\tskk-join@sato.riec.tohoku.ac.jp\n\n"
	    "!! Hit <return> key when you are ready."))
  (while (not (= ?\C-m (read-char)))
    (message
     (if skk-japanese-tut
	 "<return> 򲡤Ƥ"
       "Hit <return> key."))
    (ding))
  (setq skk-tutorial-end t))

(if (not (fboundp 'j-save-other-window))
    (fset 'j-save-other-window (symbol-function 'other-window)))

(defun j-other-window (arg)
  (interactive "p")
  (if (interactive-p)
      (error "Can't switch window now.")
    (j-save-other-window arg)))
