;; SKK (Simple Kana to Kanji conversion program)
;; version 5.30 of May 9, 1992
;; Copyright (C) 1988, 1992 Masahiko Sato (masahiko@sato.riec.tohoku.ac.jp)

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either versions 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SKK, see the file COPYING.  If not, write to the Free
;; Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Following people contributed modifications to skk.el:
;;	Masakazu Takahashi (masaka-t@ascii.co.jp)
;;	Tsugutomo Enami (enami@ptgd.sony.co.jp)
;;      Koichi Mori (kmori@bek-owl.cns.caltech.edu)
;;      Jun-ichi Nakamura (nakamura@pluto.ai.kyutech.ac.jp)
;;      Kazuo Hirokawa (hirokawa@ricsgw.rics.co.jp)
;;      Wataru Matsui (matsui@atr-rd.atr.co.jp)
;;      Tomoyuki Hiro (hiro@momo.it.okayama-u.ac.jp)

;; version 5.30 released 1992.5.9
;; version 4.33 released 1991.5.9
;; version 4.30 released 1991.4.20
;; version 3.47 released 1990.4.29
;; version 2.25 released 1989.4.15
;; version 1.13 released 1988.4.22

(provide 'skk)

(defun skk-version ()
  (interactive)
  (if (interactive-p)
      (message "SKK version %s of May 9, 1992" (skk-version))
    "5.30"))

(defun skk-server-version ()
  (interactive)
  (if skk-server-host
      (let (v)
	(set-buffer " *skkserv*")
	(erase-buffer)
	(if (process-status "skkservd")
	    (progn
	      (process-send-string "skkservd" "2")
	      (accept-process-output "skkservd")
	      (setq v (buffer-substring (point-min) (1- (point-max))))
	      (erase-buffer)
	      (process-send-string "skkservd" "3")
	      (accept-process-output "skkservd")
	      (message "SKK SERVER %s running on HOST %s." v
		       (buffer-substring
			(point-min)
			(progn
			  (goto-char (point-min))
			  (search-forward ":")
			  (- (point) 1)))))
	  (message "SKK SERVER is not active.")))
    (message "You are not using SKK SERVER.")))

;; variable declarations

(setq enable-recursive-minibuffers t)

(defvar j-count-touroku 0)
(defvar j-count-kakutei 0)

(defvar j-kakutei-early nil)

(defvar skk-isearch-message nil
  "User must not change the value of this value.  This variable is used in
conjunction with skk-isearch.")

(defvar j-skk-mode-invoked nil
  "The value is t if skk-mode was invoked in this Emacs session.")

(defvar skk-mode nil
  "The value is t if skk-mode is currently invoked on the current buffer,
it is nil otherwise.")
(make-variable-buffer-local 'skk-mode)

(defvar j-mode-line-format nil
  "Save mode-line-format of the current buffer.  This variable may also be
used to test if skk-mode has been called on the current buffer.")
(make-variable-buffer-local 'j-mode-line-format)

(defvar j-current-local-map nil
  "Save current-local-map")
(make-variable-buffer-local 'j-current-local-map)

(defvar j-vip-current-mode nil
  "Save vip-current-mode")
(make-variable-buffer-local 'j-vip-current-mode)

;; Tsugutomo Enami suggested that the variables skk-map, skk-zenkaku-map and
;; skk-abbrev-map be initially set to nil, 1990.7.13.

(defvar skk-map nil
  "Keymap for kana mode.")
(make-variable-buffer-local 'skk-map)

(defvar skk-zenkaku-map nil
  "Keymap for zenkaku mode.")
(make-variable-buffer-local 'skk-zenkaku-map)

(defvar skk-abbrev-map nil
  "Keymap for abbrev input.")
(make-variable-buffer-local 'skk-abbrev-map)

(defvar j-emacs-local-map nil
  "Original keymap for emacs mode.")
(make-variable-buffer-local 'j-emacs-local-map)

(defvar j-mode-line-skk nil)
(make-variable-buffer-local 'j-mode-line-skk)

(defvar j-prefix "")

(defvar j-mode nil)
(make-variable-buffer-local 'j-mode)

(defvar j-henkan-on nil)
(make-variable-buffer-local 'j-henkan-on)

(defvar j-henkan-start-point (make-marker)
  "Henkan start point as a marker.")
(make-variable-buffer-local 'j-henkan-start-point)

(defvar j-henkan-end-point (make-marker)
  "Henkan end point as a marker.")
(make-variable-buffer-local 'j-henkan-end-point)

(defvar j-kana-start-point (make-marker)
  "Start point of a kana character as a marker.")

(defvar j-okurigana nil
  "If t, it means that okurigana part of henkan word is being input.")
(make-variable-buffer-local 'j-okurigana)

(defvar j-okurigana-start-point (make-marker)
  "Start point of okurigana")
(make-variable-buffer-local 'j-okurigana-start-point)

(defvar j-okuri-char ""
  "The roman prefix for the okurigana part of henkan word as a string.")
(make-variable-buffer-local 'j-okuri-char)

(defvar j-henkan-count 0)
(make-variable-buffer-local 'j-henkan-count)

(defvar j-okuri-index-min 0)
(make-variable-buffer-local 'j-okuri-index-min)

(defvar j-okuri-index-max 0)
(make-variable-buffer-local 'j-okuri-index-max)

(defvar j-henkan-active nil)
(make-variable-buffer-local 'j-henkan-active)

(defvar j-katakana nil
  "t means katakana and nil means hirakana input.")
(make-variable-buffer-local 'j-katakana)

(defvar j-henkan-key)
(make-variable-buffer-local 'j-henkan-key)

(defvar j-henkan-key2 nil)
(make-variable-buffer-local 'j-henkan-key2)

(defvar j-henkan-okurigana nil)
(make-variable-buffer-local 'j-henkan-okurigana)

(defvar j-henkan-vector (vector))
(make-variable-buffer-local 'j-henkan-vector)

(defvar j-henkan-vector1 (vector))
(make-variable-buffer-local 'j-henkan-vector1)

(defvar j-henkan-vector2 (vector))
(make-variable-buffer-local 'j-henkan-vector2)

(defvar j-henkan-vector3 (vector))
(make-variable-buffer-local 'j-henkan-vector3)

(defvar j-henkan-vector4 (vector))
(make-variable-buffer-local 'j-henkan-vector4)

(defvar j-zenkaku)
(make-variable-buffer-local 'j-zenkaku)

(defvar j-abbrev nil)
(make-variable-buffer-local 'j-abbrev)

(defvar j-completion-word "")

(defvar j-zenkaku-alist
  '((?  . "") 
    (?! . "")
    (?\" . "")
    (?# . "")
    (?$ . "")
    (?% . "")
    (?& . "")
    (?' . "")
    (?( . "")
    (?) . "")
    (?* . "")
    (?+ . "")
    (?, . "")
    (?- . "")
    (?. . "")
    (?/ . "")
    (?: . "")
    (?; . "")
    (?< . "")
    (?= . "")
    (?> . "")
    (?? . "")
    (?@ . "")
    (?[ . "")
    (?\\ . "")
    (?] . "")
    (?^ . "")
    (?_ . "")
    (?{ . "")
    (?| . "")
    (?} . "")
    (?~ . "")))

(defvar j-num-list ()
  "list of numeric arguments obtained from j-henkan-key.  for instance,
if j-henkan-key is \"225ˤ\", j-num-list will be set to
(\"2\" \"25\").")
(make-variable-buffer-local 'j-num-list)

(defvar j-search-key nil
  "string used as the search key for midashi search in jisyo buffer. the value
of this variable is set by using j-henkan-key.  for instance, if j-henkan-key
is \"225ˤ\", j-search-key will be set to \"##ˤ\".")
(make-variable-buffer-local 'j-search-key)

;; skk server related variables

(defvar j-local-length 0)
(make-variable-buffer-local 'j-local-length)

(defvar j-server-called nil)
(make-variable-buffer-local 'j-server-called)

;; variables for the function input-by-code-or-menu

(defvar j-code-n1-min 161)
(defvar j-code-n1-max 244)
(defvar j-code-n2-min 161)
(defvar j-code-n2-max 254)
(defvar j-code-null 128)
(defvar j-input-by-code-or-menu-jump-default j-code-n1-min)

;; The following variable was introduced at the suggestion of Masakazu
;; Takahashi, 1991.10.1.

(defvar j-remote-shell-program
  (or (getenv "REMOTESHELL")
      (cond
       ((eq system-type 'berkeley-unix) "/usr/ucb/rsh")
       ((eq system-type 'usg-unix-v)
	(if (file-exists-p "/usr/ucb/remsh") "/usr/ucb/remsh" "/bin/rsh"))
       ((eq system-type 'hpux) "/usr/bin/remsh")
       ((eq system-type 'EWS-UX/V) "/usr/ucb/remsh")
       ((eq system-type 'pcux) "/usr/bin/rcmd")
       (t "rsh"))))

;; constants

;; Cutomization of these constants cannot be done in .emacs file.  This
;; should be done in .skk file or by using skk-mode-hook.

;; We are using `defconst' rather than `defvar' here for the following
;; reason:  The SKK tutorial program, which is intended for the beginners,
;; must be run on a vanilla SKK.  To achieve this, the tutorial program
;; loads SKK (even if it has been loaded already) and undoes the customization
;; which might have been done by the user.  (Note that `defconst' always sets
;; a new value, while `defvar' does that only when the value is unbound when
;; `defvar' is executed.)

(defconst skk-jisyo "~/.skk-jisyo"
  "*The name of the skk jisyo file.")

;; suggested by Koichi Mori, 1990.8.20
(defconst skk-jisyo-code nil
  "*If non-nil, it is used as the value of kanji-fileio-code for the jisyo
buffer.")

(defconst skk-record-file "~/.skk-record"
  "*The name of the skk record file.")

;; suggested by Koichi Mori, 1991.1.27.
(defconst skk-date-ad nil
  "*If non-nil, j-date is in A.D.  If nil, it is in Nengo (Heisei).")

;; suggested by Koichi Mori, 1991.1.27.
(defconst skk-number-style 1
  "*If nil or 0, numbers in j-date are in Hankaku. If t or 1, in Zenkaku.
Otherwise Kanji.")

(defconst skk-keep-record t
  "*If t, skk will keep record of the success rate of kana to kanji conversion
etc. in the file ~/.skk-record.")

(defconst skk-use-vip nil
  "*If t, skk will assume that the user will use vip-mode at the same time.")

(defconst skk-auto-okuri-process nil
  "*If t, skk automatically recognizes okuri-gana part.  For instance, you
can just type `Uresii ' and it will be converted as you expect. This
behavior is available only if skk-process-okuri-early is nil.")

(defconst skk-process-okuri-early nil
  "*If t, skk will start kana to kanji conversion when it reads the first
roman character for okurigana, if nil then kana to kanji conversion will
start only when it reads a complete roman representation for a kana
character.")

;; introduced by Wataru Matsui 1991.8.19
(defconst skk-egg-like-newline nil
  "*If non-nil, no newline is inserted in any henkan mode.")

(defconst skk-kakutei-early t
  "*If t, the values of j-henkan-active is also t, and skk-process-okuri-early
is nil, then skk will kakutei the current candidate when j-kana-input is
called.")

(defconst skk-mode-string "--SKK:"
  "*The value of this constant is used in the mode line when skk is in ascii
mode.")

(defconst skk-hirakana-mode-string "--"
  "*The value of this constant is used in the mode line when skk is in hirakana
mode.")

(defconst skk-katakana-mode-string "--"
  "*The value of this constant is used in the mode line when skk is in katakana
mode.")

(defconst skk-zenei-mode-string "--"
  "*The value of this constant is used in the mode line when skk is in zenei
mode.")

(defconst skk-echo t
  "*If T, echoes prefix for kana characters.")

(defconst skk-server-host (getenv "SKKSERVER")
  "*Name of the host running SKK Jisyo Server")

(defconst skk-serv (getenv "SKKSERV")
  "*Name of the SKK Jisyo Server program")

(defconst skk-portnum nil
  "*If the value is non-nil, it must be the PORTNUM defined in skkserv.h")

(defconst skk-report-server-response nil
  "*When SKK sends a request to SKK SERVER, SKK waits for the process output
by repeatedly executing the accept-process-output function.  If this constant
is t, then SKK will report the number of times the function is called.")

(defconst skk-num-type-list
  '((?0 . j-id)
    (?1 . j-zenkaku-num-str)
    (?2 . j-kanji-num-str)
    (?3 . j-kanji-num-str2)
    (?9 . j-shogi-num-str))
  "*association list of type-function pairs used in the conversion of numeric
arguments in henkan key.")

(defconst skk-char-type-vector
  [0 0 0 0 0 0 0 0
   5 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 0 0 0 0 0 0 0
   0 4 4 4 4 4 4 4
   4 4 4 4 0 4 4 4
   4 0 4 4 4 4 4 4
   0 4 4 0 0 0 0 0
   0 3 1 1 1 3 1 1
   1 3 1 1 0 1 2 3
   1 0 1 1 1 3 1 1
   2 1 1 0 0 0 0 5]
  "*char type is interpreted as follows.
0 exit rom-kana conversion (not used at present)
1 consonant that can be part of sokuon
2 other consonants (n, x)
3 vowel
4 bound to j-set-henkan-point in skk-mode
5 delete prefix")

(defconst skk-kana-rom-vector
  ["x" "a" "x" "i" "x" "u" "x" "e" "x" "o" "k" "g" "k" "g" "k" "g"
   "k" "g" "k" "g" "s" "z" "s" "j" "s" "z" "s" "z" "s" "z" "t" "d"
   "t" "d" "x" "t" "d" "t" "d" "t" "d" "n" "n" "n" "n" "n" "h" "b"
   "p" "h" "b" "p" "h" "b" "p" "h" "b" "p" "h" "b" "p" "m" "m" "m"
   "m" "m" "x" "y" "x" "y" "x" "y" "r" "r" "r" "r" "r" "x" "w" "x"
   "x" "w" "n"]
  "*kana to roman conversion giving only the first roman character
corresponding to the following kana characters.  For the characters
֤, ֤,֤ա, you might want to change the values for them to
\"z\", \"c\", \"f\".
                                  
                                  
                                  
                                  
                                  
        ")

(defconst skk-prefix-list
  '("by" "ch" "cy" "dh" "dy" "fy" "gy" "hy" "jy" "ky" "my" "py" "ry" "sh" "sy"
    "th" "ts" "ty" "xk" "xt" "xts" "xw" "xy" "zy"))

(defconst skk-rom-kana-rule-list
  '(("nn" nil ("" . ""))
    ("n'" nil ("" . ""))))

(defconst skk-roma-kana-a
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . "")) 
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xk" . ("" . ""))
    ("xw" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-i
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("ꤣ" . "ꥣ"))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xw" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-u
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("" . "")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ts" . ("" . "")) ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xt" . ("" . ""))
    ("xts" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-e
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("ꤧ" . "ꥧ"))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xk" . ("" . ""))
    ("xw" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-roma-kana-o
  '(("" . ("" . ""))
    ("b" . ("" . "")) ("by" . ("Ӥ" . "ӥ"))
    ("ch" . ("" . "")) ("cy" . ("" . ""))
    ("d" . ("" . "")) ("dh" . ("Ǥ" . "ǥ"))
    ("dy" . ("¤" . "¥"))
    ("f" . ("դ" . "ե")) ("fy" . ("դ" . "ե"))
    ("g" . ("" . "")) ("gy" . ("" . ""))
    ("h" . ("" . "")) ("hy" . ("Ҥ" . "ҥ"))
    ("j" . ("" . "")) ("jy" . ("" . ""))
    ("k" . ("" . "")) ("ky" . ("" . ""))
    ("m" . ("" . "")) ("my" . ("ߤ" . "ߥ"))
    ("n" . ("" . "")) ("ny" . ("ˤ" . "˥"))
    ("p" . ("" . "")) ("py" . ("Ԥ" . "ԥ"))
    ("r" . ("" . "")) ("ry" . ("" . ""))
    ("s" . ("" . "")) ("sh" . ("" . ""))
    ("sy" . ("" . ""))
    ("t" . ("" . "")) ("th" . ("Ƥ" . "ƥ"))
    ("ty" . ("" . ""))
    ("v" . ("" . "")) ("w" . ("" . ""))
    ("x" . ("" . "")) ("xy" . ("" . ""))
    ("y" . ("" . ""))
    ("z" . ("" . "")) ("zy" . ("" . ""))))

(defconst skk-input-vector
  [nil  nil  nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  "" nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  "" "" "" nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  "" "" nil  nil  nil  ""
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  "" nil  "" nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil]
  "*Translation table for j-self inserting characters.")

(defconst skk-zenkaku-vector
  [nil  nil  nil  nil  nil  nil  nil  nil
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   nil  nil  nil  nil  nil  nil  nil  nil 
   ""  "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" ""
   "" "" "" "" "" "" "" nil])

;; utilities

(defun j-member (x l)
  "check if X is a member of L, where string= is used to compare elements
of L against X"
  (cond ((null l) nil)
	((string= x (car l)) t)
	(t (j-member x (cdr l)))))

(defun j-assoc (key a-list)
  (let ((list a-list) (cont t) (val nil))
    (while cont
      (if (null list)
	  (setq cont nil)
	(let ((key2 (car (car list))))
	  (if (string= key key2)
	      (setq cont nil
		    val (cdr (car list)))
	    (setq list (cdr list))))))
    val))

(defun j-change-mode-line (string)
  (setq j-mode-line-skk string)
  (set-buffer-modified-p (buffer-modified-p)))

(defun j-today ()
  "Insert current date at point in Kanji."
  (interactive)
  (insert (j-date)))

;; modified by Koichi Mori, 1991.1.17

(defun j-date ()
  (let* ((str (current-time-string))
	 (y (- (string-to-int (substring str 20 24)) 1988))
	 (year (if skk-date-ad
		   (j-num (substring str 20 24))
		 (if (= y 1) "" (j-num (int-to-string y)))))
	 (mon (substring str 4 7))
	 (month
	  (j-num
	   (cdr (assoc mon
		       '(("Jan" . "1") ("Feb" . "2") ("Mar" . "3")
			 ("Apr" . "4") ("May" . "5") ("Jun" . "6")
			 ("Jul" . "7") ("Aug" . "8") ("Sep" . "9")
			 ("Oct" . "10") ("Nov" . "11") ("Dec" . "12"))))))
	 (day (j-num (substring str 8 10)))
	 (dw (substring str 0 3))
	 (day-of-week
	  (cdr (assoc dw
		      '(("Sun" . "") ("Mon" . "") ("Tue" . "")
			("Wed" . "") ("Thu" . "") ("Fri" . "")
			("Sat" . "")))))
	 )
    (concat (if skk-date-ad "" "ʿ") year "ǯ"
	    month "" day "" "\(" day-of-week "\)")))

;; modified by Masakazu Takahashi, 1991.3.27
;; modified by Koichi Mori, 1991.1.17

(defun j-num (str)
  "convert a number string to other style according to skk-number-style"
  (mapconcat (function j-num1) str ""))

(defun j-num1 (c)
  "convert a number character to other style according to skk-number-style"
  (cond ((or (null skk-number-style) (eq skk-number-style 0))
	 (cdr (assq c
		    '((?0 . "0") (?1 . "1") (?2 . "2") (?3 . "3")
		      (?4 . "4") (?5 . "5") (?6 . "6") (?7 . "7")
		      (?8 . "8") (?9 . "9")(?  . "")))))
	((or (eq skk-number-style t) (eq skk-number-style 1))
	 (cdr (assq c
		    '((?0 . "") (?1 . "") (?2 . "") (?3 . "")
		      (?4 . "") (?5 . "") (?6 . "") (?7 . "")
		      (?8 . "") (?9 . "")(?  . "")))))
	(t
	 (cdr (assq c
		    '((?0 . "") (?1 . "") (?2 . "") (?3 . "")
		      (?4 . "") (?5 . "") (?6 . "ϻ") (?7 . "")
		      (?8 . "Ȭ") (?9 . "") (?  . "")))))))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-convert-to-vector (map)
  "convert a map in list form to a map in vector form."
  (if (listp map)
      (let ((new-map (make-keymap)) (l (cdr map)) def)
	(while l
	  (setq def (cdr (car l)))
	  (aset new-map (car (car l)) (if (keymapp def)
					  (copy-keymap def)
					def))
	  (setq l (cdr l)))
	new-map)
    ;; modified by Tsugutomo Enami, 1990.7.9
    (copy-keymap map)))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-get-editor-command (l-map g-map &optional init-chars)
  "Read characters from keyboard until an editor command is formed, using
local keymap L-MAP and global keymap G-MAP. init-chars is first key or
key sequences."
  (let (char chars l-bind g-bind)
    (if init-chars
	(if (stringp init-chars)
	    (if (string= init-chars "")
		(setq char (read-char)
		      chars nil)
	      (setq char (aref init-chars 0)
		    chars (substring init-chars 1)))
	  (setq char init-chars))
      (setq char (read-char)))
    (setq last-command-char char)
    (setq l-bind (j-binding-of char l-map))
    (if (null l-bind)
	;; since local binding is empty, we concentrate on global one.
	(progn
	  (setq g-bind (j-binding-of char g-map))
	  (if (null g-bind)
	      nil ;; return nil, since both bindings are void.
	    (if (keymapp g-bind)
		(j-get-editor-command nil g-bind chars)
	      g-bind)))
      ;; local binding is nonvoid
      (if (keymapp l-bind)
	  ;; since l-bind is a keymap, we consider g-bind as well.
	  (progn
	    (setq g-bind (j-binding-of char g-map))
	    (if (null g-bind)
		(j-get-editor-command l-bind nil chars)
	      (if (keymapp g-bind)
		  ;; both bindings are keymap
		  (j-get-editor-command l-bind g-bind chars)
		;; l-bind is a keymap, so we neglect g-bind
		(j-get-editor-command l-bind nil chars))))
	;; l-bind is a command
	l-bind))))

(defun j-binding-of (char map)
  "Return key-binding of CHAR under keymap MAP.  It is nil if the binding
is void, or a command, or a keymap"
  (let ((val (if (listp map)
		 (cdr (assq char map))
	       (aref map char))))
    (cond ((null val) nil)
	  ((keymapp val)
	   (if (symbolp val) (symbol-function val) val))
	  (t
	   ;; otherwise, it is a function which is either a real function or
	   ;; a keymap fset to val.
	   (let ((fun (symbol-function val)))
	     (if (or (null fun) (keymapp fun)) fun val))))))

;; auto fill
;; modified by Masahiko Sato, 1990.3.20, 1990.6.27
;; modified by Masakazu Takahashi, 1989.8.13, 1989.11.6, 1990.6.22

(defun j-kanji-fill ()
  "Auto fill fuction for Japanese text. This function returns t if a new
line is inserted, and returns nil otherwise."
  (let ((val nil) (eolp nil) point)
    (save-excursion
      (if (>= (current-column) fill-column)
	  (let ((p (point)))
	    ;; move cursor to newline point
	    (move-to-column (1- fill-column))
	    (if (and (< (current-column) fill-column)
		     (not (eolp)))
		(forward-char 1))
	    (if (and (< (following-char) ?\200)
		     (not (looking-at " \\|\t")))
		(while (and (< (preceding-char) ?\200)
			    (not (and (/= (preceding-char) ? )
				      (/= (preceding-char) ?\t)
				      (looking-at " \\|\t")))
			    (< 0 (current-column)))
		  (backward-char 1)))
	    (kinsoku-shori)
	    ;; newline
	    (if (< (point) p)
		(progn
		  (delete-horizontal-space)
		  (setq eolp (eolp))
		  (insert ?\n)
 		  ;; modified by Masakazu Takahashi, 1990.6.22
 		  (and fill-prefix (stringp fill-prefix) (insert fill-prefix))
		  (setq point (point))
		  (setq val t))))))
    (if eolp (goto-char point))
    val))

(defun j-do-auto-fill ()
  (let* ((c (current-column)) (cont (> c fill-column)))
    (while cont
      (run-hooks 'auto-fill-hook)
      ;; if current-column doesn't change after do-auto-fill, we exit from
      ;; the while loop by setting cont to be nil.  also, it is necessary
      ;; that current-column > fill-column to continue the loop.
      (setq cont (and (< (current-column) c) (> (current-column) fill-column)))
      (setq c (current-column)))))

(defun skk-auto-fill-mode (arg)
  (interactive "P")
  (let ((auto-fill
	 (cond ((null arg) (not auto-fill-hook))
	       ((> (prefix-numeric-value arg) 0) t)
	       (t nil))))
    (auto-fill-mode (if auto-fill 1 -1))
    (or skk-mode (skk-mode 1))
    (run-hooks 'skk-auto-fill-mode-hook)))

;; kana inserting functions

(defun j-insert (table)
  (let ((char (assoc j-prefix table)) str)
    (if (null char)
	;; j-prefix not found in the table
	(setq j-prefix ""
	      unread-command-char last-command-char)
      (if (and j-kakutei-early j-henkan-active) (j-kakutei))
      (setq str (if j-katakana (cdr (cdr char)) (car (cdr char))))
      (j-insert-str str)
      (if j-okurigana
	  (j-set-okurigana)
	(setq j-prefix ""))
      ;; Modified at the suggestion of Kazuo Hirokawa, 1992.3.9.
      (if (and auto-fill-hook (not j-henkan-on)) (j-do-auto-fill)))))

;; the following function relies on the dynamic variable vip-ovwrt-limit
;; used in vip version 4.
;; this style of coding should be avoided if possible.

(defun j-ovwrt-len (len)
  (if (and skk-use-vip (boundp 'vip-ovwrt-limit))
      (min (- vip-ovwrt-limit (point))
	   (- (save-excursion (end-of-line) (point)) (point))
	   len)
    (min (- (save-excursion (end-of-line) (point)) (point)) len)))

(defun j-insert-str (str)
  "Insert STR and, if necessary, call self-insert-after-hook, and take care of
overwrite-mode"
  (insert str)
  (or j-henkan-on
      (progn
	(if self-insert-after-hook
	    (funcall self-insert-after-hook (- (point) (length str)) (point)))
	;; (boundp 'vip-ovwrt-limit) means that this function is within the
	;; dynamic scope of vip-overwrite-execute
	(if (or overwrite-mode (boundp 'vip-ovwrt-limit))
	    (j-del-char-with-pad (j-ovwrt-len (length str)))))))

(defun j-del-char-with-pad (count)
  "Delete exactly COUNT characters, by inserting a space at the end,
if necessary, for adjustment."
  (let ((p (point)) dist)
    (save-excursion
      (setq dist (- (progn (forward-char count) (point)) p)))
    (delete-char count)
    (or (= count dist)
	(progn
	  (insert " ")
	  (backward-char 1)))))

(defun j-set-okurigana ()
  (let (w-head)
    (move-marker j-henkan-end-point j-okurigana-start-point)
    (save-excursion
      ;; just in case
      (goto-char j-okurigana-start-point)
      (or (looking-at "*") (insert "*")))
    (setq w-head (buffer-substring
		  j-henkan-start-point
		  j-henkan-end-point))
    (setq j-henkan-key2
	  (concat w-head
		  "*"
		  (setq j-henkan-okurigana
			(buffer-substring
			 (1+ j-okurigana-start-point)
			 (point)))))
    (setq j-henkan-key (concat w-head j-okuri-char))
    (setq j-prefix ""
	  j-okuri-char "")
    (delete-region
     j-okurigana-start-point (1+ j-okurigana-start-point))
    (j-henkan)
    (setq j-okurigana nil)))

(defun j-insert-a ()
  (interactive)
  (j-insert skk-roma-kana-a))

(defun j-insert-i ()
  (interactive)
  (j-insert skk-roma-kana-i))

(defun j-insert-u ()
  (interactive)
  (j-insert skk-roma-kana-u))

(defun j-insert-e ()
  (interactive)
  (j-insert skk-roma-kana-e))

(defun j-insert-o ()
  (j-insert skk-roma-kana-o))

(defun j-erase-prefix ()
  "erase j-prefix characters if skk-echo is on."
  (if skk-echo
      (delete-region j-kana-start-point (point))))

(defun j-insert-prefix (&optional char)
  "insert j-prefix if skk-echo is on"
  (if skk-echo (insert (or char j-prefix))))

;; other inserting functions

(defun j-insert-comma (count)
  (interactive "p")
  (if (eq last-command 'j-completion)
      (j-previous-completion)
    (j-self-insert count)))

(defun j-insert-period (count)
  (interactive "p")
  (if (eq last-command 'j-completion)
      (progn
	(setq this-command 'j-completion)
	(j-completion nil))
    (j-self-insert count)))

;; kana input by romaji to kana conversion

(defun j-kana-input ()
  "Start romaji-to-kana conversion.  The character `o' calls this function in
order to treat `ohta' etc."
  (interactive)
  (if (and j-kakutei-early j-henkan-active) (j-kakutei))
  (let ((char (char-to-string last-command-char))
	;; don't echo key strokes in the minibuffer
	(echo-keystrokes 0)
	(cont t))
    (if skk-isearch-message (j-isearch-message))
    (if (= last-command-char ?o)
	(progn
	  (j-insert-o)
	  (if skk-isearch-message
	      (progn
		(j-isearch-message)
		(setq cont nil)
		;; thus you must type `oota' instead of `ohta' when inputting
		;; isearch string
		)
	    (setq j-prefix char)
	    (set-marker j-kana-start-point (point))))
      (setq j-prefix char)
      (set-marker j-kana-start-point (point))
      (j-insert-prefix))
    (condition-case conditions
	(while cont
	  (let* ((r-char (read-char))
		 (r-char (if (and j-henkan-on
				  (not j-henkan-active)
				  ;; we must distinguish the two cases where
				  ;; skk-echo is on and off
				  (= j-henkan-start-point
				     (if skk-echo (1- (point)) (point)))
				  (< 64 r-char) (< r-char 91))
			     ;; this case takes care of the rare case where
			     ;; one types two characters in upper case
			     ;; consequtively.  For example, one sometimes
			     ;; types "TE" when one should type "Te"
			     (+ 32 r-char)
			   r-char))
		 (input (char-to-string r-char))
		 (prefix (concat j-prefix input))
		 (next (j-assoc prefix skk-rom-kana-rule-list)))
	    (if skk-isearch-message (j-isearch-message))
	    (cond (next
		   (let ((newprefix (car next))
			 (output (car (cdr next))))
		     (j-erase-prefix)
		     (if output
			 (progn
			   (j-insert-str
			    (if j-katakana (car output) (cdr output)))
			   (if j-okurigana (j-set-okurigana))))
		     (if newprefix
			 (progn
			   (set-marker j-kana-start-point (point))
			   (insert newprefix)
			   (setq j-prefix newprefix))
		       (setq cont nil)
		       (setq j-prefix ""))))
		  ((string= j-prefix "n")
		   (cond ((= (aref skk-char-type-vector r-char) 3) ;; vowel
			  (j-erase-prefix)
			  (setq cont nil)
			  (setq unread-command-char r-char))
			 ((= r-char ?y)
			  (setq j-prefix prefix)
			  (j-insert-prefix ?y))
			 ((and j-okurigana (= r-char 32))
			  (setq unread-command-char ?n))
			 (t
			  (j-erase-prefix)
			  (j-insert-str (if j-katakana "" ""))
			  (set-marker j-kana-start-point (point))
			  (setq j-prefix "")
			  (setq cont nil)
			  (setq unread-command-char r-char))))
		  ((string= j-prefix "o")
		   (if (= r-char ?h)
		       (progn
			 (setq j-prefix prefix)
			 (j-insert-prefix "h"))
		     (setq cont nil
			   j-prefix ""
			   unread-command-char r-char)))
		  ((string= j-prefix "oh")
		   (if (= (aref skk-char-type-vector r-char) 3)
		       ;; vowel
		       (progn
			 (setq cont nil)
			 (j-erase-prefix)
			 (setq j-prefix "h")
			 (setq unread-command-char r-char))
		     (j-erase-prefix)
		     (j-insert-str (if j-katakana "" ""))
		     (setq cont nil
			   j-prefix ""
			   unread-command-char r-char)))
		  ((and (string= input j-prefix)
			(= (aref skk-char-type-vector r-char) 1))
		   (j-erase-prefix)
		   (j-insert-str (if j-katakana "" ""))
		   (set-marker j-kana-start-point (point))
		   (j-insert-prefix r-char))
		  ((j-member prefix skk-prefix-list)
		   (setq j-prefix prefix)
		   (j-insert-prefix input))
		  ((= (aref skk-char-type-vector r-char) 3) ;; vowel
		   (setq cont nil)
		   (j-erase-prefix)
		   (setq unread-command-char r-char))
		  ((= (aref skk-char-type-vector r-char) 4)
		   ;; j-set-henkan-point
		   (let ((last-command-char r-char))
		     (j-erase-prefix)
		     (j-set-henkan-point)))
		  ((= (aref skk-char-type-vector r-char) 5) ;; delete prefix
		   (setq cont nil)
		   (if (and j-okurigana
			    (= (char-after (+ j-okurigana-start-point 0)) ?*))
		       (progn
			 (delete-region j-okurigana-start-point
					(1+ j-okurigana-start-point))
			 (set-marker j-kana-start-point
				     (+ j-okurigana-start-point 0))
			 (setq j-okurigana nil)))
		   (or (string= j-prefix "")
		       (if skk-echo
			   (j-erase-prefix)
			 (message "Deleted prefix \"%s\"." j-prefix)))
		   (setq j-prefix ""))
		  (t
		   (setq cont nil)
		   (j-erase-prefix)
		   (set-marker j-kana-start-point nil)
		   (setq j-prefix "")
		   (setq unread-command-char r-char)))))
      (quit
       (setq j-prefix "")
       (j-erase-prefix)
       (set-marker j-kana-start-point nil)
       (ding)))))

;; mode setup

(defun skk-mode (&optional arg)
  "SKK mode enables input of Japanese texts without affecting the
functionality of the original mode. With negative ARG, exits from SKK mode."
  (interactive "P")
  (setq skk-mode
	(if j-mode-line-format
	    (cond ((null arg) (not skk-mode))
		  ((> (prefix-numeric-value arg) 0) t)
		  (t nil))
	  ;; if skk-mode has not been called in this buffer, always
	  ;; enter skk-mode
	  t))
  (if (and (> (minibuffer-depth) 0) (boundp 'vip-current-mode))
      (setq vip-read-string-hook
	    '(lambda ()
	       (define-key minibuffer-local-map "\C-j" 'j-kakutei)
	       (define-key minibuffer-local-map "\C-m" 'j-newline))))
  (if (not skk-mode)
      ;; exit skk-mode
      (progn
	(if skk-use-vip (vip-change-mode-to-insert))
	;; Tsugutomo Enami added the following line, 1990.7.13
	(let ((skk-mode t)) (j-kakutei))
	;; the following line modified by Tsugutomo Enami, 1990.7.13
	(setq mode-line-format j-mode-line-format)
	(use-local-map j-current-local-map)
	(set-buffer-modified-p (buffer-modified-p)))
    ;; enter skk-mode
    (if (not j-skk-mode-invoked)
	;; enter skk-mode for the first time in this session
	(progn
	  (setq j-skk-mode-invoked t)
	  ;; modified by Tsugutomo Enami, 1990.8.26
	  (load "~/.skk" t)
	  (setq j-kakutei-early
		(and skk-kakutei-early (not skk-process-okuri-early)))
	  (if skk-server-host (j-open-server))
	  (if skk-keep-record
	      (j-create-file
	       skk-record-file
	       "I have created a SKK record file for you."))
	  (let ((inhibit-quit t))
	    (save-window-excursion
	      (set-buffer (get-buffer-create " *skk-jisyo*"))
	      ;; modified at the suggestion of Masakazu Takahashi, 1992.1.26.
	      (setq case-fold-search nil)
	      (erase-buffer)
	      (if skk-jisyo-code (setq kanji-fileio-code skk-jisyo-code))
	      (or (file-exists-p skk-jisyo)
		  (j-create-file
		   skk-jisyo
		   "I have created an empty Jisyo file for you."))
	      (insert-file-contents skk-jisyo)
	      (goto-char (point-min))
	      ;; skip and delete initial comment lines if any
	      (while (and (looking-at ";; ")
			  (not (looking-at ";; okuri-ari")))
		(end-of-line)
		(forward-char 1))
	      (delete-region (point-min) (point))
	      (if (looking-at ";; okuri-ari")
		  (if (search-forward "\n;; okuri-nasi" nil t)
		      (progn
			(beginning-of-line)
			(let ((okuri-part
			       (buffer-substring (point-min) (point))))
			  (save-excursion
			    (set-buffer
			     (get-buffer-create " *skk-okuri-jisyo*"))
			    ;; modified at the suggestion of
			    ;; Masakazu Takahashi, 1992.1.26.
			    (setq case-fold-search nil)
			    (erase-buffer)
			    ;; modified by Tomoyuki Hiro, 1992.1.21
			    (if skk-jisyo-code
				(setq kanji-fileio-code skk-jisyo-code))
			    (insert okuri-part)
			    (set-buffer-modified-p nil)))
			(delete-region (point-min) (point)))
		    (error
		     (concat "Could not find the starting point for "
			     "okuri-nasi entries.")))
		(require 'skk-tools)
		(j-reformat-jisyo-buffer))
	      (if skk-use-vip
		  (progn
		    ;; we should be able to use (require 'vip) eventually.
		    (if (not (boundp 'vip-current-mode)) (load "vip"))
		    ;; check vip's version.  this modification will not be
		    ;; necessary for vip4.x.
		    (if (string= (vip-version)
				 "VIP version 3.5 of September 15, 1987")
			(progn
			  (message "")
			  (fset 'vip-change-mode 'vip-ch-mode)))))
	      ;; let this buffer be not modified initially.
	      (set-buffer-modified-p nil)))
	  (provide 'skk)))
    (if (or (not j-mode-line-format) (> (minibuffer-depth) 0))
	;; enter skk-mode for the first time in this buffer or the buffer
	;; is the minibuffer
	(progn
	  (if (and skk-use-vip (= (minibuffer-depth) 0))
	      (let ((skk-mode nil)) (vip-change-mode-to-insert)))
	  ;; save current mode
	  ;; modified by Tsugutomo Enami, 1990.7.13.
	  (setq j-mode-line-format mode-line-format
		j-current-local-map (let ((map (current-local-map)))
				      (if map (copy-keymap map) nil)))
	  ;; modified at the suggestion of Masakazu Takahashi, 1990.7.8
	  (setq j-emacs-local-map
		(let ((map (current-local-map)))
		  (if map (copy-keymap map)
		    (make-sparse-keymap))))
	  (define-key j-emacs-local-map "\C-j" 'j-kakutei)
	  (j-setup-skk-map)
	  (j-setup-skk-zenkaku-map)
	  (j-setup-skk-abbrev-map)
	  (run-hooks 'skk-mode-hook))
      (if (and skk-use-vip (= (minibuffer-depth) 0))
	  (let ((skk-mode nil)) (vip-change-mode-to-insert))))
    (setq j-mode t
	  j-katakana nil)
    (use-local-map skk-map)
    (define-key minibuffer-local-map "\C-j" 'j-kakutei)
    (define-key minibuffer-local-map "\C-m" 'j-newline)
    (define-key minibuffer-local-completion-map "\C-j" 'j-kakutei)
    (define-key minibuffer-local-completion-map "\C-m" 'j-newline)
    (define-key minibuffer-local-ns-map "\C-j" 'j-kakutei)
    (define-key minibuffer-local-ns-map "\C-m" 'j-newline)
    (setq mode-line-format
	  (cons "" (cons 'j-mode-line-skk (cdr j-mode-line-format))))
    (j-change-mode-line skk-hirakana-mode-string)))

(defun j-setup-skk-map ()
  (setq skk-map (j-convert-to-vector j-emacs-local-map))
  (define-key skk-map "\C-g" 'j-keyboard-quit)
  (define-key skk-map "\C-j" 'j-kakutei)
  (define-key skk-map "\C-m" 'j-newline)
  (define-key skk-map "\t" 'j-try-completion)
  (define-key skk-map "\ " 'j-start-henkan)
  (define-key skk-map "!" 'j-self-insert)
  (define-key skk-map "\"" 'j-self-insert)
  (define-key skk-map "#" 'j-self-insert)
  (define-key skk-map "$" 'j-display-code-for-char-at-point)
  (define-key skk-map "%" 'j-self-insert)
  (define-key skk-map "&" 'j-self-insert)
  (define-key skk-map "'" 'j-self-insert)
  (define-key skk-map "\(" 'j-self-insert)
  (define-key skk-map "\)" 'j-self-insert)
  (define-key skk-map "*" 'j-self-insert)
  (define-key skk-map "+" 'j-self-insert)
  (define-key skk-map "," 'j-insert-comma)
  (define-key skk-map "-" 'j-self-insert)
  (define-key skk-map "." 'j-insert-period)
  (define-key skk-map "/" 'j-abbrev-input)
  (define-key skk-map "0" 'j-self-insert)
  (define-key skk-map "1" 'j-self-insert)
  (define-key skk-map "2" 'j-self-insert)
  (define-key skk-map "3" 'j-self-insert)
  (define-key skk-map "4" 'j-self-insert)
  (define-key skk-map "5" 'j-self-insert)
  (define-key skk-map "6" 'j-self-insert)
  (define-key skk-map "7" 'j-self-insert)
  (define-key skk-map "8" 'j-self-insert)
  (define-key skk-map "9" 'j-self-insert)
  (define-key skk-map ":" 'j-self-insert)
  (define-key skk-map ";" 'j-self-insert)
  (define-key skk-map "<" 'j-self-insert)
  (define-key skk-map "=" 'j-self-insert)
  (define-key skk-map ">" 'j-self-insert)
  (define-key skk-map "?" 'j-self-insert)
  (define-key skk-map "@" 'j-today)
  (define-key skk-map "A" 'j-set-henkan-point)
  (define-key skk-map "B" 'j-set-henkan-point)
  (define-key skk-map "C" 'j-set-henkan-point)
  (define-key skk-map "D" 'j-set-henkan-point)
  (define-key skk-map "E" 'j-set-henkan-point)
  (define-key skk-map "F" 'j-set-henkan-point)
  (define-key skk-map "G" 'j-set-henkan-point)
  (define-key skk-map "H" 'j-set-henkan-point)
  (define-key skk-map "I" 'j-set-henkan-point)
  (define-key skk-map "J" 'j-set-henkan-point)
  (define-key skk-map "K" 'j-set-henkan-point)
  (define-key skk-map "L" 'j-zenkaku-eiji)
  (define-key skk-map "M" 'j-set-henkan-point)
  (define-key skk-map "N" 'j-set-henkan-point)
  (define-key skk-map "O" 'j-set-henkan-point)
  (define-key skk-map "P" 'j-set-henkan-point)
  (define-key skk-map "Q" 'j-set-henkan-point-subr)
  (define-key skk-map "R" 'j-set-henkan-point)
  (define-key skk-map "S" 'j-set-henkan-point)
  (define-key skk-map "T" 'j-set-henkan-point)
  (define-key skk-map "U" 'j-set-henkan-point)
  (define-key skk-map "V" 'j-set-henkan-point)
  (define-key skk-map "W" 'j-set-henkan-point)
  (define-key skk-map "X" 'j-purge-from-jisyo)
  (define-key skk-map "Y" 'j-set-henkan-point)
  (define-key skk-map "Z" 'j-set-henkan-point)
  (define-key skk-map "\[" 'j-self-insert) 
  (define-key skk-map "\\" 'j-input-by-code-or-menu)
  (define-key skk-map "\]" 'j-self-insert)
  (define-key skk-map "^" 'j-self-insert)
  (define-key skk-map "_" 'j-self-insert)
  (define-key skk-map "`" 'j-self-insert)
  (define-key skk-map "a" 'j-insert-a)
  (define-key skk-map "b" 'j-kana-input)
  (define-key skk-map "c" 'j-kana-input)
  (define-key skk-map "d" 'j-kana-input)
  (define-key skk-map "e" 'j-insert-e)
  (define-key skk-map "f" 'j-kana-input)
  (define-key skk-map "g" 'j-kana-input)
  (define-key skk-map "h" 'j-kana-input)
  (define-key skk-map "i" 'j-insert-i)
  (define-key skk-map "j" 'j-kana-input)
  (define-key skk-map "k" 'j-kana-input)
  (define-key skk-map "l" 'j-mode-off)
  (define-key skk-map "m" 'j-kana-input)
  (define-key skk-map "n" 'j-kana-input)
  (define-key skk-map "o" 'j-kana-input)
  (define-key skk-map "p" 'j-kana-input)
  (define-key skk-map "q" 'j-toggle-kana)
  (define-key skk-map "r" 'j-kana-input)
  (define-key skk-map "s" 'j-kana-input)
  (define-key skk-map "t" 'j-kana-input)
  (define-key skk-map "u" 'j-insert-u)
  (define-key skk-map "v" 'j-kana-input)
  (define-key skk-map "w" 'j-kana-input)
  (define-key skk-map "x" 'j-previous-candidate)
  (define-key skk-map "y" 'j-kana-input)
  (define-key skk-map "z" 'j-kana-input)
  (define-key skk-map "\{" 'j-self-insert)
  (define-key skk-map "|" 'j-self-insert)
  (define-key skk-map "\}" 'j-self-insert)
  (define-key skk-map "~" 'j-self-insert)
  (define-key skk-map "\177" 'j-delete-backward-char)
  ;;(define-key skk-map "\M-k" 'j-katakana-henkan)
  ;;(define-key skk-map "\M-h" 'j-hiragana-henkan)
  ;;(define-key skk-map "\M-z" 'j-zenkaku-henkan)
  )

;; modified by Masahiko Sato, 1991.4.6
;; modified by Masakazu Takahashi, 1990.5.11

(defun j-setup-skk-zenkaku-map ()
  (setq skk-zenkaku-map (j-convert-to-vector j-emacs-local-map))
  (let ((i 0))
    (while (< i 128)
      (if (aref skk-zenkaku-vector i)
	  (define-key skk-zenkaku-map (char-to-string i) 'j-zenkaku-insert) )
      (setq i (1+ i)))))

(defun j-setup-skk-abbrev-map ()
  (setq skk-abbrev-map (j-convert-to-vector j-emacs-local-map))
  (define-key skk-abbrev-map "\C-g" 'j-keyboard-quit)
  (define-key skk-abbrev-map "\C-j" 'j-kakutei)
  (define-key skk-abbrev-map "\t" 'j-try-completion)
  (define-key skk-abbrev-map "\C-m" 'j-newline)
  (define-key skk-abbrev-map "\C-q" 'j-zenkaku-henkan)
  (define-key skk-abbrev-map "." 'j-abbrev-period)
  (define-key skk-abbrev-map "," 'j-abbrev-comma)
  (define-key skk-abbrev-map "\ " 'j-start-henkan)
  (define-key skk-abbrev-map "\177" 'j-delete-backward-char)
  ;;(define-key skk-abbrev-map "\M-z" 'j-zenkaku-henkan)
  )

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-self-insert (arg)
  (interactive "P")
  (let ((str (aref skk-input-vector last-command-char)))
    (if str
	(let ((count (prefix-numeric-value arg)))
	  (while (> count 0)
	    (j-insert-str str)
	    (setq count (1- count))))
      (j-emulate-original-map arg))
    (if j-henkan-active (j-kakutei))))

(defun j-zenkaku-insert (count)
  (interactive "p")
  (let* ((str (aref skk-zenkaku-vector last-command-char))
	 (len (length str)))
    (while (> count 0)
      (j-insert-str str)
      (setq count (1- count)))))

(defun j-zenkaku-self-insert (count)
  (interactive "p")
  (while (> count 0)
    (j-insert-str
     (concat (char-to-string 163) (char-to-string (+ last-command-char 128))))
    (setq count (1- count))))

(defun j-mode-off ()
  "return to old ascii mode."
  (interactive)
  (j-kakutei)
  (setq j-mode nil
	j-prefix ""
	j-zenkaku nil)
  (use-local-map j-emacs-local-map)
  (j-change-mode-line skk-mode-string))

(defun j-zenkaku-eiji ()
  "zenkaku-eiji input"
  (interactive)
  (j-kakutei)
  (setq j-mode nil
	j-prefix ""
	j-zenkaku t)
  (use-local-map skk-zenkaku-map)
  (j-change-mode-line skk-zenei-mode-string))

(defun j-abbrev-input ()
  (interactive)
  (if (and j-henkan-on (not j-henkan-active)) (error ""))
  (j-kakutei)
  (j-set-henkan-point-subr)
  (use-local-map skk-abbrev-map)
  (setq j-abbrev t))

;; vip related functions

(defun vip-ch-mode (new-mode)
  "(NEW-MODE)  Change mode to NEW-MODE.  NEW-MODE is either emacs-mode,
vi-mode, or insert-mode."
  (if (> (minibuffer-depth) 0)
      (error "Can't change mode in minibuffer!")
    (or (eq new-mode vip-current-mode)
	(progn
	  (cond ((eq new-mode 'vi-mode)
		 (if (eq vip-current-mode 'insert-mode)
		     (progn
		       (if (and skk-mode j-mode) (j-kakutei))
		       (vip-copy-region-as-kill (point) vip-insert-point)
		       (vip-repeat-insert-command))
		   (setq vip-emacs-local-map (current-local-map)
			 vip-emacs-mode-line-buffer-identification
			 mode-line-buffer-identification
			 vip-insert-local-map (vip-copy-keymap
					       (current-local-map))))
		 (vip-change-mode-line "Vi:   ")
		 (if skk-mode (j-change-mode-line skk-mode-string))
		 (use-local-map vip-mode-map))
		((eq new-mode 'insert-mode)
		 (move-marker vip-insert-point (point))
		 (if (eq vip-current-mode 'emacs-mode)
		     (setq vip-emacs-local-map (current-local-map)
			   vip-emacs-mode-line-buffer-identification
			   mode-line-buffer-identification
			   vip-insert-local-map (vip-copy-keymap
						 (current-local-map))))
		 (vip-change-mode-line "Insert")
		 (if skk-mode
		     (j-change-mode-line
		      (if j-mode (if j-katakana skk-katakana-mode-string
				   skk-hirakana-mode-string)
			(if j-zenkaku skk-zenei-mode-string skk-mode-string))))
		 (if skk-mode
		     (if j-mode
			 (use-local-map skk-map)
		       (if j-zenkaku
			   (use-local-map skk-zenkaku-map)
			 (use-local-map vip-insert-local-map)
			 (local-set-key "\C-j" 'j-kakutei)))
		   (use-local-map vip-insert-local-map))
		 (local-set-key "\e" 'vip-change-mode-to-vi)
		 (local-set-key "\C-z" 'vip-ESC)
		 (local-set-key "\C-h"
				(if vip-help-in-insert-mode 'help-command
				  'j-delete-backward-char))
		 (local-set-key "\C-w"
				'vip-delete-backward-word))
		((eq new-mode 'emacs-mode) 
		 (vip-change-mode-line "Emacs:")
		 (if skk-mode (j-change-mode-line skk-mode-string))
		 (use-local-map vip-emacs-local-map)))
	  (setq vip-current-mode new-mode)
	  (vip-refresh-mode-line)))))

(if (not (fboundp 'j-original-kill-emacs))
    (fset 'j-original-kill-emacs (symbol-function 'kill-emacs)))

(defun kill-emacs (&optional query)
  "Save SKK Jisyo buffer, exit the Emacs job and kill it."
  (interactive "P")
  (skk-save-jisyo)
  (if (process-status "skkservd")
      (process-send-string "skkservd" "0")) ;; disconnect server
  (j-original-kill-emacs query))

(defun skk-kill-emacs-without-saving-jisyo ()
  "Kill Emacs without saving jisyo."
  (interactive)
  (if (yes-or-no-p "Do you really wish to kill emacs without saving Jisyo? ")
      (progn
	(if (process-status "skkservd")
	    (process-send-string "skkservd" "0")) ;; disconnect server
	(j-original-kill-emacs))))

(defun skk-save-jisyo ()
  "Save jisyo buffer into jisyo file."
  (interactive)
  (if (or
       (not (get-buffer " *skk-jisyo*"))
       (and
	(not (buffer-modified-p (get-buffer " *skk-jisyo*")))
	(not (buffer-modified-p (get-buffer " *skk-okuri-jisyo*")))))
      (message "No need to save Jisyo.")
    ;; check if the header lines for the two jisyo buffers are ok.
    ;; error corrected by Masakazu Takahashi, 1992.1.26.
    (save-excursion
      (set-buffer " *skk-jisyo*")
      (goto-char (point-min))
      (or (string=
	   (buffer-substring
	    (point)
	    (progn (end-of-line) (point)))
	   ";; okuri-nasi entries.")
	  (if (y-or-n-p
	       "The header line of skk-jisyo buffer is broken, shall I fix it? ")
	      (progn
		(delete-region (point-min) (point))
		(insert ";; okuri-nasi entries."))
	    (error
	     "Jisyo not saved.  Please check the buffer \" *skk-jisyo*\"")))
      (set-buffer " *skk-okuri-jisyo*")
      (goto-char (point-min))
      (or (string=
	   (buffer-substring
	    (point)
	    (progn (end-of-line) (point)))
	   ";; okuri-ari entries.")
	  (if (y-or-n-p
	       "The header line of skk-okuri-jisyo buffer is broken, shall I fix it? ")
	      (progn
		(delete-region (point-min) (point))
		(insert ";; okuri-ari entries."))
	    (error
	     "Jisyo not saved.  Please check the buffer \" *skk-okuri-jisyo*\""))))
    (message "Saving Jisyo...")
    (get-buffer-create " *record*")
    (let* ((inhibit-quit t)
	   lines
	   (skk-jisyo (expand-file-name skk-jisyo))
	   (file-modes (file-modes skk-jisyo)))
      (save-window-excursion
	;; Originally, call-process was used to execute the Unix mv command.
	;; Masakazu Takahashi modified it as follows on 1991.10.1.
	(rename-file skk-jisyo (concat skk-jisyo ".BAK") t)
	(j-create-file skk-jisyo)
	(set-file-modes skk-jisyo file-modes)
	(set-buffer " *skk-okuri-jisyo*")
	(setq lines (count-lines (point-min) (point-max)))
	(j-save-jisyo)
	(set-buffer " *skk-jisyo*")
	(setq lines (+ lines (count-lines (point-min) (point-max))))
	(j-save-jisyo)
	(message "Saving Jisyo...done.")
	(if (and skk-keep-record (> j-count-kakutei 0))
	    (progn
	      (set-buffer " *record*")
	      (erase-buffer)
	      (insert-file skk-record-file)
	      (goto-char (point-min))
	      (insert
	       (format "%s  Ͽ: %3d  : %4d  Ψ: %3d%%  : %6d\n"
		       (current-time-string)
		       j-count-touroku
		       j-count-kakutei
		       (/ (* 100 (- j-count-kakutei j-count-touroku))
			  j-count-kakutei)
		       lines))
	      (write-region (point-min) (point-max) skk-record-file nil 'nomsg)
	      (set-buffer-modified-p nil)))
	;; modified by Masakazu Takahashi, 1991.3.27
	(setq j-count-touroku 0
	      j-count-kakutei 0)
	;; set jisyo buffer to be not modified
	(set-buffer " *skk-jisyo*")
	(set-buffer-modified-p nil)
	(set-buffer " *skk-okuri-jisyo*")
	(set-buffer-modified-p nil)))))

(defun j-save-jisyo ()
  (let ((go-on t))
    (goto-char (point-min))
    (while go-on
      (set-mark (point))
      (goto-char (+ 50000 (point)))
      (end-of-line)
      (write-region (mark) (point) skk-jisyo t 'nomsg)
      (if (eobp) (setq go-on nil)))))

;; input functions

;; modified by Wataru Matsui, 1991.8.19
;; modified by Masahiko Sato, 1990.3.19
;; modified by Masakazu Takahashi, 1989.8.10

(defun j-newline (arg)
  (interactive "p")
  (let ((no-nl (and skk-egg-like-newline j-henkan-on)))
    (if (> (minibuffer-depth) 0)
	(progn
	  (j-kakutei)
	  (if no-nl
	      nil
	    (setq j-mode nil)
	    (exit-minibuffer)))
      (if (j-kakutei) (setq arg (1- arg)))
      (if no-nl
	  nil
	(j-emulate-original-map arg)))))

;; modified by Masakazu Takahashi, 1991.11.17
;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-delete-backward-char (count)
  (interactive "P")
  (if j-okurigana
      (progn
	(setq j-okurigana nil)
	(if (and (char-after (+ j-okurigana-start-point 0))
		 (= (char-after (+ j-okurigana-start-point 0)) ?*))
	    (delete-region j-okurigana-start-point
			   (1+ j-okurigana-start-point)))))
  (if (and j-henkan-on (= (point) j-henkan-start-point))
      (progn
	(setq j-henkan-count 0)
	(j-kakutei))
    ;; modified by Tsugutomo Enami, 1991.10.2
    (if j-henkan-active 
	(let ((p (+ j-henkan-end-point 0)))
	  (delete-backward-char (prefix-numeric-value count) count)
	  (if (< (point) p) (j-kakutei)) )
      (j-emulate-original-map count))))

;; input by EUC/JIS code or by menu

;; The function j-input-by-code, which was used until version 4.20,
;; is now replaced by the following new function.  The code for this
;; function was contributed by nakamura@pluto.ai.kyutech.ac.jp

;; modified by Masahiko Sato, 1991.4.4
;; modified by Jun-ichi Nakamura, 1991.3.29

(defun j-input-by-code-or-menu ()
  "This function lets the user input a JIS character either by its code or
by following the menu provided by the system."
  (interactive)
  (let* ((str (read-string "JIS or EUC code (00nn or CR for Jump Menu): "))
	 (n1 (if (string= str "") 128
	       (+ (* 16 (j-jis-string-to-hex (substring str 0 1)))
		  (j-string-to-hex (substring str 1 2)))))
	 (n2 (if (string= str "") 128
	       (+ (* 16 (j-jis-string-to-hex (substring str 2 3)))
		  (j-string-to-hex (substring str 3 4))))))
    (insert
     (if (> n1 160) (j-char-to-string n1 n2)
       (j-input-by-code-or-menu-0 n1 n2)))))

(defun j-string-to-hex (str)
  (let ((num (string-to-char str)))
  (cond ((> num 96) (- num 87)) ;; a-f
	((> num 64) (- num 55)) ;; A-F
	((> num 47) (- num 48)) ;; 0-9
	(t (error "")))))

(defun j-jis-string-to-hex (str)
  (let ((num (string-to-char str)))
  (cond ((> num 96) (- num 87)) ;; a-f
	((> num 64) (- num 55)) ;; A-F
	((> num 47) (- num 40)) ;; 0-9
	(t (error "")))))

(defun j-char-to-string (n1 n2)
  (concat (char-to-string n1) (char-to-string n2)))

(defun j-next-n2-code (n)
  (if (<= (setq n (1+ n)) j-code-n2-max) n j-code-n2-min))

(defun j-previous-n2-code (n)
  (if (<= j-code-n2-min (setq n (1- n))) n j-code-n2-max))

(defun j-next-n1-code (n)
  (if (<= (setq n (1+ n)) j-code-n1-max) n j-code-n1-min))

(defun j-previous-n1-code (n)
  (if (<= j-code-n1-min (setq n (1- n))) n j-code-n1-max))

(defun j-input-by-code-or-menu-0 (n1 n2)
  (if (= n1 j-code-null)
      (j-input-by-code-or-menu-jump n2)
    (j-input-by-code-or-menu-1 n1 n2)))

(defun j-input-by-code-or-menu-jump (n)
  (let ((kanji-char nil))
    (if (< n j-code-n1-min) (setq n j-input-by-code-or-menu-jump-default))
    (while (not kanji-char)
      (let* ((n-org n)
	     (ch-a (list (j-char-to-string n j-code-n1-min) n j-code-n1-min))
	     (ch-s (list (j-char-to-string n 177) n 177))
	     (ch-d (list (j-char-to-string n 193) n 193))
	     (ch-f (list (j-char-to-string n 209) n 209))
	     (ch-g (list (j-char-to-string n 225) n 225))
	     (ch-h (list (j-char-to-string n 241) n 241))
	     (ch-q (progn
		     (setq n (j-next-n1-code n))
		     (list (j-char-to-string n j-code-n1-min) n j-code-n1-min))
		   )
	     (ch-w (list (j-char-to-string n 177) n 177))
	     (ch-e (list (j-char-to-string n 193) n 193))
	     (ch-r (list (j-char-to-string n 209) n 209))
	     (ch-t (list (j-char-to-string n 225) n 225))
	     (ch-y (list (j-char-to-string n 241) n 241)))
	(save-excursion
	  (message
	   (concat
	    "A:" (car ch-a) "  S:" (car ch-s) "  D:" (car ch-d)
	    "  F:" (car ch-f) "  G:" (car ch-g) "  H:" (car ch-h)
	    "  Q:" (car ch-q) "  W:" (car ch-w) "  E:" (car ch-e)
	    "  R:" (car ch-r) "  T:" (car ch-t) "  Y:" (car ch-y)
	    ))
	  (let ((char (read-char)))
	    (setq kanji-char
		  (cond
		   ((= char ?a) ch-a) ((= char ?s) ch-s)
		   ((= char ?d) ch-d) ((= char ?f) ch-f)
		   ((= char ?g) ch-g) ((= char ?h) ch-h)
		   ((= char ?q) ch-q) ((= char ?w) ch-w)
		   ((= char ?e) ch-e) ((= char ?r) ch-r)
		   ((= char ?t) ch-t) ((= char ?y) ch-y)
		   ((= char ?x)
		    (if (< (setq n (- n-org 2)) j-code-n1-min)
			(setq n j-code-n1-max))
		    nil)
		   ((= char 32)  ;;; space
		    (setq n (j-next-n1-code n))
		    nil)
		   ((= char 63)  ;;; ?
		    (message
		      "%s  EUC: %2x%2x (%3d, %3d), JIS: %2x%2x (%3d, %3d)  [Hit any key to continue.]"
		      (car ch-a)
		      n-org j-code-n1-min n-org j-code-n1-min
		      (- n-org 128) (- j-code-n1-min 128)
		      (- n-org 128) (- j-code-n1-min 128))
		    (read-char)
		    (setq n n-org)
		    nil)
		   (t
		    (message
		     (format "`%c' is not valid here!" char))
		    (sit-for 1)
		    (message "")
		    (setq n n-org)
		    nil)))))))
    (setq j-input-by-code-or-menu-jump-default (car (cdr kanji-char)))
    (j-input-by-code-or-menu-1
     (car (cdr kanji-char)) (car (cdr (cdr kanji-char))))))

(defun j-input-by-code-or-menu-1 (n1 n2)
  (let ((kanji-char nil))
    (while (not kanji-char)
      (let* ((n1-org n1) (n2-org n2)
	     (ch-a (j-char-to-string n1 n2))
	     (ch-s 
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-d
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-f
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-g
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-h
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-j
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-k
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-l
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-q
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-w
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-e
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-r
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-t
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-y
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     (ch-u
	      (progn (setq n2 (j-next-n2-code n2))
		     (if (= n2 j-code-n2-min) (setq n1 (j-next-n1-code n1)))
		     (j-char-to-string n1 n2)))
	     )
	(save-excursion
	  (message
	   (concat
	    "A:" ch-a " S:" ch-s " D:" ch-d " F:" ch-f
	    " G:" ch-g " H:" ch-h " J:" ch-j " K:" ch-k
	    " L:" ch-l " Q:" ch-q " W:" ch-w " E:" ch-e
	    " R:" ch-r " T:" ch-t " Y:" ch-y " U:" ch-u
	    ))
	  (let ((char (read-char)))
	    (setq kanji-char
		  (cond
		   ((= char ?a) ch-a) ((= char ?s) ch-s)
		   ((= char ?d) ch-d) ((= char ?f) ch-f)
		   ((= char ?g) ch-g) ((= char ?h) ch-h)
		   ((= char ?j) ch-j) ((= char ?k) ch-k)
		   ((= char ?l) ch-l) ((= char ?q) ch-q)
		   ((= char ?w) ch-w) ((= char ?e) ch-e)
		   ((= char ?r) ch-r) ((= char ?t) ch-t)
		   ((= char ?y) ch-y) ((= char ?u) ch-u)
		   ((= char ?x)
		    (if (< (setq n2 (- n2 31)) j-code-n2-min)
			(progn (setq n2 (+ n2 94)
				     n1 (j-previous-n1-code n1))))
		    nil)
		   ((= char 32)  ;;; space
		    (if (= (setq n2 (j-next-n2-code n2))
			   j-code-n2-min)
			(setq n1 (j-next-n1-code n1)))
		    nil)
		   ((= char 63)  ;;; ?
		    (message
		      "%s  EUC: %2x%2x (%3d, %3d), JIS: %2x%2x (%3d, %3d)  [Hit any key to continue.]"
		      ch-a n1-org n2-org n1-org n2-org
		      (- n1-org 128) (- n2-org 128)
		      (- n1-org 128) (- n2-org 128))
		    (read-char)
		    (setq n1 n1-org n2 n2-org)
		    nil)
		   ((= char 62)  ;;; >
		    (if (= (setq n2 (j-next-n2-code n2-org))
			   j-code-n2-min)
			(setq n1 (j-next-n1-code n1-org))
		      (setq n1 n1-org))
		    nil)
		   ((= char 60) ;;; < 
		    (if (= (setq n2 (j-previous-n2-code n2-org))
			   j-code-n2-max)
			(setq n1 (j-previous-n1-code n1-org))
		      (setq n1 n1-org))
		    nil)
		   (t
		    (message "`%c' is not valid here!" char)
		    (sit-for 1)
		    (message "")
		    (setq n1 n1-org n2 n2-org)
		    nil)))))))
    kanji-char))

(defun j-display-code-for-char-at-point ()
  "Displays the code for the charater at the current point."
  (interactive)
  (if (eobp) (error "Cursor is at the end of the buffer."))
  (let* ((str
	  ;; modified by Kazuo Hirokawa, 1992.2.26.
	  (buffer-substring
	   (point)
	   (save-excursion (forward-char 1) (point))))
	 (char-list (mapcar '+ str))
	 (char1 (car char-list)))
    (if (= (length char-list) 1)
	;; ascii character
	(message "`%c'  %2x (%3d)" char1 char1 char1)
      (let* ((char1-j (- char1 128))
	     (char2 (car (cdr char-list)))
	     (char2-j (- char2 128)))
	(message "%s  EUC: %2x%2x (%3d, %3d), JIS: %2x%2x (%3d, %3d)"
		 str char1 char2 char1 char2
		 char1-j char2-j char1-j char2-j)))))

;; henkan routines

(defun j-toggle-kana (arg)
  (interactive "P")
  (if (and j-henkan-on (not j-henkan-active))
      (j-katakana-henkan arg)
    (j-kakutei)
    (setq j-katakana (not j-katakana))
    (j-change-mode-line (if j-katakana skk-katakana-mode-string
			  skk-hirakana-mode-string))))

(defun j-set-henkan-point ()
  (interactive)
  (let* ((last-char (+ 32 last-command-char))
	 (sokuon (and (string= j-prefix (char-to-string last-char))
		      (/= last-char ?o))))
    (if (or (not j-henkan-on) j-henkan-active)
	(j-set-henkan-point-subr)
      ;; prepare for the processing of okurigana if not j-okurigana and
      ;; the preceding character is not a numeric character.
      ;; if the previous char is a numeric character, we assume that the
      ;; user inteded to type the last-command-char in lower case.
      (if (and (not j-okurigana)
	       (not
		(and (not (= j-henkan-start-point (point)))
		     (let ((p1 (char-after (1- (point))))
			   (p2 (char-after (- (point) 2))))
		       (or
			;; previous char is an ascii numeric char
			(and (<= ?0 p1) (<= p1 ?9))
			;; previous char is a jis numeric char
			(and (= p2 163) (<= 176 p1) (<= p1 185)))))))
	  (if skk-process-okuri-early
	      (progn
		(move-marker j-henkan-end-point (point))
		(setq j-henkan-key
		      (concat (buffer-substring j-henkan-start-point
						(if sokuon
						    j-kana-start-point
						  (point)))
			      (if sokuon
				  (if j-katakana "" "")
				"")
			      (char-to-string last-char)))
		(if sokuon (j-erase-prefix))
		(insert (if sokuon
			    (if j-katakana "" "")
			  "")
			" ")
		(setq j-prefix "")
		(j-henkan)
		(delete-backward-char (if sokuon 2 1)))
 	    ;;; modified by Wataru Matsui, 1991.8.16
	    (if (= j-henkan-start-point (point))
		nil
	      (if sokuon
		  (progn
		    (j-erase-prefix)
		    (insert (if j-katakana "" ""))
		    (setq j-prefix "")))
	      (move-marker j-okurigana-start-point (point))
	      (insert "*")
	      (set-marker j-kana-start-point (point))
	      (setq j-okuri-char (char-to-string last-char))
	      (setq j-okurigana t)))))
    (setq unread-command-char last-char)))

;; The following two functions, j-katakana-region and j-katakana-henkan,
;; were contributed by Wataru Matsui on 1991.8.27.

;; j-katakana-region: modified by Wataru Matsui, 199l.11.19.
;;                    modified by Wataru Matsui, 1992. 1.21.
;;    Arguments can be markers as well as numbers.
;;    Cf. katakana-region in wnn-egg.el, arguments to which 
;;    should not be markers.
;;    If vcontract is non-nil, "" is converted to "".
;;    The idea for this function comes from katakana-region in wnn-egg.el.

(defun j-katakana-region (start end &optional vcontract)
  "Convert hiragana\'s in the region between START and END to katakana\'s.
Arguments start and end can be markers as well as numbers.
If vcontract is non-nil, \"\" is converted to \"\"."
  (interactive "r")
  (let ((mstart (set-marker (make-marker) start))
	(mend (set-marker (make-marker) end)))
    (goto-char mstart)
    (while (re-search-forward  "[-]" mend mend)
      (let ((ch2 (preceding-char)))
	(forward-char -2)
	(insert ?\245 ch2)
	(delete-char 2)))
    (if vcontract
	(progn
	  (goto-char mstart)
	  (while (re-search-forward  "" mend mend)
	    (forward-char -4)
	    (insert "")
	    (delete-char 4))))))

;; j-hiragana-region: contributed by Wataru Matsui, 1991.11.19.
;;                       modified by Wataru Matsui, 1992. 1.21.
;;    Arguments can be markers as well as numbers.
;;    Cf. hiragana-region in wnn-egg.el, arguments to which 
;;    should not be markers.
;;    "" and "" are not regarded as katakana's, since
;;    they don't have corresponding hiragana's.  If optional
;;    argument vexpand is non-nil, "" is converted to "".
;;    The idea for this function comes from hiragana-region in wnn-egg.el.

(defun j-hiragana-region (start end &optional vexpand)
"Convert katakana\'s in the region between START and END to hiragana\'s.
Arguments start and end can be markers as well as numbers.
If vexpand is non-nil, \"\" is converted to \"\".
\"\" and \"\" are left unchanged."
  (interactive "r")
  (let ((mstart (set-marker (make-marker) start))
	(mend (set-marker (make-marker) end)))
    (goto-char mstart)
    (while (re-search-forward  "[-]" mend mend)
      (let ((ch2 (preceding-char)))
	(forward-char -2)
	(insert ?\244 ch2)
	(delete-char 2)))
    (if vexpand
	(progn
	  (goto-char mstart)
	  (while (re-search-forward  "" mend mend)
	    (forward-char -2)
	    (insert "")
	    (delete-char 2))))))

(defun j-zenkaku-region (start end)
  "Convert ascii characters in the region between START and END to the
corresponding zenkaku characters.  This function is copied from wnn-egg.el"
  (interactive "r")
  (save-restriction
    (narrow-to-region start end)
    (goto-char (point-min))
    (while (re-search-forward "[ -~]" (point-max) (point-max))
      (let ((ch (preceding-char)))
	(if (and (<= ?  ch) (<= ch ?~))
	    (progn (delete-char -1)
		   (let ((zen (cdr (assq ch j-zenkaku-alist))))
		     (if zen (insert zen)
		       (insert ?\243 (+ ?\200 ch))))))))))

;; modified by Kazuo Hirokawa, 1991.11.21.

(defun j-katakana-henkan (arg)
  "Convert hiragana\'s between henkan start point and
henkan end point into katakana\'s.  Valid only in  mode."
  (interactive "P")
  (if j-henkan-on
      (if (or j-henkan-active j-katakana)
	  nil
        (setq j-mode t)
	(or (string= j-prefix "") (error ""))
	(if (< (point) j-henkan-start-point)
	    (error "Henkan end point must be after henkan start point."))
	(save-excursion
	  (beginning-of-line)
	  (if (> (point) j-henkan-start-point)
	      (error "Henkan key may not contain a new line character.")))
	(move-marker j-henkan-end-point (point))
	(j-katakana-region j-henkan-start-point j-henkan-end-point t)
	(or arg (j-kakutei))
	)
    (j-emulate-original-map arg)
    ))

;; contributed by Kazuo Hirokawa, 1991.11.21.

(defun j-hiragana-henkan (arg)
"Convert katakana\'s between henkan start point and
henkan end point into hiragana\'s.  Valid only in  mode."
  (interactive "P")
  (if j-henkan-on
      (if (or j-henkan-active (not j-katakana))
	  nil
	(setq j-mode t)
	(or (string= j-prefix "") (error ""))
	(if (< (point) j-henkan-start-point)
	    (error "Henkan end point must be after henkan start point."))
	(save-excursion
	  (beginning-of-line)
	  (if (> (point) j-henkan-start-point)
	      (error "Henkan key may not contain a new line character.")))
	(move-marker j-henkan-end-point (point))
	(j-hiragana-region j-henkan-start-point j-henkan-end-point t)
	(or arg (j-kakutei))
	)
    (j-emulate-original-map arg)
    ))

;; contributed by Kazuo Hirokawa, 1991.11.21.

(defun j-zenkaku-henkan (arg)
  "Convert hankaku\'s between henkan start point and
henkan end point into zenkaku\'s.  Valid only in  mode."
  (interactive "P")
  (if j-henkan-on
      (if j-henkan-active
	  nil
        (setq j-mode t)
	(or (string= j-prefix "") (error ""))
	(if (< (point) j-henkan-start-point)
	    (error "Henkan end point must be after henkan start point."))
	(save-excursion
	  (beginning-of-line)
	  (if (> (point) j-henkan-start-point)
	      (error "Henkan key may not contain a new line character.")))
	(move-marker j-henkan-end-point (point))
	(j-zenkaku-region j-henkan-start-point j-henkan-end-point)
	(or arg (j-kakutei))
	)
    (j-emulate-original-map arg)
    ))

(defun j-start-henkan (arg)
  (interactive "p")
  (if j-henkan-on
      (progn
	(setq j-mode t)
	(if j-henkan-active (j-next-candidate)
	  (or (string= j-prefix "") (error ""))
	  (if (< (point) j-henkan-start-point)
	      (error "Henkan end point must be after henkan start point."))
	  (save-excursion
	    (beginning-of-line)
	    (if (> (point) j-henkan-start-point)
		(error "Henkan key may not contain a new line character.")))
	  (move-marker j-henkan-end-point (point))
	  (setq j-henkan-key (buffer-substring j-henkan-start-point
						j-henkan-end-point)
		j-henkan-vector (vector)
		j-henkan-vector1 (vector)
		j-henkan-vector2 (vector)
		j-henkan-vector3 (vector)
		j-henkan-vector4 (vector))
	  (insert " ")
	  (j-henkan)
	  (delete-backward-char 1)
	  (if (and j-abbrev j-henkan-active) (use-local-map skk-map))))
    (j-self-insert arg)))

(defun j-set-henkan-point-subr ()
  (interactive)
  (if j-henkan-on (j-kakutei))
  (or (string= j-prefix "") (j-erase-prefix))
  (insert "")
  (or (string= j-prefix "")
      (progn
	(set-marker j-kana-start-point (point))
	(insert j-prefix)))
  (setq j-henkan-on t)
  (move-marker j-henkan-start-point (point)))

(defun j-change-marker ()
  (save-excursion
    (goto-char (- j-henkan-start-point 2))
    (if (looking-at "")
	(progn
	  (insert "")
	  (delete-char 1))
      (j-kakutei)
      (error "It seems that you have deleted ."))))

(defun j-change-marker-to-white ()
  (save-excursion
    (goto-char (- j-henkan-start-point 2))
    (if (looking-at "")
	(progn
	  (insert "")
	  (delete-char 1))
      (goto-char j-henkan-start-point)
      (insert "")
      (move-marker j-henkan-start-point (point))
      (message "It seems that you have deleted ."))))

;; functions for handling numeric arguments in henkan key.

(defun j-compute-numeric-henkan-key ()
  "return the modified version of j-henkan-key where any consecutive
substring consisting of numeric chars like \"12\" or \"\" is
replaced by a #."
  (let (p (list nil) (val nil) (key j-henkan-key))
    (save-excursion
      (set-buffer (get-buffer-create " *skk-work*"))
      (erase-buffer)
      (insert key)
      (goto-char (point-min))
      (while (re-search-forward "\\([0-9]\\|[-]\\)" nil t)
	(let (num)
	  (goto-char (match-beginning 1))
	  (setq p (point))
	  (while (looking-at "\\([0-9]\\|[-]\\)")
	    (if (looking-at "[-]")
		(progn
		  (insert (- (char-after (1+ (point))) 128))
		  (delete-char 1))
	      (forward-char 1)))
	  (setq num (buffer-substring p (point)))
	  (delete-region p (point))
	  (insert "#")
	  (setq list (append list (list num)))))
      (setq val (buffer-string)))
    (setq j-num-list list)
    val))

(defun j-num-exp (num type)
  "convert the integer string NUM into a string accoding to TYPE."
  (let ((fun (cdr (assq type skk-num-type-list))))
    (if fun (apply fun (list num))
      num)))

(defun j-id (num)
  "Identity function"
  num)

(defun j-zenkaku-num-str (num)
  "convert NUM to the string representing it in zenkaku numerals."
  (if (> (length num) 0)
      (concat (char-to-string 163)
	      (char-to-string (+ 128 (aref num 0)))
	      (j-zenkaku-num-str (substring num 1)))
    ""))

(defun j-zenkaku-num (num)
  (cdr (assq num
	     '((?0 . "") (?1 . "") (?2 . "") (?3 . "")
	       (?4 . "") (?5 . "") (?6 . "") (?7 . "")
	       (?8 . "") (?9 . "")))))

(defun j-kanji-num-str (num)
  "convert NUM to the string representing it in zenkaku kanji numerals."
  (if (> (length num) 0)
      (concat (j-kanji-num (aref num 0))
	      (j-kanji-num-str (substring num 1)))
    ""))

(defun j-kanji-num (num)
  (cdr (assq num
	     '((?0 . "") (?1 . "") (?2 . "") (?3 . "")
	       (?4 . "") (?5 . "") (?6 . "ϻ") (?7 . "")
	       (?8 . "Ȭ") (?9 . "")))))

(defun j-kanji-num-str2 (num)
  "convert NUM to the string representing it in kanji.  e.g, 1021 will be
converted to \"󽽰\""
  (let ((str (j-kanji-num-str2-subr num)))
    (if (string= "" str) "" str)))

(defun j-kanji-num-str2-subr (num &optional silent0)
  "If the optional argument SILENT0 is t, the initial 0 of NUM string
will be supressed."
  (let* ((len (length num))
	 (l (substring num 0 1))
	 (r (substring num 1))
	 (c (string-to-char l)))
    (if (string= "" r)
	(if (= c ?0) "" (j-kanji-num c))
      (or (string= l "0") (setq silent0 nil))
      (concat
       (if (or (= c ?0) (and (string= l "1") (not (= (mod len 4) 1))))
	   ""
	 (j-kanji-num c))
       (if (and (string= l "0") (or silent0 (not (= (mod len 4) 1))))
	   ""
	 (cond ((= (mod len 4) 2) "")
	       ((= (mod len 4) 3) "ɴ")
	       ((= (mod len 4) 0) "")
	       ((= len 5) (setq silent0 t) "")
	       ((= len 9) (setq silent0 t) "")
	       ((= len 13) (setq silent0 t) "")
	       ((= len 17) (setq silent0 t) "")
	       (t (error "Too big num!"))))
       (j-kanji-num-str2-subr r silent0)))))

(defun j-shogi-num-str (num)
  "convert NUM to the representation used in Shogi.  e.g., 34 will be
converted to \"\"."
  (concat
   (j-zenkaku-num (string-to-char num))
   (j-kanji-num (string-to-char (substring num 1 2)))))

(defun j-search (buffer)
  "search for KEY in BUFFER and create the vectors of candidates."
  (setq j-search-key (j-compute-numeric-henkan-key))
  (let ((key j-search-key)
	(list j-num-list) ;; used in j-compute-henkan-vectors
	(okuri-ari j-okurigana)
	(okurigana j-henkan-okurigana) (inhibit-quit t) l)
    (setq l
	  (save-excursion
	    (set-buffer buffer)
	    (goto-char (point-min))
	    (let ((w (concat key " /")) (kanji-flag nil)
		  (case-fold-search nil))
	      ;; kanji-flag must be set nil for efficiency.
	      (if (search-forward (concat "\n" w) nil t)
		  (j-compute-henkan-vectors)
		'([] [] [] [])))))
    (setq j-henkan-vector1 (nth 0 l)
	  j-henkan-vector2 (nth 1 l)
	  j-henkan-vector3 (nth 2 l)
	  j-henkan-vector4 (nth 3 l)
	  j-henkan-vector (j-vector-add j-henkan-vector3 j-henkan-vector1))))

(defun j-compute-henkan-vectors ()
  (let ((cont t) (stage 1)
	(v1 (vector)) (v2 (vector)) (v3 (vector)) (v4 (vector)))
    (while (and cont (not (eolp)))
      (let ((item (buffer-substring
		   (point)
		   (progn
		     (search-forward "/")
		     (1- (point))))))
	(cond ((not okuri-ari)
	       (if (and skk-process-okuri-early (= (aref item 0) ?\[))
		   (setq cont nil)
		 (setq v1 (vconcat v1 (vector item)))))
	      ((and (= (aref item 0) ?\[) (<= stage 2))
	       (if (string= item
			    (concat "\[" okurigana))
		   (progn
		     (setq stage 3)
		     (setq v2 (vconcat v2 (vector item))))
		 (setq stage 2)
		 (setq v2 (vconcat v2 (vector item)))))
	      ((and (= (aref item 0) ?\]) (= stage 3))
	       (setq stage 4)
	       (setq v4 (vconcat v4 (vector item))))
	      ((= stage 1)
	       (setq v1 (vconcat v1 (vector item))))
	      ((= stage 2)
	       (setq v2 (vconcat v2 (vector item))))
	      ((= stage 3)
	       (setq v3 (vconcat v3 (vector item))))
	      ((= stage 4)
	       (setq v4 (vconcat v4 (vector item)))))))
    (list v1 v2 v3 v4)))

(defun j-update-jisyo (word &optional purge)
  "update jisyo buffer so that WORD will become the first candidate
next time.  If PURGE is t, the word is  removed from the jisyo entry."
  (let ((vector1 [])
	(vector2 [])
	(vector3 [])
	(vector4 [])
	(len (length j-henkan-vector))
	okuri-ari
	(okurigana j-henkan-okurigana)
	key
	(inhibit-quit t)
	(new-entry t))
    ;; we don't have to auto-process okuri if j-num-list is non-nil
    (if (and skk-auto-okuri-process
	     (not j-num-list)
	     (or (> j-henkan-count len)
		 (and (< j-okuri-index-min j-henkan-count)
		      (<= j-henkan-count j-okuri-index-max))))
	(j-remove-common))
    (setq okuri-ari
	  (let ((char (aref j-henkan-key (1- (length j-henkan-key)))))
	    (and (> (aref j-henkan-key 0) 127)
		 (<= ?a char)
		 (<= char ?z))))
    (setq key (concat (if okuri-ari j-henkan-key j-search-key) " /"))
    (save-excursion
      (set-buffer
       (if okuri-ari " *skk-okuri-jisyo*" " *skk-jisyo*"))
      (goto-char (point-min))
      ;; delete old entry
      (if (search-forward (concat "\n" key) nil t)
	  (progn
	    (let ((l (j-compute-henkan-vectors)))
	      (setq vector1 (nth 0 l)
		    vector2 (nth 1 l)
		    vector3 (nth 2 l)
		    vector4 (nth 3 l)))
	    (setq new-entry nil)
	    (delete-region
	     (progn (beginning-of-line) (point))
	     (progn (end-of-line) (forward-char 1) (point)))))
      ;; insert new entry
      (if (string= (buffer-name (current-buffer)) " *skk-okuri-jisyo*")
	  (if new-entry
	      (let* (p)
		(goto-char (point-min))
		(forward-line 1)
		(setq p (point))
		;; modified by Kazuo Hirokawa, 1991.6.10
		(while (and (not (eobp))
			    (string< key
				     (progn
				       (search-forward " /")
				       (buffer-substring p (point)))))
		  (forward-line 1)
		  (setq p (point)))))
	(goto-char (point-min))
	(forward-line 1))
      (beginning-of-line)
      (insert key (if (and purge (or skk-process-okuri-early (not okuri-ari)))
			   ""
			   (concat word "/")))
      (let ((index 0) (len (length vector1)))
	(while (< index len)
	  (let ((w (aref vector1 index)))
	    (or (string= word w)
		(insert w "/"))
	    (setq index (1+ index)))))
      (if (and okuri-ari (not skk-process-okuri-early))
	  (progn
	    (let ((index 0) (len (length vector2)))
	      (while (< index len)
		(insert (aref vector2 index) "/")
		(setq index (1+ index))))
	    (if (equal vector3 [])
		(or purge (insert "\[" okurigana "/" word "/\]/"))
	      (let ((index 0) (len (length vector3)))
		(or purge (insert word "/"))
		(while (< index len)
		  (let ((w (aref vector3 index)))
		    (or (string= word w)
			(insert w "/"))
		    (setq index (1+ index))))))
	    (let ((index 0) (len (length vector4)))
	      (if (> len 0)
		  (let ((p (point)))
		    (save-excursion
		      (search-backward "/" nil nil 2)
		      (if (looking-at  "/\\[")
			  (progn
			    (delete-region (1+ (point)) p)
			    (setq index 1))))))
	      (while (< index len)
		(insert (aref vector4 index) "/")
		(setq index (1+ index))))))
      (if purge
	  (let ((uniq t) (index 0) (l2 (length vector2))
		(l4 (length vector4)))
	    (while (and uniq (< index l2))
	      (if (string= word (aref vector2 index))
		  (setq uniq nil)
		(setq index (1+ index))))
	    (while (and uniq (< index l4))
	      (if (string= word (aref vector4 index))
		  (setq uniq nil)
		(setq index (1+ index))))
	    (if uniq
		;; the WORD is not in vector3 nor in vector4, so purge it
		;; also from vector1 (if any)
		(save-excursion
		  (let ((q (point)))
		    (goto-char (point-min))
		    ;; added following line at the suggestion of
		    ;; Wataru Matsui, 1992.1.22
		    (end-of-line)
		    (if (search-forward word q t)
			(delete-region (1+ (point))
				       (progn
					 (search-backward word)
					 (point)))))))))
      (insert "\n")
      (forward-line -1)
      (search-forward " /")
      (if (eolp)
	  ;; the key has no words for it so delete everything
	  (delete-region (save-excursion (beginning-of-line) (point))
			 (1+ (point)))))))

(defun j-remove-common ()
  "remove trailing common kana ending parts both from midasi and word.
For instannce, (äƤ äƤ) -> key := \"t /\",
okurigana := \"ä\", word := \"\""
  (let* ((midasi j-henkan-key)
	 (midasi-len (length midasi))
	 (word-len (length word))
	 char pos2)
    (if (and (>= midasi-len 4) (>= word-len 4))
	(progn
	  ;; check if both midasi and word end with the same ascii char.
	  (if (and (string= (substring midasi -1) (substring word -1))
		   (< (string-to-char (substring midasi -1)) 128))
	      ;; if so chop off the char from midasi and word
	      (setq midasi (substring midasi 0 -1)
		    midasi-len (1- midasi-len)
		    word (substring word 0 -1)
		    word-len (1- word-len)))
	  (let ((midasi-tail (substring midasi (- midasi-len 2) midasi-len))
		(word-tail (substring word (- word-len 2) word-len)))
	    (if (and (string= midasi-tail word-tail)
		     (or
		      (and
		       (string<= "" midasi-tail) (string<= midasi-tail ""))
		      (string= "" midasi-tail)
		      (string= "" midasi-tail)
		      (string= "" midasi-tail)
		      (string= "" midasi-tail)))
		(let ((pos (- word-len 2))
		      (cont t)
		      new-word new-j-henkan-key)
		  (while (and cont (> pos 0))
		    (setq char (substring word (- pos 2) pos))
		    (if (and (string<= "" char) (string<= char ""))
			;; char is the right-most Kanji
			(setq cont nil)
		      (setq pos (- pos 2))))
		  (setq pos2 (- midasi-len (- word-len pos)))
		  ;; check if midasi and word has the same tail of length
		  ;; (- word-len pos)
		  (if (string= (substring midasi pos2 midasi-len)
			       (substring word pos word-len))
		      (let ((okuri-first (substring word pos (+ pos 2))))
			(setq okurigana
			      (if (and (or (string= okuri-first "")
					   (string= okuri-first ""))
				       (<= (+ pos 4) word-len))
				  ;; in this case okuriga consits of two
				  ;; characters, e.g., ֻĤä
				  (substring word pos (+ pos 4))
				okuri-first))
			(setq new-word (substring word 0 pos))
			(setq new-j-henkan-key
			      (concat
			       (substring midasi 0 pos2)
			       (if (string= okuri-first "")
				   "n"
				 (if (string= okuri-first "")
				     (aref skk-kana-rom-vector
					   (- (string-to-char
					       (substring okurigana 3 4))
					      161))
				   (aref skk-kana-rom-vector
					 (- (string-to-char
					     (substring okurigana 1 2))
					    161))))))
			(if (> j-henkan-count len)
			    ;; ask if register as okuri-ari word.
			    (let (;; allow keyboard quit
				  (inhibit-quit nil))
			      (if (y-or-n-p
				   (format
				    "Shall I register this as okuri-ari entry: %s /%s/ ? "
				    new-j-henkan-key new-word))
				  (setq word new-word
					j-henkan-key new-j-henkan-key))
			      (message ""))
			  (setq word new-word
				j-henkan-key new-j-henkan-key)))))))))))

(defun string<= (str1 str2)
  (or (string< str1 str2) (string= str1 str2)))

(defun j-henkan (&optional okuri-processed)
  "The main kana-to-kanji conversion routine."
  (save-excursion
    (if j-henkan-active
	(if (string= j-henkan-key "")
	    (j-kakutei)
	  (let (new-word
		(j-okuri-processed nil) (block nil) (found nil) (kakutei nil)
		(quit nil)
		(okuri-ari
		 (let ((char (aref j-henkan-key (1- (length j-henkan-key)))))
		   (and (> (aref j-henkan-key 0) 127)
			(<= ?a char)
			(<= char ?z)))))
	    (if (= j-henkan-count 0)
		(progn
		  (if (= (length j-henkan-vector) 0)
		      (j-search
		       (if okuri-ari " *skk-okuri-jisyo*" " *skk-jisyo*")))
		  (if (> (length j-henkan-vector) 0)
		      (setq found t
			    j-henkan-count 1
			    new-word (aref j-henkan-vector 0))))
	      (if (< j-henkan-count 4)
		  (if (< j-henkan-count (length j-henkan-vector))
		      (setq new-word (aref j-henkan-vector j-henkan-count)
			    found t
			    j-henkan-count (1+ j-henkan-count)))
		;; show candidates in minibuffer
		(if (or (> j-henkan-count 4)
			(< j-henkan-count (length j-henkan-vector)))
		    (j-henkan-show-candidates))))
	    (or quit
		(if found
		    (progn
		      (delete-region j-henkan-start-point j-henkan-end-point)
		      (goto-char j-henkan-start-point)
		      (let ((l (length new-word)))
			(if (and (> l 2)
				 (= (aref new-word 0) ?\()
				 (< (aref new-word 1) 128)
				 (= (aref new-word (1- l)) ?\)))
			    (condition-case conditions
				(insert
				 (eval (car (read-from-string new-word))))
			      (error
			       (insert new-word)))
			  (insert (j-numeric-convert new-word))))
		      (move-marker j-henkan-end-point (point))
		      (if kakutei
			  (let ((save-prefix j-prefix))
			    ;; save and restore j-prefix after j-kakutei
			    (j-kakutei new-word)
			    (setq j-prefix save-prefix))))
		  ;; try okurigana process if necessary
		  (if (and skk-auto-okuri-process
			   (not skk-process-okuri-early)
			   (not okuri-ari)
			   (not okuri-processed)
			   ;; suggested by Kazuo Hirokawa, 1991.10.30.
			   (> (length j-henkan-key) 1))
		      (progn
			(setq j-okuri-index-min (length j-henkan-vector))
			(setq j-henkan-vector
			      (j-vector-add j-henkan-vector (j-okuri-search)))
			(setq j-okuri-index-max (length j-henkan-vector))
			(j-henkan t))
		    ;; call skk server if necessary
		    (if (and skk-server-host (not j-server-called))
			(progn
			  (setq j-local-length (length j-henkan-vector))
			  (setq j-henkan-vector
				(j-vector-add j-henkan-vector
					      (j-search-server)))
			  (setq j-server-called t)
			  (j-henkan))
		      ;; use recursive minibuffer
		      (let ((save-start (copy-marker j-henkan-start-point))
			    (save-end (copy-marker j-henkan-end-point))
			    new-word-entered
			    ;; inhibit isearch message while doing henkan
			    (skk-isearch-message nil))
			(setq new-word-entered (j-henkan-in-minibuff))
			(move-marker j-henkan-start-point save-start)
			(move-marker j-henkan-end-point save-end)
			(if new-word-entered
			    (progn
			      (delete-region j-henkan-start-point
					     j-henkan-end-point)
			      (goto-char j-henkan-start-point)
			      (insert (j-numeric-convert new-word))
			      (move-marker j-henkan-end-point (point))
			      (if kakutei
				  (let ((save-prefix j-prefix))
				    ;; save and restore j-prefix after
				    ;; j-kakutei
				    (j-kakutei new-word)
				    (setq j-prefix save-prefix))))))))))))
      (j-change-marker)
      (setq j-henkan-active t)
      (j-henkan))))

(defun j-numeric-convert (word)
  "If j-num-list is non-nil, convert WORD by substing appropriate strings
for the occurrences of # in the WORD."
  (let ((list j-num-list))
    (if list
	(save-excursion
	 (set-buffer (get-buffer-create " *skk-work*"))
	 (erase-buffer)
	 (insert word)
	 (goto-char (point-min))
	 (while (and list (search-forward "#" nil t))
	   (setq num (car list))
	   (let ((char (following-char)))
	     (delete-region (1- (point)) (1+ (point)))
	     (insert (j-num-exp num char))
	     (setq list (cdr list))))
	 (buffer-string))
      word)))

(defun j-henkan-show-candidates ()
  "Show candidates in the minibuffer."
  (let ((cont t) (count 1) (l (length j-henkan-vector)))
    (setq block t found t)
    (delete-region j-henkan-start-point j-henkan-end-point)
    (let ((h-count j-henkan-count) (unread nil)
	  ;; Modified by Masakazu Takahashi, 1991.11.18
	  ;; Added following to enable interruption while executing
	  ;; skk-isearch.
	  (inhibit-quit nil))
      (save-excursion
	(j-call-server)
	(if (= h-count l)
	    (setq cont nil found nil))
	(while cont
	  (j-call-server)
	  (if (= h-count l)
	      (setq cont nil)
	    (message "%s%s%s%s%s%s%s%s"
		     (concat "A:" (j-numeric-convert
				   (aref j-henkan-vector h-count)))
		     (if (< (1+ h-count) l)
			 (concat "  S:"
				 (j-numeric-convert
				  (aref j-henkan-vector (1+ h-count))))
		       "")
		     (if (< (+ h-count 2) l)
			 (concat "  D:"
				 (j-numeric-convert
				  (aref j-henkan-vector (+ h-count 2))))
		       "")
		     (if (< (+ h-count 3) l)
			 (concat "  F:"
				 (j-numeric-convert
				  (aref j-henkan-vector (+ h-count 3))))
		       "")
		     (if (< (+ h-count 4) l)
			 (concat "  J:"
				 (j-numeric-convert
				  (aref j-henkan-vector (+ h-count 4))))
		       "")
		     (if (< (+ h-count 5) l)
			 (concat "  K:"
				 (j-numeric-convert
				  (aref j-henkan-vector (+ h-count 5))))
		       "")
		     (if (< (+ h-count 6) l)
			 (concat "  L:"
				 (j-numeric-convert
				  (aref j-henkan-vector (+ h-count 6))))
		       "")
		     (format "  [Ĥ %d%s]"
			     (1+ (- l (+ h-count
					 (setq count
					       (min 8 (1+ (- l h-count)))))))
			     (if (or j-server-called (not skk-server-host))
				 ""
			       "+")))
	    (setq h-count (+ h-count (1- count)))
	    (condition-case conditions
		(let* ((char (read-char))
		       (char-val-pair
			(assq char
			      '((?a . 0) (?s . 1) (?d . 2) (?f . 3) (?j . 4)
				(?k . 5) (?l . 6)))))
		  (cond (char-val-pair
			 (let ((c (cdr char-val-pair)))
			   (if (> (+ c 2) count)
			       (progn
				 (message "`%c' is not valid here!" char)
				 (sit-for 1)
				 (message "")
				 (setq h-count
				       (- h-count (1- count))))
			     (setq h-count
				   (-
				    (+ c
				       (* 7
					  (/ (+ h-count 2) 7)))
				    2))
			     (setq new-word
				   (aref j-henkan-vector (1- h-count)))
			     (setq kakutei t cont nil))))
			((= char ?x)
			 (let* ((a (/ (- h-count 5) 7))
				(b (- h-count (+ 4 (* 7 a)))))
			   (if (= a 0)
			       (setq h-count (- h-count (1- b))
				     cont nil unread t)
			     (setq h-count (- (- h-count b) 7)))))
			((= char 32)
			 (if (= h-count l)
			     (setq cont nil found nil)))
			(t
			 (message "`%c' is not valid here!" char)
			 (sit-for 1)
			 (message "")
			 (setq h-count (- h-count (1- count))))))
	      (quit
	       (setq h-count 1 cont nil unread t))))))
      (if unread (setq unread-command-char ?x quit t))
      (setq j-henkan-count h-count))))

(defun j-call-server ()
  "This function is called from j-henkan-show-candidates."
  (if (and skk-server-host
	   (< (- l h-count) 7)
	   (not j-server-called))
      (progn
	(setq j-local-length (length j-henkan-vector))
	(setq j-henkan-vector
	      (j-vector-add j-henkan-vector (j-search-server)))
	(setq j-server-called t)
	(setq l (length j-henkan-vector)))))

(defun j-henkan-in-minibuff ()
  (setq new-word
	(let (r-word (h-count j-henkan-count))
	  (condition-case conditions
	      (setq r-word
		    (let ((unread-command-char ?\C-j)
			  (skk-mode t))
		      (read-string
		       (concat
			(if j-num-list
			    j-search-key
			  (or j-henkan-key2 j-henkan-key))
			" "))))
	    (quit
	     (setq r-word "")))
	  (setq j-henkan-count h-count)
	  r-word))
  (if (string= new-word "")
      (cond ((> j-henkan-count 4)
	     (let* ((a (/ (- j-henkan-count 5) 7))
		    (b (- j-henkan-count (+ 4 (* 7 a)))))
	       (setq j-henkan-count (- j-henkan-count b))
	       (j-henkan)))
	    ((= j-henkan-count 0)
	     (setq j-henkan-active nil
		   j-henkan-key2 nil
		   ;; modified at the suggestion of Kazuo Hirokawa, 1992.1.20
		   j-server-called nil
		   )
	     (let ((j-henkan-start-point save-start))
	       (j-change-marker-to-white)))
	    (t nil))
    (setq kakutei t)
    (setq j-count-touroku (1+ j-count-touroku))
    (setq j-henkan-count (1+ j-henkan-count)))
  (not (string= new-word "")))

(defun j-next-candidate ()
  "Show next henkan candidate."
  (if j-henkan-active
      (progn
	(insert " ")
	(j-henkan)
	(delete-backward-char 1))))

(defun j-previous-candidate ()
  "Show previous henkan candidate."
  (interactive)
  (if (and j-henkan-active (not (string= j-henkan-key "")))
      (progn
	(insert " ")
	(let ((word (concat j-henkan-key " ")) new-word
	      (found nil) p q)
	  (save-excursion
	    (cond ((= j-henkan-count 1)
		   (setq j-henkan-active nil
			 j-henkan-count 0
			 j-okuri-index-min 0
			 j-okuri-index-max 0
			 j-henkan-vector []
			 j-okurigana nil
			 j-henkan-okurigana nil
			 j-henkan-key2 nil
			 j-server-called nil)
		   (setq q (point-marker))
		   (delete-region j-henkan-start-point j-henkan-end-point)
		   (goto-char j-henkan-end-point)
		   (insert j-henkan-key)
		   (if (> (length j-prefix) 0)
		       (set-marker j-kana-start-point (point)))
		   (setq p (point-marker))
		   (goto-char j-henkan-start-point)
		   (if (> (following-char) 127)
		       (progn
			 ;; roman prefix for okurigana is removed
			 (while (and (< (point) p) (> (following-char) 127))
			   (forward-char 1))
			 (if (and (<= ?a (following-char))
				  (<= (following-char) ?z))
			     (delete-region (point) p)))
		     (goto-char p))
		   (j-change-marker-to-white))
		  ((<= j-henkan-count 7)
		   (setq word (aref j-henkan-vector (- j-henkan-count 2)))
		   (delete-region j-henkan-start-point j-henkan-end-point)
		   (goto-char j-henkan-end-point)
		   (insert (j-numeric-convert word))
		   (move-marker j-henkan-end-point (point))
		   (setq j-henkan-count (1- j-henkan-count)))
		  ((< 7 j-henkan-count)
		   (setq j-henkan-count (- j-henkan-count 8))
		   (setq unread-command-char 32) ;; 32 is space char
		   )))
	  (if (and (> (point) 1) (= (char-after (1- (point))) 32))
	      (delete-backward-char 1))
	  (if (= j-henkan-count 0)
	      (progn
		(if skk-auto-okuri-process (goto-char q) (goto-char p))
		(if j-abbrev (use-local-map skk-abbrev-map))))))
    (j-kana-input)))

(defun j-keyboard-quit ()
  "If j-henkan-active, abort showing henkan candidates and return to the
j-henkan-on status; if j-henkan-on, delete region between current point and
j-henkan-start-point; otherwise same as keyboard-quit"
  (interactive)
  (if (string= j-henkan-key "")
      (if (> (minibuffer-depth) 0) (abort-recursive-edit) (keyboard-quit)))
  (if j-henkan-active
      (progn
	(setq j-henkan-count 1)
	(j-previous-candidate))
    (if j-henkan-on
	(progn
	  (j-kakutei)
	  (if (> (point) j-henkan-start-point)
	      (delete-region (point) j-henkan-start-point)))
	(if (> (minibuffer-depth) 0) (abort-recursive-edit) (keyboard-quit)))))

(defun j-okuri-search ()
  (let* ((henkan-key j-henkan-key)
	 (key (substring henkan-key 0 2))
	 (len (length henkan-key))
	 (key1 (concat "\n" key))
	 key2 len2
	 key3 len3 okuri3
	 (kanji-flag nil)
	 (inhibit-quit t)
	 p q r s
	 (v (vector))
	 )
    (save-excursion
      (set-buffer " *skk-okuri-jisyo*")
      (goto-char (point-min))
      (while (search-forward key1 nil t)
	(setq p (point))
	(search-forward " ")
	(setq key2 (concat key (buffer-substring p (- (point) 2)))
	      len2 (length key2))
	(if (and
	     (<= len2 len)
	     (string= key2 (substring henkan-key 0 (length key2))))
	    (let ((cont t))
	      (save-excursion
		(end-of-line)
		(setq q (point)))
	      (while (and cont
			  (search-forward "/[" q t))
		(setq r (point))
		(search-forward "/")
		(setq okuri3 (buffer-substring r (1- (point)))
		      key3 (concat key2 okuri3)
		      len3 (length key3))
		(if (and
		     (<= len3 len)
		     (string= key3 (substring henkan-key 0 (length key3))))
		    ;; finally found a candidate!
		    (let ((okuri
			   (concat okuri3
				   (substring henkan-key len3 len))))
		      (while (not (looking-at "\]"))
			(setq s (point))
			(search-forward "/" nil t)
			(setq v (vconcat v
					 (vector
					  (concat
					   (buffer-substring s (1- (point)))
					   okuri)))))
		      ;; it is not necessary to seach for "\[" on this line
		      ;; any more
		      (setq cont nil)))))))
      v)))

(defun j-kakutei (&optional word)
  "This function does what it says. If WORD is given, it is used as the new
word for jisyo update."
  (interactive)
  (let ((inhibit-quit t))
    (if skk-mode
	(progn
	  (j-change-mode-line (if j-katakana skk-katakana-mode-string
				skk-hirakana-mode-string))
	  (use-local-map skk-map))
      ;; if skk-mode has not been called in this buffer, call it now.
      (skk-mode 1))
    (setq j-mode t
	  j-prefix "")
    (if j-henkan-on
	(progn
	  (if (and j-henkan-active (> j-henkan-count 0))
	      (progn
		(j-update-jisyo
		 (or word (aref j-henkan-vector (1- j-henkan-count))))
		(setq j-count-kakutei (1+ j-count-kakutei))))
	  (save-excursion
	    (goto-char (- j-henkan-start-point 2))
	    (if j-henkan-active
		(if (looking-at "")
		    (delete-char 1)
		  (message "It seems that you have deleted ."))
	      (if (looking-at "")
		  (delete-char 1)
		(message "It seems that you have deleted .")))
	    (if j-okurigana
		(progn
		  (goto-char j-okurigana-start-point)
		  (if (looking-at "*") (delete-char 1))
		  (setq j-okurigana nil))))
	  (if self-insert-after-hook
	      (funcall self-insert-after-hook j-henkan-start-point (point)))
	  (if (or overwrite-mode (boundp 'vip-ovwrt-limit))
	      (j-del-char-with-pad
	       (j-ovwrt-len
		(- (point) j-henkan-start-point))))
	  (setq j-abbrev nil)))
    (setq j-henkan-active nil
	  j-henkan-on nil
	  j-okurigana nil
	  j-okuri-char ""
	  j-henkan-key nil
	  j-henkan-key2 nil
	  j-henkan-okurigana nil
	  j-henkan-vector []
	  j-henkan-count 0
	  j-okuri-index-min 0
	  j-okuri-index-max 0
	  j-server-called nil)
    ;; modified by Masakazu Takahashi, 1989.8.9
    ;; modified at the suggestion of Kazuo Hirokawa, 1992.3.9.
    (if auto-fill-hook (j-do-auto-fill))))

;; completion

(defun j-completion (first)
  (let (c-word (cont t))
    (if first
	(setq j-completion-word
	      (buffer-substring j-henkan-start-point (point))))
    (let ((inhibit-quit t)
	  ;; completion word is assumed to be a JIS Kanji word if it begins
	  ;; with a JIS character
	  (jis-completion-word (> (aref j-completion-word 0) 127)))
      (save-excursion
	(set-buffer " *skk-jisyo*")
	(if first (goto-char (point-min)))
	(let ((kanji-flag nil))
	  ;; kanji-flag must be set nil for efficiency.
	  (while (and cont
		      (or (and (looking-at (regexp-quote j-completion-word))
			       (search-forward j-completion-word))
			  (search-forward (concat "\n" j-completion-word)
					  (point-max) t)))
	    (if (not (looking-at " /"))
		(progn
		  (setq p (point))
		  (search-forward " /")
		  (setq c-word (concat j-completion-word
				       (buffer-substring p (- (point) 2))))
		  ;; if the c-word begins with a JIS character, we must check
		  ;; if it ends with an ascii character.  if it indeed ends
		  ;; with an ascii character we should ignore it.
		  (if jis-completion-word
		      (if (> (aref c-word (1- (length c-word))) 127)
			  (setq cont nil)
			(setq c-word nil))
		    (setq cont nil))))))))
    (if cont
	(progn
	  (message "No %scompletions for \"%s\""
		   (if first "" "more ") j-completion-word)
	  (ding)))
    (if c-word
	(progn
	  (delete-region j-henkan-start-point (point))
	  (insert c-word)))))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-try-completion (arg)
  (interactive "P")
  (if (and j-henkan-on (not j-henkan-active))
      (progn
	(setq this-command 'j-completion)
	(j-completion t))
    (j-emulate-original-map arg)))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-abbrev-period (arg)
  (interactive "P")
  (if (eq last-command 'j-completion)
      (progn
	(setq this-command 'j-completion)
	(j-completion nil))
    (j-emulate-original-map arg)))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-abbrev-comma (arg)
  (interactive "P")
  (if (eq last-command 'j-completion)
      (j-previous-completion)
    (j-emulate-original-map arg)))

;; modified by Kazuo Hirokawa, 1991.11.18.

(defun j-emulate-original-map (arg)
  (let* ((prefix-arg arg)
	 (command-keys (this-command-keys))
	 (n (1- (length command-keys))))
    ;; command may be invoked by M-x command-name, or key strokes with prefix
    (while (>= n 0)
      (if (not (eq this-command (key-binding (substring command-keys n))))
	  (setq n (1- n))
	(command-execute
	 (j-get-editor-command
	  j-emacs-local-map global-map (substring command-keys n)))
	(setq n -1)))))

(defun j-previous-completion ()
  (let ((inhibit-quit t)
	;; completion word is assumed to be a JIS Kanji word if it begins
	;; with a JIS character
	(jis-completion-word (> (aref j-completion-word 0) 127))
	(c-word nil))
    (save-excursion
      (set-buffer " *skk-jisyo*")
      (beginning-of-line)
      (let ((kanji-flag nil) (cont t))
	;; kanji-flag must be set nil for efficiency.
	(while (and cont
		    (not (bobp))
		    (or (search-backward (concat "\n" j-completion-word)
					 (point-min) t)
			(progn
			  (goto-char (point-min))
			  (looking-at (regexp-quote j-completion-word)))))
	  (search-forward j-completion-word)
	  (if (not (looking-at " /"))
	      (progn
		(setq p (point))
		(search-forward " /")
		(setq c-word (concat j-completion-word
				     (buffer-substring p (- (point) 2))))
		;; if the c-word begins with a JIS character, we must check
		;; if it ends with an ascii character.  if it indeed ends
		;; with an ascii character we should ignore it.
		(if jis-completion-word
		    (if (> (aref c-word (1- (length c-word))) 127)
			(setq cont nil)
		      (setq c-word nil)
		      (beginning-of-line))
		  (setq cont nil)))
	    (beginning-of-line)))))
    (delete-region j-henkan-start-point (point))
    (if c-word
	(insert c-word)
      (insert j-completion-word)
      (message "No more previous completions.")
      (ding))
    (setq this-command 'j-completion)))

(defun j-purge-from-jisyo ()
  "Purge wrong words from jisyo buffer."
  (interactive)
  (if (and j-henkan-active (not (string= j-henkan-key ""))
	   (yes-or-no-p "Really purge? "))
      (progn
	(j-update-jisyo (aref j-henkan-vector (1- j-henkan-count)) t)
	(delete-region j-henkan-start-point j-henkan-end-point)
	(j-change-marker-to-white)
	(setq j-henkan-active nil)
	(j-kakutei))))

;; skk server related functions

(defun j-open-server ()
  "Establish connection with skk server."
  (or (get-process "skkservd")
      (let ((succ nil))
	(get-buffer-create " *skkserv*")
	(if (j-open-server-subr)
	    (setq succ t)
	  (progn
	    (let ((count 7))
	      (while (> count 0)
		(message "SKK SERVER is not active, I will activate it%s"
			 (make-string count ?.))
		(if (string= skk-server-host (system-name))
		    ;; server host is local machine
		    (call-process skk-serv nil 0 nil)
		  (call-process
		   j-remote-shell-program nil 0 nil skk-server-host skk-serv))
		(sit-for 3)
		(if (j-open-server-subr)
		    (setq succ t
			  count 0)
		  (setq count (1- count)))))
	    (if succ
		(progn
		  (message "SKK SERVER is active now.")
		  (sit-for 1)))))
	(if succ
	    (save-excursion
	      ;; If your Nemacs is an old one, it may not have the function
	      ;; set-process-kanji-code.
	      (if (fboundp 'set-process-kanji-code)
		  (set-process-kanji-code (get-process "skkservd") 0))
	      (set-buffer " *skkserv*")
	      (setq kanji-fileio-code 0) ;; for old Nemacs
	      )
	  (message "Could not activate SKK SERVER.")
	  (sit-for 1)
	  ;; reset skk-server-host so as not to use server in this session
	  (setq skk-server-host nil)
	  (ding)))))

(defun j-open-server-subr ()
  (condition-case nil
      (open-network-stream
       "skkservd" " *skkserv*" skk-server-host (or skk-portnum "skkserv"))
    (error
     nil)))

(defun j-search-server ()
  "search for KEY through skkserver and create the vector of candidates."
  (let ((key j-henkan-key) (okuri-ari j-okurigana)
	(okurigana j-henkan-okurigana) (v (vector)))
    (save-excursion
      (set-buffer " *skkserv*")
      (erase-buffer)
      (or (process-status "skkservd") (j-open-server))
      (if (process-status "skkservd")
	  (let ((cont t) (count 0))
	    (process-send-string "skkservd" (concat "1" key " "))
	    (while (and cont (memq (process-status "skkservd") '(open run)))
	      (accept-process-output)
	      (setq count (1+ count))
	      (if (> (buffer-size) 0)
		  (progn
		    (if (= ?1 (char-after 1))
			;; found key successfully, so check if a whole line
			;; is received.
			(if (= ?\n (char-after (1- (point-max))))
			    (setq cont nil))
		      ;; not found or error, so exit
		      (setq cont nil)))))
	    (goto-char (point-min))
	    (if skk-report-server-response
		(message "waited for server response %d times." count))
	    (if (looking-at "1")
		(progn
		  (forward-char 2)
		  (let ((l (j-compute-henkan-vectors)))
		    (j-vector-add (nth 0 l) (nth 2 l))))
	      ;; not found or error
	      []))
	[]))))

(defun j-vector-add (u v)
  "Concatenate U and elements of V not in U."
    (let ((w u) (lv (length v)) (i 0))
      (while (< i lv)
	(let ((e (aref v i)))
	  (or (j-vector-mem e u)
	      (setq w (vconcat w (vector e))))
	  (setq i (1+ i))))
      w))

(defun j-vector-mem (e v)
  "Test if the string E is a member of the vector V."
  (let ((l (length v)) (i 0) (cont t))
    (while (and cont (< i l))
      (if (string= e (aref v i)) (setq cont nil) (setq i (1+ i))))
    (not cont)))

(defun j-create-file (file &optional message)
  "Create an empty file named FILE if the file FILE does not exist already."
  (let ((file (expand-file-name file)))
    (or (file-exists-p file)
	(progn
	  (write-region (point-min) (point-min) file nil 0)
	  (if message
	      (progn
		(message message)
		(sit-for 3)))))))

;; patch for old Nemacs (v2)
;; suggested by Masakazu Takahashi, 1992.4.12.
;; this modification is necessary since the variable
;; self-insert-after-hook is unbound in Nemacs v2 

(or (boundp 'self-insert-after-hook)
    (setq self-insert-after-hook nil))

;; end of patch for old Nemacs (v2)
