;;; rail-user-agent.el --- convert codename into Japanese -*- coding: iso-2022-7bit-ss2; -*-

;; Copyright (C) 1999 by Free Software Foundation, Inc.

;; Author: SHIMADA Mitsunobu <simm-emacs@fan.gr.jp>
;; Keywords: FLIM, SEMI, Rail

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Current version supports only MULE, FLIM, and SEMI.

;;; Code:

(defun rail-replace-character (string src dst)
  "Replace character src -> dst"
  (let ((len (length string))
        (i 0))
    (while (< i len)
      (if (eq src (aref string i))
          (aset string i dst))
      (setq i (1+ i)))
    string))

(defun rail-replace-mime-version-header ()
  "Replace MIME-Version: header"
  (catch 'replace-exit
    (save-excursion
      (let (beg end string)
        (goto-char (point-min))
        (or (search-forward "MIME-Version: " nil t)
            (throw 'replace-exit nil))
        (setq beg (point))
        (end-of-line)
        (setq end (point))
        (save-restriction
          (narrow-to-region beg end)
          (goto-char (point-min))
          (or (search-forward "\"" nil t)
              (throw 'replace-exit nil))
          (setq beg (point))
          (or (search-forward "\"" nil t)
              (throw 'replace-exit nil))
          (forward-char -1)
          (setq end (point))
          (setq string (buffer-substring beg end))
          (rail-replace-character string ?.AN~ ?.DN~)
          (rail-replace-character string ?.ANr ?.DNr)
          (delete-region beg end)
          (insert rail-ja-semi-codename))))))

(defun rail-add-hook-replace-mime-version-header ()
  (let ((list (memq 'mime-edit-translate-body mime-edit-translate-buffer-hook)))
    (setcdr list (cons 'rail-replace-mime-version-header (cdr list)))))

(if (featurep 'mime-edit)
    (rail-add-hook-replace-mime-version-header)
  (add-hook 'mime-edit-load-hook
	    '(lambda () (rail-add-hook-replace-mime-version-header))))

(if (featurep 'semi-def)
    (let ((pgpgpg-flag nil))
      (require 'mime-def)
      (load "rail-table-flim")
      (load "rail-table-semi")
      (defvar rail-additional-flim-codename-alist nil
        "Additional codename for FLIM")
      (defvar rail-additional-semi-codename-alist nil
        "Additional codename for SEMI")
      (defvar rail-en-flim-codename
	(mime-product-code-name mime-library-product)
        "ISO-8859-4 flim codename .")
      (and (featurep 'xemacs)
           (rail-replace-character rail-en-flim-codename ?.AN~ ?.DN~)
           (rail-replace-character rail-en-flim-codename ?.ANr ?.DNr))
      (defvar rail-ja-flim-codename
	(cdr (assoc rail-en-flim-codename
                    (append rail-additional-flim-codename-alist rail-flim-codename-alist)))
        "Japanese flim codename.")
      (and (eq 0 (length rail-ja-flim-codename))
           (setq rail-ja-flim-codename rail-en-flim-codename))
      (defvar rail-en-semi-codename
	(mime-product-code-name mime-user-interface-product)
        "ISO-8859-4 semi codename .")
      (and (featurep 'xemacs)
           (rail-replace-character rail-en-semi-codename ?.AN~ ?.DN~)
           (rail-replace-character rail-en-semi-codename ?.ANr ?.DNr))
      (defvar rail-ja-semi-codename
	(cdr (assoc rail-en-semi-codename
                    (append rail-additional-semi-codename-alist rail-semi-codename-alist)))
        "Japanese semi codename.")
      (and (eq 0 (length rail-ja-semi-codename))
           (setq rail-ja-semi-codename rail-en-semi-codename))
      (and (boundp 'pgp-version)
           (let ((pgp-document (get 'pgp-version 'variable-documentation)))
	     (setq pgpgpg-flag
		   (or (and (stringp pgp-document)
			    (string-match "gpg" pgp-document))
		       (and (listp pgp-document)
			    (stringp (car pgp-document))
			    (string-match "semi/semi-def.el" (car pgp-document)))))))
      (defvar rail-translate-user-agent t
        "If non-nil, translate User-Agent: field into Japanese.")
      (and rail-translate-user-agent
           (setq mime-edit-user-agent-value
                 (concat
                  (mime-product-name mime-user-interface-product)
                  "/"
                  (mapconcat #'number-to-string
                             (mime-product-version mime-user-interface-product) ".")
                  " ("
                  rail-ja-semi-codename
                  (if pgpgpg-flag ") PGPGPG " ") ")
                  (mime-product-name mime-library-product)
                  "/"
                  (mapconcat #'number-to-string
                             (mime-product-version mime-library-product) ".")
                  " ("
                  rail-ja-flim-codename
                  ") "
                  (if (featurep 'xemacs)
                      (concat (cond ((featurep 'utf-2000)
                                     (concat "UTF-2000-MULE/" rail-ja-utf-2000-version))
                                    ((featurep 'mule) "MULE"))
                              " XEmacs"
                              (if (string-match "^[0-9]+\\(\\.[0-9]+\\)" emacs-version)
                                  (concat "/"
                                          (substring emacs-version 0 (match-end 0))
                                          (if (and (boundp 'xemacs-betaname)
                                                   ;; It does not exist in XEmacs
                                                   ;; versions prior to 20.3.
                                                   xemacs-betaname)
                                              (concat " " xemacs-betaname)
                                            "")
                                          " (" rail-ja-xemacs-codename ") ("
                                          system-configuration ")")
                                " (" emacs-version ")"))
                    (let ((ver (if (string-match "\\.[0-9]+$" emacs-version)
                                   (substring emacs-version 0 (match-beginning 0))
                                 emacs-version)))
                      (if (featurep 'mule)
                          (if (boundp 'enable-multibyte-characters)
                              (concat "Emacs/" ver
                                      " (" system-configuration ")"
                                      (if enable-multibyte-characters
                                     (concat " MULE/" rail-ja-mule-version)
                                     " (with unibyte mode)")
                                      (and (featurep 'meadow)
                                           (string-match "^Meadow-" rail-ja-meadow-version)
                                           (concat " Meadow/"
                                                   (substring rail-ja-meadow-version (match-end 0)))))
                            (concat "MULE/" rail-ja-mule-version
                                    " (based on Emacs " ver ")"))
                        (concat "Emacs/" ver " (" system-configuration ")")))))))))

(provide 'rail-user-agent)

;;; rail-user-agent.el ends here
