/* simple wrapper program for STARTTLS

   Copyright (C) 1999, 2000 Free Software Foundation, Inc.

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 1999-11-19
   Keywords: TLS, OpenSSL

   This file is not part of any package.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>

#include <alloca.h>
#include <arpa/inet.h>
#include <getopt.h>
#include <netinet/in.h>
#include <poll.h>
#include <stdbool.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#include "getaddrinfo.h"
#include "gettext.h"
#include "inet_ntop.h"
#include "minmax.h"
#include "size_max.h"
#include "snprintf.h"
#include "strdup.h"
#include "vasnprintf.h"
#include "xsize.h"

extern void tls_negotiate (int, const char *, const char *);
extern int tls_write(int, const char *, int);
extern int tls_read(int, char *, int);
extern int tls_pending();

static char *opt_cert_file = NULL, *opt_key_file = NULL;
static bool opt_assuan = false;
static int tls_fd = -1;

static void
usage (progname)
     const char *progname;
{
  printf ("%s (%s) %s\n"
	  "Copyright (C) 1999 Free Software Foundation, Inc.\n"
	  "This program comes with ABSOLUTELY NO WARRANTY.\n"
	  "This is free software, and you are welcome to redistribute it\n"
	  "under certain conditions. See the file COPYING for details.\n\n"
	  "Usage: %s [options] host port\n\n"
	  "Options:\n\n"
	  " --cert-file [file]      specify certificate file\n"
	  " --key-file [file]       specify private key file\n"
	  " --help                  show this help\n",
	  progname, PACKAGE, VERSION, progname);
}

static void
do_tls_negotiate(sig)
     int sig;
{
  tls_negotiate (tls_fd, opt_cert_file, opt_key_file);
}

int
tcp_connect (hostname, service)
     const char *hostname, *service;
{
  int server, _false = 0;
  struct addrinfo *in, *in0, hints;

  memset (&hints, 0, sizeof (hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  if (getaddrinfo (hostname, service, &hints, &in0))
    return -1;

  for (in = in0; in; in = in->ai_next)
    {
      server = socket (in->ai_family, in->ai_socktype, in->ai_protocol);
      if (server < 0)
	continue;
      if (connect (server, in->ai_addr, in->ai_addrlen) < 0)
	{
	  server = -1;
	  continue;
	}
      break;
  }

  if (server < 0)
    return -1;

  setsockopt (server, SOL_SOCKET, SO_KEEPALIVE, (const char *) &_false,
	      sizeof (_false));

  return server;
}

int
redirect (fd, buffer, nbuffer, write_function)
     int fd;
     char *buffer;
     int nbuffer;
     int (*write_function) (int, const char *, int);
{
  sigset_t mask, orig_mask;
  struct pollfd writefds[1];
  int wrote;
  char *retry;

  sigemptyset (&mask);
  sigaddset (&mask, SIGALRM);

  writefds[0].fd = fd;
  writefds[0].events = POLLOUT;

  for (retry = buffer; nbuffer > 0; nbuffer -= wrote, retry += wrote)
    {
      int ready;

      sigprocmask (SIG_SETMASK, &mask, &orig_mask);
      ready = poll (writefds, 1, -1);
      sigprocmask (SIG_SETMASK, &orig_mask, NULL);
      if (ready == -1 && errno != EINTR)
	{
	  if (opt_assuan)
	    printf ("ERR 1 poll: %s\r\n", strerror (errno));
	  return 1;
	}
      wrote = (*write_function)(fd, retry, nbuffer);
      if (wrote == -1)
	{
	  if (opt_assuan)
	    printf ("ERR 1 write: %s\r\n", strerror (errno));
	  return 1;
	}
    }
}

int
main (argc, argv) 
     int argc;
     char **argv;
{
  int in = fileno (stdin), out = fileno (stdout),
    nbuffer, wrote;
  struct pollfd readfds[2];
  char buffer[BUFSIZ];
  struct sigaction act;
  sigset_t mask, orig_mask;

  int this_option_optind = optind ? optind : 1;
  int option_index = 0, c;
  static struct option long_options[] =
    {
      {"cert-file", 1, 0, 'c'},
      {"key-file", 1, 0, 'k'},
      {"help", 0, 0, 'h'},
      {"assuan", 0, 0, 'A'},
      {0, 0, 0, 0}
    };

  while (1)
    {
      c = getopt_long (argc, argv, "c:k:hA", long_options, &option_index);
      if (c == -1)
	break;
    
      switch (c)
	{
	case 'c':
	  opt_cert_file = optarg;
	  break;
	case 'k':
	  opt_key_file = optarg;
	  break;
	case 'h':
	  usage (argv[0]);
	  return 0;
	case 'A':
	  opt_assuan = true;
	  break;
	default:
	  usage (argv[0]);
	  return 1;
	}
    }

  if (optind+2 != argc)
    {
      usage (argv[0]);
      return 1;
    }

  tls_fd = tcp_connect (argv[optind], argv[optind+1]);
  if (tls_fd < 0)
    {
      perror ("tcp_connect");
      return 1;
    }

  memset (&act, 0, sizeof (act));
  act.sa_handler = do_tls_negotiate;
  sigemptyset (&act.sa_mask);
  act.sa_flags = SA_RESTART|SA_RESETHAND;
  sigaction (SIGALRM, &act, NULL);

  sigemptyset (&mask);
  sigaddset (&mask, SIGALRM);

  readfds[0].fd = in;
  readfds[1].fd = tls_fd;
  readfds[0].events = POLLIN;
  readfds[1].events = POLLIN;

  while (1)
    {
      int ready;

      sigprocmask (SIG_SETMASK, &mask, &orig_mask);
      ready = poll (readfds, 2, -1);
      sigprocmask (SIG_SETMASK, &orig_mask, NULL);
      if (ready == -1 && errno != EINTR)
	{
	  if (opt_assuan)
	    printf ("ERR 1 poll: %s\r\n", strerror (errno));
	  return 1;
	}
      if (readfds[0].revents & POLLIN)
	{
	  nbuffer = read (in, buffer, sizeof buffer -1);
	  if (nbuffer == 0)
	    goto finish;
	  redirect (tls_fd, buffer, nbuffer, tls_write);
	}
      if (readfds[1].revents & POLLIN)
	do {
	  nbuffer = tls_read (tls_fd, buffer, sizeof buffer -1);
	  if (nbuffer == 0)
	    goto finish;
	  redirect (out, buffer, nbuffer, write);
	} while (tls_pending ());
    }

 finish:
  close (in);
  close (out);
  
  return 0;
}
