;; mixi-utils.el --- Utilities for mixi object -*- coding: euc-jp -*-

;; Copyright (C) 2007 OHASHI Akira

;; Author: OHASHI Akira <bg66@koka-in.org>
;; Keywords: hypermedia

;; This file is *NOT* a part of Emacs.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, you can either send email to this
;; program's maintainer or write to: The Free Software Foundation,
;; Inc.; 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

;;; Commentary:

;; Bug reports:
;;
;; If you have bug reports and/or suggestions for improvement, please
;; send them via <URL:http://mixi.jp/view_community.pl?id=1596390>.

;;; Code:

(require 'mixi)

(defvar mixi-reply-to nil)

(defmacro with-mixi-class (object &rest body)
  `(let ((class (mixi-object-class ,object)))
     ,@body))
(put 'with-mixi-class 'lisp-indent-function 'defun)
(put 'with-mixi-class 'edebug-form-spec '(body))

(defun mixi-make-objects (url-or-function &optional range)
  (if (stringp url-or-function)
      (let ((object (mixi-make-object-from-url url-or-function)))
	(with-mixi-class object
	  (cond ((eq class 'mixi-friend)
		 (mixi-get-diaries object range))
		((eq class 'mixi-community)
		 (mixi-get-bbses object range))
		((mixi-parent-p object)
		 (mixi-get-comments object range))
		(t (error (concat (symbol-name class)
				  " is not supported yet."))))))
    (funcall url-or-function range)))

(defun mixi-make-title (object &optional add-parent)
  (with-mixi-class object
    (cond ((eq class 'mixi-comment)
	   (concat "Re: " (mixi-make-title
			   (mixi-comment-parent object) add-parent)))
	  ((eq class 'mixi-log)
	   (mixi-friend-nick (mixi-log-friend object)))
	  (t
	   (let ((prefix (when (eq class 'mixi-event) "[٥]"))
		 (subject (mixi-object-title object))
		 (suffix (when add-parent
			   (concat " ("
				   (cond ((eq class 'mixi-diary)
					  (mixi-friend-nick
					   (mixi-diary-owner object)))
					 ((eq class 'mixi-news)
					  (mixi-news-media object))
					 (t
					  (mixi-community-name
					   (mixi-bbs-community object))))
				   ")"))))
	     (concat prefix subject suffix))))))

(defun mixi-make-author (object)
  (with-mixi-class object
    (if (eq class 'mixi-news)
	(mixi-news-media object)
      (let ((owner (if (eq class 'mixi-log)
		       (mixi-log-friend object)
		     (mixi-object-owner object))))
	(mixi-friend-nick owner)))))

(defun mixi-make-date (object)
  (let* ((time (mixi-object-time object))
	 (cts (current-time-string time))
	 (day-of-week (substring cts 0 3))
	 (month (substring cts 4 7)))
    (concat day-of-week ", "
	    (format-time-string "%d" time) " "
	    month " "
	    (format-time-string "%Y %H:%M:%S %z" time))))

(defun mixi-make-id-1 (object)
  (with-mixi-class object
    (concat
     (format-time-string "%Y%m%d%H%M" (mixi-object-time object)) "."
     (cond ((eq class 'mixi-comment)
	    (concat (mixi-friend-id (mixi-comment-owner object)) "@"
		    (mixi-object-id (mixi-comment-parent object)) "."
		    (mixi-friend-id (mixi-object-owner
				     (mixi-comment-parent object))) "."))
	   ((eq class 'mixi-log)
	    (concat (mixi-friend-id (mixi-log-friend object)) "@"))
	   (t
	    (concat (mixi-object-id object) "@"
		    (if (eq class 'mixi-news)
			(mixi-news-media-id object)
		      (mixi-object-id (mixi-object-owner object))) ".")))
     (mixi-object-name object))))

(defun mixi-make-message-id (object)
  (format "<%s.mixi.jp>" (mixi-make-id-1 object)))

(defun mixi-make-tag-uri (object)
  (format "tag:mixi.jp,%s:%s"
	  (format-time-string "%Y-%m-%d" (mixi-object-time object))
	  (mixi-make-id-1 object)))

(defun mixi-make-url (object)
  (with-mixi-class object
    (cond ((eq class 'mixi-diary)
	   (mixi-expand-url (mixi-diary-page object)))
	  ((eq class 'mixi-topic)
	   (mixi-expand-url (mixi-topic-page object)))
	  ((eq class 'mixi-event)
	   (mixi-expand-url (mixi-event-page object)))
	  ((eq class 'mixi-comment)
	   (concat (mixi-make-url (mixi-comment-parent object))
		   "#comment"))
	  ((eq class 'mixi-message)
	   (mixi-expand-url (mixi-message-page object)))
	  ((eq class 'mixi-news)
	   (mixi-news-page object))
	  ((eq class 'mixi-log)
	   (mixi-expand-url (mixi-friend-page (mixi-log-friend object))))
	  ((eq class 'mixi-friend)
	   (mixi-expand-url (mixi-friend-page object))))))

(defun mixi-make-encoded-url (object)
  (mixi-url-encode-string (mixi-make-url object)))

(defun mixi-make-content (object)
  (with-mixi-class object
    (cond ((eq class 'mixi-event)
	   (concat "<dl>"
		   "<dt></dt>"
		   "<dd>" (mixi-event-date object) "</dd>\n"
		   "<dt>žꡧ</dt>"
		   "<dd>" (mixi-event-place object) "</dd>\n"
		   "<dt>ܺ١</dt>"
		   "<dd>" (mixi-event-detail object) "</dd>\n"
		   "<dt>罸¡</dt>"
		   "<dd>" (mixi-event-limit object) "</dd>\n"
		   "<dt>üԡ</dt>"
		   "<dd>" (mixi-event-members object) "</dd>\n"
		   "</dl>"))
	  ((eq class 'mixi-friend)
	   (if (mixi-object-realized-p object)
	       (let ((sex (if (eq (mixi-friend-sex object) 'male) "" ""))
		     (age (if (numberp (mixi-friend-age object))
			      (number-to-string (mixi-friend-age object))
			    "??"))
		     (birthday (if (mixi-friend-birthday object)
				   (concat
				    (mapconcat (lambda (number)
						 (number-to-string number))
					       (mixi-friend-birthday object)
					       "") "")
				 "????"))
		     (blood-type (if (mixi-friend-blood-type object)
				     (symbol-name
				      (mixi-friend-blood-type object))
				   "?"))
		     (hobby (mapconcat 'identity
				       (mixi-friend-hobby object) ", ")))
		 (concat "<dl>"
			 "<dt>̾</dt>"
			 "<dd>" (mixi-friend-name object) "</dd>\n"
			 "<dt>̡</dt>"
			 "<dd>" sex "</dd>\n"
			 "<dt>ꡧ</dt>"
			 "<dd>" (mixi-friend-address object) "</dd>\n"
			 "<dt>ǯ</dt>"
			 "<dd>" age "</dd>\n"
			 "<dt></dt>"
			 "<dd>" birthday "</dd>\n"
			 "<dt>շ</dt>"
			 "<dd>" blood-type "</dd>\n"
			 "<dt>пϡ</dt>"
			 "<dd>" (mixi-friend-birthplace object) "</dd>\n"
			 "<dt>̣</dt>"
			 "<dd>" hobby "</dd>\n"
			 "<dt>ȡ</dt>"
			 "<dd>" (mixi-friend-job object) "</dd>\n"
			 "<dt>°</dt>"
			 "<dd>" (mixi-friend-organization object) "</dd>\n"
			 "<dt>ʾҲ</dt>"
			 "<dd>" (mixi-friend-profile object) "</dd>\n"
			 "</dl>"))
	     (concat "<a href=\"" (mixi-make-url object)
		     "\">ץեɽ</a>")))
	  ((eq class 'mixi-log)
	   (mixi-make-content (mixi-log-friend object)))
	  (t (mixi-object-content object)))))

(defun mixi-make-reply-to (object)
  (setq mixi-reply-to "mixi;")
  (with-mixi-class object
    (setq mixi-reply-to
	  (concat
	   (cond ((eq class 'mixi-diary)
		  (concat mixi-reply-to "comment;diary;"
			  (mixi-friend-id (mixi-diary-owner object)) ";"
			  (mixi-diary-id object)))
		 ((mixi-bbs-p object)
		  (concat mixi-reply-to "comment;"
			  (mixi-object-name object) ";"
			  (mixi-community-id (mixi-bbs-community object)) ";"
			  (mixi-bbs-id object)))
		 ((eq class 'mixi-community)
		  (concat mixi-reply-to "topic;"
			  (mixi-community-id object)))
		 ((or (eq class 'mixi-news) (eq object (mixi-make-me)))
		  (concat mixi-reply-to "diary"))
		 ((eq class 'mixi-message)
		  (concat mixi-reply-to "message;"
			  (mixi-friend-id (mixi-message-owner object))))
		 ((or (eq class 'mixi-friend) (eq class 'mixi-log))
		  (concat mixi-reply-to "message;"
			  (mixi-friend-id object))))))))

(defconst mixi-to-regexp
  "^mixi;\\([a-z]+\\);?\\([a-z0-9]+\\)?;?\\([0-9]+\\)?;?\\([0-9]+\\)?")

(defun mixi-send-mail (to title content)
  (when (string-match mixi-to-regexp to)
    (let ((method (match-string 1 to)))
      (cond ((string= method "comment")
	     (let ((parent (match-string 2 to))
		   (owner-id (match-string 3 to))
		   (id (match-string 4 to)))
	       (if (string= parent "diary")
		   (mixi-post-comment
		    (mixi-make-diary (mixi-make-friend owner-id) id) content)
		 (let ((func (intern
			      (concat mixi-object-prefix "make-" parent))))
		   (mixi-post-comment
		    (funcall func (mixi-make-community owner-id) id)
		    content)))))
	    ((string= method "topic")
	     (mixi-post-topic (mixi-make-community (match-string 2 to))
			      title content))
	    ((string= method "diary")
	     (mixi-post-diary title content))
	    ((string= method "message")
	     (mixi-post-message (mixi-make-friend (match-string 2 to))
				title content))))))

(provide 'mixi-utils)

;;; mixi-utils.el ends here
