;;; dropfile.el --- Drag and Drop a file

;; Author: Yuuichi Teranishi <teranisi@gohome.org>
;; Keywords: Windows, Icon

;; Copyright (C) 2004 Yuuichi Teranishi <teranisi@gohome.org>

;; This file is not part of GNU Emacs

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

;;; Commentary:
;; 

;;; History:
;; 

;;; Code:

(defgroup dropfile nil
  "Drag and Drop a file."
  :group 'image)

(defcustom dropfile-program "dropfile.exe"
  "Program name of dropfile."
  :type 'file
  :group 'dropfile)

(defcustom dropfile-temporary-directory (or temporary-file-directory
					    (getenv "TEMP"))
  "Temporary file directory."
  :type 'file
  :group 'dropfile)

(defcustom dropfile-work-directory "dropfile"
  "Working directory under temporary file directory."
  :type 'file
  :group 'dropfile)

(defvar dropfile-process-buffer "*dropfile*"
  "Process buffer for dropfile.exe.")

(defun dropfile-work-directory ()
  "Get temporary working directory dropfile."
  (let ((workdir (expand-file-name dropfile-work-directory
				   dropfile-temporary-directory)))
    (unless (file-directory-p workdir)
      (make-directory workdir))
    workdir))

(defun dropfile-make-file (file)
  "Make temporary file named FILE with current buffer content.
Return the full-path."
  (let ((coding-system-for-write 'binary)
	outfile)
    (write-region (point-min) (point-max)
		  (setq outfile
			(expand-file-name (file-name-nondirectory file)
					  (dropfile-work-directory)))
		  nil 'silent)
    outfile))

;;; dropfile API.
(defun dropfile (file &optional buffer)
  "Drag and Drop a FILE.
If BUFFER is specified, a file named FILE with BUFFER content is dropped."
  (when buffer
    (setq file (with-current-buffer buffer (dropfile-make-file file))))
  (setq file (with-temp-buffer
	       (insert file)
	       (goto-char (point-min))
	       (while (search-forward "/" nil t)
		 (replace-match "\\" nil t))
	       (buffer-string)))
  (unwind-protect
      (call-process dropfile-program nil nil nil file)
    (when buffer
      (delete-file file))))

(provide 'dropfile)

;;; dropfile.el ends here
