//
// dispicon.cpp - obtain displayed icon bitmap from specified file name
//
// Copyright (c) 2004  Yuuichi Teranishi
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//

#include <Windows.h>
#include <math.h>
#include <stdio.h>
#include "version.h"
#include "ThumbnailImage.h"

#define PROG "dispicon.exe"

#define THUMBNAIL_SIZE 48

HBITMAP
IconToBitmap (HICON hIcon, int width, int height,
	      COLORREF bgColor, int thumbnail)
{
  if (!hIcon)
    return NULL;
  RECT     rect;

  rect.right = width;
  rect.bottom = height;

  rect.left = rect.top  = 0;

  HWND desktop    = GetDesktopWindow();
  HDC  screen_dev = GetDC(desktop);

  // Create a compatible DC
  HDC dst_hdc = CreateCompatibleDC(screen_dev);

  // Create a new bitmap of icon size
  HBITMAP bmp = CreateCompatibleBitmap(screen_dev,
				       rect.right,
				       rect.bottom);

  // Select it into the compatible DC
  HBITMAP old_dst_bmp = (HBITMAP)SelectObject(dst_hdc, bmp);

  // Fill the background of the compatible DC with the given color
  if (thumbnail)
    {
      bgColor = 0xFFFFFF;
    }
  HBRUSH brush = ::CreateSolidBrush(bgColor);
  FillRect(dst_hdc, &rect, brush);
  DeleteObject(brush);

  // Draw the icon into the compatible DC
  DrawIconEx(dst_hdc,
	     thumbnail? abs (rect.right - THUMBNAIL_SIZE) / 2 : 0,
	     thumbnail? abs (rect.bottom - THUMBNAIL_SIZE) / 2 : 0,
	     hIcon,
	     thumbnail? THUMBNAIL_SIZE : rect.right,
	     thumbnail? THUMBNAIL_SIZE : rect.bottom,
	     0, NULL, DI_NORMAL);

  // Restore settings
  SelectObject(dst_hdc, old_dst_bmp);
  DeleteDC(dst_hdc);
  ReleaseDC(desktop, screen_dev); 
  DestroyIcon(hIcon);
  return bmp;
}

BOOL
SaveBitmapFile (char *outFile, HBITMAP hBitmap, int bits)
{
  HDC hDC = ::GetDC(NULL);
  
  HANDLE hFile;
  DWORD	 dwWrite;
  int stat;

  BITMAPFILEHEADER	bmfh;
  LPBITMAPINFO		lpbmpinfo;
  BITMAP		sBitmap;
  BITMAPINFOHEADER	sBmiHeader;
  BITMAPINFOHEADER*	psBmiHeader;
  HANDLE		psBminfo;
  BYTE*			pbBits;
  int			iBitmapSize, iLineSize;

  if (outFile == NULL || hBitmap == NULL) return FALSE;
  if (bits != 1 && bits != 4 && bits != 8 && bits != 16 && 
      bits != 24 && bits != 32) return FALSE;
  GetObject(hBitmap, sizeof(BITMAP), &sBitmap);

  sBmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  sBmiHeader.biWidth = sBitmap.bmWidth;
  sBmiHeader.biHeight = sBitmap.bmHeight;
  sBmiHeader.biPlanes = 1;
  sBmiHeader.biBitCount = bits;
  sBmiHeader.biCompression = BI_RGB;
  sBmiHeader.biSizeImage = 0;
  sBmiHeader.biXPelsPerMeter = 0;
  sBmiHeader.biYPelsPerMeter = 0;
  sBmiHeader.biClrUsed = 0;
  sBmiHeader.biClrImportant = 0;
  
  psBminfo = (bits < 16) ?
    GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER) +
		(1 << bits) * sizeof(RGBQUAD) ) : 
    GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER));

  psBmiHeader = (BITMAPINFOHEADER *)GlobalLock(psBminfo);

  *psBmiHeader = sBmiHeader;
  lpbmpinfo = (LPBITMAPINFO)psBmiHeader;

  iLineSize = ((int)ceil( (double)sBitmap.bmWidth * (double)bits / 32.0)) * 4;
  iBitmapSize = iLineSize * sBitmap.bmHeight;
  pbBits = (BYTE *)LocalAlloc(LMEM_FIXED, sizeof(BYTE) * iBitmapSize);

  GetDIBits(hDC, hBitmap, 0, sBitmap.bmHeight, pbBits, lpbmpinfo,
	    DIB_RGB_COLORS);
  bmfh.bfType = 19778;
  bmfh.bfSize = (bits < 16) ?
    sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + 
    sizeof(RGBQUAD) * (1 << bits) + iBitmapSize :
    sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + iBitmapSize;
  bmfh.bfReserved1 = 0;
  bmfh.bfReserved2 = 0;
  bmfh.bfOffBits = (bits < 16) ? sizeof(BITMAPFILEHEADER) +
    sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * (1 << bits) :
    sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

  stat = FALSE;

  if (strcmp (outFile, "-") == 0)
    {
      hFile = GetStdHandle (STD_OUTPUT_HANDLE);
    }
  else
    {
      hFile = CreateFile(outFile, GENERIC_WRITE, FILE_SHARE_WRITE,
			 NULL, CREATE_ALWAYS, NULL, NULL);
    }
  if (hFile == INVALID_HANDLE_VALUE) goto Exit;

  stat = WriteFile(hFile, &bmfh, sizeof( BITMAPFILEHEADER ),
		   &dwWrite, NULL);
  if (stat == FALSE) goto End;

  if (bits < 16)
    {
      stat = WriteFile(hFile, lpbmpinfo,
			  sizeof(BITMAPINFOHEADER) +
			  sizeof(RGBQUAD) * (1 << bits), &dwWrite, NULL);
    }
  else
    {
      stat = WriteFile(hFile, lpbmpinfo, sizeof(BITMAPINFOHEADER),
			  &dwWrite, NULL);
    }
  if (stat == FALSE) goto End;

  stat = WriteFile(hFile, pbBits, iBitmapSize, &dwWrite, NULL);
 End:
  CloseHandle(hFile);
 Exit:
  ::ReleaseDC(NULL, hDC);
  LocalFree(pbBits);
  GlobalUnlock(psBminfo);
  GlobalFree(psBminfo);
  if (stat == FALSE) return FALSE;
  return TRUE;
}

// From mew/bin
int  Optind = 1;
char *Optarg = NULL;
#define NUL '\0'
char *warn_prog = NULL;

void
warn_exit(const char *fmt, ...)
{
  va_list ap;

  if (warn_prog != NULL)
    fprintf(stderr, "%s: ", warn_prog);
  va_start(ap, fmt);
  if (fmt != NULL)
    vfprintf(stderr, fmt, ap);
  va_end(ap);
  
  exit(1);
}

int
Getopt(int argc, char *argv[], const char *fmt)
{
	char *p, *q, c;

	Optarg = NULL;
	if (Optind >= argc)
		return EOF;
	p = argv[Optind];

	if (*p++ != '-')
		return EOF;
	c = *p;
	if (c == NUL)
		return EOF;
	if (*(p + 1) != NUL)
		warn_exit("unknown long option '-%s'.\n", p);
	if ((q = strchr(fmt, c)) == NULL)
		warn_exit("unknown option '-%c'.\n", c);
	if (++q != NULL && *q == ':') {
		if (++Optind >= argc)
			warn_exit("no parameter for '-%c'.\n", c);
		Optarg = argv[Optind];
	}
	Optind++;
	return c;
}

void
usage (char *progname)
{
  printf ("%s - Get displayed icon bitmap from specified file name\n\
Usage: %s [options] filename\n%s",
	  progname,
	  progname,
	  "Options:\n\
 -b color\n\
    specify background color by BGR hex string. (ex. #FFFFFF(default))\n\
 -d depth\n\
    specify the depth of output bitmap. (ex. 16, 24(default), 32)\n\
 -f \n\
    extract icon from existing file.\n\
 -h \n\
    print this message and exit.\n\
 -l \n\
    obtain large icon (default is small icon).\n\
 -o file\n\
    output file name (- means stdout(default)).\n\
 -s size\n\
    width and height of the output bitmap. (ex. 32 (default))\n\
");
  exit (1);
}

int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow )
{
  SHFILEINFO sfi;
  char *outFile = NULL;
  char *inFile = NULL;
  int width = 32;
  int height = 32;
  int depth = 24;
  COLORREF bgColor = 0xFFFFFF;
  int optc;
  int useExisting = FALSE;
  int thumbnail = FALSE;
  warn_prog = PROG;
  int getFileInfoArg = SHGFI_ICON | SHGFI_SMALLICON;
  HBITMAP bm;

  while ((optc = Getopt (__argc, __argv, "b:d:fhlo:s:tv")) != EOF)
    {
      switch (optc)
	{
	case 'b': // bg color
	  sscanf (Optarg, "#%X", &bgColor);
	  printf ("%X", bgColor);
	  break;
	case 'd': // depth
	  depth = atoi (Optarg);
	  break;
	case 'h': // help
	  usage (PROG);
	  break;
	case 'f':
	  useExisting = TRUE;
	  break;
	case 'l':
	  getFileInfoArg = SHGFI_ICON;
	  break;
	case 'o': // outfile
	  outFile = strdup (Optarg);
	  break;
	case 's': // size (height and width)
	  width = atoi (Optarg);
	  height = atoi (Optarg);
	  break;
	case 't':
	  thumbnail = TRUE;
	  break;
	case 'v':
	  fprintf (stderr, "%s - %s\n", PROG, VERSION);
	  exit (0);
	  break;
	}
    }
  if (!useExisting)
    getFileInfoArg |= SHGFI_USEFILEATTRIBUTES;
  if (outFile == NULL)
    {
      outFile = strdup("-");
    }
  if ((__argc - Optind) != 1)
    {
      usage (PROG);
    }
  if (thumbnail)
    {
      OLECHAR ochPath[MAX_PATH];
      OLECHAR ochFile[MAX_PATH];
      char *file;
      char *p;
      ThumbnailImage thum;

      file = __argv[Optind];
      p = file + strlen (file);
      while (*p != '\\')
	{
	  p--;
	}
      ::MultiByteToWideChar( CP_ACP, MB_PRECOMPOSED, 
			     (p + 1), -1, ochFile, MAX_PATH );
      *p = '\0';
      ::MultiByteToWideChar( CP_ACP, MB_PRECOMPOSED, 
			     file, -1, ochPath, MAX_PATH );

      // Revert.
      *p = '\\';
      bm = thum.ExtractFile (ochPath, ochFile, width, height, depth);

      if (!bm) // No bitmap found. Obtain the large icon.
	{
	  SHGetFileInfo(__argv[Optind],//trim (lpCmdLine),
			FILE_ATTRIBUTE_NORMAL,
			&sfi,sizeof(SHFILEINFO),
			SHGFI_ICON);
	  if (sfi.hIcon)
	    {
	      bm = IconToBitmap(sfi.hIcon, width, height, bgColor, TRUE);
	    }
	}
    }
  else
    {
      SHGetFileInfo(__argv[Optind],//trim (lpCmdLine),
		    FILE_ATTRIBUTE_NORMAL,
		    &sfi,sizeof(SHFILEINFO),
		    getFileInfoArg);
      if (sfi.hIcon)
	{
	  bm = IconToBitmap(sfi.hIcon, width, height, bgColor, FALSE);
	}
    }
  if (bm)
    {
      if (SaveBitmapFile (outFile, bm, depth))
	{
	  exit (0);
	}
      else
	{
	  exit (1);
	}
    }
  else
    {
      fprintf (stderr, "ERROR: cannot find icon.\n");
    }
}

