;;; -*- byte-compile-dynamic: t -*-
;;; mucs-ccl.el --- for MULE-UCS basic ccl configurations

;; Copyright (C) 1997-1999 Miyashita Hisashi

;; Keywords: mule, multilingual, 
;;           character set, coding-system, ISO10646, Unicode

;; This file is part of MULE-UCS

;; MULE-UCS is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; MULE-UCS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;; Comment:

(require 'mucs-type)
(require 'trans-util)
(require 'ccl)

;;; ------ Some notes on MUCS-CCL. ------
;;; MUCS-CCL stands for MUle-uCS CCL generation library.
;;; This module provides the following services.
;;;
;;; 1 ... more abstracted description. (that has upper compatibility.)
;;; 2 ... supply basic CCL libraries.
;;; 3 ... manages the state to produce CCL codes.
;;;           This state is called `mucs-ccl-production-state'.
;;; 4 ... register the mucs-ccl-production-state
;;;                at the location marked by users.
;;; 5 ... produce the table structures that are used by ExCCL.
;;;


;;;
;;; Table package registration manager.
;;;

(defun mucs-ccl-registered-p (table)
  (memq table (get mucs-current-package
		   'mucs-registered-ccls)))

(defun mucs-ccl-add-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (or
       (mucs-ccl-registered-p table)
       (put mucs-current-package
	    'mucs-registered-ccls
	    (cons table
		  (get mucs-current-package
		       'mucs-registered-ccls))))))

(defun mucs-ccl-delete-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (put mucs-current-package
	   'mucs-registered-ccls
	   (delq
	    (get mucs-current-package
		 'mucs-registered-ccls)))))

;;;
;;; To check invalid code
;;;

(defun mucs-ccl-if-invalid (invalid-part &optional valid-part)
  (if valid-part
      `((if (r0 == ,mucs-invalid-code)
	    (,@invalid-part)
	  (,@valid-part)))
    `((if (r0 == ,mucs-invalid-code)
	  (,@invalid-part)))))

(defun mucs-ccl-if-valid (valid-part)
  `((if (r0 != ,mucs-invalid-code)
	(,@valid-part))))

(defun mucs-ccl-if-invalid-repeat ()
  (mucs-ccl-if-invalid '((loop))))

;;;
;;; Common I/O interface.
;;;

(defun mucs-ccl-read (type serialize)
  (setq mucs-current-type type)
  (mucs-type-get-serialize-method type serialize t))

(defun mucs-ccl-write (serialize)
  (if mucs-current-type
      (mucs-ccl-if-valid
       (mucs-type-get-serialize-method
	mucs-current-type serialize nil))
    (error "Current TYPE is not set.  Cannot write.")))

;;;
;;; some simple ccl codes.
;;;

(defvar mucs-ccl-read-ex-1-octet
  `((read r0)))

(defvar mucs-ccl-read-ex-be-2-octet
  `((read r0 r6)
    (r0 = (r0 <8 r6))))

(defvar mucs-ccl-read-ex-le-2-octet
  `((read r0 r6)
    (r0 = (r6 <8 r0))))

(defvar mucs-ccl-read-ex-be-4-octet
  `((read r0 r6)
    (r0 = (r0 <8 r6))
    (read r4 r6)
    (r0 |= (r4 << 16))))

(defvar mucs-ccl-read-ex-le-4-octet
  `((read r0 r6)
    (r0 |= (r6 << 16))
    (read r4 r6)
    (r0 |= (r4 <8 r6))))

(defvar mucs-ccl-write-ex-1-octet
  `((write r0)))

(defvar mucs-ccl-write-ex-be-2-octet
  `((r6 = (r0 >> 8))
    (r4 = (r0 & ?\xff))
    (write r6 r4)))

(defvar mucs-ccl-write-ex-le-2-octet
  `((r6 = (r0 >> 8))
    (r4 = (r0 & ?\xff))
    (write r4 r6)))

(defvar mucs-ccl-write-ex-be-4-octet
  `((r4 = ((r0 >> 16) & ?\xff))
    (r6 = 0)
    (write r6 r4)
    (r6 = ((r0 >> 8) & ?\xff))
    (r4 = (r0 & ?\xff))    
    (write r6 r4)))

(defvar mucs-ccl-write-ex-le-4-octet
  `((r4 = (r0 & ?\xff))
    (r6 = ((r0 >> 8) & ?\xff))
    (write r4 r6)
    (r4 = ((r0 >> 16) & ?\xff))
    (r6 = 0)
    (write r4 r6)))


;;;
;;; generic type definition
;;;

(defun mucs-ccl-write-char-packed-1 ()
  '((r4 = (r0 << 16))
    (r0 = (r0 & ?\xFFFF))
    (write-multibyte-character r4 r0)))

(defun mucs-ccl-read-char-packed-1 ()
  '((read-multibyte-character r1 r0)
    (r0 = ((r1 << 16) | r0))))

(mucs-define-type
 'char-packed-1
 (lambda (x)
   (make-char-from-charset-id-codepoint
    (lsh x -16)
    (logand x ?\xFFFF)))
 (lambda (x)
   (logior (lsh (charset-id (char-charset x)) 16)
	   (char-codepoint x))))

(mucs-type-register-serialization
 'char-1
 'emacs-mule
 '((write r0)) ;;;; umm....
 'none)

(mucs-type-register-serialization
 'char-packed-1
 'emacs-mule
 (mucs-ccl-write-char-packed-1)
 (mucs-ccl-read-char-packed-1))

(mucs-type-register-conversion
 'char-1
 'char-packed-1
 'identity)


;;
;; Dealing with line separator problem.
;;

(defconst mucs-ccl-previous-cr-p-bit 16)

(defun mucs-ccl-previous-cr-p ()
  '(r5 & ?\x2))

(defun mucs-ccl-set-eol-state (eol)
  (if (eq eol 'cr)
      '((r5 |= ?\x2))
    `((r5 &= ,(logxor (mucs-number-mask) ?\x2)))))

(defun mucs-ccl-write-char-packed-1-dos ()
  `((if (r0 == ?\x0d)
	((if ,(mucs-ccl-previous-cr-p)
	     ((write ?\x0d))
	   ,(mucs-ccl-set-eol-state 'cr)))
      ((r6 = (r0 != ?\x0a))
       (if (,(mucs-ccl-previous-cr-p) & r6)
	   ((write ?\x0d)))
       ,@(mucs-ccl-set-eol-state nil)
       ,@(mucs-ccl-write-char-packed-1)))))

(defun mucs-ccl-write-char-1-dos ()
  `((if (r0 == ?\x0d)
	((if ,(mucs-ccl-previous-cr-p)
	     ((write ?\x0d))
	   ,(mucs-ccl-set-eol-state 'cr)))
      ((r6 = (r0 != ?\x0a))
       (if (,(mucs-ccl-previous-cr-p) & r6)
	   ((write ?\x0d)))
       ,@(mucs-ccl-set-eol-state nil)
       (write r0)))))

(mucs-type-register-serialization
 'char-1
 'emacs-mule-dos
 (mucs-ccl-write-char-1-dos)
 'none)

(mucs-type-register-serialization
 'char-packed-1
 'emacs-mule-dos
 (mucs-ccl-write-char-packed-1-dos)
 'none)


;;;
;;; CCL program generator
;;;

;;
;;  MYO fundamental section.
;;

;; NOTE:
;;    MUCS-CCL defines a simple message to pass generated program
;; and data structure, which is called MYO.
;; The followings show the specification of MYO.
;;
;;    (myo . A-LIST)
;;          Special data structure for mucs-ccl.
;;          A-LIST has data generated by CCL generator.
;;          The reserved keys of the A-LIST are as
;;          follows(All of them are SYMBOL),
;;          and all MYO MUST have these keys.
;;          (a) ... ccl
;;                   generated CCL programs
;;          (b) ... elisp
;;                   Elisps that are supporsed to evaled
;;                   before CCL programs are registered.
;;          (c) ... table-set
;;                   Generated table-set that are supporsed to be
;;                   registered before CCL programs are registered.
;;
;;

(defsubst mucs-ccl-myo-p (data)
  (eq 'myo (car-safe data)))

(defsubst mucs-ccl-check-myo (data)
  (if (not (and (mucs-ccl-myo-p data)
		(assq 'elisp (cdr data))
		(assq 'ccl (cdr data))
		(assq 'table-set (cdr data))))
      (error "Invalid MYO:%S" data)))


(defsubst mucs-ccl-empty-myo ()
  (cons 'myo (list
	      (cons 'ccl nil)
	      (cons 'elisp nil)
	      (cons 'table-set nil)))
  )

(defsubst mucs-ccl-myo-from-ccl (ccl)
  (cons 'myo (list (list 'ccl ccl)
		   (cons 'elisp nil)
		   (cons 'table-set nil))))

(defsubst mucs-ccl-myo-append (myo1 myo2)
  "Returns concatinated MYO1 and MYO2.
MYO1 and MYO2 may be destructed."
  (let ((alist1 (cdr myo1))
	(alist2 (cdr myo2))
	elem1 elem2)
    (while (setq elem1 (car alist1))
      (setq elem2 (assq (car elem1) alist2)
	    alist1 (cdr alist1))
      (if elem2
	  (progn
	    (setcdr elem1 (append (cdr elem1)
				  (cdr elem2)))
	    (setq alist2 (delq elem2 alist2)))))
    (nconc myo1 alist2)))


(defsubst mucs-ccl-myo-add-elisp (elisp myo)
  (let ((slot (assq 'elisp (cdr myo))))
    (setcdr slot (append (cdr slot) elisp))
    myo))

(defsubst mucs-ccl-myo-get-elisp (myo)
  (cdr (assq 'elisp (cdr myo))))

(defsubst mucs-ccl-myo-set-elisp (elisp myo)
  (setcdr (assq 'elisp (cdr myo)) elisp))


(defsubst mucs-ccl-myo-add-ccl (ccl myo)
  (let ((slot (assq 'ccl (cdr myo))))
    (setcdr slot (append (cdr slot) ccl))
    myo))

(defsubst mucs-ccl-myo-get-ccl (myo)
  (cdr (assq 'ccl (cdr myo))))

(defsubst mucs-ccl-myo-set-ccl (ccl myo)
  (setcdr (assq 'ccl (cdr myo)) ccl)
  myo)


(defsubst mucs-ccl-myo-add-table-set (table-set myo)
  (let ((slot (assq 'table-set (cdr myo))))
    (setcdr slot (append (cdr slot)
			 (list table-set)))
    myo))

(defsubst mucs-ccl-myo-get-table-set (myo)
  (cdr (assq 'table-set (cdr myo))))

(defsubst mucs-ccl-myo-set-table-set (ccl myo)
  (setcdr (assq 'table-set (cdr myo)) ccl))


;;
;; conversion configuration handler
;;

(defun mucs-ccl-bind-program (progs)
  (let ((myo (mucs-ccl-empty-myo))
	ret)
    (while progs
      (setq ret (car progs)
	    progs (cdr progs))
      (if (mucs-ccl-myo-p ret)
	  (setq myo (mucs-ccl-myo-append myo ret))
	(if (consp ret)
	    (setq myo (mucs-ccl-myo-add-ccl ret myo))
	  (if ret
	      (error "Unknown MUCS-CCL data:%S" ret)))))
    myo))

(defun mucs-ccl-stream-form (&rest args)
  (let ((myo (mucs-ccl-bind-program args)))
    (mucs-ccl-myo-set-ccl
     (list
      (append '(loop)
	      (mucs-ccl-myo-get-ccl myo)
	      '((repeat))))
     myo)))

;;
;; misc.
;;

(defun mucs-ccl-make-elisp-preparation-from-myo (myo)
  (append
   (mucs-ccl-myo-get-elisp myo)
   (mapcar
    (function generate-table-set-registration-program)
    (mucs-ccl-myo-get-table-set myo))))

;;
;; API.
;;

(defsubst mucs-convert (conversion object)
  (let ((ccl-object
	 (funcall
	  (mucs-type-get-ccl-representation
	   (car (mucs-conversion-get-conv-type conversion)))
	  object))
	(execute-vector
	 (make-vector 8 0)))
    (ccl-execute conversion
		 (cond ((numberp ccl-object)
			(aset execute-vector 0 ccl-object)
			execute-vector)
		       ((vectorp ccl-object)
			ccl-object)
		       (t
			(error "MULE-UCS(Convert) invalid object: %S" object))))
    (if (= (aref execute-vector 0) mucs-invalid-code)
	nil
      (funcall
       (mucs-type-get-elisp-representation
	(cdr (mucs-conversion-get-conv-type conversion)))
       (if (vectorp ccl-object)
	   execute-vector
	 (aref execute-vector 0))))))

(defun mucs-setup-defun-convsersion (conv def)
)

(defun mucs-setup-conversion (conv def)
  (let ((convtype (mucs-conversion-get-conv-type conv))
	(mucs-current-type nil)
	(mag (mucs-conversion-definition-mag def))
	(config-main (mucs-conversion-definition-main-prog def))
	(config-eof (mucs-conversion-definition-eof-prog def))
	(main-ccl-myo (mucs-ccl-empty-myo))
	(eof-ccl-myo (mucs-ccl-empty-myo))
	ccl-program
	compiled-ccl
	func ret)
    (if (consp convtype)
	(setq mucs-current-type (car convtype)))
    (setq func (lambda (config myo)
		 (while config
		   (setq ret (eval (car config))
			 config (cdr config))
		   (if (consp ret)
		       (if (mucs-ccl-myo-p ret)
			   (setq myo (mucs-ccl-myo-append myo ret))
			 (setq myo (mucs-ccl-myo-add-ccl ret myo)))
		     (if ret
			 (error "Invalid MUCS-CCL:%S" ret))))))

    (funcall func config-main main-ccl-myo)
    (funcall func config-eof eof-ccl-myo)
    (setq ccl-program (list mag
			    (mucs-ccl-myo-get-ccl main-ccl-myo)
			    (mucs-ccl-myo-get-ccl eof-ccl-myo)))
    (setq compiled-ccl (ccl-compile ccl-program))
    (mucs-conversion-set-program-and-compiled-code
     conv ccl-program compiled-ccl)
    (append
     (mucs-ccl-make-elisp-preparation-from-myo main-ccl-myo)
     (mucs-ccl-make-elisp-preparation-from-myo eof-ccl-myo)
     `((declare-ccl-program ,conv ,compiled-ccl)))))

(provide 'mucs-ccl)
