;;; mew-minibuf.el --- Minibuffer input methods for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar 23, 1997

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Completion switch
;;;

(defvar mew-input-complete-function nil)

(defun mew-input-complete ()
  "Do completion according to the global variable
\"mew-input-complete-function\"."
  (interactive)
  (if (and mew-input-complete-function (fboundp mew-input-complete-function))
      (funcall mew-input-complete-function)))

(defvar mew-exit-minibuffer-function nil)

(defun mew-exit-minibuffer ()
  "Ensure the input meets a condition."
  (interactive)
  (if (or (not (and mew-exit-minibuffer-function
                    (fboundp mew-exit-minibuffer-function)))
          (funcall mew-exit-minibuffer-function))
      (exit-minibuffer)))

(defvar mew-input-comma-function nil)

(defun mew-input-comma ()
  "Ensure the input meets a condition."
  (interactive)
  (when (or (not (and mew-input-comma-function
		      (fboundp mew-input-comma-function)))
	    (funcall mew-input-comma-function))
    (if mew-xemacs-p
	(insert (event-to-character last-command-event))
      (insert last-command-event))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Mew original completion
;;;

(defun mew-input-clear ()
  "A function to clean up side effects of window configuration
at completions."
  (save-excursion
    (set-buffer (window-buffer (minibuffer-window)))
    ;; (mew-ainfo-get-win-cfg) is shared by many functions
    ;; because minibuffer is just one!
    (mew-ainfo-set-win-cfg nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folder
;;;

(defvar mew-input-folder-hist nil)

(defun mew-input-folder (default)
  (let ((prefix (substring default 0 1))
	 folder)
    (mew-input-clear)
    (setq mew-input-complete-function (function mew-complete-folder))
    (setq folder (read-from-minibuffer (format "Folder name (%s): " default)
				       prefix
				       mew-input-map
				       nil
				       'mew-input-folder-hist))
    (if (or (string= folder "") (string= folder prefix))
	(directory-file-name default)
      (directory-file-name folder)))) ;; +foo/ -> +foo

(defun mew-input-folders (default)
  (let (prompt init folders)
    (cond
     ((consp default)
      (setq prompt "Folder name: ")
      (setq init (mew-join "," default)))
     (t
      (setq prompt (format "Folder name (%s): " default))
      (setq init "+")))
    (mew-input-clear)
    (setq mew-input-complete-function (function mew-complete-folder))
    (setq folders (read-from-minibuffer prompt
					init
					mew-input-map
					nil
					'mew-input-folder-hist))
    (setq folders (mapcar (function mew-chop) (mew-split folders ?,)))
    (when folders
      (setq folders (delete "+" folders)))
    (when (and (null folders) (stringp default))
      (setq folders (list default)))
    (if folders
	;; +foo/ -> +foo
	(mapcar (function directory-file-name) folders)
      nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Address
;;;

(defvar mew-input-address-hist nil)

(defun mew-input-address (prompt &optional default)
  (mew-input-clear)
  (setq mew-input-complete-function (function mew-complete-address))
  (let (val vals addrs ret)
    (setq val (read-from-minibuffer 
	       (if default (format prompt default) prompt)
	       ""
	       mew-input-map
	       nil
	       'mew-input-address-hist))
    (if (and default (string= val ""))
	(setq val default))
    (setq vals (mapcar (function mew-chop)  (mew-split-quoted val ?,)))
    (while vals
      (setq val (car vals))
      (setq vals (cdr vals))
      (setq addrs (mew-alias-expand val mew-addrbook-alist 0))
      (setq addrs (mapcar (function mew-addrstr-append-domain) addrs))
      (setq ret (nconc ret addrs)))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Pick pattern
;;;

(defvar mew-input-pick-pattern-hist nil)

(defun mew-input-pick-pattern ()
  (mew-input-clear)
  (setq mew-input-complete-function (function mew-complete-pick-pattern))
  (let ((keymap (copy-keymap mew-input-map)) ret)
    (define-key keymap " " nil)
    (setq ret (mew-pick-macro-expand-string
	       (read-from-minibuffer "Pick pattern: "
				     mew-pick-default-field
				     keymap
				     nil
				     'mew-input-pick-pattern-hist)))
    (mew-decode-syntax-delete)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Grep pattern
;;;

(defun mew-input-grep-pattern ()
  (read-string "Grep pattern: "))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sort key
;;;
;;; mew-sort-default-key-alist

(defvar mew-input-sort-key-hist nil)

(defun mew-sort-key-check ()
  (let* ((field:mode (mew-buffer-substring 
		      (save-excursion (beginning-of-line) (point))
		      (point-max)))
	 (mode (car (cdr (mew-split field:mode ?:))))
	 err)
    (if mode
	(unless (member mode mew-sort-modes)
	  (setq err mode)))
    (if err
        (progn
          (mew-temp-minibuffer-message (format " [No match: %s]" err))
          nil)
      t)))

(defun mew-input-sort-key (key)
  (mew-input-clear)
  (setq mew-input-complete-function (function mew-complete-sort-key))
  (let* ((mew-exit-minibuffer-function (function mew-sort-key-check))
	 (field:mode (read-from-minibuffer
		      (format "Sort by (%s)? : " key)
		      ""
		      mew-input-map
		      nil 
		      'mew-input-sort-key-hist))
	 field mode)
    (if (or (null field:mode) (string= field:mode ""))
	(setq field:mode key))
    (setq field (car (mew-split field:mode ?:)))
    (setq mode  (or (car (cdr (mew-split field:mode ?:)))
		    (cdr (assoc field mew-sort-key-alist))
		    "text"))
    (cons field mode)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Remote file
;;;

(defvar mew-input-rfile-hist nil)

(defun mew-input-rfile (prompt) ;; prompt="To:"
  (mew-input-clear)
  (setq mew-input-complete-function (function mew-complete-rfile))
  (read-from-minibuffer
   (concat prompt " ")
   ""
   mew-input-map
   nil
   'mew-input-rfile-hist))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Range
;;;

(defun mew-input-range (folder askp)
  (let* (default comp range ret)
    (if (or (mew-folder-draftp folder)
	    (mew-folder-queuep folder)
	    (mew-folder-mdropp folder))
	;; folders whose messages are dynamically removed.
	(setq default mew-range-str-all)
      (setq default mew-range-str-update))
    (when askp
      (setq comp (mapcar (function (lambda (x) (cons x x)))
			 mew-input-range-list))
      (setq range (completing-read (format "Range (%s): " default) comp)))
    (if (or (string= range "") (null range))
	(setq range default))
    (cond
     ((string= range mew-range-str-all)
      (setq ret mew-range-all))
     ((string= range mew-range-str-update)
      (save-excursion
	(set-buffer folder)
	(goto-char (point-max))
	(if (bobp)
	    (setq ret mew-range-all) ;; buffer is empty.
	  (forward-line -1)
	  (mew-summary-goto-message)
	  (setq ret
		(concat (int-to-string
			 (1+ (string-to-int (mew-summary-message-number))))
			"-")))))
     ((string-match ":" range)
      (setq ret range))
     (t
      ;; If range starts with "0", Summary buffer is erased.
      ;; 1-20 -> 01-20
      ;; -20 -> 0-20
      (setq ret (concat mew-range-all range))))
    (mew-decode-syntax-delete)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffer
;;;

(defun mew-input-buffer (default)
  (let* ((comp (mapcar (lambda (x) (list (buffer-name x))) (buffer-list)))
	 (buf (completing-read (format "Buffer (%s): " default) comp)))
    (if (string= buf "")
	default
      buf)))

(defun mew-input-draft-buffer (default)
  (let ((regex (mew-folder-regex (file-name-as-directory mew-draft-folder)))
	comp buf)
    (setq comp (delq nil (mapcar
			  (lambda (x)
			    (when (string-match regex (buffer-name x))
			      (list (buffer-name x))))
			  (buffer-list))))
    (if (and (= (length comp) 1)
	     (string= default (car (car comp))))
	default
      (setq buf (completing-read (format "Buffer (%s): " default) comp))
      (if (string= buf "")
	  default
	buf))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File
;;;

(defun mew-input-file-name (&optional prompt default)
  (let ((msg (or prompt "File: "))
	(use-dialog-box nil)
	file)
    (cond
     ((null default)
      (setq file mew-home))
     ((or (string-match (format "^[~%s]" mew-path-separator) default)
        ;; allow drive letter -- "^\\([A-Za-z]:\\|[~%s]\\)"
        (string-match (format "^[A-Za-z]:%s" mew-path-separator) default))
      (setq file default))
     (t
      (setq file (concat mew-home default))))
    (expand-file-name (read-file-name msg file file))))

(defun mew-input-directory-name (&optional default)
  (let ((dir (expand-file-name
	      (read-file-name "Directory : " default default t))))
    (if (file-directory-p dir)
	dir
      (mew-warn "%s is not directory" dir)
      (mew-input-directory-name default))))

(defun mew-convert-to-home-dir (dir)
  (let* ((chome (file-name-as-directory mew-home))
	 (ehome (expand-file-name chome)))
    (if (string-match ehome dir)
	(concat chome (substring dir (match-end 0) nil))
      dir)))

(defvar mew-summary-previous-directory nil)
(defvar mew-draft-previous-directory nil)

(defmacro mew-mode-input-file-name (prompt file preservep previous modedir)
  `(let (dir ret def)
     (if (and ,file (file-name-absolute-p ,file))
	 (setq def (mew-convert-to-home-dir ,file))
       (if ,preservep
	   (setq dir (or ,previous ,modedir))
	 (setq dir , modedir))
       (setq dir (and dir (file-name-as-directory dir)))
       (setq def (concat dir ,file)))
     (setq ret (mew-input-file-name ,prompt def))
     (if ,preservep
	 (setq ,previous (file-name-directory (mew-convert-to-home-dir ret))))
     ret))

(defun mew-summary-input-file-name (&optional prompt file)
  (mew-mode-input-file-name prompt file mew-summary-preserve-dir
			    mew-summary-previous-directory mew-save-dir))

(defun mew-draft-input-file-name (&optional prompt file)
  (mew-mode-input-file-name prompt file mew-draft-preserve-dir
			    mew-draft-previous-directory mew-copy-dir))

(defmacro mew-mode-input-directory-name (preservep previous modedir)
  `(if ,preservep
       (let (dir ret)
	 (setq dir (file-name-as-directory (or ,previous ,modedir)))
	 (setq ret (mew-input-directory-name dir))
	 (setq ,previous (mew-convert-to-home-dir ret))
	 ret)
     (mew-input-directory-name default-directory)))

(defun mew-summary-input-directory-name ()
  (mew-mode-input-directory-name
   mew-summary-preserve-dir mew-summary-previous-directory mew-save-dir))

(defun mew-draft-input-directory-name ()
  (mew-mode-input-directory-name
   mew-draft-preserve-dir mew-draft-previous-directory mew-copy-dir))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; String
;;;

(defun mew-input-string (prompt subdir default)
  (let ((input (read-string (format prompt subdir default) "")))
    (if (string= input "") default input)))

(defun mew-input-general (prompt alist &optional require-match initial)
  (let* ((completion-ignore-case t)
	 (question (if initial (format "%s (%s) : " prompt initial)
		     (format "(%s) : " prompt)))
	 (value (completing-read question alist nil require-match nil)))
    (if (and initial (string= value "")) initial value)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Type
;;;

(defun mew-input-type (prompt filename default type-list)
  (let ((completion-ignore-case t)
	(type))
    (setq type (completing-read
		(format prompt filename default)
		(mapcar (function (lambda (x) (cons x x))) type-list)
		nil
		t
		""))
    (if (string= type "") default type)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Case
;;;

(defun mew-case-check ()
  (let* ((case (mew-buffer-substring 
		(save-excursion (beginning-of-line) (point))
		(point-max)))
         (lst (mew-split case ?,))
	 err)
    (catch 'nomatch
      (while lst
	(unless (member (car lst) mew-config-cases)
	  (throw 'nomatch (setq err (car lst))))
	(setq lst (cdr lst))))
    (if err
        (progn
          (mew-temp-minibuffer-message (format " [No match: %s]" err))
          nil)
      t)))

(defun mew-input-case (default msg &optional edit)
  (let (case lst ret)
    (unless default (setq default mew-case-default))
    (mew-input-clear)
    (setq mew-input-complete-function (function mew-complete-case))
    (let ((mew-exit-minibuffer-function (function mew-case-check))
          (mew-input-comma-function (function mew-case-check)))
      (if edit
	  (setq case (read-from-minibuffer
		      (format "%s case value: " msg)
		      default
		      mew-input-map))
	(setq case (read-from-minibuffer
		    (format "%s case value (%s): " msg default)
		    ""
		    mew-input-map))))
    (if (string= case "")
	default
      (setq lst (mew-split case ?,))
      (while lst
	(if (member (car lst) mew-config-cases)
	    (setq ret (cons (car lst) ret)))
	(setq lst (cdr lst)))
      (mapconcat (function identity) (nreverse ret) ","))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Mark
;;;

(defun mew-input-mark ()
  (let ((ociea cursor-in-echo-area) char)
    (unwind-protect
	(progn
	  (message "Input mark : ")
	  (setq cursor-in-echo-area t)
	  (setq char (read-char))
	  (unless (char-equal char ?\r)
	    (message "Input mark : %s" (char-to-string char))))
      (setq cursor-in-echo-area ociea))
    (cond
     ((char-equal char ?\r) char)
     ((mew-markdb-by-mark char) char)
     (t (message "Mark %c is not supported" char)
	nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Password
;;;

(defvar mew-passwd-alist nil)
(defvar mew-passwd-timer-id nil)

(defsubst mew-passwd-get-passwd (key)
  (nth 1 (assoc key mew-passwd-alist)))
(defsubst mew-passwd-get-counter (key)
  (nth 2 (assoc key mew-passwd-alist)))

(defun mew-passwd-set-passwd (key val)
  (if (assoc key mew-passwd-alist)
      (setcar (nthcdr 1 (assoc key mew-passwd-alist)) val)
    (setq mew-passwd-alist (cons (list key val 0) mew-passwd-alist))))
(defun mew-passwd-set-counter (key val)
  (if (assoc key mew-passwd-alist)
      (setcar (nthcdr 2 (assoc key mew-passwd-alist)) val)))

(defun mew-passwd-get-keys ()
  (mapcar (function car) mew-passwd-alist))

(defsubst mew-passwd-reset ()
  (setq mew-passwd-alist nil))

(defun mew-passwd-setup ()
  (if mew-passwd-timer-id (cancel-timer mew-passwd-timer-id))
  (setq mew-passwd-timer-id
	(mew-timer (* mew-passwd-timer-unit 60)
		   (function mew-passwd-timer))))

(defun mew-passwd-clean-up ()
  (mew-passwd-reset)
  (if mew-passwd-timer-id (cancel-timer mew-passwd-timer-id))
  (setq mew-passwd-timer-id nil))

(defun mew-passwd-timer ()
  (let ((keys (mew-passwd-get-keys)) key)
    (while keys
      (setq key (car keys))
      (setq keys (cdr keys))
      (if (< (mew-passwd-get-counter key) mew-passwd-lifetime)
	  (mew-passwd-set-counter key (1+ (mew-passwd-get-counter key)))
	;; time out
	(mew-passwd-set-passwd key nil)
	(mew-passwd-set-counter key 0)))))

(defun mew-input-passwd (prompt key)
  (if (and key mew-use-cached-passwd)
      (if (mew-passwd-get-passwd key)
	  (progn
	    (mew-timing)
	    (if mew-passwd-reset-timer (mew-passwd-set-counter key 0))
	    (mew-passwd-get-passwd key))
	(let ((pass (mew-read-passwd prompt)))
	  (mew-passwd-set-passwd key pass)
	  (mew-passwd-set-counter key 0)
	  pass))
    (mew-read-passwd prompt)))

(defun mew-read-passwd (prompt)
  (let ((inhibit-input-event-recording t))
    (condition-case nil
	(read-passwd prompt)
      ;; If read-passwd causes an error, let's return "" so that
      ;; the password process will safely fail.
      (error ""))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Language name
;;;

;; read-language-name should be used. Unfortunately, that of
;; XEmacs 21.1.14 is broken. This function should be obsoleted
;; when Mew quit support of XEmacs 21.1.14

(defun mew-input-language-name (prompt &optional default) 
  (let* ((completion-ignore-case t) 
         (name (completing-read prompt 
                                language-info-alist 
                                nil 
                                t nil nil))) 
    (if (string= name "") (setq name default)) 
    name)) 

(provide 'mew-minibuf)

;;; Copyright Notice:

;; Copyright (C) 1997-2002 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-minibuf.el ends here
