;;; iiimcf.el --- IIIM Client Framework Support on Emacs
;;;                   

;; Copyright (C) 2000 MIYASHITA Hisashi

;; Author: MIYASHITA Hisashi <himi@m17n.org>

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This module supplies client framework on Emacs.
;; It consists of (1) higher level event manager and dispatcher;
;; (2) minimum default event handlers mainly for default responce
;; to the server(s); (3) keyevent translater; (4) IIIM input-context
;; state absorption layer ;and so on.
;; This module use iiimp.el for low level protocol communication
;; with IIIM server sides.

;;; Code:

(require 'iiimp)

;;; version.

(defconst iiimcf-version "0.1 (Hiei)")

;;; error handling.

(defun iiimcf-prevent-error (&rest args)
  nil)

(defun iiimcf-error (&rest args)
  (signal 'iiimcf-error (apply 'format args)))
(put 'iiimcf-error 'error-conditions '(iiimcf-error error))

;;; Emacs event and IIIM keyevent translator.

;; Because IIIMF keyevent is based on Java environment, IIIMECF tries
;; to emulate Java KeyEvent data format.  But, owing to the various
;; problems, this emulation is maybe incomplete.
;;   The KeyEvent class of Java maintains keycode, keychar, and
;; modifiers.  Among these, the meanings of keychar seem to be clear,
;; i.e., it is a character code generated by the event.  The meanings
;; of keycode is, however, ambiguous because it holds a virtual key
;; code that specifies a certain key. Eventually we cannot help assuming
;; a generalized keyboard.  For example, VK_A virtual key code itself does
;; not specify `A' at all.  it specifies a certain key, which probably
;; says `A' on the key top :-), but a Japanese keyboard driver may translate
;; it to a Japanese hiragana character: `chi'.
;;   In the Emacs environment, (event-basic-type) has a weak
;; correspondence to the above virtual key code.  However,
;; there is a serious incompatibility between these.  
;; In the Java environment, virtual key codes from VK_A to VK_Z are
;; mapped into numbers from 65(`A' in ASCII) to 90(`Z' in ASCII).
;; In the Emacs environment, numbers of (event-basic-type) generated
;; from the corresponding key events range from 97(`a' in ASCII)
;; to 122('z' in ASCII).

(defconst iiimcf-event-spec-alist
  '((component-event . 1)
    (container-event . 2)
    (focus-event . 4)
    (key-event . 8)
    (mouse-event . 16)
    (mouse-motion-event . 32)
    (window-event . 64)
    (action-event . 128)
    (adjustment-event . 256)
    (item-event . 512)
    (text-event . 1024)))

(defconst iiimcf-modifier-spec-alist
  '((shift . 1)
    (control . 2)
    (meta . 4)
    (alt . 8)
    (button-1 . 16)
    (button-2 . 8)
    (button-3 . 4)))

;; ((event-symbol keycode [keychar]) ...)
(defconst iiimcf-keycode-spec-alist
  '((component-moved 100 65535)
    (component-resized 101 65535)
    (component-shown 102 65535)
    (component-hidden 103 65535)
    (key-typed 400 65535)
    (key-pressed 401 65535)
    (key-released 402 65535)
    (return 10)
    (13 10)
    (backspace 8)
    (tab 9)
    (cancel 3 65535)
    (clear 12 65535)
    (shift 16 65535)
    (control 17 65535)
    (alt 18)
    (pause .19)
    (capslock 20)
    (escape 27)
    (prior 33 65535)
    (next 34 65535)
    (end 35 65535)
    (home 36 65535)
    (left 37 65535)
    (up 38 65535)
    (right 39 65535)
    (down 40 65535)
    (f1 112 65535)
    (f2 113 65535)
    (f3 114 65535)
    (f4 115 65535)
    (f5 116 65535)
    (f6 117 65535)
    (f7 118 65535)
    (f8 119 65535)
    (f9 120 65535)
    (f10 121 65535)
    (f11 122 65535)
    (f12 123 65535)
    (delete 127)
    (numlock 144 65535)
    (scroll 145 65535)
    (insert 155 65535)
    (help 156 65535)
    (meta 157 65535)
    (convert 28 65535)
    (nonconvert 29 65535)
    (accept 30 65535)
    (modechange 31 65535)
    (kana 21 65535)
    (kanji 25 65535)
    (undefined 0)
    (char-undefined 0)))

(defun iiimcf-translate-iiim-keymodifier (kmod)
  (let ((mod (logior (lsh (car kmod) 16)
		     (cdr kmod)))
	(cand iiimcf-modifier-spec-alist)
	result)
    (while cand
      (if (/= 0 (logand mod (cdr (car cand))))
	  (setq result (cons (car (car cand)) result)))
      (setq cand (cdr cand)))
    result))

(defsubst iiimcf-event-to-char (event)
  (cond ((fboundp (function event-to-character))
	 (setq event (event-to-character event))
	 (if event
	     (char-to-ucs event)))
	((and (numberp event)
	      ;; Practically, normal events don't propagate
	      ;; non-ASCII characters.
	      (>= event 0)
	      (< event 128))
	 event)
	(t nil)))

(defsubst iiimcf-basic-event-type-to-keycode (base)
  (if (and (numberp base)
	   (>= base 97)   ;; corresponds to VK_A
	   (<= base 122)) ;; corresponds to VK_Z
      (- base 32)
    base))

(defun iiimcf-translate-key (event)
  (let ((mods (event-modifiers event))
	(base (or (event-basic-type event)
		  event))
	(kchar (or (iiimcf-event-to-char event)
		   0))
	(kmod 0)
	(ts (current-time))
	kcode slot)
    (while mods
      (setq kmod
	    (logior
	     kmod
	     (cdr (assq
		   (car mods)
		   iiimcf-modifier-spec-alist)))
	    mods (cdr mods)))
    (setq slot (assq base iiimcf-keycode-spec-alist))
    (if slot
	(setq kcode (nth 1 slot)
	      kchar (or (nth 2 slot)
			kcode))
      (setq kcode (iiimcf-basic-event-type-to-keycode base)))
    (if (numberp kcode)
	(list (cons 0 kcode)
	      (cons 0 kchar)
	      (cons 0 kmod)
	      (cons (nth 1 ts) (nth 2 ts))))))

(defun iiimcf-translate-iiim-keyevent (keyevent)
  (let* ((kcode (car keyevent))
	 (kchar (nth 1 keyevent))
	 (kmod (nth 2 keyevent))
	 (mod (logior (lsh (car kmod) 16)
		      (cdr kmod)))
	 (char (logior (lsh (car kchar) 16)
		       (cdr kchar)))
	 key)
    (cond ((/= (car kcode) 0) nil)
	  ((setq key
		 (rassq (cdr kcode)
			iiimcf-keycode-spec-alist))
	   (event-convert-list
	    (nconc (iiimcf-translate-iiim-keymodifier kmod)
		   (list (car key)))))
	  ((= mod 0)
	   (ucs-to-char char))
	  ((/= mod 0)
	   (event-convert-list
	    (nconc (iiimcf-translate-iiim-keymodifier kmod)
		   (list (cdr kcode)))))
	  (t (cdr kcode)))))

;;; Action alist for LWE facilities.
(defconst iiimcf-lwe-action-alist
  '((mode-switch . "goto")
    (convert . "convert")
    (backspace . "backspace")
    (convert-s . "convert-s")
    (unconvert . "unconvert")
    (next . "next")
    (next-s . "next-s")
    (previous . "previous")
    (previous-s . "previous-s")
    (forward . "forward")
    (backword . "backword")
    (move-top . "move-top")
    (move-bottom . "move-bottom")
    (clear . "clear")
    (expand . "expand")
    (expand-s . "expand-s")
    (shrink . "shrink")
    (shrink-s . "shrink-s")
    (expand-noconv . "expand-noconv")
    (expand-noconv-s . "expand-noconv-s")
    (shrink-noconv . "shrink-noconv")
    (shrink-noconv-s . "shrink-noconv-s")
    (fix . "fix")
    ;; Maybe added in the future?
    ))

;;;
;;; Client descriptor.
;;;

(defun iiimcf-make-client-descriptor (appname)
  (let ((arch "*")
	(vendor "*")
	(osname "*"))
    (if (string-match "\\([^\\-]+\\)-\\([^\\-]+\\)-\\(.+\\)"
		      system-configuration)
	(setq arch (match-string 1 system-configuration)
	      vendor (match-string 2 system-configuration)
	      osname (match-string 3 system-configuration)))
    (mapcar
     (lambda (x)
       (if (or (string= (symbol-value x) "*")
	       (string= (symbol-value x) ""))
	   (set x "Unknown")))
     '(arch vendor osname))

  (list
   'client-descriptor
   (list
    (format "%s IIIMECF/%s Emacs/%s"
	    appname iiimcf-version emacs-version)
    osname
    arch
    "Unknown"))))

(defun iiimcf-send-client-descriptor (com-id im-id appname)
  (iiimp-send-message
   'iiimp-im-setimvalues
   com-id im-id
   (list
    (iiimcf-make-client-descriptor appname)))
  (iiimcf-message-manager
   com-id
   (list
    (list 'iiimp-im-setimvalues-reply
	  com-id im-id))))

;;;
;;; Event manager.
;;;
;;   IIIMCF for Emacs provides a simple message dispatcher
;; like generic function, but it is a very fragile version.
;; MESSAGE format is a list like the following.
;;     (<MESSAGE-TYPE> <MESSAGE-DEPENDENT-VALUES>...)
;;

(defvar iiimcf-event-before-handler-alist
  nil)

(defvar iiimcf-event-normal-handler-alist
  nil)

(defvar iiimcf-event-after-handler-alist
  nil)

(defvar iiimcf-event-handler-alist nil)

(defvar iiimcf-pumped-message-list nil)

(defun iiimcf-register-handler
  (function specializer &optional qualifier)
  (let* ((hl (cond ((eq qualifier :before)
		    'iiimcf-event-before-handler-alist)
		   ((eq qualifier :after)
		    'iiimcf-event-after-handler-alist)
		   (t
		    'iiimcf-event-normal-handler-alist)))
	 (slot (cons specializer function))
	 (prob (member slot iiimcf-event-handler-alist)))
    (if (null prob)
	(set hl (cons slot (symbol-value hl))))
    (setq iiimcf-event-handler-alist
	  (append
	   iiimcf-event-before-handler-alist
	   iiimcf-event-normal-handler-alist
	   iiimcf-event-after-handler-alist))
    nil))

(defun iiimcf-remove-handler
  (function specializer &optional qualifier)
  "Remove the specified handler from IIIMCF event manager.
If you specify nil to FUNCTION or SPECIALIZER, remove
all handlers that match the rest."
  (let ((sym (cond ((eq qualifier :before)
		    'iiimcf-event-before-handler-alist)
		   ((eq qualifier :after)
		    'iiimcf-event-after-handler-alist)
		   (t
		    'iiimcf-event-normal-handler-alist)))
	prob slot ll)
    (cond ((and function specializer)
	   (setq slot (cons specializer function))
	   (set sym (delete slot (symbol-value sym))))
	  (function
	   (setq ll (symbol-value sym))
	   (while
	       (prog1
		   (setq slot (rassq function ll))
		 (setq ll (delq slot ll))))
	   (set sym ll))
	  (specializer
	   (setq ll (symbol-value sym))
	   (while
	       (prog1
		   (setq slot (assoc specializer ll))
		 (setq ll (delq slot ll))))
	   (set sym ll))
	  (t
	   (set sym nil)))
    (setq iiimcf-event-handler-alist
	  (append
	   iiimcf-event-before-handler-alist
	   iiimcf-event-normal-handler-alist
	   iiimcf-event-after-handler-alist))
    nil))

;; Should we cache the result for efficiency?
(defsubst iiimcf-message-match-specializer-p
  (obj specializer)
  (while
      (and
       (or (equal (car obj) (car specializer))
	   (eq (car specializer) 'any))
       (setq obj (cdr obj)
	     specializer (cdr specializer))))
  (null specializer))

(defun iiimcf-message-manager (com-id w-specs)
  (let ((iiimcf-pumped-message-list
	 iiimcf-pumped-message-list)
	(exit (null w-specs))
	ws mes ret hl hle func applied newcell)
    (while (setq mes (iiimp-wait-message
		      com-id (not exit)))
      (setq newcell (cons mes nil))
      (if iiimcf-pumped-message-list
	  (setcdr iiimcf-pumped-message-list
		  newcell))
      (setq iiimcf-pumped-message-list newcell)
      (iiimp-add-debug-log (format "Received:%S\n" mes))
      (setq hl iiimcf-event-handler-alist
	    applied nil)
      (while (setq hle (car hl))
	(if (iiimcf-message-match-specializer-p mes (car hle))
	    (progn
	      (setq func (cdr hle))
	      (if (not (memq func applied))
		  (progn
		    (funcall func mes)
		    (setq applied (cons func applied))))))
	(setq hl (cdr hl)))
      (while (and (not exit)
		  (progn
		    (setq ws w-specs
			  mes (car iiimcf-pumped-message-list))
		    (while
			(and ws
			     (if (iiimcf-message-match-specializer-p
				  mes (car ws))
				 (progn
				   (setq exit t
					 ret mes)
				   nil)
			       (setq ws (cdr ws))
			       t)))
		    (and (cdr iiimcf-pumped-message-list)
			 (setq iiimcf-pumped-message-list
			       (cdr iiimcf-pumped-message-list)))))))
    ret))

(defun iiimcf-im-destroyic-reply-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-trigger-notify-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-trigger-notify-handler-default)
 '(iiimp-im-trigger-notify)
 :after)

(defun iiimcf-im-trigger-notify-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-trigger-notify-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-trigger-notify-handler-default)
 '(iiimp-im-trigger-notify)
 :after)

(defun iiimcf-im-setimvalues-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-setimvalues-reply
   (nth 1 mes) (nth 2 mes)))
(iiimcf-register-handler
 (function iiimcf-im-setimvalues-handler-default)
 '(iiimp-im-setimvalues)
 :after)

(defun iiimcf-im-forward-event-handler-default (mes)
  (iiimp-add-debug-log "REPLY:\n")
  (iiimp-send-message
   'iiimp-im-forward-event-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-forward-event-handler-default)
 '(iiimp-im-forward-event)
 :after)

;; OPERATIONS that have been processed are removed from
;; the corresponding MES slot, which is (nth 5 mes).
(defun iiimcf-im-forward-event-with-operations-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-forward-event-with-operations-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes) (nth 5 mes)))
(iiimcf-register-handler
 (function iiimcf-im-forward-event-handler-default)
 '(iiimp-im-forward-event-with-operations)
 :after)

(defun iiimcf-im-preedit-start-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-preedit-start-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-preedit-start-handler-default)
 '(iiimp-im-preedit-start)
 :after)

(defun iiimcf-im-preedit-draw-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-preedit-draw-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-preedit-draw-handler-default)
 '(iiimp-im-preedit-draw)
 :after)

(defun iiimcf-im-preedit-done-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-preedit-done-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-preedit-done-handler-default)
 '(iiimp-im-preedit-done)
 :after)

(defun iiimcf-im-status-start-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-status-start-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-status-start-handler-default)
 '(iiimp-im-status-start)
 :after)

(defun iiimcf-im-status-draw-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-status-draw-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-status-draw-handler-default)
 '(iiimp-im-status-draw)
 :after)

(defun iiimcf-im-status-done-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-status-done-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-status-done-handler-default)
 '(iiimp-im-status-done)
 :after)

;;; LOOKUP default handlers

(defun iiimcf-im-lookup-choice-start-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-lookup-choice-start-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-lookup-choice-start-handler-default)
 '(iiimp-im-lookup-choice-start)
 :after)

(defun iiimcf-im-lookup-choice-draw-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-lookup-choice-draw-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-lookup-choice-draw-handler-default)
 '(iiimp-im-lookup-choice-draw)
 :after)

(defun iiimcf-im-lookup-choice-process-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-lookup-choice-process-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-lookup-choice-process-handler-default)
 '(iiimp-im-lookup-choice-process)
 :after)

(defun iiimcf-im-lookup-choice-done-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-lookup-choice-done-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)))
(iiimcf-register-handler
 (function iiimcf-im-lookup-choice-done-handler-default)
 '(iiimp-im-lookup-choice-done)
 :after)


;;; AUX default handlers

(defun iiimcf-im-aux-start-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-aux-start-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)
   (nth 4 mes) (nth 5 mes)))
(iiimcf-register-handler
 (function iiimcf-im-aux-start-handler-default)
 '(iiimp-im-aux-start)
 :after)

(defun iiimcf-im-aux-draw-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-aux-draw-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)
   (nth 4 mes) (nth 5 mes)))
(iiimcf-register-handler
 (function iiimcf-im-aux-draw-handler-default)
 '(iiimp-im-aux-draw)
 :after)

(defun iiimcf-im-aux-done-handler-default (mes)
  (iiimp-send-message
   'iiimp-im-aux-done-reply
   (nth 1 mes) (nth 2 mes) (nth 3 mes)
   (nth 4 mes) (nth 5 mes)))
(iiimcf-register-handler
 (function iiimcf-im-aux-done-handler-default)
 '(iiimp-im-aux-done)
 :after)

(provide 'iiimcf)

;; iiimcf.el ends here.
