/* Modul %M% Release %R% Level %L% Branch %B% Sequenz %S% Created %E% %U% */

/* Copyright 1994 University of Stuttgart
 * This Graph Widget layouts (hierarchical) directed Graphs
 * This Work is based on the X11R5 tree widget
*/

/*
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE UNIVERSITY OF STUTTGART OR
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.

 * Except as contained in this notice, the name of the University of
 * Stuttgart or the names of the authors shall not be used in
 * advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from the
 * University of Stuttgart and the authors.
*/

/*
 * $XConsortium: Tree.c,v 1.42 91/02/20 20:06:07 converse Exp $
 *
 * Copyright 1990 Massachusetts Institute of Technology
 * Copyright 1989 Prentice Hall
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation.
 * 
 * M.I.T., Prentice Hall and the authors disclaim all warranties with regard
 * to this software, including all implied warranties of merchantability and
 * fitness.  In no event shall M.I.T., Prentice Hall or the authors be liable
 * for any special, indirect or cosequential damages or any damages whatsoever
 * resulting from loss of use, data or profits, whether in an action of
 * contract, negligence or other tortious action, arising out of or in
 * connection with the use or performance of this software.
 * 
 * Authors:  Jim Fulton, MIT X Consortium,
 *           based on a version by Douglas Young, Prentice Hall
 * 
 * This widget is based on the Tree widget described on pages 397-419 of
 * Douglas Young's book "The X Window System, Programming and Applications 
 * with Xt OSF/Motif Edition."  The layout code has been rewritten to use
 * additional blank space to make the structure of the graph easier to see
 * as well as to support vertical trees.
 */

/* Author: Roland Zink
           Universitaet Stuttgart
           IPVR
           Breitwiesenstrasse 20-22
           D 70565 Stuttgart (Germany)
           EMail Roland.Zink@informatik.uni-stuttgart.de
*/

#if defined(SCCSID)
static char sccs_id[] = "%A%";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>
#include "GraphP.h"
#include "Graph_layout.h"

#define VERTICEDATA(v) (GRAPH_CONSTRAINT((v)->w)->graph.verticeData)

#if NeedFunctionPrototypes
typedef int (*CompareFunc_t)(const void *,const void *);
#else
typedef int (*CompareFunc_t)();
#endif


/*------------------------------------------------------------------------
            Check if widget w is a non visible dummy node
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
Boolean IsDummyNode(Vertice_t *v)
#else
Boolean IsDummyNode(v)
Vertice_t *v;
#endif
{ return v->is_dummy == TRUE; }

#if NeedFunctionPrototypes
static void fprintvertice(FILE *fp,Vertice_t *v)
#else
static void fprintvertice(fp,v)
FILE *fp;
Vertice_t *v;
#endif
{ if( IsDummyNode(v) )
    { fprintf(fp,"Dummy   "); }
  else
    { fprintf(fp,"Vertice "); }

  fprintf(fp,"level=%d,elem=%g,x=%d,y=%d,width=%d,height=%d ",v->level,v->elem,v->x,v->y,v->width,v->height);
  fprintf(fp," widget=%p,name=%s,x=%d,y=%d,width=%d,height=%d\n",v->w,v->w->core.name,v->w->core.x,v->w->core.y,v->w->core.width,v->w->core.height);

}

/*------------------------------------------------------------------------
        replace a layoutedge in an edgelist through two others
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void ReplaceInsertLayoutEdgeInEdgeList(LayoutEdgeList_t *list,
                                              LayoutEdge_t *x,
                                              LayoutEdge_t *y,
                                              LayoutEdge_t *z)
#else
static void ReplaceInsertLayoutEdgeInEdgeList(list,x,y,z)
LayoutEdgeList_t *list;
LayoutEdge_t *x;
LayoutEdge_t *y;
LayoutEdge_t *z;
#endif
{ int pos;
  int i;

  pos = FindLayoutEdgeInEdgeList(list,x);
  if( pos < list->n_edges )
    { if( list->n_edges == list->max_edges )
        { list->max_edges += (list->max_edges/2) + 2;
          list->edges = (LayoutEdge_t **)XtRealloc((char *)list->edges,
                     (unsigned int)((list->max_edges)*sizeof(LayoutEdge_t *)));
        }
      for( i=list->n_edges; i>pos; i-- )
	 { list->edges[i] = list->edges[i-1]; }
      list->n_edges++;
      list->edges[pos] = y;
      list->edges[pos+1] = z;
    }
  else
    {
#if defined(__STDC__)
      fprintf(stderr,__FILE__":ReplaceInsertLayoutEdgeInEdgeList():edge not found\n");
#else
      fprintf(stderr,":ReplaceInsertLayoutEdgeInEdgeList():edge not found\n");
#endif
    }
}




/*------------------------------------------------------------------------
        test if a LayoutEdge is short and reversed
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Boolean LayoutEdgeIsShortReversed(LayoutEdge_t *e)
#else
static Boolean LayoutEdgeIsShortReversed(e)
LayoutEdge_t *e;
#endif
{ if( (e->virtual->layouts.n_edges == 1) && 
      (e->edge.from == e->virtual->edge.to) &&
      (e->edge.to   == e->virtual->edge.from) )
    { return True; }
  return False;
}

/*------------------------------------------------------------------------
        reverse the direction of a Edge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void TurnEdge(Edge_t *e)
#else
static void TurnEdge(e)
Edge_t *e;
#endif
{ Vertice_t *x;

  x       = e->from;
  e->from = e->to;
  e->to   = x;
}

/*------------------------------------------------------------------------
        reverse the direction of a LayoutEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void TurnLayoutEdge(LayoutEdge_t *e)
#else
static void TurnLayoutEdge(e)
LayoutEdge_t *e;
#endif
{ TurnEdge(&e->edge); }

/*------------------------------------------------------------------------
        Goes the LayoutEdge to a Managed Widget
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Boolean IsValidOutEdge(Vertice_t *v,LayoutEdge_t *e)
#else
static Boolean IsValidOutEdge(v,e)
Vertice_t *v;
LayoutEdge_t *e;
#endif
{ if( (e->edge.from == v) && (e->edge.from != e->edge.to) && 
      (XtIsManaged(e->edge.to->w) || IsDummyNode(e->edge.to)) )
    { return True; }
  return False;
}

/*------------------------------------------------------------------------
        Comes the LayoutEdge from a Managed Widget
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Boolean IsValidInEdge(Vertice_t *x,LayoutEdge_t *e)
#else
static Boolean IsValidInEdge(x,e)
Vertice_t *x;
LayoutEdge_t *e;
#endif
{ if( (e->edge.to == x) && (e->edge.from != e->edge.to) && 
      (XtIsManaged(e->edge.from->w) || IsDummyNode(e->edge.from)) )
    { return True; }
  return False;
}



/*------------------------------------------------------------------------
        insert a vertice in a LayoutEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void SplitLayoutEdge(LayoutEdge_t *e, Vertice_t *new)
#else
static void SplitLayoutEdge(e,new)
LayoutEdge_t *e;
Vertice_t *new;
#endif
{ VirtualEdge_t *virtual;
  LayoutEdgeList_t *layouts;
  LayoutEdge_t *x;
  LayoutEdge_t *y;
  Edge_t edge;

  virtual = e->virtual;

  edge.from = e->edge.from;
  edge.to   = new;
  x = CreateLayoutEdge(edge,virtual);
  edge.from = new;
  edge.to   = e->edge.to;
  y = CreateLayoutEdge(edge,virtual);

  layouts = &virtual->layouts;
  ReplaceInsertLayoutEdgeInEdgeList(layouts,e,x,y);

  layouts = &e->edge.from->edges;
  ReplaceLayoutEdgeInEdgeList(layouts,e,x);
  layouts = &e->edge.to->edges;
  ReplaceLayoutEdgeInEdgeList(layouts,e,y);
  InsertLayoutEdgeInEdgeList(&new->edges,x);
  InsertLayoutEdgeInEdgeList(&new->edges,y);

  DestroyLayoutEdge(e);
}

/*------------------------------------------------------------------------
        remove a vertice in a long edge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void JoinLayoutEdges(LayoutEdge_t *e1, LayoutEdge_t *e2)
#else
static void JoinLayoutEdges(e1,e2)
LayoutEdge_t *e1;
LayoutEdge_t *e2;
#endif
{ VirtualEdge_t *virtual;
  Edge_t edge;
  LayoutEdge_t *e;

  if( (e1->virtual != e2->virtual) ||
      (e1->edge.to != e2->edge.from) )
    {
#if defined(__STDC__)
      fprintf(stderr,__FILE__":JoinLayoutEdges():inconsistent edges!\n");
#else
      fprintf(stderr,":JoinLayoutEdges():inconsistent edges!\n");
#endif
      return;
    }
  virtual = e1->virtual;
  edge.from = e1->edge.from;
  edge.to   = e2->edge.to;
  e = CreateLayoutEdge(edge,virtual);

  ReplaceLayoutEdgeInEdgeList(&e1->edge.from->edges,e1,e);
  ReplaceLayoutEdgeInEdgeList(&e2->edge.to->edges,e2,e);
  DeleteLayoutEdgeInEdgeList(&e1->edge.to->edges,e1);
  DeleteLayoutEdgeInEdgeList(&e2->edge.from->edges,e2);

  DeleteLayoutEdgeInEdgeList(&virtual->layouts,e2);
  ReplaceLayoutEdgeInEdgeList(&virtual->layouts,e1,e);

  DestroyLayoutEdge(e1);
  DestroyLayoutEdge(e2);
}




/*------------------------------------------------------------------------
           Create a widget as dummy node/
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Vertice_t *make_dummy_node(LayoutData_t *graph)
#else
static Vertice_t *make_dummy_node(graph)
LayoutData_t *graph;
#endif
{ Widget w; GraphConstraints gc;
/*
  printf("%p\n",graph->w);
*/
  w = XtVaCreateWidget ("dummy", widgetClass, (Widget)graph->w,
                        XtNwidth, 1, XtNheight, 1, XtNborderWidth, 0, NULL);
  gc=GRAPH_CONSTRAINT(w);
  gc->graph.vertice.is_dummy = TRUE;
  return &gc->graph.vertice;
}

#if NeedFunctionPrototypes
static Vertice_t *pop_dummy(LayoutData_t *graph)
#else
static Vertice_t *pop_dummy(graph)
LayoutData_t *graph;
#endif
{ Vertice_t *d;

  if( graph->n_dummies > 0 )
    { graph->n_dummies--;
      d=graph->dummies[graph->n_dummies];
    }
  else
    { d=make_dummy_node(graph); }

  return d;
}

#if NeedFunctionPrototypes
static void push_dummy(LayoutData_t *graph,Vertice_t *d)
#else
static void push_dummy(graph,d)
LayoutData_t *graph;
Vertice_t *d;
#endif
{ int dummy;

  dummy = graph->n_dummies;
  graph->n_dummies++;
  if( graph->n_dummies > graph->max_dummies ) /* enough memory */
    { graph->max_dummies += (graph->max_dummies/2) + 2;
      graph->dummies = (Vertice_t **)XtRealloc( (XtPointer)graph->dummies,
                     (unsigned int)((graph->max_dummies)*sizeof(Vertice_t *)));
    }
  graph->dummies[dummy] = d;
}














#if NeedFunctionPrototypes
static void check_size_and_position(GraphWidget graphw)
#else
static void check_size_and_position(graphw)
GraphWidget graphw;
#endif
{ int i; Widget child; GraphConstraints gc;

  for(i=0; i<graphw->composite.num_children; i++ )
     { child = graphw->composite.children[i];
       gc = GRAPH_CONSTRAINT(child);
       if( XtIsManaged(child) )
         { if( (child->core.x != gc->graph.vertice.x) ||
               (child->core.y != gc->graph.vertice.y) ||
               ((child->core.width+2*child->core.border_width) !=
                 gc->graph.vertice.width) ||
               ((child->core.height+2*child->core.border_width) !=
                 gc->graph.vertice.height))
             {
/*
               fprintf(stdout,"Warning: Widget and internal information unsynchronized\n");
               fprintvertice(stdout,&gc->graph.vertice);
*/
               gc->graph.vertice.x = child->core.x;
               gc->graph.vertice.y = child->core.y;
               gc->graph.vertice.width = child->core.width + 
                                         2*child->core.border_width;
               gc->graph.vertice.height = child->core.height + 
                                          2*child->core.border_width;
             }
         }
     }
}








#if NeedFunctionPrototypes
static void make_dag1(Vertice_t *v)
#else
static void make_dag1(v)
Vertice_t *v;
#endif
{ int i; Vertice_t *to;

  v->in_current_search_path = TRUE;
  for(i=0; i<v->edges.n_edges; i++ )
     { if( IsValidOutEdge(v,v->edges.edges[i]) )
         { to = v->edges.edges[i]->edge.to;
           if( !to->is_dag )                           /* cycle detected ? */
             { if( to->in_current_search_path )        /* yes turn edge */
                 { TurnLayoutEdge(v->edges.edges[i]); }
               else                                /* no depth-first-search */
                 { make_dag1(to); }
             }
         }
     }
  v->in_current_search_path = FALSE;  /* we back out of this node */
  v->is_dag = TRUE;            /* the subgraph under this node is now a dag */
}


#if NeedFunctionPrototypes
static void make_dag(GraphWidget graphw)
#else
static void make_dag(graphw)
GraphWidget graphw;
#endif
{ int i; Widget child; GraphConstraints gc;

  for(i=0; i<graphw->composite.num_children; i++ )
     { child = graphw->composite.children[i];
       gc = GRAPH_CONSTRAINT(child);
       if( XtIsManaged(child) && !gc->graph.vertice.is_dag )
         { make_dag1(&gc->graph.vertice); }
     }
}

/* level und elem je wieder auf -1 setzen um make_levels laufen lassen zu koennen */
/* Dummy nodes wieder freigeben oder wiederverwenden */
#if NeedFunctionPrototypes
static void clear_level(LayoutData_t *graph,Level_t *l)
#else
static void clear_level(graph,l)
LayoutData_t *graph;
Level_t *l;
#endif
{ int i,j;
  Vertice_t *v;
  LayoutEdge_t *e1;
  LayoutEdge_t *e2;

  for( i=0; i<l->n_elems; i++)
    { v=l->sequence[i];
      v->is_dag = FALSE;
      v->level = -1;
      v->elem  = -1;
      if( v->is_dummy == TRUE )  /* dummy node ? */
        { if( v->edges.n_edges == 2 )
            { e1 = v->edges.edges[0];
              e2 = v->edges.edges[1];
              JoinLayoutEdges(e1,e2);
            }
          if( (v->edges.n_edges == 0) || (v->edges.n_edges == 2) )
            { push_dummy(graph,v); }
          else
            {
#if defined(__STDC__)
          fprintf(stderr,__FILE__":clear_level(): inconsistent dummy found\n");
#else
          fprintf(stderr,":clear_level(): inconsistent dummy found\n");
#endif
            }
        }
      else
        { for( j=0; j<v->edges.n_edges; j++ )
	     { if( LayoutEdgeIsShortReversed(v->edges.edges[j]) )
                 { TurnLayoutEdge(v->edges.edges[j]); }
             }
         }
    }
}

#if NeedFunctionPrototypes
static void clear_levels(LayoutData_t *graph)
#else
static void clear_levels(graph)
LayoutData_t *graph;
#endif
{ int i;

  for( i=0; i<graph->n_levels; i++ )
     { clear_level(graph,&graph->levels[i]); }

  graph->n_levels = 0;
}

#if NeedFunctionPrototypes
static void add_level(LayoutData_t *graph)
#else
static void add_level(graph)
LayoutData_t *graph;
#endif
{ int level; int i;

  level = graph->n_levels;
  graph->n_levels++;
  if( graph->n_levels > graph->max_levels ) /* enough memory ? */
    { graph->max_levels += (graph->max_levels / 2) + 2; /* alloc memory */
      graph->levels = (Level_t *)XtRealloc( (XtPointer)graph->levels,
                   (unsigned int)((graph->max_levels) * sizeof(Level_t)));
      for( i=level; i<graph->max_levels; i++ )
         { graph->levels[i].sequence = NULL;      /* yet no memory */
           graph->levels[i].max_elems = 0;        /* for subarrarys */
         }
    }
  graph->levels[level].n_elems = 0;
}

/* insert vertice in a level */
#if NeedFunctionPrototypes
static void add_vertice_to_level(Level_t *l,Vertice_t *v)
#else
static void add_vertice_to_level(l,v)
Level_t *l;
Vertice_t *v;
#endif
{ int elem;

  elem = l->n_elems;
  l->n_elems++;
  if( l->n_elems > l->max_elems ) /* enough memory */
    { l->max_elems += (l->max_elems/2) + 2; /* no: alloc memory */
      l->sequence = (Vertice_t **)XtRealloc( (XtPointer)l->sequence,
                      (unsigned int)((l->max_elems)*sizeof(Vertice_t *)));
    }
  l->sequence[elem] = v;
  v->elem = elem;
/*
  fprintvertice(stdout,v);
*/
}

/* (level may change later) */
#if NeedFunctionPrototypes
static void set_level(Vertice_t *v,int level)
#else
static void set_level(v,level)
Vertice_t *v;
int level;
#endif
{ int i;
  Vertice_t *to;

  if( v->level >= level )
    { return; }
  else
    { v->level = level;
      for( i=0; i<v->edges.n_edges; i++ )
         { if( IsValidOutEdge(v,v->edges.edges[i]) )
             { to = v->edges.edges[i]->edge.to;
               set_level(to,level+1);
             }
         }
    }
}

#if NeedFunctionPrototypes
static Boolean IsVerticeWithoutAncestors(Vertice_t *x,Vertice_t *title)
#else
static Boolean IsVerticeWithoutAncestors(x,title)
Vertice_t *x;
Vertice_t *title;
#endif
{ int i;

  if( x->is_dummy == TRUE )
    { return False; }
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidInEdge(x,x->edges.edges[i]) && 
           (x->edges.edges[i]->edge.from != title) )
         { return False; }
     }
  return True;
}

#if NeedFunctionPrototypes
static void make_first_level(GraphWidget w)
#else
static void make_first_level(w)
GraphWidget w;
#endif
{ int i; Widget child;
  Vertice_t *title;

  add_level(&w->graph.layout);
  if( w->composite.num_children > 0 )
    { child = w->composite.children[0];
      title = &GRAPH_CONSTRAINT(child)->graph.vertice;
      if( XtIsManaged(child) )
        { set_level(title,0);
          add_vertice_to_level(&w->graph.layout.levels[0],title);
        }
    }
  /* search nodes with no ancestors and insert these nodes to level 1 */
  add_level(&w->graph.layout);
  for(i=1; i<w->composite.num_children; i++ )
     { child = w->composite.children[i];

       if( XtIsManaged(child) && 
           IsVerticeWithoutAncestors(&GRAPH_CONSTRAINT(child)->graph.vertice,title) )
         { 
           set_level(&GRAPH_CONSTRAINT(child)->graph.vertice,1);
           add_vertice_to_level(&w->graph.layout.levels[1],
                                &GRAPH_CONSTRAINT(child)->graph.vertice);
         }
     }
}

#if NeedFunctionPrototypes
static void make_level1(LayoutData_t *graph,int level,Vertice_t *x)
#else
static void make_level1(graph,level,x)
LayoutData_t *graph;
int level;
Vertice_t *x;
#endif
{ int i;
  Vertice_t *to;
  Vertice_t *dummy;

  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidOutEdge(x,x->edges.edges[i]) )
         { to = x->edges.edges[i]->edge.to;
           if(to->level==level)   /* OK */
             { if( to->elem < 0 )
                 { set_level(to,level);
                   add_vertice_to_level(&graph->levels[level],to); }
             }
           else 
             { dummy=pop_dummy(graph);
               dummy->level = level;
               add_vertice_to_level(&graph->levels[level],dummy);
               SplitLayoutEdge(x->edges.edges[i],dummy);
             }
         }
     }
}

#if NeedFunctionPrototypes
static void make_level(LayoutData_t *graph,int level)
#else
static void make_level(graph,level)
LayoutData_t *graph;
int level;
#endif
{ int i; Level_t *l; Vertice_t *x; int n_elems;

  if( (level<2) /*|| (level!=w->graph.n_levels)*/ ) { return; }

  add_level(graph);
  l = &graph->levels[level-1];
  n_elems = l->n_elems;
  for(i=0;i<n_elems;i++)
    { x=l->sequence[i];
      make_level1(graph,level,x);
    }
}

#if NeedFunctionPrototypes
static void make_levels(GraphWidget w)
#else
static void make_levels(w)
GraphWidget w;
#endif
{ int i; Level_t *l;

  make_first_level(w);
  l = &w->graph.layout.levels[1];
  for( i=2; l->n_elems>0; i++ )
     { make_level(&w->graph.layout,i);
       l = &w->graph.layout.levels[i];
     }
  w->graph.layout.n_levels--;
}

#if NeedFunctionPrototypes
static double get_downBarycenter(LayoutData_t *graph,Vertice_t *x)
#else
static double get_downBarycenter(graph,x)
LayoutData_t *graph;
Vertice_t *x;
#endif
{ int i;
  Vertice_t *to;
  double barycenter;
  int num;

  barycenter=0; num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidOutEdge(x,x->edges.edges[i]) )
         { to = x->edges.edges[i]->edge.to;
           barycenter += to->elem;
           num++;
         }
     }
  if( num > 0 )
    { barycenter= barycenter / num; }

  return barycenter;
}

#if NeedFunctionPrototypes
static double get_upBarycenter(LayoutData_t *graph,Vertice_t *x)
#else
static double get_upBarycenter(graph,x)
LayoutData_t *graph;
Vertice_t *x;
#endif
{ int i;
  Vertice_t *from;
  double barycenter;
  int num;

  barycenter=0; num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidInEdge(x,x->edges.edges[i]) )
         { from = x->edges.edges[i]->edge.from;
           barycenter += from->elem;
           num++;
         }
     }
  if( num > 0 )
    { barycenter= barycenter / num; }

  return barycenter;
}

#if NeedFunctionPrototypes
static void apply_to_level(LayoutData_t *graph,int level,Level_t *l,
             void (*fun)(LayoutData_t *graph,int level,int elem,Vertice_t *x))
#else
static void apply_to_level(graph,level,l,fun)
LayoutData_t *graph;
int level;
Level_t *l;
void (*fun)();
#endif
{ int i;

  for( i=0; i<l->n_elems; i++ )
     { fun(graph,level,i,l->sequence[i]); }
}

#if NeedFunctionPrototypes
static void apply_to_levels(LayoutData_t *graph,
                        void (*fun)(LayoutData_t *graph,int level,Level_t *l))
#else
static void apply_to_levels(graph,fun)
LayoutData_t *graph;
void (*fun)();
#endif
{ int i; Level_t *l;

  for( i=0; i<graph->n_levels; i++)
     { l = &graph->levels[i];
       fun(graph,i,l);
     }
}

#if NeedFunctionPrototypes
static void apply_to_levels_rev(LayoutData_t *graph,
                        void (*fun)(LayoutData_t *graph,int level,Level_t *l))
#else
static void apply_to_levels_rev(graph,fun)
LayoutData_t *graph;
void (*fun)();
#endif
{ int i; Level_t *l;

  for( i=graph->n_levels-1; i>=0; i--)
     { l = &graph->levels[i];
       fun(graph,i,l);
     }
}

#if NeedFunctionPrototypes
static void set_upBarycenter(LayoutData_t *graph,int level,int elem,
                             Vertice_t *x)
#else
static void set_upBarycenter(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ x->elem = get_upBarycenter(graph,x); }

#if NeedFunctionPrototypes
static void set_downBarycenter(LayoutData_t *graph,int level,int elem,
                             Vertice_t *x)
#else
static void set_downBarycenter(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ x->elem = get_downBarycenter(graph,x); }

#if NeedFunctionPrototypes
static void set_updownBarycenter(LayoutData_t *graph,int level,int elem,
                             Vertice_t *x)
#else
static void set_updownBarycenter(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ x->elem = (get_upBarycenter(graph,x) + get_downBarycenter(graph,x))/2; }

#if NeedFunctionPrototypes
static void set_elem_to_elem(LayoutData_t *graph,int level,int elem,
                             Vertice_t *x)
#else
static void set_elem_to_elem(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ x->elem = elem; }

#if NeedFunctionPrototypes
static int compare_vertices_byBarycenter(Vertice_t **x,Vertice_t **y)
#else
static int compare_vertices_byBarycenter(x,y)
Vertice_t **x;
Vertice_t **y;
#endif
{ if( ((*x)->elem) > ((*y)->elem) )
    { return 1; }
  else if( ((*x)->elem) == ((*y)->elem) )
    { return 0; }
  else
    { return -1; }
}

#if NeedFunctionPrototypes
static void sort_level_by_Barycenter(LayoutData_t *graph,int level,Level_t *l)
#else
static void sort_level_by_Barycenter(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ qsort(l->sequence,l->n_elems,sizeof(Vertice_t *),
        (CompareFunc_t)compare_vertices_byBarycenter);
}

#if NeedFunctionPrototypes
static void set_upBarycenter_level(LayoutData_t *graph,int level,Level_t *l)
#else
static void set_upBarycenter_level(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_upBarycenter);
  sort_level_by_Barycenter(graph,level,l);
  apply_to_level(graph,level,l,set_elem_to_elem);
}

#if NeedFunctionPrototypes
static void set_downBarycenter_level(LayoutData_t *graph,int level,Level_t *l)
#else
static void set_downBarycenter_level(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_downBarycenter);
  sort_level_by_Barycenter(graph,level,l);
  apply_to_level(graph,level,l,set_elem_to_elem);
}

#if NeedFunctionPrototypes
static void set_updownBarycenter_level(LayoutData_t *graph,int level,Level_t *l)
#else
static void set_updownBarycenter_level(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_updownBarycenter);
  sort_level_by_Barycenter(graph,level,l);
  apply_to_level(graph,level,l,set_elem_to_elem);
}

#if NeedFunctionPrototypes
static void set_upBarycenter_levels(LayoutData_t *graph)
#else
static void set_upBarycenter_levels(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels_rev(graph,set_upBarycenter_level); }

#if NeedFunctionPrototypes
static void set_downBarycenter_levels(LayoutData_t *graph)
#else
static void set_downBarycenter_levels(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels(graph,set_downBarycenter_level); }

#if NeedFunctionPrototypes
static void set_updownBarycenter_levels_up(LayoutData_t *graph)
#else
static void set_updownBarycenter_levels_up(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels_rev(graph,set_updownBarycenter_level); }

#if NeedFunctionPrototypes
static void set_updownBarycenter_levels_down(LayoutData_t *graph)
#else
static void set_updownBarycenter_levels_down(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels(graph,set_updownBarycenter_level); }

#if NeedFunctionPrototypes
static void minimize_edge_crossings(LayoutData_t *graph)
#else
static void minimize_edge_crossings(graph)
LayoutData_t *graph;
#endif
{ int i;

  set_downBarycenter_levels(graph);
  set_upBarycenter_levels(graph);
  for(i=0; i<4; i++ )
    { set_updownBarycenter_levels_down(graph);
      set_updownBarycenter_levels_up(graph);
    }
}

#if NeedFunctionPrototypes
static int NumValidOutEdges(Vertice_t *x)
#else
static int NumValidOutEdges(x)
Vertice_t *x;
#endif
{ int i;
  int num;

  num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidOutEdge(x,x->edges.edges[i]) )
         { num++;
         }
     }
  return num;
}

#if NeedFunctionPrototypes
static int NumValidInEdges(Vertice_t *x)
#else
static int NumValidInEdges(x)
Vertice_t *x;
#endif
{ int i;
  int num;

  num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidInEdge(x,x->edges.edges[i]) )
         { num++;
         }
     }
  return num;
}

#if NeedFunctionPrototypes
static int getCenterPosDown(LayoutData_t *graph,Vertice_t *x)
#else
static int getCenterPosDown(graph,x)
LayoutData_t *graph;
Vertice_t *x;
#endif
{ int i;
  Vertice_t *from;
  int center_pos;
  int num;

  center_pos=0; num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidOutEdge(x,x->edges.edges[i]) )
         { from = x->edges.edges[i]->edge.to;

           switch( graph->w->graph.gravity )
           { case SouthGravity:
             case NorthGravity: center_pos += from->x + from->width/2;
                                break;
             case WestGravity:
             case EastGravity: center_pos += from->y + from->height/2;
                           break;
	   }
           num++;
         }
     }
/*
  center_pos = center_pos + (x->x + x->width/2);
  num++;
*/
  if( num == 0 )
    { switch( graph->w->graph.gravity )
      { case SouthGravity:
        case NorthGravity: center_pos = x->x+x->width/2; break;
	case EastGravity:
        case WestGravity:  center_pos = x->y+x->height/2; break;
      }
    }
  else if( num == 1 )
    { if( NumValidInEdges(from) > 1 )
        { switch( graph->w->graph.gravity )
          { case SouthGravity:
            case NorthGravity: center_pos = x->x+x->width/2; break;
	    case EastGravity:
            case WestGravity:  center_pos = x->y+x->height/2; break;
          }
        }
    }
  else
    { center_pos = (center_pos/num); }

  return center_pos;
}

#if NeedFunctionPrototypes
static int getCenterPosUp(LayoutData_t *graph,Vertice_t *x)
#else
static int getCenterPosUp(graph,x)
LayoutData_t *graph;
Vertice_t *x;
#endif
{ int i;
  Vertice_t *from;
  int center_pos;
  int num;
/*
  if( IsDummyNode(x) )
    { x = x; }
*/
  center_pos=0; num=0;
  for( i=0; i<x->edges.n_edges; i++ )
     { if( IsValidInEdge(x,x->edges.edges[i]) )
         { from = x->edges.edges[i]->edge.from;

           switch( graph->w->graph.gravity )
           { case SouthGravity:
             case NorthGravity: center_pos += from->x + from->width/2;
                                break;
             case WestGravity:
             case EastGravity: center_pos += from->y + from->height/2;
                               break;
	   }

           num++;
         }
     }
/*
  center_pos = center_pos + (x->x + x->width/2);
  num++;
*/
  if( num == 0 )
    { switch( graph->w->graph.gravity )
      { case SouthGravity:
        case NorthGravity: center_pos = x->x+x->width/2; break;
	case EastGravity:
        case WestGravity:  center_pos = x->y+x->height/2; break;
      }
    }
  else if( num == 1 )
    { if( NumValidOutEdges(from) > 1 )
        { switch( graph->w->graph.gravity )
          { case SouthGravity:
            case NorthGravity: center_pos = x->x+x->width/2; break;
	    case EastGravity:
            case WestGravity:  center_pos = x->y+x->height/2; break;
          }
        }
    }
  else
    { center_pos = (center_pos/num); }

  return center_pos;
}

#if NeedFunctionPrototypes
static void set_position(LayoutData_t *graph,int level,int elem,Vertice_t *x)
#else
static void set_position(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ if( (x->w->core.x != x->x) || (x->w->core.y != x->y) )
    { XtMoveWidget(x->w,x->x,x->y); }
/*  fprintvertice(stdout,x); */
}

#if NeedFunctionPrototypes
static void set_pos_pr_up(LayoutData_t *graph,int level,int elem,Vertice_t *x)
#else
static void set_pos_pr_up(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ int center_pos;
  int previous_pos;

  center_pos = getCenterPosDown(graph,x);
  switch( graph->w->graph.gravity )
  { case SouthGravity:
    case NorthGravity: center_pos = center_pos - x->width/2; break;
    case EastGravity:
    case WestGravity:  center_pos = center_pos - x->height/2; break;
  }

  if( elem == 0 )
    { previous_pos = -1000000; }
  else
    { switch( graph->w->graph.gravity )
      { case SouthGravity:
        case NorthGravity:
             previous_pos = graph->levels[level].sequence[elem-1]->x +
                            graph->levels[level].sequence[elem-1]->width +
                            graph->w->graph.hpad;
             break;
        case EastGravity:
        case WestGravity:
             previous_pos = graph->levels[level].sequence[elem-1]->y +
                            graph->levels[level].sequence[elem-1]->height+
                            graph->w->graph.vpad;
             break;
      }
    }
  if( center_pos < previous_pos )
    { switch( graph->w->graph.gravity )
      { case SouthGravity: case NorthGravity: x->x = previous_pos; break;
        case EastGravity:  case WestGravity:  x->y = previous_pos; break;
      }
    }
  else
    { switch( graph->w->graph.gravity )
      { case SouthGravity: case NorthGravity: x->x = center_pos; break;
        case EastGravity:  case WestGravity:  x->y = center_pos; break;
      }
    }
/*  fprintvertice(stdout,x); */
}

#if NeedFunctionPrototypes
static void set_pos_pr_down(LayoutData_t *graph,int level,int elem,Vertice_t *x)
#else
static void set_pos_pr_down(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ int center_pos;
  int previous_pos;

  center_pos = getCenterPosUp(graph,x);
  switch( graph->w->graph.gravity )
  { case SouthGravity:
    case NorthGravity: center_pos = center_pos - x->width/2; break;
    case EastGravity:
    case WestGravity:  center_pos = center_pos - x->height/2; break;
  }
  if( elem == 0 )
    { previous_pos = -1000000; }
  else
    { switch( graph->w->graph.gravity )
      { case SouthGravity:
        case NorthGravity:
             previous_pos = graph->levels[level].sequence[elem-1]->x +
                            graph->levels[level].sequence[elem-1]->width +
                            graph->w->graph.hpad;
             break;
        case EastGravity:
        case WestGravity:
             previous_pos = graph->levels[level].sequence[elem-1]->y +
                            graph->levels[level].sequence[elem-1]->height+
                            graph->w->graph.vpad;
             break;
      }
    }
  if( center_pos < previous_pos )
    { switch( graph->w->graph.gravity )
      { case SouthGravity: case NorthGravity: x->x = previous_pos; break;
        case EastGravity:  case WestGravity:  x->y = previous_pos; break;
      }
    }
  else
    { switch( graph->w->graph.gravity )
      { case SouthGravity: case NorthGravity: x->x = center_pos; break;
        case EastGravity:  case WestGravity:  x->y = center_pos; break;
      }
    }
/*  fprintvertice(stdout,x); */
}

#if NeedFunctionPrototypes
static void pr_correct_position(LayoutData_t *graph,int level,int elem,Vertice_t *x)
#else
static void pr_correct_position(graph,level,elem,x)
LayoutData_t *graph;
int level;
int elem;
Vertice_t *x;
#endif
{ switch( graph->w->graph.gravity )
  { case SouthGravity: case NorthGravity: x->x = x->x-graph->left; break;
    case EastGravity:  case WestGravity:  x->y = x->y-graph->left; break;
  }
}

#if NeedFunctionPrototypes
static void pr_layout_level_up(LayoutData_t *graph,int level,Level_t *l)
#else
static void pr_layout_level_up(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_pos_pr_up);
  switch( graph->w->graph.gravity )
  { case SouthGravity:
    case NorthGravity: l->maxwidth = l->sequence[l->n_elems-1]->x + 
                       l->sequence[l->n_elems-1]->width +
                       graph->w->graph.hpad/2;
                       break;
    case EastGravity:
    case WestGravity:  l->maxheight = l->sequence[l->n_elems-1]->y + 
                       l->sequence[l->n_elems-1]->height +
                       graph->w->graph.vpad/2;
                       break;
  }
}

#if NeedFunctionPrototypes
static void pr_layout_level_down(LayoutData_t *graph,int level,Level_t *l)
#else
static void pr_layout_level_down(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_pos_pr_down);
  switch( graph->w->graph.gravity )
  { case SouthGravity:
    case NorthGravity: l->maxwidth = l->sequence[l->n_elems-1]->x + 
                       l->sequence[l->n_elems-1]->width +
                       graph->w->graph.hpad/2;
                       break;
    case EastGravity:
    case WestGravity:  l->maxheight = l->sequence[l->n_elems-1]->y + 
                       l->sequence[l->n_elems-1]->height +
                       graph->w->graph.vpad/2;
                       break;
  }
}

#if NeedFunctionPrototypes
static void set_position_level(LayoutData_t *graph,int level,Level_t *l)
#else
static void set_position_level(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ apply_to_level(graph,level,l,set_position); }

#if NeedFunctionPrototypes
static void pr_correct_positions_level(LayoutData_t *graph,int level,Level_t *l)
#else
static void pr_correct_positions_level(graph,level,l)
LayoutData_t *graph;
int level;
Level_t *l;
#endif
{ if( level != 0 )
    { apply_to_level(graph,level,l,pr_correct_position); }
  switch( graph->w->graph.gravity )
  { case SouthGravity:
    case NorthGravity: l->maxwidth = l->sequence[l->n_elems-1]->x + 
                       l->sequence[l->n_elems-1]->width +
                       graph->w->graph.hpad/2;
                       break;
    case EastGravity:
    case WestGravity:  l->maxheight = l->sequence[l->n_elems-1]->y + 
                       l->sequence[l->n_elems-1]->height +
                       graph->w->graph.vpad/2;
                       break;
  }
}

#if NeedFunctionPrototypes
static void pr_layout_levels_up(LayoutData_t *graph)
#else
static void pr_layout_levels_up(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels_rev(graph,pr_layout_level_up); }

#if NeedFunctionPrototypes
static void pr_layout_levels_down(LayoutData_t *graph)
#else
static void pr_layout_levels_down(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels(graph,pr_layout_level_down); }

#if NeedFunctionPrototypes
static void pr_correct_positions_levels(LayoutData_t *graph)
#else
static void pr_correct_positions_levels(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels(graph,pr_correct_positions_level); }

#if NeedFunctionPrototypes
static void set_position_levels(LayoutData_t *graph)
#else
static void set_position_levels(graph)
LayoutData_t *graph;
#endif
{ apply_to_levels(graph,set_position_level); }

#if NeedFunctionPrototypes
static void pr_layout_graph(LayoutData_t *graph)
#else
static void pr_layout_graph(graph)
LayoutData_t *graph;
#endif
{ int i; int j; Level_t *l; int left;

  for( i=0; i<2; i++ )
     { pr_layout_levels_down(graph);
       pr_layout_levels_up(graph);

       left = 1000000;
       for( j=1; j<graph->n_levels; j++)
          { l = &graph->levels[j];
            if( l->n_elems > 0 )
              { switch( graph->w->graph.gravity )
                { case SouthGravity:
                  case NorthGravity: if( l->sequence[0]->x < left )
                                       { left = l->sequence[0]->x; }
                                     break;
                  case EastGravity:
                  case WestGravity:  if( l->sequence[0]->y < left )
                                       { left = l->sequence[0]->y; }
                                     break;
		}
              }
          }
       switch( graph->w->graph.gravity )
       { case SouthGravity:
         case NorthGravity: left=left - graph->w->graph.hpad/2; break;
         case EastGravity:
         case WestGravity:  left=left - graph->w->graph.vpad/2; break;
       }
       graph->left = left;
       pr_correct_positions_levels(graph);

     }
}

#if NeedFunctionPrototypes
static void maxdim_level(LayoutData_t *graph,int level)
#else
static void maxdim_level(graph,level)
LayoutData_t *graph;
int level;
#endif
{ int i; Level_t *l;
  Vertice_t *v;
  Dimension width; Dimension height;
  Bool horiz = IsHorizontal(graph->w);

  width = 0;
  height = 0;
  l = &graph->levels[level];
  for( i=0; i<l->n_elems; i++ )
     { v=l->sequence[i];
       if( horiz )
         { height += v->height;
           height += graph->w->graph.vpad;
           if( width < (Dimension)v->width ) { width = v->width; }
         }
       else
         { width += v->width;
           width += graph->w->graph.hpad;
           if( height < (Dimension)v->height ) { height = v->height; }
         }
     }
  l->maxwidth = width;
  l->maxheight = height;
}

#if NeedFunctionPrototypes
static void maxdim(LayoutData_t *graph)
#else
static void maxdim(graph)
LayoutData_t *graph;
#endif
{ Dimension maxwidth; Dimension maxheight; int i; Level_t *l;
  Bool horiz = IsHorizontal(graph->w);

  maxwidth  = 0;
  maxheight = 0;
  for(i=0;i<graph->n_levels;i++)
     { l = &graph->levels[i];
       maxdim_level(graph,i);
       if( horiz )
         { if( maxheight < l->maxheight )
             { maxheight = l->maxheight; }
           maxwidth += l->maxwidth;
           maxwidth += graph->w->graph.hpad;
         }
       else
         { if( maxwidth < l->maxwidth )
             { maxwidth = l->maxwidth; }
           maxheight += l->maxheight;
           maxheight += graph->w->graph.vpad;
         }
     }
  graph->maxwidth = maxwidth;
  graph->maxheight = maxheight;
}

#if NeedFunctionPrototypes
static void setmaxdim(LayoutData_t *graph)
#else
static void setmaxdim(graph)
LayoutData_t *graph;
#endif
{ Dimension maxwidth; Dimension maxheight; int i; Level_t *l;
  Bool horiz = IsHorizontal(graph->w);

  maxwidth  = 0;
  maxheight = 0;
  for(i=0;i<graph->n_levels;i++)
     { l = &graph->levels[i];
       if( horiz )
         { if( maxheight < l->maxheight )
             { maxheight = l->maxheight; }
           maxwidth += l->maxwidth;
           maxwidth += graph->w->graph.hpad;
         }
       else
         { if( maxwidth < l->maxwidth )
             { maxwidth = l->maxwidth; }
           maxheight += l->maxheight;
           maxheight += graph->w->graph.vpad;
         }
     }
  graph->maxwidth = maxwidth;
  graph->maxheight = maxheight;
}

#if NeedFunctionPrototypes
static void layout_level(LayoutData_t *graph,int level,
                         Dimension width,Dimension height)
#else
static void layout_level(graph,level,width,height)
LayoutData_t *graph;
int level;
Dimension width;
Dimension height;
#endif
{ int i; Level_t *l;
  Vertice_t *v;
  Dimension padding, centerx, centery;

  l = &graph->levels[level];

  if( IsHorizontal(graph->w) )
    { padding = (int)(graph->maxheight - l->maxheight) / l->n_elems;
      height += padding/2;
    }
  else
    { padding = (int)(graph->maxwidth - l->maxwidth) / l->n_elems;
      width += padding/2;
    }

  for(i=0; i<l->n_elems; i++)
     { v=l->sequence[i];
       switch( graph->w->graph.gravity )
       { case WestGravity: centerx= (int)(l->maxwidth - v->width)/2;
                           centery=0;
                           break;
         case EastGravity: centerx=(int)(l->maxwidth - v->width)/2;
                           centery=0;
                           break;
         case NorthGravity: centerx=0;
                            centery=(int)(l->maxheight - v->height)/2;
                            break;
         case SouthGravity: centerx=0;
                            centery=(int)(l->maxheight - v->height)/2;
                            break;
       }
       v->x = width+centerx;
       v->y = height+centery;
       switch( graph->w->graph.gravity )
       { case WestGravity: height += v->height + graph->w->graph.vpad +padding;
                           break;
         case EastGravity: height += v->height + graph->w->graph.vpad +padding;
                           break;
         case NorthGravity: width += v->width + graph->w->graph.hpad + padding;
                            break;
         case SouthGravity: width += v->width + graph->w->graph.hpad + padding;
                            break;
       }
     }
}

#if NeedFunctionPrototypes
static void layout_levels(LayoutData_t *graph)
#else
static void layout_levels(graph)
LayoutData_t *graph;
#endif
{ int i; Level_t *l;
  Dimension width; Dimension height;

  switch( graph->w->graph.gravity )
  { case WestGravity: width=graph->w->graph.hpad/2;
                      height=graph->w->graph.vpad/2;
                      break;
    case EastGravity: width=graph->maxwidth-graph->w->graph.hpad/2;
                      height=graph->w->graph.vpad/2;
                      break;
    case NorthGravity: height=graph->w->graph.vpad/2;
                       width=graph->w->graph.hpad/2;
                       break;
    case SouthGravity: height = graph->maxheight-graph->w->graph.vpad/2;
                       width=graph->w->graph.vpad/2;
                       break;
  }

  for( i=0; i<graph->n_levels; i++)
     { l = &graph->levels[i];
       switch( graph->w->graph.gravity )
       { case WestGravity: break;
         case EastGravity: width -= l->maxwidth; break;
         case NorthGravity: break;
         case SouthGravity: height -= l->maxheight; break;
       }
       layout_level(graph,i,width,height);
       switch( graph->w->graph.gravity )
       { case WestGravity: width += l->maxwidth + graph->w->graph.hpad; break;
         case EastGravity: width -= graph->w->graph.hpad; break;
         case NorthGravity: height += l->maxheight + graph->w->graph.vpad; break;
         case SouthGravity: height -= graph->w->graph.vpad; break;
       }
     }
}


#if NeedFunctionPrototypes
void execute_layout_alg(GraphWidget gw)
#else
void execute_layout_alg(gw)
GraphWidget gw;
#endif
{
  check_size_and_position(gw);
  clear_levels(&gw->graph.layout);
  make_dag(gw);
  make_levels(gw);
  minimize_edge_crossings(&gw->graph.layout);

  maxdim(&gw->graph.layout);
  layout_levels(&gw->graph.layout);

  pr_layout_graph(&gw->graph.layout);
  setmaxdim(&gw->graph.layout);

  set_position_levels(&gw->graph.layout);
}
