/********************************************************************************/
/* 										*/
/* 	fileselect.c : Zugriffsfunktionen auf das Dateisystem fuer Ghostview	*/
/*		       Basiert auf dem Dateidialog von Xe			*/
/*		       Roland Krause 1997					*/
/*										*/
/********************************************************************************/

/* Exportiert folgende Funktionen :
 *
 * InitFileDialog() :	Initialisiert das Modul
 * ExitFileDialog() :   Gibt durch das Modul belegte Ressourcen frei
 * OpenFileDialog() :	Oeffnet Dateidialog zur Auswahl einer Datei
 */

/* --- Include -------------------------------------------------------- */

#include <stdio.h>		/* Standard-Ein/Ausgabe */
#include <string.h>		/* String-Funktionen */
#include <malloc.h>		/* Dynamische Speicherverwaltung */
#include <unistd.h>		/* Systemfunktionen */
#include <sys/types.h>		/* Typen fuer Zugriff auf Dateisystem */
#include <dirent.h>		/* Zugriff auf Verzeichnisse */
#include <sys/stat.h>		/* Dateiinformationen */
#include <pwd.h>		/* User ID -> String umwandeln */

#include <X11/Intrinsic.h>	/* Xt-Header */
#include <X11/StringDefs.h>	/* Konstante fuer Ressourcen */

#include <X11/Shell.h>		/* Shell-Widget */
#include <X11/XawPlus/Paned.h>	/* Das Panel-Widget als Rahmen */
#include <X11/XawPlus/Form.h>	/* Rahmen fuer Button und Label */
#include <X11/XawPlus/Label.h>	/* Fuer die Titelzeile */
#include <X11/XawPlus/Command.h>   /* Buttons */
#include <X11/XawPlus/AsciiText.h> /* Text */
#include <X11/XawPlus/Viewport.h>  /* Scrollbars fuer das List-Widget */
#include <X11/XawPlus/IconList.h>  /* List-Widget fuer Dateiliste */

#include "gv.h"			/* Ghostview header */

/* --- Icons fuer Dateiliste von BuildFileList() ------------------------ */

/* Achtung : Alle Icons muessen in Breite Hoehe und Farbtiefe
 * den in filefunc.h definierten Parametern ICON_WIDTH und ICON_HEGHT
 * ( z. Zt. = 16x16 ) entsprechen !
 */

#include "bitmaps/file.xpm"		/* Normale Datei */
#include "bitmaps/psfile.xpm"		/* Postscript-Datei */
#include "bitmaps/folder.xpm"		/* Normales Verzeichnis */
#include "bitmaps/tool.xpm"		/* Ausfuehrbare Datei */
#include "bitmaps/shadefile.xpm"	/* Nicht lesbare Datei */
#include "bitmaps/shadefold.xpm"	/* Nicht ausfuerbares Verzeichnis */

/* --- Prototypen ------------------------------------------------------- */

char *getcwd(char *, size_t);	/* Ermittelt akt. Arbeitsverzeichnis */

typedef enum			/* Datentyp fuer CheckFileType */
{
  XE_FAULT,			/* Ungueltiger Eintrag */
  XE_FILE,			/* Normale Datei */
  XE_DIRECTORY			/* Verzeichnis */
}
  XeFileType;

/* --- Konstante -------------------------------------------------------- */

#define ICON_WIDTH	16	/* Konstante fuer die Icon-Bitmaps */
#define ICON_HEIGHT	16

#define	PATH_LEN	200	/* Laenge Pfadnamen */
#define LIST_LEN	36	/* Laenge Eintrag in Liste ohne Dateiname */
#define FNAME_LEN	14	/* Minimale Laenge Dateinamen in Liste */
#define FNAME_IN_LIST	12	/* Position Dateiname in Eintrag Dateiliste */

/* --- Globale Variable ------------------------------------------------- */

static Widget		mainShell,		/* Shell-Widget des Programms */
			fileShell,		/* Widget fuer Shell Dateiauswahldialog */
			fileList,		/* List Widget fuer Dateiliste */
			selectFile;		/* Text-Widget fuer gewaehlte Datei */
static char		AktPath[PATH_LEN]; 	/* Aktueller Pfadname */
static XawIconList	*Directory;		/* Dateiliste */
static int		IconDepth = 1;		/* Tiefe Icon-Pixmap */
static int		LenFName =  FNAME_LEN;	/* Aktuelle Laenge Dateinamen */

/* Pixmaps fuer die oben angegebenen Icons und Clip-Masken dazu
 */
static Pixmap folderPix		= XtUnspecifiedPixmap;
static Pixmap folderClip	= XtUnspecifiedPixmap;
static Pixmap shadefoldPix	= XtUnspecifiedPixmap;
static Pixmap shadefoldClip	= XtUnspecifiedPixmap;
static Pixmap filePix		= XtUnspecifiedPixmap;
static Pixmap fileClip		= XtUnspecifiedPixmap;
static Pixmap psfilePix		= XtUnspecifiedPixmap;
static Pixmap psfileClip	= XtUnspecifiedPixmap;
static Pixmap shadefilePix	= XtUnspecifiedPixmap;
static Pixmap shadefileClip	= XtUnspecifiedPixmap;
static Pixmap toolPix		= XtUnspecifiedPixmap;
static Pixmap toolClip		= XtUnspecifiedPixmap;

/**********************************************************************
 *
 *	Hilfsfunktionen
 *
 **********************************************************************/

/* Gibt den Speicher fuer die durch BuildFileList() angelegte
 * Dateiliste wieder frei.
 */
static void FreeFileList()
{
  int i = 0;

  if (Directory != NULL)		/* Liste angelegt ? */
  {
    while (Directory[i].string != NULL)	/* Eintraege freigeben */
    {
       free(Directory[i].string);
       i++;
    }
    free((void *)Directory);			/* Liste freigeben */
    Directory = NULL;
  }
}

/* ------------------------------------------------------------------ */

/* Vergleicht zwei Strings zum Sortieren der Dateiliste
 * in Funktion BuildFileList(). Ergebnis ist wie bei
 * strcmp() oder vergleichbare Funktionen.
 * Als Stringlaenge wird die globale Variable `LenFName' verwendet !
 */
static int CompareEntries(ent1, ent2)
char **ent1, **ent2;
{
  XawIconList *p1 = (XawIconList *)ent1; 
  XawIconList *p2 = (XawIconList *)ent2;

  if ((p1->bitmap == folderPix) || (p1->bitmap == shadefoldPix))
    if ((p2->bitmap != folderPix) && (p2->bitmap != shadefoldPix)) return (-1);
    else return (strncmp(&(p1->string[FNAME_IN_LIST]), &(p2->string[FNAME_IN_LIST]), LenFName));
  else
    if ((p2->bitmap == folderPix) || (p2->bitmap == shadefoldPix)) return (1);
    else return (strncmp(&(p1->string[FNAME_IN_LIST]), &(p2->string[FNAME_IN_LIST]), LenFName));
}

/* ------------------------------------------------------------------ */

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt das korrekte Icon fuer die angegebene Datei
 */
static Pixmap GetIcon(FileName, FileFlags)
char		*FileName;
unsigned short	FileFlags;
{
  int StrLen;

  /* Zunaechst pruefen, ob Datei ein Verzeichnis ist.
   * Wenn es sich um ein Verzeichnis handelt und die
   * die Datei ist ausfuehrbar, wird das Verzeichnis-Icon verwendet.
   * Ist die Datei nicht lesbar, wird ein graues Verzeichnis-Icon
   * verwendet.
   */
  if (FileFlags & S_IFDIR)
    if (access(FileName, X_OK) == 0) return(folderPix);
    else return(shadefoldPix);
  else
  /* Datei ist kein Verzeichnis:
   * Wenn die Datei nicht lesbar ist, wird ein graues Datei-Icon
   * verwendet. Wenn die Datei lesbar und ausfuehrbar ist, wird
   * ein Programm-Icon benutzt. Wenn die Datei lesbar und nicht
   * ausfuehrbar ist, wird ein normales Datei-Icon verwendet.
   */
    if (access(FileName, R_OK) == 0)
      if (FileFlags & (S_IEXEC|S_IXGRP|S_IXOTH)) return(toolPix);
      else
      {
	 /* Dateien mit der Endung '.ps' bekommen das Postscript-Icon */

	 StrLen = strlen(FileName);
	 if ((StrLen > 3) && 
	      ((strcmp(&FileName[StrLen-3], ".ps") == 0) ||
	       (strcmp(&FileName[StrLen-4], ".eps") == 0)||
	       (strcmp(&FileName[StrLen-3], ".PS") == 0) ||
	       (strcmp(&FileName[StrLen-4], ".EPS") == 0))) return(psfilePix);
         else return(filePix);
      }
    else return(shadefilePix);
}

/* ------------------------------------------------------------------ */

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt die richtige Clip-Maske fuer ein Icon
 */
static Pixmap GetClip(Icon)
Pixmap Icon;
{
  if (Icon == filePix) return(fileClip);
  if (Icon == psfilePix) return(psfileClip);
  if (Icon == folderPix) return(folderClip);
  if (Icon == toolPix) return(toolClip);
  if (Icon == shadefilePix) return(shadefileClip);
  if (Icon == shadefoldPix) return(shadefoldClip);
  return(XtUnspecifiedPixmap);
}

/* ------------------------------------------------------------------ */

/* Hilfsfunktion fuer BuildFileList():
 * Baut einen String fuer einen Eintrag in der Dateiliste auf.
 * Der Dateiname wird mit der Laenge 'FNameLen' eingefuegt.
 */
static void BuildListEntry(ListEntry, Name, FNameLen, FileInfo)
char	    *ListEntry, *Name;
int	     FNameLen;
struct stat *FileInfo;
{
   struct passwd *usr;			/* Fuer User ID in ASCII */
   char		 FileName[PATH_LEN],	/* Puffer fuer Dateiname */
		 UserName[PATH_LEN],	/* dto. fuer Name Eigentuemer */
		 Blanks[PATH_LEN];	/* Zum auffuellen der Dateinamen */

   /* Zunaechst den Puffer zum auffuellen der Dateinamen initialisieren */

   (void)memset(Blanks, ' ', FNameLen);
   Blanks[FNameLen] = '\0';

   /* Dateinamen und Name Eigentuemer ermitteln und
    * String auf mindestens 14 Zeichen auffuellen
    */
   strcpy(FileName, Name);
   strcat(FileName, Blanks);
   if ((usr = getpwuid(FileInfo->st_uid)) != NULL) strcpy(UserName, usr->pw_name);
   else *UserName = '\0';
   strcat(UserName, Blanks);

   /* String zusammenbauen: Ergebnis darf maximal aus LIST_LEN
    * Zeichen bestehen !
    */
   sprintf(ListEntry, " %c%c%c%c%c%c%c%c%c  %-.*s %-.10s %lu",
	FileInfo->st_mode & S_IREAD  ? 'r' : '-',
	FileInfo->st_mode & S_IWRITE ? 'w' : '-',
	FileInfo->st_mode & S_IEXEC  ? 'x' : '-',
	FileInfo->st_mode & S_IRGRP  ? 'r' : '-',
	FileInfo->st_mode & S_IWGRP  ? 'w' : '-',
	FileInfo->st_mode & S_IXGRP  ? 'x' : '-',
	FileInfo->st_mode & S_IROTH  ? 'r' : '-',
	FileInfo->st_mode & S_IWOTH  ? 'w' : '-',
	FileInfo->st_mode & S_IXOTH  ? 'x' : '-',
	FNameLen, FileName, UserName, FileInfo->st_size);
}

/* ------------------------------------------------------------------ */

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt die Laenge des laengsten Dateinamens
 * in einem geoffneten Verzeichnis
 */
static int CheckFNames(dir)
DIR *dir;
{
  struct dirent *dp;		/* Eintrag Verzeichnisdatei */
  int MaxLen = FNAME_LEN;	/* Ermittelte Laenge */
 int  Len;

  while ((dp = readdir(dir)) != NULL)
  {
     Len = strlen(dp->d_name);
     if (Len > MaxLen) MaxLen = Len;
  }
  rewinddir(dir);
  return(MaxLen);
}

/**********************************************************************
 *
 *	Callbacks
 *
 **********************************************************************/

/* Traegt den in der Liste angeklickten Dateinamen im Textfeld
 * fuer den ausgewahlten Dateinamen ein. Bei einem Verzeichnis
 * wird dieses geoeffnet und als Liste ausgegeben. Im Textfeld
 * wird entweder der selektierte Dateiname oder das selektierte
 * Verzeichnis ausgegeben.
 */
static void SetFileEntry(w, clientData, callData)
Widget w;
XtPointer clientData, callData;
{
   XawIconListReturnStruct	*SelectEntry;
   XawIconList			*FileList;
   char				*FileName, ActualPath[200];

   static XawIconList *BuildFileList();
   static char	      *GetFileName();
   static XeFileType   CheckFileType();
   static void	       ChangeDir();
   static void	       ResetDir();

   SelectEntry = (XawIconListReturnStruct *)callData;

   /* Pruefen, ob Verzeichnis angeklickt wurde. Ggfs. Verzeichnis
    * wechseln, neue Dateiliste aufbauen
    */

   switch (CheckFileType(SelectEntry->list_index))
   {
     case XE_DIRECTORY:	/* Eintrag ist gueltiges Verzeichnis */

	ChangeDir(SelectEntry->list_index);
	if ((FileList = BuildFileList()) != NULL)
	{
	  /* Neue Dateiliste aufbauen aktuelles Verzeichnis im
           * Textfeld fuer den Dateinamen eintragen
	   */
	  (void)strcpy(ActualPath, AktPath);
	  if (ActualPath[1] != '\0') (void)strcat(ActualPath, "/");
	  XtVaSetValues(selectFile, XtNstring, ActualPath, NULL);
	  XawIconListChange(fileList, FileList, 0, 0, ICON_WIDTH, ICON_HEIGHT,
			    IconDepth, True);
	}
	else	/* Fehler : Dateiliste kann nicht aufgebaut werden */
	{
	  ResetDir();
	  XBell(XtDisplay(mainShell), 0);
	  XtPopdown(fileShell);
	}
        break;

     case XE_FILE:		/* Eintrag ist normale Datei */

	FileName = GetFileName(SelectEntry->list_index);
	XtVaSetValues(selectFile, XtNstring, FileName, NULL);
	break;

     default:			/* ungueltiger Eintrag */
	XBell(XtDisplay(mainShell), 0);
        break;
   }
}

/* ------------------------------------------------------------------ */

/* Schliesst die Dialogbox vom Dateidialog ohne eine Datei
 * zu oeffnen
 */
static void CloseFileDialog(w, clientData, callData)
Widget w;
XtPointer clientData, callData;
{
  XtPopdown(fileShell);
}

/* ------------------------------------------------------------------ */

/* Schliesst die Dialogbox vom Dateidialog und oeffnet
 * eine Datei, wenn das moeglich ist.
 */
static void OpenFile(w, clientData, callData)
Widget w;
XtPointer clientData, callData;
{
  FILE		*fd;
  struct stat	FileInfo;
  char 		*FileName = NULL;

  XtVaGetValues(selectFile, XtNstring, &FileName, NULL);

  /* Pruefen, ob schon eine Datei ausgewaehlt wurde:
   * Wenn nicht, endet der String mit `/'
   */
  if (FileName[strlen(FileName)-1] != '/')
  {
    /* Datei wurde ausgewaehlt */

    XtPopdown(fileShell);

    /* Datei zum Lesen oeffnen und Ausgabe mit Ghostscript anstossen */

    if ((fd = fopen(FileName,"r")) != NULL)
    {
	if (oldfilename) XtFree(oldfilename);   /* Alte Strings freigeben */
	oldfilename = NULL;
	if (filename)
	{
	   oldfilename = XtNewString(filename); /* und mit neuem Namen allokieren */
    	   XtFree(filename);
	}
        filename = XtNewString(FileName);

	if (psfile) fclose(psfile);
	psfile = fd;
	stat(filename, &FileInfo);
	mtime = FileInfo.st_mtime;
	new_file(0);
	show_page(0);
    }
    else XBell(XtDisplay(mainShell), 0); /* Datei kann nicht gelesen werden */
  }
  else XBell(XtDisplay(mainShell), 0);   /* Verzeichnis ausgewaehlt */
}

/**********************************************************************
 *
 *	Interne Funktionen
 *
 **********************************************************************/

/* Erzeugt einen Dialog zum Auswaehlen einer Datei, die geladen oder
 * gespeichert werden soll. Als Basis dient eine 'transient shell',
 * die mit XtPopup() und XtPopdown() aufgeblendet oder abgeblendet
 * werden kann. Auf dieser Shell wird ein Panel angelegt, das als Basis
 * fuer alle Manager-Widgets dient. Die Widgets auf dem Panel werden
 * von oben nach unten angelegt. Im oberen Feld wird ein List-Widget
 * auf einem View-Widget fuer die Dateiliste erzeugt. Im mittleren
 * Feld wird ein Text-Widget fuer den gewaehlten Dateinamen mit einem
 * vorangestellten Text-Label angelegt. Im untersten Feld werden
 * die Buttons OK und ABBRUCH angelegt. Vor dem Aufblenden des
 * Dialogs muss dem List-Widget eine Liste zugewiesen werden.
 */
static void CreateFileDialogWidget()
{
  Widget	panel,				/* Manager fuer alle Objekte */
		view, fileForm, buttonBox,	/* fuer Liste, Datei u. Buttons */
		fileLabel, ok, abort;

  static char	*NoCR = "<Key>Return: no-op()";	/* CR bei Eingaben ignorieren */
  XtTranslations TransTab;

  /* Zunaechst Shell-Widget zum aufblenden und Panel-Widget als Manager
   * fuer alle anderen Widgets erzeugen
   */
  fileShell   = XtVaCreatePopupShell("fileDialogShell", transientShellWidgetClass,
		mainShell, XtNallowShellResize, TRUE, NULL);

  panel       = XtVaCreateManagedWidget("fileDialogPanel", panedWidgetClass, 
		fileShell, XtNallowResize, TRUE, NULL);

  /* Auf dem Panel wird ein Viewport fuer die Dateiliste und zwei
   * Form-Widgets fuer den Dateinamen und fuer die Buttons OK und
   * ABBRUCH erzeugt
   */
  view	      = XtVaCreateManagedWidget("listView", viewportWidgetClass, panel,
		XtNshowGrip, FALSE,  XtNforceBars, TRUE,
		XtNallowVert, TRUE, XtNallowHoriz, TRUE,
		XtNuseBottom, TRUE, XtNuseRight, TRUE,
		NULL);

  fileForm    = XtVaCreateManagedWidget("fileForm", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

  buttonBox   = XtVaCreateManagedWidget("fileButtons", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

  /* List-Widget auf dem Viewport fuer Datei-Liste erzeugen */

   fileList   = XtVaCreateManagedWidget("selectList", iconListWidgetClass, view,
		XtNforceColumns, TRUE, NULL);
   XtAddCallback(fileList, XtNcallback, SetFileEntry, NULL);

  /* Eingabefeld und Label fuer gewahlten Dateinamen anlegen.
   * CR soll im Textfeld ignoriert werden, weil nur eine Zeile genutzt wird.
   */
  fileLabel   = XtVaCreateManagedWidget("selectLabel", labelWidgetClass, fileForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		NULL);

  selectFile  = XtVaCreateManagedWidget("fileName", asciiTextWidgetClass, fileForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNeditType, XawtextEdit, XtNfromHoriz, fileLabel,
		NULL);
  TransTab = XtParseTranslationTable(NoCR);
  XtOverrideTranslations(selectFile, TransTab);

  /* Ok- und Abbruch-Button auf Form-Widget anlegen
   */
  ok          = XtVaCreateManagedWidget("ok", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		NULL);
  XtAddCallback(ok, XtNcallback, OpenFile, NULL);

  abort       = XtVaCreateManagedWidget("abort", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		XtNfromHoriz, ok, NULL);
  XtAddCallback(abort, XtNcallback, CloseFileDialog, NULL);
}

/* ------------------------------------------------------------------ */

/* Erzeugt aus einem String der Liste, die mit BuildFileList
 * erzeugt wurde, einen vollstaendigen Dateinamen
 */
static char *GetFileName(Index)
int Index;
{
  static char	Filename[PATH_LEN];
  char		*p, *q;

  strcpy(Filename, AktPath);		/* Pfad kopieren */
  p = Filename;	
  while (*p != '\0') p++;		/* Ende Pfadname suchen */
  if (Filename[1] != '\0') *p++ = '/';	/* root - Verzeichnis ? */
  q = &(Directory[Index].string[FNAME_IN_LIST]);
  while (*q != ' ') *p++ = *q++;	/* Dateiname anhaengen */
  *p = '\0';				/* Stringende markieren */
  return(Filename);
}

/* Prueft, ob Eintrag in der Dateiliste zu einem
 * ausfuehrbaren Verzeichnis gehoert. In diesem Fall
 * wurde dem Eintrag die Bitmap 'folderPix' zugeordnet.
 */
static XeFileType CheckFileType(Index)
int Index;
{
  if ((Directory[Index].bitmap == filePix) ||		/* Gewoehnliche Datei */
      (Directory[Index].bitmap == psfilePix) ||
      (Directory[Index].bitmap == shadefilePix) ||
      (Directory[Index].bitmap == toolPix)) return(XE_FILE);

  if (Directory[Index].bitmap == folderPix) return(XE_DIRECTORY);
  return(XE_FAULT);					/* Sonstiges: Fehler */
}

/* ------------------------------------------------------------------ */

/* Wechselt relativ das aktuell gesetzte Verzeichnis.
 * Bei '.' passiert nichts, bei '..' wird der Pfad um einen Eintrag
 * verkuerzt und bei normalen Namen wird eine Stufe tiefer in dieses
 * Verzeichnis gewechselt. 'ListEntry' muss ein Eintrag aus einer
 * mit BuildFileList() erzeugten Dateiliste sein !
 */
static void ChangeDir(Index)
int Index;
{
   char *p, *q, PathName[PATH_LEN];

   p = PathName;			/* Pfadname isolieren */
   q = &(Directory[Index].string[FNAME_IN_LIST]);
   while (*q != ' ') *p++ = *q++;	/* Dateiname kopieren */
   *p = '\0';				/* Stringende markieren */

   if (strcmp(PathName, ".") != 0)
   {
      if (strcmp(PathName, "..") == 0)
      {
	 /* Wenn wir noch nicht im root - Verzeichnis angekommen
	  * sind, wird der Pfad um einen Verzeichniseintrag verkuerzt.
	  */
	 p = strrchr(AktPath, '/');
	 if ((p != NULL) && (p != AktPath)) *p = '\0';
	 else AktPath[1] = '\0';
      }
      else	/* Pfad um 'PathName' verlaengern */
      {
	 if (AktPath[1] != '\0') strcat(AktPath, "/");
	 strcat(AktPath, PathName);
      }
   }
}

/* ------------------------------------------------------------------ */

/* Setzt den aktuellen Verzeichnispfad auf den vorherigen
 * Pfad zurueck. Funktion wird benoetigt, um nach dem Wechsel
 * in ein nicht lesbares Verzeichnis den Pfadwechsel rueckgaengig
 * zu machen.
 */
static void ResetDir()
{
  char *p;

  /* Wenn wir noch nicht im root - Verzeichnis angekommen
   * sind, wird der Pfad um einen Verzeichniseintrag verkuerzt.
   */
   p = strrchr(AktPath, '/');
   if ((p != NULL) && (p != AktPath)) *p = '\0';
   else AktPath[1] = '\0';
}

/* ------------------------------------------------------------------ */

/* Erzeugt eine Dateiliste fuer das List-Widget
 * Funktionsergebnis ist im Fehlerfall ein NULL-Zeiger,
 * sonst ein Zeiger auf die Liste
 */
static XawIconList *BuildFileList()
{
   DIR           *dir;			/* Dateizeiger fuer Verzeichnis */
   struct dirent *dp;			/* Eintrag Verzeichnisdatei */
   struct stat   fileinfo;		/* Informationen ueber Datei */
   char		 FileName[PATH_LEN];	/* Puffer fuer Dateiname */
   int		 Files;

   FreeFileList();
   if ((Directory = (XawIconList *)malloc(sizeof(XawIconList))) != NULL)
   {
      Directory[0].bitmap   = XtUnspecifiedPixmap;
      Directory[0].clipMask = XtUnspecifiedPixmap;
      Directory[0].string   = NULL;
      Files = 0;
      if ((dir = opendir(AktPath)) != NULL)
      {
	 /* Zunaechst Laenge des laengsten Dateinamens ermitteln */

	 LenFName = CheckFNames(dir);

	 /* Alle Dateien des Verzeichnisses untersuchen */

	 while ((dp = readdir(dir)) != NULL)
	 {
	   sprintf(FileName, "%s/%s", AktPath, dp->d_name);
	   if (stat(FileName, &fileinfo) == 0)
	   {
	      /* Pruefen, ob regulaere Datei oder Verzeichnis.
	       * Andere Dateien werden nicht beruecksichtigt.
	       */
	      if (S_ISDIR(fileinfo.st_mode) || S_ISREG(fileinfo.st_mode))
	      {
		/* Eintrag in Verzeichnis gefunden: Speicher beschaffen,
		 * Neuen Eintrag fuer die Liste aufbauen
		 */
		Files++;
		Directory = (XawIconList *)realloc((char *)Directory, (Files+1) * sizeof(XawIconList));
		Directory[Files].string   = NULL;
		Directory[Files].bitmap   = XtUnspecifiedPixmap;
		Directory[Files].clipMask = XtUnspecifiedPixmap;
		if ((Directory[Files-1].string = malloc(LIST_LEN+LenFName)) != NULL)
		{
		  /* Eintrag fuer die Liste aufbauen und geeignetes Icon
		   * fuer die Datei auswaehlen. Achtung : Die Laenge
		   * des String darf nicht groesser als LIST_LEN+LenFName sein !
		   */
		  BuildListEntry(Directory[Files-1].string, dp->d_name, LenFName, &fileinfo);
		  Directory[Files-1].bitmap   = GetIcon(FileName, fileinfo.st_mode);
		  Directory[Files-1].clipMask = GetClip(Directory[Files-1].bitmap);
		}
		else	/* Fehler beim Aufbau der Liste: Kein Speicher */
		{
		   FreeFileList();
		   return (NULL);
		}
	      }
	   }
	 }
	 /* Ok: Verzeichnis schliessen, Eintraege sortieren
	  * und Zeiger auf die Liste zurueckgeben
	  */
	 closedir(dir);
	 qsort(Directory, Files, sizeof(XawIconList), CompareEntries);
	 return (Directory);
      }
      /* Fehler: Verzeichnis kann nicht geoeffnet werden
       */
      free((void *)Directory);
      Directory = NULL;
    }
    return (NULL);
}

/**********************************************************************
 *
 *	Exportierte Funktionen
 *
 **********************************************************************/

/* Initialisierung: Bestimmt den Pfadnamen des Arbeitsverzeichnisses
 * und erzeugt Pixmaps fuer die Icons in der Dateiliste. Das Shell-
 * Widget muss zur Erzeugung der Bitmaps bereits exisitieren !
 */
int InitFileDialog(topLevelShell)
Widget topLevelShell;
{
  Display  *disp;
  Window   win;
  int	   ScrNum;

  Directory = NULL;
  if (getcwd(AktPath, 200) != NULL)
  {
     mainShell = topLevelShell;

     /* Pixmaps fuer Dateiliste erzeugen */

     disp	  = XtDisplay(mainShell);
     win	  = XtWindow(mainShell);

     ScrNum    = XDefaultScreen(disp);
     IconDepth = XDefaultDepthOfScreen(XScreenOfDisplay(disp,ScrNum));

     (void)XpmCreatePixmapFromData(disp,win,folder_xpm,&folderPix,&folderClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,shadefold_xpm,&shadefoldPix,&shadefoldClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,file_xpm,&filePix,&fileClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,psfile_xpm,&psfilePix,&psfileClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,shadefile_xpm,&shadefilePix,&shadefileClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,tool_xpm,&toolPix,&toolClip,NULL);
     CreateFileDialogWidget();
     return (TRUE);
  }
  return (FALSE);
}
/* ------------------------------------------------------------------ */

/* Gibt alle belegten Ressourcen wieder frei
 */
void ExitFileDialog()
{
   Display *disp = XtDisplay(mainShell);

   XFreePixmap(disp, filePix);		/* Zunaechst die Pixmaps freigeben */
   XFreePixmap(disp, psfilePix);
   XFreePixmap(disp, shadefilePix);
   XFreePixmap(disp, folderPix);
   XFreePixmap(disp, shadefoldPix);
   XFreePixmap(disp, toolPix);
   XFreePixmap(disp, fileClip);		/* XPM: Clip-Masken freigeben */
   XFreePixmap(disp, psfileClip);
   XFreePixmap(disp, shadefileClip);
   XFreePixmap(disp, folderClip);
   XFreePixmap(disp, shadefoldClip);
   XFreePixmap(disp, toolClip);
   FreeFileList();			/* Speicher Dateiliste freigeben */
}

/* ------------------------------------------------------------------ */

/* Oeffnet das Fenster zum Auswaehlen einer Datei.
 * Das Fenster wird an relativer Position zum Ghostview-Fenster aufgeblendet.
 */
void OpenFileDialog()
{
  int		XPos, YPos;
  char		ActualPath[PATH_LEN];
  XawIconList	*FileList;

  XtVaGetValues(mainShell, XtNx, &XPos, XtNy, &YPos, NULL);
  XtVaSetValues(fileShell, XtNx, XPos+100, XtNy, YPos+40, NULL);
  if ((FileList = BuildFileList()) != NULL)
  {
     (void)strcpy(ActualPath, AktPath);
     if (ActualPath[1] != '\0') (void)strcat(ActualPath, "/");
     XtVaSetValues(selectFile, XtNstring, ActualPath, NULL);
     XawIconListChange(fileList, FileList, 0, 0, 
			ICON_WIDTH, ICON_HEIGHT, IconDepth, True);
     XtPopup(fileShell, XtGrabExclusive);
  }
  else XBell(XtDisplay(mainShell), 0);	/* Liste kann nicht angelegt werden */
}
