/*+****************************************************************************/
/*+                                                                           */
/*+ Copyright (c) 1992-1997 Bruce M. Corwin                                   */
/*+                                                                           */
/*+****************************************************************************/
/*+****************************************************************************/
/*+                                                                           */
/*+ Module Name: load_menu.c                                                  */
/*+                                                                           */
/*+ Program ID:  vtwidget                                                     */
/*+                                                                           */
/*+ Functions:  free_menu, load_menu, strcicmp,                               */
/*+             strcincmp, collect, process_line,                             */
/*+             find_command, parse_line, singlespcQ,                         */
/*+             XmVtLoadPopup, StoreOption,                                   */
/*+             SetUpPopupMenu, XmVtUnloadPopup,                              */
/*+             PopupMenuPost, CreateOneOption,                               */
/*+             CreateOneCascade, CreateOneSep,                               */
/*+             PopupCallback, StoreMenuShell                                 */
/*+                                                                           */
/*+****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/MenuShell.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/TextF.h>
#include <Xm/Form.h>
#include "VtP.h"
/***************************************************************/
/* The following are definitions for this module.              */
/***************************************************************/
#define  READMODE           ( "r" )
#define  F_TITLE            ( 0 )
#define  F_MENU             ( 1 )
#define  F_EXEC             ( 2 )
#define  F_SEPARATOR        ( 4 )
#define  F_RESTART          ( 5 )
#define  F_QUIT             ( 34 )
#define  F_QUIT_MWM         ( 36 )
#define  F_SEND             ( 37 )
#define  F_SEND_CR          ( 38 )
#define  F_WRITE            ( 39 )
#define  NUMDIRECTS         (1000)
#define  XMASK              ( 'X' )
#define  NULLCHR            ( 0 )
#define  SPC                ( ',' )
#define  COMMA              ( ',' )
#define  COLON              ( ':' )
#define  DQUOTE             ( '"' )
#define  BSLASH             ( '\\' )
#define  FSLASH             ( '/' )
#define  FSLASHSTR          ( "/" )
#define  SPCMASK            ( ' ' )
#define  NEWLINE            ( '\n' )
#define  TAB                ( '\t' )
#define  HOMEPATH           ( "HOME" )
#define  NULLTITLE          ( "NULLTITLE" )
/***************************************************************/
/* End of definitions                                          */
/***************************************************************/
/***************************************************************/
/* The following are global variables for this module.         */
/***************************************************************/
static char *com_str[] =
{
    "f.title",
    "f.menu",
    "f.exec",
    "f.refresh",
    "f.separator",
    "f.restart",
    "f.separator",
    "f.action",
    "f.circle_down",
    "f.circle_up",
    "f.kill",
    "f.lower",
    "f.maximize",
    "f.minimize",
    "f.move",
    "f.normalize",
    "f.occupy_all",
    "f.remove",
    "f.resize",
    "f.toggle_frontpanel",
    "f.workspace_presence",
    "f.pack_icons",
    "f.restore",
    "f.raise",
    "f.quit_app",
    "f.iconify",
    "f.nop",
    "f.focus",
    "f.unfocus",
    "f.showiconmgr",
    "f.hideiconmgr",
    "f.nop",
    "f.destroy",
    "f.delete",
    "f.quit",
    "f.set_behavior",
    "f.quit_mwm",
    "f.send",
    "f.send_cr",
    "f.write",
    NULL
};
/***************************************************************/
/* End of global variables                                     */
/***************************************************************/
/***************************************************************/
/* The following are API calls provided to the Vt.c module     */
/***************************************************************/
void XmVtLoadPopup (Widget parent, char *root, char *location, char *app);
void XmVtUnloadPopup (Widget parent);
/***************************************************************/
/* End of API for Vt.c                                         */
/***************************************************************/
/***************************************************************/
/* The following are static function internal to this module.  */
/***************************************************************/
static MENU *load_menu (char *menupath, int *count);
static void free_menu (MENU * vtmenu, int count);
static int collect (int ch, char *buffer);
static int strcicmp (char *firststr, char *secondstr);
static int strcincmp (char *firststr, char *secondstr, int len);
static int process_line (char *buffer, MENU * retval, int *count);
static int find_command (char *command_str);
static int parse_line (char **elements, char *line, char *mask, char quotechr);
static void singlespcQ (char *string, char *mask, char quotechr);
static Widget SetUpPopupMenu (Widget parent, char *name, char *title);
static Widget CreateOneOption (Widget parent, char *name, char *label, char *value);
static Widget CreateOneCascade (Widget parent, char *name, char *label, char *value, MENU * menu_array, int count, Widget shell);
static Widget CreateOneSep (Widget parent, char *name);
static void PopupMenuPost (Widget wgt, XtPointer client_data, XEvent * event, Boolean * ctd);
static void PopupCallback (Widget sent_widget, char *client_dat, XtPointer call_back);
static void StoreOption (Widget parent, Widget new_popup, char *name, char *root, MENU * new_menu, int count);
static void StoreMenuShell (Widget parent, Widget shell);
/***************************************************************/
/* End of static functions.                                    */
/***************************************************************/

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: free_menu                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to clean up all dynamic memory used by  */
/*+              a menu type structure.                                       */
/*+                                                                           */
/*+ Parameters: vtmenu        - MENU *     - This is a menu structure to      */
/*+                                          free.                            */
/*+             count         - int        - Count of characters to write     */
/*+                                                                           */
/*+ Input:      vtmenu, count                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void free_menu (MENU * vtmenu, int count)
{
    int inx = 0;

    for (inx = 0; inx < count; inx++)
    {
	free (vtmenu[inx].menu);
	free (vtmenu[inx].description);
	free (vtmenu[inx].data);
    }
    free (vtmenu);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: load_menu                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to load a given menu file to derive a   */
/*+              dynamic popup menu for the Vt widget right hand mouse press  */
/*+              action.                                                      */
/*+                                                                           */
/*+ Parameters: menupath      - char *     - This is the full path to a menu  */
/*+                                          file to load.                    */
/*+             count         - int *      - Count of characters to write     */
/*+                                                                           */
/*+ Input:      menupath, count                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static MENU *load_menu (char *menupath, int *count)
{
    FILE *menu_file = NULL;
    MENU *retval = NULL;
    int ch = 0;
    int line_found = 0;
    int process = -1;
    int crlf = -1;
    int within_block = 0;
    char buffer[BUFSIZ];
    char *local_buffer = NULL;

    menu_file = fopen (menupath, READMODE);
    if (menu_file != NULL)
    {
	retval = (MENU *) malloc (NUMDIRECTS * sizeof (MENU));
	buffer[0] = 0;
	while ((ch = getc (menu_file)) != EOF)
	{
	    if (isspace (ch) && ch != 10)
		ch = ' ';
	    if (ch == '#' && crlf)
		process = 0;
	    if (ch == '!' && crlf)
		process = 0;
	    if (ch == 10 && crlf)
		process = 0;
	    if (process)
	    {
		line_found = collect (ch, buffer);
		if (line_found)
		{
		    local_buffer = buffer;
		    while (*local_buffer == ' ' && *local_buffer != 0)
			local_buffer++;
		    if (strcincmp (local_buffer, "menu ", 5) == 0)
		    {
			within_block = -1;
		    }
		    if (within_block)
			process_line (local_buffer, retval, count);
		    if (local_buffer[0] == '}')
			within_block = 0;
		    buffer[0] = 0;
		}
	    }
	    crlf = 0;
	    if (ch == 10)
	    {
		crlf = -1;
		process = -1;
	    }
	}
	fclose (menu_file);
    }
    return (retval);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: strcicmp                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine works exactly like strcmp except that the       */
/*+              comparison is case insensitive. (Notice the 'ci' in the      */
/*+              name.)                                                       */
/*+                                                                           */
/*+ Parameters: firststr      - char *     - First string to compare          */
/*+             secondstr     - char *     - Second string to compare.        */
/*+                                                                           */
/*+ Input:      firststr, secondstr                                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int strcicmp (char *firststr, char *secondstr)
{
    int rcode = 0;
    int inx = 0;
    char *ifirst = NULL;
    char *isecond = NULL;

    ifirst = (char *) malloc (strlen (firststr) + 1);
    isecond = (char *) malloc (strlen (secondstr) + 1);
    for (inx = 0; inx < strlen (firststr); inx++)
	ifirst[inx] = toupper (firststr[inx]);
    ifirst[inx] = 0;
    for (inx = 0; inx < strlen (secondstr); inx++)
	isecond[inx] = toupper (secondstr[inx]);
    isecond[inx] = 0;
    rcode = strcmp (ifirst, isecond);
    free (ifirst);
    free (isecond);
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: strcincmp                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine works exactly like strncmp except that the      */
/*+              comparison is case insensitive. (Notice the 'ci' in the      */
/*+              name.)                                                       */
/*+                                                                           */
/*+ Parameters: firststr      - char *     - First string to compare          */
/*+             secondstr     - char *     - Second string to compare.        */
/*+             len           - int        - Length of string                 */
/*+                                                                           */
/*+ Input:      firststr, secondstr, len                                      */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int strcincmp (char *firststr, char *secondstr, int len)
{
    int rcode = 0;
    int inx = 0;
    char *ifirst = NULL;
    char *isecond = NULL;

    ifirst = (char *) malloc (strlen (firststr) + 1);
    isecond = (char *) malloc (strlen (secondstr) + 1);
    for (inx = 0; inx < strlen (firststr); inx++)
	ifirst[inx] = toupper (firststr[inx]);
    for (inx = 0; inx < strlen (secondstr); inx++)
	isecond[inx] = toupper (secondstr[inx]);
    rcode = strncmp (ifirst, isecond, len);
    free (ifirst);
    free (isecond);
    return (rcode);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: collect                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to collect preprocessed lines that have */
/*+              already had comments and blank lines removed.                */
/*+                                                                           */
/*+ Parameters: ch            - int        - This is a single character.      */
/*+             buffer        - char *     - Buffer of output data            */
/*+                                                                           */
/*+ Input:      ch, buffer                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int collect (int ch, char *buffer)
{
    int retval = 0;
    int len = 0;

    if (ch != 10)
    {
	len = strlen (buffer);
	buffer[len] = ch;
	buffer[len + 1] = 0;
    }
    else
    {
	retval = -1;
    }
    return (retval);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: process_line                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to process each line of the vtrc file   */
/*+              individualy. It scans each line for various pieces of        */
/*+              information such as menu name, command to execute on that    */
/*+              menu item, data to operate upon and a description for the    */
/*+              menu item.                                                   */
/*+                                                                           */
/*+ Parameters: buffer        - char *     - Buffer of output data            */
/*+             retval        - MENU *     - This is an output value.         */
/*+             count         - int *      - Count of characters to write     */
/*+                                                                           */
/*+ Input:      buffer, count                                                 */
/*+                                                                           */
/*+ Output:     retval                                                        */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int process_line (char *buffer, MENU * retval, int *count)
{
    static int outside = -1;
    static char menu_name[BUFSIZ];
    int inx = 0;
    int argcnt = 0;
    char mask[BUFSIZ];
    char nullstr[2];
    char quote = '"';
    char *elements[100];
    char *desc = NULL;
    char *data = NULL;
    char *description = NULL;
    char *command_str = NULL;
    int command = 0;

    nullstr[0] = 0;
    singlespcQ (buffer, mask, quote);
    argcnt = parse_line ((char **) (&elements), buffer, mask, quote);
    if (elements[0][0] != '{' && elements[0][0] != '}')
    {
	if (outside)
	{
	    if (elements[1] != NULL)
	    {
		strcpy (menu_name, elements[1]);
	    }
	}
	else
	{
	    desc = elements[0];
	    if (elements[1] != NULL)
	    {
		if (strcincmp (elements[argcnt - 1], "f.", 2) == 0)
		{
		    data = nullstr;
		    command_str = elements[argcnt - 1];
		}
		else
		{
		    data = elements[argcnt - 1];
		    command_str = elements[argcnt - 2];
		}
		command = find_command (command_str);
		retval[*count].menu = (char *) malloc (strlen (menu_name) + 1);
		strcpy (retval[*count].menu, menu_name);
		retval[*count].description = (char *) malloc (strlen (desc) + 1);
		strcpy (retval[*count].description, desc);
		retval[*count].data = (char *) malloc (strlen (data) + 1);
		strcpy (retval[*count].data, data);
		retval[*count].command = command;
		(*count)++;
	    }
	}
    }
    else
    {
	switch (elements[0][0])
	{
	    case '{':
		outside = 0;
		break;
	    case '}':
		outside = -1;
		break;
	}
    }
    return (0);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: find_command                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine determines what token should represent the      */
/*+              current command. It returns a -1 if the command is not a     */
/*+              supported command. It returns the index number for a         */
/*+              supported command.                                           */
/*+                                                                           */
/*+ Parameters: command_str   - char *     - This is the command string to    */
/*+                                          evaluate.                        */
/*+                                                                           */
/*+ Input:      command_str                                                   */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int find_command (char *command_str)
{
    int inx = 0;
    int retval = -1;

    while (com_str[inx] != NULL)
    {
	if (strcicmp (com_str[inx], command_str) == 0)
	{
	    retval = inx;
	    break;
	}
	inx++;
    }
    if (retval == -1)
	fprintf
	    (
		stderr,
		"Could not find action %s in internal table.\n", command_str
	    );
    return (retval);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: parse_line                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Find all the commands and parameters on a line and place     */
/*+              these into string arrays for future parsing.                 */
/*+                                                                           */
/*+ Parameters: elements      - char **    - Elements parsed from line.       */
/*+             line          - char *     - Line of input to parse.          */
/*+             mask          - char *     - Mask showing a string.           */
/*+             quotechr      - char       - Defined quote character.         */
/*+                                                                           */
/*+ Input:      line, quotechr                                                */
/*+                                                                           */
/*+ Output:     elements, mask                                                */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 11/07/95    Bruce M. Corwin   Initial Release                             */
/*+                                                                           */
/*+****************************************************************************/
static int parse_line (char **elements, char *line, char *mask, char quotechr)
{
    int inx = 0;
    int jnx = 0;
    int len = 0;
    int argcnt = 0;
    len = strlen (line);
    for (inx = 0; inx < len; inx++)
    {
	if (isspace (line[inx]) && mask[inx] != XMASK)
	{
	    line[inx] = NULLCHR;
	}
    }
    if (len != 0)
	elements[argcnt++] = line;
    for (inx = 0; inx < len; inx++)
    {
	if (line[inx] == NULLCHR)
	    elements[argcnt++] = line + inx + 1;
    }
    elements[argcnt] = NULL;
    for (jnx = 0; jnx < argcnt; jnx++)
    {
	if (elements[jnx][0] == quotechr)
	{
	    elements[jnx]++;
	    elements[jnx][strlen (elements[jnx]) - 1] = NULLCHR;
	}
    }
    return (argcnt);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: singlespcQ                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function removes all extranious spaces or tabs and      */
/*+              replaces them with a single space or tab. EXCEPT within a    */
/*+              double quotes delimited string. This is how this function    */
/*+              differs from the regular singlespc function. It also adds a  */
/*+              simple mask to be used by other functions that will be a     */
/*+              string of Xs where a string resides in the original input    */
/*+              string.                                                      */
/*+                                                                           */
/*+ Parameters: string        - char *     - This is the input string to      */
/*+                                          single space.                    */
/*+             mask          - char *     - Mask showing a string.           */
/*+             quotechr      - char       - Defined quote character.         */
/*+                                                                           */
/*+ Input:      string, quotechr                                              */
/*+                                                                           */
/*+ Output:     mask                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 11/07/95    Bruce M. Corwin   Initial Release                             */
/*+                                                                           */
/*+****************************************************************************/
static void singlespcQ (char *string, char *mask, char quotechr)
{
    int inx = 0;
    int jnx = 0;
    char buffer[BUFSIZ];
    int spccnt = 0;
    int quotemode = 1;
    int oldquote = 1;
    int slashmode = 1;
    int convert = -1;
    for (inx = 0; inx < strlen (string); inx++)
    {
	if (string[inx] == quotechr)
	    quotemode = quotemode * -1;
	if (inx > 0 && string[inx - 1] == BSLASH && string[inx] == quotechr)
	{
	    quotemode = quotemode * -1;
	    if (inx > 1 && string[inx - 2] == BSLASH)
	    {
		quotemode = quotemode * -1;
	    }
	}
	convert = -1;
	if (inx > 0 && string[inx - 1] == BSLASH)
	{
	    jnx--;
	    convert = -1;
	    if (inx > 1 && string[inx - 2] == BSLASH)
	    {
		jnx++;
		convert = 0;
	    }
	    if (convert)
	    {
		switch (string[inx])
		{
		    case ('n'):
			string[inx] = NEWLINE;
			break;
		    case ('t'):
			string[inx] = TAB;
			break;
		}
	    }
	}
	if (isspace (string[inx]))
	    spccnt++;
	else
	    spccnt = 0;
	if (quotemode == -1)
	    spccnt = 0;
	if (spccnt < 2)
	{
	    if (quotemode == -1)
		mask[jnx] = XMASK;
	    else
		mask[jnx] = SPCMASK;
	    if (oldquote == -1 && quotemode == 1)
		mask[jnx] = XMASK;
	    buffer[jnx++] = string[inx];
	}
	oldquote = quotemode;
    }
    buffer[jnx] = NULLCHR;
    mask[jnx] = NULLCHR;
    strcpy (string, buffer);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: XmVtLoadPopup                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine will create all of the menus provided in the    */
/*+              referenced menu file and will make the referenced root menu  */
/*+              the base popup menu.                                         */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             root          - char *     - This is the menu that should be  */
/*+                                          the root menu.                   */
/*+             location      - char *     - This is the location of the menu */
/*+                                          file.                            */
/*+             app           - char *     - This is the application name to  */
/*+                                          obtain the popup menu for.       */
/*+                                                                           */
/*+ Input:      parent, root, location, app                                   */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void XmVtLoadPopup (Widget parent, char *root, char *location, char *app)
{
    XmVtWidget tw = (XmVtWidget) parent;
    char opname[BUFSIZ];
    char name[BUFSIZ];
    char path[BUFSIZ];
    char title[BUFSIZ];
    char exec_str[BUFSIZ];
    int count = 0;
    int inx = 0;
    int next_num = 0;
    Widget new_popup = NULL;
    Widget one_option = NULL;
    MENU *new_menu = NULL;
    char *total_path = NULL;
    char *env = NULL;
    char home[BUFSIZ];

    tw->vt.destroy_count = 0;
    tw->vt.menu_count = 0;
    tw->vt.popup_menu = NULL;
    tw->vt.popup_destroy_list = NULL;

    env = getenv (HOMEPATH);
    if (env != NULL)
	strcpy (home, env);
    else
	strcpy (home, "/users/nohome");
    sprintf
	(path, "/usr/lib/X11/%%T/%%N%%S:%s/%%S:./%%S:%s/.vue/vuewmrc:%s/.mwmrc", home, home, home);
    total_path = (char *) XtResolvePathname
	(XtDisplay (parent), app, "system", location, path, NULL, 0, NULL);
    if (total_path != NULL)
    {
	strcpy (name, root);
	new_menu = load_menu (total_path, &count);
	tw->vt.popup_menu = new_menu;
	tw->vt.menu_count = count;
	strcpy (title, NULLTITLE);
	for (inx = 0; inx < count; inx++)
	{
	    if (
		   strcmp (root, new_menu[inx].menu) == 0 &&
		   new_menu[inx].command == F_TITLE
		)
		strcpy (title, new_menu[inx].description);
	}
	if (strcmp (title, NULLTITLE) != 0)
	{
	    new_popup = SetUpPopupMenu (parent, name, title);
	    StoreOption (parent, new_popup, name, root, new_menu, count);
	}
    }
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: StoreOption                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to add an option onto the current menu. */
/*+              It can add a regular option, a separator or a cascading      */
/*+              button. This can be done for either a popup or a pulldown    */
/*+              menu.                                                        */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             new_popup     - Widget     - The new popup menu stub to add   */
/*+                                          options to.                      */
/*+             name          - char *     - The name of the widget.          */
/*+             root          - char *     - This is the menu that should be  */
/*+                                          the root menu.                   */
/*+             new_menu      - MENU *     - This is the new menu information */
/*+                                          structure.                       */
/*+             count         - int        - Count of characters to write     */
/*+                                                                           */
/*+ Input:      parent, new_popup, name, root, new_menu, count                */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void StoreOption (Widget parent, Widget new_popup, char *name, char *root, MENU * new_menu, int count)
{
    char opname[BUFSIZ];
    char exec_str[BUFSIZ];
    int inx = 0;
    int next_num = 0;
    Widget one_option = NULL;


    for (inx = 0; inx < count; inx++)
    {
	if (strcmp (root, new_menu[inx].menu) == 0)
	{
	    sprintf (opname, "%sPB%d", name, next_num++);
	    sprintf
		(exec_str, "%d:%s", new_menu[inx].command, new_menu[inx].data);

	    switch (new_menu[inx].command)
	    {
		case F_TITLE:
		    break;
		case F_MENU:
		    one_option = CreateOneCascade
			(new_popup, opname, new_menu[inx].description, exec_str, new_menu, count, parent);
		    break;
		case F_EXEC:
		    one_option = CreateOneOption
			(new_popup, opname, new_menu[inx].description, exec_str);
		    break;
		case F_SEPARATOR:
		    one_option = CreateOneSep
			(new_popup, opname);
		    break;
		default:
		    one_option = CreateOneOption
			(new_popup, opname, new_menu[inx].description, exec_str);
		    break;
	    }
	}
    }
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: SetUpPopupMenu                                             */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to set up the initial menu 'stub'. It   */
/*+              prepares the base menu shells and menu widgets and performs  */
/*+              various initialization tasks.                                */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             name          - char *     - The name of the widget.          */
/*+             title         - char *     - Title for new menu.              */
/*+                                                                           */
/*+ Input:      parent, name, title                                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static Widget SetUpPopupMenu (Widget parent, char *name, char *title)
{
    XmVtWidget tw = (XmVtWidget) parent;
    char working_name[BUFSIZ];
    Widget dynamic_menu_shell = NULL;
    Widget dynamic_menu = NULL;
    Widget dynamic_menu2_p1_title = NULL;
    Widget dynamic_menu2_p1_b3 = NULL;

    dynamic_menu_shell = XtVaCreatePopupShell (name,
					     xmMenuShellWidgetClass, parent,
					       XmNwidth, 1,
					       XmNheight, 1,
					       XmNallowShellResize, TRUE,
					       XmNoverrideRedirect, TRUE,
					       NULL);
    StoreMenuShell (parent, dynamic_menu_shell);

    sprintf (working_name, "base_%s", name);
    dynamic_menu = XtVaCreateWidget (working_name,
				 xmRowColumnWidgetClass, dynamic_menu_shell,
				     XmNmenuAccelerator, "<KeyUp>F4",
				     XmNrowColumnType, XmMENU_POPUP,
				     NULL);

    tw->vt.dynamic_menu = dynamic_menu;

    sprintf (working_name, "%s_title", name);
    dynamic_menu2_p1_title = XtVaCreateManagedWidget (working_name,
					   xmLabelWidgetClass, dynamic_menu,
					RES_CONVERT (XmNlabelString, title),
						      NULL);

    sprintf (working_name, "%s_sep", name);
    dynamic_menu2_p1_b3 = XtVaCreateManagedWidget (working_name,
				       xmSeparatorWidgetClass, dynamic_menu,
					    XmNseparatorType, XmDOUBLE_LINE,
						   NULL);

    XtAddEventHandler (parent, ButtonPressMask,
		       False, PopupMenuPost, dynamic_menu);
    tw->vt.menu_event_added = True;
    return (dynamic_menu);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: XmVtUnloadPopup                                            */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine will remove a previously created Vt Popup menu  */
/*+              created with the XmVtLoadPopup routine. It frees memory and  */
/*+              destroys menu widgets. This routine is used when either of   */
/*+              the resources RootMenu or MenuFile are changed. This should  */
/*+              be called before a new XmVtLoadPopup is called.              */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+                                                                           */
/*+ Input:      parent                                                        */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void XmVtUnloadPopup (Widget parent)
{
    XmVtWidget tw = (XmVtWidget) parent;
    int inx = 0;

    if (tw->vt.menu_event_added)
    {
	XtRemoveEventHandler (parent, ButtonPressMask,
			      False, PopupMenuPost, tw->vt.dynamic_menu);
    }
    free_menu (tw->vt.popup_menu, tw->vt.menu_count);
    for (inx = 0; inx < tw->vt.destroy_count; inx++)
    {
	XtDestroyWidget ((tw->vt.popup_destroy_list)[inx]);
    }
    free (tw->vt.popup_destroy_list);
    tw->vt.popup_destroy_list = NULL;
    tw->vt.destroy_count = 0;
    tw->vt.menu_count = 0;
    tw->vt.menu_event_added = False;
    tw->vt.popup_menu = NULL;
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: PopupMenuPost                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function displays a dynamic menu.                       */
/*+                                                                           */
/*+ Parameters: wgt           - Widget     - Input widget to callback.        */
/*+             client_data   - XtPointer  - Data for callback                */
/*+             event         - XEvent *   - X event for situation            */
/*+             ctd           - Boolean *  - This is a boolean value sent to  */
/*+                                          the dynamic instantiation        */
/*+                                          routine.                         */
/*+                                                                           */
/*+ Input:      wgt, client_data, event, ctd                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void PopupMenuPost (Widget wgt, XtPointer client_data, XEvent * event, Boolean * ctd)
{
    Widget menu = (Widget) client_data;
    int which_button;

    XtVaGetValues (menu, XmNwhichButton, &which_button, NULL);

    if (event->xbutton.button == which_button)
    {
	XmMenuPosition (menu, (XButtonPressedEvent *) event);
	XtManageChild (menu);
    }
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: CreateOneOption                                            */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is used to create a basic button option on a   */
/*+              popup menu.                                                  */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             name          - char *     - The name of the widget.          */
/*+             label         - char *     - Label for a button on the menu.  */
/*+             value         - char *     - Value to paste                   */
/*+                                                                           */
/*+ Input:      parent, name, label, value                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static Widget CreateOneOption (Widget parent, char *name, char *label, char *value)
{
    char *storevalue = NULL;
    Widget dynamic_menu_pb = NULL;
    dynamic_menu_pb = XtVaCreateManagedWidget (name,
					    xmPushButtonWidgetClass, parent,
					RES_CONVERT (XmNlabelString, label),
					       NULL);
    storevalue = (char *) malloc (strlen (value) + 1);
    strcpy (storevalue, value);
    XtAddCallback (dynamic_menu_pb, XmNactivateCallback,
		   (XtCallbackProc) PopupCallback,
		   storevalue);
    return (dynamic_menu_pb);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: CreateOneCascade                                           */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is used to create a cascade button on a popup  */
/*+              menu.                                                        */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             name          - char *     - The name of the widget.          */
/*+             label         - char *     - Label for a button on the menu.  */
/*+             value         - char *     - Value to paste                   */
/*+             menu_array    - MENU *     - This is the main menu array to   */
/*+                                          create sub menus with.           */
/*+             count         - int        - Count of characters to write     */
/*+             base          - Widget     - Base angle for hue               */
/*+                                                                           */
/*+ Input:      parent, name, label, value, menu_array, count, base           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static Widget CreateOneCascade (Widget parent, char *name, char *label, char *value, MENU * menu_array, int count, Widget base)
{
    char *storevalue = NULL;
    Widget new_dynamic_menu = NULL;
    Widget dynamic_menu_shell = NULL;
    Widget dynamic_menu_pb = NULL;
    Widget dynamic_menu2_p1_title = NULL;
    Widget dynamic_menu2_p1_b3 = NULL;
    char working_name[BUFSIZ];
    char title[BUFSIZ];
    int inx = 0;
    char *dummy = NULL;
    char *root = NULL;


    dummy = strtok (value, ":");
    root = strtok (NULL, "");

    sprintf (working_name, "shell_for_%s", name);

    dynamic_menu_shell = XtVaCreatePopupShell (working_name,
					       xmMenuShellWidgetClass, base,
					       XmNwidth, 1,
					       XmNheight, 1,
					       XmNallowShellResize, TRUE,
					       XmNoverrideRedirect, TRUE,
					       NULL);

    StoreMenuShell (base, dynamic_menu_shell);

    sprintf (working_name, "next_base_%s", name);
    new_dynamic_menu = XtVaCreateWidget (working_name,
				 xmRowColumnWidgetClass, dynamic_menu_shell,
					 XmNmenuAccelerator, "<KeyUp>F4",
					 XmNrowColumnType, XmMENU_PULLDOWN,
					 NULL);

    sprintf (working_name, "%s_next_title", name);

    strcpy (title, "NULL");
    for (inx = 0; inx < count; inx++)
    {
	if (
	       strcmp (root, menu_array[inx].menu) == 0 &&
	       menu_array[inx].command == F_TITLE
	    )
	    strcpy (title, menu_array[inx].description);
    }

    if (strcmp (title, "NULL") != 0)
    {
	dynamic_menu2_p1_title = XtVaCreateManagedWidget (working_name,
				       xmLabelWidgetClass, new_dynamic_menu,
					RES_CONVERT (XmNlabelString, title),
							  NULL);

	sprintf (working_name, "%s_next_sep", name);
	dynamic_menu2_p1_b3 = XtVaCreateManagedWidget (working_name,
				   xmSeparatorWidgetClass, new_dynamic_menu,
					    XmNseparatorType, XmDOUBLE_LINE,
						       NULL);

    }

    sprintf (working_name, "%s_sub_options", name);
    StoreOption (base, new_dynamic_menu, working_name, root, menu_array, count);





    dynamic_menu_pb = XtVaCreateManagedWidget (name,
					 xmCascadeButtonWidgetClass, parent,
					RES_CONVERT (XmNlabelString, label),
					     XmNsubMenuId, new_dynamic_menu,
					       NULL);
    storevalue = (char *) malloc (strlen (value) + 1);
    strcpy (storevalue, value);
    XtAddCallback (dynamic_menu_pb, XmNactivateCallback,
		   (XtCallbackProc) PopupCallback,
		   storevalue);
    return (dynamic_menu_pb);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: CreateOneSep                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is used to create a separator on a popup menu. */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             name          - char *     - The name of the widget.          */
/*+                                                                           */
/*+ Input:      parent, name                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static Widget CreateOneSep (Widget parent, char *name)
{
    Widget dynamic_menu_sep = NULL;

    dynamic_menu_sep = XtVaCreateManagedWidget (name,
					     xmSeparatorWidgetClass, parent,
						NULL);

    return (dynamic_menu_sep);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: PopupCallback                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is the callback for all of the basic and       */
/*+              cascading menu buttons for a popup menu in the Vt widget.    */
/*+                                                                           */
/*+ Parameters: sent_widget   - Widget     - This is the widget that is       */
/*+                                          sending the callback.            */
/*+             client_dat    - char *     - Client data passed to callback   */
/*+             call_back     - XtPointer  - This variable defines callback   */
/*+                                          reasons.                         */
/*+                                                                           */
/*+ Input:      sent_widget, client_dat, call_back                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void PopupCallback (Widget sent_widget, char *client_dat, XtPointer call_back)
{
    char *com_str = NULL;
    char *data = NULL;
    int command = 0;
    char buffer[BUFSIZ];
    char modified[BUFSIZ];
    Widget posting_widget = NULL;
    posting_widget = XtParent (XtParent (XtParent (sent_widget)));
    strcpy (buffer, (char *) client_dat);
    com_str = strtok (buffer, ":");
    data = strtok (NULL, "");
    sscanf (com_str, "%d", &command);

    switch (command)
    {
	case F_MENU:
	    break;
	case F_WRITE:
	    XmVtWrite (posting_widget, data, strlen (data));
	    break;
	case F_EXEC:
	    sprintf (modified, "%s &\n", data);
	    XmVtSend (posting_widget, modified);
	    break;
	case F_SEND_CR:
	    sprintf (modified, "%s\n", data);
	    XmVtSend (posting_widget, modified);
	    break;
	case F_SEND:
	    XmVtSend (posting_widget, data);
	    break;
	case F_QUIT:
	case F_QUIT_MWM:
	    exit (0);
	    break;
    }
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: StoreMenuShell                                             */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to store the main menu shells for a     */
/*+              complete heirarchial menu structure. These menu shells are   */
/*+              then used to destroy the completed menu when the             */
/*+              XmVtUnloadPopup function is called.                          */
/*+                                                                           */
/*+ Parameters: parent        - Widget     - Parent widget for new menu.      */
/*+             shell         - Widget     - Menu shell widget to store away  */
/*+                                          in the Vt widget.                */
/*+                                                                           */
/*+ Input:      parent, shell                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void StoreMenuShell (Widget parent, Widget shell)
{
    XmVtWidget tw = (XmVtWidget) parent;
    int destroy_count = 0;


    destroy_count = tw->vt.destroy_count;
    if (destroy_count == 0)
    {
	tw->vt.popup_destroy_list = malloc (sizeof (Widget));
    }
    else
    {
	tw->vt.popup_destroy_list =
	    realloc (
			tw->vt.popup_destroy_list,
			sizeof (Widget) * (destroy_count + 1)
	    );
    }
    tw->vt.popup_destroy_list[destroy_count] = shell;
    (tw->vt.destroy_count)++;
}
