/*+****************************************************************************/
/*+                                                                           */
/*+ Copyright (c) 1992-1997 Bruce M. Corwin                                   */
/*+                                                                           */
/*+****************************************************************************/
/*+****************************************************************************/
/*+                                                                           */
/*+ Module Name: screen.c                                                     */
/*+                                                                           */
/*+ Program ID:  vtwidget                                                     */
/*+                                                                           */
/*+ Functions:  init_data, reset, write_s, chop,                              */
/*+             insert_char, non_print, parse_esc,                            */
/*+             parse_color, parse_hls, isforcing,                            */
/*+             equalize, get_red, get_green, get_blue,                       */
/*+             calc_ix, insert_blanks, right_edge,                           */
/*+             delete_lines, insert_lines,                                   */
/*+             reverse_index, bottom_edge, tab_over,                         */
/*+             delete_char, half_scroll_up, scroll_up,                       */
/*+             clear_eol, clear_bol, clear_aol,                              */
/*+             clear_up, clear_down, clear_all,                              */
/*+             eight_bit, strcatch, sync_cursor,                             */
/*+             execute_esc, message, special_message,                        */
/*+             speccmp, sub_proc_one, sub_proc_three,                        */
/*+             sub_proc_two, reserve, fixrems,                               */
/*+             multi_attr, cup_move, cuu_move, cud_move,                     */
/*+             cuf_move, cub_move, dynamic_tmpnam,                           */
/*+             isnonview, self_test, newline,                                */
/*+             cancel_eol, tab_clear, tab_set,                               */
/*+             tab_clear_all, resize_data, free_data,                        */
/*+             free_vtscreen, alc_vtscreen, s_attr,                          */
/*+             s_data, s_line_attr, s_cset, s_flag,                          */
/*+             g_attr, g_data, g_line_attr, g_cset,                          */
/*+             g_flag, s_common, s_swap, s_modify,                           */
/*+             s_multi, paint_data                                           */
/*+                                                                           */
/*+****************************************************************************/

/***************************************************************/
/* The following are the include directives for this module.   */
/***************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#ifndef SIMPLE
#include         <Xm/XmP.h>
#include         <Xm/ScrolledW.h>
#endif
#include         "screen.h"
#include         "VtP.h"
/*************************************************************/
/* End of include directives for this module.                */
/*************************************************************/
/*************************************************************/
/* The following are the define directives for this module.  */
/*************************************************************/
#ifdef   LPR
#define  DATA_COM           ("lpr")
#else
#define  DATA_COM           ("lp")
#endif

#define  ABM                ( 5 )
#define  BELL               ( 7 )
#define  BACKSPACE          ( 8 )
#define  TAB                ( 9 )
#define  LINEFEED           ( 10 )
#define  NP                 ( 11 )
#define  VT                 ( 12 )
#define  RETURN             ( 13 )
#define  OPENCS             ( 14 )
#define  CLOSECS            ( 15 )
#define  ESC                ( 27 )
#define  CSI                ( 0x9B )
#define  RI                 ( 0x8D )

#define  BRACKET            ( '[' )
#define  MINX               ( 1 )
#define  MINY               ( 1 )
#define  MAXX               ( 80 )
#define  MAXY               ( 25 )
#define  S_MOD              ( 1 )
#define  S_SAME             ( 0 )
#define  WRITEMODE          ( "w" )
#define  DEBUGFL            ( "Vt_debug.txt" )
#define  CS_USASCII         ( 0 )
#define  CS_LINES           ( 1 )
#define  CS_UKASCII         ( 2 )
#define  W_NORMAL           ( 0 )
#define  W_SCROLL           ( 1 )
#define  W_HSCROLL          ( 2 )
#define  RED                ( 120 )
#define  GREEN              ( 240 )
#define  BLUE               ( 0 )
#define  MAXRGB             ( 255 )
#ifndef  TRUE
#define  TRUE               ( -1 )
#endif
#ifndef  FALSE
#define  FALSE              ( 0 )
#endif
/*************************************************************/
/* End of define directives for this module.                 */
/*************************************************************/
/*************************************************************/
/* The following are provided as an external API for a Vt.c  */
/*************************************************************/
void init_data (XmVtWidget tw, int maxx_value, int maxy_value, int minx_value, int miny_value);
int write_s (XmVtWidget tw, char *buffer, int count);
int strcatch (char *string, int data);
void resize_data (XmVtWidget tw, int maxx_value, int maxy_value);
void free_data (XmVtWidget tw);
/*************************************************************/
/* These particular API routines access low level data screen */
/* information and should always be checked when changes are */
/* made to such structures.                                  */
/*************************************************************/
void s_attr (XmVtWidget tw, int xc, int yc, attribute attr);
void s_data (XmVtWidget tw, int xc, int yc, int data);
void s_cset (XmVtWidget tw, int xc, int yc, int cset);
void s_flag (XmVtWidget tw, int xc, int yc, int flag);
void s_line_attr (XmVtWidget tw, int yc, int ldata);
int s_multi (XmVtWidget tw, char *buffer, int count);
void s_common (XmVtWidget tw, int xc, int yc, int count, attribute attr, int data, int cset);
void s_swap (XmVtWidget tw, int destination, int source);
void s_modify (XmVtWidget tw, int line_pos);
attribute g_attr (XmVtWidget tw, int xc, int yc);
int g_data (XmVtWidget tw, int xc, int yc);
int g_cset (XmVtWidget tw, int xc, int yc);
int g_flag (XmVtWidget tw, int xc, int yc);
int g_line_attr (XmVtWidget tw, int yc);
void paint_data (XmVtWidget tw);
/*************************************************************/
/* End of functions for external API for Vt.c module         */
/*************************************************************/
/*************************************************************/
/* The following are the API from an external Vt.c module    */
/*************************************************************/
extern char *convert_esc (char *format, char *addon);
extern void ok_update (XmVtWidget tw);
extern void ok_revive (XmVtWidget tw);
extern void message_do (XmVtWidget tw);
extern void special_msg_do (XmVtWidget tw, char *inputdata);
extern void beep (XmVtWidget tw);
extern int send_keys (XmVtWidget tw, char *buffer);
extern void place_cursor (XmVtWidget tw);
extern void remove_cursor (XmVtWidget tw);
extern void showgraphic (XmVtWidget tw);
extern void change_colors (XmVtWidget tw, char *rgb, int map, int mapping, int bintense, int screen_flag, int writing_flag);
extern void change_width (XmVtWidget tw, int scr_width);
extern void answer (XmVtWidget tw);
extern int plot (XmVtWidget tw, char *pldata, attribute attr, char cset, int xcoord, int ycoord);
/*************************************************************/
/* End of Vt.c API functions                                 */
/*************************************************************/
/*************************************************************/
/* The following are low level internal routines.            */
/* These routines access the low level data structures.      */
/* These should be checked like the above API routines if    */
/* low level screen data structures are changed.             */
/*************************************************************/
static void free_vtscreen (int row, int strwidth, VTLINE * vtscreen_name);
static VTLINE *alc_vtscreen (int row, int strwidth);
/*************************************************************/
/* End of low level internal functions.                      */
/*************************************************************/
/*************************************************************/
/* The following are normal internal functions.              */
/*************************************************************/
static void self_test (XmVtWidget tw);
static void newline (XmVtWidget tw);
static void cancel_eol (XmVtWidget tw);
static void message (XmVtWidget tw);
static int parse_hls (XmVtWidget tw, char *input, char *rgb, int *map);
static int isforcing (char single);
static int speccmp (char *firststr, char *secondstr, int len, int *alen);
static int insert_char (XmVtWidget tw, int single);
static int non_print (XmVtWidget tw, int single);
static int parse_esc (XmVtWidget tw, int single);
static void special_message (XmVtWidget tw);
static void fixrems (char *string, int first, int last);
static void reserve (XmVtWidget tw, int first, int last);
static void delete_lines (XmVtWidget tw, int number);
static void tab_over (XmVtWidget tw);
static void reset (XmVtWidget tw);
static void cup_move (XmVtWidget tw, int xc, int yc);
static void cuu_move (XmVtWidget tw, int number);
static void cud_move (XmVtWidget tw, int number);
static void cub_move (XmVtWidget tw, int number);
static void cuf_move (XmVtWidget tw, int number);
static void multi_attr (XmVtWidget tw, char *mattr);
static void right_edge (XmVtWidget tw);
static void bottom_edge (XmVtWidget tw);
static void clear_all (XmVtWidget tw);
static void eight_bit (XmVtWidget tw);
static void reverse_index (XmVtWidget tw);
static void insert_lines (XmVtWidget tw, int number);
static void insert_blanks (XmVtWidget tw, int number);
static void clear_down (XmVtWidget tw);
static void clear_up (XmVtWidget tw);
static void clear_eol (XmVtWidget tw);
static void clear_bol (XmVtWidget tw);
static void clear_aol (XmVtWidget tw);
static void tab_clear (XmVtWidget tw);
static void tab_set (XmVtWidget tw);
static void tab_clear_all (XmVtWidget tw);
static void delete_char (XmVtWidget tw, int number);
static void sync_cursor (XmVtWidget tw);
static void scroll_up (XmVtWidget tw);
static void half_scroll_up (XmVtWidget tw);
static int execute_esc (XmVtWidget tw);
static int sub_proc_one (XmVtWidget tw);
static int sub_proc_two (XmVtWidget tw);
static int sub_proc_three (XmVtWidget tw);
static int parse_color (XmVtWidget tw);
static int get_red (int hue, int lum, int sat);
static int get_green (int hue, int lum, int sat);
static int get_blue (int hue, int lum, int sat);
static void equalize (int *red, int *green, int *blue);
static int calc_ix (int base, int inhue);
static char *dynamic_tmpnam (void);
static int isnonview (unsigned int value);
static int chop (XmVtWidget tw, char *buffer, int count);
/*************************************************************/
/* End of internal functions.                                */
/*************************************************************/
/*************************************************************/
/* The following are global varables.                        */
/*************************************************************/
FILE *debug_fp = stdout;
attribute clr_attr = BLANK_ATTR;
/*************************************************************/
/* End of global varables.                                   */
/*************************************************************/
/*************************************************************/
/* The following are static structures.                      */
/*************************************************************/
static int deflums[16] =
{0, 50, 100, 50, 50, 50, 50, 46, 26, 46, 43, 46, 46, 46, 46, 80};
static int defhues[16] =
{0, 0, 120, 240, 60, 300, 180, 0, 0, 0, 120, 240, 60, 300, 180, 0};
static int defsats[16] =
{0, 60, 71, 60, 60, 60, 60, 0, 0, 28, 37, 28, 28, 28, 28, 0};
/*************************************************************/
/* End of static structures.                                 */
/*************************************************************/

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: init_data                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: init_data initializes all the internal data structures for   */
/*+              the screen. It allocates space sets the cursor in places and */
/*+              clears the screen.                                           */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             maxx_value    - int        - Maximum X coord Value            */
/*+             maxy_value    - int        - Maximum Y coord Value            */
/*+             minx_value    - int        - Minimum X coord Value            */
/*+             miny_value    - int        - Minimum Y coord Value            */
/*+                                                                           */
/*+ Input:      tw, maxx_value, maxy_value, minx_value, miny_value            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void init_data (XmVtWidget tw, int maxx_value, int maxy_value, int minx_value, int miny_value)
{
    int inx = 0;
    int jnx = 0;
    int maxhist = tw->vt.max_history;

    tw->vt.data_name = NULL;
    tw->vt.graphics_name = NULL;
    tw->vt.gr_cset_level = 0;
    tw->vt.pass_thru = FALSE;
    tw->vt.newline_mode = FALSE;
    tw->vt.cursor_mode = TRUE;
    tw->vt.smooth_mode = FALSE;
    tw->vt.mouse_mode = FALSE;
    tw->vt.application_key_mode = FALSE;
    tw->vt.inverse_mode = FALSE;
    tw->vt.autowrap_mode = TRUE;
    tw->vt.onlcr_mode = FALSE;
    tw->vt.insert_mode = FALSE;
    tw->vt.decom_mode = FALSE;
    tw->vt.last_ansi_fg = 0;
    tw->vt.last_ansi_bg = 0;
    tw->vt.save_ansi_fg = 0;
    tw->vt.save_ansi_bg = 0;
    tw->vt.stored_cx = 0;
    tw->vt.stored_cy = 0;
    tw->vt.minx = minx_value;
    tw->vt.miny = miny_value;
    tw->vt.maxx = maxx_value;
    tw->vt.maxy = maxy_value;
    tw->vt.real_miny = miny_value;
    tw->vt.real_maxy = maxy_value;
    tw->vt.real_minx = minx_value;
    tw->vt.real_maxx = maxx_value;
    tw->vt.vtscreen = alc_vtscreen (tw->vt.maxy + 1 + maxhist, tw->vt.maxx + 1);
    tw->vt.screen_cx = tw->vt.minx;
    tw->vt.screen_cy = tw->vt.miny;
    tw->vt.save_cx = 1;
    tw->vt.save_cy = 1;
    reset_attr (&(tw->vt.attributes));
    reset_attr (&(tw->vt.save_attributes));
    tw->vt.char_set[0] = CS_USASCII;
    tw->vt.char_set[1] = CS_LINES;
    for (inx = 0; inx < NUMMAPS; inx++)
    {
	tw->vt.lums[inx] = deflums[inx];
	tw->vt.sats[inx] = defsats[inx];
	tw->vt.hues[inx] = defhues[inx];
    }
    tw->vt.poundflag = FALSE;
    tw->vt.eflag = FALSE;
    tw->vt.bflag = FALSE;
    tw->vt.qflag = FALSE;
    tw->vt.fflag = FALSE;
    tw->vt.dflag = FALSE;
    tw->vt.sflag = FALSE;
    tw->vt.jflag = FALSE;
    tw->vt.pflag = FALSE;
    tw->vt.aflag = FALSE;
    tw->vt.nflag = FALSE;
    tw->vt.esc_sequence[0] = 0;
    tw->vt.first_row = tw->vt.max_history;
    tw->vt.scroll_value = 0;
    for (inx = 1; inx <= tw->vt.maxx; inx++)
	if ((inx - 1) % 8)
	    tw->vt.tabs[inx] = ' ';
	else
	    tw->vt.tabs[inx] = 'I';
    tw->vt.tabs[tw->vt.maxx + 1] = 0;
    for (jnx = 0; jnx <= (tw->vt.real_maxy + tw->vt.max_history); jnx++)
    {
	for (inx = tw->vt.real_minx; inx <= tw->vt.real_maxx; inx++)
	{
	    s_data (tw, inx, jnx, ' ');
	    s_attr (tw, inx, jnx, clr_attr);
	    s_cset (tw, inx, jnx, CS_USASCII);
	    s_flag (tw, inx, jnx, S_MOD);
	}
    }
    if (!tw->vt.debug_stdout && tw->vt.debug)
	debug_fp = fopen (DEBUGFL, WRITEMODE);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: reset                                                      */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Reset initializes all of the soft parameters for the screen. */
/*+              This includes where the cursor is and clearing the screen as */
/*+              well as setting insert and attributes.                       */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 11/07/95    Bruce M. Corwin   Initial Release                             */
/*+ 04/22/97    Bruce M. Corwin   Made sure to initialize more variables.     */
/*+                                                                           */
/*+****************************************************************************/
static void reset (XmVtWidget tw)
{
    int inx = 0;
    tw->vt.last_ansi_fg = 0;
    tw->vt.last_ansi_bg = 0;
    tw->vt.save_ansi_fg = 0;
    tw->vt.save_ansi_bg = 0;
    tw->vt.minx = tw->vt.real_minx;
    tw->vt.miny = tw->vt.real_miny;
    tw->vt.maxx = tw->vt.real_maxx;
    tw->vt.maxy = tw->vt.real_maxy;
    tw->vt.screen_cx = tw->vt.minx;
    tw->vt.screen_cy = tw->vt.miny;
    tw->vt.save_cx = 1;
    tw->vt.save_cy = 1;
    tw->vt.stored_cx = 0;
    tw->vt.stored_cy = 0;
    tw->vt.mouse_mode = FALSE;
    tw->vt.smooth_mode = FALSE;
    tw->vt.autowrap_mode = TRUE;
    tw->vt.onlcr_mode = FALSE;
    tw->vt.insert_mode = FALSE;
    tw->vt.cursor_mode = TRUE;
    tw->vt.decom_mode = FALSE;
    tw->vt.save_cset_level = 0;
    tw->vt.application_key_mode = FALSE;
    reset_attr (&(tw->vt.attributes));
    reset_attr (&(tw->vt.save_attributes));
    tw->vt.char_set[0] = CS_USASCII;
    tw->vt.char_set[1] = CS_LINES;
    tw->vt.gr_cset_level = 0;
    for (inx = 1; inx <= tw->vt.maxx; inx++)
	if ((inx - 1) % 8)
	    tw->vt.tabs[inx] = ' ';
	else
	    tw->vt.tabs[inx] = 'I';
    tw->vt.tabs[tw->vt.maxx + 1] = 0;
    for (inx = 0; inx < NUMMAPS; inx++)
    {
	tw->vt.lums[inx] = deflums[inx];
	tw->vt.sats[inx] = defsats[inx];
	tw->vt.hues[inx] = defhues[inx];
    }
    clear_all (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: write_s                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: write_s works just like write except no file descriptor is   */
/*+              passed because write_s only writes to the screen.            */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             buffer        - char *     - Buffer of output data            */
/*+             count         - int        - Count of characters to write     */
/*+                                                                           */
/*+ Input:      tw, buffer, count                                             */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int write_s (XmVtWidget tw, char *buffer, int count)
{
    char *pointer = NULL;
    int rcode = 0;
    int inx = 0;
    int jnx = 0;
    char temp = 0;
    int reference = 0;

    reference = tw->vt.first_row;
    remove_cursor (tw);
    buffer[count] = 0;
    jnx = count;
    pointer = buffer;
    while (jnx != 0)
    {
	inx = chop (tw, pointer, jnx);
	jnx = jnx - inx;
	pointer = pointer + inx;
    }
    sync_cursor (tw);
    bottom_edge (tw);

    ok_update (tw);

    if (tw->vt.cursor_mode)
	place_cursor (tw);

    if (reference != tw->vt.first_row)
    {
	ok_revive (tw);
    }
    return (count);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: chop                                                       */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is used to chop an input buffer up into as     */
/*+              large of blocks as possible that can be digested for output  */
/*+              into the widget screen data. The bigger the chunks the       */
/*+              better the performance because these routines are called     */
/*+              often because of the fact that this is the main data stream. */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             buffer        - char *     - Buffer of output data            */
/*+             count         - int        - Count of characters to write     */
/*+                                                                           */
/*+ Input:      tw, buffer, count                                             */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int chop (XmVtWidget tw, char *buffer, int count)
{
    int inx = 0;
    if (tw->vt.pass_thru && !(tw->vt.eflag) && buffer[inx] != ESC)
	fprintf (tw->vt.data_fp, "%c", buffer[inx]);
    if (
	   !isnonview (buffer[inx]) &&
	   !(tw->vt.eflag) &&
	   !(tw->vt.pass_thru)
	)
    {
	if (tw->vt.insert_mode)
	{
	    insert_char (tw, (unsigned char) buffer[inx]);
	    inx++;
	}
	else
	{
	    while (inx < count && !isnonview (buffer[inx]))
		inx++;
	    inx = s_multi (tw, buffer, inx);
	}
    }
    else
    {
	non_print (tw, (unsigned char) buffer[inx]);
	inx++;
	bottom_edge (tw);
    }
    return (inx);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: insert_char                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: insert_char inserts one character at the current position.   */
/*+              It will move characters out of the way to make room for the  */
/*+              character to be printed.                                     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             single        - int        - Single chr to print              */
/*+                                                                           */
/*+ Input:      tw, single                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int insert_char (XmVtWidget tw, int single)
{
    int jnx = 0;
    int l_data = 0;
    attribute l_attr = BLANK_ATTR;
    int l_flag = 0;
    int l_cset = 0;
    static int lastcnt = 0;
    char current_cs = CS_USASCII;
    current_cs = tw->vt.char_set[tw->vt.gr_cset_level];
    if (tw->vt.screen_cx == tw->vt.maxx)
	lastcnt++;
    else
	lastcnt = 0;
    if (lastcnt > 1)
	tw->vt.screen_cx++;
    right_edge (tw);
    for (jnx = tw->vt.maxx; jnx > tw->vt.screen_cx; jnx--)
    {
	l_data = g_data (tw, jnx - 1, tw->vt.screen_cy);
	s_data (tw, jnx, tw->vt.screen_cy, l_data);
	l_attr = g_attr (tw, jnx - 1, tw->vt.screen_cy);
	s_attr (tw, jnx, tw->vt.screen_cy, l_attr);
	l_cset = g_cset (tw, jnx - 1, tw->vt.screen_cy);
	s_cset (tw, jnx, tw->vt.screen_cy, l_cset);
	s_flag (tw, jnx, tw->vt.screen_cy, S_MOD);
    }
    s_data (tw, tw->vt.screen_cx, tw->vt.screen_cy, single);
    s_flag (tw, tw->vt.screen_cx, tw->vt.screen_cy, S_MOD);
    s_attr (tw, tw->vt.screen_cx, tw->vt.screen_cy, tw->vt.attributes);
    s_cset (tw, tw->vt.screen_cx, tw->vt.screen_cy, current_cs);
    tw->vt.screen_cx++;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: non_print                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Execute the non printable character actions like Carage      */
/*+              return, line feed and escape.                                */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             single        - int        - Single chr to print              */
/*+                                                                           */
/*+ Input:      tw, single                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int non_print (XmVtWidget tw, int single)
{
    int rcode = 0;
    int permit_bell = -1;
    if (tw->vt.eflag)
    {
	parse_esc (tw, single);
	permit_bell = 0;
    }
    if (!tw->vt.pass_thru)
    {
	if (single == ABM)
	    answer (tw);
	if (single == LINEFEED && tw->vt.onlcr_mode)
	    tw->vt.screen_cx = 1;
	if (single == LINEFEED || single == NP || single == VT)
	{
	    tw->vt.screen_cy++;
	}
	if (single == BELL)
	{
	    if (permit_bell)
		beep (tw);
	}
	if (single == BACKSPACE)
	{
	    cancel_eol (tw);
	    tw->vt.screen_cx--;
	}
	if (single == TAB)
	    tab_over (tw);
	if (single == RETURN)
	{
	    cancel_eol (tw);
	    tw->vt.screen_cx = 1;
	}
	if (single == RI)
	    reverse_index (tw);
	if (single == OPENCS)
	{
	    tw->vt.gr_cset_level = 1;
	}
	if (single == CLOSECS)
	{
	    tw->vt.gr_cset_level = 0;
	}
    }
    if (single == ESC)
    {
	if (tw->vt.eflag)
	{
	    tw->vt.eflag = FALSE;
	    tw->vt.leftcnt = 0;
	    tw->vt.rightcnt = 0;
	    tw->vt.cflag = 0;
	    tw->vt.poundflag = 0;
	    tw->vt.bflag = 0;
	    tw->vt.qflag = 0;
	    tw->vt.fflag = 0;
	    tw->vt.dflag = 0;
	    tw->vt.sflag = 0;
	    tw->vt.jflag = 0;
	    tw->vt.pflag = 0;
	    tw->vt.aflag = 0;
	    tw->vt.nflag = 0;
	    tw->vt.color_flag = 0;
	    tw->vt.spcflag = 0;
	    tw->vt.esc_sequence[0] = 0;
	}
	else
	{
	    tw->vt.eflag = TRUE;
	}
    }
    if (single == CSI)
    {
	tw->vt.eflag = TRUE;
	parse_esc (tw, BRACKET);
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: parse_esc                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: parse_esc isolates an escape sequence to be later executed   */
/*+              by execute_esc.                                              */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             single        - int        - Single chr to print              */
/*+                                                                           */
/*+ Input:      tw, single                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int parse_esc (XmVtWidget tw, int single)
{
    char send[BUFSIZ];
    char sys_command[BUFSIZ];
    int two_char_command = 0;
    if (single == '[')
	tw->vt.bflag = TRUE;
    if (single == '?' && tw->vt.bflag)
	tw->vt.qflag = TRUE;
    if (isdigit (single))
    {
	tw->vt.dflag = TRUE;
	tw->vt.poundflag = FALSE;
    }
    if (single == '#')
	tw->vt.poundflag = TRUE;
    if (single == ';')
	tw->vt.sflag = TRUE;
    if (single == '}' && !tw->vt.bflag)
	tw->vt.cflag = TRUE;
    if (single == ']')
	tw->vt.cflag = TRUE;
    if (single == '+')
	tw->vt.jflag = TRUE;
    if (single == '*')
	tw->vt.aflag = TRUE;
    if (single == ' ')
	tw->vt.spcflag = TRUE;
    if (single == ')')
    {
	tw->vt.nflag = TRUE;
	(tw->vt.rightcnt)++;
    }
    if (single == '(')
    {
	tw->vt.pflag = TRUE;
	(tw->vt.leftcnt)++;
    }
    if (single == 'P')
	tw->vt.color_flag = TRUE;
    two_char_command = tw->vt.nflag ||
	tw->vt.poundflag ||
	tw->vt.jflag ||
	tw->vt.aflag ||
	tw->vt.pflag ||
	tw->vt.cflag ||
	tw->vt.color_flag ||
	tw->vt.spcflag ||
	tw->vt.bflag;
    if (!two_char_command)
	tw->vt.fflag = TRUE;
    if (tw->vt.bflag && !(tw->vt.cflag) && (isalpha (single) || single == '@' || single == '~' || single == '}'))
	tw->vt.fflag = TRUE;
    if (tw->vt.nflag && !(tw->vt.cflag) && !(tw->vt.color_flag) && isalnum (single))
	tw->vt.fflag = TRUE;
    if (tw->vt.jflag && !(tw->vt.cflag) && isalnum (single))
	tw->vt.fflag = TRUE;
    if (tw->vt.aflag && !(tw->vt.cflag) && isalnum (single))
	tw->vt.fflag = TRUE;
    if (tw->vt.spcflag && !(tw->vt.cflag) && isalnum (single))
	tw->vt.fflag = TRUE;
    if (tw->vt.pflag && !(tw->vt.cflag) && !(tw->vt.color_flag) && isalnum (single))
	tw->vt.fflag = TRUE;
    if (tw->vt.cflag && !isprint (single))
	tw->vt.fflag = TRUE;
    if (!tw->vt.cflag && tw->vt.color_flag && tw->vt.leftcnt > 0 && tw->vt.leftcnt == tw->vt.rightcnt)
	tw->vt.fflag = TRUE;
    strcatch (tw->vt.esc_sequence, single);
    if (tw->vt.fflag)
    {
	if (!tw->vt.pass_thru)
	    execute_esc (tw);
	else
	{
	    if (strcmp (tw->vt.esc_sequence, vt100[PRINTER_OFF]) == 0)
	    {
		tw->vt.pass_thru = FALSE;
		fclose (tw->vt.data_fp);
		tw->vt.data_fp = NULL;
		sprintf (sys_command, "%s %s", DATA_COM, tw->vt.data_name);
		system (sys_command);
		unlink (tw->vt.data_name);
		free (tw->vt.data_name);
		tw->vt.data_name = NULL;
	    }
	    if (strcmp (tw->vt.esc_sequence, vt100[GRAPHIC_OFF]) == 0)
	    {
		tw->vt.pass_thru = FALSE;
		fclose (tw->vt.data_fp);
		tw->vt.data_fp = NULL;
		showgraphic (tw);
		unlink (tw->vt.graphics_name);
		free (tw->vt.graphics_name);
		tw->vt.graphics_name = NULL;
	    }
	}
	tw->vt.leftcnt = 0;
	tw->vt.rightcnt = 0;
	tw->vt.cflag = 0;
	tw->vt.poundflag = 0;
	tw->vt.eflag = 0;
	tw->vt.bflag = 0;
	tw->vt.qflag = 0;
	tw->vt.fflag = 0;
	tw->vt.dflag = 0;
	tw->vt.sflag = 0;
	tw->vt.jflag = 0;
	tw->vt.pflag = 0;
	tw->vt.aflag = 0;
	tw->vt.nflag = 0;
	tw->vt.color_flag = 0;
	tw->vt.spcflag = 0;
	tw->vt.esc_sequence[0] = 0;
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: parse_color                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine properly decodes complete DEC color sequences   */
/*+              for setting terminal color maps. These color maps are a      */
/*+              different type than the X color maps. They are related       */
/*+              specificaly for DEC VT240 type color usage.                  */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int parse_color (XmVtWidget tw)
{
    int rcode = 0;
    int screen_flag = 0;
    int writing_flag = 0;
    int mapping = 0;
    int map = 0;
    int bintense = 0;
    int capture = 0;
    int scnt = -1;
    int inx = 0;
    char buffer[BUFSIZ];
    char chunk[BUFSIZ];
    char rgb[BUFSIZ];
    chunk[0] = 0;
    strcpy (buffer, tw->vt.esc_sequence + 2);
    buffer[strlen (buffer) - 1] = 0;
    for (inx = 0; inx < strlen (buffer); inx++)
    {
	if (buffer[inx] == 'M' && scnt == 0)
	{
	    mapping = TRUE;
	    bintense = FALSE;
	}
	if (buffer[inx] == 'I' && scnt == 0)
	{
	    bintense = TRUE;
	    mapping = FALSE;
	}
	if (buffer[inx] == 'S' && !mapping && !bintense)
	    screen_flag = TRUE;
	if (buffer[inx] == 'W' && !mapping && !bintense)
	    writing_flag = TRUE;
	if (buffer[inx] == '(')
	    scnt++;
	if (buffer[inx] == ')')
	    scnt--;
	if (
	       mapping || bintense &&
	  isdigit (buffer[inx]) || buffer[inx] == 'M' || buffer[inx] == 'I')
	{
	    capture = TRUE;
	}
	if (capture)
	    strcatch (chunk, (int) buffer[inx]);
	if (buffer[inx] == ')' && capture)
	{
	    capture = FALSE;
	    parse_hls (tw, chunk, rgb, &map);
	    change_colors (tw, rgb, map, mapping, bintense, screen_flag, writing_flag);
	    chunk[0] = 0;
	}
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: parse_hls                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine properly decodes just the Hue, Luminiance and   */
/*+              Saturation part of a DEC color sequence. It returns the X    */
/*+              RGB color value that is appropriate. It also finds which     */
/*+              terminal color map to load with the new color.               */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             input         - char *     - Color sequence to parse          */
/*+             rgb           - char *     - X rgb color string               */
/*+             map           - int *      - Color map ID to load             */
/*+                                                                           */
/*+ Input:      tw, input, map                                                */
/*+                                                                           */
/*+ Output:     rgb                                                           */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int parse_hls (XmVtWidget tw, char *input, char *rgb, int *map)
{
    int oldlum = 0;
    int rcode = 0;
    int jnx = 0;
    char *inx = 0;
    char rgb_command = 0;
    char map_str[BUFSIZ];
    char hue_str[BUFSIZ];
    char lum_str[BUFSIZ];
    char sat_str[BUFSIZ];
    int keep_old = 0;
    int force_flag = 0;
    int map_flag = -1;
    int hue_flag = 0;
    int lum_flag = 0;
    int sat_flag = 0;
    int hue = 0;
    int lum = 0;
    int sat = 0;
    int red = 0;
    int green = 0;
    int blue = 0;
    strcpy (hue_str, "000");
    strcpy (lum_str, "000");
    strcpy (sat_str, "100");
    for (inx = input; *inx != 0; inx++)
    {
	if (map_flag)
	{
	    if (*inx == 'M' || *inx == 'I')
		*inx = ' ';
	    map_str[jnx++] = *inx;
	    if (*inx == '(')
	    {
		map_flag = 0;
		map_str[jnx - 1] = 0;
		jnx = 0;
		if (map_str[0] == 0)
		    *map = 0;
		else
		    *map = atoi (map_str);
	    }
	}
	if (hue_flag)
	{
	    hue_str[jnx++] = *inx;
	    if (isalpha (*inx) || isspace (*inx) || *inx == ')')
	    {
		hue_flag = 0;
		hue_str[jnx - 1] = 0;
		jnx = 0;
		hue = atoi (hue_str);
	    }
	}
	if (lum_flag)
	{
	    lum_str[jnx++] = *inx;
	    if (isalpha (*inx) || isspace (*inx) || *inx == ')')
	    {
		lum_flag = 0;
		lum_str[jnx - 1] = 0;
		jnx = 0;
		lum = atoi (lum_str);
	    }
	}
	if (sat_flag)
	{
	    sat_str[jnx++] = *inx;
	    if (isalpha (*inx) || isspace (*inx) || *inx == ')')
	    {
		sat_flag = 0;
		sat_str[jnx - 1] = 0;
		jnx = 0;
		sat = atoi (sat_str);
	    }
	}
	if (*inx == 'H')
	    hue_flag = -1;
	if (*inx == 'L')
	    lum_flag = -1;
	if (*inx == 'S')
	    sat_flag = -1;
	if (isforcing (*inx))
	{
	    force_flag = -1;
	    rgb_command = *inx;
	    break;
	}
    }
    if (keep_old)
    {
	lum = tw->vt.lums[*map];
    }
    if (force_flag)
    {
	switch (rgb_command)
	{
	    case 'D':
		red = 0;
		green = 0;
		blue = 0;
		break;
	    case 'R':
		red = 255;
		green = 0;
		blue = 0;
		break;
	    case 'G':
		red = 0;
		green = 255;
		blue = 0;
		break;
	    case 'B':
		red = 0;
		green = 0;
		blue = 255;
		break;
	    case 'C':
		red = 0;
		green = 255;
		blue = 255;
		break;
	    case 'Y':
		red = 255;
		green = 255;
		blue = 0;
		break;
	    case 'M':
		red = 255;
		green = 0;
		blue = 255;
		break;
	    case 'W':
		red = 255;
		green = 255;
		blue = 255;
		break;
	}
    }
    else
    {
	red = get_red (hue, lum, sat);
	green = get_green (hue, lum, sat);
	blue = get_blue (hue, lum, sat);
	equalize (&red, &green, &blue);
    }
    sprintf (
		rgb,
		"#%02X%02X%02X",
		red,
		green,
		blue
	);
    tw->vt.hues[*map] = hue;
    tw->vt.lums[*map] = lum;
    tw->vt.sats[*map] = sat;
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: isforcing                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This boolean check determines whether a character is one of  */
/*+              those which forces a set RGB value to be loaded into a       */
/*+              terminal colormap.                                           */
/*+                                                                           */
/*+ Parameters: single        - char       - Single chr to print              */
/*+                                                                           */
/*+ Input:      single                                                        */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int isforcing (char single)
{
    int rcode = 0;
    rcode = (
		single == 'D' ||
		single == 'R' ||
		single == 'G' ||
		single == 'B' ||
		single == 'C' ||
		single == 'Y' ||
		single == 'M' ||
		single == 'W'
	);
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: equalize                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to equalize rgb values that are out of  */
/*+              range. This occurs when large values of luminance are        */
/*+              specified. The color is basicaly made more white by          */
/*+              spreading this extra intensity to the other normal ranged    */
/*+              colors.                                                      */
/*+                                                                           */
/*+ Parameters: red           - int *      - Red value 0 - 255                */
/*+             green         - int *      - Green value 0 - 255              */
/*+             blue          - int *      - Blue value 0 - 255               */
/*+                                                                           */
/*+ Input:      red, green, blue                                              */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void equalize (int *red, int *green, int *blue)
{
    int check = 0;
    float over = 0.0;
    if (*red > check)
    {
	check = *red;
    }
    if (*green > check)
    {
	check = *green;
    }
    if (*blue > check)
    {
	check = *blue;
    }
    if (check <= MAXRGB)
	return;
    over = ((float) check - (float) MAXRGB) / (float) MAXRGB;
    if (check == *red)
    {
	*red = MAXRGB;
	if (check != *green)
	    *green = *green + (MAXRGB - *green) * over;
	if (check != *blue)
	    *blue = *blue + (MAXRGB - *blue) * over;
    }
    if (check == *green)
    {
	*green = MAXRGB;
	if (check != *blue)
	    *blue = *blue + (MAXRGB - *blue) * over;
	if (check != *red)
	    *red = *red + (MAXRGB - *red) * over;
    }
    if (check == *blue)
    {
	*blue = MAXRGB;
	if (check != *red)
	    *red = *red + (MAXRGB - *red) * over;
	if (check != *green)
	    *green = *green + (MAXRGB - *green) * over;
    }
    if (*red > MAXRGB)
	*red = MAXRGB;
    if (*green > MAXRGB)
	*green = MAXRGB;
    if (*blue > MAXRGB)
	*blue = MAXRGB;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: get_red                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Obtain a Red rgb value from hue, luminance and saturation.   */
/*+                                                                           */
/*+ Parameters: hue           - int        - Hue angle of color               */
/*+             lum           - int        - Luminance of color               */
/*+             sat           - int        - Saturation of color              */
/*+                                                                           */
/*+ Input:      hue, lum, sat                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int get_red (int hue, int lum, int sat)
{
    int rvalue = 0;
    float r_lum = lum / 100.0;
    float r_sat = sat / 100.0;
    float raw_hue = 0.0;
    raw_hue = calc_ix (RED, hue) / 100.0;
    rvalue = (2 * raw_hue * r_sat - r_sat + 1.0) * 100.0 * r_lum;
    return (rvalue);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: get_green                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Obtain a Green rgb value from hue, luminance and saturation. */
/*+                                                                           */
/*+ Parameters: hue           - int        - Hue angle of color               */
/*+             lum           - int        - Luminance of color               */
/*+             sat           - int        - Saturation of color              */
/*+                                                                           */
/*+ Input:      hue, lum, sat                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int get_green (int hue, int lum, int sat)
{
    int rvalue = 0;
    float r_lum = lum / 100.0;
    float r_sat = sat / 100.0;
    float raw_hue = 0.0;
    raw_hue = calc_ix (GREEN, hue) / 100.0;
    rvalue = (2 * raw_hue * r_sat - r_sat + 1.0) * 100.0 * r_lum;
    return (rvalue);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: get_blue                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Obtain a Blue rgb value from hue, luminance and saturation.  */
/*+                                                                           */
/*+ Parameters: hue           - int        - Hue angle of color               */
/*+             lum           - int        - Luminance of color               */
/*+             sat           - int        - Saturation of color              */
/*+                                                                           */
/*+ Input:      hue, lum, sat                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int get_blue (int hue, int lum, int sat)
{
    int rvalue = 0;
    float r_lum = lum / 100.0;
    float r_sat = sat / 100.0;
    float raw_hue = 0.0;
    raw_hue = calc_ix (BLUE, hue) / 100.0;
    rvalue = (2 * raw_hue * r_sat - r_sat + 1.0) * 100.0 * r_lum;
    return (rvalue);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: calc_ix                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine calculates the main internal translation        */
/*+              weighting facter for determining RGB values.                 */
/*+                                                                           */
/*+ Parameters: base          - int        - Base angle for hue               */
/*+             inhue         - int        - Hue value to check               */
/*+                                                                           */
/*+ Input:      base, inhue                                                   */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int calc_ix (int base, int inhue)
{
    int retvalue = 0;
    int span = 0;
    float calc = 0.0;
    if (inhue - base < 180)
	span = (inhue - base);
    else
	span = 360 + (base - inhue);
    span = abs (span);
    if (span > 180)
	span = 360 - span;
    if (span < 60)
    {
	retvalue = MAXRGB;
    }
    else
    {
	if (span < 120)
	{
	    retvalue = (120 - span) * MAXRGB / 60;
	}
    }
    return (retvalue);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: insert_blanks                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Insert any number of spaces at the current cursor.           */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void insert_blanks (XmVtWidget tw, int number)
{
    int jnx = 0;
    int inx = 0;
    int endpos = 0;
    int l_data = 0;
    int l_cset = 0;
    attribute l_attr = BLANK_ATTR;
    if (tw->vt.insert_mode)
    {
	for (inx = 0; inx < number; inx++)
	{
	    write_s (tw, " ", 1);
	}
    }
    else
    {
	for (inx = 0; inx < number; inx++)
	{
	    endpos = tw->vt.screen_cx;
	    for (jnx = tw->vt.maxx; jnx > endpos; jnx--)
	    {
		l_data = g_data (tw, jnx - 1, tw->vt.screen_cy);
		s_data (tw, jnx, tw->vt.screen_cy, l_data);
		l_attr = g_attr (tw, jnx - 1, tw->vt.screen_cy);
		s_attr (tw, jnx, tw->vt.screen_cy, l_attr);
		l_cset = g_cset (tw, jnx - 1, tw->vt.screen_cy);
		s_cset (tw, jnx, tw->vt.screen_cy, l_cset);
		s_flag (tw, jnx, tw->vt.screen_cy, S_MOD);
	    }
	    s_data (tw, tw->vt.screen_cx, tw->vt.screen_cy, ' ');
	    s_flag (tw, tw->vt.screen_cx, tw->vt.screen_cy, S_MOD);
	    s_attr (tw, tw->vt.screen_cx, tw->vt.screen_cy, tw->vt.attributes);
	    s_cset (tw, tw->vt.screen_cx, tw->vt.screen_cy, CS_USASCII);
	}
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: right_edge                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Check to see if the cursor is at the right edge, if true     */
/*+              move to next line.                                           */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void right_edge (XmVtWidget tw)
{
    if (tw->vt.screen_cx > tw->vt.maxx)
    {
	tw->vt.screen_cx = tw->vt.minx;
	tw->vt.screen_cy++;
	bottom_edge (tw);
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: delete_lines                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Delete any number of blank lines starting at the current     */
/*+              cursor.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void delete_lines (XmVtWidget tw, int number)
{
    int inx = 0;
    for (inx = 0; inx < number; inx++)
	half_scroll_up (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: insert_lines                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Insert any number of blank lines at the current cursor.      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void insert_lines (XmVtWidget tw, int number)
{
    int knx = 0;
    int inx = 1;
    int jnx = 1;
    int l_data = 0;
    attribute l_attr = BLANK_ATTR;
    int l_cset = 0;
    for (knx = 0; knx < number; knx++)
    {
	for (jnx = tw->vt.maxy; jnx > tw->vt.screen_cy; jnx--)
	{
	    for (inx = tw->vt.minx; inx <= tw->vt.maxx; inx++)
	    {
		l_data = g_data (tw, inx, jnx - 1);
		s_data (tw, inx, jnx, l_data);
		l_attr = g_attr (tw, inx, jnx - 1);
		s_attr (tw, inx, jnx, l_attr);
		l_cset = g_cset (tw, inx, jnx - 1);
		s_cset (tw, inx, jnx, l_cset);
		s_flag (tw, inx, jnx, S_MOD);
	    }
	}
	for (inx = tw->vt.minx; inx <= tw->vt.maxx; inx++)
	{
	    s_data (tw, inx, tw->vt.screen_cy, ' ');
	    s_attr (tw, inx, tw->vt.screen_cy, clr_attr);
	    s_cset (tw, inx, tw->vt.screen_cy, CS_USASCII);
	    s_flag (tw, inx, tw->vt.screen_cy, S_MOD);
	}
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: reverse_index                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Insert one line at the cursor.                               */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void reverse_index (XmVtWidget tw)
{
    int inx = 1;
    int jnx = 1;
    int l_data = 0;
    attribute l_attr = BLANK_ATTR;
    int l_cset = 0;
    int l_line_attr = 0;
    int min = 0;
    int max = 0;
    int inscroll = 0;

    inscroll = (tw->vt.screen_cy <= tw->vt.maxy && tw->vt.screen_cy >= tw->vt.miny);
    if (tw->vt.decom_mode || inscroll)
    {
	max = tw->vt.maxy;
	min = tw->vt.miny;
    }
    else
    {
	max = tw->vt.real_maxy;
	min = tw->vt.real_miny;
    }
    if (tw->vt.screen_cy == min)
    {
	for (jnx = max; jnx > tw->vt.screen_cy; jnx--)
	{
	    s_swap (tw, jnx, jnx - 1);
	}
	s_common
	    (
		tw, tw->vt.minx, tw->vt.screen_cy, tw->vt.columns,
		tw->vt.attributes, ' ', CS_USASCII
	    );
	s_line_attr (tw, tw->vt.screen_cy, 0);
    }
    else
    {
	tw->vt.screen_cy--;
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: bottom_edge                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Check to see if the cursor is at the bottom edge if true     */
/*+              scroll up one line.                                          */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void bottom_edge (XmVtWidget tw)
{
    int inscroll = 0;

    inscroll = (tw->vt.screen_cy - 1 <= tw->vt.maxy && tw->vt.screen_cy >= tw->vt.miny);
    if (tw->vt.decom_mode || inscroll)
    {
	if (tw->vt.screen_cy > tw->vt.maxy)
	    scroll_up (tw);
    }
    else
    {
	if (tw->vt.screen_cy > tw->vt.real_maxy)
	    scroll_up (tw);
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: tab_over                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Tab over to the next screen defined tab.                     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void tab_over (XmVtWidget tw)
{
    int inx = 0;
    inx = tw->vt.screen_cx + 1;
    while (tw->vt.tabs[inx] != 'I' && inx < tw->vt.maxx)
    {
	s_flag (tw, inx, tw->vt.screen_cy, S_MOD);
	inx++;
    }
    tw->vt.screen_cx = inx;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: delete_char                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Delete any number of characters from the cursor forward.     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void delete_char (XmVtWidget tw, int number)
{
    attribute l_attr = BLANK_ATTR;
    int l_data = 0;
    int l_cset = 0;
    int inx = 1;
    int jnx = 1;
    for (inx = tw->vt.screen_cx; inx <= tw->vt.maxx - number; inx++)
    {
	l_data = g_data (tw, inx + number, tw->vt.screen_cy);
	s_data (tw, inx, tw->vt.screen_cy, l_data);
	l_attr = g_attr (tw, inx + number, tw->vt.screen_cy);
	s_attr (tw, inx, tw->vt.screen_cy, l_attr);
	l_cset = g_cset (tw, inx + number, tw->vt.screen_cy);
	s_cset (tw, inx, tw->vt.screen_cy, l_cset);
	s_flag (tw, inx, tw->vt.screen_cy, S_MOD);
    }
    for (jnx = inx; jnx <= tw->vt.maxx; jnx++)
    {
	s_data (tw, jnx, tw->vt.screen_cy, ' ');
	s_attr (tw, jnx, tw->vt.screen_cy, clr_attr);
	s_cset (tw, jnx, tw->vt.screen_cy, CS_USASCII);
	s_flag (tw, jnx, tw->vt.screen_cy, S_MOD);
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: half_scroll_up                                             */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Scroll up lower part of the screen up to the cursor.         */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void half_scroll_up (XmVtWidget tw)
{
    int inx = 1;
    int jnx = 1;
    int l_data = 0;
    attribute l_attr = BLANK_ATTR;
    int l_cset = 0;
    int l_line_attr = 0;

    for (jnx = tw->vt.screen_cy + 1; jnx <= tw->vt.maxy; jnx++)
    {
	s_swap (tw, jnx - 1, jnx);
    }
    s_common
	(
	    tw, tw->vt.minx, tw->vt.maxy, tw->vt.columns,
	    clr_attr, ' ', CS_USASCII
	);
    s_line_attr (tw, tw->vt.maxy, 0);
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: scroll_up                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Scroll the screen up.                                        */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void scroll_up (XmVtWidget tw)
{
    int inx = 1;
    int jnx = 1;
    int loop = 0;
    int l_data = 0;
    attribute l_attr = BLANK_ATTR;
    int l_cset = 0;

    if (
	   tw->vt.miny == tw->vt.real_miny &&
	   tw->vt.maxy == tw->vt.real_maxy
	)
    {
	(tw->vt.first_row)++;
    }
    else
    {
	for (jnx = tw->vt.miny + 1; jnx <= tw->vt.maxy; jnx++)
	{
	    s_swap (tw, jnx - 1, jnx);
	}
    }
    s_common
	(
	    tw, tw->vt.minx, tw->vt.maxy, tw->vt.columns,
	    l_attr, ' ', CS_USASCII
	);
    s_line_attr (tw, tw->vt.maxy, 0);
    tw->vt.screen_cy--;
    tw->vt.scroll_line_count++;
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_eol                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Clear to end of line.                                        */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_eol (XmVtWidget tw)
{
    int xcoo = 1;
    for (xcoo = tw->vt.screen_cx; xcoo <= tw->vt.maxx; xcoo++)
    {
	s_data (tw, xcoo, tw->vt.screen_cy, ' ');
	s_attr (tw, xcoo, tw->vt.screen_cy, tw->vt.attributes);
	s_cset (tw, xcoo, tw->vt.screen_cy, CS_USASCII);
	s_flag (tw, xcoo, tw->vt.screen_cy, S_MOD);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_bol                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Clear from begining of line to the cursor.                   */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_bol (XmVtWidget tw)
{
    int xcoo = 1;
    for (xcoo = tw->vt.minx; xcoo <= tw->vt.screen_cx; xcoo++)
    {
	s_data (tw, xcoo, tw->vt.screen_cy, ' ');
	s_attr (tw, xcoo, tw->vt.screen_cy, tw->vt.attributes);
	s_cset (tw, xcoo, tw->vt.screen_cy, CS_USASCII);
	s_flag (tw, xcoo, tw->vt.screen_cy, S_MOD);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_aol                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Clear all of line.                                           */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_aol (XmVtWidget tw)
{
    int xcoo = 1;
    for (xcoo = tw->vt.minx; xcoo <= tw->vt.maxx; xcoo++)
    {
	s_data (tw, xcoo, tw->vt.screen_cy, ' ');
	s_attr (tw, xcoo, tw->vt.screen_cy, tw->vt.attributes);
	s_cset (tw, xcoo, tw->vt.screen_cy, CS_USASCII);
	s_flag (tw, xcoo, tw->vt.screen_cy, S_MOD);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_up                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine clears the screen up from the cursor position   */
/*+              to the first character position of the screen.               */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_up (XmVtWidget tw)
{
    int xcoo = 1;
    int ycoo = 1;
    for (ycoo = tw->vt.screen_cy; ycoo >= tw->vt.real_miny; ycoo--)
    {
	for (xcoo = tw->vt.real_minx; xcoo <= tw->vt.real_maxx; xcoo++)
	{
	    if (
		   (ycoo == tw->vt.screen_cy && xcoo <= tw->vt.screen_cx) ||
		   (ycoo < tw->vt.screen_cy)
		)
	    {
		s_data (tw, xcoo, ycoo, ' ');
		s_attr (tw, xcoo, ycoo, tw->vt.attributes);
		s_cset (tw, xcoo, ycoo, CS_USASCII);
		s_flag (tw, xcoo, ycoo, S_MOD);
	    }
	}
	s_line_attr (tw, ycoo, 0);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_down                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Clear downward from cursor.                                  */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_down (XmVtWidget tw)
{
    int xcoo = 1;
    int ycoo = 1;
    for (ycoo = tw->vt.screen_cy; ycoo <= tw->vt.real_maxy; ycoo++)
    {
	for (xcoo = tw->vt.real_minx; xcoo <= tw->vt.real_maxx; xcoo++)
	{
	    if (
		   (ycoo == tw->vt.screen_cy && xcoo >= tw->vt.screen_cx) ||
		   (ycoo > tw->vt.screen_cy)
		)
	    {
		s_data (tw, xcoo, ycoo, ' ');
		s_attr (tw, xcoo, ycoo, tw->vt.attributes);
		s_cset (tw, xcoo, ycoo, CS_USASCII);
		s_flag (tw, xcoo, ycoo, S_MOD);
	    }
	}
	s_line_attr (tw, ycoo, 0);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear_all                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Clear all the screen.                                        */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void clear_all (XmVtWidget tw)
{
    int ycoo = 1;
    for (ycoo = tw->vt.real_miny; ycoo <= tw->vt.real_maxy; ycoo++)
    {
	s_common
	    (
	      tw, tw->vt.real_minx, ycoo, tw->vt.columns, tw->vt.attributes,
		' ', CS_USASCII
	    );
	s_line_attr (tw, ycoo, 0);
    }
    ok_update (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: eight_bit                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to intiate eight bit emulation. Normaly */
/*+              this includes at least a load of the default color maps.     */
/*+              Since this 'terminal' ceases to exist at the end of its      */
/*+              usage there is no default set of colors. Therefore at this   */
/*+              time this routine does nothing.                              */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void eight_bit (XmVtWidget tw)
{
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: strcatch                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Concatenate a character to a string.                         */
/*+                                                                           */
/*+ Parameters: string        - char *     - This is the input string to      */
/*+                                          single space.                    */
/*+             data          - int        - Data to append                   */
/*+                                                                           */
/*+ Input:      string, data                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int strcatch (char *string, int data)
{
    string[strlen (string) + 1] = 0;
    string[strlen (string)] = data;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: sync_cursor                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Make sure that the screen cursor is not outside the bounds   */
/*+              of the screen.                                               */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void sync_cursor (XmVtWidget tw)
{
    int inscroll = 0;
    inscroll = (tw->vt.screen_cy <= tw->vt.maxy && tw->vt.screen_cy >= tw->vt.miny);
    if (tw->vt.decom_mode || inscroll)
    {
	if (tw->vt.screen_cy < tw->vt.miny)
	{
	    tw->vt.screen_cy = tw->vt.miny;
	}
	if (tw->vt.screen_cx < tw->vt.minx)
	{
	    tw->vt.screen_cx = tw->vt.minx;
	}
	if (tw->vt.screen_cx > tw->vt.maxx)
	{
	    tw->vt.screen_cx = tw->vt.maxx;
	}
	if (tw->vt.screen_cy > tw->vt.maxy)
	{
	    tw->vt.screen_cy = tw->vt.maxy;
	}
    }
    else
    {
	if (tw->vt.screen_cy < tw->vt.real_miny)
	{
	    tw->vt.screen_cy = tw->vt.real_miny;
	}
	if (tw->vt.screen_cx < tw->vt.real_minx)
	{
	    tw->vt.screen_cx = tw->vt.real_minx;
	}
	if (tw->vt.screen_cx > tw->vt.real_maxx)
	{
	    tw->vt.screen_cx = tw->vt.real_maxx;
	}
	if (tw->vt.screen_cy > tw->vt.real_maxy)
	{
	    tw->vt.screen_cy = tw->vt.real_maxy;
	}
    }
    s_flag (tw, tw->vt.screen_cx, tw->vt.screen_cy, S_MOD);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: execute_esc                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Execute the commands associated with an escape sequence.     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int execute_esc (XmVtWidget tw)
{
    int inx = 0;
    int jnx = 0;
    int len = 0;
    int alen = 0;
    int rcode = 0;
    char *view = NULL;
    int vtcommand = BADCOMMAND;
    char local[BUFSIZ];
    char sendbuf[BUFSIZ];
    char junk[BUFSIZ];
    view = tw->vt.esc_sequence;
    for (inx = 0; inx < NUMVT100 - 1; inx++)
    {
	len = strlen (vt100[inx]);
	if (speccmp (view, vt100[inx], len, &alen) == 0)
	{
	    vtcommand = inx;
	    break;
	}
    }
    if (tw->vt.debug)
    {
	if (vtcommand != BADCOMMAND)
	    fprintf (debug_fp, "Interpreted Escape Sequence: ESC%s\n", view);
	else
	    fprintf (debug_fp, "Don't know  Escape Sequence: ESC%s\n", view);
    }
    switch (vtcommand)
    {
	case BADCOMMAND:
	    strcpy (local, view);
	    rcode = -1;
	    break;
	case EIGHT_BIT:
	    eight_bit (tw);
	    break;
	case CLEAR:
	    clear_all (tw);
	    break;
	case CLEAR_DOWN2:
	case CLEAR_DOWN:
	    clear_down (tw);
	    break;
	case CLEAR_UP:
	    clear_up (tw);
	    break;
	case CLEAR_EOL:
	case CLEAR_EOL2:
	    clear_eol (tw);
	    break;
	case CLEAR_BOL:
	    clear_bol (tw);
	    break;
	case CLEAR_AOL:
	    clear_aol (tw);
	    break;
	case SEND_ID:
	case SEND_ID2:
	case SEND_ID3:
	    sprintf (sendbuf, "%c[?62;1;2;3;4;6;8c", ESC);
	    send_keys (tw, sendbuf);
	    break;
	case NORMAL_KEY:
	    break;
	case APPLIC_KEY:
	    break;
	case PRINTER_ON:
	    tw->vt.pass_thru = TRUE;
	    tw->vt.data_name = dynamic_tmpnam ();
	    tw->vt.data_fp = fopen (tw->vt.data_name, WRITEMODE);
	    if (tw->vt.data_fp == NULL)
		tw->vt.data_fp = stdout;
	    break;
	case PRINTER_OFF:
	    tw->vt.pass_thru = FALSE;
	    break;
	case DSTREP:
	    sprintf (sendbuf, "\033[0n");
	    send_keys (tw, sendbuf);
	    break;
	case REPORT:
	    sprintf (
			sendbuf,
			"%c[%d;%dR",
			ESC,
			tw->vt.screen_cy,
			tw->vt.screen_cx
		);
	    send_keys (tw, sendbuf);
	    break;
	case DA0_X:
	    sprintf (sendbuf, "\033[2;1;1;112;112;1;0x");
	    send_keys (tw, sendbuf);
	    break;
	case DA1_X:
	    sprintf (sendbuf, "\033[3;1;1;112;112;1;0x");
	    send_keys (tw, sendbuf);
	    break;
	case DA2_X:
	    if (tw->vt.report_message != NULL)
		send_keys (tw, tw->vt.report_message);
	    else
		send_keys (tw, "Report message not defined.");
	    break;
	case PRIVATE_MD:
	    rcode = sub_proc_one (tw);
	    break;
	case CURSOR_MV:
	    rcode = sub_proc_two (tw);
	    break;
	case SAVE:
	    tw->vt.save_cx = tw->vt.screen_cx;
	    tw->vt.save_cy = tw->vt.screen_cy;
	    tw->vt.save_cset_level = tw->vt.gr_cset_level;
	    tw->vt.save_attributes = tw->vt.attributes;
	    memcpy (tw->vt.save_char_set, tw->vt.char_set, NUMLEV);
	    tw->vt.save_ansi_fg = tw->vt.last_ansi_fg;
	    tw->vt.save_ansi_bg = tw->vt.last_ansi_bg;
	    break;
	case REVERSE_IN:
	    reverse_index (tw);
	    break;
	case RESTORE:
	    tw->vt.screen_cx = tw->vt.save_cx;
	    tw->vt.screen_cy = tw->vt.save_cy;
	    tw->vt.gr_cset_level = tw->vt.save_cset_level;
	    tw->vt.attributes = tw->vt.save_attributes;
	    memcpy (tw->vt.char_set, tw->vt.save_char_set, NUMLEV);
	    tw->vt.last_ansi_fg = tw->vt.save_ansi_fg;
	    tw->vt.last_ansi_bg = tw->vt.save_ansi_bg;
	    break;
	case SELECTCS0A:
	    tw->vt.char_set[0] = CS_LINES;
	    break;
	case SELECTCS0B:
	    tw->vt.char_set[0] = CS_USASCII;
	    break;
	case SELECTCS0C:
	    tw->vt.char_set[0] = CS_USASCII;
	    break;
	case SELECTCS0D:
	    tw->vt.char_set[0] = CS_UKASCII;
	    break;
	case SELECTCS0E:
	    tw->vt.char_set[0] = CS_USASCII;
	    break;
	case SELECTCS1A:
	    tw->vt.char_set[1] = CS_LINES;
	    break;
	case SELECTCS1B:
	    tw->vt.char_set[1] = CS_USASCII;
	    break;
	case SELECTCS1C:
	    tw->vt.char_set[1] = CS_USASCII;
	    break;
	case SELECTCS1D:
	    tw->vt.char_set[1] = CS_UKASCII;
	    break;
	case SELECTCS1E:
	    tw->vt.char_set[1] = CS_USASCII;
	    break;
	case SOFT_RESET:
	case RESET:
	    reset (tw);
	    break;
	case MESSAGE:
	    message (tw);
	    break;
	case SPECMSG:
	    special_message (tw);
	    break;
	case SWITCH_G2:
	    tw->vt.gr_cset_level = 2;
	    break;
	case SWITCH_G3:
	    tw->vt.gr_cset_level = 3;
	    break;
	case COLOR_LOAD:
	    parse_color (tw);
	    break;
	case BACK_SLASH:
	    break;
	case REQUEST_7:
	    break;
	case REQUEST_8:
	    break;
	case GRAPHIC_ON:
	    tw->vt.pass_thru = TRUE;
	    tw->vt.graphics_name = dynamic_tmpnam ();
	    tw->vt.data_fp = fopen (tw->vt.graphics_name, WRITEMODE);
	    if (tw->vt.data_fp == NULL)
		tw->vt.data_fp = stdout;
	    break;
	case GRAPHIC_OFF:
	    tw->vt.pass_thru = FALSE;
	    break;
	case INDEX:
	    tw->vt.screen_cy++;
	    break;
	case TABSET:
	    tab_set (tw);
	    break;
	case TABCLR:
	case TABCLR2:
	    tab_clear (tw);
	    break;
	case TABCLRALL:
	    tab_clear_all (tw);
	    break;
	case DONEWLINE:
	    newline (tw);
	    break;
	case TEST:
	    self_test (tw);
	    break;
	case EXIT_VT52:
	    tw->vt.ansi_mode = TRUE;
	    break;
	case DECSWL:
	    s_line_attr (tw, tw->vt.screen_cy, 0);
	    break;
	case DECDWL:
	    s_line_attr (tw, tw->vt.screen_cy, 1);
	    break;
	case DECDHL_TP:
	    s_line_attr (tw, tw->vt.screen_cy, 2);
	    break;
	case DECDHL_BT:
	    s_line_attr (tw, tw->vt.screen_cy, 3);
	    break;
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: message                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: message parses an escape sequence that is sent to provide a  */
/*+              message to the terminal. This message is stored in the vt    */
/*+              structure.                                                   */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void message (XmVtWidget tw)
{
    strcpy (tw->vt.message, tw->vt.esc_sequence + 3);
    tw->vt.message[strlen (tw->vt.message) - 1] = 0;
    message_do (tw);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: special_message                                            */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine parses an escape sequence that is sent to       */
/*+              provide a special_message to the VT widget. This message is  */
/*+              a command sequence that is used to access special features   */
/*+              such as GIF picture loading.                                 */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void special_message (XmVtWidget tw)
{
    char inputdata[BUFSIZ];
    char filename[BUFSIZ];
    char identity = 0;
    strcpy (inputdata, tw->vt.esc_sequence + 1);
    inputdata[strlen (inputdata) - 1] = 0;
    special_msg_do (tw, inputdata);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: speccmp                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is a Special string compare that works alot     */
/*+              like strncmp except that ; characters are removed from the   */
/*+              first string before the comparison is made. This basicaly    */
/*+              means that the ; character is ignored.                       */
/*+                                                                           */
/*+ Parameters: firststr      - char *     - First string to compare          */
/*+             secondstr     - char *     - Second string to compare.        */
/*+             len           - int        - Length of string                 */
/*+             alen          - int *      - Actual length of match           */
/*+                                                                           */
/*+ Input:      firststr, secondstr, len, alen                                */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int speccmp (char *firststr, char *secondstr, int len, int *alen)
{
    int inx = 0;
    int jnx = 0;
    int retval = 0;
    (*alen) = 0;
    for (inx = 0; inx < len; inx++)
    {
	while (firststr[jnx] == ';')
	{
	    jnx++;
	    (*alen) = 1;
	}
	if (firststr[jnx] != secondstr[inx])
	    retval = 1;
	jnx++;
    }
    return (retval);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: sub_proc_one                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Sub process an 'escape[?ParmsCode' seqence for parameters.   */
/*+              Most of these will be hard coded in the execute_esc          */
/*+              function. Only those that are not being processed will end   */
/*+              up here.                                                     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int sub_proc_one (XmVtWidget tw)
{
    Boolean dummy = TRUE;
    Boolean *action = NULL;
    int inx = 0;
    int rcode = 0;
    int xcrd = 1;
    int ycrd = 1;
    int last = 1;
    int len = 0;
    int subcode = 0;
    int dollar = 0;
    int subdata = 0;
    Boolean save_column = FALSE;
    Boolean save_inverse = FALSE;
    Boolean save_decom = FALSE;
    char rems[BUFSIZ];
    char sendbuf[BUFSIZ];
    char *element = NULL;

    strcpy (rems, tw->vt.esc_sequence + 2);
    subcode = rems[strlen (rems) - 1];
    dollar = rems[strlen (rems) - 2];
    if (dollar == '$')
	rems[strlen (rems) - 2] = 0;
    else
	rems[strlen (rems) - 1] = 0;
    element = strtok (rems, ";");
    while (element != NULL)
    {
	sscanf (element, "%d", &subdata);
	save_column = tw->vt.column_mode;
	save_inverse = tw->vt.inverse_mode;
	save_decom = tw->vt.decom_mode;
	action = &dummy;
	switch (subdata)
	{
	    case 1:
		action = &(tw->vt.application_key_mode);
		break;
	    case 2:
		action = &(tw->vt.ansi_mode);
		break;
	    case 3:
		action = &(tw->vt.column_mode);
		break;
	    case 4:
		action = &(tw->vt.smooth_mode);
		break;
	    case 5:
		action = &(tw->vt.inverse_mode);
		break;
	    case 6:
		action = &(tw->vt.decom_mode);
		break;
	    case 7:
		action = &(tw->vt.autowrap_mode);
		break;
	    case 8:
		action = &(tw->vt.autorepeat_mode);
		break;
	    case 9:
		action = &(tw->vt.mouse_mode);
		break;
	    case 25:
		action = &(tw->vt.cursor_mode);
		break;
	    default:
		rcode = -1;
		if (tw->vt.debug)
		    fprintf (
				debug_fp,
		      "DEC Sub command '%c' not processed yet. Data = %d\n",
				subcode,
				subdata
			);
		break;
	}
	element = strtok (NULL, ";");
	switch (subcode)
	{
	    case 'l':
		*action = FALSE;
		break;
	    case 'h':
		*action = TRUE;
		break;
	    case 'p':
		if (dollar == '$')
		{
		    if (rcode)
		    {
			sprintf (sendbuf, "\033[?%d;0$y", subdata);
		    }
		    else
		    {
			if (*action)
			    sprintf (sendbuf, "\033[?%d;1$y", subdata);
			else
			    sprintf (sendbuf, "\033[?%d;2$y", subdata);
		    }
		    send_keys (tw, sendbuf);
		}
		break;
	}
    }
    if (save_column != tw->vt.column_mode)
    {
	if (tw->vt.column_mode)
	    change_width (tw, 132);
	else
	    change_width (tw, 80);
    }
    if (save_inverse != tw->vt.inverse_mode)
    {
	ok_revive (tw);
    }
    if (save_decom != tw->vt.decom_mode)
    {
	if (tw->vt.decom_mode)
	{
	    tw->vt.screen_cy = tw->vt.miny;
	    tw->vt.screen_cx = tw->vt.minx;
	}
	else
	{
	    tw->vt.screen_cy = tw->vt.real_miny;
	    tw->vt.screen_cx = tw->vt.real_minx;
	}
    }
    return (rcode);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: sub_proc_three                                             */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to sub process an 'escape[Pal' type     */
/*+              escape sequence. This type sequence is used to behavior mode */
/*+              type flags in the emulation typicaly. A subcode of l means   */
/*+              to reset the flag. A subcode of h means to set the flag and  */
/*+              a subcode of $p means to report the flags state back to the  */
/*+              host.                                                        */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int sub_proc_three (XmVtWidget tw)
{
    Boolean dummy = TRUE;
    Boolean *action = NULL;
    int inx = 0;
    int rcode = 0;
    int xcrd = 1;
    int ycrd = 1;
    int last = 1;
    int len = 0;
    int subcode = 0;
    int dollar = 0;
    int subdata = 0;
    char rems[BUFSIZ];
    char *element = NULL;
    char sendbuf[BUFSIZ];

    strcpy (rems, tw->vt.esc_sequence + 1);
    subcode = rems[strlen (rems) - 1];
    dollar = rems[strlen (rems) - 2];
    if (dollar == '$')
	rems[strlen (rems) - 2] = 0;
    else
	rems[strlen (rems) - 1] = 0;
    element = strtok (rems, ";");
    while (element != NULL)
    {
	sscanf (element, "%d", &subdata);
	action = &dummy;
	switch (subdata)
	{
	    case 4:
		action = &(tw->vt.insert_mode);
		break;
	    case 20:
		action = &(tw->vt.newline_mode);
		break;
	    default:
		rcode = -1;
		if (tw->vt.debug)
		    fprintf (
				debug_fp,
		     "ANSI Sub command '%c' not processed yet. Data = %d\n",
				subcode,
				subdata
			);
		break;
	}
	element = strtok (NULL, ";");
	switch (subcode)
	{
	    case 'l':
		*action = FALSE;
		break;
	    case 'h':
		*action = TRUE;
		break;
	    case 'p':
		if (dollar == '$')
		{
		    if (rcode)
		    {
			sprintf (sendbuf, "\033[%d;0$y", subdata);
		    }
		    else
		    {
			if (*action)
			    sprintf (sendbuf, "\033[%d;1$y", subdata);
			else
			    sprintf (sendbuf, "\033[%d;2$y", subdata);
		    }
		    send_keys (tw, sendbuf);
		}
		break;
	}
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: sub_proc_two                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: Sub process an 'escape[ParmsCode' seqence for parameters.    */
/*+              Some of these will be hard coded in the execute_esc          */
/*+              function. Several heavily used functions like relocating the */
/*+              cursor are located here.                                     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int sub_proc_two (XmVtWidget tw)
{
    int inx = 0;
    int xcrd = 1;
    int ycrd = 1;
    int first = 1;
    int last = 1;
    int len = 0;
    int rcode = 0;
    int subcode = 0;
    char rems[BUFSIZ];
    strcpy (rems, tw->vt.esc_sequence + 1);
    subcode = rems[strlen (rems) - 1];
    rems[strlen (rems) - 1] = 0;
    switch (subcode)
    {
	case '@':
	    sscanf (rems, "%d", &first);
	    insert_blanks (tw, first);
	    break;
	case 'A':
	    sscanf (rems, "%d", &first);
	    cuu_move (tw, first);
	    break;
	case 'B':
	    sscanf (rems, "%d", &first);
	    cud_move (tw, first);
	    break;
	case 'C':
	    sscanf (rems, "%d", &first);
	    cuf_move (tw, first);
	    break;
	case 'D':
	    sscanf (rems, "%d", &first);
	    cancel_eol (tw);
	    if (strlen (rems) == 0)
		first = 1;
	    cub_move (tw, first);
	    break;
	case 'L':
	    sscanf (rems, "%d", &first);
	    insert_lines (tw, first);
	    break;
	case 'M':
	    sscanf (rems, "%d", &first);
	    delete_lines (tw, first);
	    break;
	case 'P':
	    sscanf (rems, "%d", &first);
	    cancel_eol (tw);
	    delete_char (tw, first);
	    break;
	case 'X':
	    sscanf (rems, "%d;%d", &first, &last);
	    delete_char (tw, first);
	    cancel_eol (tw);
	    break;
	case 'f':
	case 'H':
	    cancel_eol (tw);
	    fixrems (rems, first, last);
	    sscanf (rems, "%d;%d", &ycrd, &xcrd);
	    cup_move (tw, xcrd, ycrd);
	    break;
	case 'm':
	    multi_attr (tw, rems);
	    break;
	case 'r':
	    first = tw->vt.real_miny;
	    last = tw->vt.real_maxy;
	    fixrems (rems, first, last);
	    sscanf (rems, "%d;%d", &first, &last);
	    reserve (tw, first, last);
	    break;
	case 'l':
	case 'h':
	case 'p':
	case '~':
	case '}':
	    sub_proc_three (tw);
	    break;
	case ESC:
	    half_scroll_up (tw);
	    break;
	default:
	    if (tw->vt.debug)
		fprintf (
			    debug_fp,
			"Sub command '%c' not processed yet. Data = '%s'\n",
			    subcode,
			    rems
		    );
	    rcode = -1;
	    break;
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: reserve                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine reserves a range of lines on the screen for     */
/*+              scrolling. This permits localized scrolling of part of the   */
/*+              screen. This is commonly used in editors.                    */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             first         - int        - Default first parm               */
/*+             last          - int        - Default last parm                */
/*+                                                                           */
/*+ Input:      tw, first, last                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void reserve (XmVtWidget tw, int first, int last)
{
    tw->vt.miny = first;
    tw->vt.maxy = last;
    if (tw->vt.decom_mode)
	tw->vt.screen_cy = tw->vt.miny;
    else
	tw->vt.screen_cy = tw->vt.real_miny;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: fixrems                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to fix up an escape sequence that has a */
/*+              'alternative' format for the numeric parameters. Anyway,     */
/*+              this alternative format involves no values in the area where */
/*+              you would expect them such as: ESC[;7H which means go to     */
/*+              column 7 in the first row.                                   */
/*+                                                                           */
/*+ Parameters: string        - char *     - This is the input string to      */
/*+                                          single space.                    */
/*+             first         - int        - Default first parm               */
/*+             last          - int        - Default last parm                */
/*+                                                                           */
/*+ Input:      string, first, last                                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void fixrems (char *string, int first, int last)
{
    char temp[BUFSIZ];
    char chr2 = 0;
    char chr1 = 0;
    chr1 = string[0];
    chr2 = string[strlen (string) - 1];
    if (chr1 != ';' && chr2 != ';')
    {
	return;
    }
    if (chr2 == ';')
    {
	sprintf (temp, "%d", last);
	strcat (string, temp);
    }
    if (chr1 == ';')
    {
	sprintf (temp, "%d%s", first, string);
	strcpy (string, temp);
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: multi_attr                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function handles those cases of setting more than one   */
/*+              character attribute at one time with the attribute sequence. */
/*+              Unfortunatly this routine is short because it is recursive   */
/*+              it calls execute_esc to set the proper attributes. This can  */
/*+              be changed if this is not proper. I do not mind recursive    */
/*+              code if its easy to read. However it can be hard to read if  */
/*+              it is complicated. This is not.                              */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             mattr         - char *     - Multiple attributes              */
/*+                                                                           */
/*+ Input:      tw, mattr                                                     */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void multi_attr (XmVtWidget tw, char *mattr)
{
    char one[BUFSIZ];
    char temp[BUFSIZ];
    char *parsed = NULL;
    int attcode = 0;
    strcpy (temp, mattr);
    parsed = strtok (temp, ";");
    if (parsed == NULL)
    {
	strcpy (temp, "0");
	parsed = temp;
    }
    while (parsed != NULL)
    {
	if (strlen (parsed) == 0)
	{
	    attcode = 0;
	}
	else
	{
	    strcpy (one, parsed);
	    sscanf (one, "%d", &attcode);
	}
	if (attcode >= 30 && attcode <= 39)
	{
	    if (tw->vt.last_ansi_fg)
		clr (&(tw->vt.attributes), tw->vt.last_ansi_fg);
	    tw->vt.last_ansi_fg = attcode;
	}
	if (attcode >= 40 && attcode <= 49)
	{
	    if (tw->vt.last_ansi_bg)
		clr (&(tw->vt.attributes), tw->vt.last_ansi_bg);
	    tw->vt.last_ansi_bg = attcode;
	}
	switch (attcode)
	{
	    case ATT_NORMAL:
		reset_attr (&(tw->vt.attributes));
		break;
	    case ATT_BLINK:
		set (&(tw->vt.attributes), ATT_BLINK);
		break;
	    case ATT_BOLD:
		set (&(tw->vt.attributes), ATT_BOLD);
		break;
	    case ATT_DIM:
		set (&(tw->vt.attributes), ATT_DIM);
		break;
	    case ATT_UNDERSCORE:
		set (&(tw->vt.attributes), ATT_UNDERSCORE);
		break;
	    case ATT_INVERSE:
		set (&(tw->vt.attributes), ATT_INVERSE);
		break;
	    case ATT_ANSI_30:
		set (&(tw->vt.attributes), ATT_ANSI_30);
		break;
	    case ATT_ANSI_31:
		set (&(tw->vt.attributes), ATT_ANSI_31);
		break;
	    case ATT_ANSI_32:
		set (&(tw->vt.attributes), ATT_ANSI_32);
		break;
	    case ATT_ANSI_33:
		set (&(tw->vt.attributes), ATT_ANSI_33);
		break;
	    case ATT_ANSI_34:
		set (&(tw->vt.attributes), ATT_ANSI_34);
		break;
	    case ATT_ANSI_35:
		set (&(tw->vt.attributes), ATT_ANSI_35);
		break;
	    case ATT_ANSI_36:
		set (&(tw->vt.attributes), ATT_ANSI_36);
		break;
	    case ATT_ANSI_37:
		set (&(tw->vt.attributes), ATT_ANSI_37);
		break;
	    case ATT_ANSI_40:
		set (&(tw->vt.attributes), ATT_ANSI_40);
		break;
	    case ATT_ANSI_41:
		set (&(tw->vt.attributes), ATT_ANSI_41);
		break;
	    case ATT_ANSI_42:
		set (&(tw->vt.attributes), ATT_ANSI_42);
		break;
	    case ATT_ANSI_43:
		set (&(tw->vt.attributes), ATT_ANSI_43);
		break;
	    case ATT_ANSI_44:
		set (&(tw->vt.attributes), ATT_ANSI_44);
		break;
	    case ATT_ANSI_45:
		set (&(tw->vt.attributes), ATT_ANSI_45);
		break;
	    case ATT_ANSI_46:
		set (&(tw->vt.attributes), ATT_ANSI_46);
		break;
	    case ATT_ANSI_47:
		set (&(tw->vt.attributes), ATT_ANSI_47);
		break;
	    case ATT_CURSOR:
		set (&(tw->vt.attributes), ATT_CURSOR);
		break;
	    case ATT_SELECT:
		set (&(tw->vt.attributes), ATT_SELECT);
		break;
	    case ATT_SPECIAL:
		set (&(tw->vt.attributes), ATT_SPECIAL);
		break;
	    case ATT_ARMED:
		set (&(tw->vt.attributes), ATT_ARMED);
		break;
	    case ATT_GRAPHICS:
		set (&(tw->vt.attributes), ATT_GRAPHICS);
		break;
	    case ATT_WIDGET:
		set (&(tw->vt.attributes), ATT_WIDGET);
		break;
	    case ATT_CURSOR2:
		set (&(tw->vt.attributes), ATT_CURSOR2);
		break;
	    case ATT_NOBLINK:
		clr (&(tw->vt.attributes), ATT_BLINK);
		break;
	    case ATT_NOBOLD:
		clr (&(tw->vt.attributes), ATT_BOLD);
		break;
	    case ATT_NODIM:
		clr (&(tw->vt.attributes), ATT_DIM);
		break;
	    case ATT_NOUNDERSCORE:
		clr (&(tw->vt.attributes), ATT_UNDERSCORE);
		break;
	    case ATT_NOINVERSE:
		clr (&(tw->vt.attributes), ATT_INVERSE);
		break;
	    case ATT_NOCURSOR:
		clr (&(tw->vt.attributes), ATT_CURSOR);
		break;
	    case ATT_NOSELECT:
		clr (&(tw->vt.attributes), ATT_SELECT);
		break;
	    case ATT_NOSPECIAL:
		clr (&(tw->vt.attributes), ATT_SPECIAL);
		break;
	    case ATT_NOARMED:
		clr (&(tw->vt.attributes), ATT_ARMED);
		break;
	    case ATT_NOGRAPHICS:
		clr (&(tw->vt.attributes), ATT_GRAPHICS);
		break;
	    case ATT_NOWIDGET:
		clr (&(tw->vt.attributes), ATT_WIDGET);
		break;
	    case ATT_NOCURSOR2:
		clr (&(tw->vt.attributes), ATT_CURSOR2);
		break;
	}
	parsed = strtok (NULL, ";");
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cup_move                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function moves the cursor to a specified location.      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, xc, yc                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cup_move (XmVtWidget tw, int xc, int yc)
{
    if (tw->vt.decom_mode)
    {
	tw->vt.screen_cx = xc;
	tw->vt.screen_cy = yc + (tw->vt.miny - tw->vt.real_miny);
	if (tw->vt.screen_cy > tw->vt.maxy)
	    tw->vt.screen_cy = tw->vt.maxy;
	if (tw->vt.screen_cy < tw->vt.miny)
	    tw->vt.screen_cy = tw->vt.miny;
	if (tw->vt.screen_cx > tw->vt.maxx)
	    tw->vt.screen_cx = tw->vt.maxx;
	if (tw->vt.screen_cx < tw->vt.minx)
	    tw->vt.screen_cx = tw->vt.minx;
    }
    else
    {
	tw->vt.screen_cx = xc;
	tw->vt.screen_cy = yc;
	if (tw->vt.screen_cy > tw->vt.real_maxy)
	    tw->vt.screen_cy = tw->vt.real_maxy;
	if (tw->vt.screen_cy < tw->vt.real_miny)
	    tw->vt.screen_cy = tw->vt.real_miny;
	if (tw->vt.screen_cx > tw->vt.real_maxx)
	    tw->vt.screen_cx = tw->vt.real_maxx;
	if (tw->vt.screen_cx < tw->vt.real_minx)
	    tw->vt.screen_cx = tw->vt.real_minx;
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cuu_move                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function moves the cursor up.                           */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cuu_move (XmVtWidget tw, int number)
{
    int inscroll = 0;
    if (number == 0)
	number = 1;
    inscroll = (tw->vt.screen_cy <= tw->vt.maxy && tw->vt.screen_cy >= tw->vt.miny);
    tw->vt.screen_cy = tw->vt.screen_cy - number;
    if (tw->vt.decom_mode || inscroll)
    {
	if (tw->vt.screen_cy < tw->vt.miny)
	    tw->vt.screen_cy = tw->vt.miny;
    }
    else
    {
	if (tw->vt.screen_cy < tw->vt.real_miny)
	    tw->vt.screen_cy = tw->vt.real_miny;
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cud_move                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function moves the cursor down.                         */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cud_move (XmVtWidget tw, int number)
{
    int inscroll = 0;
    if (number == 0)
	number = 1;
    inscroll = (tw->vt.screen_cy <= tw->vt.maxy && tw->vt.screen_cy >= tw->vt.miny);
    tw->vt.screen_cy = tw->vt.screen_cy + number;
    if (tw->vt.screen_cy > tw->vt.maxy)
	tw->vt.screen_cy = tw->vt.maxy;
    if (tw->vt.decom_mode || inscroll)
    {
	if (tw->vt.screen_cy > tw->vt.maxy)
	    tw->vt.screen_cy = tw->vt.maxy;
    }
    else
    {
	if (tw->vt.screen_cy > tw->vt.real_maxy)
	    tw->vt.screen_cy = tw->vt.real_maxy;
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cuf_move                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function moves the cursor forward.                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cuf_move (XmVtWidget tw, int number)
{
    if (number == 0)
	number = 1;
    tw->vt.screen_cx = tw->vt.screen_cx + number;
    if (tw->vt.screen_cx > tw->vt.maxx)
	tw->vt.screen_cx = tw->vt.maxx;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cub_move                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function moves the cursor backwards.                    */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             number        - int        - Number of actions to take        */
/*+                                                                           */
/*+ Input:      tw, number                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cub_move (XmVtWidget tw, int number)
{
    if (number == 0)
	number = 1;
    tw->vt.screen_cx = tw->vt.screen_cx - number;
    if (tw->vt.screen_cx < tw->vt.minx)
	tw->vt.screen_cx = tw->vt.minx;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: dynamic_tmpnam                                             */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function works similar to tmpnam (on which it relies)   */
/*+              except for two differences. First, it does not take any      */
/*+              parameters like tmpnam. It returns the value directly.       */
/*+              Secondly, it returns a dynamic value that must be freed      */
/*+              after use. This allows saving the value for future use such  */
/*+              as deleting the file.                                        */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static char *dynamic_tmpnam ()
{
    char *retval = NULL;
    char *internal_value = NULL;

    internal_value = tmpnam (NULL);
    retval = (char *) malloc (strlen (internal_value) + 1);
    strcpy (retval, internal_value);
    return (retval);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: isnonview                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine finds those characters that are non viewable    */
/*+              for the Vt widget. it returns True if the passed character   */
/*+              is not viewable. The general rules for the vtwidget is to    */
/*+              not print any characters less that character 28 except for   */
/*+              character 22. The reason for this set up is to allow the     */
/*+              printing of DOS ANSI graphics characters (something rxvt and */
/*+              color_xterm do not do). There still may be some DOS ANSI     */
/*+              graphics characters not printed, I have no references here.  */
/*+              I just test with a nice ANSI artwork file I found.           */
/*+                                                                           */
/*+ Parameters: value         - unsigned int                                  */
/*+                                        - Value to paste                   */
/*+                                                                           */
/*+ Input:      value                                                         */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static int isnonview (unsigned int value)
{
    int rcode = 0;

    rcode = (value < 28 && value != 22);
    return (rcode);
}





/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: self_test                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function fills the screen with 'E's. This is used as a  */
/*+              self test for most DEC terminals.                            */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void self_test (XmVtWidget tw)
{
    int ycoo = 1;
    attribute attr;

    reset_attr (&attr);
    for (ycoo = tw->vt.miny; ycoo <= tw->vt.maxy; ycoo++)
    {
	s_common (tw, tw->vt.minx, ycoo, tw->vt.columns, attr, 'E', CS_USASCII);
	s_line_attr (tw, ycoo, 0);
    }
    ok_update (tw);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: newline                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function begins a new line. It does not need to check   */
/*+              any modes unline the normal carriage return. It is basicly a */
/*+              carriage return/ line feed.                                  */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void newline (XmVtWidget tw)
{
    cancel_eol (tw);
    tw->vt.screen_cx = tw->vt.minx;
    tw->vt.screen_cy++;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: cancel_eol                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is called right before a control sequence that */
/*+              will change the cursor position. It cancels a preparation    */
/*+              for autowrap that might be in progress.                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void cancel_eol (XmVtWidget tw)
{
    tw->vt.stored_cx = 0;
    tw->vt.stored_cy = 0;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: tab_clear                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function clears a tab at the current cursor location.   */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void tab_clear (XmVtWidget tw)
{
    tw->vt.tabs[tw->vt.screen_cx] = ' ';
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: tab_set                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets a tab at the current cursor location.     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void tab_set (XmVtWidget tw)
{
    tw->vt.tabs[tw->vt.screen_cx] = 'I';
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: tab_clear_all                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function clears all tabs.                               */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void tab_clear_all (XmVtWidget tw)
{
    int inx = 0;
    for (inx = tw->vt.minx; inx <= tw->vt.maxx; inx++)
    {
	tw->vt.tabs[inx] = ' ';
    }
}
/*****************************************************************************/
/*                                                                           */
/* The following routines all access low level screen data structures.       */
/* Some of them are static but many are called from the Vt.c module as well  */
/* as from within this module.  If you change the vtscreen data structures   */
/* be very certain that these routines are working properly.  I have         */
/* localized these routines here to find them if such changes need to be     */
/* made.  The reason that they are not part of a separate module is that     */
/* when I would optimize routines above they would suddenly access low level */
/* data elements meaning I would need to move them to the other module.      */
/* This movement becomes tedious and it becomes hard to define something     */
/* as low level when it calls so many high level routines.                   */
/*                                                                           */
/*****************************************************************************/

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: resize_data                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: resize data changes the values of the rows and columns of    */
/*+              the screen. It does some dynamic memory changes and ensures  */
/*+              all memory is accounted for.                                 */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             maxx_value    - int        - Maximum X coord Value            */
/*+             maxy_value    - int        - Maximum Y coord Value            */
/*+                                                                           */
/*+ Input:      tw, maxx_value, maxy_value                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void resize_data (XmVtWidget tw, int maxx_value, int maxy_value)
{
    VTLINE *new_screen = NULL;

    int r_data = 0;
    int r_lattr = 0;
    attribute r_attr = BLANK_ATTR;
    int r_cset = 0;
    int jnx = 0;
    int knx = 0;
    int inx = 0;
    int limitx = 0;
    int limity = 0;
    int addition = 0;
    int new_fr = 0;
    int new_cy = 0;
    int no_skip = -1;
    int maxhist = tw->vt.max_history;
    int last_length = tw->vt.prev_max_history + tw->vt.real_maxy;

    addition = (maxy_value - tw->vt.real_maxy);
    new_cy = tw->vt.screen_cy + addition;
    new_fr = 2 * (maxy_value + maxhist) - addition;
    if (new_fr < 0)
	new_fr = -1 * new_fr;
    if (new_cy < 1)
    {
	new_fr = new_fr + new_cy - 1;
	new_cy = 1;
    }
    new_screen = alc_vtscreen (maxy_value + maxhist + 1, maxx_value + 1);
    if (maxx_value > tw->vt.real_maxx)
	limitx = tw->vt.real_maxx;
    else
	limitx = maxx_value;
    knx = ((tw->vt.first_row) % last_length);
    for (jnx = 0; jnx < maxy_value + maxhist; jnx++)
    {
	if (knx >= last_length)
	    knx = knx - last_length;
	for (inx = 0; inx <= maxx_value; inx++)
	{
	    r_data = ' ';
	    reset_attr (&r_attr);
	    r_cset = CS_USASCII;
	    r_lattr = 0;
	    if (jnx < last_length + addition && inx <= limitx && no_skip)
	    {
		r_data = tw->vt.vtscreen[knx].data[inx];
		r_cset = tw->vt.vtscreen[knx].cset[inx];
		r_attr.attrib0 = tw->vt.vtscreen[knx].attrib0[inx];
		r_attr.attrib1 = tw->vt.vtscreen[knx].attrib1[inx];
		r_attr.attrib2 = tw->vt.vtscreen[knx].attrib2[inx];
		r_attr.attrib3 = tw->vt.vtscreen[knx].attrib3[inx];
		r_attr.attrib4 = tw->vt.vtscreen[knx].attrib4[inx];
		r_attr.attrib5 = tw->vt.vtscreen[knx].attrib5[inx];
		r_attr.attrib6 = tw->vt.vtscreen[knx].attrib6[inx];
		r_attr.attrib7 = tw->vt.vtscreen[knx].attrib7[inx];
		r_lattr = tw->vt.vtscreen[knx].line_attribute;
	    }
	    new_screen[jnx].data[inx] = r_data;
	    new_screen[jnx].cset[inx] = r_cset;
	    new_screen[jnx].flag[inx] = S_MOD;
	    new_screen[jnx].attrib0[inx] = r_attr.attrib0;
	    new_screen[jnx].attrib1[inx] = r_attr.attrib1;
	    new_screen[jnx].attrib2[inx] = r_attr.attrib2;
	    new_screen[jnx].attrib3[inx] = r_attr.attrib3;
	    new_screen[jnx].attrib4[inx] = r_attr.attrib4;
	    new_screen[jnx].attrib5[inx] = r_attr.attrib5;
	    new_screen[jnx].attrib6[inx] = r_attr.attrib6;
	    new_screen[jnx].attrib7[inx] = r_attr.attrib7;
	    new_screen[jnx].line_attribute = r_lattr;
	}
	knx++;
	if (jnx == tw->vt.real_maxy && addition > 0)
	{
	    no_skip = 0;
	}
	if (!no_skip && jnx == tw->vt.real_maxy + addition)
	{
	    no_skip = -1;
	    knx = knx - addition;
	}
    }
    free_data (tw);
    if (!tw->vt.debug_stdout && tw->vt.debug)
	debug_fp = fopen (DEBUGFL, WRITEMODE);

    tw->vt.vtscreen = new_screen;
    tw->vt.maxx = maxx_value;
    tw->vt.maxy = maxy_value;
    tw->vt.real_maxy = maxy_value;
    tw->vt.real_maxx = maxx_value;
    sync_cursor (tw);
    tw->vt.save_cx = tw->vt.screen_cx;
    tw->vt.save_cy = tw->vt.screen_cy;
    tw->vt.first_row = new_fr;
    tw->vt.screen_cy = new_cy;
    for (inx = 1; inx <= tw->vt.maxx; inx++)
	if ((inx - 1) % 8)
	    tw->vt.tabs[inx] = ' ';
	else
	    tw->vt.tabs[inx] = 'I';
    tw->vt.tabs[tw->vt.maxx + 1] = 0;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: free_data                                                  */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: free_data frees all allocated dynamic memory associated with */
/*+              the screen. It also frees other dynamic allocations as well  */
/*+              as closes some files that may be open.                       */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void free_data (XmVtWidget tw)
{
    int maxhist = tw->vt.prev_max_history;
    free_vtscreen (tw->vt.maxy + maxhist, tw->vt.maxx, tw->vt.vtscreen);
    tw->vt.prev_max_history = tw->vt.max_history;
    if (!tw->vt.debug_stdout && tw->vt.debug)
	fclose (debug_fp);
    if (tw->vt.data_name != NULL)
    {
	free (tw->vt.data_name);
	fclose (tw->vt.data_fp);
	unlink (tw->vt.data_name);
    }
    if (tw->vt.graphics_name != NULL)
    {
	free (tw->vt.graphics_name);
	fclose (tw->vt.data_fp);
	unlink (tw->vt.graphics_name);
    }
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: free_vtscreen                                              */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to free all of the allocated memory in  */
/*+              a VTLINE array. A VTLINE array is the memory structure used  */
/*+              to store all of the screen information.                      */
/*+                                                                           */
/*+ Parameters: row           - int        - Count of rows in 2D array        */
/*+             strwidth      - int        - Actual string width of strings   */
/*+             vtscreen_name - VTLINE *   - This is the current vtscreen to  */
/*+                                          manipulate.                      */
/*+                                                                           */
/*+ Input:      row, strwidth, vtscreen_name                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
static void free_vtscreen (int row, int strwidth, VTLINE * vtscreen_name)
{
    int inx = 0;
    for (inx = 0; inx < row; inx++)
    {
	free (vtscreen_name[inx].data);
	free (vtscreen_name[inx].cset);
	free (vtscreen_name[inx].flag);
	free (vtscreen_name[inx].attrib0);
	free (vtscreen_name[inx].attrib1);
	free (vtscreen_name[inx].attrib2);
	free (vtscreen_name[inx].attrib3);
	free (vtscreen_name[inx].attrib4);
	free (vtscreen_name[inx].attrib5);
	free (vtscreen_name[inx].attrib6);
	free (vtscreen_name[inx].attrib7);
    }
    free (vtscreen_name);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: alc_vtscreen                                               */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to allocate the memory used for the     */
/*+              VTLINE array used to store the screen information.           */
/*+                                                                           */
/*+ Parameters: row           - int        - Count of rows in 2D array        */
/*+             strwidth      - int        - Actual string width of strings   */
/*+                                                                           */
/*+ Input:      row, strwidth                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 11/07/95    Bruce M. Corwin   Initial Release                             */
/*+ 04/22/97    Bruce M. Corwin   Made sure to initialize line_attribute      */
/*+                                                                           */
/*+****************************************************************************/
static VTLINE *alc_vtscreen (int row, int strwidth)
{
    VTLINE *ret_array;
    int inx = 0;
    ret_array = (VTLINE *) malloc (row * sizeof (VTLINE));
    for (inx = 0; inx < row; inx++)
    {
	ret_array[inx].line_attribute = 0;
	ret_array[inx].data = (char *) malloc (strwidth);
	ret_array[inx].cset = (char *) malloc (strwidth);
	ret_array[inx].flag = (char *) malloc (strwidth);
	ret_array[inx].attrib0 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib1 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib2 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib3 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib4 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib5 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib6 = (unsigned char *) malloc (strwidth);
	ret_array[inx].attrib7 = (unsigned char *) malloc (strwidth);
    }
    return (ret_array);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_attr                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets the attribute section of the screen       */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+             attr          - attribute  - Type of characters               */
/*+                                                                           */
/*+ Input:      tw, xc, yc, attr                                              */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_attr (XmVtWidget tw, int xc, int yc, attribute attr)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    tw->vt.vtscreen[ya].attrib0[xc] = attr.attrib0;
    tw->vt.vtscreen[ya].attrib1[xc] = attr.attrib1;
    tw->vt.vtscreen[ya].attrib2[xc] = attr.attrib2;
    tw->vt.vtscreen[ya].attrib3[xc] = attr.attrib3;
    tw->vt.vtscreen[ya].attrib4[xc] = attr.attrib4;
    tw->vt.vtscreen[ya].attrib5[xc] = attr.attrib5;
    tw->vt.vtscreen[ya].attrib6[xc] = attr.attrib6;
    tw->vt.vtscreen[ya].attrib7[xc] = attr.attrib7;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_data                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets the data section of the screen buffer.    */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+             data          - int        - Data to append                   */
/*+                                                                           */
/*+ Input:      tw, xc, yc, data                                              */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_data (XmVtWidget tw, int xc, int yc, int data)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    tw->vt.vtscreen[ya].data[xc] = data;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_line_attr                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets the line attribute for a line of the      */
/*+              screen. buffer.                                              */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             yc            - int        - Y coordinate                     */
/*+             ldata         - int        - Line attribute information.      */
/*+                                                                           */
/*+ Input:      tw, yc, ldata                                                 */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_line_attr (XmVtWidget tw, int yc, int ldata)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    tw->vt.vtscreen[ya].line_attribute = ldata;
    memset (tw->vt.vtscreen[ya].flag, S_MOD, tw->vt.columns + 1);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_cset                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets the character set section of the screen   */
/*+              buffer.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+             cset          - int        - Character set                    */
/*+                                                                           */
/*+ Input:      tw, xc, yc, cset                                              */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_cset (XmVtWidget tw, int xc, int yc, int cset)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    tw->vt.vtscreen[ya].cset[xc] = cset;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_flag                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets the modify flag section of the screen     */
/*+              buffer.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+             flag          - int        - Modified flag                    */
/*+                                                                           */
/*+ Input:      tw, xc, yc, flag                                              */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_flag (XmVtWidget tw, int xc, int yc, int flag)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    tw->vt.vtscreen[ya].flag[xc] = flag;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: g_attr                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function gets the attribute section of the screen       */
/*+              buffer.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, xc, yc                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
attribute g_attr (XmVtWidget tw, int xc, int yc)
{
    int ya = 0;
    attribute retval;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    retval.attrib0 = tw->vt.vtscreen[ya].attrib0[xc];
    retval.attrib1 = tw->vt.vtscreen[ya].attrib1[xc];
    retval.attrib2 = tw->vt.vtscreen[ya].attrib2[xc];
    retval.attrib3 = tw->vt.vtscreen[ya].attrib3[xc];
    retval.attrib4 = tw->vt.vtscreen[ya].attrib4[xc];
    retval.attrib5 = tw->vt.vtscreen[ya].attrib5[xc];
    retval.attrib6 = tw->vt.vtscreen[ya].attrib6[xc];
    retval.attrib7 = tw->vt.vtscreen[ya].attrib7[xc];
    return (retval);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: g_data                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function gets the data section of the screen buffer.    */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, xc, yc                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int g_data (XmVtWidget tw, int xc, int yc)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    return (tw->vt.vtscreen[ya].data[xc]);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: g_line_attr                                                */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function gets the line attributes for a line of the     */
/*+              screen.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, yc                                                        */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int g_line_attr (XmVtWidget tw, int yc)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    return (tw->vt.vtscreen[ya].line_attribute);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: g_cset                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function gets the character set section of the screen   */
/*+              buffer.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, xc, yc                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int g_cset (XmVtWidget tw, int xc, int yc)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    return (tw->vt.vtscreen[ya].cset[xc]);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: g_flag                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function gets the modify flag section of the screen     */
/*+              buffer.                                                      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+                                                                           */
/*+ Input:      tw, xc, yc                                                    */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int g_flag (XmVtWidget tw, int xc, int yc)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    return (tw->vt.vtscreen[ya].flag[xc]);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_common                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function sets many characteristics across a defined     */
/*+              section of a line. It will set the data, attributes, and     */
/*+              character set information for a whole line if necesary. This */
/*+              function is built to provide optimization for the many calls */
/*+              made to clear lines or set lines to a particular value.      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             xc            - int        - X coordinate                     */
/*+             yc            - int        - Y coordinate                     */
/*+             count         - int        - Count of characters to write     */
/*+             attr          - attribute  - Type of characters               */
/*+             data          - int        - Data to append                   */
/*+             cset          - int        - Character set                    */
/*+                                                                           */
/*+ Input:      tw, xc, yc, count, attr, data, cset                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_common (XmVtWidget tw, int xc, int yc, int count, attribute attr, int data, int cset)
{
    int ya = 0;
    int maxhist = tw->vt.max_history;
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    memset ((tw->vt.vtscreen[ya].data + xc), data, count);
    memset ((tw->vt.vtscreen[ya].flag + xc), S_MOD, count);
    memset ((tw->vt.vtscreen[ya].cset + xc), cset, count);
    memset ((tw->vt.vtscreen[ya].attrib0 + xc), attr.attrib0, count);
    memset ((tw->vt.vtscreen[ya].attrib1 + xc), attr.attrib1, count);
    memset ((tw->vt.vtscreen[ya].attrib2 + xc), attr.attrib2, count);
    memset ((tw->vt.vtscreen[ya].attrib3 + xc), attr.attrib3, count);
    memset ((tw->vt.vtscreen[ya].attrib4 + xc), attr.attrib4, count);
    memset ((tw->vt.vtscreen[ya].attrib5 + xc), attr.attrib5, count);
    memset ((tw->vt.vtscreen[ya].attrib6 + xc), attr.attrib6, count);
    memset ((tw->vt.vtscreen[ya].attrib7 + xc), attr.attrib7, count);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_swap                                                     */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This function is used to swap one line and another line by   */
/*+              reassigning pointers. This function is used for optimization */
/*+              of reverse index and scrolling type functions. It sets the   */
/*+              source line to modified (The source becomes the destination  */
/*+              then during the swap.) The destination line is considered    */
/*+              unimportant and so it is never identified as modified. I     */
/*+              found this to be faster than a s_copy function that used     */
/*+              memcpy to actualy move data. Now I just reassign pointers by */
/*+              swaping VTLINE structures.                                   */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             destination   - int        - This is the destination line to  */
/*+                                          copy to.                         */
/*+             source        - int        - This is the source line to copy  */
/*+                                          from.                            */
/*+                                                                           */
/*+ Input:      tw, destination, source                                       */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_swap (XmVtWidget tw, int destination, int source)
{
    VTLINE temp;
    int sa = 0;
    int da = 0;
    int count = 0;
    int maxhist = tw->vt.max_history;

    sa = (tw->vt.first_row - tw->vt.scroll_value + source) % (maxhist + tw->vt.real_maxy);
    da = (tw->vt.first_row - tw->vt.scroll_value + destination) % (maxhist + tw->vt.real_maxy);
    count = (tw->vt.columns + 1);
    memset (tw->vt.vtscreen[sa].flag, S_MOD, count);
    temp = tw->vt.vtscreen[da];
    tw->vt.vtscreen[da] = tw->vt.vtscreen[sa];
    tw->vt.vtscreen[sa] = temp;
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_modify                                                   */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine is used to set the S_MOD flag on an entire line */
/*+              of the Vt screen. This allows faster updates to this flag    */
/*+              when large sections of the screen need to be repainted.      */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             line_pos      - int        - This is the line position to set */
/*+                                          to S_MOD.                        */
/*+                                                                           */
/*+ Input:      tw, line_pos                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void s_modify (XmVtWidget tw, int line_pos)
{
    int sa = 0;
    int da = 0;
    int count = 0;
    int maxhist = tw->vt.max_history;

    da = (tw->vt.first_row - tw->vt.scroll_value + line_pos) % (maxhist + tw->vt.real_maxy);
    count = (tw->vt.columns + 1);
    memset (tw->vt.vtscreen[da].flag, S_MOD, count);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: s_multi                                                    */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine works a little like a combination of s_data,    */
/*+              s_attr, s_flag and s_cset. It also works with a complete     */
/*+              string. It adds multiple components to the vtscreen element  */
/*+              of the Vt widget. This routine could probably also be named  */
/*+              add_string because it works alot like insert_char except it  */
/*+              adds the data instead of inserting it and it adds a full     */
/*+              string of characters. This routine can only enough           */
/*+              characters to fill a line. If more characters are sent to it */
/*+              than what it needs to fill a line it will ignore them. It    */
/*+              does however return the number of characters it actualy      */
/*+              outputs.                                                     */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+             buffer        - char *     - Buffer of output data            */
/*+             count         - int        - Count of characters to write     */
/*+                                                                           */
/*+ Input:      tw, buffer, count                                             */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int s_multi (XmVtWidget tw, char *buffer, int count)
{
    int inx = 0;
    char current_cs = CS_USASCII;
    int localcnt = 0;
    int new_cx = 0;
    int total_cx = 0;
    int ya = 0;
    int yc = 0;
    int maxhist = tw->vt.max_history;

    current_cs = tw->vt.char_set[tw->vt.gr_cset_level];
    if (tw->vt.stored_cx != 0 && tw->vt.stored_cy != 0)
    {
	tw->vt.screen_cx = tw->vt.stored_cx;
	tw->vt.screen_cy = tw->vt.stored_cy;
	tw->vt.stored_cx = 0;
	tw->vt.stored_cy = 0;
	bottom_edge (tw);
    }
    yc = tw->vt.screen_cy;
    total_cx = tw->vt.screen_cx + count - 1;
    if (total_cx >= tw->vt.maxx)
    {
	localcnt = tw->vt.maxx - tw->vt.screen_cx + 1;
    }
    else
    {
	localcnt = count;
    }
    new_cx = tw->vt.screen_cx + localcnt;
    if (new_cx == tw->vt.maxx + 1)
    {
	tw->vt.stored_cx = tw->vt.minx;
	tw->vt.stored_cy = tw->vt.screen_cy + 1;
    }
    else
    {
	tw->vt.stored_cx = 0;
	tw->vt.stored_cy = 0;
    }
    ya = (tw->vt.first_row - tw->vt.scroll_value + yc) % (maxhist + tw->vt.real_maxy);
    memcpy
	((tw->vt.vtscreen[ya].data + tw->vt.screen_cx), buffer, localcnt);
    memset
	(
	    (tw->vt.vtscreen[ya].flag + tw->vt.screen_cx),
	    S_MOD,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].cset + tw->vt.screen_cx),
	    current_cs,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib0 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib0,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib1 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib1,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib2 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib2,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib3 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib3,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib4 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib4,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib5 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib5,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib6 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib6,
	    localcnt
	);
    memset
	(
	    (tw->vt.vtscreen[ya].attrib7 + tw->vt.screen_cx),
	    tw->vt.attributes.attrib7,
	    localcnt
	);
    if (tw->vt.screen_cx == tw->vt.maxx)
    {
	new_cx = tw->vt.screen_cx;
    }
    tw->vt.screen_cx = new_cx;
    if (!tw->vt.autowrap_mode)
	localcnt = count;
    return (localcnt);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: paint_data                                                 */
/*+                                                                           */
/*+ Program ID:    vtwidget                                                   */
/*+                                                                           */
/*+ Description: This routine paints the current data to the output device.   */
/*+              It accesses the memory structures of vt.vtscreen directly so */
/*+              if any changes are made to these structures coorisponding    */
/*+              changes need to be made here. This routine is rather hard to */
/*+              understand, especialy if you do not understand how Vt.c is   */
/*+              processing information it receives from screen.c. It has     */
/*+              been optimized to accelerate the painting of data on the Vt  */
/*+              widget window. This is my only excuse for such hard to read  */
/*+              code. I basicaly loop through the data line by line column   */
/*+              by column looking for modified data (S_MOD). I then try to   */
/*+              find similar chunks of screen info (same attributes and      */
/*+              cset) and plot them with the plot routine. At some point     */
/*+              another optimization will be to simply plot the data         */
/*+              directly from the vt.vtscreen area instead of memcpy-ing it  */
/*+              to a pldata area. This memcpy takes a little time when you   */
/*+              are looping it so much. This routine was originaly in Vt.c   */
/*+              but was moved to this module when it was rewritten to access */
/*+              screen data structures directly. It is now a low level       */
/*+              routine even though it calls the plot routine in Vt.c.       */
/*+                                                                           */
/*+ Parameters: tw            - XmVtWidget - VT Widget data                   */
/*+                                                                           */
/*+ Input:      tw                                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 02/15/96    Bruce M. Corwin   Initial Release                             */
/*+ 02/04/97    Bruce M. Corwin   Changed the value of S_MOD to 1             */
/*+                               instead of -1 for signed problems.          */
/*+                                                                           */
/*+****************************************************************************/
void paint_data (XmVtWidget tw)
{
    int xcoo = 1;
    int back = 1;
    int ycoo = 1;
    int skip = 0;
    attribute attr = BLANK_ATTR;
    char cset = CS_USASCII;
    char pldata[BUFSIZ];
    int ya = 0;
    int maxhist = tw->vt.max_history;

    pldata[0] = 0;
    tw->vt.scroll_line_count = 0;
    for (ycoo = tw->vt.real_miny; ycoo <= tw->vt.real_maxy; ycoo++)
    {
	ya = (tw->vt.first_row - tw->vt.scroll_value + ycoo) % (maxhist + tw->vt.real_maxy);
	for (xcoo = tw->vt.real_minx; xcoo <= tw->vt.real_maxx; xcoo++)
	{
	    back = xcoo;
	    attr.attrib0 = tw->vt.vtscreen[ya].attrib0[xcoo];
	    attr.attrib1 = tw->vt.vtscreen[ya].attrib1[xcoo];
	    attr.attrib2 = tw->vt.vtscreen[ya].attrib2[xcoo];
	    attr.attrib3 = tw->vt.vtscreen[ya].attrib3[xcoo];
	    attr.attrib4 = tw->vt.vtscreen[ya].attrib4[xcoo];
	    attr.attrib5 = tw->vt.vtscreen[ya].attrib5[xcoo];
	    attr.attrib6 = tw->vt.vtscreen[ya].attrib6[xcoo];
	    attr.attrib7 = tw->vt.vtscreen[ya].attrib7[xcoo];
	    cset = tw->vt.vtscreen[ya].cset[xcoo];
	    if (tw->vt.vtscreen[ya].flag[xcoo] == S_MOD)
	    {
		while
		    (
			xcoo <= tw->vt.real_maxx &&
			tw->vt.vtscreen[ya].flag[xcoo] == S_MOD &&
			tw->vt.vtscreen[ya].attrib0[xcoo] == attr.attrib0 &&
			tw->vt.vtscreen[ya].attrib1[xcoo] == attr.attrib1 &&
			tw->vt.vtscreen[ya].attrib2[xcoo] == attr.attrib2 &&
			tw->vt.vtscreen[ya].attrib3[xcoo] == attr.attrib3 &&
			tw->vt.vtscreen[ya].attrib4[xcoo] == attr.attrib4 &&
			tw->vt.vtscreen[ya].attrib5[xcoo] == attr.attrib5 &&
			tw->vt.vtscreen[ya].attrib6[xcoo] == attr.attrib6 &&
			tw->vt.vtscreen[ya].attrib7[xcoo] == attr.attrib7 &&
			tw->vt.vtscreen[ya].cset[xcoo] == cset
		    )
		    xcoo++;
		memcpy (pldata, tw->vt.vtscreen[ya].data + back, xcoo - back);
		pldata[xcoo - back] = 0;
		memset (tw->vt.vtscreen[ya].flag + back, S_SAME, xcoo - back);
		plot (tw, pldata, attr, cset, back, ycoo);
		xcoo--;
	    }
	}
    }
    unmanage_unpainted_widgets (tw);
}
