/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: transformer.c,v 1.8.2.1 96/03/14 11:51:48 leon Exp $ */
/*
 * This is a translation of the Transformer C++ class defined in InterViews2.6,
 * to pure C code, YEAH!
 *
 * Arnaud Le Hors - 20 May 94
 */

/*
 * Copyright (c) 1987, 1988, 1989 Stanford University
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Stanford not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Stanford makes no representations about
 * the suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * STANFORD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * Implementation of transformation matrix operations.
 */

#include "transformer.h"
#include <math.h>

#ifndef M_PI
#define M_PI            3.14159265358979323846
#endif

#ifdef __STDC__
static const float RADPERDEG = M_PI / 180.0;
#else
#define RADPERDEG (M_PI / 180.0)
#endif

/*
  
  The initial round function is:
  inline int round (double x) { return x > 0 ? int(x+0.5) : -int(-x+0.5); }
  got 3 bugs converting this:
  - forgot the () around x
  - had x evaluated twice
  - had a conversion problem with
  #define round(x) ( (rtmp=(x)) > 0 ? ((rtmp) + 0.5) : -(-(rtmp) + 0.5) )
    That I replaced with the function

static int round(float x)
{
    return (x > 0 ? (x + 0.5) : -(-x + 0.5) );
}

*/
#if 0
static int round(float x)
{
    return (x > 0 ? (x + 0.5) : -(-x + 0.5) );
}
#endif

static int floattoint(float x)
{
    return (x > 0 ? (x + 0.5) : -(-x + 0.5) );
}

#define round(a) a

Transformer 
TransformerMake()
{
    Transformer t = (Transformer) malloc(sizeof(TransformerStruct));

    if (t) {
	/* identity */
	t->flags = T_IDENTITY;	/* allow auto deletion if = identity */
	t->mat00 = t->mat11 = 1;
	t->mat01 = t->mat10 = t->mat20 = t->mat21 = 0;
    }
    return t;
}

Transformer 
TransformerMakeCopy(Transformer t)
{
    Transformer t2 = (Transformer) malloc(sizeof(TransformerStruct));

    if (t2)
	TransformerSet(t2, t);
    t2->flags = t->flags;
    return t2;
}

void
TransformerTranslate(Transformer t, float dx, float dy)
{
    t->mat20 += dx;
    t->mat21 += dy;
    t->flags |= T_TRANSLATED;
}
 
void
TransformerScale(Transformer t, float sx, float sy)
{
    t->mat00 *= sx;
    t->mat01 *= sy;
    t->mat10 *= sx;
    t->mat11 *= sy;
    t->mat20 *= sx;
    t->mat21 *= sy;
    t->flags |= T_SCALED;
}

void
TransformerRotate(Transformer t, float angle)
{
    float tmp1, tmp2, m00, m01, m10, m11, m20, m21;

    angle *= RADPERDEG;
    tmp1 = cos(angle);
    tmp2 = sin(angle);

    m00 = t->mat00 * tmp1;
    m01 = t->mat01 * tmp2;
    m10 = t->mat10 * tmp1;
    m11 = t->mat11 * tmp2;
    m20 = t->mat20 * tmp1;
    m21 = t->mat21 * tmp2;

    t->mat01 = t->mat00 * tmp2 + t->mat01 * tmp1;
    t->mat11 = t->mat10 * tmp2 + t->mat11 * tmp1;
    t->mat21 = t->mat20 * tmp2 + t->mat21 * tmp1;
    t->mat00 = m00 - m01;
    t->mat10 = m10 - m11;
    t->mat20 = m20 - m21;
    t->flags |= T_ROTATED;
}

void
TransformerPremultiply(Transformer t, Transformer t2)
{
    float tmp1 = t->mat00;
    float tmp2 = t->mat10;

    t->mat00 = t2->mat00 * tmp1 + t2->mat01 * tmp2;
    t->mat10 = t2->mat10 * tmp1 + t2->mat11 * tmp2;
    t->mat20 += t2->mat20 * tmp1 + t2->mat21 * tmp2;

    tmp1 = t->mat01;
    tmp2 = t->mat11;

    t->mat01 = t2->mat00 * tmp1 + t2->mat01 * tmp2;
    t->mat11 = t2->mat10 * tmp1 + t2->mat11 * tmp2;
    t->mat21 += t2->mat20 * tmp1 + t2->mat21 * tmp2;

    t->flags |= t2->flags;
}

void
TransformerPostmultiply(Transformer t, Transformer t2)
{
    float tmp = t->mat00 * t2->mat01 + t->mat01 * t2->mat11;

    t->mat00 = t->mat00 * t2->mat00 + t->mat01 * t2->mat10;
    t->mat01 = tmp;

    tmp = t->mat10 * t2->mat01 + t->mat11 * t2->mat11;
    t->mat10 = t->mat10 * t2->mat00 + t->mat11 * t2->mat10;
    t->mat11 = tmp;

    tmp = t->mat20 * t2->mat01 + t->mat21 * t2->mat11;
    t->mat20 = t->mat20 * t2->mat00 + t->mat21 * t2->mat10;
    t->mat21 = tmp;

    t->mat20 += t2->mat20;
    t->mat21 += t2->mat21;
    
    t->flags |= t2->flags;
}

void
TransformerInvert(Transformer t)
{
    float d = TransformerDet(t);
    float t00 = t->mat00;
    float t20 = t->mat20;

    t->mat20 = (t->mat10 * t->mat21 - t->mat11 * t->mat20) / d;
    t->mat21 = (t->mat01 * t20 - t->mat00 * t->mat21) / d;
    t->mat00 = t->mat11 / d;
    t->mat11 = t00 / d;
    t->mat10 = -t->mat10 / d;
    t->mat01 = -t->mat01 / d;
}

void
TransformerTransform(Transformer t, KnPosition x, KnPosition y, KnPosition *tx, KnPosition *ty)
{
    KnPosition tmpx;
    tmpx = round(x * t->mat00 + y * t->mat10 + t->mat20);
    *ty = round(x * t->mat01 + y * t->mat11 + t->mat21);
    *tx = tmpx;
}

/* transform an XPoint
 * be carefull to preserve original x, so that target point may be source point
 * returns: 
 */
void
TransformerTransformPoint(Transformer t, XPoint *p, XPoint *top)
{
    KnPosition tmpx;
    if(t->flags == T_TRANSLATED) {
	/* optimize for translation only */
	top->x = round(p->x + t->mat20);
	top->y = round(p->y + t->mat21);
    }
    else {
	tmpx = round(p->x * t->mat00 + p->y * t->mat10 + t->mat20);
	top->y = round(p->x * t->mat01 + p->y * t->mat11 + t->mat21);
	top->x = tmpx;
    }
}



void
TransformerTransformPointList(Transformer t, XPoint points[], Cardinal num, 
			      XPoint topoints[])
{
    register XPoint *o, *n;
    XPoint *lim;

    lim = &points[num];
    for (o = points, n = topoints; o < lim; o++, n++) {
	TransformerTransformPoint(t, o, n);
    }
}


void
TransformerTransformBoxPoints(Transformer t,
			      KnPosition x, KnPosition y,
			      KnDimension w, KnDimension h,
			      XPoint *points)
{
    float tmp0, tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7;
    
    tmp0 = (float)x * t->mat00;
    tmp1 = (float)x * t->mat01;
    tmp2 = (float)y * t->mat10;
    tmp3 = (float)y * t->mat11;
    tmp4 = (float)w * t->mat00;
    tmp5 = (float)w * t->mat01;
    tmp6 = (float)h * t->mat10;
    tmp7 = (float)h * t->mat11;

    points[0].x = tmp0 + tmp2 + t->mat20;
    points[0].y = tmp1 + tmp3 + t->mat21;
    
    points[1].x = tmp0 + tmp4 + tmp2 + t->mat20;
    points[1].y = tmp1 + tmp5 + tmp3 + t->mat21;
    
    points[2].x = tmp0  + tmp4 + tmp2 + tmp6 + t->mat20;
    points[2].y = tmp1 + tmp5 + tmp3 + tmp7 + t->mat21;
    
    points[3].x = tmp0 + tmp2 + tmp6 + t->mat20;
    points[3].y = tmp1 + tmp3 + tmp7 + t->mat21;
}



void
TransformerTransformFloat(Transformer t, float x, float y, float *tx, float *ty)
{
    *tx = x * t->mat00 + y * t->mat10 + t->mat20;
    *ty = x * t->mat01 + y * t->mat11 + t->mat21;
}

void
TransformerInvTransform(Transformer t, KnPosition tx, KnPosition ty, KnPosition *x, KnPosition *y)
{
    float d = TransformerDet(t);
    float a = ((float)(tx) - t->mat20) /d;
    float b = ((float)(ty) - t->mat21) /d;

    *x = round(a * t->mat11 - b * t->mat10);
    *y = round(b * t->mat00 - a * t->mat01);
}

void
TransformerInvTransformFloat(Transformer t, float tx, float ty, float *x, float *y)
{
    float d = TransformerDet(t);
    float a = (tx - t->mat20) / d;
    float b = (ty - t->mat21) / d;

    *x = a * t->mat11 - b * t->mat10;
    *y = b * t->mat00 - a * t->mat01;
}

void
TransformerTransformList(Transformer t, KnPosition x[], KnPosition y[], Cardinal n, KnPosition tx[], KnPosition ty[])
{
    register KnPosition *ox, *oy, *nx, *ny;
    KnPosition *lim;

    lim = &x[n];
    for (ox = x, oy = y, nx = tx, ny = ty; ox < lim; ox++, oy++, nx++, ny++) {
	TransformerTransform(t, *ox, *oy, nx, ny);
    }
}

void
TransformerInvTransformList(Transformer t, KnPosition tx[], KnPosition ty[], Cardinal n, KnPosition x[], KnPosition y[])
{
    register KnPosition *ox, *oy, *nx, *ny;
    KnPosition *lim;
    float a, b, d = TransformerDet(t);

    lim = &tx[n];
    for (ox = tx, oy = ty, nx = x, ny = y; ox < lim; ox++, oy++, nx++, ny++) {
	a = ((float)(*ox) - t->mat20) /d;
	b = ((float)(*oy) - t->mat21) /d;

	*nx = round(a * t->mat11 - b * t->mat10);
	*ny = round(b * t->mat00 - a * t->mat01);
    }
}

int
TransformerEqual(Transformer t, Transformer t2)
{
    return
	t->mat00 == t2->mat00 && t->mat01 == t2->mat01 &&
	t->mat10 == t2->mat10 && t->mat11 == t2->mat11 &&
	t->mat20 == t2->mat20 && t->mat21 == t2->mat21;
}
