/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: rect.c,v 1.11.2.2 96/03/14 16:39:57 leon Exp $ */
#include <math.h>
#include "rect.h"
#include "TagP.h"
#include "misc.h"
#include "utils.h"
#ifdef LOOK3D
#include <Xm/Xm.h>
#endif /* LOOK3D */

/*********************\
* 		      *
*  KlO  KnRect          *
*  BODY		      *
* 		      *
\*********************/



/* type desc */
KlType KnRectClass;
KlType KnFRectClass;


/* init class fields and call super class Init proc */
void
KnRectInit(KnRect rect,
	   Widget knvas,
	   KnO parent, 
	   KnPosition x,
	   KnPosition y,
	   KnDimension width,
	   KnDimension height)
{
    KnOInit((KnO)rect, knvas, parent, x, y);
    rect->width = width;
    rect->height = height;
}


/* allocates storage space for a new rect
 * 
 * returns: the newly created KnRect
 */
KnO
KnCreateRect(Widget knvas, KnO parent, KnPosition x, KnPosition y, KnDimension width, KnDimension height)
{
    KnRect rect;
    rect =  (KnRect)KlOMake(KnRectClass);
    KnRectInit(rect, knvas, parent, x, y, width, height);
    return (KnO)rect;
}


void
KnRectCopyFields(KnRect new, KnRect old)
{
    KnOCopyFields(new, old);
}


void
KnRectDraw(KnRect rect, KnDrawingContext ct, Transformer tf) 
{
    Transformer t;    
    XPoint points[5];
    XRectangle box;
    TagObject tag;
    tag = (TagObject)KnGetTag(ct->w, (KnO)rect);

    rect->flags &= ~KnCLEARED_FLAG;
    t = tf;
    if(t) {

	TransformerTransformBoxPoints(t, rect->x, rect->y,
				      rect->width, rect->height,
				      points);

	points[4].x = points[0].x;
	points[4].y = points[0].y;
	
	GetBoundingBox(points, 4, &box);

	if(KnRectInRegion(ct->clip, box.x, box.y, box.width, box.height)) {
#ifdef LOOK3D_FOR_RECT
	    if(tag->tag.shadowThickness) {
		GC top_gc, bottom_gc;
		KnDimension top_thick, bottom_thick;
		int i;
		
		KnTagComputeShadow(tag, &top_gc, &bottom_gc,
				   &top_thick, &bottom_thick);
		KnSetRegion(ct->dpy, top_gc, ct->clip);
		KnSetRegion(ct->dpy, bottom_gc, ct->clip);
		XSetLineAttributes(ct->dpy, top_gc,
				   2,
				   tag->tag.line_style, tag->tag.cap_style,
				   tag->tag.join_style);
		XSetLineAttributes(ct->dpy, bottom_gc,
				   2,
				   tag->tag.line_style, tag->tag.cap_style,
				   tag->tag.join_style);
		for(i = 0; i < 5; i++) {
		    points[i].x -= top_thick;
		    points[i].y -= top_thick;
		}
		XDrawLines(ct->dpy, ct->win,  top_gc, points, 5, 
			   CoordModeOrigin);
		for(i = 0; i < 5; i++) {
		    points[i].x += top_thick+bottom_thick;
		    points[i].y += top_thick+bottom_thick;
		}
		XDrawLines(ct->dpy, ct->win,  bottom_gc, points, 5, 
			   CoordModeOrigin);
		for(i = 0; i < 5; i++) {
		    points[i].x -= bottom_thick;
		    points[i].y -= bottom_thick;
		}
	    }
#endif /* LOOK3D_FOR_RECT */
	    XDrawLines(ct->dpy, ct->win,  tag->tag.gc, points, 5, 
		       CoordModeOrigin);
	}
    }
    else {
	/* no transformer need to be applied, optimize redraw */
	box.x = rect->x;
	box.y = rect->y;
	box.width = rect->width;
	box.height = rect->height;
	if(KnRectInRegion(ct->clip, box.x, box.y, 
			 box.width, box.height)) { 
#ifdef LOOK3D_FOR_RECT
	    if(tag->tag.shadowThickness) {
		GC top_gc, bottom_gc;
		KnDimension top_thick, bottom_thick;
		KnTagComputeShadow(tag, &top_gc, &bottom_gc,
				   &top_thick, &bottom_thick);
		KnSetRegion(ct->dpy, top_gc, ct->clip);
		KnSetRegion(ct->dpy, bottom_gc, ct->clip);
		
		XDrawRectangle(ct->dpy, ct->win, top_gc, 
			       box.x- top_thick, box.y- top_thick, 
			       box.width, box.height);
		XDrawRectangle(ct->dpy, ct->win, bottom_gc, 
			       box.x+ bottom_thick, box.y+ bottom_thick, 
			       box.width, box.height);
	    }
#endif /* LOOK3D_FOR_RECT */
	    XDrawRectangle(ct->dpy, ct->win, tag->tag.gc, 
			   box.x, box.y, 
			   box.width, box.height);
	}
    }
}




void
KnRectCenter(KnRect rect, KnPosition *x, KnPosition *y)
{
    *x = rect->x + rect->width / 2;
    *y = rect->y + rect->height / 2;
}


void
KnRectSize(KnRect rect, KnDimension *w, KnDimension *h)
{
    XPoint points[3];
    register int tmp0, tmp1;

    points[0].x = rect->x;
    points[1].x = points[2].x = rect->x + rect->width;
    points[0].y = points[1].y = rect->y;
    points[2].y = rect->y + rect->height;

    if(rect->t) {
	TransformerTransformPointList(rect->t, points, 3, points);
    }
    tmp0 = points[1].x - points[0].x;
    tmp1 = points[1].y - points[0].y;
    *w = sqrt(tmp0*tmp0 + tmp1*tmp1);
    tmp0 = points[2].x - points[1].x;
    tmp1 = points[2].y - points[1].y;
    *h = sqrt(tmp0*tmp0 + tmp1*tmp1);
	     
}


void
KnRectBoundingBox(KnRect rect, Widget view, XRectangle *box)
{
    XPoint points[4];

    points[0].x = points[3].x = rect->x;
    points[1].x = points[2].x = rect->x + rect->width;
    points[0].y = points[1].y = rect->y;
    points[2].y = points[3].y = rect->y + rect->height;

    KnTransformPointList(view, rect, points, 4);
    GetBoundingBox(points, 4, box);
}


void
KnRectClear(KnRect rect, Widget w)
{
    XPoint points[4];
    TagObject tag;
    XRectangle extra;
    
    if(rect->flags & KnCLEARED_FLAG) return;
    rect->flags |= KnCLEARED_FLAG;
    tag = (TagObject)KnGetTag(w, (KnO)rect);

    points[0].x = points[3].x = rect->x - tag->tag.line_width;
    points[1].x = points[2].x = rect->x + rect->width + tag->tag.line_width;
    points[0].y = points[1].y = rect->y - tag->tag.line_width;
    points[2].y = points[3].y = rect->y + rect->height + tag->tag.line_width;

#ifdef LOOK3D
    extra.x = extra.y = 0;
    extra.width = extra.height = tag->tag.shadowThickness;
    KlSend_clearbox(rect, points, 4, &extra);
#else /* LOOK3D */
    KlSend_clearbox(rect, points, 4, NULL);
#endif /* LOOK3D */
    
    KnOClearAnchors((KnO)rect, w);
}



KnO
KnRectContains(KnRect rect, Widget w, Int x, Int y)
{
    Boolean res;
    KnPosition rx = x, ry = y;

    if(rect->t) {
	TransformerInvTransform(rect->t, x, y, &rx, &ry);
    }

    res = (rx >= rect->x) && (rx <= (KnPosition)(rect->x + rect->width)) &&
	(ry >= rect->y) && (ry <= (KnPosition)(rect->y + rect->height));
    return res ? (KnO)rect : NULL;
}




/* init class fields and call super class Init proc */
void
KnFRectInit(KnFRect rect, Widget knvas, KnO parent, 
	 KnPosition x, KnPosition y, KnDimension width, KnDimension height)
{
    KnRectInit((KnRect)rect, knvas, parent, x, y, width, height);
}


/* allocates storage space for a new filled rect
 * 
 * returns: the newly created KnRect
 */
KnO
KnCreateFRect(Widget knvas, KnO parent, KnPosition x, KnPosition y, KnDimension width, KnDimension height)
{
    KnFRect rect;
    rect =  (KnFRect)KlOMake(KnFRectClass);
    KnFRectInit(rect, knvas, parent, x, y, width, height);
    return (KnO)rect;
}

#define ROTATED(t) ((t->mat10)||(t->mat01))


void
KnFRectDraw(KnRect rect, KnDrawingContext ct, Transformer tf) 
{
    Transformer t;    
    XPoint points[5];
    XRectangle box;
    TagObject tag;
    tag = (TagObject)KnGetTag(ct->w, (KnO)rect);

    rect->flags &= ~KnCLEARED_FLAG;

    t = tf;
    if(t) {
	TransformerTransformBoxPoints(t, rect->x, rect->y,
				      rect->width, rect->height,
				      points);

	points[4].x = points[0].x;
	points[4].y = points[0].y;
    }
    if(t &&ROTATED(t)) {
	GetBoundingBox(points, 4, &box);
	if(KnRectInRegion(ct->clip, box.x, box.y, 
			 box.width, box.height)) {
#ifdef LOOK3D
	    if(tag->tag.shadowThickness) {
		KnDrawFilledRelief(ct, tag, points, 5);
	    }
#endif /* LOOK3D */
	    XFillPolygon(ct->dpy, ct->win, tag->tag.gc, points, 5, 
			 Convex, CoordModeOrigin);
	    return;
	}
    }

    /* no rotation applied, optimize redraw */
    if(t) {
	box.x = points[0].x;
	box.y = points[0].y;
	box.width = points[1].x - box.x;
	box.height = points[2].y - box.y;
    }
    else {
	box.x = rect->x;
	box.y = rect->y;
	box.width = rect->width;
	box.height = rect->height;
    }
	if(KnRectInRegion(ct->clip, box.x, box.y, 
			 box.width, box.height)) {
	    XFillRectangle(ct->dpy, ct->win, tag->tag.gc, 
			   box.x, box.y, 
			   box.width, box.height);
#ifdef LOOK3D
	    if(tag->tag.shadowThickness) {
		/* 3d look is only available for non rotated rectangles */
		KnSetRegion(ct->dpy, tag->tag.top_shadow_gc, ct->clip);
		KnSetRegion(ct->dpy, tag->tag.bottom_shadow_gc, ct->clip);
		_XmDrawShadows(XtDisplay(ct->w), ct->win,
			       tag->tag.top_shadow_gc, tag->tag.bottom_shadow_gc,
			       box.x, box.y, box.width, box.height,
			       tag->tag.shadowThickness, tag->tag.shadowType);
	    }
#endif /* LOOK3D */
	    if(DrawSelection(rect)) {
		KnDrawSelectedBox((KnTag)tag, ct, &box);
	    }
    }
}



/* declare new   selectors
 */
void
KnRectImport()
{
	
}



/* class initializations for KnRect objects
 * 
 * returns: 
 */
void
KnRectClassInitialize()
{
     /* declare  types */
    KlDeclareSubType(&KnRectClass, "KnRect", KnOClass, sizeof(struct KnRectStruct));
    KlDeclareSubType(&KnFRectClass, "KnFRect", KnRectClass, sizeof(struct KnRectStruct));

    /* declare methods */
    KlDeclareMethod(KnRectClass, KnOSelContains, KnRectContains);
    KlDeclareMethod(KnRectClass, KnOSelClear, KnRectClear);
    KlDeclareMethod(KnRectClass, KnOSelDraw, KnRectDraw);
    KlDeclareMethod(KnRectClass, KnOSelBoundingBox, KnRectBoundingBox);
    KlDeclareMethod(KnRectClass, KnOSelCenter, KnRectCenter);
    KlDeclareMethod(KnRectClass, KnOSelSize, KnRectSize);
    
    KlDeclareMethod(KnFRectClass, KnOSelDraw, KnFRectDraw);

}



