/* Copyright (c) 1995 by Groupe Bull.  All Rights Reserved */
/* $Id: poly.c,v 1.4 95/09/28 11:58:31 leon Exp $ */



#include "poly.h"
#include "TagP.h"


static XPoint *
DuplicatePointList(XPoint *src_points, int num)
{
    XPoint *points;
    int nbytes;
    
    nbytes = sizeof(XPoint) * num;
    points = (XPoint *)XtMalloc(nbytes);
    memcpy(points, src_points, nbytes);
    return points;
}

void
KnPolyInit(KnPoly obj,		/* the graphical object */
	   Widget knvas,	/* ignored. required for consistency */
	   KnO parent,		/* the object's parent */
	   XPoint points[],
	   int npoints)
{
    int i;
    
    KnOInit((KnO)obj, knvas, parent, 0, 0);
    obj->t = TransformerMake();
    obj->t->flags &= ~T_IDENTITY ; /* suppress T_IDENTITY flag */
    obj->points = (XPoint*)XtMalloc(sizeof(XPoint)*npoints);
    for(i = 0; i< npoints; i++) {
	obj->points[i] = points[i];
    }
    obj->npoints = npoints;
}


void
KnPolyCopyFields(KnPoly new, KnPoly old)
{
    KnOCopyFields((KnO)new, (KnO)old);
    new->points = DuplicatePointList(old->points, old->npoints);
}



void
KnPolyFree(KnPoly obj)
{
    XtFree((void*)obj->points);
    KnOFree((KnO)obj);
}



/* Could use KnCreatePath to create a relative line */

KnO
KnCreatePolyline(Widget knvas, KnO parent, XPoint points[], int npoints)
{
    KnPoly obj;
    obj = (KnPoly)KlOMake(KnPolylineClass);
    KnPolyInit(obj, knvas, parent, points, npoints);
    return (KnO)obj;
}



KnO
KnCreatePolygon(Widget knvas, KnO parent, XPoint points[], int npoints)
{
    KnPoly obj;
    obj = (KnPoly)KlOMake(KnPolygonClass);
    KnPolyInit(obj, knvas, parent, points, npoints);
    return (KnO)obj;
}



void
KnPolyClear(KnPoly obj, Widget w)
{
    XPoint *points;

    if(obj->flags & KnCLEARED_FLAG) return;
    obj->flags |= KnCLEARED_FLAG;

    points = DuplicatePointList(obj->points, obj->npoints);
    
    KlSend_clearbox(obj, points, obj->npoints, NULL); 

    XtFree((void*)points);
}

void
KnPolylineDraw(KnPoly obj, KnDrawingContext ct, Transformer tf)
{
    Transformer t;    
    XPoint *points;
    XRectangle box;
    TagObject tag;
    
    tag = (TagObject)KnGetTag(ct->w, (KnO)obj);
    obj->flags &= ~KnCLEARED_FLAG;
    
    points = DuplicatePointList(obj->points, obj->npoints);;
    
    t = tf;
    if(t) {
	TransformerTransformPointList(t, points, obj->npoints, points);
    }

    GetBoundingBox(points, obj->npoints, &box);

    if(KnRectInRegion(ct->clip, box.x, box.y, 
		     box.width, box.height)) {
	XDrawLines(ct->dpy, ct->win,  tag->tag.gc, points,
		   obj->npoints, CoordModeOrigin);
    }
    XtFree((void *)points);
}

void
KnPolygonDraw(KnPoly obj, KnDrawingContext ct, Transformer tf)
{
    Transformer t;    
    XPoint *points;
    XRectangle box;
    TagObject tag;
    
    tag = (TagObject)KnGetTag(ct->w, (KnO)obj);
    obj->flags &= ~KnCLEARED_FLAG;
    
    points = DuplicatePointList(obj->points, obj->npoints);;
    
    t = tf;
    if(t) {
	TransformerTransformPointList(t, points, obj->npoints, points);
    }

    GetBoundingBox(points, obj->npoints, &box);

    if(KnRectInRegion(ct->clip, box.x, box.y, 
		     box.width, box.height)) {
#ifdef LOOK3D
	if(tag->tag.shadowThickness) {
	    KnDrawFilledRelief(ct, tag, points, obj->npoints);
	}
#endif /* LOOK3D */
	XFillPolygon(ct->dpy, ct->win,  tag->tag.gc, points,
		     obj->npoints, Complex, CoordModeOrigin);
    }
    XtFree((void *)points);
}




KnO
KnPolyContains(KnPoly obj, Widget w, Int x, Int y)
{
    Boolean res;
    KnPosition rx = x, ry = y;
    XRectangle box;

    if(obj->t) {
	TransformerInvTransform(obj->t, x, y, &rx, &ry);
    }
    
    GetBoundingBox(obj->points, obj->npoints, &box);

    res = (rx >= box.x) && (rx <= (KnPosition)(box.x + box.width)) &&
	(ry >= box.y) && (ry <= (KnPosition)(box.y + box.height));

    return res ? (KnO)obj : NULL;
}




void
KnPolyCenter(KnPoly obj, KnPosition *x, KnPosition *y)
{
    register int i;
    int bx, by;
    bx = obj->points[0].x;
    by = obj->points[0].y;
    for(i = 1; i < obj->npoints; i++) {
	bx += obj->points[i].x;
	by += obj->points[i].y;
    }
    *x = bx / obj->npoints;
    *y = by / obj->npoints;
}



void
KnPolyBoundingBox(KnPoly obj, Widget view, XRectangle *box)
{
    XPoint *points;

    points = DuplicatePointList(obj->points, obj->npoints);

    KnTransformPointList(view, obj, points, 4);
    GetBoundingBox(points, 4, box);

    XtFree((void*)points);
}



/* type desc */
KlType KnPolyClass;
KlType KnPolylineClass;
KlType KnPolygonClass;


void
KnPolyImport()
{
    /* no new selector */
}





/* CLASS INITIALIZATIONS FOR KnNewClass OBJECTS */
void
KnPolyClassInitialize()
{
     /* DECLARE  TYPES */
    KlDeclareSubType(&KnPolyClass, "KnPoly", KnOClass,
		     sizeof(struct KnPolyStruct));
    KlDeclareSubType(&KnPolylineClass, "KnPolyline", KnPolyClass,
		     sizeof(struct KnPolylineStruct));
    KlDeclareSubType(&KnPolygonClass, "KnPolygon", KnPolyClass,
		     sizeof(struct KnPolygonStruct));


    KlDeclareMethod(KnPolyClass, KlSelFree, KnPolyFree);
    KlDeclareMethod(KnPolyClass, KnOSelCopyFields, KnPolyCopyFields);
    
    KlDeclareMethod(KnPolyClass, KnOSelContains, KnPolyContains);
    KlDeclareMethod(KnPolyClass, KnOSelClear, KnPolyClear);
    KlDeclareMethod(KnPolyClass, KnOSelBoundingBox, KnPolyBoundingBox);
    KlDeclareMethod(KnPolyClass, KnOSelCenter, KnPolyCenter);
    
    KlDeclareMethod(KnPolylineClass, KnOSelDraw, KnPolylineDraw);
    KlDeclareMethod(KnPolygonClass, KnOSelDraw, KnPolygonDraw);

/*
    KlDeclareMethod(KnPolyClass, KnOSelSize, KnPolySize);
    */
    

}
