/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: line.c,v 1.9 95/09/28 11:57:54 leon Exp $ */
#include <math.h>
#include "line.h"
#include "TagP.h"
#ifdef LOOK3D
#include <Xm/Xm.h>
#endif /* LOOK3D */


/* type desc */
KlType KnLineClass;




/* init class fields and call super class Init proc */
void
KnLineInit(KnLine l, Widget knvas, KnO parent, KnPosition x, KnPosition y, 
	 KnPosition tox, KnPosition toy)
{
    KnOInit((KnO)l, knvas, parent, x, y);
    l->width = tox - x;
    l->height = toy - y;
}


/* allocates storage space for a new line
 * 
 * returns: the newly created KnO
 */
KnO
KnCreateLine(Widget knvas, KnO parent, KnPosition x, KnPosition y, 
	 KnPosition tox, KnPosition toy)
{
    KnLine line;
    line =  (KnLine)KlOMake(KnLineClass);
    KnLineInit(line, knvas, parent, x, y, tox, toy);
    return (KnO)line;
}



void
KnLineDraw(KnLine line, KnDrawingContext ct, Transformer tf) 

{
    Transformer t;    
    XPoint points[2];
    XRectangle box;
    TagObject tag;
    unsigned int orientation;
    
    tag = (TagObject)KnGetTag(ct->w, (KnO)line);
    t = tf;
    points[0].x = line->x;
    points[0].y = line->y;
    points[1].x = line->x + line->width;
    points[1].y = line->y + line->height;
    if(t) {
	TransformerTransformPointList(t, points, 2, points);
    }
    if(((points[0].x != points[1].x) &&
	     (points[0].y != points[1].y))) {
   /* if the line is not horizontal, nor vertical, we cannot draw 3dlook */
	GetBoundingBox(points, 2, &box);
	if(KnRectInRegion(ct->clip, box.x, box.y, 
			 box.width, box.height)) {
	    XDrawLines(ct->dpy, ct->win, tag->tag.gc, points, 2, 
		       CoordModeOrigin);
	    if(DrawSelection(line))
		KnDrawSelectedBox(tag, ct, &box);
	    return;
	}
    }

    /* 3d look possible */
    GetBoundingBox(points, 2, &box);
#ifdef LOOK3D
    if(tag->tag.shadowThickness) {
	/* 3d look required */
	if(points[0].y == points[1].y) {
	    /* horizontal line */
	    orientation = XmHORIZONTAL;
	    box.height = tag->tag.shadowThickness*2;
	}
	else {
	    /* vertical line */
	    box.width = tag->tag.shadowThickness*2;
	    orientation = XmVERTICAL;
	}
    }
#endif LOOK3D
	
    if(KnRectInRegion(ct->clip, box.x, box.y, 
		     box.width, box.height)) {
#ifdef LOOK3D
	if(tag->tag.shadowThickness) {
	    KnSetRegion(ct->dpy, tag->tag.top_shadow_gc, ct->clip);
	    KnSetRegion(ct->dpy, tag->tag.bottom_shadow_gc, ct->clip);
	    _XmDrawShadows(XtDisplay(ct->w), ct->win,
			     tag->tag.top_shadow_gc, tag->tag.bottom_shadow_gc,
			     box.x, box.y,
			     box.width, box.height,
			     tag->tag.shadowThickness, 
			     tag->tag.shadowType);
	    if(DrawSelection(line))
		KnDrawSelectedBox(tag, ct, &box);
	    return;
	}
#endif /* LOOK3D */
	/* 3d look possible, but shadow thickness = 0 */
	XDrawLine(XtDisplay(ct->w), ct->win, tag->tag.gc,
		  points[0].x, points[0].y,
		  points[1].x, points[1].y);
	if(DrawSelection(line))
	    KnDrawSelectedBox(tag, ct, &box);
    }
}


void
KnLineClear(KnLine line, Widget w)
{
    XPoint points[2];
    TagObject tag;
    
    tag = (TagObject)KnGetTag(w, (KnO)line);

    points[0].x = line->x - tag->tag.line_width;
    points[0].y = line->y - tag->tag.line_width;
    points[1].x = line->x + line->width + tag->tag.line_width;
    points[1].y = line->y + line->height + tag->tag.line_width;

#ifdef LOOK3D
    if(tag->tag.shadowThickness) {
	XRectangle extra;
	extra.x = extra.y = 0;
	extra.width = extra.height = 2*tag->tag.shadowThickness;
	KlSend_clearbox(line, points, 2, &extra);
	return;
    }
#endif /* LOOK3D */
    /* called with no LOOK3D or shadowThickness = 0 */
    KlSend_clearbox(line, points, 2, NULL);
}



KnO
KnLineContains(KnLine line, Widget w, Int x, Int y)
{
    Boolean res;
    KnPosition rx = x, ry = y;
    XRectangle box;
    XPoint points[2];
    TagObject tag;

    tag = (TagObject)KnGetTag(w, (KnO)line);
    if(line->t) {
	TransformerInvTransform(line->t, x, y, &rx, &ry);
    }
    points[0].x = line->x;
    points[0].y = line->y;
    points[1].x = line->x+line->width;
    points[1].y = line->y+line->height;
    GetBoundingBox(points, 2, &box);
    
#ifdef LOOK3D
    if(tag->tag.shadowThickness) {
	box.width +=  2*tag->tag.shadowThickness;
	box.height +=  2*tag->tag.shadowThickness;
    }
#endif /* LOOK3D */
    res = (rx >= box.x) && (rx <= (KnPosition)(box.x+box.width)) &&
	(ry >= box.y) && (ry <= (KnPosition)(box.y + box.height));
    return res ? (KnO)line : NULL;
}




void
KnLineCenter(KnLine line, KnPosition *x, KnPosition *y)
{
    *x = line->x + line->width / 2;
    *y = line->y + line->height / 2;
}



void
KnLineBoundingBox(KnLine line, Widget view, XRectangle *box)
{
    XPoint points[4];

    points[0].x = points[3].x = line->x;
    points[1].x = points[2].x = line->x + line->width;
    points[0].y = points[1].y = line->y;
    points[2].y = points[3].y = line->y + line->height;

    KnTransformPointList(view, line, points, 4);
    GetBoundingBox(points, 4, box);
}

void
KnLineSize(KnLine line, KnDimension *w, KnDimension *h)
{
    XPoint points[2];
    register int tmp0, tmp1;
    
    points[0].x = line->x;
    points[0].y = line->y;
    points[1].x = line->x + line->width;
    points[1].y = line->y + line->height;

    if(line->t) {
	TransformerTransformPointList(line->t, points, 2, points);
    }
    tmp0 = points[1].x - points[0].x;
    tmp1 = points[1].y - points[0].y;
    *w = sqrt(tmp0*tmp0 + tmp1*tmp1);
    /* line's height is always 0 */
    *h = 0;
}



/* declare new   selectors
 */
void
KnLineImport()
{
	
}



/* class initializations for KnLine objects
 * 
 * returns: 
 */
void
KnLineClassInitialize()
{
     /* declare  type */
    KlDeclareSubType(&KnLineClass, "KnLine", KnOClass, sizeof(struct KnLineStruct));

    /* declare methods */
    KlDeclareMethod(KnLineClass, KnOSelDraw, KnLineDraw);
    KlDeclareMethod(KnLineClass, KnOSelClear, KnLineClear);
    KlDeclareMethod(KnLineClass, KnOSelContains, KnLineContains);
    KlDeclareMethod(KnLineClass, KnOSelBoundingBox, KnLineBoundingBox);
    KlDeclareMethod(KnLineClass, KnOSelCenter, KnLineCenter);
    KlDeclareMethod(KnLineClass, KnOSelSize, KnLineSize);

}
