/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: libkn.h,v 1.11.2.1 96/03/27 16:30:41 leon Exp $ */
#ifndef __LIBKN_H
#define __LIBKN_H



#include <X11/Intrinsic.h>
#include <Kn/kno.h>
#include <Kn/interactor.h>



#ifdef __cplusplus
extern "C" {
#endif




/* The Callback Information structure given to any knvas callback
   
 * Contains usefull information for callbacks accuring on a knvas widget
 
 */
typedef struct _InteractorCallbackStruct {
    int reason;			/* The reason of the callback (i.e SELECT,
				   DRAG, or DROP) */
    XEvent *e;			/* The XEvent that triggered the callback */
    KnInteractor interactor;	/* the interactor that triggered the
				   callback */
    KnO target;			/* The event target: a KnO */
    KnPosition tox, toy;	/* The target's relative coordinates of the
				   event. These coordinates may be modified in
				   the callback and will be used by
				   Interactors */
    Boolean accept;		/* flag for refusing the interaction. Entering
				   the callback, this flag is True. set it to
				   False will cancel the interaction */
    Boolean select;		/* on a DROP interaction, this flag may be set
				   to False, to say that the object must be
				   unselected */
} InteractorCallbackStruct;




/* Translate a graphical object
   
 * This function will translate the specified object of dx, dy deltas given in
 * its father coordinates. It calls the Clear method before and after the
 * translation.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.

 * USE: application programming
 
 * returns: */
void
KnTranslate(Widget w,		/* the knvas widget */
	    KnO kno,		/* the object to translate */
	    KnPosition dx,	/* horizontal delta */
	    KnPosition dy);	/* vertical delta */



/* Move a graphical object
   
 * Move the graphical object to the specified location, given in its father
 * coordinates. It calls the Clear method before and after the
 * movment.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.

 * USE: application programming
 
 * returns:
 */
void
KnMove(Widget w,		/* the knvas widget */
	   KnO kno,		/* the graphical object to move */
	   KnPosition tox,	/* horizontal position */
	   KnPosition toy);	/* vertical position */



/* Rotate a graphical object
   
 * Rotate the graphical object of the specified angle, around its center
 * (returned by the KnOSelCenter method). Object's center is the point returned
 * by the "center" method. It calls the Clear method before and after the
 * rotation.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.

 * USE: application programming
 
 * returns: */
void
KnRotate(Widget w,		/* the knvas widget */
	 KnO kno,			/* the graphical object to rotate */
	 int angle);



/* Resize a graphical object
   
 * Resize the graphical object according to the sx and sy scale factors. It
 * calls the Clear method before and after the resize.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.

 * USE: application programming
 
 * returns:
 */
void
KnResize(Widget w,		/* the knvas widget */
	 KnO kno,			/* the graphical object */
	 float sx,		/* the horizontal scale factor */
	 float sy);		/* the vertical scale factor */


/* Select a graphical object
   
 * Set the flag KnSELECTED_FLAG for the graphical object. If the parameter
 * 'visible' is set to True, the function sets the KnSHOW_SELECTION_FLAG flag
 * too. If the KnSHOW_SELECTION_FLAG flag is modified, it calls the Clear
 * method.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.

 * USE: application programming
 
 * returns:
 */
void
KnSelect(Widget w,		/* the knvas widget */
	 KnO kno,		/* the graphical object */
	 Boolean visible);	/* visible status */


/* Unselect a graphical object
   
 * Set the flags KnSELECTED_FLAG and KnSHOW_SELECTION_FLAG to False. If the
 * object selection was previously visible, it calls the Clear method.

 * WARNING: Graphical update requires an explicit call to KnvasUpdateDisplay or
 * KnvasUpdateLayers.
 
 * returns:
 */
void
KnUnselect(Widget w,		/* the knvas widget */
	   KnO kno);		/* the graphical object */



/* Knvas specific main loop

 * This is intended to replace the XtAppMainLoop call doing the same plus
 * Garbage collection calls. XtAppMainLoop replacement is not mandatory, the
 * KnAppMainLoop function is just intended to provide automatic garbagge
 * collection of grahical objects.

 * returns: NEVER RETURNS
 */
void
KnAppMainLoop(XtAppContext context); /* the Xt application context */



/* Sets an interactor on a kno
   
 * Replace the last object's behavior with the interactor's behavior. There is
 * no operation performed on interactors. The old one is simply replaced. If
 * you want to combine behaviors, you should use KnInteractorMerge.

 * USE: application programming
 
 * returns:
 */
void
KnSetInteractor(Widget knvas, 	/* a knvas widget */
		KnO kno,		/* the graphic obj */
		KnInteractor i); /* the interactor */


/* Add a callback on a graphical object
   
 * Adds the function 'proc' on the callback list named 'name'. KnOs should have
 * only one callback list that should be called by any interactor to notify
 * interaction to the objects. The proc should have the standard Xt callback
 * signature. A callback set on a kno will be available for all the views
 * sharing this kno.

 * WARNING: in order to minimize object size, knos have only one calbback
 * defined, that is XtNinteractorCallback. The callback name is required here
 * to be consistent with the corresdonding Xt functions.

 * USE: application programming
 
 * returns: */
void
KnAddCallback(Widget knvas,	/* a knvas widget */
	      KnO kno,		/* the graphic obj */
	      String name,	/* callback name:  XtNinteractorCallback */
	      XtCallbackProc proc, /* standard Xt callback proc */
	      XtPointer client); /* client data */


/* remove a callback from a graphic obj
   
 * This removes the function 'proc' from the kno callback list.

 * WARNING: in order to minimize object size, knos have only one callback
 * defined, that is XtNinteractorCallback. The callback name is required here
 * to be consistent with the corresdonding Xt functions.

 * USE: application programming
 
 * returns:
 */
void
KnRemoveCallback(Widget knvas,	/* a knvas widget */
		 KnO kno, 	/* the graphic obj */
		 String name, 	/* callback name:  XtNinteractorCallback */
		 XtCallbackProc proc, /* standard Xt callback proc */
		 XtPointer client); /* client data */


/* call a callback list on a graphical object
   
 * This function is similar to XtCallCallbacks. It must be used (by interactor
 * for example) to call the callbacks defined on a kno. the knvas parameter
 * will be passed as the widget to the callback. 'call' should be a pointer to
 * s InteractorCallbackStruct variable. The caller must set the call fields,
 * before calling the callback.

 * USE: class writing */
void
KnCallCallbacks(Widget knvas, 	/* a knvas widget */
		KnO kno,		/* the graphic obj */
		String name, 	/* callback name:  XtNinteractorCallback */
		XtPointer call); /* call data */





/* Set objet's tag
   
 * The object's tag will be set to 'tag'. 
 *
 * WARNING
 * To improve graphical speed, no redraw nor clear is performed, so that
 * an unexpected expose should redraw a part of the object with the new tag,
 * the other part (not redrawn, keeping the old tag).

 * USE: application programming

 * returns: */
void
KnSetTag(Widget knvas,		/* a Knvas widget */
	 KnO kno,		/* the graphical object */
	 KnTag tag);		/* the new tag */



/* Get a kno's tag
   
 * If the knvas owns a tag whose tag id is the tid of the object, it returns
 * this tag, else, it returns the default knvas tag. It allows in an object
 * class to retrieve the tag defined for the view we are currently
 * drawing. Using this function only to access a tag is mandatory for multi
 * display considerations.

 * USE: class writing

 * returns: The Tag whose tag id is the tid of the object
 */
KnTag
KnGetTag(Widget knvas,		/* The Knvas widget */
	 KnO kno);		/* The kno we want the tag */

/* Destroys a Tag.

 * This function destroys a tag and the associated GCs, removing
 * references from the to the tag.

 * USE: application programming

 * returns: */
void
KnDestroyTag(Widget knvas,   /* The Knvas widget used when creating the tag */
	     KnTag kntag);	/* The Tag to be destroyed */


/* Reparents a graphical object
   
 * This functions reparents a KnO in the objects hierachy. The kno is detached
 * fron its old parent and attached to its new parent. It is cleared from its
 * old position and drawn to its new position.

 * USE: application programming

 * returns: */
void
KnReparent(Widget w,		/* The Knvas widget */
	   KnO kno,		/* the object to reparent */
	   KnO parent);		/* the new parent */




/* Destroy a graphical object
   
 * This functions removes a graphical object from the objects hierachy. The
 * object is cleared before being removed.

 * Next call to KnvasUpdateDisplay will commit this change.

 * USE: application programming

 * returns: */
void
KnRemove(Widget w,		/* The Knvas widget */
	 KnO kno);		/* the kno to remove */





/* Get object's position
   
 * Returns the object / parent's relative coodinates;

 * USE: application programming

 * returns: */
void
KnGetPosition(Widget w,		/* unused - requiered for consistency */
	   KnO kno,		/* the kno */
	   KnPosition *x,		/* returns the x coordinate */
	   KnPosition *y);		/* returns the y coordinate */





/* Set sensitive status.

 * The sensitive status of a KnO defines whether or not the KnO should be
 * considedred as a possible target during event dispatching. Default value is
 * True, so that any created object may dispatch event as soon as it has an
 * interactor. Setting this to False make the object transparent to event
 * dispatch (all will happen just as if the object would not exist. Setting
 * this to false on a Group or Layer disable event dispatch for children, too.

 * USE: application programming

 * returns: 

 */
void
KnSetSensitive(Widget w,	/* unused - requiered for consistency */
	       KnO kno,		/* the kno */
	       Boolean sensitive); /* new sensitive status */



/* Unmaps a graphical object
   
 * Any graphical object (any subclass of KnO) may be in a MAP or UNMAPPED
 * state. When a kno is mapped, it may be drawn on the screen and receive X
 * events. When a kno is unmapped, it is no longer displayed and won't receive
 * X events.

 * This function toggles the state of kno to UNMAPPED. If kno is a subclass of
 * group, this will disable recursively redraw and event dispatching.

 * If the object was not yet unmapped, it deletes it from the screen, calling
 * its 'clear' method. Next call to KnvasUpdateDisplay will commit this change.

 * USE: application programming
 
 * returns:
 
 */
void
KnUnmap(Widget w,		/* A knvas widget containing the kno */
	KnO kno);		/* the kno to unmap */


    
/* Maps a graphical object
   
 * Any graphical object (any subclass of KnO) may be in a MAP or UNMAPPED
 * state. When a kno is mapped, it may be drawn on the screen and receive X
 * events. When a kno is unmapped, it is no longer displayed and won't receive
 * X events.

 * This function toggles the state of kno to MAPPED. If kno is a subclass of
 * group, this will enable recursively redraw and event dispatching.

 * If the object was not yet mapped, it redraws it on the screen, calling its
 * 'clear' method. Next call to KnvasUpdateDisplay will commit this change.

 * USE: application programming 
 
 * returns:
 
 */
void
KnMap(Widget w,			/* A knvas widget containing the kno */
      KnO kno);			/* the kno to map */



#ifndef __C2MAN__

void
KnClear(Widget w, KnO kno);


void
KnAttach(Widget knvas, KnO kno, KnAnchor anchor);


void
KnDettach(Widget knvas, KnO kno, KnAnchor anchor);

#endif
    

#ifdef __cplusplus
} /* Close scope of 'extern "C"' */
#endif


#endif /* __LIBKN_H */
