/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: kno.c,v 1.12.2.2 96/03/12 17:38:14 leon Exp $ */
#include "kno.h"
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "misc.h"
#include "knoP.h"


/* type desc */
KlType KnOClass;

/* new selectors */

Selector KnOSelBoundingBox;
Selector KnOSelCenter;
Selector KnOSelClear;
Selector KnOSelClearBox;
Selector KnOSelCopyFields;
Selector KnOSelContains;
Selector KnOSelDestroy;
Selector KnOSelDispatch;
Selector KnOSelDraw;
Selector KnOSelDrawSelection;
Selector KnOSelMove;
Selector KnOSelPosition;
Selector KnOSelResize;
Selector KnOSelRotate;
Selector KnOSelSelect;
Selector KnOSelSize;
Selector KnOSelTranslate;
Selector KnOSelUnselect;


/* new traits */
Trait KnOTrait_kno;




/*
 * this method is invoked when copying an object from another.  all the field
 * have already been bitwised copied, modify here those which have to be
 * duplicated
 */
void
KnOCopyFields(KnO newo, KnO old)
{
    /* interactors can be shared */
    if(newo->interactor)
	KlIncRef(newo->interactor);
    /* transformers cannot be shared, duplicate it */
    if(newo->t)
	newo->t = TransformerMakeCopy(newo->t);
    /* this is a potential bug: what should we do with this ? */
    newo->anchors = NULL;
    newo->interactorCallback = NULL;
}


/* allocates storage space for a new kno and copy from the second
 * WARNING: DOES NOT NOTIFY TO PARENT
 * returns: the newly created KnO */
KnO
KnOCopy(KnO old)
{
    KnO kno;
    unsigned int rc;
    kno =  (KnO)KlOMake(old->type);
    /* save ref count, initialized by KlOMake */
    rc = kno->reference_count;
    /* quick and dirty copy */
    memcpy(kno, old, (int)KlMallocSizeOfBucket (KlTypeSizeGet(old->type)));
    kno->reference_count = rc;
    /* then allow initialization of special fields */
    KlSend(KnOSelCopyFields, kno, (kno, old));
    return kno;
}





/* init kno fields.
 * This must be called by any sub class In order to speed up
 * graphical operations, we do not set a transformer just only for object
 * translation. a transformer will be created only for Rescale and resize
 * operations.
 * returns: */
void
KnOInit(KnO s, Widget knvas, KnO parent, 
	  KnPosition x, KnPosition y)
{
    TagObject defaultTag;
    s->parent = parent;		/* DO NOT IncRef parent to avoid cycles */
    s->interactor = NULL;
    s->flags = KnSENSITIVE_FLAG | KnMAPPED_FLAG;
    s->interactorCallback = NULL;
    s->t = 0;
    s->x = x;
    s->y = y;
    defaultTag = (TagObject) KnvasGetDefaultTag(knvas);
    if(parent) {
	KnGroupAdd((KnGroup)parent, s);
	s->tid  = defaultTag->tag.id;
    }
    s->anchors = NULL;
}


/* Free allocated space and the gc
 * returns: 
 */
void
KnOFree(KnO kno)
{
    if(kno->interactor)
	KlDecRef(kno->interactor);
    if(kno->t)
	XtFree((char*)kno->t);
    if(kno->anchors)
	KlDecRef(kno->anchors);
    if(kno->interactorCallback) {
	XtFree((char*)kno->interactorCallback);
    }
    KlFree(kno);
}


void
KnODestroy(KnO kno)
{
    KnUnmanage(kno);
}






static void
KnODrawSelection(KnO kno, Display *dpy, Window win, 
		   XRectangle *box, KnTag kntag)
{
    TagObject tag = (TagObject)kntag;
    XFillRectangle(dpy, win, tag->tag.gc, box->x, box->y, 6, 6);
    XFillRectangle(dpy, win, tag->tag.gc, 
		   box->x+box->width-6, box->y+box->height-6, 6, 6);
    XFillRectangle(dpy, win, tag->tag.gc, box->x+box->width-6, box->y, 6, 6);
    XFillRectangle(dpy, win, tag->tag.gc, box->x, box->y+box->height-6, 6, 6);
}



static void
KnORotate(KnO kno, KnPosition x, KnPosition y, short angle)
{
    float dx, dy;
    if(!kno->t) {
	kno->t = TransformerMake();
/*	TransformerTranslate(kno->t, kno->x, kno->y);
	kno->x = kno->y = 0;*/

    }
    dx = x;
    dy = y;
    TransformerTranslate(kno->t, -dx, -dy); 
    TransformerRotate(kno->t, angle);
    TransformerTranslate(kno->t, dx, dy);

    if(TransformerIdentity(kno->t)) {
	free(kno->t);
	kno->t = NULL;
    }
}


/*
 * not static: used in rect.c
 */
void
KnOResize(KnO kno, float *sx, float *sy)
{
    float dx, dy;
    if(!kno->t) {
	kno->t = TransformerMake();
	TransformerTranslate(kno->t, kno->x, kno->y);
	kno->x = kno->y = 0;

    }
    dx = kno->t->mat20;
    dy = kno->t->mat21;
    TransformerTranslate(kno->t, -dx, -dy); 
    TransformerScale(kno->t, *sx, *sy);
    TransformerTranslate(kno->t, dx, dy);

    if(TransformerIdentity(kno->t)) {
	free(kno->t);
	kno->t = NULL;
    }
}






void
KnOTranslate(KnO kno, Int dx, Int dy)
{
    if(!kno->t) {
	kno->x += dx;
	kno->y += dy;
    }
    else {
	TransformerTranslate(kno->t, dx, dy);
	if(TransformerIdentity(kno->t)) {
	    free(kno->t);
	    kno->t = NULL;
	}
    }
    if(kno->anchors && kno->anchors->size)
	KnOTranslateAnchors(kno, dx, dy);
}



static Int
KnODispatch(KnO kno, Widget w, XEvent *event, Int rx, Int ry)
{
    KnvasWidget cw = (KnvasWidget) w;
    XtTMRec tm;

    if(KlSend_contains(kno, w, (Int)rx, (Int)ry)) {
	cw->knvas.target = kno;
	if(kno->interactor) {
	    tm = w->core.tm;
	    w->core.tm = kno->interactor->tm;
	    /* this will always return True, because the event filter, is ok.
	       should install the translation's event filter in the widget */
	    _XtTranslateEvent(w, event);
	    kno->interactor->tm = w->core.tm;
	    w->core.tm = tm;
	    return (Int)cw->knvas.dispatched;
	}
    }
    return (Int)False;
}




/* declare new   selectors
 */
void
KnOImport()
{
    KnOSelBoundingBox      = KlDeclareSelector(3, "boundingbox");
    KnOSelCenter           = KlDeclareSelector(3, "center");
    KnOSelClear            = KlDeclareSelector(2, "clear");
    KnOSelClearBox         = KlDeclareSelector(4, "clearbox");
    KnOSelContains         = KlDeclareSelector(4, "contains");
    KnOSelDestroy          = KlDeclareSelector(1, "destroy");
    KnOSelDispatch         = KlDeclareSelector(3, "dispatch");
    KnOSelDraw             = KlDeclareSelector(3, "draw");
    KnOSelDrawSelection    = KlDeclareSelector(5, "draw-selection");
    KnOSelMove             = KlDeclareSelector(3, "move");
    KnOSelPosition         = KlDeclareSelector(3, "position");
    KnOSelResize           = KlDeclareSelector(3, "resize");
    KnOSelRotate           = KlDeclareSelector(4, "rotate");
    KnOSelSelect           = KlDeclareSelector(1, "select");
    KnOSelSize             = KlDeclareSelector(3, "size");
    KnOSelTranslate        = KlDeclareSelector(3, "translate");
    KnOSelUnselect         = KlDeclareSelector(1, "unselect");


    KnOTrait_kno =
	KlDeclareNewTrait("kno");
	
}




static void
KnOClearBox(KnO kno, XPoint *points, Cardinal n, XRectangle *extra)
{
    KnO o;
    KnLayer layer;
    int i;
    /* optimize method call: do not recurse */
    for(o = kno; NULL != o->parent; o = o->parent) {
	if(o->t) {
	    TransformerTransformPointList(o->t, points, n, points);
	}
    }
    layer = (KnLayer)o;
    for(i = 0; i < layer->parents->size; i++) {
	KnvasClearArea((Widget)layer->parents->list[i], points, n, extra);
    }
}



/* default select method,
 * does nothing
 * returns: 
 */
static void
KnOSelect(KnO kno)
{

}


/* default unselect method,
 * does nothing
 * returns: 
 */
static void
KnOUnselect(KnO kno)
{

}


static void
KnOPosition(KnO kno, KnPosition *x, KnPosition *y)
{
    if(kno->t) {
	TransformerTransform(kno->t, kno->x, kno->y, x, y);
    }
    else {
	*x = kno->x;
	*y = kno->y;
    }
}


static void
KnOMove(KnO kno, Int tox, Int toy)
{
    KnPosition x, y;
    Int dx, dy;

    KnOPosition(kno, &x, &y);
    dx = tox - (Int)x;
    dy = toy - (Int)y;
    KnOTranslate(kno, dx, dy);
}



void
KnOClearAnchors(KnO kno, Widget w)
{   
    Cardinal i;
    if(kno->anchors) {
	for(i = 0; i < kno->anchors->size; i++) {
	    KlSend_clear(kno->anchors->list[i], w);
	}
    }
}


void
KnOTranslateAnchors(KnO kno, Int dx, Int dy)
{
    Cardinal i;
    if(kno->anchors) {
	for(i = 0; i < kno->anchors->size; i++) {
	    KlSend_translate((KnO)kno->anchors->list[i], dx, dy);
	}
    }
}




static void
KnOSize(KnO kno, KnDimension *w, KnDimension *h)
{
    *w = *h = 0;
}


/* This is required for compatibily with klm */
typedef void(*InitHook)(void);
InitHook kno_init_hook = 0;


/* class initializations for KnO objects
 * 
 * returns: 
 */
void
KnOClassInitialize()
{
     /* declare  type */
    KlDeclareType(&KnOClass, "KnO", sizeof(struct KnOStruct));

    /* declare methods */
    KlDeclareMethod(KnOClass, KlSelFree, KnOFree);
    KlDeclareMethod(KnOClass, KlSelCopy, KnOCopy);
    KlDeclareMethod(KnOClass, KnOSelClearBox, KnOClearBox);
    KlDeclareMethod(KnOClass, KnOSelCopyFields, KnOCopyFields);
    KlDeclareMethod(KnOClass, KnOSelDestroy, KnODestroy);
    KlDeclareMethod(KnOClass, KnOSelDispatch, KnODispatch);
    KlDeclareMethod(KnOClass, KnOSelDrawSelection, KnODrawSelection);
    KlDeclareMethod(KnOClass, KnOSelMove, KnOMove);
    KlDeclareMethod(KnOClass, KnOSelPosition, KnOPosition);
    KlDeclareMethod(KnOClass, KnOSelResize, KnOResize);
    KlDeclareMethod(KnOClass, KnOSelRotate, KnORotate);
    KlDeclareMethod(KnOClass, KnOSelSelect, KnOSelect);
    KlDeclareMethod(KnOClass, KnOSelSize, KnOSize);
    KlDeclareMethod(KnOClass, KnOSelTranslate, KnOTranslate);
    KlDeclareMethod(KnOClass, KnOSelUnselect, KnOUnselect);

    if (kno_init_hook) {
	/* This is required for compatibily with klm */
	(*kno_init_hook)();
    }
    
    KlDeclareTrait(KnOClass, KnOTrait_kno);

    /* declare companion classes */
}


/* computes the bounding box of the n points
 * 
 * returns: 
 */
void
GetBoundingBox(XPoint points[], Cardinal n, XRectangle *box)
{
    register KnPosition minx, miny, maxx, maxy;
    register int i;
    minx = maxx = points[0].x;
    miny = maxy = points[0].y;
    for(i = 1; i <n; i++) {
	if(points[i].x < minx)
	    minx = points[i].x;
	else if(points[i].x > maxx)
	    maxx = points[i].x;
	if(points[i].y < miny)
	    miny = points[i].y;
	else if(points[i].y > maxy)
	    maxy = points[i].y;
    }
    box->x = minx;
    box->y = miny;
    box->width = maxx - minx;
    box->height = maxy - miny;
}

