/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: interactor.h,v 1.9 95/09/28 11:57:29 leon Exp $ */
#ifndef __KN_INTERACTOR_H
#define __KN_INTERACTOR_H

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#ifdef ISM
#include "wool.h"
#include "wl_list.h"
#else
#include <KlLib.h>
#endif /* ISM */
#include <Kn/hash.h>



#ifdef __cplusplus
extern "C" {
#endif


    
extern KnHashtable interactorTable;

#define XtNvisibleSelection "visibleSelection"
#define XtCVisibleSelection "VisibleSelection"
#define XtNbutton "button"
#define XtCButton "Button"
#define XtNconstraints "constraints"
#define XtCConstraints "Constraints"
#define XtNvaryCursor "varyCursor"
#define XtCVaryCursor "VaryCursor"
#define XtNcallbackPolicy "callbackPolicy"
#define XtCCallbackPolicy "CallbackPolicy"

#define XtRCallbackPolicy "CallbackPolicy"

    
extern void KnInteractorImport();
extern void KnInteractorClassInitialize();

typedef enum _KnButton {
    KnLEFT = 1,
    KnMIDDLE = 2,
    KnRIGHT = 3
} KnButton;

typedef enum _KnCallbackPolicy {
    KnCALLBACK_NEVER = 0,
    KnCALLBACK_ON_PICK = (1<<0),
    KnCALLBACK_ON_DRAG = (1<<1),
    KnCALLBACK_ON_DROP = (1<<2),
    KnCALLBACK_ON_BUTTON = KnCALLBACK_ON_PICK | KnCALLBACK_ON_DROP,
    KnCALLBACK_ALWAYS = 0xFF
} KnCallbackPolicy;

enum {
    KnCR_PICK = (1<<0),
    KnCR_DRAG = (1<<1),
    KnCR_DROP = (1<<2)
};

#define KnInteractorPart\
    KlKLONE_HEADER;\
    XtTMRec tm;\
    KnButton button;\
    Boolean visibleSelection;\
    Boolean varyCursor;\
    String constraints;\
    KnCallbackPolicy callbackPolicy;\
    Boolean useGhosts




typedef struct KnInteractorStruct {
	KnInteractorPart;
} *KnInteractor;


/* The Interactor Class
   
 * INHERITS: none
 
 * OVERRIDES: no method overrided
 
 * FIELDS: new fields defined in the KnInteractor class are:
 
- XtTranslations translations
- KnButton button
- Boolean visibleSelection
- Boolean varyCursor
- String constraints
- KnCallbackPolicy callbackPolicy
- Boolean useGhosts


 * DESCRIPTION
   
 * An Interactor is a set of translations that are intended to provide a
 * complete (eventualy complex) behavior for graphical objects. Interactors are
 * highly coupled with the knvas widget internal structure and writing an
 * interactor may be complex. Goal is to encapsulate and manipulate
 * translations tables. An Interactor may be added on an object so that it will
 * give it its behavior. An Interactor may be shared between many knos, so that
 * you only need to build an interactor once to set the same behavior on many
 * objects. Each built interactor fetch resources from the database at creation
 * time, so you can customize interactor via Xt resources.

 * INTERACTION FEEDBACK

 * By default, feedback is provided using an opaque redraw (i.e the objects
 * themselves are modified and redrawn). For those who do not have fast enough
 * display, interactors also provides a "cheap" mode: setting the "useGhosts"
 * resource enables this and interaction's feedback is done drawing ghosts. Be
 * aware that using ghosts is constraining from a programmer's point view: It
 * does a global X server grab and forbids any object's modification during an
 * interaction. Two other resources can be used to control interaction
 * feedback: "varyCursor" specifies wether or not the cursor should be animated
 * during an interaction. "visibleSelection" specifies if the object should be
 * drawn as selected.
 
 * RESOURCES: (Name, Class, Type, Default Value)
 
- XtNbutton, XtCButton, KnButton, KnLEFT
- XtNvisibleSelection, XtCVisibleSelection, Boolean, True
- XtNvaryCursor, XtCVaryCursor, Boolean, True
- XtNconstraints, XtCConstraints, String, "hv"
- XtNcallbackPolicy, XtCCallbackPolicy, KnCallbackPolicy, KnCALLBACK_ALWAYS

 * USE: application programming */
extern KlType KnInteractorClass;


/* Builds a new Interactor object
   
 * This builds a new Interactor, whose translation table will be compiled from
 * the ascii description contained in the 'translations' string. This is the
 * generic constructor that may be used to build custom interactors, based on
 * user translation tables. To build standards interactors, you'd better use
 * the specific interactor construction functions.

 * USE: application programming
 
 * returns: The newly created Interactor*/
KnInteractor
KnCreateInteractor(Widget w,	/* a knvas widget */
		   String translations); /* a set of translations */


/* Interactor Initializer
   
 * Compile the ascci description of the behavior into an Xt translation table,
 * using XtParseTranslationTable.
 
 * USE: class writing
 */
void
KnInteractorInit(KnInteractor interactor, /* the interactor */
		 String translations, /* a set of translations */
		 Widget w);	/* the knvas widget we are attached to */


/* Combine interactors

 * Merge interactor i2 into interactor i1, using XtOverrideTranslationTable.

 * returns: the modified i1 interactor
 */
KnInteractor
KnInteractorMerge(Widget w,	/* a knvas widget */
 		  KnInteractor i1,
 		  KnInteractor i2);




/* Builds a move interactor

 * The move interactor is intended to move any object with a mouse
 * interaction. The name parameter is used to retrieve resources from the
 * database. Resources are: 'button', 'visibleSelection', 'constraints',
 * 'varyCursor', 'callbackPolicy'.


 * TRANSLATIONS:  Set of translations for moveBehavior:
 
- <Btn?Down>: Select(<visible>)
- <Btn?Motion>: MoveDrag(<constraints>)
- <Btn?Up>: MoveDrop()

 * returns: */
KnInteractor
KnCreateMoveInteractor(Widget knvas, /* a knvas widget */
		       char *name); /* interactor's name */






/* Builds a resize interactor

 * The resize interactor is intended to resize any object with a mouse
 * interaction. The ResizeInteractor fetch resources from database at creation
 * time.

 * RESOURCES: Set of resources defined for a ResizeInteractor

- visibleSelection (True) 
- button           (1)
- constraints      (none)
- varyCursor       (True)
- callbackPolicy   (ALWAYS)

 * TRANSLATIONS:  Set of translations for resizeBehavior:
 
- <Btn?Down>: Select()
- <Btn?Motion>: ResizeDrag()
- <Btn?Up>: ResizeDrop()

 * returns: */
KnInteractor
KnCreateResizeInteractor(Widget knvas, /* a knvas widget */
			 char *name); /* interactor's name */






/* Builds a select interactor

 * The select interactor is intended to select any object with a mouse
 * interaction. The SelectInteractor fetch resources from database at creation
 * time.

 * RESOURCES: Set of resources defined for a SelectInteractor

- visibleSelection (True) 
- button           (1)
- constraints      (none)
- varyCursor       (True)
- callbackPolicy   (ALWAYS)

 * TRANSLATIONS:  Set of translations for selectBehavior:
 
- <Btn?Down>: Select()
- <Btn?Motion>: SelectDrag()
- <Btn?Up>: SelectDrop()

 * returns: */
KnInteractor
KnCreateSelectInteractor(Widget knvas, /* a knvas widget */
			 char *name); /* interactor's name */


void KnInteractorRegisterCoverters();


void KnInteractorRealize(KnInteractor i, Widget w);


#ifdef __cplusplus
} /* Close scope of 'extern "C"' */
#endif


#endif /* __KN_INTERACTOR_H */
