/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: interactor.c,v 1.9.2.1 96/03/27 17:17:02 leon Exp $ */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "interactor.h"
#include "select-move.h"
#include "resize.h"
#include "hash.h"
#include "cvt.h"



/* type desc */
KlType KnInteractorClass;




static XtActionsRec
actions[] = {
    {"Select", SelectProc},

    {"SelectDrag", SelectDragProc},
    {"SelectDrop", SelectDropProc},
    
    {"MoveDrag", MoveDragProc},
    {"MoveDrop", MoveDropProc},

    {"ResizeDrag", ResizeDragProc},
    {"ResizeDrop", ResizeDropProc},
};



static XtResource 
interactorResources [] = {
#define offset(field) XtOffset(KnInteractor, field)
  {
      XtNvisibleSelection, XtCVisibleSelection,
      XtRBoolean, sizeof(Boolean), offset(visibleSelection),
      XtRImmediate, (caddr_t)False
  },
  {
      XtNbutton, XtCButton,
      XtRInt, sizeof(int), offset(button),
      XtRImmediate, (caddr_t)KnLEFT
  }, 
  {
      XtNconstraints, XtCConstraints,
      XtRString, sizeof(String), offset(constraints),
      XtRImmediate, (caddr_t)0
  },
  {
      XtNvaryCursor, XtCVaryCursor,
      XtRBoolean, sizeof(Boolean), offset(varyCursor),
      XtRImmediate, (caddr_t)True
  },
  {
      XtNcallbackPolicy, XtCCallbackPolicy,
      XtRCallbackPolicy, sizeof(KnCallbackPolicy), offset(callbackPolicy),
      XtRImmediate, (caddr_t)KnCALLBACK_ALWAYS
  }, 
  {
      "useGhosts", "UseGhosts",
      XtRBoolean, sizeof(Boolean), offset(useGhosts),
      XtRImmediate, (caddr_t)False
  }, 
  
#undef offset
};


static Boolean
CvtStringToCallbackPolicy(Display *dpy, XrmValue *args, Cardinal *num_args, 
			  XrmValue *from, XrmValue *to,
			  XtPointer *converter_data)
{
    int i;
    char s[32];
    static struct _names {
	char *name;
	int style;
    } names[] = {
	{"callback_never", KnCALLBACK_NEVER},
	{"callback_on_pick", KnCALLBACK_ON_PICK},
	{"callback_on_drag", KnCALLBACK_ON_DRAG},
	{"callback_on_drop", KnCALLBACK_ON_DROP},
	{"callback_on_button", KnCALLBACK_ON_BUTTON},
	{"callback_always", KnCALLBACK_ALWAYS},

    };
    StringToLower((char *)from->addr, s);
    for(i = 0; i < XtNumber(names); i++) {
	if(!strcmp(s, names[i].name)) {
	    done(&(names[i].style), int);
	    return True;
	}
    }
    XtDisplayStringConversionWarning(dpy, (char *)from->addr,
				     XtRCallbackPolicy);
    return False;
}


void KnInteractorRealize(KnInteractor i, Widget w)
{
    XtTMRec tm;
    XtTranslations xlations;

    /* save current state and prevent for toolkit free */
    tm = w->core.tm;
    w->core.tm.translations = NULL;
    w->core.tm.proc_table = NULL;
    
    xlations = i->tm.translations;
    i->tm.translations = NULL;
    
    XtVaSetValues(w, XtNtranslations, xlations, 0);
    i->tm = w->core.tm;
    
    w->core.tm = tm;
}

/* init class fields and call super class Init proc */
void
KnInteractorInit(KnInteractor interactor, String translations, Widget w)
{
    memset(&(interactor->tm), 0, sizeof(XtTMRec));
    interactor->tm.translations = XtParseTranslationTable(translations);
    if(XtIsRealized(w)) {
	KnInteractorRealize(interactor, w);
    }
    else {
	/* delay to realize time */
	KnvasRegisterInteractor(w, interactor);
    }
}



/* allocates storage space for a new KnInteractor object */
KnInteractor
KnCreateInteractor(Widget w, String translations)
{  
    KnInteractor interactor;
    XtAppContext app_context;
    int actions_registered = False;

    app_context = XtWidgetToApplicationContext(w);
    if(!actions_registered) {
	XtAppAddActions(app_context, actions, XtNumber(actions));
	actions_registered = True;
    }
    interactor =  (KnInteractor)KlOMake(KnInteractorClass);
    KnInteractorInit(interactor, translations, w);
    return interactor;
}




KnInteractor
KnInteractorMerge(Widget w, KnInteractor i1, KnInteractor i2)
{
    XtTMRec tm;

    /* save current state and prevent from toolkit free */
    tm = w->core.tm;
    w->core.tm.translations = NULL;
    w->core.tm.proc_table = NULL;

    w->core.tm = i1->tm;
    XtOverrideTranslations(w, i2->tm.translations);
    i1->tm = w->core.tm;

    /* restore old state, preventing from free */
    w->core.tm = tm;
    return i1;
}




/* declare new   selectors
 */
void
KnInteractorImport()
{
	
}

void
KnInteractorRegisterConverters ()
{
    XtSetTypeConverter(XtRString, XtRCallbackPolicy, CvtStringToCallbackPolicy,
		       NULL, 0, XtCacheNone, NULL);
}


/* class initializations for KnInteractor objects
 * 
 * returns: 
 */
void
KnInteractorClassInitialize()
{
     /* declare  type */
    KlDeclareType(&KnInteractorClass, "KnInteractor", sizeof(struct KnInteractorStruct));

    /* declare methods */
    KlDeclareMethod(KnInteractorClass, KlSelMake, KnCreateInteractor);

    /* initializes standard interactors */
}



static void
ApplyResources(KnInteractor newi, struct KnInteractorStruct *resources)
{
    newi->button = resources->button;
    newi->visibleSelection = resources->visibleSelection;
    newi->varyCursor = resources->varyCursor;
    newi->constraints = resources->constraints;
    newi->callbackPolicy = resources->callbackPolicy;
    newi->useGhosts = resources->useGhosts;
}

KnHashtable interactorTable = NULL;

static KnInteractor
InitializeInteractor3(Widget knvas, char *name, char *type, char *behavior)
{
    static count = 0;
    char translations[1024];
    char ident[32];
    char fullClass[32];
    KnInteractor newi;
    struct KnInteractorStruct resources;

    /* this is a unique indentifier for the new interactor */
    sprintf(ident, "%s%d", type, count++);
    sprintf(fullClass, "Kn%sInteractor", type);
    
    XtGetSubresources(knvas, &resources, name, fullClass,
		      interactorResources, 
		      XtNumber(interactorResources), 0, 0);
    
    sprintf(translations, behavior,
	    resources.button, ident,
	    resources.button, ident,
	    resources.button, ident);
    newi =  KnCreateInteractor(knvas, translations);
    ApplyResources(newi, &resources);

    /* register the new interactor in the sym table */
    if (NULL == interactorTable)
	interactorTable = KnHashtableMake(13);
    KnHashtableAdd(interactorTable, ident, newi);
    KlIncRef(newi);

    return newi;
}


KnInteractor
KnCreateMoveInteractor(Widget knvas, char *name)
{
    return InitializeInteractor3(knvas, name, "Move", moveBehavior);
}


KnInteractor
KnCreateResizeInteractor(Widget knvas, char *name)
{
    return InitializeInteractor3(knvas, name, "Resize", resizeBehavior);
}


KnInteractor
KnCreateSelectInteractor(Widget knvas, char *name)
{
    return InitializeInteractor3(knvas, name, "Select", selectBehavior);
}

