/* Copyright (c) 1995 by Groupe Bull.  All Rights Reserved */
/* $Id: Tag.c,v 1.5 96/01/04 11:47:24 leon Exp $ */


#include <X11/StringDefs.h>
#include <ctype.h>		/* for isalpha */
#include "TagP.h"
#include "Knvas.h"
#ifdef LOOK3D
#include <Xm/Xm.h>
#endif /* LOOK3D */
#include "cvt.h"



/*
 * external declaration
 */
void
StringToLower(char  *source, char  *dest);
void
KnvasClearTaggedObjects(Widget w, KnTag kntag);


#ifdef _NO_PROTO

static void ClassInitialize();
static Boolean SetValues();
static void Destroy();
static void Initialize();

#else /* NO_PROTO */

static void ClassInitialize(WidgetClass wc);
static void Initialize(Widget w, Widget nw, ArgList args, Cardinal *n);
static Boolean SetValues(Widget cw, Widget rw, Widget nw,
			 ArgList args, Cardinal *num_args);
static void Destroy(Widget w);

#endif /* NO_PROTO */


/* no string defs, because we do not want to allow SetValue */
static XtResource 
resources [] = {
#define offset(field) XtOffset(TagObject, tag.field)
  {
      XtNforeground, XtCForeground,
      XtRPixel, sizeof(Pixel), offset(foreground),
      XtRImmediate, (caddr_t)1
  }, 
  {
      XtNbackground, XtCBackground,
      XtRPixel, sizeof(Pixel), offset(background),
      XtRImmediate, (caddr_t)1
  }, 
  {
      XtNlineWidth, XtCLineWidth,
      XtRInt, sizeof(int), offset(line_width),
      XtRImmediate, (caddr_t)0
  }, 
  {
      XtNlineStyle, XtCLineStyle,
      XtRLineStyle, sizeof(int), offset(line_style),
      XtRImmediate, (caddr_t)LineSolid
  }, 
  {
      XtNcapStyle, XtCCapStyle,
      XtRCapStyle, sizeof(int), offset(cap_style),
      XtRImmediate, (caddr_t)CapButt
  }, 
  {
      XtNjoinStyle, XtCJoinStyle,
      XtRJoinStyle, sizeof(int), offset(join_style),
      XtRImmediate, (caddr_t)JoinMiter
  },
  {
      XtNfont, XtCFont,
      XtRFontStruct, sizeof(XFontStruct *), offset(font),
      XtRString, XtDefaultFont
  },
  {
      XtNsymColors, XtCSymColors,
      XtRSymColors, sizeof(SymColors), offset(sym),
      XtRImmediate, NULL
  },
  {
      XtNvisibleAnchors, XtCVisibleAnchors,
      XtRBoolean, sizeof(Boolean), offset(visibleAnchors),
      XtRImmediate, (caddr_t)False
  },
  {
      XtNselectionLook, XtCSelectionLook,
      XtRSelectionLook, sizeof(SelectionLook), offset(selectionLook),
      XtRImmediate, (caddr_t)KnSELECTION_FANCY
  },
#ifdef LOOK3D  
  {
      XmNshadowThickness, XmCShadowThickness,
      XmRDimension, sizeof(KnDimension), offset(shadowThickness),
      XtRImmediate, (caddr_t)0
  },
  {
      XmNshadowType, XmCShadowType,
      XmRShadowType, sizeof(unsigned char), offset(shadowType),
      XtRImmediate, (caddr_t)XmSHADOW_OUT
  },
#endif /* LOOK3D */
  
#undef offset
};



TagClassRec tagClassRec = {
	{					/* core fields */
	(WidgetClass) &objectClassRec, 		/* superclass		*/
	"KnTag",				/* class_name		*/
	sizeof(TagRec),				/* widget_size		*/
	NULL,					/* class_initialize	*/
	ClassInitialize,			/* class_part_initialize*/
	False,					/* class_inited		*/
	Initialize,				/* initialize		*/
	NULL,					/* initialize_hook	*/
	NULL,					/* obj1			*/
	NULL,					/* obj2			*/
	0,					/* obj3			*/
	resources,				/* resources		*/
	XtNumber(resources),			/* num_resources	*/
	NULLQUARK,				/* xrm_class		*/
	0,					/* obj4			*/
	0,					/* obj5			*/
	0,					/* obj6			*/
	0,					/* obj7			*/
	Destroy,				/* destroy		*/
	NULL,					/* obj8			*/
	NULL,					/* obj9			*/
	SetValues,				/* set_values		*/
	NULL,					/* set_values_hook	*/
	NULL,			 		/* obj10		*/
	NULL,					/* get_values_hook	*/
	NULL,					/* obj11		*/
	XtVersion,				/* version		*/
	NULL,					/* callback_private	*/
	NULL,					/* obj12		*/
	NULL,					/* obj13		*/
	NULL,					/* obj14		*/
	NULL,					/* extension		*/
	},
	{ /* tag class fields */
	0,    					/* empty		*/
	}
};

WidgetClass tagObjectClass = (WidgetClass)&tagClassRec;


/* parse the string to build the color table.

 <table> ::= <colordesc> (',' <colordesc>) *
 <colordesc> ::= <ws> <ident> <ws> '=' <ws> <name> <ws>
 <ident> ::= [a-zA-Z0-9_]+
 <name> ::= [a-zA-Z][a-zA-Z ]*
 <ws> ::= [ \t\n]
   
 * automata description: - state 0: beginning or one color has ben parsed
 
 * returns: The created color table. each pixel value is set to 0, so that the
 * symbolic name will be used by the Xpm lib */
static XpmColorSymbol *
ParseColorTable(char *from, int *num)
{
    XpmColorSymbol *sym = NULL;
    char *buffer, *p, *s, *t;
    int state = 0, pos = 0;
    buffer = XtNewString(from);
    for(p = buffer; '\0' != *p; p++) {
	switch(state) {
	case 0:			/* <colordesc> */
	    switch(*p) {
	    case ' ' :
	    case '\n' :
	    case '\t' : break;
	    default:
		s = p;
		state = 1;
		if(NULL == sym) {
		    sym = XtNew(XpmColorSymbol);
		    pos = 0;
		}
		else {
		    sym = (XpmColorSymbol*)XtRealloc((void *)sym, sizeof(XpmColorSymbol)*(pos+1));
		}		    
	    }
	    break;
	case 1 :		/* <ident> */
	    if(isalpha(*p)) break;
	    *p = '\0';
	    sym[pos].name = XtNewString(s);
	    state = 2;
	    break;
	case 2:			/* <ws> '=' <ws> */
	    if(isalpha(*p)) {
		s = p;
		t = NULL;
		state = 3;
	    };
	    break;
	case 3:			/* <name> */
	    if(isalpha(*p)) break;
	    if(' ' == *p) {
		t = p;
		state = 4;
		break;
	    }
	    if(',' == *p) {
		if(NULL == t) t = p;
		*t = '\0';
		sym[pos].value = XtNewString(s);
		pos++;
		state = 0;
	    }
	    break;
	case 4:			/* skip white spaces in <name> */
	    if(' ' == *p) break;
	    if(isalpha(*p)) {
		state = 3;
		t = NULL;
		break;
	    }
	    if(',' == *p) {
		*p = '\0';
		sym[pos].value = XtNewString(s);
		pos++;
		state = 0;
	    }
	} /* switch */
    } /* for */
    if(4 == state || 3 == state) {
	if(NULL == t) t = p;
	*t = '\0';
	sym[pos].value = XtNewString(s);
	pos++;
	state = 0;
    }
    XtFree(buffer);
    if(0 != state) {
	*num = 0;
	return NULL;
    }
    else {
	*num = pos;
	return sym;
    }
}


Boolean
CvtStringToColorTable(Display *dpy, XrmValue *args, Cardinal *num_args, 
	       XrmValue *from, XrmValue *to, XtPointer *converter_data)
{
    int n;
    XpmColorSymbol *sym;
    SymColors table;
    sym = 
	ParseColorTable((char *)from->addr, &n);
    if(NULL == sym) {
	XtDisplayStringConversionWarning(dpy,
					 (char *)from->addr, XtRSymColors);
	return False;
    }
    else {
	table = XtNew(SymColorsRec);
	table->colors = sym;
	table->n = n;
	done(&table, void *);
    }
    return True;
}





Boolean
CvtStringToSelectionLook(Display *dpy, XrmValue *args, Cardinal *num_args, 
	       XrmValue *from, XrmValue *to, XtPointer *converter_data)
{
    int i;
    char s[32];
    static struct _names {
	char *name;
	int style;
    } names[] = {
	{"unvisible", KnSELECTION_UNVISIBLE},
	{"simple", KnSELECTION_SIMPLE},
	{"fancy", KnSELECTION_FANCY},
	{"none", KnSELECTION_UNVISIBLE},
    };
    StringToLower((char *)from->addr, s);
    for(i = 0; i < XtNumber(names); i++) {
	if(!strcmp(s, names[i].name)) {
	    done(&(names[i].style), int);
	    return True;
	}
    }
    XtDisplayStringConversionWarning(dpy, (char *)from->addr,
				     XtRSelectionLook);
    return False;
}


Boolean
CvtStringToLineStyle(Display *dpy, XrmValue *args, Cardinal *num_args, 
	       XrmValue *from, XrmValue *to, XtPointer *converter_data)
{
    int i;
    char s[32];
    static struct _names {
	char *name;
	int style;
    } names[] = {
	{"solid", LineSolid},
	{"doubledash", LineDoubleDash},
	{"onoffdash", LineOnOffDash},
    };
    StringToLower((char *)from->addr, s);
    for(i = 0; i < XtNumber(names); i++) {
	if(!strcmp(s, names[i].name)) {
	    done(&(names[i].style), int);
	    return True;
	}
    }
    XtDisplayStringConversionWarning(dpy, (char *)from->addr, "LineStyle");
    return False;
}




Boolean
CvtStringToCapStyle(Display *dpy, XrmValue *args, Cardinal *num_args, 
	       XrmValue *from, XrmValue *to, XtPointer *converter_data)
{
    int i;
    char s[32];
    static struct _names {
	char *name;
	int style;
    } names[] = {
	{"capnotlast", CapNotLast},
	{"capbutt", CapButt},
	{"capround", CapRound},
	{"capprojecting", CapProjecting},
    };
    StringToLower((char *)from->addr, s);
    for(i = 0; i < XtNumber(names); i++) {
	if(!strcmp(s, names[i].name)) {
	    done(&(names[i].style), int);
	    return True;
	}
    }
    XtDisplayStringConversionWarning(dpy, (char *)from->addr, "CapStyle");
    return False;
}




Boolean
CvtStringToJoinStyle(Display *dpy, XrmValue *args, Cardinal *num_args, 
	       XrmValue *from, XrmValue *to, XtPointer *converter_data)
{
    int i;
    char s[32];
    static struct _names {
	char *name;
	int style;
    } names[] = {
	{"joinmiter", JoinMiter},
	{"joinround", JoinRound},
	{"joinbevel", JoinBevel},
    };
    StringToLower((char *)from->addr, s);
    for(i = 0; i < XtNumber(names); i++) {
	if(!strcmp(s, names[i].name)) {
	    done(&(names[i].style), int);
	    return True;
	}
    }
    XtDisplayStringConversionWarning(dpy, (char *)from->addr, "JoinStyle");
    return False;
}



static void 
#ifdef _NO_PROTO
Initialize(w, nw, args, n)
    Widget w;
    Widget nw;
    ArgList args;
    Cardinal *n;
#else /* NO_PROTO */
Initialize(
    Widget w,
    Widget nw,
    ArgList args,
    Cardinal *n)
#endif /* NO_PROTO */
{
    TagObject tag = (TagObject)nw;
#define THIS tag->tag

    THIS.gc = None;
    THIS.knvas = XtParent(nw);

    /* sanity checks */
    if(THIS.line_width < 0) THIS.line_width = 0;

    /* notify to our parent */
    /* will probably become a notify method, one day ... */
    KnvasRegisterTag(THIS.knvas, (KnTag)tag);
#undef THIS
}



/* allocates storage space for a new tag
 * gc is not created, because this requires the window id 
   (so the widget must be realized)
 * returns: the newly created KnTag
 */
KnTag
KnCreateTag(Widget knvas, String name)
{
    KnTag tag;
    tag = (KnTag)XtCreateWidget(name, tagObjectClass, knvas, NULL, 0);
    return tag;
}



#ifdef _NO_PROTO
static void Destroy(w)
    Widget w;
#else /* NO_PROTO */
static void Destroy(Widget w)
#endif /* NO_PROTO */
{
    TagObject tag = (TagObject)w;
#define THIS tag->tag
    if(None != THIS.gc)
	XFreeGC(XtDisplay(THIS.knvas), THIS.gc);
#ifdef LOOK3D
    if(None != THIS.top_shadow_gc)
	XFreeGC(XtDisplay(THIS.knvas), THIS.top_shadow_gc);
    if(None != THIS.bottom_shadow_gc)
	XFreeGC(XtDisplay(THIS.knvas), THIS.bottom_shadow_gc);
#endif /* LOOK3D */
#undef THIS
}



static Boolean 
#ifdef _NO_PROTO
SetValues(cw, rw, nw, args, num_args)
    Widget cw;
    Widget rw;
    Widget nw;
    ArgList args;
    Cardinal *num_args;
#else /* NO_PROTO */
SetValues(
    Widget cw,
    Widget rw,
    Widget nw,
    ArgList args,
    Cardinal *num_args)
#endif /* NO_PROTO */
{
    TagObject ntag = (TagObject)nw;
    TagObject ctag = (TagObject)cw;
    Boolean redraw = False;
    XGCValues values;
    Mask mask = 0L;

    if(ntag->tag.foreground != ctag->tag.foreground) {
	redraw = True;
	mask |= GCForeground;
	values.foreground = ntag->tag.foreground;
    }
    if(ntag->tag.background != ctag->tag.background) {
	redraw = True;
	mask |= GCBackground;
	values.background = ntag->tag.background;
    }
    if(ntag->tag.line_style != ctag->tag.line_style) {
	redraw = True;
	mask |= GCLineStyle;
	values.line_style = ntag->tag.line_style;
    }
    if(ntag->tag.line_width != ctag->tag.line_width) {
	redraw = True;
	mask |= GCLineWidth;
	values.line_width = ntag->tag.line_width;
    }
    if(ntag->tag.cap_style != ctag->tag.cap_style) {
	redraw = True;
	mask |= GCCapStyle;
	values.cap_style = ntag->tag.cap_style;
    }
    if(ntag->tag.join_style != ctag->tag.join_style) {
	redraw = True;
	mask |= GCJoinStyle;
	values.join_style = ntag->tag.join_style;
    }
    if(ntag->tag.font != ctag->tag.font) {
	redraw = True;
	mask |= GCFont;
	values.font = ntag->tag.font->fid;
    }

    /* WARNING !!! This breaks out multi display */
    if(redraw && XtIsRealized(ntag->tag.knvas)) {
	XChangeGC(XtDisplay(ntag->tag.knvas), ntag->tag.gc, mask, &values);
#ifdef LOOK3D
	{
	    /* update shadow GCs */
	    Pixel top_pixel, bottom_pixel;
	    XmGetColors(XtScreen(ntag->tag.knvas),
			DefaultColormap(XtDisplay(ntag->tag.knvas), 0),
			ntag->tag.foreground, NULL, &top_pixel, &bottom_pixel,
			NULL);
	    values.foreground = top_pixel;
	    XChangeGC(XtDisplay(ntag->tag.knvas), ntag->tag.top_shadow_gc,
		      mask, &values);
	    values.foreground = bottom_pixel;
	    XChangeGC(XtDisplay(ntag->tag.knvas), ntag->tag.bottom_shadow_gc,
		      mask, &values);
	    
	}
	
	KnvasClearTaggedObjects(ntag->tag.knvas, (KnTag)ntag);
#endif /* LOOK3D */
    }
    return False;
}



/* Creates the gc.
 * set defaults values from resources
 * returns: 
 */
void
KnTagRealize(KnTag tago)
{
    TagObject tag = (TagObject)tago;
#define THIS tag->tag
    XGCValues vals;
    unsigned long mask;

    mask = GCForeground | GCBackground | GCLineWidth | GCLineStyle
	| GCGraphicsExposures | GCCapStyle | GCJoinStyle | GCFont
	| GCTileStipXOrigin | GCTileStipYOrigin;
    vals.background = THIS.background;
    vals.foreground = THIS.foreground;
    vals.line_style = THIS.line_style;
    vals.line_width = THIS.line_width;
    vals.cap_style = THIS.cap_style;
    vals.join_style = THIS.join_style;
    vals.font = THIS.font->fid;
    vals.graphics_exposures = False;
    vals.ts_x_origin = 0;
    vals.ts_y_origin = 0;
    THIS.gc = XCreateGC(XtDisplay(THIS.knvas), XtWindow(THIS.knvas),
			 mask, &vals);
#ifdef LOOK3D
    {
	Pixel top_pixel, bottom_pixel;
	XmGetColors(XtScreen(THIS.knvas),
		    DefaultColormap(XtDisplay(THIS.knvas), 0),
		    THIS.foreground, NULL, &top_pixel, &bottom_pixel,
		    NULL);
	vals.foreground = top_pixel;
	THIS.top_shadow_gc =
	    XCreateGC(XtDisplay(THIS.knvas), XtWindow(THIS.knvas),
		      mask, &vals);
	vals.foreground = bottom_pixel;
	THIS.bottom_shadow_gc =
	    XCreateGC(XtDisplay(THIS.knvas), XtWindow(THIS.knvas),
		      mask, &vals);
    }
#endif /* LOOK3D */
}





static void
#ifdef _NO_PROTO
ClassInitialize(wc)
    WidgetClass wc;
#else /* NO_PROTO */
ClassInitialize(WidgetClass wc)
#endif /* NO_PROTO */
{
    /* do not use XmRepTypeRegister to be Motif independant */
    XtSetTypeConverter(XtRString, XtRLineStyle, CvtStringToLineStyle, NULL, 0,
		       XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XtRCapStyle, CvtStringToCapStyle, NULL, 0,
		       XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XtRJoinStyle, CvtStringToJoinStyle, NULL, 0,
		       XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XtRSymColors, CvtStringToColorTable, NULL, 0,
		       XtCacheAll, NULL);
    XtSetTypeConverter(XtRString, XtRSelectionLook,
		       CvtStringToSelectionLook, NULL, 0,
		       XtCacheAll, NULL);
}




#ifdef LOOK3D
void
KnTagComputeShadow(KnTag kntag,
		   GC *top_gc,
		   GC *bottom_gc,
		   KnDimension *top_thick,
		   KnDimension *bottom_thick)
{
    TagObject tag = (TagObject) kntag;
#define THIS tag->tag    
    *top_thick = 1;
    *bottom_thick = 1;
    if((XmSHADOW_OUT == THIS.shadowType) ||
       (XmSHADOW_ETCHED_OUT == THIS.shadowType)) {
	*top_gc =  THIS.top_shadow_gc;
	*bottom_gc = THIS.bottom_shadow_gc;
	*bottom_thick = THIS.shadowThickness;
    }
    else {
	*top_gc =  THIS.bottom_shadow_gc;
	*bottom_gc = THIS.top_shadow_gc;
	*top_thick = THIS.shadowThickness;
    }
#undef THIS
}
#endif /* LOOK3D */
