/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * mapscan.c
 * kirk johnson
 * august 1993
 *
 * RCS $Id: scan.c,v 1.16 1995/09/25 01:12:11 tuna Exp $
 *
 * Copyright (C) 1989, 1990, 1993, 1994, 1995 Kirk Lauritz Johnson
 *
 * Parts of the source code (as marked) are:
 *   Copyright (C) 1989, 1990, 1991 by Jim Frost
 *   Copyright (C) 1992 by Jamie Zawinski <jwz@lucid.com>
 *
 * Permission to use, copy, modify and freely distribute xearth for
 * non-commercial and not-for-profit purposes is hereby granted
 * without fee, provided that both the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.
 *
 * Unisys Corporation holds worldwide patent rights on the Lempel Zev
 * Welch (LZW) compression technique employed in the CompuServe GIF
 * image file format as well as in other formats. Unisys has made it
 * clear, however, that it does not require licensing or fees to be
 * paid for freely distributed, non-commercial applications (such as
 * xearth) that employ LZW/GIF technology. Those wishing further
 * information about licensing the LZW patent should contact Unisys
 * directly at (lzw_info@unisys.com) or by writing to
 *
 *   Unisys Corporation
 *   Welch Licensing Department
 *   M/S-C1SW19
 *   P.O. Box 500
 *   Blue Bell, PA 19424
 *
 * The author makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express or
 * implied warranty.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "earthmisc.h"
#include "mapdata.h"
#include "mapscan.h"
#include "XearthI.h"

void			mso_alloc(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  t_status		status;
  int			i;

  if ((mso->scanbits = XEARTH_ARR_NEW(sizeof (ScanBit),
				      "mso_alloc_scanbits",
				      &status)) == NULL)
    {
      XearthWarning("XEARTH_ARR_NEW");
      exit(1);
    }
  if ((mso->edgexings = XEARTH_ARR_NEW(sizeof (EdgeXing),
				       "mso_alloc_edgexings",
				       &status)) == NULL)
    {
      XearthWarning("XEARTH_ARR_NEW");
      exit(1);
    }
  
  if ((mso->scanbuf = XEARTH_ALLOC_PROC(sizeof (XearthArr *) * msi->hght,
					"xearth",
					"mso_alloc:scanbuf",
					&status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  for (i=0; i<msi->hght; i++)
    {
      if ((mso->scanbuf[i] = XEARTH_ARR_NEW(sizeof (double),
					    "xearth",
					    &status)) == NULL)
	{
	  XearthWarning("XEARTH_ARR_NEW");
	  exit(1);
	}
#ifdef NOTDEF
      printf("%d %p\n",i,mso->scanbuf[i]);
#endif
    }
}

void			mso_reset(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  int			i;

  ARR_EMPTY(mso->scanbits);
  ARR_EMPTY(mso->edgexings);
  for (i=0; i<msi->hght; i++)
    {
      ARR_EMPTY(mso->scanbuf[i]);
    }
}

void			mso_cleanup(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  int			i;

  XearthArrDelete(mso->edgexings);
  XearthArrDelete(mso->scanbits);
  for (i=0; i<msi->hght; i++)
    {
#ifdef NOTDEF
      printf("%d %d %p\n",i,ARR_COUNT(mso->scanbuf[i]),mso->scanbuf[i]);
#endif
      XearthArrDelete(mso->scanbuf[i]);
    }
  XEARTH_FREE_PROC(mso->scanbuf,
		   "xearth",
		   "mso_alloc:scanbuf");
}

static int double_comp(a, b)
     const void *a;
     const void *b;
{
  double val_a;
  double val_b;
  int    rslt;

  val_a = *((const double *) a);
  val_b = *((const double *) b);

  if (val_a < val_b)
    rslt = -1;
  else if (val_a > val_b)
    rslt = 1;
  else
    rslt = 0;

  return rslt;
}


static int scanbit_comp(a, b)
     const void *a;
     const void *b;
{
  return (((const ScanBit *) a)->y - ((const ScanBit *) b)->y);
}


static int orth_edgexing_comp(a, b)
     const void *a;
     const void *b;
{
  double angle_a;
  double angle_b;
  int    rslt;

  angle_a = ((const EdgeXing *) a)->angle;
  angle_b = ((const EdgeXing *) b)->angle;

  if (angle_a < angle_b)
    rslt = -1;
  else if (angle_a > angle_b)
    rslt = 1;
  else
    rslt = 0;

  return rslt;
}


static int merc_edgexing_comp(a, b)
     const void *a;
     const void *b;
{
  double val_a;
  double val_b;
  int    rslt;

  val_a = ((const EdgeXing *) a)->angle;
  val_b = ((const EdgeXing *) b)->angle;

  if (val_a < val_b)
  {
    rslt = -1;
  }
  else if (val_a > val_b)
  {
    rslt = 1;
  }
  else if (val_a == 0)
  {
    val_a = ((const EdgeXing *) a)->y;
    val_b = ((const EdgeXing *) b)->y;

    if (val_a < val_b)
      rslt = -1;
    else if (val_a > val_b)
      rslt = 1;
    else
      rslt = 0;
  }
  else if (val_a == 2)
  {
    val_a = ((const EdgeXing *) a)->y;
    val_b = ((const EdgeXing *) b)->y;

    if (val_a > val_b)
      rslt = -1;
    else if (val_a < val_b)
      rslt = 1;
    else
      rslt = 0;
  }
  else
  {
    /* keep lint happy */
    rslt = 0;
    assert(0);
  }

  return rslt;
}


void			scan_map(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  int i;
  t_status	status;

  mso->view_pos_info.cos_lat = cos(msi->view_lat * (M_PI/180));
  mso->view_pos_info.sin_lat = sin(msi->view_lat * (M_PI/180));
  mso->view_pos_info.cos_lon = cos(msi->view_lon * (M_PI/180));
  mso->view_pos_info.sin_lon = sin(msi->view_lon * (M_PI/180));
  mso->view_pos_info.cos_rot = cos(msi->view_rot * (M_PI/180));
  mso->view_pos_info.sin_rot = sin(msi->view_rot * (M_PI/180));

  if (msi->proj_type == ProjTypeOrthographic)
    mso->proj_scale = ((msi->hght < msi->wdth) ? msi->hght : msi->wdth) * (msi->view_mag / 2) * 0.99;
  else /* (msi->proj_type == ProjTypeMercator) */
    mso->proj_scale = (msi->view_mag * msi->wdth) / (2 * M_PI);

  mso->proj_xofs      = (double) msi->wdth / 2 + msi->shift_x;
  mso->proj_yofs      = (double) msi->hght / 2 + msi->shift_y;
  mso->inv_proj_scale = 1 / mso->proj_scale;

  /* the first time through, allocate scanbits and edgexings;
   * on subsequent passes, simply reset them.
   */
  if (msi->proj_type == ProjTypeOrthographic)
  {
    orth_scan_outline(msi,mso);
    orth_scan_curves(msi,mso);
  }
  else /* (msi->proj_type == ProjTypeMercator) */
  {
    merc_scan_outline(msi,mso);
    merc_scan_curves(msi,mso);
  }

  qsort(mso->scanbits->elts,
	ARR_COUNT(mso->scanbits),
	sizeof (ScanBit), scanbit_comp);
}

void orth_scan_outline(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  mso->min_y = msi->hght;
  mso->max_y = -1;

  orth_scan_arc(msi,mso,1.0, 0.0, 0.0, 1.0, 0.0, (2*M_PI));

  get_scanbits(msi,mso,64);
}


void orth_scan_curves(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  int     i;
  int     cidx;
  int     npts;
  int     val;
  short  *raw;
  double *pos;
  double *prev;
  double *curr;

  cidx = 0;
  raw  = map_data;
  while (1)
  {
    npts = raw[0];
    if (npts == 0) break;
    val  = raw[1];
    raw += 2;

    pos   = orth_extract_curve(mso,npts, raw);
    prev  = pos + (npts-1)*3;
    curr  = pos;
    mso->min_y = msi->hght;
    mso->max_y = -1;

    for (i=0; i<npts; i++)
    {
      orth_scan_along_curve(msi,mso,prev, curr, cidx);
      prev  = curr;
      curr += 3;
    }
    XEARTH_FREE_PROC(pos,
		     "xearth",
		     "*:pos");
    if (ARR_COUNT(mso->edgexings) > 0)
      orth_handle_xings(msi,mso);
    if (mso->min_y <= mso->max_y)
      get_scanbits(msi,mso,val);

    cidx += 1;
    raw  += 3*npts;
  }
}


double *orth_extract_curve(mso,npts, data)
     t_map_scan_output	*mso;
     int    npts;
     short *data;
{
  int     i;
  int     x, y, z;
  double  scale;
  double *pos;
  double *rslt;
  t_status status;

  if ((rslt = XEARTH_ALLOC_PROC(sizeof (double) * 3 * npts,
				"xearth",
				"orth_extract_curve:pos",
				&status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }

  x     = 0;
  y     = 0;
  z     = 0;
  scale = 1.0 / MAP_DATA_SCALE;
  pos   = rslt;

  for (i=0; i<npts; i++)
  {
    x += data[0];
    y += data[1];
    z += data[2];

    pos[0] = x * scale;
    pos[1] = y * scale;
    pos[2] = z * scale;

    XFORM_ROTATE(pos, mso->view_pos_info);

    data += 3;
    pos  += 3;
  }

  return rslt;
}


void orth_scan_along_curve(msi,mso,prev, curr, cidx)
     t_map_scan_input	*msi;
     t_map_scan_output	*mso;
     double *prev;
     double *curr;
     int     cidx;
{
  double    extra[3];
  EdgeXing *xing;
  t_status	status;

  if (prev[2] <= 0)             /* prev not visible */
  {
    if (curr[2] <= 0)
      return;                   /* neither point visible */

    orth_find_edge_xing(prev, curr, extra);

    /* extra[] is an edge crossing (entry point) */
    if ((xing = XearthArrNext(mso->edgexings,
			 &status)) == NULL)
      {
	XearthWarning("arr_next");
	exit(1);
      }
    xing->type  = XingTypeEntry;
    xing->cidx  = cidx;
    xing->x     = extra[0];
    xing->y     = extra[1];
    xing->angle = atan2(extra[1], extra[0]);

    prev = extra;
  }
  else if (curr[2] <= 0)        /* curr not visible */
  {
    orth_find_edge_xing(prev, curr, extra);

    /* extra[] is an edge crossing (exit point) */
    if ((xing = XearthArrNext(mso->edgexings,
			 &status)) == NULL)
      {
	XearthWarning("arr_next");
	exit(1);
      }
    xing->type  = XingTypeExit;
    xing->cidx  = cidx;
    xing->x     = extra[0];
    xing->y     = extra[1];
    xing->angle = atan2(extra[1], extra[0]);

    curr = extra;
  }

  scan(msi,mso,XPROJECT(prev[0]), YPROJECT(prev[1]),
       XPROJECT(curr[0]), YPROJECT(curr[1]));
}


void orth_find_edge_xing(prev, curr, rslt)
     double *prev;
     double *curr;
     double *rslt;
{
  double tmp;
  double r0, r1;

  tmp = curr[2] / (curr[2] - prev[2]);
  r0 = curr[0] - tmp * (curr[0] - prev[0]);
  r1 = curr[1] - tmp * (curr[1] - prev[1]);

  tmp = sqrt((r0*r0) + (r1*r1));
  rslt[0] = r0 / tmp;
  rslt[1] = r1 / tmp;
  rslt[2] = 0;
}


void orth_handle_xings(msi,mso)
t_map_scan_input *msi;
t_map_scan_output *mso;
{
  int       i;
  int       nxings;
  EdgeXing *xings;
  EdgeXing *from;
  EdgeXing *to;

  xings  = (EdgeXing *) mso->edgexings->elts;
  nxings = ARR_COUNT(mso->edgexings);

  assert((nxings % 2) == 0);
  qsort(xings, (unsigned) nxings, sizeof(EdgeXing), orth_edgexing_comp);

  if (xings[0].type == XingTypeExit)
  {
    for (i=0; i<nxings; i+=2)
    {
      from = &(xings[i]);
      to   = &(xings[i+1]);

      if ((from->type != XingTypeExit) ||
          (to->type != XingTypeEntry))
        xing_error(msi,__FILE__, __LINE__, i, nxings, xings);

      orth_scan_arc(msi,mso,from->x, from->y, from->angle,
                    to->x, to->y, to->angle);
    }
  }
  else
  {
    from = &(xings[nxings-1]);
    to   = &(xings[0]);

    if ((from->type != XingTypeExit) ||
        (to->type != XingTypeEntry) ||
        (from->angle < to->angle))
      xing_error(msi,__FILE__, __LINE__, nxings-1, nxings, xings);

    orth_scan_arc(msi,mso,from->x, from->y, from->angle,
                  to->x, to->y, to->angle+(2*M_PI));

    for (i=1; i<(nxings-1); i+=2)
    {
      from = &(xings[i]);
      to   = &(xings[i+1]);

      if ((from->type != XingTypeExit) ||
          (to->type != XingTypeEntry))
        xing_error(msi,__FILE__, __LINE__, i, nxings, xings);

      orth_scan_arc(msi,mso,from->x, from->y, from->angle,
                    to->x, to->y, to->angle);
    }
  }

  ARR_EMPTY(mso->edgexings);
}


void orth_scan_arc(msi,mso,x_0, y_0, a_0, x_1, y_1, a_1)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     double x_0, y_0, a_0;
     double x_1, y_1, a_1;
{
  int    i;
  int    lo, hi;
  double angle, step;
  double prev_x, prev_y;
  double curr_x, curr_y;

  assert(a_0 < a_1);

  step = mso->inv_proj_scale * 10;
  if (step > 0.05) step = 0.05;
  lo = ceil(a_0 / step);
  hi = floor(a_1 / step);

  prev_x = XPROJECT(x_0);
  prev_y = YPROJECT(y_0);

  for (i=lo; i<=hi; i++)
  {
    angle  = i * step;
    curr_x = XPROJECT(cos(angle));
    curr_y = YPROJECT(sin(angle));
    scan(msi,mso,prev_x, prev_y, curr_x, curr_y);

    prev_x = curr_x;
    prev_y = curr_y;
  }

  curr_x = XPROJECT(x_1);
  curr_y = YPROJECT(y_1);
  scan(msi,mso,prev_x, prev_y, curr_x, curr_y);
}


void merc_scan_outline(msi,mso)
t_map_scan_input *msi;
t_map_scan_output *mso;
{
  double left, right;
  double top, bottom;

  mso->min_y = msi->hght;
  mso->max_y = -1;

  left   = XPROJECT(-M_PI);
  right  = XPROJECT(M_PI);
  top    = YPROJECT(BigNumber);
  bottom = YPROJECT(-BigNumber);

  scan(msi,mso,right, top, left, top);
  scan(msi,mso,left, top, left, bottom);
  scan(msi,mso,left, bottom, right, bottom);
  scan(msi,mso,right, bottom, right, top);

  get_scanbits(msi,mso,64);
}


void merc_scan_curves(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  int     i;
  int     cidx;
  int     npts;
  int     val;
  short  *raw;
  double *pos;
  double *prev;
  double *curr;

  cidx = 0;
  raw  = map_data;
  while (1)
  {
    npts = raw[0];
    if (npts == 0) break;
    val  = raw[1];
    raw += 2;

    pos   = merc_extract_curve(mso,npts, raw);
    prev  = pos + (npts-1)*5;
    curr  = pos;
    mso->min_y = msi->hght;
    mso->max_y = -1;

    for (i=0; i<npts; i++)
    {
      merc_scan_along_curve(msi,mso,prev, curr, cidx);
      prev  = curr;
      curr += 5;
    }
    XEARTH_FREE_PROC(pos,
		     "xearth",
		     "*:pos");
    if (ARR_COUNT(mso->edgexings) > 0)
      merc_handle_xings(msi,mso);
    if (mso->min_y <= mso->max_y)
      get_scanbits(msi,mso,val);

    cidx += 1;
    raw  += 3*npts;
  }
}


double *merc_extract_curve(mso,npts, data)
     t_map_scan_output *mso;
     int    npts;
     short *data;
{
  int     i;
  int     x, y, z;
  double  scale;
  double *pos;
  double *rslt;
  t_status status;

  if ((rslt = XEARTH_ALLOC_PROC(sizeof (double) * 5 * npts,
				"xearth",
				"merc_extract_curve:pos",
				&status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  x     = 0;
  y     = 0;
  z     = 0;
  scale = 1.0 / MAP_DATA_SCALE;
  pos   = rslt;

  for (i=0; i<npts; i++)
  {
    x += data[0];
    y += data[1];
    z += data[2];

    pos[0] = x * scale;
    pos[1] = y * scale;
    pos[2] = z * scale;

    XFORM_ROTATE(pos, mso->view_pos_info);

    /* apply mercator projection
     */
    pos[3] = MERCATOR_X(pos[0], pos[2]);
    pos[4] = MERCATOR_Y(pos[1]);

    data += 3;
    pos  += 5;
  }

  return rslt;
}


void merc_scan_along_curve(msi,mso,prev, curr, cidx)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     double *prev;
     double *curr;
     int     cidx;
{
  double    px, py;
  double    cx, cy;
  double    dx;
  double    mx, my;
  EdgeXing *xing;
  t_status status;

  px = prev[3];
  cx = curr[3];
  py = prev[4];
  cy = curr[4];
  dx = cx - px;

  if (dx > 0)
  {
    /* curr to the right of prev
     */

    if (dx > ((2*M_PI) - dx))
    {
      /* vertical edge crossing to the left of prev
       */

      /* find exit point (left edge) */
      mx = - M_PI;
      my = merc_find_edge_xing(prev, curr);

      /* scan from prev to exit point */
      scan(msi,mso,XPROJECT(px), YPROJECT(py), XPROJECT(mx), YPROJECT(my));

      /* (mx, my) is an edge crossing (exit point) */
      
      if ((xing = XearthArrNext(mso->edgexings,
			   &status)) == NULL)
	{
	  XearthWarning("arr_next");
	  exit(1);
	}
      xing->type  = XingTypeExit;
      xing->cidx  = cidx;
      xing->x     = mx;
      xing->y     = my;
      xing->angle = 2; /* left edge */

      /* scan from entry point (right edge) to curr */
      mx = M_PI;
      scan(msi,mso,XPROJECT(mx), YPROJECT(my), XPROJECT(cx), YPROJECT(cy));

      /* (mx, my) is an edge crossing (entry point) */
      if ((xing = XearthArrNext(mso->edgexings,
			   &status)) == NULL)
	{
	  XearthWarning("arr_next");
	  exit(1);
	}
      xing->type  = XingTypeEntry;
      xing->cidx  = cidx;
      xing->x     = mx;
      xing->y     = my;
      xing->angle = 0; /* right edge */
    }
    else
    {
      /* no vertical edge crossing
       */
      scan(msi,mso,XPROJECT(px), YPROJECT(py), XPROJECT(cx), YPROJECT(cy));
    }
  }
  else
  {
    /* curr to the left of prev
     */
    dx = - dx;

    if (dx > ((2*M_PI) - dx))
    {
      /* vertical edge crossing to the right of prev
       */

      /* find exit point (right edge) */
      mx = M_PI;
      my = merc_find_edge_xing(prev, curr);

      /* scan from prev to exit point */
      scan(msi,mso,XPROJECT(px), YPROJECT(py), XPROJECT(mx), YPROJECT(my));

      /* (mx, my) is an edge crossing (exit point) */
      if ((xing = XearthArrNext(mso->edgexings,
			   &status)) == NULL)
	{
	  XearthWarning("arr_next");
	  exit(1);
	}
      xing->type  = XingTypeExit;
      xing->cidx  = cidx;
      xing->x     = mx;
      xing->y     = my;
      xing->angle = 0; /* right edge */

      /* scan from entry point (left edge) to curr */
      mx = - M_PI;
      scan(msi,mso,XPROJECT(mx), YPROJECT(my), XPROJECT(cx), YPROJECT(cy));

      /* (mx, my) is an edge crossing (entry point) */
      if ((xing = XearthArrNext(mso->edgexings,
			   &status)) == NULL)
	{
	  XearthWarning("arr_next");
	  exit(1);
	}
      xing->type  = XingTypeEntry;
      xing->cidx  = cidx;
      xing->x     = mx;
      xing->y     = my;
      xing->angle = 2; /* left edge */
    }
    else
    {
      /* no vertical edge crossing
       */
      scan(msi,mso,XPROJECT(px), YPROJECT(py), XPROJECT(cx), YPROJECT(cy));
    }
  }
}


double merc_find_edge_xing(prev, curr)
     double *prev;
     double *curr;
{
  double ratio;
  double scale;
  double z1, z2;
  double rslt;

  if (curr[0] != 0)
  {
    ratio = (prev[0] / curr[0]);
    z1 = prev[1] - (ratio * curr[1]);
    z2 = prev[2] - (ratio * curr[2]);
  }
  else
  {
    z1 = curr[1];
    z2 = curr[2];
  }

  scale = ((z2 > 0) ? -1 : 1) / sqrt((z1*z1) + (z2*z2));
  z1 *= scale;

  rslt = MERCATOR_Y(z1);

  return rslt;
}


void merc_handle_xings(msi,mso)
t_map_scan_input	*msi;
t_map_scan_output *mso;
{
  int       i;
  int       nxings;
  EdgeXing *xings;
  EdgeXing *from;
  EdgeXing *to;

  xings  = (EdgeXing *) mso->edgexings->elts;
  nxings = ARR_COUNT(mso->edgexings);

  assert((nxings % 2) == 0);
  qsort(xings, (unsigned) nxings, sizeof(EdgeXing), merc_edgexing_comp);

  if (xings[0].type == XingTypeExit)
  {
    for (i=0; i<nxings; i+=2)
    {
      from = &(xings[i]);
      to   = &(xings[i+1]);

      if ((from->type != XingTypeExit) ||
          (to->type != XingTypeEntry))
        xing_error(msi,__FILE__, __LINE__, i, nxings, xings);

      merc_scan_edge(msi,mso,from, to);
    }
  }
  else
  {
    from = &(xings[nxings-1]);
    to   = &(xings[0]);

    if ((from->type != XingTypeExit) ||
        (to->type != XingTypeEntry) ||
        (from->angle < to->angle))
      xing_error(msi,__FILE__, __LINE__, nxings-1, nxings, xings);

    merc_scan_edge(msi,mso,from, to);

    for (i=1; i<(nxings-1); i+=2)
    {
      from = &(xings[i]);
      to   = &(xings[i+1]);

      if ((from->type != XingTypeExit) ||
          (to->type != XingTypeEntry))
        xing_error(msi,__FILE__, __LINE__, i, nxings, xings);

      merc_scan_edge(msi,mso,from, to);
    }
  }

  ARR_EMPTY(mso->edgexings);
}


void merc_scan_edge(msi,mso,from, to)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     EdgeXing *from;
     EdgeXing *to;
{
  int    s0, s1, s_new;
  double x_0, x_1, x_new;
  double y_0, y_1, y_new;

  s0 = from->angle;
  x_0 = XPROJECT(from->x);
  y_0 = YPROJECT(from->y);

  s1 = to->angle;
  x_1 = XPROJECT(to->x);
  y_1 = YPROJECT(to->y);

  while (s0 != s1)
  {
    switch (s0)
    {
    case 0:
      x_new = XPROJECT(M_PI);
      y_new = YPROJECT(BigNumber);
      s_new = 1;
      break;

    case 1:
      x_new = XPROJECT(-M_PI);
      y_new = YPROJECT(BigNumber);
      s_new = 2;
      break;

    case 2:
      x_new = XPROJECT(-M_PI);
      y_new = YPROJECT(-BigNumber);
      s_new = 3;
      break;

    case 3:
      x_new = XPROJECT(M_PI);
      y_new = YPROJECT(-BigNumber);
      s_new = 0;
      break;

    default:
      /* keep lint happy */
      x_new = 0;
      y_new = 0;
      s_new = 0;
      assert(0);
    }

    scan(msi,mso,x_0, y_0, x_new, y_new);
    x_0 = x_new;
    y_0 = y_new;
    s0 = s_new;
  }

  scan(msi,mso,x_0, y_0, x_1, y_1);
}


void xing_error(msi,file, line, idx, nxings, xings)
     t_map_scan_input *msi;
     const char *file;
     int         line;
     int         idx;
     int         nxings;
     EdgeXing   *xings;
{
  fflush(stdout);
  fprintf(stderr, "xearth %s: incorrect edgexing sequence (%s:%d)\n",
          VersionString, file, line);
  fprintf(stderr, " (cidx %d) (view_lat %.16f) (view_lon %.16f)\n",
          xings[idx].cidx, msi->view_lat, msi->view_lon);
  fprintf(stderr, "\n");
  exit(1);
}


void scan(msi,mso,x_0, y_0, x_1, y_1)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     double x_0, y_0;
     double x_1, y_1;
{
  int    i;
  int    lo_y, hi_y;
  double x_value;
  double x_delta;
  t_status status;

  if (y_0 < y_1)
  {
    lo_y = ceil(y_0 - 0.5);
    hi_y = floor(y_1 - 0.5);

    if (hi_y == (y_1 - 0.5))
      hi_y -= 1;
  }
  else
  {
    lo_y = ceil(y_1 - 0.5);
    hi_y = floor(y_0 - 0.5);

    if (hi_y == (y_0 - 0.5))
      hi_y -= 1;
  }

  if (lo_y < 0)     lo_y = 0;
  if (hi_y >= msi->hght) hi_y = msi->hght-1;

  if (lo_y > hi_y)
    return;                     /* no scan lines crossed */

  if (lo_y < mso->min_y) mso->min_y = lo_y;
  if (hi_y > mso->max_y) mso->max_y = hi_y;

  x_delta = (x_1 - x_0) / (y_1 - y_0);
  x_value = x_0 + x_delta * ((lo_y + 0.5) - y_0);

  for (i=lo_y; i<=hi_y; i++)
  {
    double	*v;

    if ((v = XearthArrNext(mso->scanbuf[i],
		      &status)) == NULL)
	{
	  XearthWarning("arr_next");
	  exit(1);
	}
    *v = x_value;
    x_value += x_delta;
  }
}


void get_scanbits(msi,mso,val)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     int val;
{
  int      i, j;
  int      lo_x, hi_x;
  unsigned nvals;
  double  *vals;
  ScanBit *scanbit;
  t_status status;

  for (i=mso->min_y; i<=mso->max_y; i++)
  {
    vals  = (double *) mso->scanbuf[i]->elts;
    nvals = ARR_COUNT(mso->scanbuf[i]);
    assert((nvals % 2) == 0);
    qsort(vals, nvals, sizeof(double), double_comp);

    for (j=0; j<nvals; j+=2)
    {
      lo_x = ceil(vals[j] - 0.5);
      hi_x = floor(vals[j+1] - 0.5);

      if (lo_x < 0)     lo_x = 0;
      if (hi_x >= msi->wdth) hi_x = msi->wdth-1;

      if (lo_x <= hi_x)
      {
	if ((scanbit = XearthArrNext(mso->scanbits,
				&status)) == NULL)
	  {
	    XearthWarning("arr_next");
	    exit(1);
	  }
	scanbit->y    = i;
        scanbit->lo_x = lo_x;
        scanbit->hi_x = hi_x;
        scanbit->val  = val;
      }
    }

    ARR_EMPTY(mso->scanbuf[i]);
  }
}
