/* $Id: XCC.c,v 1.3 1995/01/31 23:03:10 cwikla Exp $ */
/*
 * Copyright 1994,1995 John L. Cwikla
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appears in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of John L. Cwikla or
 * Wolfram Research, Inc not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  John L. Cwikla and Wolfram Research, Inc make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 * John L. Cwikla and Wolfram Research, Inc disclaim all warranties with
 * regard to this software, including all implied warranties of
 * merchantability and fitness, in no event shall John L. Cwikla or
 * Wolfram Research, Inc be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of
 * use, data or profits, whether in an action of contract, negligence or
 * other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * Author:
 *  John L. Cwikla
 *  X Programmer
 *  Wolfram Research Inc.
 *
 *  cwikla@wri.com
*/

#include <X11/Xlib.h>
#include <X11/Xlibint.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "XCCP.h"

#define NUMBER(a) ((int)(sizeof(a)/sizeof(a[0])))

/*
** Find a standard colormap from a property, and make sure the visual matches
** the one we are using!
*/

#if NeedFunctionProtoTypes
static int _findGoodCube(XCC _xcc, Atom _atom, XStandardColormap *_matchedCube)
#else
static int _findGoodCube(_xcc, _atom, _matchedCube)
XCC _xcc;
Atom _atom;
XStandardColormap *_matchedCube;
#endif /* NeedFunctionProtoTypes */
{
	XStandardColormap *cubes, *match;
	int status;
	int count;
	int i;

	if (!_atom)
		return 0;

	status = XGetRGBColormaps(_xcc->dpy, RootWindow(_xcc->dpy, DefaultScreen(_xcc->dpy)),
				&cubes, &count, _atom);

	match = NULL;
	if (status)
	{
		status = 0;
		for(i=0;(match == NULL) && (i<count);i++)
		{
			if (cubes[i].visualid == _xcc->visualInfo->visualid)
			{
				match = cubes+i;
				status = 1;
			}
		}
	}

	if (match)
		*_matchedCube = *match;

	if (cubes)
		XFree((char *)cubes);

	return status;
}

/*
** Find a standard cmap if it exists.
*/

Atom ColorAtomList[] =
{
	XA_RGB_DEFAULT_MAP,
	XA_RGB_BEST_MAP,
	XA_RGB_GRAY_MAP,
};
Atom GrayAtomList[] =
{
	XA_RGB_GRAY_MAP,
	XA_RGB_DEFAULT_MAP,
};

#if NeedFunctionProtoTypes
static int _lookForStdCmap(XCC _xcc, Atom _atom)
#else
static int _lookForStdCmap(_xcc, _atom)
XCC _xcc;
Atom _atom;
#endif /* NeedFunctionProtoTypes */
{
	int status;
	int i;

	status = 0;

	if ((status = _findGoodCube(_xcc, _atom, &_xcc->stdCmap)) == 0)
		switch(_xcc->visualInfo->class)
		{
			case TrueColor: /* HMMM? */
			case StaticColor:
			case PseudoColor:
			case DirectColor:

				for(i=0;i<NUMBER(ColorAtomList);i++)
					if ((status = _findGoodCube(_xcc, ColorAtomList[i], &_xcc->stdCmap)) != 0)
						break;

				break;

			case StaticGray:
			case GrayScale:
		
				for(i=0;i<NUMBER(GrayAtomList);i++)
					if ((status = _findGoodCube(_xcc, GrayAtomList[i], &_xcc->stdCmap)) != 0)
						break;
				break;
		}

	if (!status)
		return 0;


	_xcc->mode = MODE_STDCMAP;
	_xcc->numColors = _xcc->stdCmap.red_max * _xcc->stdCmap.red_mult + 
					 _xcc->stdCmap.green_max * _xcc->stdCmap.green_mult +
					 _xcc->stdCmap.blue_max * _xcc->stdCmap.blue_mult;

	return status;
}

/*
** If we die, we go to the world of B+W
*/

#if NeedFunctionProtoTypes
static void _initBW(XCC _xcc)
#else
static void _initBW(_xcc)
XCC _xcc;
#endif /* NeedFuntionProtoTypes */
{
	XColor color;

	_xcc->mode = MODE_BW;

	color.red = color.blue = color.green = 0;
	if (!XAllocColor(_xcc->dpy, _xcc->colormap, &color))
		_xcc->blackPixel = 0;
	else
		_xcc->blackPixel = color.pixel;

	color.red = color.blue = color.green = 0xFFFF;
	if (!XAllocColor(_xcc->dpy, _xcc->colormap, &color))
		_xcc->whitePixel = _xcc->blackPixel ? 0 : 1;
	else
		_xcc->whitePixel = color.pixel;

	_xcc->numColors = 2;
}

/*
** Make our life easier and ramp our grays. Note
** that each lookup is /= 2 of the previous.
*/

#if NeedFunctionProtoTypes
static void _initGray(XCC _xcc)
#else
static void _initGray(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	XColor *clrs, *cstart;
	register int i;
	double dinc;

	_xcc->numColors = _xcc->visual->map_entries;

	_xcc->CLUT = (unsigned long *)Xmalloc(sizeof(unsigned long) * _xcc->numColors);
	cstart = (XColor *)Xmalloc(sizeof(XColor) * _xcc->numColors);

retryGray:

	dinc = 65535.0/_xcc->numColors;

	clrs = cstart;
	for(i=0;i<_xcc->numColors;i++)
	{
		clrs->red = clrs->blue = clrs->green = dinc * i;
		if (!XAllocColor(_xcc->dpy, _xcc->colormap, clrs))
		{
			XFreeColors(_xcc->dpy, _xcc->colormap, _xcc->CLUT, i, 0);

			_xcc->numColors /= 2;

			if (_xcc->numColors > 1)
				goto retryGray;
			else
			{
				XFree((char *)_xcc->CLUT);
				_xcc->CLUT = NULL;
				_initBW(_xcc);
				XFree((char *)cstart);
				return;
			}
		}
		_xcc->CLUT[i] = clrs++->pixel;
	}

	XFree((char *)cstart);

	_xcc->stdCmap.colormap = _xcc->colormap;
	_xcc->stdCmap.base_pixel = 0;
	_xcc->stdCmap.red_max = _xcc->visual->map_entries-1;
	_xcc->stdCmap.green_max = 0;
	_xcc->stdCmap.blue_max = 0;
	_xcc->stdCmap.red_mult = 1;
	_xcc->stdCmap.green_mult = _xcc->stdCmap.blue_mult = 0;

	_xcc->mode = MODE_MY_GRAY;
}

/*
** These are the color visuals that require a CLUT
*/

#if NeedFunctionProtoTypes
static void _initColor(XCC _xcc)
#else
static void _initColor(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	XColor *clrs, *cstart;
	register int red, green, blue;
	double dinc;
	int cubeval;
	int count;

	cubeval = 1;
	while((cubeval*cubeval*cubeval) < _xcc->visual->map_entries)
		cubeval++;
	cubeval--;
	_xcc->numColors = cubeval * cubeval * cubeval;

	_xcc->CLUT = (unsigned long *)Xmalloc(sizeof(unsigned long) * _xcc->numColors);
	cstart = (XColor *)Xmalloc(sizeof(XColor) * _xcc->numColors);

retryColor:

	_xcc->stdCmap.red_max = cubeval - 1;
	_xcc->stdCmap.green_max = cubeval - 1;
	_xcc->stdCmap.blue_max = cubeval - 1;
	_xcc->stdCmap.red_mult = cubeval * cubeval;
	_xcc->stdCmap.green_mult = cubeval;
	_xcc->stdCmap.blue_mult = 1;
	_xcc->stdCmap.base_pixel = 0;

	clrs = cstart;
	count = 0;
	for (red = 0; red <= _xcc->stdCmap.red_max; red++) 
	{
		for (green = 0; green <= _xcc->stdCmap.green_max; green++) 
		{
			for(blue = 0; blue <= _xcc->stdCmap.blue_max; blue++)
			{
				clrs->red = 0xFFFF * red / _xcc->stdCmap.red_max;
				clrs->green = 0xFFFF * green / _xcc->stdCmap.green_max;
				clrs->blue = 0xFFFF * blue / _xcc->stdCmap.blue_max;

				if (!XAllocColor(_xcc->dpy, _xcc->colormap, clrs))
				{
					XFreeColors(_xcc->dpy, _xcc->colormap, _xcc->CLUT, count, 0);

					cubeval--;

					if (cubeval > 1)
						goto retryColor;
					else
					{
						XFree((char *)_xcc->CLUT);
						_xcc->CLUT = NULL;
						_initBW(_xcc);
						XFree((char *)cstart);
						return;
					}

				}
				_xcc->CLUT[count++] = clrs++->pixel;
			}
		}
	}

	_xcc->numColors = count;

	XFree((char *)cstart);

	_xcc->mode = MODE_STDCMAP;
}

/*
** Get our shifts and masks
*/

#if NeedFunctionProtoTypes
static void _initTrueColor(XCC _xcc)
#else
static void _initTrueColor(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	register unsigned long rmask, gmask, bmask;

	_xcc->mode = MODE_TRUE;

	rmask = _xcc->masks.red = _xcc->visualInfo->red_mask;
	_xcc->shifts.red = 0;
	_xcc->bits.red = 0;
	while (!(rmask & 1))
	{
		rmask >>= 1;
		_xcc->shifts.red++;
	}
	while((rmask & 1))
	{
		rmask >>= 1;
		_xcc->bits.red++;
	}

	gmask = _xcc->masks.green = _xcc->visualInfo->green_mask;
	_xcc->shifts.green = 0;
	_xcc->bits.green = 0;
	while (!(gmask & 1))
	{
		gmask >>= 1;
		_xcc->shifts.green++;
	}
	while(gmask & 1)
	{
		gmask >>= 1;
		_xcc->bits.green++;
	}

	bmask = _xcc->masks.blue = _xcc->visualInfo->blue_mask;
	_xcc->shifts.blue = 0;
	_xcc->bits.blue = 0;
	while (!(bmask & 1))
	{
		bmask >>= 1;
		_xcc->shifts.blue++;
	}
	while(bmask & 1)
	{
		bmask >>= 1;
		_xcc->bits.blue++;
	}

	_xcc->numColors = ((_xcc->visualInfo->red_mask) | 
						(_xcc->visualInfo->green_mask) | 
						(_xcc->visualInfo->blue_mask)) + 1;
}

/*
** Cheat here! Make the direct color visual work like
** a true color! I'm pretty sure this is right!
*/

#if NeedFunctionProtoTypes
static void _initDirectColor(XCC _xcc)
#else
static void _initDirectColor(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	int n, count;
	XColor *clrs, *cstart;
	unsigned long rval, bval, gval, maxval;
	unsigned long *rtable;
	unsigned long *gtable;
	unsigned long *btable;
	double dinc;

	_initTrueColor(_xcc); /* for shift stuff */

	rval = _xcc->visualInfo->red_mask;
	gval = _xcc->visualInfo->green_mask;
	bval = _xcc->visualInfo->blue_mask;

	while(!(rval & 1))
		rval >>= 1;
	while(!(gval & 1))
		gval >>= 1;
	while(!(bval & 1))
		bval >>= 1;

	rtable = (unsigned long *)Xmalloc(sizeof(unsigned long) * (rval+1));
	gtable = (unsigned long *)Xmalloc(sizeof(unsigned long) * (gval+1)); 
	btable = (unsigned long *)Xmalloc(sizeof(unsigned long) * (bval+1));

	maxval = (rval > gval) ? rval : gval;
	maxval = (maxval > bval) ? maxval : bval;

	cstart = (XColor *)Xmalloc(sizeof(XColor) * (maxval+1));
	_xcc->CLUT = (unsigned long *)Xmalloc(sizeof(unsigned long) * (maxval+1));

retrydirect:

	for(n=0;n<=rval;n++)
		rtable[n] = 65535.0/rval * n;
	for(n=0;n<=gval;n++)
		gtable[n] = 65535.0/gval * n;
	for(n=0;n<=bval;n++)
		btable[n] = 65535.0/bval * n;

	maxval = (rval > gval) ? rval : gval;
	maxval = (maxval > bval) ? maxval : bval;

	count = 0;
	clrs = cstart;
	_xcc->numColors = (bval + 1) * (gval + 1) * (rval + 1);
	for(n=0;n<=maxval;n++)
	{
		dinc = (double)n/(double)maxval;
		clrs->red = rtable[(int)(dinc * rval)];
		clrs->green = gtable[(int)(dinc * gval)];
		clrs->blue = btable[(int)(dinc * bval)];
		if (XAllocColor(_xcc->dpy, _xcc->colormap, clrs))
		{
			_xcc->CLUT[count++] = clrs->pixel;
			clrs++;
		}
		else
		{
			XFreeColors(_xcc->dpy, _xcc->colormap, _xcc->CLUT, count, 0);

			bval >>= 1;
			gval >>= 1;
			rval >>= 1;

			_xcc->masks.red = (_xcc->masks.red >> 1) & _xcc->visualInfo->red_mask;
			_xcc->masks.green = (_xcc->masks.green >> 1) & _xcc->visualInfo->green_mask;
			_xcc->masks.blue = (_xcc->masks.green >> 1) & _xcc->visualInfo->blue_mask;
			
			_xcc->shifts.red++;
			_xcc->shifts.green++;
			_xcc->shifts.blue++;

			_xcc->bits.red--;
			_xcc->bits.green--;
			_xcc->bits.blue--;

			 _xcc->numColors = (bval + 1) * (gval + 1) * (rval + 1);

			if (_xcc->numColors > 1)
				goto retrydirect;
			else
			{
				XFree((char *)_xcc->CLUT);
				_xcc->CLUT = NULL;
				_initBW(_xcc);
				break;
			}
		}
	}

	XFree(rtable);
	XFree(gtable);
	XFree(btable);
	XFree(cstart);
}

#if NeedFunctionProtoTypes
XCC XCCCreate(Display *_dpy, Visual *_visual, int _usePrivateColormap, int _useStdCmaps, 
							Atom _stdCmapAtom, Colormap *_colormap)
#else
XCC XCCCreate(_dpy, _visual, _usePrivateColormap, _useStdCmaps, _stdCmapAtom, _colormap)
Display *_dpy;
Visual *_visual;
int _usePrivateColormap;
int _useStdCmaps;
Atom _stdCmapAtom;
Colormap *_colormap;
#endif /* NeedFunctionProtoTypes */
{
	XCC cc;
	int n;
	XVisualInfo visInfo;
	int status;
	int retryCount;

	cc = (XCC)Xmalloc(sizeof(struct _XColorContext));

	if (cc == NULL)
		return NULL;

	cc->dpy = _dpy;
	cc->visual = _visual;
	cc->colormap = DefaultColormap(_dpy, DefaultScreen(_dpy));
	cc->CLUT = NULL;
	cc->mode = MODE_UNDEFINED;
	cc->needToFreeColormap = 0;

	visInfo.visualid = XVisualIDFromVisual(_visual);
	cc->visualInfo = XGetVisualInfo(_dpy, VisualIDMask, &visInfo, &n);

	status = 0;
	if (!_usePrivateColormap && _useStdCmaps)
		status = _lookForStdCmap(cc, _stdCmapAtom);

	retryCount = 0;
	while(retryCount < 2)
	{
		if (!status)
		{
			if (_usePrivateColormap || (cc->visual != DefaultVisual(_dpy, DefaultScreen(_dpy))))
			{
				cc->colormap = XCreateColormap(_dpy, RootWindow(_dpy, DefaultScreen(_dpy)), 
								cc->visual, AllocNone);
				cc->needToFreeColormap = (cc->colormap != DefaultColormap(_dpy, DefaultScreen(_dpy)));
			
			}
			switch(_visual->class)
			{
				case StaticGray:
				case GrayScale:
					_initGray(cc);
					break;

				case TrueColor: /* shifts */
					_initTrueColor(cc);
					break;
			
				case DirectColor: /* shifts & fake CLUT */
					_initDirectColor(cc);
					break;

				case StaticColor:
				case PseudoColor:
					_initColor(cc);
					break;
			}
		}
		else
			cc->colormap = cc->stdCmap.colormap;

		if ( (cc->mode == MODE_BW) && (cc->visualInfo->depth > 1))
		{
			_usePrivateColormap = 1;
			retryCount++;
		}
		else
			break;
	}

	*_colormap = cc->colormap;

	return cc;
}

/*
** This doesn't currently free black/white. Hmm...
*/

#if NeedFunctionProtoTypes
void  XCCFree(XCC _xcc)
#else
void XCCFree(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	if (_xcc == NULL)
		return;

	if (_xcc->CLUT != NULL)
	{
		XFreeColors(_xcc->dpy, _xcc->colormap, _xcc->CLUT, _xcc->numColors, 0);
		XFree((char *)_xcc->CLUT);
	}

	if (_xcc->needToFreeColormap)
		XFreeColormap(_xcc->dpy, _xcc->colormap);

	XFree((char *)_xcc);
}

#if NeedFunctionProtoTypes
unsigned long XCCGetPixel(XCC _xcc, unsigned int _red, unsigned int _green, unsigned int _blue)
#else
unsigned long XCCGetPixel(_xcc, _red, _green, _blue)
XCC _xcc;
unsigned int _red;
unsigned int _green;
unsigned int _blue;
#endif /* NeedFunctionProtoTypes */
{
	unsigned long ired, igreen, iblue;
	XColor color;

	switch(_xcc->mode)
	{
		case MODE_BW:
		{
			double value;

			value = (double)_red/65535.0 * 0.3 + (double)_green/65535.0 * 0.59 + (double)_blue/65535.0 * 0.11;
			if (value > 0.5)
				return _xcc->whitePixel;
			else
				return _xcc->blackPixel;
		}
		case MODE_MY_GRAY:
				_red = _red * 0.3 + _green * 0.59 + _blue * 0.1;
				/* fall thru */

		case MODE_STDCMAP:
			if ((ired = _red * (_xcc->stdCmap.red_max + 1) / 0xFFFF)
				> _xcc->stdCmap.red_max)
				ired = _xcc->stdCmap.red_max;
			ired *= _xcc->stdCmap.red_mult;

			if ((igreen = _green * (_xcc->stdCmap.green_max + 1) / 0xFFFF)
				> _xcc->stdCmap.green_max)
				igreen = _xcc->stdCmap.green_max;
			igreen *= _xcc->stdCmap.green_mult;

			if ((iblue = _blue * (_xcc->stdCmap.blue_max + 1) / 0xFFFF)
				> _xcc->stdCmap.blue_max)
				iblue = _xcc->stdCmap.blue_max;
			iblue *= _xcc->stdCmap.blue_mult;

			if (_xcc->CLUT != NULL)
				return _xcc->CLUT[_xcc->stdCmap.base_pixel + ired + igreen + iblue];
			else
				return _xcc->stdCmap.base_pixel + ired + igreen + iblue;

		case MODE_TRUE:
			_red >>= 16 - _xcc->bits.red;
			_green >>= 16 - _xcc->bits.green;
			_blue >>= 16 - _xcc->bits.blue;

			ired = _red << _xcc->shifts.red;

			igreen = _green << _xcc->shifts.green;

			iblue = _blue << _xcc->shifts.blue;

			return (ired & _xcc->masks.red) +
					(igreen & _xcc->masks.green) +
					(iblue & _xcc->masks.blue);

		default:
			color.red = _red;
			color.green = _green;
			color.blue = _blue;

			if (!XAllocColor(_xcc->dpy, _xcc->colormap, &color))
				return 0;
			else
				return color.pixel;
	}
}

#if NeedFunctionProtoTypes
int XCCGetNumColors(XCC _xcc)
#else
int XCCGetNumColors(_xcc)
XCC _xcc;
#endif /* NeedFunctionProtoTypes */
{
	return _xcc->numColors;
}
