# version 1.1 (9 Feb '95)
# (Motif version of XfwfTabs)

@class XmTabs (XmPrimitive)  @file=XmTabs

@ The |XmTabs| widget displays a series of tabs, similar to the
alphabet tabs along the top of index cards. One tab, the front one, is
completely visible, the others are partially hidden behind it. Each of
the tabs can be clicked on with the mouse.

@PUBLIC

@ |lefttabs| is the number of partially visible tabs to the left of
the main one. The main tab always occupies the same amount of space,
but the tabs to the left become narrower when |lefttabs| is increased.

	@var int lefttabs = 0

@ The number of tabs to the right of the main one. The higher the
number, the smaller the tabs will be.

	@var int righttabs = 0

@ The labels on each of the tabs must be simple strings. There is no
support for multiline labels or different fonts in this version of
|XfwfTabs|. The array |labels| should contain no less than |lefttabs +
righttabs + 1| strings. The widget makes private copies of the array
and of the strings.

The leftmost tab displays |labels[0]|. Note that the labels may be too
large to fit in the tab. In that case they are clipped. The left tabs
show only the initial part of the labels, the right tabs show the last
part of their labels and the main (front) tab shows the middle part of
its label.

	@var StringArray labels = NULL

@ The color of the left and right tabs. The main (front) tab is drawn
in the |background| color. Unless the |tabcolor| has been specified
explicitly, the left and right tabs have the same color as the main
tab.

	@var Pixel tabcolor = <CallProc> copy_bg

@ The text is drawn in the font which is given as the |font| resource.

	@var <FontStruct> XFontStruct *font = <String> XtDefaultFont

@ When the user clicks on a tab, a callback function is called. The
|call_data| argument to the callback will be a relative number: 0 if
the user clicked on the main tab, -1 if he clicked on the tab
immediately to the left, +1 if he clicked on the first right tab, and
higher numbers for tabs that are further removed from the main one.

	@var <Callback> XtCallbackList activateCallback = NULL

@ Labels in the left and right tabs are drawn left resp. right
justified. The |textmargin| resource gives the number of pixels
between the edge of the tab and the label.

	@var int textmargin = 3

@PRIVATE

@ A GC for drawing the labels.

	@var GC textgc

@ A GC for drawing the top shadow lines.

	@var GC topgc

@ A GC for drawing the top shadow lines.

	@var GC bottomgc

@ A GC for filing the main tab.

	@var GC fillgc

@CLASSVARS

@ Set a few class variables.

	@var compress_motion = True
	@var compress_exposure = XtExposeCompressMultiple
	@var compress_enterleave = True

@METHODS

@proc class_initialize
{
    XtSetTypeConverter(XtRString, "StringArray", cvtStringToStringArray,
		       NULL, 0, XtCacheNone, NULL);
}

@ The |initialize| method checks the resources and creates the local
variables.

For the time being, there is no keyboard traversal.

@proc initialize
{
    String *h;
    int i;

    $traversal_on = FALSE;
    $topgc = NULL;
    create_topgc($);
    $bottomgc = NULL;
    create_bottomgc($);
    $textgc = NULL;
    create_textgc($);
    $fillgc = NULL;
    create_fillgc($);
    if ($labels) {
	h = (String*) XtMalloc(($lefttabs + $righttabs + 1) * sizeof(*h));
	for (i = $lefttabs + $righttabs; i >= 0; i--)
	    h[i] = XtNewString($labels[i]);
	$labels = h;
    }
}

@ The |set_values| method checks changed resources and recomputes some
local variables. In this case the GC's may have to be changed.

@proc set_values
{
    Boolean redraw = False;
    String *h;
    int i;

    if ($background_pixel != $old$background_pixel
	|| $background_pixmap != $old$background_pixmap
	|| $shadow_thickness != $old$shadow_thickness) {
	create_topgc($);
	create_bottomgc($);
    }
    if ($foreground != $old$foreground || $font != $old$font) {
	create_textgc($);
	redraw = True;
    }
    if ($textmargin != $old$textmargin
	|| $highlight_thickness != $old$highlight_thickness) {
	redraw = True;
    }
    if ($tabcolor != $old$tabcolor) {
	create_fillgc($);
	redraw = True;
    }
    if ($labels != $old$labels) {
	if ($labels) {
	    h = (String*) XtMalloc(($lefttabs + $righttabs + 1) * sizeof(*h));
	    for (i = $lefttabs + $righttabs; i >= 0; i--) {
		h[i] = XtNewString($labels[i]);
	    }
	    $labels = h;
	}
	if ($old$labels) {
	    for (i = $old$lefttabs + $old$righttabs; i >= 0; i--)
		XtFree($old$labels[i]);
	    XtFree((XtPointer) $old$labels);
	}
	redraw = True;
    }
    if ($lefttabs != $old$lefttabs || $righttabs != $old$righttabs)
	redraw = True;

    return redraw;
}

@ When the widget is realized or resized, a new shape mask is
installed. The shape mask makes the corners transparent.

@proc realize
{
    #realize($, mask, attributes);
    set_shape($);
}

@proc resize
{
    #resize($);
    set_shape($);
}

@ The |expose| method draws the tabs. First the tabs to the left, then
the tabs to the right, and finally the main one. Three private
routines do most of the work. The inherited |expose| method is called
to draw the frame, if any, although the tabs should look best without
a frame.

@proc expose
{
    int i;
    Dimension width, height;
    Position x, y;

    if (! XtIsRealized($)) return;

    for (i = 0; i < $lefttabs; i++)
	draw_left_tab($, region, i);
    for (i = $lefttabs + $righttabs; i > $lefttabs; i--)
	draw_right_tab($, region, i);
    draw_main_tab($, region);
    /* Focus highlight? */
}

@ TO DO: There should be some internal keyboard traversal, and also
new |border_highlight| and |border_unhighlight| methods.

@proc border_highlight
{
}

@proc border_unhighlight
{
}

@UTILITIES

@ The |draw_left_tab| routine draws a narrow tab to the left of the
main one. The |poly| variable holds the corner points of the area
occupied by this tab. The label is written inside this area,
left-aligned.

@proc draw_left_tab_old($, Region region, int i)
{
    Region clip;
    XPoint poly[7];
    Dimension w, delta, shad, h;
    Position x1, y1;
    Display *dpy = XtDisplay($);
    int j = $lefttabs - i + 1;
    h = $highlight_thickness;
    w = ($width - 2 * h)/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */
    shad = $shadow_thickness/2;
    x1 = h + i * w;

    /*
     *     5 o--------o 6
     *      /
     *   4 o
     *    /
     * 3 o        o 0
     *   |        |
     *   |        |
     *   |        |
     * 2 o--------o 1
     */
    poly[0].x = x1 + w;			poly[0].y = h + 4 + j * shad;
    poly[1].x = x1 + w;			poly[1].y = $height - 1 - shad;
    poly[2].x = x1 + shad;		poly[2].y = $height - 1 - shad;
    poly[3].x = x1 + shad;		poly[3].y = h + 4 + j * shad;
    poly[4].x = x1 + 1 + shad;		poly[4].y = h + 1 + j * shad;
    poly[5].x = x1 + 4 + shad;		poly[5].y = h + j * shad;
    poly[6].x = x1 + w + 4 + shad;	poly[6].y = h + j * shad;
	
    clip = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(dpy, $textgc, clip);

    XFillPolygon(dpy, XtWindow($), $fillgc, poly, XtNumber(poly),
		 Convex, CoordModeOrigin);

    if ($labels) {
	y1 = ($height - h - j * shad - $font->ascent - $font->descent)/2
	    + h + j * shad;
	XDrawString(dpy, XtWindow($), $textgc, x1 + shad + $textmargin,
		    y1 + $font->ascent, $labels[i], strlen($labels[i]));
    }
    XDrawLines(dpy, XtWindow($), $topgc, poly + 2, 5, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_left_tab($, Region region, int i)
{
    Region clip;
    XPoint poly[4];
    Dimension w, delta, shad;
    Position x1, y1, x2;
    Display *dpy = XtDisplay($);

    w = $width/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */
    shad = $shadow_thickness/2;
    x1 = i * w;

    /*
     *     2 o--------o 3
     *      /
     *     /
     *    /
     * 1 o--------o 0
     */
    poly[0].x = x1 + w;			poly[0].y = $height - shad;
    poly[1].x = x1 + shad;		poly[1].y = $height - shad;
    poly[2].x = x1 + shad + delta;	poly[2].y = shad;
    poly[3].x = x1 + w + shad + delta;	poly[3].y = shad;
	
    clip = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(dpy, $textgc, clip);

    XFillPolygon(dpy, XtWindow($), $fillgc, poly, XtNumber(poly),
		 Convex, CoordModeOrigin);

    if ($labels) {
	y1 = ($height - $font->ascent - $font->descent)/2;
	x2 = x1 + shad + $textmargin
	    + (y1 + $font->ascent + $font->descent)/2;
	XDrawString(dpy, XtWindow($), $textgc, x2,
		    y1 + $font->ascent, $labels[i], strlen($labels[i]));
    }
    XDrawLines(dpy, XtWindow($), $topgc, poly + 1, 3, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_right_tab_old($, Region region, int i)
{
    Region clip;
    XPoint poly[7];
    Dimension w, delta, textw, len, shad, h;
    Position x1, y1;
    Display *dpy = XtDisplay($);
    int j = i - $lefttabs + 1;

    h = $highlight_thickness;
    w = ($width - 2 * h)/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */
    shad = $shadow_thickness/2;

    /*
     * 6 o--------o 5
     *             \
     *              o 4
     *               \
     *     0 o        o 3
     *       |        |
     *       |        |
     *       |        |
     *     1 o--------o 2
     */
    x1 = $width - h - w * ($lefttabs + $righttabs - i);
    poly[0].x = x1 - w;			poly[0].y = 4 + j * shad;
    poly[1].x = x1 - w;			poly[1].y = $height - 1 - shad;
    poly[2].x = x1 - shad;		poly[2].y = $height - 1 - shad;
    poly[3].x = x1 - shad;		poly[3].y = h + 4 + j * shad;
    poly[4].x = x1 - 1 - shad;		poly[4].y = h + 1 + j * shad;
    poly[5].x = x1 - 4 - shad;		poly[5].y = h + 0 + j * shad;
    poly[6].x = x1 - w - 4 - shad;	poly[6].y = h + 0 + j * shad;

    clip = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    if (region) XIntersectRegion(region, clip, clip);
    XSetRegion(dpy, $textgc, clip);

    XFillPolygon(dpy, XtWindow($), $fillgc, poly, XtNumber(poly),
		 Convex, CoordModeOrigin);

    if ($labels) {
	len = strlen($labels[i]);
	textw = XTextWidth($font, $labels[i], len);
	y1 = ($height - h - j * shad - $font->ascent - $font->descent)/2
	    + h + j * shad;
	XDrawString(dpy, XtWindow($), $textgc, x1 - shad - $textmargin - textw,
		    y1 + $font->ascent, $labels[i], len);
    }
    XDrawLines(dpy, XtWindow($), $bottomgc, poly + 2, 4, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $topgc, poly + 5, 2, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_right_tab($, Region region, int i)
{
    Region clip;
    XPoint poly[4];
    Dimension w, delta, textw, len, shad;
    Position x1, y1, x2;
    Display *dpy = XtDisplay($);

    w = $width/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */
    shad = $shadow_thickness/2;

    /*
     * 3 o--------o 2
     *             \
     *              \
     *               \
     *     0 o--------o 1
     */
    x1 = $width - w * ($lefttabs + $righttabs - i);
    poly[0].x = x1 - w;			poly[0].y = $height - shad;
    poly[1].x = x1 - shad;		poly[1].y = $height - shad;
    poly[2].x = x1 - shad - delta;	poly[2].y = shad;
    poly[3].x = x1 - w - shad - delta;	poly[3].y = shad;

    clip = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    if (region) XIntersectRegion(region, clip, clip);
    XSetRegion(dpy, $textgc, clip);

    XFillPolygon(dpy, XtWindow($), $fillgc, poly, XtNumber(poly),
		 Convex, CoordModeOrigin);

    if ($labels) {
	len = strlen($labels[i]);
	textw = XTextWidth($font, $labels[i], len);
	y1 = ($height - $font->ascent - $font->descent)/2;
	x2 = x1 - shad - $textmargin - textw
	    - (y1 + $font->ascent + $font->descent)/2;
	XDrawString(dpy, XtWindow($), $textgc, x2,
		    y1 + $font->ascent, $labels[i], len);
    }
    XDrawLines(dpy, XtWindow($), $bottomgc, poly + 1, 2, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $topgc, poly + 2, 2, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_main_tab_old($, Region region)
{
    Dimension textw, w1, w2, delta, len, shad, h;
    Display *dpy = XtDisplay($);
    Position x1, y1;
    Region clip;
    XPoint poly[10];

    h = $highlight_thickness;
    delta = $height/2;
    w1 = $lefttabs == 0 ? 0
	: $lefttabs * (($width - 2 * h)/2/($lefttabs + $righttabs));
    w1 += h;
    w2 = $righttabs == 0 ? 0
	: $righttabs * (($width - 2 * h)/2/($lefttabs +$righttabs));
    w2 += h;
    shad = $shadow_thickness/2;

    /*
     *           4 o-------------o 5
     *            /               \
     *         3 o                 o 6
     *          /                   \
     *       2 o                     o 7
     *         |                     |
     *         |                     |
     *         |                     |
     * 0 o-----o 1                 8 o------o 9
     */
    poly[0].x = h;			poly[0].y = $height - 1;
    poly[1].x = w1 + shad;		poly[1].y = $height - 1;
    poly[2].x = w1 + shad;		poly[2].y = h + 4 + shad;
    poly[3].x = w1 + 1 + shad;		poly[3].y = h + 1 + shad;
    poly[4].x = w1 + 4 + shad;		poly[4].y = h + 0 + shad;
    poly[5].x = $width - w2 - 4 - shad;	poly[5].y = h + 0 + shad;
    poly[6].x = $width - w2 - 1 - shad;	poly[6].y = h + 1 + shad;
    poly[7].x = $width - w2 - shad;	poly[7].y = h + 4 + shad;
    poly[8].x = $width - w2 - shad;	poly[8].y = $height - 1;
    poly[9].x = $width - h;		poly[9].y = $height - 1;

    clip = XPolygonRegion(poly + 1, 8, EvenOddRule);

    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(dpy, $textgc, clip);

    if ($labels) {
	len = strlen($labels[$lefttabs]);
	textw = XTextWidth($font, $labels[$lefttabs], len);
	x1 = (w1 + $width - w2)/2 - textw/2;
	y1 = ($height - h - shad - $font->ascent - $font->descent)/2
	    + h + shad + $font->ascent;
	XDrawString(dpy, XtWindow($), $textgc, x1, y1, $labels[$lefttabs],len);
    }
    XDrawLines(dpy, XtWindow($), $topgc, poly, 6, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $bottomgc, poly + 5, 4, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $topgc, poly + 8, 2, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_main_tab($, Region region)
{
    Dimension textw, w1, w2, delta, len, shad;
    Display *dpy = XtDisplay($);
    Position x1, y1;
    Region clip;
    XPoint poly[6];

    delta = $height/2;
    w1 = $lefttabs == 0 ? 0
	: $lefttabs * ($width/2/($lefttabs + $righttabs));
    w2 = $righttabs == 0 ? 0
	: $righttabs * ($width/2/($lefttabs +$righttabs));
    w2 = $width - w2;
    shad = $shadow_thickness/2;

    /*
     *           2 o-------------o 3
     *            /               \
     *           /                 \
     *          /                   \
     * 0 o-----o 1                 4 o------o 5
     */
    poly[0].x = 0;			poly[0].y = $height - 1 - shad;
    poly[1].x = w1 + shad;		poly[1].y = $height - 1 - shad;
    poly[2].x = w1 + shad + delta;	poly[2].y = shad;
    poly[3].x = w2 - shad - delta;	poly[3].y = shad;
    poly[4].x = w2 - shad;		poly[4].y = $height - 1 - shad;
    poly[5].x = $width;			poly[5].y = $height - 1 - shad;

    clip = XPolygonRegion(poly + 1, 4, EvenOddRule);
    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(dpy, $textgc, clip);

    if ($labels) {
	len = strlen($labels[$lefttabs]);
	textw = XTextWidth($font, $labels[$lefttabs], len);
	x1 = (w1 + w2 - textw)/2;
	y1 = ($height - $font->ascent - $font->descent)/2 + $font->ascent;
	XDrawString(dpy, XtWindow($), $textgc, x1, y1, $labels[$lefttabs],len);
    }
    XDrawLines(dpy, XtWindow($), $topgc, poly, 4, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $bottomgc, poly + 3, 2, CoordModeOrigin);
    XDrawLines(dpy, XtWindow($), $topgc, poly + 4, 2, CoordModeOrigin);
    XDestroyRegion(clip);
}

@ The GCs are not shared, because the clip masks are changed
sometimes.

The current shadows are only drawn with colors, not with pixmaps. This
has to be changed some day.

@proc create_topgc($)
{
    XtGCMask mask = GCForeground | GCLineWidth;
    XGCValues values;

    if ($topgc != NULL) XFreeGC(XtDisplay($), $topgc);
    values.foreground = $top_shadow_color;
    values.line_width = $shadow_thickness;
    $topgc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
		       mask, &values);
}

@proc create_bottomgc($)
{
    XtGCMask mask = GCForeground | GCLineWidth;
    XGCValues values;

    if ($bottomgc != NULL) XFreeGC(XtDisplay($), $bottomgc);
    values.foreground = $bottom_shadow_color;
    values.line_width = $shadow_thickness;
    $bottomgc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
			  mask, &values);
}

@proc create_textgc($)
{
    XtGCMask mask = GCForeground | GCFont;
    XGCValues values;

    if ($textgc != NULL) XFreeGC(XtDisplay($), $textgc);
    values.foreground = $foreground;
    values.font = $font->fid;
    $textgc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
			mask, &values);
}

@proc create_fillgc($)
{
    XtGCMask mask = GCForeground;
    XGCValues values;

    if ($fillgc != NULL) XFreeGC(XtDisplay($), $fillgc);
    values.foreground = $tabcolor;
    $fillgc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
			mask, &values);
}

@ |copy_bg| copies the |background| color to the |main_color|.

@proc copy_bg($, int offset, XrmValue* value)
{
    value->addr = (XtPointer) &$background_pixel;
}

@ |set_shape| is called from |realize| and from |resize|, to set the
shape of the window.

@proc set_shape_old($)
{
    Region region;
    Dimension delta;
    XPoint poly[8];

    if (! XtIsRealized($)) return;
    delta = $height/2;

    /*
     *      3 o----------------o 4
     *       /                  \
     *    2 o                    o 5
     *     /                      \
     *  1 o                        o 6
     *    |                        |
     *    |                        |
     *    |                        |
     *  0 o------------------------o 7
     */
    poly[0].x = 0;		poly[0].y = $height;
    poly[1].x = 0;		poly[1].y = 4;
    poly[2].x = 1;		poly[2].y = 1;
    poly[3].x = 4;		poly[3].y = 0;
    poly[4].x = $width - 4;	poly[4].y = 0;
    poly[5].x = $width - 1;	poly[5].y = 1;
    poly[6].x = $width;		poly[6].y = 4;
    poly[7].x = $width;		poly[7].y = $height;

    region = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    XShapeCombineRegion(XtDisplay($), XtWindow($), ShapeBounding,
			0, 0, region, ShapeSet);
    XDestroyRegion(region);
}

@proc set_shape($)
{
    Region region;
    Dimension delta;
    XPoint poly[4];

    if (! XtIsRealized($)) return;
    delta = $height/2;

    /*
     *      1 o----------------o 2
     *       /                  \
     *      /                    \
     *     /                      \
     *  0 o------------------------o 3
     */
    poly[0].x = 0;		poly[0].y = $height;
    poly[1].x = delta;		poly[1].y = 0;
    poly[2].x = $width - delta;	poly[2].y = 0;
    poly[3].x = $width;		poly[3].y = $height;

    region = XPolygonRegion(poly, XtNumber(poly), EvenOddRule);
    XShapeCombineRegion(XtDisplay($), XtWindow($), ShapeBounding,
			0, 0, region, ShapeSet);
    XDestroyRegion(region);
}

@TRANSLATIONS

@trans <Btn1Down>,<Btn1Up>: activate()

@ACTIONS

@proc activate
{
    Dimension w, delta, ww, w1, w2;
    int i;

    delta = ($height - event->xbutton.y)/2;
    w = $lefttabs + $righttabs == 0 ? 0 : $width/2/($lefttabs + $righttabs);
    w1 = $lefttabs == 0 ? 0 : $lefttabs * w;
    w2 = $righttabs == 0 ? 0 : $righttabs * w;

    if (event->xbutton.x < w1 + delta) {
	/* Click left of main tab */
	if ($lefttabs != 0) {
	    ww = w + delta;
	    for (i = -$lefttabs; event->xbutton.x >= ww; i++, ww += w) ;
	    XtCallCallbackList($, $activateCallback, (XtPointer) i);
	}
    } else if (event->xbutton.x < $width - w2 - delta) {
	/* Click on main tab */
	XtCallCallbackList($, $activateCallback, (XtPointer) 0);
    } else {
	/* Click right of main tab */
	if ($righttabs != 0) {
	    ww = $width - w - delta;
	    for (i = $righttabs; event->xbutton.x < ww; i--, ww -= w) ;
	    XtCallCallbackList($, $activateCallback, (XtPointer) i);
	}
    }
}

@IMPORTS

@incl <stdio.h>
@incl <X11/extensions/shape.h>
@incl <Xfwf/Converters.h>
