/*
** TextLayout
**	Layout character content lines
**
** Copyright 1992-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if SYSV_INCLUDES
#	include <memory.h>	/* memset() */
#endif
#if ANSI_INCLUDES
#	include <stdlib.h>
#endif
#include <string.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/TextP.h>
#include "TextImport.h"
#include "TextLayout.h"
#include "TextFont.h"
#include "FrameSupport.h"

#ifndef INT_MAX
#	define INT_MAX (~((unsigned)0) >> 1)
#endif

/*
** LAYOUT_CONTENT_TAG
**	is an internal and special tag value that is associated with the
**	snips that are generated by this layout process self. It is used
**	to differentiate between application generated layout and normal
**	layout content. Currently application generated layout content
**	gets NULL tag.
*/
#define LAYOUT_CONTENT_TAG (XeTextTag)(-1)

/*
**	This module has only one externally callable function
**	XeTextLayout, which lays out the content of
**	single widget. Layout process needs to keep up quite a
**	lot of state information. This state is kept in static
**	variables local to this module to lessen the need to
**	pass many between local functions here.
*/
typedef struct SequenceState
    {
	short maxAscent;	/* Max character backward extent */
	short maxDescent;	/* Max character forward extent */
	XeTextTag tag;		/* Current tag at this point */
	int wNow;		/* Total width up to this point */
	int spaces;		/* Number adjustable spaces *inside* line */
	int framed;		/* Non-zero, if framed state */
    } SequenceState;

/*
** ExpectEntry
**	stores the original Snip layout information if it was laid out
**	before this layout process. This is needed only because of the need
**	to generate good expose area and terminate layout process as
**	soon as possible after no changes start to happen to the line
**	boxes (This is a huge ugly hack of jugling x,y values in Snips
**	and I don't like it a bit.. --msa)
*/
typedef struct ExpectEntry
    {
	Snip *s;
	int xWidth;
	Position x, y;
	short ascent, descent, offset, column;
    } ExpectEntry;

/*
** FontCacheInfo
**	Pack Weight, Italic and size information into single keyword.
*/
#define FontCacheInfo(m,size) \
	(((m.bits & Weight_MASK) != 0) | \
	 (((m.bits & Italicized_MASK) != 0) << 1) | \
	 (size << 2))

typedef struct FontCacheEntry
    {
	unsigned long fontkey;	/* Posture, weight, size (FontCacheInfo) */
	XeFontList fontlist;	/* Font List */
	char *charset;		/* Character Set Id */
	XFontStruct *font;	/* Associated font */
    } FontCacheEntry;

typedef struct FontCache
    {
	struct FontCache *next;	/* Link to the next font cache */
	Display *display;	/* Display for which cache is computed. */
	int cache_hit;
	int cache_miss;
	int cache_full;
	int cache_reset;	/* Non-zero, if cache should be cleared */
	FontCacheEntry data[211]; /* Size should be a *PRIME* */
    } FontCache;

static FontCache *font_cache_list;

/*
** LayoutContext
**	collects all of the layout state variables into single structure
**	which will be dynamically allocated for each layout process (cannot
**	use global variables because it can happen that the layout process
**	is called for another widget in the middle of layout -- for example,
**	in some callback function).
*/
typedef struct LayoutContext
    {
	XeTextWidget w;		/* Back Link to the Widget */
	FontCache *font_cache;	/* Font cache being used */
	int page;		/* Current Page 0... */
	int column;		/* Current Column 0... */
	int column_x, column_y;	/* Column origin point */
	int offset_x, offset_y;	/* Available area offsets */
	int width, height;	/* Available area width and height */
	int column_end;		/* Column end y-coordinate */
	int lineLength;		/* Current available line length */
	int lineOffset;		/* Offset of from the line Home */
	int lineSpacing;	/* Current line spacing */
	short backwardExtent;	/* Line box backward extent */
	short forwardExtent;	/* Line box forward extent */
	short previousExtent;	/* Forward extent of the previous line box */
	short			/* Framing values for framed texts */
		top_framing, bottom_framing, left_framing, right_framing;
	int yShift;		/* Font baseline shift from line baseline */
	/*
	** Wrap around parameters
	*/
	int leftWidgetIndent;
	int rightWidgetIndent;
	int rightWidgetIndentLine; /* Portion generated by current line! */
	
	SequenceState current;	/* Current sequence state */
	SequenceState lastSpace;/* State at last space */
	Snip *firstSnip;	/* First Snip of the sequence */
	Snip *beginLine;	/* First Snip of the line box */
    } LayoutContext;

/*
** XeInsetSnipExtents
**	Compute dimensions of a inset Snip directly into the
**	Snip structure (fill in fields xWidth, ascent and descent)
*/
#if NeedFunctionPrototypes
static void XeInsetSnipExtents(LayoutContext *, Snip *);
#endif
static void XeInsetSnipExtents(cx, t)
LayoutContext *cx;
Snip *t;
    {
	XeTextConstraints tc = (XeTextConstraints)
		t->content.widget->core.constraints;
	int width, height;

	/*
	** Compute width for the 'MAX' constraint.
	*/
	if (tc->frame.position.type == XeFramePositionType_FIXED)
	    {
		/*
		** The code below is just throwing in some values as
		** placeholders (use FIXED and 'max' combination at
		** own risk).
		*/
		width = t->content.widget->core.width - tc->frame.position.x;
		if (width < 0)
			width = 0;
		height = t->content.widget->core.height - tc->frame.position.y;
		if (height < 0)
			height = 0;
	    }
	else
	    {
		/*
		** For horizontal 'MAX', use the remaining available space
		*/
		width = cx->lineLength - cx->current.wNow;
		/*
		** Compute height for the 'MAX' vertical constraint. This
		** is a guess at best, only the content that precedes the
		** inset is accounted...
		**
		** *NOTE* this width or height will be used only if the
		**	constraint selects the MAX in that dimension,
		**	otherwise these are ignored.
		**
		** -- Unfortunately this logic will not work, if there
		** -- are borders that require space around! This need
		** -- be rethought (either subrtract border space here
		** -- or in _XeFrameQueryExtents. Until then application
		** -- should avoid MAX constraint (or use at own risk).
		** -- msa
		*/
		height = cx->current.maxAscent + cx->current.maxDescent;
		if (height < cx->backwardExtent + cx->forwardExtent)
			height = cx->backwardExtent + cx->forwardExtent;
		if (height == 0)
			height = cx->lineSpacing; /* If all else fails.. */
	    }
	_XeFrameQueryExtents
		(t->content.widget, width, height, &tc->text.FE);
	t->xWidth = tc->text.FE.width;
	t->offset = 0;
	if (tc->frame.position.type == XeFramePositionType_FIXED ||
	    tc->frame.position.type == XeFramePositionType_START ||
	    tc->frame.position.type == XeFramePositionType_END)
		t->floating = True;
	else
		t->floating = False;
	/*
	** Fix ascent and descent only for BASELINE alignments.
	** Only Base line aligned widgets will affect line box.
	** Fixed position widgets will not have effect on line box.
	*/
	if (tc->frame.position.type == XeFramePositionType_FIXED)
		t->ascent = t->descent = 0;
	else switch (tc->frame.alignment)
	    {	
	    case XeFrameAlignment_TOP:
	    case XeFrameAlignment_BOTTOM:
	    case XeFrameAlignment_CENTER:
		/* Line Box related alignments, don't fix these yet */
		t->ascent = t->descent = 0;
		break;
	    default:
	    case XeFrameAlignment_ABOVE:
		/* Above baseline */
		t->ascent = tc->text.FE.height - 1;
		t->descent = 1;
		break;
	    case XeFrameAlignment_AROUND:
		/* Centered around baseline */
		t->ascent = tc->text.FE.height / 2;
		t->descent = tc->text.FE.height - t->ascent;
		break;
	    case XeFrameAlignment_BELOW:
		/* Below baseline */
		t->ascent = 0;
		t->descent = tc->text.FE.height;
		break;
	    }
    }

#if NeedFunctionPrototypes
static void XeTextSnipExtents(LayoutContext *, Snip *);
#endif

static void XeTextSnipExtents(cx, t)
LayoutContext *cx;
Snip *t;
    {
	XCharStruct XE;
	int dir, fontAscent, fontDescent;
	SnipData *h = t->content.head;
	FontCache *font_cache = cx->font_cache;
	XeTextWidget w = cx->w;

	if (h == NULL)
	    {
		t->xWidth = 0;
		t->ascent = 0;
		t->descent = 0;
		t->offset = 0;
		return;
	    }
	if (h->font == NULL)
	    {
		/*
		** Font struct still undefined, load it
		*/
		int fn = Font_NUMBER(t->mode.bits);
		XeFontList fId;
		int hit, i;
		int pSize;
		unsigned long fontkey;

		if (fn >= w->text.num_fonts)
			fn = 0; /* Try for default font */
		if (w->text.fonts == NULL || w->text.num_fonts == 0)
		    {
			pSize = cx->lineSpacing;
			fId = 0;
		    }
		else
		    {
			pSize = w->text.fonts[fn].font_size;
			fId = w->text.fonts[fn].font_list;
		    }
		if (t->mode.size_modification)
			pSize = (t->mode.size_modification * pSize + 50) / 100;
		fontkey = FontCacheInfo(t->mode, pSize);
		hit = ((fontkey ^ (unsigned long)h->character_set) ^
		       (unsigned long)fId) % XtNumber(font_cache->data);
		for (i = hit ;; )
		    {
			if (font_cache->data[i].font == NULL)
			    {
				font_cache->cache_miss++;
				break;
			    }
			else if (font_cache->data[i].fontkey == fontkey &&
				 font_cache->data[i].fontlist == fId &&
				 font_cache->data[i].charset==h->character_set)
			    {
				font_cache->cache_hit++;
				break;
			    }
			i += 1;
			if (i == XtNumber(font_cache->data))
				i = 0;
			if (i == hit)
			    {
				/*
				** Nasty thing.. should really not happen.
				** Just take over the first hash entry. Not
				** really right solution, but we assume for
				** now that we really don't get here...
				*/
				font_cache->data[i].font = NULL;
				font_cache->cache_full++;
				font_cache->cache_reset = TRUE;
			    }
		    }
		if (font_cache->data[i].font == NULL)
		    {
			font_cache->data[i].fontkey = fontkey;
			font_cache->data[i].fontlist = fId;
			font_cache->data[i].charset = h->character_set;
			font_cache->data[i].font = XeFindFont
				(font_cache->display, h->character_set, fId,
				 t->mode.bits & Weight_MASK,
				 t->mode.bits & Italicized_MASK, pSize);
			
		    }
		h->font = font_cache->data[i].font;
	    }
        /*
         * If you check the return value from XTextExtents[16]() you'll
         * find that it occasionally fails on the printing routines.
         * (why? I dunno), so let's make sure that Ascent and Descent have
         * sane values, as Descent is sometimes used to calculate line
         * spacing.  -EZA
         */
        
        fontDescent = fontAscent = 0;
        
	if (h->bytes == 2)
		XTextExtents16((XFontStruct *)h->font,
			       (XChar2b *)t->data, t->length,
			       &dir, &fontAscent, &fontDescent, &XE);
	else
		XTextExtents((XFontStruct *)h->font, t->data, t->length,
			     &dir, &fontAscent, &fontDescent, &XE);
	t->xWidth = XE.width;
	/*
	** Using Font ascent/descent gives better result for now..
	*/
	t->ascent = fontAscent;
	t->descent = fontDescent;
	t->offset = 0;
    }

/*
** XeTextSnipWidth
**	return the width of the text string.
**
**	*NOTE*
**		Works only if the Snip has been laid out (e.g. already
**		has a valid font assigned!!!)
*/
int XeTextSnipWidth(w, t, s, n)
XeTextWidget w;
Snip *t;
char *s;
int n;
    {
	SnipData *h = t->content.head;

	if (h == NULL)
		return 0;
	if (h->font == NULL)
		return 0;
		/* XeTextSnipExtents(w, t); ...Force loading of Font */
	return h->bytes == 2 ?
		XTextWidth16(h->font, (XChar2b *)s, n) :
			XTextWidth(h->font, s, n);
    }


/*
** FindTabulationStop
**	Find and return a pointer the tabulation stop definition.
**	Returns NULL, if not defined for the reference.
**
**	The tabulation position is selected by the reference, which
**	may result positioning over existing text. The tabulated text
**	sections can occur in any order regardless of their final
**	placements on the line.
**
*/
#if NeedFunctionPrototypes
static XeTabStop *FindTabulationStop(XeTextWidget, int);
#endif
static XeTabStop *FindTabulationStop(w, reference)
XeTextWidget w;
int reference;
    {
	XeTabStop *t = w->text.line_layout_table;

	if (t == NULL)
		return NULL;
	for ( ; t->reference >= 0; t++)
	    {
		if (t->reference == reference)
			return t;
	    }
	return NULL;
    }
/*
** STAB_POSITION
**	is a macro to compute the absolute tabulation position from
**	the XeTabStop. Current experimental logic for the position is
**
**	> 0,	tabulation position value is absolute
**	< 0,	tabulation position value is added to set width
**	== 0,	if alignment = END, then setwidth, otherwise 0.
*/
#define STAB_POSITION(cx, ts) \
	(cx->column_x + \
	 (ts->position > 0 ? ts->position : \
	  ts->position < 0 ? ts->position + cx->width : \
	  ts->alignment == XeAlignment_END ? cx->width : 0))
/*
** HorizontalTab
**	The tabulation position is selected by searching forward from
**	the current position. LineLayoutTable is used, if present,
**	otherwise tab stops are assumed to be evenly spaced at every
**	eight character position (width of the character position is
**	defined as the width of the SPACE in current font).
*/
#if NeedFunctionPrototypes
static int HorizontalTab(LayoutContext *, int, int *);
#endif
static int HorizontalTab(cx, x, tabref)
LayoutContext *cx;
int x;
int *tabref;
    {
	XeTextWidget w = cx->w;
	XeTabStop *t = w->text.line_layout_table;
	int tabWidth, position;
	Snip *r;

	if (t != NULL)
	    {
		/* Horizontal Tabulation, try using lineLayoutTable */

		XeTabStop *s = NULL;
		int d = ~((unsigned int)0)>>1, e;

		for (; t->reference >= 0; t++)
		    {
			e = STAB_POSITION(cx, t) - x;
			if (e > 0 && e < d)
			    {
				d = e;
				s = t;
			    }
		    }
		if (s != NULL)
		    {
			*tabref = s->reference;
			return cx->column_x + s->position; /* Found one */
		    }
	    }
	/*
	** Horizontal Tabulation, no lineLayoutTable, or current position
	** is past all defined positions there. Assume evenly spaced
	** tabulation positions and choose the next suitable.
	*/
	/* Use the first font used in the widget. This may not be
	   actually right, some widget wide specific default font should
	   be used. Fix later --msa
	*/
	tabWidth = 0;
	for (r = w->text.first; r != NULL; r = r->next)
		if (!r->widget && r->content.head != NULL)
		    {
			tabWidth = XeTextSnipWidth(w, r, "    ", 1) * 8;
			break;
		    }
	if (tabWidth == 0)
		tabWidth = 8 * 6; /* Just put something in there... --msa */
	x -= cx->column_x;
	position = ((x + tabWidth) / tabWidth) * tabWidth;
	return position + cx->column_x;
    }

/*
** ComputeWidth
**	Compute the width of the current Snip
*/
#if NeedFunctionPrototypes
static void ComputeWidth(LayoutContext *, Snip *);
#endif

static void ComputeWidth(cx, t)
LayoutContext *cx;
Snip *t;
    {
	int tmp;

	switch (t->mode.bits & PartialLine_MASK)
	    {
	    case PartialLine_NONE:
		cx->yShift = 0;
		break;
	    case PartialLine_UP:
		cx->yShift = -(cx->lineSpacing / 3);
		break;
	    case PartialLine_DOWN:
		cx->yShift = cx->lineSpacing / 3;
		break;
	    }
	t->y += cx->yShift;
	if (t->widget)
		XeInsetSnipExtents(cx, t);
	else
		XeTextSnipExtents(cx, t);
	if (t->mode.bits & Framing_MASK && !t->floating)
	    {
		t->ascent += cx->top_framing;
		t->descent += cx->bottom_framing;
		/*
		** NOTE: while framed is True, the reservation for
		** frame ending (right_framing) is kept in wNow, and
		** subtracted when framing explicitly ends. This attempts
		** to get lines right with wrapping, which requires implicit
		** ending/start of framing at end of lines.
		*/
		if (!cx->current.framed)
		    {
			/* Starting framed section, must leave space for
			   the frame graphics */
			t->xWidth += cx->left_framing;
			t->offset += cx->left_framing;
			cx->current.wNow += cx->right_framing;
		    }
		if (HasEndLine(t) || t->next == NULL ||
		    !(Framing_MASK & t->next->mode.bits))
			/* Ending framed section, must leave space for
			   the frame graphics (newline breaks framing
			   temporarily */
		    {
			t->xWidth += cx->right_framing;
			cx->current.wNow -= cx->right_framing;
			cx->current.framed = False;
		    }
		else
			cx->current.framed = True;
	    }
	cx->current.wNow += t->xWidth;
	tmp = t->ascent - cx->yShift;
	if (tmp > cx->current.maxAscent)
		cx->current.maxAscent = tmp;
	tmp = t->descent + cx->yShift;
	if (tmp > cx->current.maxDescent)
		cx->current.maxDescent = tmp;
    }

/*
** Position_XXXX functions will do the appropriate changes into the
** Snip chain to achieve the positioning of the content.
**
** Each will return a pointer to the last Snip that actually is included
** into this sequence. Usually it will be the supplied parameter 'last',
** except for the TEXT.
*/

/*
** CloseLineBox
**	Do final touches to the line box placement. Compute the amount of
**	vertical adjustment for the baseline (relative to the baseline of
**	the previous line) and update all snips within line.
**
**	Returns non-zero, if the line must be relaid with new position.
**	May update the (x,y) position.
*/
#if NeedFunctionPrototypes
static int CloseLineBox(LayoutContext *, Snip *, int *, int *);
#endif
static int CloseLineBox(cx, end, x, y)
LayoutContext *cx;
Snip *end;
int *x, *y;
    {
	int y_adj = cx->previousExtent, tmp;
	int y_max = *y + cx->forwardExtent;
	Snip *t;
	XeTextConstraints tc;

	if (cx->w->composite.num_children > 0)
	    {
		for (t = cx->beginLine;;)
		    {
			if (t->widget && t->content.widget &&
			    XtIsManaged(t->content.widget))
			    {
				/*
				** Do some adjustments to extents according
				** to the alignment.
				** NOTE: The tricky thing is that if the
				** element is "text framed", the ascent and
				** descent have already been adjusted for
				** framing. That's why +='s are used... --msa
				*/
				tc = (XeTextConstraints)
					t->content.widget->core.constraints;
				switch (tc->frame.alignment)
				    {
				    case XeFrameAlignment_TOP:
					/* Top of the linebbox */
					t->ascent += cx->backwardExtent;
					t->descent += tc->text.FE.height
						- cx->backwardExtent;
					break;
				    case XeFrameAlignment_BOTTOM:
					/* Bottom of the linebox */
					t->ascent += tc->text.FE.height -
						cx->forwardExtent;
					t->descent += cx->forwardExtent;
					break;
				    case XeFrameAlignment_CENTER:
					/* Centered in the line box */
					tmp = (tc->text.FE.height +
					       cx->backwardExtent -
					       cx->forwardExtent) / 2;
					t->ascent += tmp;
					t->descent += tc->text.FE.height - tmp;
					break;
				    default:
					/* Other alignemnts have already been
					** computed.
					*/
					break;
				    }
				switch (tc->frame.position.type)
				    {
				    case XeFramePositionType_FIXED:
					t->descent = 0;
					t->ascent = 0;
					break;
				    default:
				    case XeFramePositionType_INLINE:
					/*
					** Update forward extent only if
					** widget is INLINE.
					*/
					if (t->descent > cx->forwardExtent)
						cx->forwardExtent = t->descent;
					/* FALL THROUGH!! */
				    case XeFramePositionType_START:
				    case XeFramePositionType_END:
					/*
					** Update backward extent for START,
					** END and INLINE positioning
					*/
					if (t->ascent > cx->backwardExtent)
						cx->backwardExtent = t->ascent;
					/*
					** Even if the inset does not affect
					** the line box dimensions, one cannot
					** place start/end aligned inset to
					** cross page boundary--update y_max!
					*/
					if (y_max < t->y + t->descent)
						y_max = t->y + t->descent;
					break;
				    }
			    }
			if (t == end)
				break;
			t = t->next;
		    }
	    }
	/*
	** Proportional spacing is used only if the line has some content
	** in it. Otherwise fixed lineSpacing is assumed.
	*/
	if (cx->w->text.proportional&&(cx->forwardExtent||cx->backwardExtent))
	    {
		cx->previousExtent = cx->forwardExtent;
		y_adj += cx->backwardExtent;
	    }
	else
	    {
		cx->previousExtent = cx->lineSpacing / 3;
		/*
		** If fixed line spacing is selected, then use the backward
		** extent of current line, if this is the first line.
		** (otherwise the widget might clip the content)
		*/
		if (*y == cx->column_y && cx->backwardExtent > 0)
			y_adj += cx->backwardExtent;
		else
			y_adj += cx->lineSpacing - cx->previousExtent;
	    }
	y_max += y_adj;
	/*
	** Move the snips of the linebox into final positions
	** (do this before checking page overflow, so that if the layout
	** decides not to do relayout, the positions are somewhat reasonable,
	** this happens often for the last line of the content)
	*/
	for (t = cx->beginLine;;)
	    {
		t->y += y_adj;
		if (t == end)
			break;
		t = t->next;
	    }
	/*
	** If the linebox would extend past column end, align the linebox
	** to the beginning of the next "page" (unless this would be the
	** first linebox on the page).
	*/
	if (y_max > cx->column_end && *y > cx->column_y)
	    {
		if (++cx->column == cx->w->text.set.columns)
		    {
			cx->column = 0;
			cx->page += 1;
		    }
		cx->column_x = cx->column *
			(cx->width + cx->w->text.column_separation)
				+ cx->offset_x;
		*x = cx->column_x;
		cx->column_y = cx->page *
			(cx->height + cx->w->text.page_separation)
				+ cx->offset_y;
		*y = cx->column_y;
		cx->column_end = cx->column_y + cx->height;
		return 1; /* Must redo layout for this line! */
	    }
	*y += y_adj;
	return 0;
    }

/*
** WrapUpSequence
**	Do the finishing touches for the positioning functions:
**	search for endseq etc.
*/
#if NeedFunctionPrototypes
static Snip *WrapUpSequence(LayoutContext *, SequenceState *, Snip *, int);
#endif
static Snip *WrapUpSequence(cx, s, end, x_adj)
LayoutContext *cx;
SequenceState *s;
Snip *end;
int x_adj;
    {
	Snip *t;

	if (end->widget && end->content.widget)
	    {
		/*
		** Widget Snip has *always* endseq set. This means that
		** in a sequence, there can be only one Widget, and that
		** is the terminating Snip (end). Thus, it is sufficient
		** to check the end only...
		*/
		XeTextConstraints tc = (XeTextConstraints)
			end->content.widget->core.constraints;

		if (tc->frame.position.type == XeFramePositionType_START)
		    {
			/*
			** Left aligned floating widget. Place the left
			** edge of the widget to the starting point of the
			** current sequence and arrange the whole sequence
			** to be shifted right by the width of the widget.
			*/
			x_adj += end->xWidth;
			end->offset = cx->w->text.indentation + cx->column_x +
				cx->leftWidgetIndent - end->x - x_adj;
		    }
		else if (tc->frame.position.type==XeFramePositionType_END)
		    {
			/*
			** Right aligned floating widget. Place the right
			** edge of the widget to the right edge of the
			** column.
			*/
			end->offset = cx->width + cx->column_x
				- cx->rightWidgetIndent
				- end->xWidth - end->x - x_adj;
		    }
	    }
	if (s->maxAscent > cx->backwardExtent)
		cx->backwardExtent = s->maxAscent;
	if (s->maxDescent > cx->forwardExtent)
		cx->forwardExtent = s->maxDescent;
	for (t = cx->firstSnip; ;t = t->next)
	    {
		t->x += x_adj;
		if (t == end)
			break;
	    }
	return end;
    }

/*
** Position_TEXT
**	Position a TEXT sequence.
**
**	last	the last Snip actually taking part in positioning.
**
**	end	the actual Snip ending the sequence (this differs
**		from last, if line end contains spaces).
*/
static Snip *Position_TEXT(cx, text, last, end)
LayoutContext *cx;
SequenceState *text;
Snip *last, *end;
    {
	int offset = 0;
	int wExtra = cx->lineLength - text->wNow;
	XeAlignment alignment;
	/*
	** Force START aligment, if no real lineLength or if the sequence
	** is actually ended with Horizontal TAB. Use the alignment given
	** by the resource, unless the Snip specifies it explicitly by
	** Justify_MODE.
	*/
	if (cx->lineLength <= 0 ||
	    (end->tab && end->tabref == Snip_NEXTTABSTOP))
		alignment = XeAlignment_START;
	else if (end->quad)
		alignment = end->quad;
	else if (end->mode.bits & Justify_MASK)
		alignment = Justify_MODE(end->mode.bits);
	else
		alignment = cx->w->text.alignment;
	switch (alignment)
	    {
	    default:
	    case XeAlignment_START_FILL:
	    case XeAlignment_START:
		break;
	    case XeAlignment_END_FILL:
	    case XeAlignment_END:
		offset += wExtra;
		break;
	    case XeAlignment_CENTER_FILL:
	    case XeAlignment_CENTER:
		offset += wExtra / 2;
		break;
	    case XeAlignment_JUSTIFIED:
		/*
		** For the justifying to be applicable, the line
		** must include adjustable spaces, there must be
		** positive amount of loose space. Additionally,
		** unless this is "forced quad", this must not be the
		** "hard line end" or last line of the block.
		*/
		if (text->spaces && wExtra > 0 &&
		    (end->quad || 
		     (IsLayoutContent(end->mode.bits) && end->next != NULL)))
		    {
			Snip *t;
			int wSpace = wExtra / text->spaces;
			int adjust = 0;

			text->wNow += wExtra;
			wExtra -= text->spaces * wSpace;
			for (t = cx->firstSnip; t != last; t = t->next)
			    {
				if (t->space)
				    {
					adjust += wSpace;
					if (wExtra)
					    {
						adjust += 1;
						wExtra -= 1;
					    }
				    }
				t->x += adjust;
			    }
			for (;; t = t->next)
			    {
				t->x += adjust;
				if (t == end)
					break;
			    }
		    }
		break;
	    }
	return WrapUpSequence(cx, text, end, offset);
    }

/*
** Position_ITEM
**	Position ITEM identifier
*/
static Snip *Position_ITEM(cx, item, last)
LayoutContext *cx;
SequenceState *item;
Snip *last;
    {
	int offset;
	/*
	** Compute the offset of the item from the line home position,
	** taking alignment into account.
	*/
	if (cx->w->text.itemization->identifier_alignment == XeAlignment_START)
		offset = cx->w->text.itemization->identifier_start_offset;
	else
	    {
		offset = cx->w->text.itemization->identifier_end_offset -
			item->wNow;
		/*
		** Non-ODA implementation addition: if the item would be
		** positioned outside the available area, then start it
		** from the start edge of the available area.
		*/
		if (offset + cx->lineOffset < cx->column_x)
			offset = cx->column_x - cx->lineOffset;
	    }
	/*
	** If a floating widget is on the left, compute as if line home
	** position was to the right of it.. (does not really work! --msa)
	*/
	offset += cx->leftWidgetIndent;
	return WrapUpSequence(cx, item, last, offset);
    }

/*
** LookFor
**	Search for characters from a bounded string. Return
**	a pointer to the first character found or NULL, if none.
**	(comparable with strpbrk).
*/
static char *LookFor(d, s, n)
char *d, *s; int n;
    {
	unsigned char c;
	register unsigned char *p;
	unsigned char *q = (unsigned char *)s + n;

	if (d == NULL || s == NULL || n <= 0)
		return NULL; /* This really means erroneus call.. */
	for ( ; (c = *d) != 0; d++)
		for (p = (unsigned char *)s; p < q; ++p)
			if (*p == c)
				return (char *)p;
	return NULL;
    }

/*
** Position_STAB
**	Position STAB sequence
*/
static Snip *Position_STAB(cx, stab, last, tabref)
LayoutContext *cx;
SequenceState *stab;
Snip *last;
int tabref;
    {
	static XeTabStop none;

	int offset, position;
	XeTabStop *ts = FindTabulationStop(cx->w, tabref);
	Snip *t;

	if (ts == NULL)
		/* Error? Couldn't find the definition for the tab stop!!
		** Just default to All Zeroes Tabulation_Stop for now...
	        */
		ts = &none;
	/*
	** Get the real position from the XeTabStop
	*/
	position = STAB_POSITION(cx, ts);
	/*
	** Compute initial offset as if 'start aligned' was selected
	*/
	offset = position - cx->firstSnip->x;
	/*
	** Compute offset from start of the positioning area to
	** the beginning of the positioned text.
	*/
	switch (ts->alignment)
	    {
	    case XeAlignment_END:
		offset -= stab->wNow;
		break;
	    case XeAlignment_CENTER:
		offset -=  stab->wNow / 2;
		break;
	    case XeAlignment_AROUND:
		if (ts->alignment_string[0] == '\0')
		    {
			/*
			** Aligned around specified without giving the
			** characters, an error, but treat as if aligned
			** around character was not found, assume 'end_aligned'
			*/
			offset -= stab->wNow;
			break;
		    }
		for (t = cx->firstSnip; ; )
		    {
			char *s = NULL;

			/*
			** Temporary kludge: Do not search alignment chars
			** from snips that are multibyte (not right now
			** either). Should verify that the aligment char
			** is from same character set as the Snip and search
			** only in that case... --msa
			*/
			if (!t->widget &&
			    t->content.head && t->content.head->bytes == 1)
				s = LookFor(ts->alignment_string,
					    t->data, t->length);
			if (s != NULL)
			    {
				/*
				** Found the character around which
				** the text should be aligned. Align point
				** will be the PP of the align character.
				*/
				offset -= t->x - cx->firstSnip->x +
					XeTextSnipWidth(cx->w, t, t->data,
							s - t->data);
				break;
			    }
			if (t == last)
			    {
				/*
				** Aligned around was specified, but none
				** of the alignment characters occurred
				** within tabulated text. Default to
				** end-aligned in that case.
				*/
				offset -= stab->wNow;
				break;
			    }
			t = t->next;
		    }
		break;
	    case XeAlignment_START:
	    default:
		break;
	    }
	return WrapUpSequence(cx, stab, last, offset);
    }

/*
** SplitLine
**	Split the line from the specified Snip (s) and return a pointer
**	to a Snip that will belong to the next line.
*/
static Snip *SplitLine(cx, text, last)
LayoutContext *cx;
SequenceState *text;
Snip *last;
    {
	Snip *t = last;
	Snip *s;
	int x, y;

	x = t->x;
	y = t->y;
	/*
	** Eat up all extra spaces to the end of the line
	*/
	for ( ; ; )
	    {
		if (t->space)
		    {
			t->xWidth = 0;
			t->x = x;
			t->y = y;
		    }
		if (t->endseq)
			break;
		s = t->next;
		if (s == NULL || !s->space)
			break;
		t = s;
	    }
	/*
	** If the last element doesn't end the sequence, then generate
	** the line ending sequence here.
	*/
	if (!t->endseq)
	    {
		if (text->framed)
			/* Reserve space for the framing end */

			t->xWidth += cx->right_framing;
		s = _XeInsertSnip(&t->next);
		s->endseq = Snip_EndLine;
		s->mode = t->mode;
		s->mode.bits &= ~Content_MASK;
		s->mode.bits |= Content_FORMATTED;
		s->mode.tag = LAYOUT_CONTENT_TAG;
		s->layout = TRUE;
		s->valid = TRUE;
		s->column = cx->column;
		s->x = x;
		s->y = y;
		t = s;
	    }
	return Position_TEXT(cx, text, last, t);
    }

/*
** SkipLine
**	Generate an empty layout line in *FRONT* of the specified Snip (mark).
*/
static Snip *SkipLine(cx, mark)
LayoutContext *cx;
Snip *mark;
    {
	Snip *s;

	s = _XeInsertSnip(mark->back);
	s->endseq = Snip_EndLine;
	s->mode.bits &= ~Content_MASK;
	s->mode.bits |= Content_FORMATTED;
	s->mode.tag = LAYOUT_CONTENT_TAG;
	s->layout = TRUE;
	s->valid = TRUE;
	s->column = cx->column;
	s->x = mark->x;
	s->y = mark->y;
	return s;
    }

/*
** UpdateExposeArea
**	Extend expose area from the Snip
*/
#if NeedFunctionPrototypes
static void UpdateExposeArea(XRectangle *, int, int, int, int);
#endif
static void UpdateExposeArea(r, x, y, width, height)
XRectangle *r;
int x, y, width, height;
    {
	if (width == 0 || height == 0)
		return;
	if (r->width == 0)
	    {
		r->x = x;
		r->y = y;
		r->width = width;
		r->height = height;
	    }
	else
	    {
		if (x < r->x)
		    {
			r->width += r->x - x;
			r->x = x;
		    }
		if (x + width > r->x + r->width)
			r->width = x + width - r->x;
		if (y < r->y)
		    {
			r->height += r->y - y;
			r->y = y;
		    }
		if (y + height > r->y + r->height)
			r->height = y + height - r->y;
	    }
    }

/*
** UpdateExpose
**	Compare Snips in the line box to the saved coordinates of the
**	previous Snips. Update expose region to cover all changes.
**
**	Return 'mode', which has the following interpretation
**
**	mode=-1	searching for valid=FALSE position,
**	mode=1	have found valid=FALSE, now searching for valid=TRUE,
**	mode=0	have found valid=TRUE, layout process can break now.
**	mode=2	do not search for anything. Layout will not terminate
**		until end of content.
*/
#define LOOKFOR_MODE (-1)
#define LOOKFOR_NONE 2

#if NeedFunctionPrototypes
static int UpdateExpose(XRectangle *,ExpectEntry *,ExpectEntry *,Snip *,int);
#endif
static int UpdateExpose(expose, start, end, last, mode)
XRectangle *expose;
ExpectEntry *start, *end;
Snip *last;
int mode;
    {
	int changes = 0;
	Snip *s;

	if (!expose)
	    {
		for ( ;start < end && (s = start->s) != last; start++)
		    {
			if (IsEditableContent(s->mode.bits))
			    {
				if (!s->valid ||
				    start->x != s->x || start->y != s->y)
					changes++;
				if (s->valid)
				    {
					if (mode == 1)
						mode = 0;
				    }
				else if (mode == -1)
					mode = 1;
			    }
			s->layout = s->valid = TRUE;
		    }
	    }
	else for ( ;start < end && (s = start->s) != last; start++)
	    {
		if (IsEditableContent(s->mode.bits))
		    {
			if (!s->valid ||
			    start->x != s->x ||
			    start->y != s->y ||
			    start->ascent != s->ascent ||
			    start->descent != s->descent ||
			    start->xWidth != s->xWidth ||
			    start->offset != s->offset)
			    {
				UpdateExposeArea
					(expose, s->x, s->y - s->ascent,
					 s->xWidth, s->ascent + s->descent);
				changes++;
				if (s->layout)
					UpdateExposeArea
						(expose, start->x,
						 start->y - start->ascent,
						 start->xWidth,
						 start->ascent+start->descent);
			    }
			if (s->valid)
			    {
				if (mode == 1)
					mode = 0;
			    }
			else if (mode == -1)
				mode = 1;
		    }
		else
			UpdateExposeArea(expose, s->x, s->y - s->ascent,
					 s->xWidth, s->ascent + s->descent);
		s->layout = s->valid = TRUE;
	    }
	
	/*
	** Restore the original (x,y) for the snips that were processed,
	** but didn't get included into the linebox.
	*/
	while (start < end)
	    {
		start->s->x = start->x;
		start->s->y = start->y;
		start->s->xWidth = start->xWidth;
		start->s->ascent = start->ascent;
		start->s->descent = start->descent;
		start->s->offset = start->offset;
		start->s->column = start->column;
		/*
	        ** A quick patch: If there was inset that was tried
		** for insertion into line, but didn't actually fit. The
		** contraint text.FE structure may have been changed by
		** this and should be also restored--unfortunately it is
		** not saved. Try to fix this by preventing the layout
		** termination (bump the 'changes' upward)...
		*/
		if (start->s->widget)
			changes++;
		start++;
	    }
	return mode ? mode : changes != 0;
    }

/*
** AdjustForWidgets
**	checks if the terminating inset Snip has a widget that is
**	floated to the start/end of sequence and activates a
**	wraparound. In that case one has to use only the 'x' from
**	the widget and ignore the xWidth. (The real positioning of
**	the widget is made by offset field).
*/
static int AdjustForWidgets(cx, s)
LayoutContext *cx;
Snip *s;
    {
	if (s->widget && s->content.widget)
	    {
		if (s->floating)
		    {
			XeTextConstraints tc;
			tc = (XeTextConstraints)
				s->content.widget->core.constraints;
			/*
			** A rather ugly hack to get the line width
			** right with right aligned floating widgets
			*/
			if (tc->frame.position.type == XeFramePositionType_END)
			    {
				cx->rightWidgetIndent += tc->text.FE.width;
				cx->rightWidgetIndentLine += tc->text.FE.width;
			    }
			return s->x;
		    }
	    }
	return s->x + s->xWidth;
    }

/*
** FindInsetIndents
**	compute the extra left and right indentation that is in effect
**	at current point (y).
*/
static void FindInsetIndents(cx, y)
LayoutContext *cx;
int y;
    {
	int i, indent, y_end, x_begin, x_end;
	XeTextConstraints tc;
	Widget child;
	Snip *s;

	cx->leftWidgetIndent = 0;
	cx->rightWidgetIndent = 0;
	cx->rightWidgetIndentLine = 0;
	for (i = 0; i < cx->w->composite.num_children; ++i)
	    {
		XeFramePositionType type;

		if ((child = cx->w->composite.children[i]) == NULL)
			continue;
		if ((tc = (XeTextConstraints)child->core.constraints) == NULL)
			continue;
		if ((s = tc->text.inset) == NULL)
			continue;
		if (!s->valid)
			continue;
		type = tc->frame.position.type;
		if (type == XeFramePositionType_FIXED)
		    {
			/*
			** This is a first crack at dealing with fixed
			** position insets.
			*/
			int left_room, right_room;
			/*
			** Whether inset affects the current line or not
			** is at best a guess work...
			*/
			if (tc->frame.position.y > y + cx->lineSpacing)
				continue;
			y_end = tc->frame.position.y + tc->text.FE.height;
			x_begin = tc->frame.position.x;
			x_end = x_begin + tc->text.FE.width;
			left_room = x_begin -
				(cx->column_x + cx->w->text.indentation);
			right_room = cx->column_x + cx->width - x_end;
			if (left_room > right_room)
				type = XeFramePositionType_END;
			else
				type = XeFramePositionType_START;
		    }
		else if (s->column < cx->column)
		    {
			/*
			** Totally ad hoc hack here! If the column of the inset
			** is before the current column, treat it here as if it
			** had fixed position (an attempt to deal with extra
			** large insets spanning columns). Does not work well,
			** if it spans pages too--would need page number in
			** Snip also. Now it will work across pages only for
			** columns less than inset column. May seem mysterious
			** and confusing to the user). I hope this does not
			** make layout unstable... -- msa
			*/
			int left_room, right_room;

			if (s->y - s->ascent > y + cx->lineSpacing)
				continue;
			y_end = s->y + s->descent;
			x_begin = s->x + s->offset;
			x_end = x_begin + tc->text.FE.width;
			left_room = x_begin -
				(cx->column_x + cx->w->text.indentation);
			right_room = cx->column_x + cx->width - x_end;
			if (left_room > right_room)
				type = XeFramePositionType_END;
			else
				type = XeFramePositionType_START;
		    }
		else if (s->y > y || s->column != cx->column)
			/*
			** Insert starts after the current position or
			** belongs to another column (cannot allow floating
			** insets interact across columns--unstable!)
			*/
			continue;
		else
		    {
			x_begin = s->x + s->offset;
			x_end = x_begin + tc->text.FE.width;
			y_end = s->y - s->ascent + tc->text.FE.height;
		    }
		if (y_end < y)
			continue;
		if (x_begin >= cx->column_x + cx->width)
			continue;
		if (x_end < cx->column_x)
			continue;
		if (type == XeFramePositionType_START)
		    {
			indent= x_end - cx->column_x - cx->w->text.indentation;
			if (cx->leftWidgetIndent < indent)
				cx->leftWidgetIndent = indent;
		    }
		else if (type == XeFramePositionType_END)
		    {
			indent= cx->column_x + cx->width - x_begin;
			if (cx->rightWidgetIndent < indent)
				cx->rightWidgetIndent = indent;
		    }
	    }
    }

/*
** _XeTextLayout
**	Layout widget character content into widget *internal*
**	"Processable Formatted Form". This function can be called
**	any number of times.
**
**	from = 	NULL, redo the layout fully, ignore all existing
**		old layout information.
**
**	from != NULL, start layout process from the first possible
**		linebox before this point (Snip).
**
**	If expose is non-NULL, the pointed XRectangle will be expanded
**	to contain the area of the layout (usefull only if from != NULL).
**	The rectangle must be properly initialize before call
**	(expose->width = 0 is sufficient to make area initially empty).
*/
int _XeTextLayout(w, set, from, expose_region)
XeTextWidget w;
XeTextLayoutGeometry *set;
Snip *from;
Region expose_region;
    {
	typedef enum
	    {
		Sequence_TEXT,
		Sequence_ITEM,
		Sequence_STAB
	    } LayoutSequence;

	/*
	** lay, addLayout
	**	if addLayout == non-zero, then we are processing application
	**	inserted layout content before the Snip pointed by 'lay'.
	**	(note! lay may also be NULL, if inserting occurs after
	**	the last editable Snip). When addLayout == 0, 'lay' is
	**	NULL. A special value -1 is used when the layout process
	**	should start in addLayout mode, but not call BEGIN_LINE
	**	callbacks.
	** expect_default, expect, expect_point, expect_end
	**	are used in line building to detect when layout hits a point
	**	that doesn't change. Fixed array expect_default is used and
	**	it should be sufficient for most cases. However, the code
	**	is prepared to allocate a larger array dynamically on demand.
	*/
	Snip *s, *p, **h, *lay;
	XeTextTag tag;
	int i, x, y, yHighMark, xHighMark;
	ExpectEntry expect_default[200];
	ExpectEntry *expect = NULL, *expect_point = &expect_default[0];
	ExpectEntry *expect_end = &expect_default[XtNumber(expect_default)];
	int layout = w->text.format;
	int lookfor, tabref, firstSeq, addLayout, itemization = 0;
	LayoutSequence sequence = Sequence_TEXT; /* .. init to silence gcc */
	XRectangle expose_rect, *expose;
	LayoutContext cx;

	static LayoutContext init_cx;

	cx = init_cx;
	/*
	** Keep track of the bounding box of exposed area only if
	** expose_region is requested by the caller. (Init box to empty)
	*/
	expose_rect.width = 0;
	expose = expose_region ? &expose_rect : NULL;
	/*
	** Locate Font Cache (or create it on first call)
	*/
	for (cx.font_cache = font_cache_list; ;)
		if (cx.font_cache == NULL)
		    {
			cx.font_cache = (FontCache *)
				XtMalloc(sizeof(FontCache));
			cx.font_cache->cache_reset = TRUE;
			cx.font_cache->display = XtDisplay((Widget)w);
			cx.font_cache->next = font_cache_list;
			font_cache_list = cx.font_cache;
			break;
		    }
		else if (cx.font_cache->display == XtDisplay((Widget)w))
			break;
		else
			cx.font_cache = cx.font_cache->next;
	if (cx.font_cache->cache_reset)
	    {
		for (i = 0; i < XtNumber(cx.font_cache->data); i++)
			cx.font_cache->data[i].font = NULL;
		cx.font_cache->cache_hit = 0;
		cx.font_cache->cache_miss = 0;
		cx.font_cache->cache_reset = 0;
	    }
	if (set->columns != w->text.set.columns ||
	    set->width != w->text.set.width ||
	    set->height != w->text.set.height ||
	    set->column_width != w->text.set.column_width ||
	    set->column_height != w->text.set.column_height)
		from = NULL; /* Geometry changed, cannot do partial layout! */
	w->text.set = *set;
	cx.height = set->height - w->text.page_separation;
	cx.width = set->width - w->text.column_separation;
	if (cx.width <= 0)
		layout = False;
	cx.w = w;
	cx.lineSpacing = w->text.line_spacing;
#define FrameWidth(w,e) (w)->text.framed_rendition[e].value.space + \
	(w)->text.framed_rendition[e].value.width

	cx.top_framing = FrameWidth(w, FRAME_TOP);
	cx.bottom_framing = FrameWidth(w, FRAME_BOTTOM);
	cx.left_framing = FrameWidth(w, FRAME_LEFT);
	cx.right_framing = FrameWidth(w, FRAME_RIGHT);
#undef FrameWidth

	firstSeq = True;
	if (cx.lineSpacing == 0)
	    {
		/*
		** If lineSpacing == 0, then the default lineSpacing
		** will be the font height of the primary (default) font
		** using the default character set. Unfortunately, one
		** needs to do some complex stuff to get it...
		*/
		Snip *list = NULL;
                char alphabet[]="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
		s = _XeInsertSnip(&list);
		s->content.head =
			(SnipData *)XtCalloc(1, XtOffsetOf(SnipData, s[0]));
		s->data = &s->content.head->s[0];
		s->content.head->bytes = 1;
		s->content.head->refs = 1;
		s->content.head->font = NULL;
		s->content.head->character_set = XeTextCharsetDefault();
                s->data = alphabet;
                s->length = strlen(alphabet);

                /*!!!LOOOK HERE!!!!*/
		XeTextSnipExtents(&cx, s);
		/* ..assuming fontAscent and fontDescent are put into Snip */
                /*!!!LOOOK HERE!!!!*/
		cx.lineSpacing = s->descent + s->ascent;
		_XeDeleteSnip(&list);
	    }
	yHighMark = 0;
	cx.offset_x = cx.column_x =
		w->text.column_separation - w->text.column_separation / 2;
	cx.offset_y = cx.column_y =
		w->text.page_separation - w->text.page_separation / 2;
	/*
	** cx.height <= 0 is an indicator that no vertical paging
	** should occur. In that case use the largest value available for
	** 'end'.
	*/
	if (cx.height > 0)
		cx.column_end = cx.column_y + cx.height;
	else
		cx.column_end = INT_MAX;
	/*
	** Initialize layout parameters
	*/
	y = cx.column_y;
	x = cx.column_x + w->text.indentation;
	h = &w->text.first;
	tag = 0;
	lay = NULL;
	addLayout = FALSE;
	if (from == NULL)
	    {
		w->text.x = x;
		w->text.y = y;
		xHighMark = cx.offset_y;
		lookfor = LOOKFOR_NONE;
	    }
	else
	    {
		int gotcha, editable, base_y, max_y, min_y;
		/*
		** Skip over already laid out material, start layout from the
		** beginning of the previous line from the line has been
		** modified.
		*/
		for (s = from; ;s = s->next)
			if (s == NULL)
				return 0; /* Nothing to layout! */
			else if (!s->valid)
				break;
		/*
		** Search backwards for the line end marker from
		** which a new layout process can be started. It will
		** be the first line end having one of the following
		** conditions:
		** - line end is an editable hard line end, or
		** - line end is followed by valid layout which includes
		**   an editable space, or
		** - line end is followed by a full valid line which
		**   includes at least one valid editable component.
		*/
		for (gotcha = FALSE, editable = 0;;)
		    {
			if (s->back == &w->text.first)
				break;
			s = PreviousSnip(s);
			if (!s->valid)
			    {
				gotcha = FALSE;
				editable = 0;
				continue;
			    }
			if (IsEditableContent(s->mode.bits))
			    {
				editable += 1;
				if (s->space)
					gotcha = TRUE;
				else if (HasEndLine(s))
					break;
			    }
			else if (HasEndLine(s))
				if (gotcha && editable)
					break;
				else
				    {
					gotcha = TRUE;
					editable = 0;
				    }
		    }
		/*
		** Now 's' points to the Snip (HasEndLine) *AFTER* which
		** the new layout will start. If the proportional spacing has
		** has been requested, we have to "guess" the forwardExtent
		** of the preceding line is, and set previousExtent from it.
		** If the line is has only white space or no proportional
		** spacing, use formula "lineSpacing / 3".
		** Additionally, have to find out the initial value for the
		** previous 'tag'.
		*/
		p = s;
		gotcha = FALSE;
		base_y = max_y = min_y = p->y;
		do
		    {
			if (!gotcha && IsEditableContent(p->mode.bits))
			    {
				tag = p->mode.tag;
				gotcha = TRUE;
			    }
			if (!p->floating)
			    {
				if (p->y + p->descent > max_y)
					max_y = p->y + p->descent;
				if (p->y - p->ascent < min_y)
					min_y = p->y - p->ascent;
			    }
			if (p->back == &w->text.first)
				break;
			p = PreviousSnip(p);
		    } while (!HasEndLine(p));
		if (w->text.proportional && (max_y - min_y) > 0)
			cx.previousExtent = max_y - base_y;
		else
			cx.previousExtent = cx.lineSpacing / 3;
		/*
	        ** Do we have the 'tag'? If not, search more backwards...
		*/
		if (!gotcha)
		    {
			while (!IsEditableContent(p->mode.bits) &&
			       p->back != &w->text.first)
				p = PreviousSnip(p);
			tag = p->mode.tag;
		    }
		if (s->back == &w->text.first)
		    {
			tag = 0;
			cx.previousExtent = 0;
		    }
		else
		    {
			yHighMark = y = s->y;
			cx.page = s->y / set->column_height;
			cx.column_y = cx.page *
				(cx.height + w->text.page_separation)
					+ cx.offset_y;
			if (cx.height > 0)
				cx.column_end = cx.column_y + cx.height;
			cx.column = s->column;
			cx.column_x = cx.column *
				(cx.width + w->text.column_separation)
					+ cx.offset_x;
			firstSeq = HasEndParagraph(s);
			h = &s->next;
		    }
		/*
		** If the starting point is application generated layout
		** information, the actual layout process must start in
		** "add layout" mode until next editable content. Also,
		** assume the tag change has already been processed at
		** this point.
		*/
		if (!IsEditableContent(s->mode.bits) &&
		    s->mode.tag != LAYOUT_CONTENT_TAG)
			for (p = s; (p = p->next) != NULL; )
				if (IsEditableContent(p->mode.bits))
				    {
					lay = p;
					addLayout = -1;
					tag = p->mode.tag;
					break;
				    }
		xHighMark= w->text.w - w->text.column_separation + cx.offset_x;
		lookfor = LOOKFOR_MODE;
	    }
	p = cx.beginLine = NULL;
	while (*h)
	    {
		static SequenceState initial = {0,0,0,0};

		Snip *space = NULL;	/* Set Non-NULL, if last space found */

		cx.current = initial;	/* Initialize 'current' state */
		cx.current.tag = tag;
		cx.firstSnip = *h;
		if (cx.beginLine == NULL)
		    {
			cx.beginLine = cx.firstSnip;
			cx.forwardExtent = 0;
			cx.backwardExtent = 0;
			itemization = FALSE;
			expect = expect_point;
			cx.lineOffset = cx.column_x + w->text.indentation;
			if (firstSeq && w->text.itemization &&
			    w->text.itemization->identifier_alignment !=
			    XeAlignment_NONE)
				sequence = Sequence_ITEM;
			else
			    {
				sequence = Sequence_TEXT;
				if (firstSeq)
					cx.lineOffset +=
						w->text.first_line_offset;
			    }
			firstSeq = False;
			if (w->composite.num_children > 0)
			    {
				FindInsetIndents(&cx, y + cx.previousExtent);
				cx.lineOffset += cx.leftWidgetIndent;
			    }
		    }
		cx.yShift = 0;
		x = cx.lineOffset;
		cx.lineLength = cx.column_x + cx.width
			- x - cx.rightWidgetIndent;
		do
		    {
			/*
			** Locate the next Snip to process. Delete all
			** Snips generated by the previous layout process.
			** If the content ends without a proper endseq
			** Snip, then add one on behalf of the layout process.
			*/
			while (1)
			    {
				if ((s = *h) == NULL)
				    {
					/*
					** Have run into the end of content
					** while deleting layout information,
					** if the last Snip in already laid
					** out content is endseq, all is done.
					*/
					if (p && p->endseq)
						goto end_content;
					s = _XeInsertSnip(h);
					s->endseq = Snip_End;
					/*
					** Copy mode from some previous Snip,
					** if any available (aligment!).
					*/
					if (p)
						s->mode = p->mode;
					s->mode.bits &= ~Content_MASK;
					s->mode.bits |= Content_FORMATTED;
					s->mode.tag = LAYOUT_CONTENT_TAG;
					if (cx.firstSnip == NULL)
						cx.firstSnip = s;
					if (cx.beginLine == NULL)
						cx.beginLine = s;
					break;
				    }
				else if (addLayout)
					break;
				else if (IsLayoutContent(s->mode.bits))
				    {
					if (s == cx.firstSnip)
						cx.firstSnip = s->next;
					if (s == cx.beginLine)
						cx.beginLine = s->next;
					if (expose && s->layout && s->xWidth)
						UpdateExposeArea
							(expose,
							 s->x,
							 s->y - s->ascent,
							 s->xWidth,
							 s->ascent+s->descent);
					_XeDeleteSnip(h);
				    }
				else if (tag != s->mode.tag &&
					 w->text.layout_callbacks)
				    {
					XeTextLayoutCallbackData data;

					data.reason = XeCR_LAYOUT_CHANGE_TAG;
					data.previous = tag;
					data.current = s->mode.tag;
					data.context =
						_XeMakeLayoutContext(w, h);
					data.feed = _XeTextFeedContent;
					lay = s;
					addLayout = TRUE;
					cx.current.tag = tag = s->mode.tag;
					XtCallCallbackList
						((Widget)w,
						 w->text.layout_callbacks,
						 (XtPointer *)&data);
					_XeTextEndContent
						(data.context, (Region)0,
						 (long *)NULL);
					if (s == cx.firstSnip)
						cx.firstSnip = *h;
					if (s == cx.beginLine)
						cx.beginLine = *h;
					s = *h;
					break;
				    }
				else
				    {
					cx.current.tag = tag = s->mode.tag;
					break;
				    }
			    }
			if (s == cx.beginLine && w->text.layout_callbacks &&
			    addLayout >= 0)
			    {
				XeTextLayoutCallbackData data;

				data.reason = XeCR_LAYOUT_BEGIN_LINE;
				data.previous = data.current = tag;
				data.context = _XeMakeLayoutContext(w, h);
				data.feed = _XeTextFeedContent;
				if (!addLayout)
				    {
					lay = s;
					addLayout = TRUE;
				    }
				XtCallCallbackList
					((Widget)w, w->text.layout_callbacks,
					 (XtPointer *)&data);
				_XeTextEndContent
					(data.context,(Region)0,(long *)NULL);
				s = cx.beginLine = cx.firstSnip = *h;
			    }
			if (lay == s)
			    {
				/* Layout Insertion terminated */
				addLayout = FALSE;
				lay = NULL;
				tag = cx.current.tag = s->mode.tag;
			    }
			if (expect == expect_end)
			    {
				ExpectEntry *e = expect_point;
				int old = expect_end - expect_point;
				int new = old + XtNumber(expect_default);
				if (expect_point == expect_default)
				    {
					expect_point = (ExpectEntry *)
						XtMalloc(sizeof(ExpectEntry) *
							 new);
					memcpy((void *)expect_point, (void *)e,
					       sizeof(ExpectEntry) * old);
				    }
				else
					expect_point = (ExpectEntry *)
						XtRealloc((char *)expect_point,
							  sizeof(ExpectEntry) *
							  new);
				expect = &expect_point[old];
				expect_end = &expect_point[new];
			    }
			expect->s = s;
			expect->x = s->x;
			expect->y = s->y;
			expect->xWidth = s->xWidth;
			expect->ascent = s->ascent;
			expect->descent = s->descent;
			expect->offset = s->offset;
			expect->column = s->column;
			expect++;
			s->x = x;
			s->y = y;
			s->column = cx.column;
			/*
			** All snips before the current one (s) will
			** fit into the line. If the current one
			** represents a valid breaking point, remember
			** this point by saving the line state into
			** 'lastSpace' *before* adding any effects
			** from this snip. (used by TEXT sequence only)
			*/
			if (s->space)
			    {
				cx.lastSpace = cx.current;
				space = s;
				cx.current.spaces += 1;
			    }
			ComputeWidth(&cx, s);
			if (sequence == Sequence_TEXT)
			    {
				if (cx.current.wNow > cx.lineLength)
				    {
					if (space &&
					    (layout ||
					     (Justify_WORD_FILL &
					      space->mode.bits)))
					    {
						s = SplitLine
						    (&cx,&cx.lastSpace,space);
						tag = cx.lastSpace.tag;
					    }
					else if (cx.rightWidgetIndent >
						 cx.rightWidgetIndentLine ||
						 cx.leftWidgetIndent > 0)
					    {
						s = SkipLine(&cx,cx.beginLine);
						cx.beginLine=cx.firstSnip=s;
					    }
					else if (s->endseq)
						s = Position_TEXT
							(&cx,&cx.current,s,s);
				    }
				else if (s->endseq)
					s = Position_TEXT(&cx,&cx.current,s,s);
			    }
			else if (s->endseq)
				if (sequence == Sequence_ITEM)
				    {
					s = Position_ITEM(&cx, &cx.current, s);
					itemization = TRUE;
					sequence = Sequence_TEXT;
				    }
				else
					s = Position_STAB
						(&cx, &cx.current, s, tabref);
			y = s->y - cx.yShift;
			x = s->x + s->xWidth;
			h = &s->next;
			p = s;
		    } while (!s->endseq);
		if (s->widget) /* WARNING! Works because endseq in insets! */
			x = AdjustForWidgets(&cx, s);

		if (x > xHighMark)
			xHighMark = x;
		if (HasEndLine(s))
		    {
			if (CloseLineBox(&cx, s, &x, &y))
			    {
				/*
				** The line box must be relaid into a new
				** position. Use UpdateExpose just to restore
				** the original layout info back to snips.
				*/
				UpdateExpose((XRectangle *)NULL,
					     expect_point, expect,
					     expect_point->s, 0);
				h = cx.beginLine->back;
				cx.previousExtent = 0;
				cx.beginLine = NULL;
				y = cx.column_y;
				/* need to set firstSeq? */
				continue;
			    }
			lookfor = UpdateExpose(expose, expect_point, expect,
					       s->next, lookfor);
			if (y > yHighMark)
				yHighMark = y;
			if (!lookfor)
				goto clean_up;
			/*
			** The next line will start a new line box
			*/
			firstSeq = HasEndParagraph(s);
			cx.beginLine = NULL;
		    }
		else if (y > yHighMark)
			yHighMark = y;
		sequence = Sequence_TEXT;
		if (s->tab)
			if ((tabref = s->tabref) == Snip_NEXTTABSTOP)
			    {
				x = HorizontalTab(&cx, x, &tabref);
				if (tabref != Snip_NEXTTABSTOP)
					sequence = Sequence_STAB;
			    }
			else
				sequence = Sequence_STAB;
		/*
		** NOTE:If next sequence starts a linebox (beginLine == NULL)
		**	then lineOffset will be reset and settings below are
		**	ignored.
		*/
		/*
		** If the sequence just ended was an 'item' then
		** the next sequence should start from the home
		** position adjusted with the first_line_offset.
		**
		** Additional Non-ODA(?) implementation: if the
		** item would overlap the normal text, then do not
		** use the fixed start (instead, just continue
		** filling the line from where the item ended).
		*/
		if (itemization)
		    {
			int offset = cx.column_x + w->text.indentation
				+ w->text.first_line_offset;

			if (x < offset)
				x = offset;
			itemization = False;
		    }
		cx.lineOffset = x;
	    }
    end_content:
	/*
	** Layout process has reached the end of the content
	*/
	if (p && HasEndLine(p))
	    {
		/*
		** The content terminated by End of Line. This means that
		** the content includes one *EMPTY* line at end. (This is
		** ODA character content interpretation).
		*/
		y += cx.previousExtent + cx.lineSpacing;
	    }
	else if (cx.beginLine)
	    {
		cx.column_y = y; /* Kludge! Disables "paging" branch! */
		(void)CloseLineBox(&cx, p, &x, &y);
		(void)UpdateExpose(expose, expect_point, expect, p->next, 0);
		y += cx.previousExtent;
	    }
	else
		y += cx.lineSpacing; /* Empty Content! */
	if (y > yHighMark)
		yHighMark = y;
	/*
	** Extend dimensions to cover the floating insets (if any)
	*/
	for (i = 0; i < w->composite.num_children; ++i)
	    {
		XeTextConstraints tc;
		Widget child = w->composite.children[i];

		if (!child)
			continue;
		tc = (XeTextConstraints)child->core.constraints;
		if (tc == NULL || (s = tc->text.inset) == NULL || !s->floating)
			continue;
		x = s->x + s->offset + tc->text.FE.width;
		y = s->y - s->ascent + tc->text.FE.height;
		if (xHighMark < x)
			xHighMark = x;
		if (yHighMark < y)
			yHighMark = y;
	    }
	w->text.h = yHighMark;
	w->text.w = xHighMark;
    clean_up:
	/*
	** Layout process jumps here directly, when it notices that it does
	** not need to lay out the all of the content and breaks out.
	*/
	xHighMark += w->text.column_separation - cx.offset_x;
	yHighMark += w->text.page_separation - cx.offset_y;
	if (xHighMark > w->text.w)
		w->text.w = xHighMark;
	if (yHighMark > w->text.h)
		w->text.h = yHighMark;
	if (expose && expose->width > 0)
	    {
		/*
		** .. should not really use XRectangle for expose in layout
		** phase, the coordinates may wrap over, fix someday! --msa
		*/
		expose->x += w->basic.x_translation;
		expose->y += w->basic.y_translation;
		XUnionRectWithRegion(expose, expose_region, expose_region);
	    }
	if (expect_point != expect_default)
		XtFree((char *)expect_point);
	return 0;
    }
