/*
** Copyright 1992-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>

extern int atoi( /* No header for this in SYSV */
#if NeedFunctionPrototypes
		char *
#endif
		);

#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <ctype.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xew/XewInit.h>
#include <X11/Xew/TextP.h>

#include "TextImport.h"
#include "TextLayout.h"
#include "TextFont.h"
#include "FrameSupport.h"
#include "Color.h"

#define TEXT_MIN_WIDTH 1
#define TEXT_MIN_HEIGHT 1

#ifndef INT_MAX
#	define INT_MAX (~((unsigned)0) >> 1)
#endif

static void
	Highlight(), Unhighlight();

static XtActionsRec actions[] =
     {
	{"highlight",	Highlight},
	{"unhighlight",	Unhighlight},
     };

static XeFramePosition init_FramePosition = {XeFramePositionType_INLINE};
static XeFrameDimension init_FrameDimension;
static XeFrameBorder init_FrameBorder;
static XeFrameAlignment init_FrameAlignment = XeFrameAlignment_ABOVE;
static XeFrameBorder framed_FrameBorder = {XeBorderType_SOLID,1,1};
static int init_Columns = 1;

#define offset(field) XtOffsetOf(XeTextRec, text.field)

static XtResource resources[] =
    {
	{
	    XtNalignment, XtCAlignment,
	    XtRXeAlignment, sizeof(XeAlignment), offset(alignment),
	    XtRImmediate, (XtPointer)XeAlignment_START,
	},
	{
	    XtNexportFormat, XtCExportFormat,
	    XtRXeTextExport, sizeof(XeTextExport), offset(export_format),
	    XtRImmediate, (XtPointer)XeTextExport_ODIF,
	},
	{XtNfirstLineOffset, XtCOffset, XtRInt, sizeof(int),
		 offset(first_line_offset), XtRImmediate, 0},
	{XtNfonts, XtCFonts, XtRXeFonts, sizeof(XeFont *),
	     offset(fonts), XtRImmediate, (XtPointer)NULL},
	{XtNformat, XtCFormat, XtRBoolean, sizeof(Boolean),
	     offset(format), XtRImmediate, (XtPointer)0},
	{XtNframeBottomRendition, XtCFrameRendition, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(framed_rendition[FRAME_BOTTOM]),
		 XeRFrameBorder, (XtPointer)&framed_FrameBorder},
	{XtNframeBottomContrast, XtCFrameBottomContrast, XtRInt, sizeof(int),
		 offset(shadow_contrast[FRAME_BOTTOM]),
		 XtRImmediate, (XtPointer)40},
	{XtNframeInvertShadow, XtCFrameInvertShadow,XtRBoolean,sizeof(Boolean),
		 offset(invert_shadow), XtRImmediate, (XtPointer)False},
	{XtNframeLeftRendition, XtCFrameRendition, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(framed_rendition[FRAME_LEFT]),
		 XeRFrameBorder, (XtPointer)&framed_FrameBorder},
	{XtNframeRightRendition, XtCFrameRendition, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(framed_rendition[FRAME_RIGHT]),
		 XeRFrameBorder, (XtPointer)&framed_FrameBorder},
	{XtNframeTopRendition, XtCFrameRendition, XeRFrameBorder,
		 sizeof(XeFrameBorder), offset(framed_rendition[FRAME_TOP]),
		 XeRFrameBorder, (XtPointer)&framed_FrameBorder},
	{XtNframeTopContrast, XtCFrameTopContrast, XtRInt, sizeof(int),
		 offset(shadow_contrast[FRAME_TOP]),
		 XtRImmediate,(XtPointer)20},
	{XtNgraphicRendition, XtCRendition, XtRString, sizeof(String),
		 offset(graphic_rendition), XtRImmediate, (XtPointer)NULL},
	{XtNindentation, XtCIndentation, XtRInt, sizeof(int),
		 offset(indentation), XtRImmediate, 0 },
	{XtNcolumnHeight, XtCColumnHeight, XtRInt, sizeof(int),
		 offset(column_height), XtRImmediate, 0 },
	{XtNcolumns, XtCColumns, XtRInt, sizeof(int),
		 offset(columns), XtRInt, (XtPointer)&init_Columns },
	{XtNcolumnSeparation, XtCColumnSeparation, XtRInt, sizeof(int),
		 offset(column_separation), XtRImmediate, 0 },
	{XtNcolumnWidth, XtCColumnWidth, XtRInt, sizeof(int),
		 offset(column_width), XtRImmediate, 0 },
	{XtNinitialState, XtCSequence,  XtRString, sizeof(String),
		 offset(initial_state), XtRImmediate, (XtPointer)NULL},
	{XtNitemization,XtCItemization, XtRXeItemization,
		 sizeof(XeItemization *),
		 offset(itemization), XtRImmediate, (XtPointer)NULL},
	{XtNkerningOffset, XtCKerningOffset, XtRKerningOffset,
		 sizeof(Dimension *),
		 offset(kerning_offset), XtRImmediate, (XtPointer)NULL},
	{XtNlayoutCallback, XtCLayoutCallback, XtRCallback, sizeof(XtPointer),
		 offset(layout_callbacks), XtRCallback, (XtPointer)NULL},
	{XtNlineLayoutTable, XtCLineLayoutTable, XtRXeLineLayoutTable,
		 sizeof(XeTabStop *),
		 offset(line_layout_table), XtRImmediate, (XtPointer)NULL},
	{XtNlineSpacing, XtCLineSpacing, XtRInt, sizeof(int),
		 offset(line_spacing), XtRImmediate, 0},
	{XtNpageSeparation, XtCPageSeparation, XtRInt, sizeof(int),
		 offset(page_separation), XtRImmediate, 0},
	{XtNproportional, XtCProportional, XtRBoolean, sizeof(Boolean),
	     offset(proportional), XtRImmediate, (XtPointer)0},
    };
#undef offset

/* using 'OFFSET' instead of 'offset', because offset is a field also! */

#define OFFSET(field) XtOffsetOf(XeFrameConstraintsRec, frame.field)

static XtResource subresources[] =
    {
	{XtNbottomShadowContrast, XtCBottomShadowContrast, XtRInt, sizeof(int),
		 OFFSET(shadow_contrast[FRAME_BOTTOM]),
		 XtRImmediate,(XtPointer)40},
	{XtNframeAlignment, XtCFrameAlignment, XeRFrameAlignment,
		 sizeof(XeFrameAlignment), OFFSET(alignment),
		 XeRFrameAlignment, (XtPointer)&init_FrameAlignment},
	{XtNframePosition, XtCFramePosition, XeRFramePosition,
		 sizeof(XeFramePosition), OFFSET(position),
		 XeRFramePosition, (XtPointer)&init_FramePosition},
	{XtNhorizontal, XtCFrameDimension, XeRFrameDimension,
		 sizeof(XeFrameDimension), OFFSET(dimension[FRAME_HORIZONTAL]),
		 XeRFrameDimension, (XtPointer)&init_FrameDimension},
	{XtNinvertShadow, XtCInvertShadow, XtRBoolean, sizeof(Boolean),
		 OFFSET(invert_shadow), XtRImmediate, (XtPointer)False},
	{XtNleadingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_LEADING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNleadingOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_LEADING]), XtRImmediate, 0},
	{XtNleadingSeparation, XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_LEADING]),XtRImmediate,0},
	{XtNleftBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_LEFT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNleftOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_LEFT]), XtRImmediate, 0},
	{XtNleftSeparation, XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_LEFT]),XtRImmediate,0},
	{XtNrightBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_RIGHT]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNrightOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_RIGHT]), XtRImmediate, 0},
	{XtNrightSeparation, XtCFrameSeparation, XtRInt,
		 sizeof(int),OFFSET(separation[FRAME_RIGHT]),XtRImmediate,0},
	{XtNtopShadowContrast, XtCTopShadowContrast, XtRInt, sizeof(int),
		 OFFSET(shadow_contrast[FRAME_TOP]),
		 XtRImmediate,(XtPointer)20},
	{XtNtrailingBorder, XtCFrameBorder, XeRFrameBorder,
		 sizeof(XeFrameBorder), OFFSET(border[FRAME_TRAILING]),
		 XeRFrameBorder, (XtPointer)&init_FrameBorder},
	{XtNtrailingOffset, XtCFrameOffset, XtRInt,
		 sizeof(int), OFFSET(offset[FRAME_TRAILING]), XtRImmediate, 0},
	{XtNtrailingSeparation, XtCFrameSeparation, XtRInt, sizeof(int),
		 OFFSET(separation[FRAME_TRAILING]),XtRImmediate,0},
	{XtNvertical, XtCFrameDimension, XeRFrameDimension,
		 sizeof(XeFrameDimension), OFFSET(dimension[FRAME_VERTICAL]),
		 XeRFrameDimension, (XtPointer)&init_FrameDimension},
    };

#undef OFFSET

#if NeedFunctionPrototypes
static void FreeColors(XeTextWidget);
#else
static void FreeColors();
#endif

static void ClassPartInitialize(), ClassInitialize();
static void Initialize(), Realize(), Redisplay(), Resize(), Destroy();
static Boolean SetValues(), FrameSetValues();
static XtGeometryResult QueryGeometry();
static XtGeometryResult GeometryManager();
static void InsertChild(), DeleteChild(), ConstraintDestroy(), ChangeManaged();

static void PrintInitialize();
static void PrintCancel(), PrintDestroy();
static XtPointer Print();
static void PrintConstraintInitialize(), PrintConstraintDestroy();

/*
** REDISPLAY
**	Call widget instance expose method. As Text may be subclassed, this
**	method is not necessarily the 'Redisplay' function on this module.
*/
#define REDISPLAY(w,e,r) \
	(*(w)->core.widget_class->core_class.expose) \
	((Widget)(w), (XEvent *)(e), (Region)(r))

#ifdef USING_MOTIF_122
static XmNavigability WidgetNavigable() ;

/*******************************************/
/*  Declaration of class extension records */

XmBaseClassExtRec _XeTextbaseClassExtRec = {
    NULL,
    NULLQUARK,
    XmBaseClassExtVersion,
    sizeof(XmBaseClassExtRec),
    NULL,				/* InitializePrehook	*/
    NULL,				/* SetValuesPrehook	*/
    NULL,				/* InitializePosthook	*/
    NULL,				/* SetValuesPosthook	*/
    NULL,				/* secondaryObjectClass	*/
    NULL,				/* secondaryCreate	*/
    NULL,		                /* getSecRes data	*/
    { 0 },				/* fastSubclass flags	*/
    NULL,				/* get_values_prehook	*/
    NULL,				/* get_values_posthook	*/
    NULL,                               /* classPartInitPrehook */
    NULL,                               /* classPartInitPosthook*/
    NULL,                               /* ext_resources        */
    NULL,                               /* compiled_ext_resources*/
    0,                                  /* num_ext_resources    */
    FALSE,                              /* use_sub_resources    */
    WidgetNavigable,                    /* widgetNavigable      */
    NULL,                               /* focusChange          */
};
#endif

#define SuperClass (&xeBasicClassRec)

XeTextClassRec xeTextClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) SuperClass,
    /* class_name		*/	"XeText",
    /* widget_size		*/	sizeof(XeTextRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	ClassPartInitialize,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	XtExposeCompressMultiple,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	XtInheritTranslations,
    /* query_geometry		*/	QueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
#ifdef USING_MOTIF_122
    /* extension		*/	(XtPointer)&_XeTextbaseClassExtRec,
#else
    /* extension		*/	NULL,
#endif
  },
  { /* composite class fields	*/
    /* geometry_manager		*/	GeometryManager,
    /* change_managed		*/	ChangeManaged,
    /* insert_child		*/	InsertChild,
    /* delete_child		*/	DeleteChild,
    /* extension		*/	NULL
  },
  { /* constraint class fields	*/
    /* subresources		*/	subresources,
    /* subresource_count	*/	XtNumber(subresources),
    /* constraint_size		*/	sizeof(XeTextConstraintsRec),
    /* initialize		*/	_XeFrameInitialize,
    /* destroy			*/	ConstraintDestroy,
    /* set_values		*/	FrameSetValues,
    /* extension		*/	NULL
  },
#ifdef USING_MOTIF_122
  { /* manager class record	*/
    /* translations		*/	XtInheritTranslations,
    /* syn_resources		*/	NULL,
    /* num_syn_resources	*/	0,
    /* syn_constraint_resources	*/	NULL,
    /* num_syn_constraint_resources */	0,
    /* parent_process		*/	XmInheritParentProcess,
    /* extension		*/	NULL  
  },
#endif
  { /* basic fields */
    /* print_initialize		*/	PrintInitialize,
    /* print			*/	Print,
    /* print_cancel		*/	PrintCancel,
    /* print_destroy		*/	PrintDestroy,
    /* print_constraint_initialize */	PrintConstraintInitialize,
    /* print_constraint_destroy */	PrintConstraintDestroy

  },
  { /* XeText fields */
    /* not used			*/	0
  }
};

WidgetClass xeTextWidgetClass = (WidgetClass)&xeTextClassRec;


static CvtEnumInfo cvtAlignment[] =
    {
	{"none",	XeAlignment_NONE },
	{"start",	XeAlignment_START },
	{"end",		XeAlignment_END },
	{"center",	XeAlignment_CENTER },
	{"justified",	XeAlignment_JUSTIFIED },
	{"around",	XeAlignment_AROUND },
	{NULL,		sizeof(XeAlignment) },
    };

static CvtEnumInfo cvtExportFormat[] =
    {
	{"string",	XeTextExport_STRING },
	{"string-f",	XeTextExport_STRING_F },
	{"odif",	XeTextExport_ODIF },
	{"odif-f",	XeTextExport_ODIF_F },
	{"odif-fp",	XeTextExport_ODIF_FP },
	{NULL,		sizeof(XeTextExport) },
    };


#define	CvtDone(type, value)				\
    do {						\
	if (toVal->addr != NULL)			\
	    {						\
		if (toVal->size < sizeof(type))		\
		    {					\
			toVal->size = sizeof(type);	\
			return False;			\
		    }					\
		*(type*)(toVal->addr) = (value);	\
	    }						\
	else						\
	    {						\
		static type static_val;			\
		static_val = (value);			\
		toVal->addr = (XPointer)&static_val;	\
	    }						\
	toVal->size = sizeof(type);			\
	return True;					\
    } while (0)

/*
** XeCvtStringToFonts
**	Convert a string into Fonts (XeFont *)
**
** <fonts>	::= <font> [ ";" <fonts> ]
** <font>	::= <size> [ <fontlist> ]
** <fontlist>	::= <fontstring> [ "=" <characterset> ]
** <fontstring>	::= <Any X11 font specification string>
** <size>	::= <positive integer>
** <characterset> ::= <Any supported X11 characterset name>
*/
static Boolean XeCvtStringToFonts
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeFont *f;
	XeFont fonts[10];
	char *s = (char *) fromVal->addr;
	char *r;
	int i, j;
	XeFontList font_list = NULL;

	for (i = 0; i < XtNumber(fonts); )
	    {
		/* Extract size */

		while (isspace(*s))
			s += 1;
		fonts[i].font_size = atoi(s);
		while (isdigit(*s))
			s += 1;
		while (isspace(*s))
			s += 1;
		/*
		** Extract font list. If the font list portion is
	        ** omitted, use the same list as was specified to
		** the previous entry
		*/
		r = strchr(s, ';');
		j = r ? r - s : strlen(s);
		if (j > 0)
			font_list = XeMakeFontList(s, j);
		fonts[i].font_list = font_list;
		i += 1;
		if (r == NULL)
			break;
		s = r + 1;
	    }
	/*
	** Allocate space for all entries and for a sentinel. Initialize.
	*/
	f = (XeFont *)XtMalloc((i+1) * sizeof(XeFont));
	memcpy((char *)f, (char *)fonts, i * sizeof(XeFont));
	f[i].font_size = 0;
	f[i].font_list = 0;
	CvtDone(XeFont *, f);
    }

/*
** XeItemizationDestructor
*/
static void XeItemizationDestructor(app, to, converter_data, args, num_args)
XtAppContext app;
XrmValue *to;
XtPointer converter_data;
XrmValue *args;
Cardinal *num_args;
    {
	XtFree((char *)converter_data);
    }

/*
** XeCvtStringToItemization
**	Convert a string into Itemization (XeItemization).
**
** <itemization>::= <alignment> [ "," [ <start> ] ["," <end>]]
** <alignment>	::= "no" | "start" | "end"
** <start>	::= <integer>
** <end>	::= <integer>
*/
Boolean XeCvtStringToItemization
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeItemization itemization;
	int need_allocate = True;
	char *s = (char *) fromVal->addr;
	char *r;
	char buf[100];
	int len;
	XrmValue toTmp, fromTmp;
	String params[1];
	Cardinal num_params;

	if (*num_args != 1)
	    {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"wrongParameters","XeCvtStringToItemization",
				"XewError",
		       "String to XeItemization conversion needs one argument",
				(String *)NULL, (Cardinal *)NULL);
		return False;
	    }
	if (converter_data == NULL)
	    {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"noConverterData","XeCvtStringToItemization",
				"XewError",
		       "String to XeItemization needs converter_data",
				(String *)NULL, (Cardinal *)NULL);
		return False;
	    }
	/*
	** Initialize default values
	*/
	itemization.identifier_alignment = XeAlignment_START;
	itemization.identifier_start_offset = - *((Dimension *) args[0].addr);
	itemization.identifier_end_offset = 0;
	for ( ;; ) /* Just to enable us to use 'break' below */
	    {
		/*
		** Extract <Alignment>
		*/
		if (s == NULL)
			break;
		r = strchr(s, ',');
		len = r ? r - s : strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		if (strcmp("no",buf) == 0 || strcmp("none", buf) == 0)
		    {
			need_allocate = False;
			break;
		    }
		else if (strcmp(buf, "start") == 0)
			itemization.identifier_alignment = XeAlignment_START;
		else if (strcmp(buf, "end") == 0)
			itemization.identifier_alignment = XeAlignment_END;
		else
		    {
			params[0] = (String)buf;
			num_params = 1;
			XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
					"XeCvtStringToItemization",
					"unknownKeyword",
					"XewError",
			       "Cannot convert \"%s\" to identifier alignment",
					params,
					&num_params);
			return False;
		    }
		if (r == NULL)
			break;
		/*
		** Extract <start>
		*/
		s = r + 1;
		r = strchr(s, ',');
		len = r ? r - s : strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		fromTmp.size = len + 1;
		fromTmp.addr = (XtPointer)buf;
		toTmp.size = sizeof(itemization.identifier_start_offset);
		toTmp.addr = (XtPointer)&itemization.identifier_start_offset;
		if (!XtCallConverter(dpy, XeCvtStringToHorizontalValue,
				     (XrmValue *)NULL, 0,
				     &fromTmp,
				     &toTmp,
				     (XtCacheRef *)NULL))
			return False;
		if (r == NULL)
			break;
		/*
		** Extract <end>
		*/
		s = r + 1;
		len = strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		fromTmp.size = len + 1;
		fromTmp.addr = (XtPointer)buf;
		toTmp.size = sizeof(itemization.identifier_end_offset);
		toTmp.addr = (XtPointer)&itemization.identifier_end_offset;
		if (!XtCallConverter(dpy, XeCvtStringToHorizontalValue,
				     (XrmValue *)NULL, 0,
				     &fromTmp,
				     &toTmp,
				     (XtCacheRef *)NULL))
			return False;
		break; /* This was not really a loop! Fall through.. */
	    }

	/* Cannot use macro CvtDone because the memory should be
	   allocated to Itemization only on succesful return... --msa */

	if (toVal->addr != NULL)
	    {
		if (toVal->size < sizeof(XeItemization *))
		    {
			toVal->size = sizeof(XeItemization *);
			return False;
		    }
	    }
	else
	    {
		static XeItemization *value;
		toVal->addr = (XtPointer)&value;
	    }
	if (need_allocate)
	    {
		*converter_data = XtMalloc(sizeof(XeItemization));
		**(XeItemization **)converter_data = itemization;
	    }
	else
		*converter_data = NULL;
	*(XeItemization **)(toVal->addr) = (XeItemization *)(*converter_data);
	toVal->size = sizeof(XeItemization *);
	return True;
    }

/*
** ExtractString
**	Extract a string terminated either by ',', ';', '#' or NUL.
**
**	Ignore leading and trailing spaces. Returns a pointer to the
**	terminating character.
**
** (This function is copied from FrameSupport.c, should use same eventually)
*/
static char *ExtractString(s, str, len)
char *s, *str;
int len;
    {
	int i;
	
	if (len && s)
	    {
		while (isspace(*s))
			++s;
		if (*s == '"')
		    {
			/* Quoted string */
			i = 0;
			while (*++s)
				if (*s == '"')
				    {
					s++;
					break;
				    }
				else if (i < len)
					str[i++] = *s;
		    }			
		else
		    {
			/* Unquoted string token */

			for (i=0; *s!=',' && *s!=';' && *s!='#' && *s; ++s)
				if (i < len)
					str[i++] = *s;
			while (i > 0 && isspace(str[i-1]))
				i--;
		    }
		str[i < len ? i : len - 1] = 0;
	    }
	return s;
    }

/*
** XeLineLayoutTableDestructor
*/
static void XeLineLayoutTableDestructor
	(app, to, converter_data, args, num_args)
XtAppContext app;
XrmValue *to;
XtPointer converter_data;
XrmValue *args;
Cardinal *num_args;
    {
	XtFree((char *)converter_data);
    }
/*
** XeCvtStringToLineLayoutTable
**	Convert a string into LineLayoutTable (XeTabStop *).
**
** <table>	::= <tabstop> [ ";" <table> ]
** <tabstop>	::= <position> ["#" <reference> ]
**			[ "," <alignment> [ "," <around> ]]
**	-- If hte <reference> is omitted, it will be implicitly
**	--  defaulted to the sequence number of the tab (first is 1).
** <reference>	::= <string of one to four digits>
** <positition>	::= <non-negative-integer>
** <alignment>	::= "start" | "end" | "center" | "around"
** <around>	::= <string of characters not including ";">
*/
static Boolean XeCvtStringToLineLayoutTable
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	static XeTabStop init_stop = {0, 0, XeAlignment_START, };

	XeTabStop *tabs = NULL;
	char *s = (char *)fromVal->addr, *r;
	char str[20];
	int n, i;

	if (s != NULL)
	    {
		/*
		** Make a guess about the number of tab stops by counting
		** the number of ';' characters in a string (this can only
		** be too large, if the <around> strings contain ';'s,
		** otherwise the estimate is exact).
		*/
		for (n = 1, r = s; (r = strchr(r, ';')) != NULL; ++r, ++n)
			/* Nothing here */;
		tabs = (XeTabStop *)XtMalloc((n+1) * sizeof(XeTabStop));
		for (i = 0;;)
		    {
			tabs[i] = init_stop;

			/* <position> string is supposed to be a number */
			s = ExtractString(s, str, XtNumber(str));
			tabs[i].position = atoi(str);
			if (*s == '#')
			    {
				/* <reference> */
				s = ExtractString(s + 1, str, XtNumber(str));
				tabs[i].reference = atoi(str);
			    }
			else
				tabs[i].reference = i + 1;
			if (*s == ',')
			    {
				/* <alignment> */
				CvtEnumInfo *list = cvtAlignment;

				s = ExtractString(s + 1, str, XtNumber(str));
				XmuCopyISOLatin1Lowered(str, str);
				for (;list->name != NULL; list++)
					if (strcmp(str, list->name) == 0)
					    {
						tabs[i].alignment =
							list->value;
						break;
					    }
			    }
			if (*s == ',')
			    {
				/* <around> */
				s = ExtractString
					(s + 1,
					 tabs[i].alignment_string,
					 sizeof(tabs[i].alignment_string));
			    }
			i += 1;
			s = strchr(s, ';');
			if (s == NULL)
				break;
			s += 1;
		    }
		tabs[i].reference = -1; /* Terminator Entry */
	    }
	*converter_data = (XtPointer)tabs;
	/* This has the same problem as XeItemization. If CvtDone returns
	   with False, tabs is not released ever? This needs to be checked!
	   --msa */
	CvtDone(XeTabStop *, tabs);
    }

static void ClassInitialize()
    {

	static XtConvertArgRec convertArg[] =
	    {
		{
		    XtResourceString,
		    (XtPointer) XtNindentation,
		    sizeof(Dimension),
		}
	    };
	static XtConvertArgRec convertAlignmentArg[] =
	    {
		{
		    XtAddress,
		    (XtPointer) &cvtAlignment[0],
		    sizeof(XtPointer),
		}
	    };
	static XtConvertArgRec convertExportFormatArg[] =
	    {
		{
		    XtAddress,
		    (XtPointer) &cvtExportFormat[0],
		    sizeof(XtPointer),
		}
	    };


	XewInitializeWidgetSet();
	XtAddConverter(XtRString, XtRLong, XmuCvtStringToLong, NULL, 0);

	XtSetTypeConverter(XtRString, XtRXeAlignment,
			   XeCvtStringToEnum,
			   convertAlignmentArg, XtNumber(convertAlignmentArg),
			   XtCacheNone,
			   (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XtRXeTextExport,
			   XeCvtStringToEnum,
			   convertExportFormatArg,
			   XtNumber(convertExportFormatArg),
			   XtCacheNone,
			   (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XtRXeFonts,
			   XeCvtStringToFonts,
			   NULL, 0,
			   XtCacheAll,
			   (XtDestructor)NULL);
	/*
	** Note: Itemization needs to know the current indentation.
	** (? How do we guarantee, that indentation has been computed
	** at the time this converter is used? --msa
	*/
	XtSetTypeConverter(XtRString, XtRXeItemization,
			   XeCvtStringToItemization,
			   convertArg, 1,
			   XtCacheAll | XtCacheRefCount,
			   XeItemizationDestructor);
	XtSetTypeConverter(XtRString, XtRXeLineLayoutTable,
			   XeCvtStringToLineLayoutTable,
			   NULL, 0,
			   XtCacheAll | XtCacheRefCount,
			   XeLineLayoutTableDestructor);
	_XeFrameConverters();
#ifdef USING_MOTIF_122
        _XeTextbaseClassExtRec.record_type = XmQmotif;
#endif

    }

static void ClassPartInitialize(class)
WidgetClass class;
    {
    }

/*
** GetInsetGeometry
**	Get current geometry of Inset
*/
static int GetInsetGeometry(t, w, current)
XeTextWidget t;
Widget w;
XtWidgetGeometry *current;
    {
	int x, y, b2;
	Snip *inset;
	XeTextConstraints tc;

	tc = (XeTextConstraints)w->core.constraints;
	if (tc == NULL)
		return False;
	inset = tc->text.inset;
	if (inset == NULL)
		return False;
	if (tc->frame.position.type == XeFramePositionType_FIXED)
	    {
		x = tc->frame.position.x;
		y = tc->frame.position.y;
	    }
	else
	    {
		x = inset->x + inset->offset;
		y = inset->y - inset->ascent;
	    }
	x += t->basic.x_translation;
	y += t->basic.y_translation;
	/*
	** A quick fix: guess a correction for y, if the snip
	** appears to be framed and inline. A very specific
	** fix that can easily be broken by unwary changes.
	** Something more "algorithmic" needed. One should not need
	** this kind of detailed knowledge of the layout process
	** and algorithm used at this level!!!
	** --msa
	*/
	if (!inset->floating && (inset->mode.bits & Framing_MASK))
		y += t->text.framed_rendition[FRAME_TOP].value.space +
			t->text.framed_rendition[FRAME_TOP].value.width;
	/* end fix */
	b2 = 2 * w->core.border_width;
	current->width = tc->text.FE.window.width - b2;
	current->height = tc->text.FE.window.height - b2;
	current->x = x + tc->text.FE.window.x;
	current->y = y + tc->text.FE.window.y;
	current->request_mode = CWX | CWY | CWWidth | CWHeight;
	return True;
    }


/*
** _XeTextConfigureChildren
**	just configure child widgets. (Not happy with this.. --msa)
*/
void _XeTextConfigureChildren(t)
XeTextWidget t;
    {
	int i;
	Widget w;
	XtWidgetGeometry set;

	for (i = 0; i < t->composite.num_children; i++)
	    {
		w = t->composite.children[i];
		if (!XtIsManaged(w) || !GetInsetGeometry(t, w, &set))
			continue;
		if (t->text.printing)
		    {
			/*
			** If printing is set, assume we are really
			** working with the cloned tree. As currently
			** each cloned widget really has the same
			** window, doing reconfigure to it is not
			** the best thing... (especially, when
			** there might not be any real window!!)
			*/
			w->core.x = set.x;
			w->core.y = set.y;
			if (w->core.width != set.width ||
			    w->core.height != set.height)
			    {
				w->core.width = set.width;
				w->core.height = set.height;
				/*
				** Call resize method directly if size changed
				** (only if Xew widget, unknown widget resize
				** cannot be called, because it might cause
				** expose method being executed too..)
				*/
#ifndef USING_MOTIF_122
				/*
				** MOTIF has some strange 'ResizeWrapper'
				** around the resize method, and this seems
				** require 'per display information', which
				** 'printer display' does not have! Cannot
				** do anything else, but skip the resize
				** totally for MOTIF subclassed Xew...
				*/
				if (w->core.widget_class->core_class.resize &&
				    XtIsSubclass(w, xeBasicWidgetClass))
					(*w->core.widget_class->
					 core_class.resize)(w);
#endif
			    }
		    }
		else
			XtConfigureWidget
				(w, set.x, set.y,
				 set.width, set.height,
				 w->core.border_width);
	    }
    }

/*
** GetTextLayoutGeometry
**	compute a column/paging attributes for the text layout assuming
**	the given width and height of the widget window.
**
**	Return non-zero, if the computed text layout geometry differs
**	from the current setting.
*/
static int GetTextLayoutGeometry(w, set, width, height)
XeTextWidget w;
XeTextLayoutGeometry *set;
int width, height;
    {
	set->columns = w->text.columns;
	set->column_width = w->text.column_width;
	set->column_height = w->text.column_height;

	switch ((set->column_height > 0) +
		(set->column_width > 0) * 2 +
		(set->columns > 0) * 4)
	    {
	    case 0:	/* None specified, no paging (default) */
		set->column_height = INT_MAX;
		/* FALL TRHOUGH */
	    case 1:	/* Only column height: "Simple Vertical Paging" */
		set->columns = 1;
		set->column_width = width;
		break;

	    case 2:	/* Only Column width: "Page to window" */
		set->column_height = height;
		/* FALL THROUGH */
	    case 3:	/* Only Column width and height */
		set->columns = width / set->column_width;
		break;

	    case 4:	/* Only columns */
		set->column_height = height;
		/* FALL THROUGH */
	    case 5:	/* Columns and height */
		set->column_width = width / set->columns;
		break;

	    case 6:	/* Columns and width */
		set->column_height = height;
		break;
	    default:
		break;
	    }
	/*
	** Above may result invalid values, adjust some.
	*/
	if (set->columns < 1)
		set->columns = 1;
	else if (set->columns > MAX_TEXT_COLUMNS)
		set->columns = MAX_TEXT_COLUMNS;

	if (set->column_width == 0)
	    {
		set->columns = 1;
		set->column_width = INT_MAX;
	    }
	if (set->column_height == 0)
		set->column_height = INT_MAX;

	if (set->column_width != INT_MAX)
		set->width = set->column_width;
	else
		set->width = 0;
	if (set->column_height != INT_MAX)
		set->height = set->column_height;
	else
		set->height = 0;

	return	set->columns != w->text.set.columns ||
		set->width != w->text.set.width ||
		set->height != w->text.set.height ||
		set->column_width != w->text.set.column_width ||
		set->column_height != w->text.set.column_height;
    }

/*
** RedoTextLayout
**	totally redo the text layout and configure children
*/
#if NeedFunctionPrototypes
static void RedoTextLayout(XeTextWidget, XeTextLayoutGeometry *, int);
#endif

static void RedoTextLayout(t, set, commit)
XeTextWidget t;
XeTextLayoutGeometry *set;
int commit;
    {
	_XeTextLayout(t, set, (Snip *)NULL, (Region)NULL);
	if (commit)
		_XeTextConfigureChildren(t);
    }

static int LimitDimension(d, limit)
int d;
XeFrameDimension *limit;
    {
	if (limit->value > 0)
		return limit->value;
	if (limit->min > 0 && d < limit->min)
		d = limit->min;
	if (limit->max > 0 && d > limit->max)
		d = limit->max;
	return d;
    }

/*
** QueryGeometry
**
**	This is just a quick hack. To get the *real* answer to query,
**	one would have to redo the layout with suggested width and
**	compute the height.
*/
static XtGeometryResult QueryGeometry(w, request, preferred)
Widget w;
XtWidgetGeometry *request;
XtWidgetGeometry *preferred;
    {
	XeTextWidget t = (XeTextWidget)w;
	XeTextLayoutGeometry set;

	preferred->request_mode = CWWidth | CWHeight;
	preferred->width = (request->request_mode & CWWidth) ?
		request->width : 0;
	preferred->height = (request->request_mode & CWHeight) ?
		request->height : 0;
	preferred->width = LimitDimension
		(preferred->width, &t->basic.dimension[0]);
	preferred->height= LimitDimension
		(preferred->height,&t->basic.dimension[1]);
	if (GetTextLayoutGeometry(t,&set,preferred->width,preferred->height))
	    {
		/* ...this is something I may yet regret... --msa */

		XeTextLayoutGeometry old_set = t->text.set;

		_XeTextLayout(t, &set, (Snip *)NULL, (Region)0);
		preferred->width = t->text.w;
		preferred->height = t->text.h;
		_XeTextLayout(t, &old_set, (Snip *)NULL, (Region)0);
	    }
	else
	    {
		preferred->width = t->text.w;
		preferred->height = t->text.h;
	    }
	preferred->width = LimitDimension
		(preferred->width, &t->basic.dimension[0]);
	preferred->height= LimitDimension
		(preferred->height,&t->basic.dimension[1]);

	if (((request->request_mode&(CWWidth|CWHeight))==(CWWidth|CWHeight)) &&
	    request->width == preferred->width &&
	    request->height == preferred->height)
		return XtGeometryYes;
	else if (preferred->width == w->core.width &&
		 preferred->height == w->core.height)
		return XtGeometryNo;
	else
		return XtGeometryAlmost;
    }

/*
** SetupContent
**	initialize widget state from the current resources. Returns
**	true, if redisplay is required.
*/
static int SetupContent(w, dolayout)
XeTextWidget w;
int dolayout;
    {
	int num;
	XeFont *f;
	XeTabStop *t;
	/*
	** Initialize num_fonts, num_tabs
	*/
	num = 0;
	if ((f = w->text.fonts) != NULL)
		for ( ; f->font_size > 0; f++)
			num += 1;
	w->text.num_fonts = num;
	num = 0;
	if ((t = w->text.line_layout_table) != NULL)
		for ( ;  t->reference >= 0; t++)
			num += 1;
	w->text.num_tabs = num;

	/*
	** Content initialization (note that the defaults for content
	** related fields have been already dealt in Basic.c)
	*/
	if (w->basic.content_loaded == False)
	    {
		char *data;
		int length, is_copy;

		if (w->text.inserting)
			_XeTextEndContent
				(w->text.inserting, (Region)0, (long)NULL);
		w->text.inserting = _XeTextBeginContent(w);
		data = _XeGetContent((XeBasicWidget)w, &length, &is_copy);
		_XeTextFeedContent(w->text.inserting, data, (long)length);
		if (is_copy)
			XtFree(data);
		_XeTextEndContent(w->text.inserting, (Region)0, (long *)NULL);
		w->text.inserting = NULL;
		w->basic.content_loaded = True;
		dolayout = True;
	    }
	FreeColors(w);
	if (w->text.enable_display < 0)
		dolayout = 0;
	if (dolayout)
	    {
		XeTextLayoutGeometry set;
		(void)GetTextLayoutGeometry(w, &set,
					    w->core.width, w->core.height);
		RedoTextLayout(w, &set, True);
	    }
	if ((dolayout && w->basic.resize) || w->core.width == 0)
		w->core.width = w->text.w;
	if ((dolayout && w->basic.resize) || w->core.height == 0)
		w->core.height = w->text.h;
	w->core.width = LimitDimension(w->core.width,&w->basic.dimension[0]);
	w->core.height = LimitDimension(w->core.height,&w->basic.dimension[1]);
	if (w->core.height < TEXT_MIN_HEIGHT)
		w->core.height = TEXT_MIN_HEIGHT;
	if (w->core.width < TEXT_MIN_WIDTH)
		w->core.width = TEXT_MIN_WIDTH;
	XSetState(XtDisplay(w),w->text.mygc, w->basic.foreground_pixel,
		  w->core.background_pixel, GXcopy, ~0L);
	XSetState(XtDisplay(w),w->text.mygcXOR, ~0L, 0, GXxor,
		  w->basic.foreground_pixel ^ w->core.background_pixel);
#if 0
	/* Part of experiment that is not yet ready */
	XSetSubwindowMode(XtDisplay(w), w->text.mygc, IncludeInferiors);
	XSetSubwindowMode(XtDisplay(w), w->text.mygcXOR, IncludeInferiors);
#endif
	return dolayout;
    }

static void Initialize(request, init)
Widget request, init;
    {
	XeTextWidget w = (XeTextWidget)init;

	if (!w->text.alignment)
		w->text.alignment = XeAlignment_START;
	w->text.mygc  =
		XCreateGC(XtDisplay(w), RootWindowOfScreen(XtScreen(w)), 0, 0);
	w->text.mygcXOR =
		XCreateGC(XtDisplay(w), RootWindowOfScreen(XtScreen(w)), 0, 0);
	XSetLineAttributes(XtDisplay(w), w->text.mygc,
			   1, LineSolid, CapButt, JoinMiter);
	/*
	** Initialize Border drawing GC
	*/
	w->text.mygcBorder = _XeFrameCreateGC
		(init, &w->text.line,
		 w->text.framed_rendition, XtNumber(w->text.framed_rendition),
		 w->text.shadow_contrast);
	w->text.w = w->text.h = 0;
	w->text.first = NULL;
	w->text.colors = NULL;
	w->text.inserting = NULL;
	(void)GetTextLayoutGeometry
		(w, &w->text.set, w->core.width, w->core.height);
	if (w->text.initial_state)
		w->text.initial_state = XtNewString(w->text.initial_state);
	if (w->text.itemization)
	    {
		XeItemization *tmp = XtNew(XeItemization);
		memcpy((void *)tmp, (void *)w->text.itemization, sizeof(*tmp));
		w->text.itemization = tmp;
	    }
       	w->text.highlight = False;
	w->text.enable_display = 0;
	w->text.printing = False;
	SetupContent(w, True);
    }

/*
**	EXPOSE HANDLING
**	===============
*/

/*
** CornerMark
**	remembers one corner of a rectangular Area.
*/
typedef enum
    {
	MARK_Single,
	MARK_Double,
	MARK_Crossed,
	MARK_Overlined,
	MARK_Framed,
	MARK_Encircled,
	MARK_EnumSize	/* Number of elements in Enum */
    } MarkerId;

typedef struct CornerMark
    {
	int x, y, h;
	Snip *s;
    } CornerMark;

typedef struct ExposeContext
    {
	XeTextWidget w;		/* Current Text Widget */
	Region my_r;		/* Expose Area */
	SnipMode p;		/* Current Snip Mode */
	CornerMark markers[(int)MARK_EnumSize];
    } ExposeContext;

/*
** FindExtents
**	Find extents for a Snip sequence (s, e). The terminating 'e'
**	is not included.
**
**	NOTE: THIS FUNCTION ASSUMES CORRECT CALLING:
**	- 'e' must follow 's'
*/
static void FindExtents(s, e, ascent, descent)
Snip *s, *e;
int *ascent;
int *descent;
    {
	int base = s->y; /* Compute ascent/descent relative to this! */

	*ascent = 0;
	*descent = 0;
	for ( ; s && s != e; s = s->next)
		if (!s->floating)
		    {
			int adj = base - s->y;

			if (s->ascent + adj > *ascent)
				*ascent = s->ascent + adj;
			if (s->descent - adj > *descent)
				*descent = s->descent - adj;
		    }
    }

/*
** MarkArea
**	Start (h != 0) or End (h == 0) rendition effect.
*/
#if NeedFunctionPrototypes
static void MarkArea(ExposeContext *, Snip *, MarkerId, int, int, int);
#endif
static void MarkArea(cx, s, id, x, y, h)
ExposeContext *cx;
Snip *s;
MarkerId id;
int x, y, h;
    {
	register CornerMark *p = &cx->markers[(int)id];

	if (p->h != 0)
	    {
		/*
		** Found closing Marker, Draw Area
		*/
		XeFrameBorderP *f;
		int ascent, descent;
		int width = x - p->x;
		XeTextWidget w = cx->w;

		switch (id)
		    {
		    case MARK_Framed:
			FindExtents(p->s, s, &ascent, &descent);
			p->y -= ascent;
			p->h = ascent + descent;
			if (cx->my_r &&
			    XRectInRegion(cx->my_r, p->x, p->y, width, p->h) ==
			    RectangleOut)
				break;
			f = w->text.framed_rendition;
			_XeDrawBorderRectangle
				((Widget)w,
				 w->text.mygcBorder,
				 &w->text.line,
				 p->x, p->y, width, p->h,
				 &f[FRAME_LEFT], &f[FRAME_RIGHT],
				 &f[FRAME_TOP], &f[FRAME_BOTTOM],
				 w->text.invert_shadow != 0);
			break;
		    case MARK_Crossed:
			FindExtents(p->s, s, &ascent, &descent);
			p->y -= ascent / 2;
			/* FALL TRHOUGH */
		    default:
			if (cx->my_r &&
			    XRectInRegion(cx->my_r, p->x, p->y, width, p->h) ==
			    RectangleOut)
				break;
			XFillRectangle(XtDisplay(w), XtWindow(w),
				       w->text.mygc,
				       p->x , p->y, width, p->h);
			break;
		    }
	    }
	p->s = s;
	p->x = x;
	p->y = y;
	p->h = h;
    }

static void ChangeUnderline(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	switch (cx->p.bits & Underline_MASK)
	    {
	    case Underline_DOUBLE:
		MarkArea(cx, s, MARK_Double, x, y, 0);
	    case Underline_SINGLE:
		MarkArea(cx, s, MARK_Single, x, y, 0);
		break;
	    default:
		break;
	    }
	if (s == NULL)
		return;
	x = s->x + cx->w->basic.x_translation;
	y = s->y + cx->w->basic.y_translation;
	switch (s->mode.bits & Underline_MASK)
	    {
	    case Underline_DOUBLE:
		MarkArea(cx, s, MARK_Double, x, y + 3, 1);
	    case Underline_SINGLE:
		MarkArea(cx, s, MARK_Single, x, y + 1, 1);
		break;
	    default:
		break;
	    }
    }

static void ChangeCrossedOut(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	if (cx->p.bits & CrossedOut_MASK)
		MarkArea(cx, s, MARK_Crossed, x, y, 0);
	if (s == NULL)
		return;
	x = s->x + cx->w->basic.x_translation;
	y = s->y + cx->w->basic.y_translation;
	if (s->mode.bits & CrossedOut_MASK)
		MarkArea(cx, s, MARK_Crossed, x, y, 1);
    }

static void ChangeFraming(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	/* Only FRAMED implemented, no ENCIRCLED yet */

	if (cx->p.bits & Framing_MASK)
		MarkArea(cx, s, MARK_Framed, x, y, 0);
	if (s == NULL)
		return;
	x = s->x + cx->w->basic.x_translation;
	y = s->y + cx->w->basic.y_translation;
	if (s->mode.bits & Framing_MASK)
		MarkArea(cx, s, MARK_Framed, x, y, 1);
    }

/*
** GetColors
**	Allocate array of 8 pixel values and allocate the 8 basic
**	colors that will be used for the text effects:
**
**	0=black, 1=red, 2=green, 3=yellow,
**	4=blue, 5=magenta, 6=cyan, 7=white
*/

static void GetColors(w)
XeTextWidget w;
    {
	static XeSample rgb[24] =
	    {
		/* black,red,green,yellow,blue,magenta,cyan,white */

		0, ~0,  0, ~0,  0, ~0,  0, ~0,	/* red */
		0,  0, ~0, ~0,  0,  0, ~0, ~0,	/* green */
		0,  0,  0,  0, ~0, ~0, ~0, ~0,	/* blue */
	    };
	
	XeSample tmp[24];

	memcpy((void *)tmp, (void *)rgb, sizeof(tmp));
	w->text.colors = (unsigned long *)XtMalloc(8 * sizeof(unsigned long));
	(void)XeAllocColors(XtDisplay(w), XtWindow(w), w->core.colormap,
			    XeColormapUse_SHARED, 8, 8,
			    &tmp[0], &tmp[8], &tmp[16],
			    w->text.colors);
    }

static void FreeColors(w)
XeTextWidget w;
    {
	if (w->text.colors == NULL)
		return;
	XeFreeColors(XtDisplay(w), w->core.colormap, 8, w->text.colors);
	XtFree((void *)w->text.colors);
	w->text.colors = NULL;
    }

static void ChangeForeground(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	unsigned long pixel;
	XeTextWidget w = cx->w;

	if (w->text.colors == NULL)
		GetColors(w);
	if (s && (s->mode.bits & Foreground_MASK) &&
	    w->text.colors[Foreground_COLOR(s->mode.bits)-1]!=FRAME_NO_PIXEL)
		pixel = w->text.colors[Foreground_COLOR(s->mode.bits)-1];
	else
		pixel = w->basic.foreground_pixel;
	if (s && (s->mode.bits & ImageInversion_MASK))
		XSetBackground(XtDisplay(w), w->text.mygc, pixel);
	else
		XSetForeground(XtDisplay(w), w->text.mygc, pixel);
    }

static void ChangeBackground(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	unsigned long pixel;
	XeTextWidget w = cx->w;

	if (w->text.colors == NULL)
		GetColors(w);
	if (s && (s->mode.bits & Background_MASK) &&
	    w->text.colors[Background_COLOR(s->mode.bits)-1]!=FRAME_NO_PIXEL)
		pixel = w->text.colors[Background_COLOR(s->mode.bits)-1];
	else
		pixel = w->core.background_pixel;
	if (s && (s->mode.bits & ImageInversion_MASK))
		XSetForeground(XtDisplay(w), w->text.mygc, pixel);
	else
		XSetBackground(XtDisplay(w), w->text.mygc, pixel);
    }

static void FlushPendingGraphics(cx, s, x, y)
ExposeContext *cx;
Snip *s;
int x, y;
    {
	if (cx->p.bits & Underline_MASK)
	    {
		MarkArea(cx, s, MARK_Double, x, y, 0);
		MarkArea(cx, s, MARK_Single, x, y, 0);
		cx->p.bits &= ~Underline_MASK;
	    }
	if (cx->p.bits & CrossedOut_MASK)
	    {
		MarkArea(cx, s, MARK_Crossed, x, y, 0);
		cx->p.bits &= ~CrossedOut_MASK;
	    }
	if (cx->p.bits & Framing_MASK)
	    {
		MarkArea(cx, s, MARK_Framed, x, y, 0);
		cx->p.bits &= ~Framing_MASK;
	    }
    }

static void Redisplay(w, e, r)
Widget w;
XExposeEvent *e;
Region r;
    {
	register XeTextWidget t = (XeTextWidget)w;
	Snip *s;
	int x, y, highlight;
	XRectangle clip;
	ExposeContext cx;
	Pixel bg_pixel;
#if NeedFunctionPrototypes
#ifndef _XConst
#define _XConst const
#endif
	int (*f_draw16)(Display *,Drawable,GC,int,int,_XConst XChar2b *,int);
	int (*f_draw)(Display *,Drawable,GC,int,int,_XConst char *,int);
#else
	int (*f_draw16)();
	int (*f_draw)();
#endif

	memset((void *)&cx, 0, sizeof(cx));
	cx.my_r = r;
	cx.w = t;
	if (!XtIsRealized(w))
		return;
	/*
	** Choose the background pixel. The background pixel will be
	** different if highlight is on and "background highlighting"
	** is selected.
	**
	** A temporary hack: use background highlighting if the frame_redition
	** TOP frame has been specified as a "shadow", and use the color
	** defined for the TOP framed rendition as hightlight. [Really
	** needs a separate resources for highlight (color/contrast) --msa]
	*/
	highlight = t->text.highlight && t->core.sensitive;
	if (highlight &&
	    t->text.framed_rendition[FRAME_TOP].value.type
	    == XeBorderType_SHADOW &&
	    t->text.framed_rendition[FRAME_TOP].pixel != FRAME_NO_PIXEL)
		bg_pixel = t->text.framed_rendition[FRAME_TOP].pixel;
	else
		bg_pixel = t->core.background_pixel;

	if (cx.my_r == NULL && e != NULL)
	    {
		/*
		** No Region, make one from exposed area. This should be
		** exceptional, widgets should have the compress exposures
		** mode set and always have a region!
		*/
		clip.x = e->x;
		clip.y = e->y;
		clip.width = e->width;
		clip.height = e->height;
		cx.my_r = XCreateRegion();
		XUnionRectWithRegion(&clip, cx.my_r, cx.my_r); 
	    }
	if (cx.my_r)
	    {
		XSetRegion(XtDisplay(w), t->text.mygc, cx.my_r);
		XSetRegion(XtDisplay(w), t->text.mygcXOR, cx.my_r);
		XSetRegion(XtDisplay(w), t->text.mygcBorder, cx.my_r);
	    }
	else
	    {
		XSetClipMask(XtDisplay(w), t->text.mygc, None);
		XSetClipMask(XtDisplay(w), t->text.mygcXOR, None);
		XSetClipMask(XtDisplay(w), t->text.mygcBorder, None);
	    }

	XSetBackground(XtDisplay(w), t->text.mygc, bg_pixel);
	if (e == NULL || bg_pixel != t->core.background_pixel)
	    {
		/*
		** If the XEvent parameter is NULL, then assume this is
		** internally generated expose event and the covered
		** area is not necessarily initially cleared. Clear the
		** area covered by the expose region (my_r). [The following
		** code is the simplest I could invent for the task, if
		** anyone has better, let me know --msa].
		** When also the Region is not given, make this a special
		** case that refreshes the whole window.
		*/
		XSetForeground(XtDisplay(w), t->text.mygc, bg_pixel);
		if (cx.my_r)
		    {
			XClipBox(cx.my_r, &clip);
			XFillRectangle(XtDisplay(w),XtWindow(w),t->text.mygc,
				       clip.x,clip.y,clip.width,clip.height);
		    }
		else
			XFillRectangle(XtDisplay(w),XtWindow(w),t->text.mygc,
				       t->basic.x_translation,
				       t->basic.y_translation,
				       t->core.width, t->core.height);
	    }
	XSetForeground(XtDisplay(w), t->text.mygc, t->basic.foreground_pixel);
	if (t->composite.num_children > 0)
	    {
		int i;
		
		for (i = 0; i < t->composite.num_children; ++i)
		    {
			XeFrameConstraints fc;
			Widget child = t->composite.children[i];
			if (!XtIsManaged(child))
				continue;
			if (XtIsSubclass(child, xeFrameWidgetClass))
				continue; /* Frames draw borders themselves */
			fc = (XeFrameConstraints)child->core.constraints;
			_XeDrawChildBorder(w, t->text.mygcBorder,
					   &t->text.line, child,
					   &fc->frame.border[FRAME_LEFT],
					   &fc->frame.border[FRAME_RIGHT],
					   &fc->frame.border[FRAME_TRAILING],
					   &fc->frame.border[FRAME_LEADING],
					   fc->frame.invert_shadow != 0);
		    }
	    }
	f_draw16 = XDrawString16;
	f_draw = XDrawString;
	y = t->text.y + t->basic.y_translation;
	x = t->text.x + t->basic.x_translation;
	for (s = t->text.first; s; s = s->next)
	    {
		SnipData *h = s->content.head;
		unsigned int m = cx.p.bits ^ s->mode.bits;

		if (m) 
		    {
			if (m & Underline_MASK)
				ChangeUnderline(&cx, s, x, y);
			if (m & CrossedOut_MASK)
				ChangeCrossedOut(&cx, s, x, y);
			if (m & Framing_MASK)
				ChangeFraming(&cx, s, x, y);
			if (m & (Foreground_MASK|ImageInversion_MASK))
				ChangeForeground(&cx, s, x, y);
			if (m & (Background_MASK|ImageInversion_MASK))
				ChangeBackground(&cx, s, x, y);
			if (s->mode.bits&(ImageInversion_MASK|Background_MASK))
			    {
				f_draw16 = XDrawImageString16;
				f_draw = XDrawImageString;
			    }
			else
			    {
				f_draw16 = XDrawString16;
				f_draw = XDrawString;
			    }
		    }
		cx.p = s->mode;
		x = s->x + t->basic.x_translation;
		y = s->y + t->basic.y_translation;

		if (!s->space && s->xWidth > 0 &&
		    (cx.my_r == NULL ||
		     XRectInRegion(cx.my_r, x, y - s->ascent, s->xWidth,
				   s->ascent + s->descent) != RectangleOut) &&
		    s->data != NULL)
		    {
                        XSetFont(XtDisplay(w), t->text.mygc, h->font->fid);
			if (h->bytes == 2)
				(*f_draw16)
					(XtDisplay(w), XtWindow(w),
					 t->text.mygc,
					 (int)x + s->offset, (int)y,
					 (XChar2b *)s->data, s->length);
			else
				(*f_draw)
					(XtDisplay(w), XtWindow(w),
					 t->text.mygc,
					 (int)x + s->offset, (int)y,
					 s->data, s->length);

                        
		    }
		if (!s->floating)
			x += s->xWidth;
		if (HasEndLine(s) &&
		    (cx.p.bits&(Underline_MASK|CrossedOut_MASK|Framing_MASK)))
			FlushPendingGraphics(&cx, s->next, x, y);
	     }
	FlushPendingGraphics(&cx, s, x, y); /* In case endseq was omitted! */
	XSetForeground(XtDisplay(w), t->text.mygc, t->basic.foreground_pixel);
	XSetBackground(XtDisplay(w), t->text.mygc, t->core.background_pixel);
	if (cx.my_r != r && cx.my_r)
		XDestroyRegion(cx.my_r);
	if (t->basic.expose_callbacks&&t->core.widget_class==xeTextWidgetClass)
	    {
		XeExposeCallbackData data;
		data.reason = XeCR_EXPOSE;
		data.event = e;
		data.region = r;
		XtCallCallbackList
			(w, t->basic.expose_callbacks, (XtPointer)&data);
	    }
       	if (highlight && bg_pixel == t->core.background_pixel)
		XDrawRectangle(XtDisplay(t), XtWindow(t), t->text.mygc,
			       0, 0, t->core.width - 1, t->core.height - 1);
    }
/*
**	****************
**	Printing support
**	****************
*/
static void _ResetSnips(w, pp)
XeTextWidget w;
XePrintProcess pp;
    {
	Snip *t;

	for (t = w->text.first; t; t= t->next)
		if (!t->widget && t->content.head)
			t->content.head->font = NULL;
    }

static GC MakePrintGC(printer, page, display, gc)
Display *printer;
Drawable page;
Display *display;
GC gc;
    {
	unsigned long value_mask;
	XGCValues values;

	/*
	** create a mask that will retrieve/set all members of XGCValues
	** (except those that are server/display specific, and need to
	** to be transferred explicitly, if needed!)
	*/
	value_mask = (1L << (GCLastBit+1)) - 1;
	value_mask &= ~(GCFont | GCTile | GCStipple | GCClipMask |
			GCDashList | GCForeground | GCBackground);
	
	XGetGCValues(display, gc, value_mask, &values);
	return XCreateGC(printer, page, value_mask, &values);
    }

static void PrintInitialize(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	XeTextWidget clone = (XeTextWidget)c;
	XeTextWidget t = (XeTextWidget)w;
	Pixel pixels[XtNumber(clone->text.framed_rendition)];
	XeSample r[XtNumber(clone->text.framed_rendition)];
	XeSample g[XtNumber(clone->text.framed_rendition)];
	XeSample b[XtNumber(clone->text.framed_rendition)];
	
	int i, j;

	/*
	** *NOTE*
	**	Currently, nothing special is done if print_initialize is
	**	called while another print is active. The current relayout
	**	really does not work properly, with simultaenous multiple
	**	printings.. Waiting for ideas what should be done about
	**	this, but for now, just let it pass (one may get X protocol
	**	error due to bad font id, if printers are different, or
	**	just plain garbled output....) -- msa
	*/
	t->text.printing = True;
	t->text.enable_display -= 1;
	/*
	** The right thing would be make a copy of all content to the
	** clone, but trying for a faster method first: move the content
	** to the clone widget, and leave original empty for the duration
	** of the printing.
	*/
	t->text.first = NULL;
	t->text.inserting = NULL;
	_ResetSnips(clone, pp);
	clone->text.printing = True;
	clone->text.mygc = MakePrintGC
		(XtDisplay(c), XtWindow(c), XtDisplay(w), t->text.mygc);
	clone->text.mygcXOR = MakePrintGC
		(XtDisplay(c), XtWindow(c), XtDisplay(w), t->text.mygcXOR);
	/*
	** Translate frame_rendition. First, one need to translate the
	** pixel values, if any..
	*/
	for (i = j = 0; i < XtNumber(t->text.framed_rendition); ++i)
		if (t->text.framed_rendition[i].pixel != FRAME_NO_PIXEL)
			pixels[j++] = t->text.framed_rendition[i].pixel;
	if (j > 0)
	    {
		XeQueryColors(XtDisplay(w),t->core.colormap, j, pixels,r,g,b);
		clone->core.colormap = XeAllocColors
			(XtDisplay(c), XtWindow(c), c->core.colormap,
			 XeColormapUse_SHARED, j, j, r, g, b, pixels);
	    }
	for (i = j = 0; i < XtNumber(t->text.framed_rendition); ++i)
		if (t->text.framed_rendition[i].pixel != FRAME_NO_PIXEL)
			clone->text.framed_rendition[i].pixel = pixels[j++];
	
	clone->text.mygcBorder = _XeFrameCreateGC
		((Widget)clone, &clone->text.line,
		 clone->text.framed_rendition,
		 XtNumber(clone->text.framed_rendition),
		 clone->text.shadow_contrast);
	clone->text.colors = NULL;
	/*
	** Do paging of the text content by adjusting some resources on
	** clone structure (this may take too much control away from the
	** application, and it might be reconsidered whether this is the
	** right place to do such adjustments)
	*/
	if (!XtParent(clone))
	    {
		clone->text.column_height = pp->printer->page_height;
		(void)GetTextLayoutGeometry
			(clone, &clone->text.set,
			 pp->printer->page_width, pp->printer->page_height);
	    }
	else
		(void)GetTextLayoutGeometry(clone, &clone->text.set, 0, 0);
	if (clone->composite.num_children > 0)
	    {
		/*
		** Ugly and very "brittle" fix to a problem: the layout
		** below needs to find the preferred geometry of the
		** insets, but PrintConstraint Initialize methods are
		** not called yet and the widget pointers in Snips are
		** not correct. The fix... patch up the widget pointers
		** here...
		*/
		Widget inset;
		int i;
		XeTextConstraints tc;

		for (i = 0; i < clone->composite.num_children; i++)
		    {
			inset = clone->composite.children[i];
			if (!inset)
				continue;
			tc = (XeTextConstraints)inset->core.constraints;
			if (tc && tc->text.inset)
				tc->text.inset->content.widget = inset;
		    }
	    }
	RedoTextLayout(clone, &clone->text.set, True);
	if (!XtParent(clone))
	    {
		clone->core.width = clone->text.w;
		clone->core.height = clone->text.h;
	    }
    }

static XtPointer Print(w, c, x, y, pp)
Widget w, c;
int x, y;
XePrintProcess pp;
    {
	XeTextWidget clone = (XeTextWidget)c;

	clone->basic.x_translation = x;
	clone->basic.y_translation = y;
	REDISPLAY(c, (XEvent *)NULL, (Region)0);
	return NULL;
    }

static void PrintCancel(w, clone, pp, client_data)
Widget w, clone;
XePrintProcess pp;
XtPointer client_data;
    {
    }

static void PrintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	XeTextWidget t = (XeTextWidget)w;
	XeTextWidget clone = (XeTextWidget)c;

	XFreeGC(XtDisplay(clone), clone->text.mygc);
	XFreeGC(XtDisplay(clone), clone->text.mygcXOR);
	_XeFrameFreeGC(c, clone->text.mygcBorder,
		       clone->text.framed_rendition,
		       XtNumber(clone->text.framed_rendition));
	FreeColors(clone);
	t->text.printing = False;
	/*
	** Restore content back to the original widget. NOTE! Need to
	** copy back the 'first' from clone to the original widget, because
	** the original first might have been pointing to a layout generated
	** content, which has been deleted by now...
	*/
	t->text.first = clone->text.first;
	t->text.inserting = clone->text.inserting;
	_ResetSnips(t, pp);
	RedoTextLayout(t, &t->text.set, True);
	if ((t->text.enable_display += 1) == 0)
		REDISPLAY(t, NULL, 0);
    }

static void PrintConstraintInitialize(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	XeTextConstraints tc = (XeTextConstraints)w->core.constraints;

	_XeFrameInitialize(w, c, (ArgList)0, (Cardinal)0);
	/*
	** Replace the widget pointer in the Snip
	*/
	if (tc && tc->text.inset)
		tc->text.inset->content.widget = c;
    }

static void PrintConstraintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	XeTextConstraints tc = (XeTextConstraints)c->core.constraints;

	_XeFrameDestroy(c);
	/*
	** Restore the original widget pointer into the Snip
	*/
	if (tc && tc->text.inset)
		tc->text.inset->content.widget = w;
    }

/*
**	*****************
**	Actions functions
**	*****************
**	Highlight, Unhighlight
*/
static void ChangeHighlight(t, state)
XeTextWidget t;
int state;
    {
	if (t->text.highlight != (Boolean)state)
	    {
		t->text.highlight = state;
		REDISPLAY(t, NULL, 0);
	    }
    }

static void Highlight(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ChangeHighlight((XeTextWidget)w, True);
    }

static void Unhighlight(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ChangeHighlight((XeTextWidget)w, False);
    }

/* END Actions functions */

static Boolean CheckBorder(old, set)
XeFrameBorderP *old, *set;
    {
	int i;

	for (i = 0; i < 4; ++i)
		if (old[i].value.width != set[i].value.width ||
		    old[i].value.space != set[i].value.space)
			return True;
	return False;
    }

static Boolean SetValues(current, request, set, args, num_args)
Widget current, request, set;
ArgList args;
Cardinal *num_args;
    {
	XeTextWidget cw = (XeTextWidget)current;
	XeTextWidget nw = (XeTextWidget)set;
	int dolayout, redisplay, i;

	if (!nw->core.sensitive && nw->text.highlight)
	    {
		nw->text.highlight = False;
		redisplay = True;
	    }
	else
		redisplay = False;
	dolayout =
		/* Only request full layout, if layout width has been
		   changed, ignore all height changes, and width change
		   if column_width is defined */
		(!nw->text.column_width && cw->core.width != nw->core.width) ||
		 cw->text.column_width != nw->text.column_width ||
		 cw->text.columns != nw->text.columns ||
		 cw->text.column_height != nw->text.column_height ||
		 cw->text.column_separation != nw->text.column_separation ||
		 cw->text.page_separation != nw->text.page_separation ||
		 /* Other parameters affecting layout */
		 cw->text.alignment != nw->text.alignment ||
		 cw->text.first_line_offset!=nw->text.first_line_offset ||
		 cw->text.fonts != nw->text.fonts ||
		 cw->text.format != nw->text.format ||
		 cw->text.indentation != nw->text.indentation ||
		 cw->text.itemization != nw->text.itemization ||
		 cw->text.line_spacing != nw->text.line_spacing ||
		 cw->text.line_layout_table!=nw->text.line_layout_table ||
		 cw->text.proportional != nw->text.proportional ||
		 cw->text.invert_shadow != nw->text.invert_shadow;
	for (i = 0; i < XtNumber(cw->text.framed_rendition); ++i)
		redisplay |= _XeChangeBorderColor
			(set, cw->text.framed_rendition,
			 nw->text.framed_rendition, i,
			 cw->text.shadow_contrast, nw->text.shadow_contrast);
	dolayout |= CheckBorder(cw->text.framed_rendition,
				 nw->text.framed_rendition);
	if (cw->text.initial_state != nw->text.initial_state)
	    {
		/*
		** Make a local copy of the initial_state resource
		*/
		XtFree(cw->text.initial_state);
		if (nw->text.initial_state)
			nw->text.initial_state =
				XtNewString(nw->text.initial_state);
		if (nw->text.initial_state && nw->text.inserting)
			_XeTextFeedContent(nw->text.inserting,
					   nw->text.initial_state,
					   strlen(nw->text.initial_state));
	    }
	if (cw->text.itemization != nw->text.itemization)
	    {
		/*
		** Make a local copy of the itemization resource
		*/
		XeItemization *new = nw->text.itemization;
		XtFree((char *)cw->text.itemization);
		if (new)
		    {
			nw->text.itemization = XtNew(XeItemization);
			memcpy((void *)nw->text.itemization,
			       (void *)new, sizeof(XeItemization));
		    }
	    }
	redisplay |= SetupContent(nw, dolayout);
#if 0
	/* This piece of "optimization" has the problem that the core
	   width and height may have been changed in SetupContent and
	   the clearing of the window in Redisplay based on these values
	   is not usually correct => garbage is left on screen...
	   The code left here as a reminder of what not to do!!!

	   -- msa */

	if (redisplay && nw->text.enable_display >= 0)
		/*
		** To avoid flickering, when widget is used to display
		** frequently changing text, just do the redisplay here
		** now!
		*/
		REDISPLAY(nw, NULL, 0);
	return False;
#else
	return redisplay && nw->text.enable_display >= 0;
#endif
    }

static void Resize(w)
Widget w;
    {
	XeTextWidget t = (XeTextWidget)w;
	XeTextLayoutGeometry layout;

	if (t->text.enable_display < 0)
		return;
	/*
	** Ignore full re-layout, if set_width doesn't change.
	*/
	if (GetTextLayoutGeometry(w, &layout, w->core.width, w->core.height))
	    {
		RedoTextLayout(t, &layout, True);
		if (!t->text.printing)
			REDISPLAY(t, NULL, 0);
	    }
    }

static Boolean FrameSetValues(old, request, set, args, num_args)
Widget old, request, set;
ArgList args;
Cardinal *num_args;
    {
	XeTextWidget t = (XeTextWidget)XtParent(set);
	XeTextConstraints tc = (XeTextConstraints)set->core.constraints;
	int result = _XeFrameSetValues(old, request, set, args, num_args);

	if (t->text.enable_display < 0)
		return 0;
	_XeFrameExtents(set, set->core.width, set->core.height, &tc->text.FE);
	if (result & XeFrameSetValues_GEOMETRY)
	    {
		/*
		** The values affect the geometry. Must recompute the
		** layout.
		*/
		XtWidgetGeometry current;

		_XeTextLayout(t, &t->text.set, (Snip *)NULL, (Region)NULL);
		/*
		** This is icky.. changing core fields here only to get
		** XtSetValues to call GeometryManager.. which again has
		** to redo all layout.. blech!! --msa
		*/
		if (GetInsetGeometry(t, set, &current))
		    {
			set->core.x = current.x;
			set->core.y = current.y;
			set->core.width = current.width;
			set->core.height = current.height;
			REDISPLAY(t, NULL, 0);
		    }
	    }
	else if ((result & XeFrameSetValues_FRAMING) &&
		 XtIsRealized((Widget)t))
	    {
		/*
		** Only framing attributes changed. Generate expose only to
		** the area child and framing.
		*/
		/* should also get the paramters of the 'old' extents
		   and generate expose for that too!!! --msa */
		XClearArea(XtDisplay(t), XtWindow((Widget)t),
			   set->core.x - tc->text.FE.window.x,
			   set->core.y - tc->text.FE.window.y,
			   tc->text.FE.width, tc->text.FE.height, True);
	    }
	return 0;
    }

static void Realize(w, valueMask, attributes)
Widget w;
Mask *valueMask;
XSetWindowAttributes *attributes;
    {
	XeTextLayout(w);
	(*SuperClass->core_class.realize) (w, valueMask, attributes);
    }

static void Destroy(w)
Widget w;
    {
	XeTextWidget tw = (XeTextWidget)w;

	/* Destroy the Snip chain */

	if (tw->text.inserting)
		(void)_XeTextEndContent
			(tw->text.inserting,(Region)0,(long *)NULL);
	while (tw->text.first)
		_XeDeleteSnip(&tw->text.first);

	/* Release GC's */

	XFreeGC(XtDisplay(w), tw->text.mygc);
	XFreeGC(XtDisplay(w), tw->text.mygcXOR);
	_XeFrameFreeGC(w, tw->text.mygcBorder, (XeFrameBorderP *)NULL, 0);
	
	XtFree(tw->text.initial_state);
	XtFree((char *)tw->text.itemization);
	/* Release fonts -- not copied currently */
	/* Release kerning_offset -- not copied currently */
	/* Release line_layout_table -- not copied currently */

	/* Release text colors */
	FreeColors(tw);
    }

/*
** **********************************************************
** COMPOSITE/CONSTRAINT WIDGET SPECIFIC METHDOS AND FUNCTIONS
** **********************************************************
*/
static void InsertChild(w)
register Widget w;
    {
	XeTextWidget tw = (XeTextWidget)XtParent(w);
	XeTextConstraints tc = (XeTextConstraints)w->core.constraints;

	if (!tw->text.inserting)
		/*
		** Insert Position is not open. Just insert in front
		*/
		tw->text.inserting = _XeTextInsertContent
			(tw, tw->text.first, 0, XeTextInsertMode_RESET);
	tc->text.inset = _XeTextInsertWidget(tw->text.inserting, w);
	/*
	** insert the child widget in the composite children list with the
	** superclass insert_child routine.
	*/
	(*SuperClass->composite_class.insert_child)(w);
    }

static void DeleteChild(w)
Widget w;
    {
	/*
	** delete the child widget in the composite children list with the
	** superclass delete_child routine.
	*/
	XeTextWidget tw = (XeTextWidget)XtParent(w);
	XeTextConstraints tc = (XeTextConstraints)w->core.constraints;

	if (tc->text.inset)
		_XeTextDeleteWidget(tw, tc->text.inset);
	tc->text.inset = NULL;
	(*SuperClass->composite_class.delete_child) (w);
    }

static void ConstraintDestroy(w)
Widget w;
    {
	XeTextWidget tw = (XeTextWidget)XtParent(w);
	XeTextConstraints tc = (XeTextConstraints)w->core.constraints;

	if (tc->text.inset)
		_XeTextDeleteWidget(tw, tc->text.inset);
	tc->text.inset = NULL;
	_XeFrameDestroy(w);
    }

/*
** ChangeManaged
**	Just do a complete re-layout of text on each change.
*/
static void ChangeManaged(w)
Widget w;
    {
	XeTextWidget tw = (XeTextWidget)w;

#ifdef USING_MOTIF_122
        /* update the traversal */
        _XmNavigChangeManaged(w);
#endif
	if (tw->text.enable_display >= 0)
		XeTextLayout(w);
    }

/*
** GeometryManager method
**
**	(A quick hack. Should take a closer look at this!! --msa )
*/
static XtGeometryResult GeometryManager(w, request, reply)
Widget w;
XtWidgetGeometry *request, *reply;
    {
	XeTextWidget tw = (XeTextWidget)XtParent(w);
	XtWidgetGeometry rq, current;
	XtGeometryMask rm = request->request_mode;
	int width, height;
	XeTextLayoutGeometry set, old;
	XtGeometryResult my_result = XtGeometryNo;

	rq.request_mode = rm;
	rq.x = (rm & CWX) ? request->x : w->core.x;
	rq.y = (rm & CWY) ? request->y : w->core.y;
	rq.width = (rm & CWWidth) ? request->width : w->core.width;
	rq.height = (rm & CWHeight) ? request->height : w->core.height;

	if (rq.x == w->core.x && rq.y == w->core.y &&
	    rq.width == w->core.width && rq.height == w->core.height)
		return XtGeometryNo;
	/*
	** Do some ad hoc heuristics here: if the dimension requests
	** MAX (== 0), then prefer the current size (should at
	** least work with XeFrame parent, in which case the core fields
	** already may include the effect of the MAX contraint
	*/
	width = tw->basic.dimension[0].value ? 0 : tw->core.width;
	height = tw->basic.dimension[1].value ? 0 : tw->core.height;

	(void)GetTextLayoutGeometry(tw, &set,
			      LimitDimension(width, &tw->basic.dimension[0]),
			      LimitDimension(height, &tw->basic.dimension[1]));
	old = tw->text.set;
	_XeTextLayout(tw, &set, (Snip *)NULL, (Region)NULL);
	width = LimitDimension(tw->text.w, &tw->basic.dimension[0]);
	height = LimitDimension(tw->text.h, &tw->basic.dimension[1]);
	if (tw->basic.resize &&
	    (height != tw->core.height || width != tw->core.width))
	    {
		/*
		** The new dimensions for parent differ from the current
		** size. Ask/inform grandparent about size change.
		**
		** (The code needs more work! One should probably do a
		** relayout if, parent gives Almost return for the width
		** --msa)
		*/
		XtGeometryResult result;
		XtWidgetGeometry my_req, my_rep;

		my_req.request_mode = CWWidth | CWHeight;
		if (XtCWQueryOnly & rm)
			my_req.request_mode |= XtCWQueryOnly;
		my_req.width = width;
		my_req.height = height;
		result = XtMakeGeometryRequest((Widget)tw, &my_req, &my_rep);
		if (result == XtGeometryNo)
		    {
			my_result = XtGeometryNo;
			goto restore_layout;
		    }
		else if (result == XtGeometryAlmost)
		    {
			if ((rm & XtCWQueryOnly) == 0)
				result = XtMakeGeometryRequest
					((Widget)tw, &my_rep,
					 (XtWidgetGeometry *)NULL);
		    }
	    }
	if (!reply)
		reply = &current;
	if (!GetInsetGeometry(tw, w, reply))
	    {
		my_result = XtGeometryNo; /* Probaly being destroyed... */
		goto restore_layout;
	    }
	if ((!(rm & CWWidth) || rq.width == reply->width) &&
	    (!(rm & CWHeight) || rq.height == reply->height) &&
	    (!(rm & CWX) || rq.x == reply->x) &&
	    (!(rm & CWY) || rq.y == reply->y))
	    {
		if (rm & XtCWQueryOnly)
		    {
			my_result = XtGeometryYes;
			goto restore_layout;
		    }
		if (tw->text.enable_display >= 0)
		    {
			_XeTextConfigureChildren(tw);
			if (XtIsRealized((Widget)tw))
				REDISPLAY(tw, NULL, 0);
		    }
		return XtGeometryDone;
	    }
        my_result = reply ? XtGeometryAlmost : XtGeometryNo;
    restore_layout:
	_XeTextLayout(tw, &old, (Snip *)NULL, (Region)NULL);
	return my_result;
    }

/*
** User callable convenience function support
** ==========================================
*/

/*
** IsTextWidget
**	Returns True, if the widget is of class xeText and
**	False otherwise (also, a warning message is generated).
*/
static int IsTextWidget(w)
Widget w;
    {
	String params[2];

	if (XtIsSubclass(w, xeTextWidgetClass))
		return True;
	params[0] = w->core.name;
	params[1] = w->core.widget_class->core_class.class_name;
	XeWidgetWarningMsg
		(w, "bugTextEdNotText",
		 "Widget %s is class %s, and not xeTextWidget",
		 (String *)params, (Cardinal)2);
	return False;
    }

/*
** XeTextLayout
*/
void XeTextLayout(w)
Widget w;
    {
	if (IsTextWidget(w))
	    {
		XeTextWidget t = (XeTextWidget)w;
		XeTextLayoutGeometry layout;

		(void)GetTextLayoutGeometry
			(w, &layout, t->core.width, t->core.height);
		RedoTextLayout(t, &layout, True);
		REDISPLAY(w, NULL, 0);
	    }
    }


#ifdef USING_MOTIF_122
/*
** WidgetNavigable
**	routine to handle traversal to children of this widget, if
**	there are any, or else the widget itself, if no children. 
*/
static XmNavigability WidgetNavigable(wid)
Widget wid;
    {   
	if(wid->core.sensitive && wid->core.ancestor_sensitive &&
	   ((XmManagerWidget) wid)->manager.traversal_on)
	    {
		XmNavigationType nav_type =
			((XmManagerWidget) wid)->manager.navigation_type;

		if(nav_type == XmSTICKY_TAB_GROUP ||
		   nav_type == XmEXCLUSIVE_TAB_GROUP ||
		   (nav_type == XmTAB_GROUP && !_XmShellIsExclusive( wid)))
			return XmTAB_NAVIGABLE;
		return XmCONTROL_NAVIGABLE;
	    }
	return XmNOT_NAVIGABLE;
    }
#endif
