/*
** RasterGIF.c
**
** Code is based on existing code...
**
** gif2ras.c - Converts from a Compuserve GIF (tm) image to a Sun Raster image.
**
** Copyright (c) 1988, 1989 by Patrick J. Naughton
**
** Author: Patrick J. Naughton
** naughton@wind.sun.com
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation.
**
** This file is provided AS IS with no warranties of any kind.  The author
** shall have no liability with respect to the infringement of copyrights,
** trade secrets or any patents by this file or any part thereof.  In no
** event will the author be liable for any lost revenue or profits or
** other special, indirect and consequential damages.
**
** Modifications and fitting into Xew Widget set:
**
** Copyright 1992 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>
#	include <malloc.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#else
char *malloc();
void free();
#endif
#endif
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/RasterP.h>
#include "RasterGIF.h"


static void RasterImportWarning(w, msg)
XeRasterWidget w;
char *msg;
    {
	XeWidgetWarningMsg
		((Widget)w, "rasterImportWarning", msg, (String *)NULL, 0);
    }

#define EXTENSION     0x21
#define IMAGESEP      0x2c
#define TRAILER       0x3b
#define INTERLACEMASK 0x40
#define COLORMAPMASK  0x80

typedef unsigned char byte;

static int BitOffset = 0,	/* Bit Offset of next code */
	XC = 0, YC = 0,		/* Output X and Y coords of current pixel */
	Pass = 0,		/* Used by output routine if interlaced pic */
	OutCount = 0,		/* Decompressor output 'stack count' */
	RWidth, RHeight,	/* screen dimensions */
	LeftOfs, TopOfs,	/* image offset */
	ColorMapSize,		/* number of colors */
	Background,		/* background color */
	CodeSize,		/* Code size, read from GIF header */
	InitCodeSize,		/* Starting code size, used during Clear */
	Code,			/* Value returned by ReadCode */
	MaxCode,		/* limiting value for current code size */
	ClearCode,		/* GIF clear code */
	EOFCode,		/* GIF end-of-information code */
	CurCode, OldCode, InCode,/* Decompressor variables */
	FirstFree,		/* First free code, generated per GIF spec */
	FreeCode,		/* Decompressor,next free slot in hash table */
	FinChar,		/* Decompressor variable */
	BitMask,		/* AND mask for data size */
	ReadMask,		/* Code AND mask for current code size */
	Misc;                   /* miscellaneous bits (interlace, local cmap)*/



    /* The hash table used by the decompressor */
static int Prefix[4096];
static int Suffix[4096];

    /* An output array used by the decompressor */
static int OutCode[4097];

static int   gif89 = 0;
static char *id87 = "GIF87a";
static char *id89 = "GIF89a";

static int EGApalette[16][3] = {
  {0,0,0},       {0,0,128},     {0,128,0},     {0,128,128}, 
  {128,0,0},     {128,0,128},   {128,128,0},   {200,200,200},
  {100,100,100}, {100,100,255}, {100,255,100}, {100,255,255},
  {255,100,100}, {255,100,255}, {255,255,100}, {255,255,255} };
  

static void DoInterlace(raw, Index)
XeRawImage *raw;
byte Index;
    {
	static byte *ptr = NULL;
	static int   oldYC = -1;
	
	if (oldYC != YC) 
	    {
		ptr = raw->data + YC * raw->width; 
		oldYC = YC;
	    }
	if (YC < raw->height)
		*ptr++ = Index;
	/*
	** Update the X-coordinate, and if it overflows,
	** update the Y-coordinate
	*/
	if (++XC == raw->width) 
	    {
		/*
		** deal with the interlace as described in the GIF spec.
		** Put the decoded scan line out to the screen if we haven't
		** gone past the bottom of it
		*/
		XC = 0;
		switch (Pass)
		    {
		    case 0:
			YC += 8;
			if (YC >= raw->height)
			    { 
				Pass++; 
				YC = 4; 
			    }
			break;
		    case 1:
			YC += 8;
			if (YC >= raw->height)
			    {
				Pass++; 
				YC = 2;
			    }
			break;
		    case 2:
			YC += 4;
			if (YC >= raw->height)
			    {
				Pass++;
				YC = 1; 
			    }
			break;
		    case 3:
			YC += 2;
			break;
		    default:
			break;
		    }
	    }
    }

/*
** Fetch the next code from the raster data stream.  The codes can be
** any length from 3 to 12 bits, packed into 8-bit bytes, so we have to
** maintain our location in the Raster array as a BIT Offset.  We compute
** the byte Offset into the raster array by dividing this by 8, pick up
** three bytes, compute the bit Offset into our 24-bit chunk, shift to
** bring the desired code to the bottom, then mask it off and return it. 
*/
static int ReadCode(Raster)
byte *Raster;
    {
	int RawCode, ByteOffset;
	
	ByteOffset = BitOffset / 8;
	RawCode = Raster[ByteOffset] + (Raster[ByteOffset + 1] << 8);
	if (CodeSize >= 8)
		RawCode += ( ((int) Raster[ByteOffset + 2]) << 16);
	RawCode >>= (BitOffset % 8);
	BitOffset += CodeSize;
	
	return RawCode & ReadMask;
    }

XeRawImage *XeImport_GIF(w)
XeRasterWidget w;
    {
	register byte ch;
	register byte *ptr, *ptr1, *r, *g, *b;
	register int i, block;
	int npixels, maxpixels, aspect, transparent;
	byte *RawGIF = NULL, *RawEnd;
	int RawLength;
	int Interlace, HasColormap;
	XeRawImage *raw = NULL;

	/* initialize variables */

	BitOffset = XC = YC = Pass = OutCount = npixels = maxpixels = 0;
	gif89 = 0;
	transparent = -1;

	RawGIF = (byte *)_XeGetContentCopy((XeBasicWidget)w, &RawLength);
	if (RawGIF == NULL || RawLength < 14)
	    {
		RasterImportWarning(w, "No content or truncated GIF");
		goto import_failed;
	    }
	RawEnd = RawGIF + RawLength;
	ptr = RawGIF;
	if (strncmp((char *)ptr, id87, 6) == 0)
		gif89 = 0;
	else if (strncmp((char *)ptr, id89, 6) == 0)
		gif89 = 1;
	else
	    {
		RasterImportWarning(w, "not a GIF file");
		goto import_failed;
	    }
	ptr += 6;
	
	/* Get variables from the GIF screen descriptor */
	
	ch = *ptr++;
	RWidth = ch + 0x100 * *ptr++;	/* screen dimensions... not used. */
	ch = *ptr++;
	RHeight = ch + 0x100 * *ptr++;
	
	ch = *ptr++;
	HasColormap = ((ch & COLORMAPMASK) ? True : False);
	
	raw = _XeCreateRawImage(1);	/* GIF is always one channel for now */
	raw->resolution_unit = XeImageUnit_NONE;
	raw->x_resolution = 1.0;	/* a dummy really --msa */
	raw->y_resolution = 1.0;	/* a dummy really --msa */
	raw->bits_per_sample = (ch & 7) + 1;
	raw->bits_per_component = 8;	/* GIF unpacks into byte/pixel */
	ColorMapSize = 1 << raw->bits_per_sample;
	BitMask = ColorMapSize - 1;
	raw->alloc_map = True;
	r = raw->color_map = (unsigned char *)XtMalloc(3 * ColorMapSize);
	g = r + ColorMapSize;
	b = g + ColorMapSize;
	Background = *ptr++;		/* background color... not used. */
	aspect = *ptr++;
	if (aspect) 
	    {
		if (!gif89)
		    {
			RasterImportWarning(w,
				  "corrupt GIF file (screen descriptor)");
			goto import_failed;
		    }
		/* gif89 aspect ratio */
		raw->y_resolution = (float) (aspect + 15) / 64.0;
	    }
	/* Read in global colormap. */

	if (HasColormap)
		for (i=0; i < ColorMapSize; i++)
		    {
			r[i] = *ptr++;
			g[i] = *ptr++;
			b[i] = *ptr++;
		    }
	else
		/*
		** no colormap in GIF file. put std EGA palette (repeated)
		** into colormap, for lack of anything better to do
		*/
		for (i=0; i < ColorMapSize; i++)
		    {
			r[i] = EGApalette[i&15][0];
			g[i] = EGApalette[i&15][1];
			b[i] = EGApalette[i&15][2];
		    }
	/*
	** possible things at this point are:
	**   an application extension block
	**   a comment extension block
	**   an (optional) graphic control extension block
	**       followed by either an image
	**	   or a plaintext extension
	*/
	while ((block = *ptr++) != IMAGESEP)
	    {
		if (block == EXTENSION)
		    {  /* parse extension blocks */
			int fn, blocksize, aspnum, aspden;
		    
			/* read extension block */
			fn = *ptr++;
			if (fn == 'R')
			    {
				/* GIF87 aspect extension */

				blocksize = *ptr++;
				if (blocksize == 2)
				    {
					aspnum = *ptr++;
					aspden = *ptr++;
					if (aspden > 0 && aspnum > 0) 
						raw->y_resolution = 
							(float) aspnum /
							(float) aspden;
					else
						raw->y_resolution = 1.0;
				    }
				else
					ptr += blocksize;
			    }
			else if (fn == 0xf9)
			    {
				/* GIF89 Graphic Control Extension */
				blocksize = *ptr++;
				if (*ptr & 1)	/* Transparent Color Flag */
					transparent = ptr[3];
				ptr += blocksize;
			    }
			/* Skip until terminator */
			while ((blocksize = *ptr++) > 0)
				ptr += blocksize;
		    }
		else if (block == TRAILER) 
		    {
			RasterImportWarning
				(w,"no image data found in GIF file");
			goto import_failed;
		    }
		else
		    {
			RasterImportWarning
				(w, "Unknown block type found in file.");
			goto import_failed;
		    }
	    }
	
	/* read in values from the image descriptor */

	ch = *ptr++;
	LeftOfs = ch + 0x100 * *ptr++;
	ch = *ptr++;
	TopOfs = ch + 0x100 * *ptr++;
	ch = *ptr++;
	raw->width = ch + 0x100 * *ptr++;
	ch = *ptr++;
	raw->height = ch + 0x100 * *ptr++;
	Misc = *ptr++;
	Interlace = ((Misc & INTERLACEMASK) ? True : False);
	
	if (Misc & 0x80)

		/* check this! is allocated colormap large enough? --msa */

		for (i=0; i< 1 << ((Misc&7)+1); i++)
		    {
			r[i] = *ptr++;
			g[i] = *ptr++;
			b[i] = *ptr++;
		    }
	
	if (!HasColormap && !(Misc&0x80))
	    {
		/* no global or local colormap */
		;
	    }
	/*
	** Start reading the raster data. First we get the intial code size
	** and compute decompressor constant values, based on this code size.
	*/
	CodeSize = *ptr++;
	ClearCode = (1 << CodeSize);
	EOFCode = ClearCode + 1;
	FreeCode = FirstFree = ClearCode + 2;
	/*
	** The GIF spec has it that the code size is the code size used to
	** compute the above values is the code size given in the file, but the
	** code size used in compression/decompression is the code size given
	** in the file plus one. (thus the ++).
	*/
	CodeSize++;
	InitCodeSize = CodeSize;
	MaxCode = (1 << CodeSize);
	ReadMask = MaxCode - 1;
	/*
	** UNBLOCK:
	** Read the raster data.  Here we just compact it from the from a
	** series of blocks into one long data stream, which makes life much
	** easier for ReadCode().
	*/
	for (ptr1 = RawGIF; (i = *ptr++) > 0; ptr1 += i, ptr += i)
	    {
		if ((ptr - RawGIF) > RawLength)
			goto truncated; /* Truncated file? */
		memcpy((void *)ptr1, (void *)ptr, i);
	    }

	maxpixels = raw->width * raw->height;
	raw->channel[0].addr = raw->data = ptr = (byte *)malloc(maxpixels);
	if (ptr == NULL)
	    {
		RasterImportWarning(w, "Out of memory at GIF import");
		goto import_failed;
	    }
	raw->alloc_data = True;
	raw->channel[0].w = raw->bytes_per_line = raw->width;
	raw->channel[0].h = raw->height;
	raw->channel[0].line = raw->width;
	raw->channel[0].inc = 1;
	/*
	** Decompress the file, continuing until you see the GIF EOF code.
	** One obvious enhancement is to add checking for corrupt files here.
	*/
	Code = ReadCode(RawGIF);
	while (Code != EOFCode)
	    {
		/*
		** Clear code sets everything back to its initial value,
		** then reads the immediately subsequent code as uncompressed
		** data.
		*/
		if (Code == ClearCode)
		    {
			CodeSize = InitCodeSize;
			MaxCode = (1 << CodeSize);
			ReadMask = MaxCode - 1;
			FreeCode = FirstFree;
			Code = ReadCode(RawGIF);
			CurCode = OldCode = Code;
			FinChar = CurCode & BitMask;
			if (!Interlace)
				*ptr++ = FinChar;
			else
				DoInterlace(raw, FinChar);
			npixels++;
		    }
		else 
		    {
			/*
			** If not a clear code, must be data: save same as
			** CurCode and InCode. if we're at maxcode and didn't
			** get a clear, stop loading
			*/
			if (FreeCode>=4096)
				break; /* freecode blew up */
			CurCode = InCode = Code;
			/*
			** If greater or equal to FreeCode, not in the hash
			** table yet; repeat the last character decoded
			*/
			if (CurCode >= FreeCode)
			    {
				CurCode = OldCode;
				if (OutCount > 4096)
					break;/* outcount1 blew */
				OutCode[OutCount++] = FinChar;
			    }
			/*
			** Unless this code is raw data, pursue the chain
			** pointed to by CurCode through the hash table to
			** its end; each code in the chain puts its associated
			** output code on the output queue.
			*/
			while (CurCode > BitMask)
			    {
				if (OutCount > 4096)
					break;   /* corrupt file */
				OutCode[OutCount++] = Suffix[CurCode];
				CurCode = Prefix[CurCode];
			    }
			
			if (OutCount > 4096)
				break; /* outcount blew up */
			/*
			** The last code in the chain is treated as raw data.
			*/
			FinChar = CurCode & BitMask;
			OutCode[OutCount++] = FinChar;
			/*
			** Now we put the data out to the Output routine.
			** It's been stacked LIFO, so deal with it that way...
			**
			** safety thing:  prevent exceeding range of 'pic'
			*/
			if (npixels + OutCount > maxpixels)
				OutCount = maxpixels-npixels;
			npixels += OutCount;
			if (!Interlace)
				for (i=OutCount-1; i>=0; i--)
					*ptr++ = OutCode[i];
			else
				for (i=OutCount-1; i>=0; i--)
					DoInterlace(raw, OutCode[i]);
			OutCount = 0;
			/*
			** Build the hash table on-the-fly. No table is
			** stored in the file.
			*/
			Prefix[FreeCode] = OldCode;
			Suffix[FreeCode] = FinChar;
			OldCode = InCode;
			/*
			** Point to the next slot in the table. If we exceed
			** the current MaxCode value, increment the code size
			** unless it's already 12.  If it is, do nothing: the
			** next code decompressed better be CLEAR
			*/
			FreeCode++;
			if (FreeCode >= MaxCode) 
			    {
				if (CodeSize < 12) 
				    {
					CodeSize++;
					MaxCode *= 2;
					ReadMask = (1 << CodeSize) - 1;
				    }
			    }
		    }
		Code = ReadCode(RawGIF);
		if (npixels >= maxpixels)
			break;
	    }
	/*
	** A kludgy support for transparency index: Just replace the
	** colormap entry for this with the values of the *current*
	** background color. Will not work properly if background is
	** later changed or if some colorallocation method assigns
	** new "close" color for that index... --msa
	*/
	if (transparent >= 0 && transparent < ColorMapSize)
	    {
		XColor query;
		query.pixel = w->core.background_pixel;
		XQueryColor(XtDisplay(w), w->core.colormap, &query);
		r[transparent] = query.red >> 8;
		g[transparent] = query.green >> 8;
		b[transparent] = query.blue >> 8;
	    }
	/*
	** Finish up the import
	*/
	XtFree((void *)RawGIF);
	if (npixels != maxpixels && !Interlace)
		/* clear to EOBuffer */
		memset((void *)(raw->data + npixels), 0, maxpixels-npixels);
	raw->image_class = XeImageClass_PALETTE;
	raw->color_space = XeImageSpace_RGB;
	raw->samples_per_pixel = 1;
	return raw;
/*
** Truncated GIF file
*/
    truncated:
	RasterImportWarning(w, "Truncated GIF stream");
/*
** Failed in load, cleanup
*/
    import_failed:
	XtFree((void *)RawGIF);
	_XeDestroyRawImage(raw);
	return NULL;
    }

