/*
** Print.c
**
**	Contains generic print "driver" functions for the Xew. This module
**	does not do any printing self, but controls the calling of print
**	methods and asynchronous printing process.
**
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/Xew/BasicP.h>
#include <X11/Xew/Raster.h>
#include "ImageTools.h"
#include "PrintP.h"
#include "GetImage.h"
#include "Color.h"

#if SYSV_INCLUDES
#	include <memory.h>
#	include <malloc.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#else
char *malloc();
void free();
#endif
#endif

/*
** CountTree
**	Count the total number of widgets in the widget tree
*/
static int CountTree(w)
Widget w;
    {
	int count = 1;

	if (XtIsComposite(w))
	    {
		int i;
		CompositeWidget cw = (CompositeWidget)w;

		for (i = 0; i < cw->composite.num_children; i++)
			count += CountTree(cw->composite.children[i]);
	    }
	return count;
    }

/*
** PrintInializeCore
**	This function does the tasks of print_initialize method for
**	the core class.
*/
void PrintInitializeCore(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	Pixel pixels[2];
	XeSample r[2], g[2], b[2];
	int screen = DefaultScreen(pp->printer->printer);
#if USING_XPRINTER
	/* XPrinter doesn't seem to use the screen, we'd better
	** make one up! This is persistently used by XtDisplay
	** replace screen w/ bogus screen w/ member pointing to
	** printer display so that XtDisplay Macro will work.
	*/
	c->core.screen = (Screen *)XtMalloc(sizeof(Screen));
	memcpy((char *)c->core.screen, (char *)w->core.screen, sizeof(Screen));
	c->core.screen->display = pp->printer->printer;
#else
	c->core.screen = DefaultScreenOfDisplay(pp->printer->printer);
	c->core.colormap = DefaultColormap(pp->printer->printer, screen);
	c->core.depth = DefaultDepth(pp->printer->printer, screen);
#endif
	c->core.window = pp->printer->page;
	/*
	** Translate original colors to printer colors
	*/
	pixels[0] = w->core.background_pixel;
	pixels[1] = w->core.border_pixel;
	XeQueryColors(XtDisplay(w), w->core.colormap, 2, pixels, r, g, b);
	if (!pp->printer->background)
		r[0] = g[0] = b[0] = XeSample_MAX;	/* 'white' */
	c->core.colormap = XeAllocColors
		(XtDisplay(c), XtWindow(c), c->core.colormap,
		 XeColormapUse_SHARED, 2, 2, r, g, b, pixels);
	c->core.background_pixel = pixels[0];
	c->core.border_pixel = pixels[1];
	/*
	** Transfer pixmaps
	**
	** (Currently not implemented--just drop them)
	*/
	c->core.background_pixmap = XtUnspecifiedPixmap;
	c->core.border_pixmap = XtUnspecifiedPixmap;
    }

/*
** PrintDestroyCore
**
**	This function does the tasks of print_destroy method for
**	the core class.
*/
static void PrintDestroyCore(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	Pixel pixels[2];

	pixels[0] = c->core.background_pixel;
	pixels[1] = c->core.border_pixel;
	XeFreeColors(XtDisplay(c), c->core.colormap, 2, pixels);
#if USING_XPRINTER
	XtFree(c->core.screen);
#endif
    }

/*
** DoPrintInitialize
**	calls print_initialize methods in superclass to subclass order
**	for all classes that are subclasses of XeBasic, and do equivalent
**	of print_initialize for the core class.
**
**	A recursive function that returns
**		0, if class is not a subclass of XeBasic
**		1, if class is a subclass of XeBasic
*/
static int DoPrintInitialize(w, c, pp, class)
Widget w, c;
XePrintProcess pp;
WidgetClass class;
    {
	if (class == NULL)
		c->core.screen = NULL;
	else if (class == coreWidgetClass)
		PrintInitializeCore(w, c, pp);
	else if (DoPrintInitialize(w, c, pp, class->core_class.superclass) ||
		 class == xeBasicWidgetClass)
	    {
		XeBasicWidgetClass bc = (XeBasicWidgetClass)class;
		/*
		** Current class is XeBasic or some subclass of it,
		** if it has a print_initialize method, call it!
		*/
		if (bc->basic_class.print_initialize)
			(*bc->basic_class.print_initialize)(w, c, pp);
		return 1;
	    }
	return 0;
    }

/*
** DoPrintDestroy
**	calls print_destroy methods in subclass to superclass order for
**	all classes that are subclasses of XeBasic, and do equivalent
**	of print_destroy for the core class.
*/
static void DoPrintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	if (XtIsSubclass(w, xeBasicWidgetClass))
	    {
		/*
		** Only Widget classes that are subclasses of XeBasic
		** have the print_destroy method!
		*/
		WidgetClass class;

		for (class = w->core.widget_class;;)
		    {
			XeBasicWidgetClass bc = (XeBasicWidgetClass)class;

			if (bc->basic_class.print_destroy)
				(*bc->basic_class.print_destroy)(w, c, pp);
			if (class == xeBasicWidgetClass)
				break;
			class = class->core_class.superclass;
		    }
	    }
	if (c->core.screen)
		PrintDestroyCore(w, c, pp);
    }


/*
** DoPrintConstraintInitialize
**	calls print_constraint_initialize methods in superclass to subclass
**	order for all classes that are subclasses of XeBasic.
**
**	A recursive function that returns
**		0, if class is not a subclass of XeBasic
**		1, if class is a subclass of XeBasic
*/
static int DoPrintConstraintInitialize(w, c, pp, class)
Widget w, c;
XePrintProcess pp;
WidgetClass class;
    {
	if (class == NULL || class == coreWidgetClass)
		return 0;
	else if (class == xeBasicWidgetClass ||
		 DoPrintConstraintInitialize
		 (w, c, pp,class->core_class.superclass))
		 
	    {
		XeBasicWidgetClass bc = (XeBasicWidgetClass)class;
		/*
		** Current class is XeBasic or some subclass of it,
		** if it has a print_constraint_initialize method, call it!
		*/
		if (bc->basic_class.print_constraint_initialize)
			(*bc->basic_class.print_constraint_initialize)(w,c,pp);
		return 1;
	    }
	return 0;
    }

/*
** DoPrintConstraintDestroy
**	calls print_constraint_destroy methods in subclass to superclass
**	order for all classes that are subclasses of XeBasic.
*/
static void DoPrintConstraintDestroy(w, c, pp)
Widget w, c;
XePrintProcess pp;
    {
	Widget parent = XtParent(c);
	
	if (parent && XtIsSubclass(parent, xeBasicWidgetClass))
	    {
		/*
		** Only Widget classes that are subclasses of XeBasic
		** can have the print_constraint_destroy method!
		*/
		WidgetClass class;

		for (class = parent->core.widget_class;;)
		    {
			XeBasicWidgetClass bc = (XeBasicWidgetClass)class;

			if (bc->basic_class.print_constraint_destroy)
				(*bc->basic_class.print_constraint_destroy)
					(w, c, pp);
			if (class == xeBasicWidgetClass)
				break;
			class = class->core_class.superclass;
		    }
	    }
    }

/*
** BuildTree
**	builds a cloned widget tree for printing process.
**
** *NOTE*
**	Current cloned tree is very skimpy, not forming a proper widget
**	hierarchy... walking on a thin ice here...
**
** *NOTE*
**	The Tree is built into tree array in postfix order, the last node
**	will be the root of the tree.
*/
static void BuildTree(w, parent, pp)
Widget w;	/* A widget to be cloned */
Widget parent;	/* An already cloned widget that is the parent or NULL */
XePrintProcess pp;
    {
	Widget clone;
	int is_composite;
	int is_xew;

	if (w->core.being_destroyed)
		return;

	clone =	(Widget)XtMalloc(w->core.widget_class->core_class.widget_size);
	if (!clone)
		return;	/* Oops.. out of memory... */	
	memcpy((char *)clone, (char *)w,
	       w->core.widget_class->core_class.widget_size);
	clone->core.parent = parent;
	if (clone->core.constraints && parent)
	    {
		/*
		** The parent must be a constraint widget, duplicate the
		** constraint instance structure.
		*/
		ConstraintWidgetClass class = (ConstraintWidgetClass)
			parent->core.widget_class;
		
		clone->core.constraints = (XtPointer)malloc
			(class->constraint_class.constraint_size);
		if (!clone->core.constraints)
		    {
			free((char *)clone);
			return; /* Another oops, out of memory! */
		    }
		memcpy((char *)clone->core.constraints,
		       (char *)w->core.constraints,
		       class->constraint_class.constraint_size);
	    }
	is_composite = XtIsComposite(w);
	if (is_composite)
	    {
		CompositeWidget cw = (CompositeWidget)w;
		CompositeWidget ccw = (CompositeWidget)clone;
		int i;

		ccw->composite.children = (WidgetList)
			XtMalloc(ccw->composite.num_slots *
				 sizeof(ccw->composite.children[0]));
		ccw->composite.num_children = 0;
		/*
		** Process children in the order specified by
		** the original composite (preserve order in
		** clone children array!!!)
		*/
		for (i = 0; i < cw->composite.num_children; ++i)
			BuildTree(cw->composite.children[i], clone, pp);
	    }
	/*
	** NOTE, problem here! Could call DoPrintInialize
	** 1) before the child processing
	**	=> children would not be initialized while initializing
	**	   a parent (a problem with XeText layout which may need
	**	   to request child resize)
	**
	** 2) after the child processing (current solution)
	**	=> parent is not initialized while child initialize is
	**	   called. The only problem might be a possible call to
	**	   to the geometry manager (or other "parent services").
	** (But, Geometry Managing in the cloned tree is currently a
	**  dangerous happening anyway, needs further consideration...! )
	*/
	is_xew = DoPrintInitialize(w, clone, pp, w->core.widget_class);
	pp->tree[pp->count].w = w;
	pp->tree[pp->count].clone = clone;
	if (parent && XtIsComposite(parent))
	    {
		CompositeWidget cw = (CompositeWidget)parent;
		cw->composite.children[cw->composite.num_children++] = clone;
	    }
	if (is_xew)
	    {
		XeBasicWidgetClass c =(XeBasicWidgetClass)w->core.widget_class;
		pp->tree[pp->count].print = c->basic_class.print;
		pp->tree[pp->count].cancel = c->basic_class.print_cancel;
	    }
	else if (!is_composite)
	    {
		/*
		** For now, generate the default print method for
		** non-composites only (the default does a heavy
		** XGetImage/XPutImage operation, is mostly wasted on most
		** composite manager widgets)
		*/
		pp->tree[pp->count].print = _XeDefaultPrint;
		pp->tree[pp->count].cancel = _XeDefaultPrintCancel;
	    }
	else
	    {
#if PRINT_UNKNOWN_COMPOSITES
		/*
		** Enable this branch if you really want the unknown
		** composites printed with XGetImage/XPutImage method
		*/
		pp->tree[pp->count].print = _XeDefaultPrint;
		pp->tree[pp->count].cancel = _XeDefaultPrintCancel;
#else
		pp->tree[pp->count].print = NULL;
		pp->tree[pp->count].cancel = NULL;
#endif
	    }
	/*
	** Can do constraint initialize only after the parent has been
	** initialized... thus, arrange calls from parent level build
	** instead of actual child level.
	*/
	if (XtIsConstraint(clone))
	    {
		CompositeWidget wcw = (CompositeWidget)w;
		CompositeWidget ccw = (CompositeWidget)clone;
		int i;
			
		for (i = 0; i < ccw->composite.num_children; ++i)
		    {
			Widget wc = wcw->composite.children[i];
			Widget cc = ccw->composite.children[i];
			
			DoPrintConstraintInitialize
				(wc, cc, pp, clone->core.widget_class);
		    }
	    }
	pp->count += 1;
    }

/*
** FinishTree
*/
static void FinishTree(tree, count, pp)
PrintNode *tree;
int count;
XePrintProcess pp;
    {
	int i;

	for (i = 0; i < count; ++i)
	    {
		if (tree[i].w->core.constraints && tree[i].clone->core.parent)
			DoPrintConstraintDestroy(tree[i].w, tree[i].clone, pp);
		DoPrintDestroy(tree[i].w, tree[i].clone, pp);
		if (XtIsComposite(tree[i].w))
		    {
			CompositeWidget cw = (CompositeWidget)tree[i].clone;
			XtFree((char *)cw->composite.children);
		    }
		if (tree[i].w->core.constraints && tree[i].clone->core.parent)
			free((char *)tree[i].clone->core.constraints);
		XtFree((char *)tree[i].clone);
	    }
    }

/*
** _XePrintProcess
**	Asynchronous printing process main loop.
**
**	On entry, the print->current is the index of the last completed
**	widget (or initial count).
*/
XePrintProcess _XePrintProcess(pp)
XePrintProcess pp;
    {
	int x, y;

	pp->printing = NULL;	/* Has completed previous print, if any */

	while (pp->running)
	    {
		
		if (pp->paused)
			return pp;

		while (--pp->current >= 0)
		    {
			Widget clone, tmp;
			int bw;
			PrintNode *node = &pp->tree[pp->current];


			tmp = clone = node->clone;
			if (!clone->core.screen)
				continue;
			x = - pp->x, y = - pp->y;
			if (XtParent(tmp))
			    {
				/*
				** Do not print internal Unmanaged widgets.
				** (The top level widget will get printed
				** whether managed or not).
				*/
				if (!XtIsManaged(tmp))
					continue;
				while (XtParent(tmp))
				    {
					x +=tmp->core.x+tmp->core.border_width;
					y +=tmp->core.y+tmp->core.border_width;
					tmp = XtParent(tmp);
				    }
			    }
			/*
			** Does this widget intersect with the current page?
			** (If not, then print method call for this page
			** can be skipped)
			*/
			bw = clone->core.border_width;
			if (x - bw >= pp->printer->page_width ||
			    y - bw >= pp->printer->page_height ||
			    x + clone->core.width + bw < 0 ||
			    y + clone->core.height + bw < 0)
				continue;
			/*
			** If the widget has borders, draw them here
			*/
			if (bw > 0)
			    {
				XRectangle r[4];
				r[0].x = x - bw;
				r[0].y = y - bw;
				r[0].width = clone->core.width + 2 * bw + 1;
				r[0].height = bw + 1;
				r[1] = r[0];
				r[1].y = y + clone->core.height;
				r[2].x = x - bw;
				r[2].y = y;
				r[2].width = bw + 1;
				r[2].height = clone->core.height + 1;
				r[3] = r[2];
				r[3].x = x + clone->core.width;
				XSetForeground
					(XtDisplay(clone),
					 pp->gc,
					 clone->core.border_pixel);
				if (clone->core.border_pixmap !=
				    XtUnspecifiedPixmap)
				    {
					XSetTile(XtDisplay(clone), pp->gc,
						 clone->core.border_pixmap);
					XSetFillStyle(XtDisplay(clone),
						      pp->gc, FillTiled);
				    }
				XFillRectangles
					(XtDisplay(clone), XtWindow(clone),
					 pp->gc, r, 4);
				if (clone->core.border_pixmap !=
				    XtUnspecifiedPixmap)
					XSetFillStyle(XtDisplay(clone),
						      pp->gc, FillSolid);
			    }
			/*
			** Do the background
			*/
			XSetForeground(XtDisplay(clone), pp->gc,
				       clone->core.background_pixel);
			XFillRectangle(XtDisplay(clone), XtWindow(clone),
				       pp->gc, x, y,
				       clone->core.width, clone->core.height);
			/*
			** The print method returns NON-ZERO, if the print
			** was started asynchronously (or if the continuation
			** is already dealt with). Returns ZERO, if method
			** completed in this call. Non-ZERO return means that
			** PrintProcess will get later called via a callback.
			*/
			if (node->print)
			    {
				pp->printing = (*node->print)
					(node->w, node->clone, x, y, pp);
				if (pp->printing)
					return pp;
			    }
		    }
		/*
		** One page has been completed, advance page
		*/
		pp->x += pp->printer->page_width;
		if (pp->x >= pp->width)
		    {
			pp->x = 0;
			pp->y += pp->printer->page_height;
			if (pp->y >= pp->height)
				break; /* *** Print completed normally *** */
		    }
		pp->page += 1;
		pp->paused = 0;
		if (pp->callback)
		    {
			pp->calling = True;
			(*pp->callback)(pp->print_root, pp,
					pp->page,pp->client_data);
			pp->calling = False;
		    }
		pp->current = pp->count;
	    }
	/*
	** Printing completed, do the final callback
	** and cleanup.
	*/
	pp->running = False;
	FinishTree(&pp->tree[0], pp->count, pp);
	XFreeGC(pp->printer->printer, pp->gc);
	if (pp->callback)
	    {
		pp->calling = True;
		(*pp->callback)(pp->print_root, pp, 0, pp->client_data);
		pp->calling = False;
	    }
	XtFree((char *)pp);
	return NULL;
    }

/*
** XePrint
**	starts printing process for printing all the content of the widget
**	to the specified printer.
**
**	Returns non-NULL, if printing is in progress asynchronously,
**
**	Returns NULL, if all printing completed within the call.
*/
XePrintProcess XePrint(w, p, callback, client_data)
Widget w;
XePrinter *p;
XePrintPageProc callback;
XtPointer client_data;
    {
	XePrintProcess pp;
	int count;
	
	/*
	** Count the max possible number of widgets in the tree and
	** build the printing tree in depth first order into the
	** allocated widget array.
	*/
	count = CountTree(w);
	pp = (XePrintProcessRec *)XtMalloc
		(XtOffsetOf(XePrintProcessRec,tree[0])+count*sizeof(PrintNode));
	pp->running = True;
	pp->paused = False;
	pp->calling = False;
	pp->printer = p;
	pp->callback = callback;
	pp->client_data = client_data;
	pp->print_root = w;
	pp->count = 0;
	pp->page = 1;
	if (pp->callback)
	    {
		pp->calling = True;
		(*pp->callback)(pp->print_root, pp, pp->page, pp->client_data);
		pp->calling = False;
	    }
	BuildTree(w, NULL, pp);
	pp->current = pp->count;
	pp->x = 0;
	pp->y = 0;
	if (pp->count > 0)
	    {
		pp->clone_root = pp->tree[pp->count-1].clone;
		pp->width = pp->clone_root->core.width;
		pp->height = pp->clone_root->core.height;
	    }
	else
	    {
		pp->clone_root = None;
		pp->width = 0;
		pp->height = 0;
	    }
	pp->gc = XCreateGC(p->printer, p->page, 0L, 0);
	/*
	** If the caller did not give any callback procedure, the printing
	** cannot inform about completion and thus cannot inform when the
	** pp get destroyed. Return pp to application only if callback
	** is exists.
	*/
	if (pp->callback)
		return _XePrintProcess(pp);
	else
	    {
		(void)_XePrintProcess(pp);
		return NULL;
	    }
    }

/*
** XePrintCancel
**	terminate current printing process. The page callback is called
**	with page==0 to indicate completion of the printing process.
**
** *NOTE*
**	After application has received the final page callback (page==0),
**	the pp context pointer returned by XePrint is invalid. Calling
**	XePrintCancel with invalid pp will cause unpredictable results,
**	because the data pointed by 'pp' has been released already.
*/
void XePrintCancel(pp)
XePrintProcess pp;
    {
	if (!pp || !pp->running)
		return;
	if (pp->printing)
	    {
		PrintNode *node = &pp->tree[pp->current];
		if (node->cancel)
		    {
			(*node->cancel)
				(node->w, node->clone, pp, pp->printing);
			pp->printing = NULL;
		    }
	    }
	pp->running = False;
	if (pp->calling)
		return;	/* in a page callback, the normal use... */
	/*
	** This should happen only when paused
	*/
	_XePrintProcess(pp);
    }

/*
** XePrintPause
**	pause the current printing process at after the next widget
**	completes (if active).
**
** *NOTE*
**	Really intended to be used from the page callback function, if
**	printing process needs to wait for page change for some reason.
*/
void XePrintPause(pp)
XePrintProcess pp;
    {
	if (!pp || !pp->running)
		return;
	pp->paused = True;
    }

/*
** XePrintResume
**	resume printing previosly stopped by XePrintPause
*/
void XePrintResume(pp)
XePrintProcess pp;
    {
	if (!pp || !pp->running)
		return;
	if (pp->paused)
	    {
		pp->paused = False;
		/*
		** 'paused' may have been set while some widget printing
		** is still in progress, and pause may not have yet been
		** 'realized', or if this is called from the page callback,
		** the printing will resume when callback returns.
		*/
		if (!pp->printing && !pp->calling)
			_XePrintProcess(pp);
	    }
    }

/*
** XeRawImageFromXImage
**	create an XeRawImage from an XImage. Returns a pointer to a
**	new XeRawImage.
*/
static XeRawImage *XeRawImageFromXImage(display, window, image, copy_data)
Display *display;
Window window;
XImage *image;
int copy_data;
    {
	XeRawImage *raw = NULL;
	int size;
	unsigned char *data = NULL, *p;
	Pixel pixel;

	if (!image)
		return NULL;
	size = image->height * image->bytes_per_line;
	if (image->format == XYBitmap)
	    {
		raw = XeCreateRawImage(1);
		if (!raw)
			goto convert_failed;
		if (copy_data)
		    {
			data = (unsigned char *)malloc(size);
			if (!data)
				goto convert_failed;
			memcpy(data, image->data, size);
		    }
		else
		    {
			data = (unsigned char *)image->data;
			image->data = NULL;
		    }
		raw->image_class = XeImageClass_BILEVEL;
		raw->width = image->width;
		raw->height = image->height;
		raw->lsbit_first = (image->bitmap_bit_order == LSBFirst);
		raw->samples_per_pixel = 1;
		raw->bits_per_component = 1;
		raw->bytes_per_line = image->bytes_per_line;
		raw->data = data;
		raw->alloc_data = True;
		raw->channel[0].addr = raw->data;
		raw->channel[0].line = raw->bytes_per_line;
		raw->channel[0].inc = 1;
		raw->channel[0].w = raw->width;
		raw->channel[0].h = raw->height;
	    }
	else
	    {
		XWindowAttributes attr;
		/* rm, gm, bm, rs, gs, bs initialized to silence compiler
		   warnings... The actual values have no significance */
		unsigned long rm = 0, gm = 0, bm = 0, m;
		int rs = 0, gs = 0, bs = 0;
		int i, x, y, channels;

		/*
		** The work arrays (pixels, r, g, b) are allocted as a single
		** chunk, and the work variables are only pointers inside it
		*/
		char *colors;
		Pixel *pixels;
		XeSample *r, *g, *b;

		XGetWindowAttributes(display, window, &attr);


		colors = (char *)XtMalloc
			(attr.visual->map_entries * 
			 (sizeof(*pixels) + 3 * sizeof(*r)));
		/*
		** Code below assumes that the Pixel has the most strict
		** alignment requirement, and XeSample is a type that
		** can be aligned to whatever results from reserving the
		** pixels first...
		*/
		pixels = (Pixel *)colors;
		r = (XeSample *)&pixels[attr.visual->map_entries];
		g = (XeSample *)&r[attr.visual->map_entries];
		b = (XeSample *)&g[attr.visual->map_entries];
		
		if (attr.visual->class == DirectColor ||
		    attr.visual->class == TrueColor)
		    {
			rm = attr.visual->red_mask;
			gm = attr.visual->green_mask;
			bm = attr.visual->blue_mask;

			for (m = 1, rs = 0; m && !(m & rm); ++rs, m <<= 1);
			for (m = 1, gs = 0; m && !(m & gm); ++gs, m <<= 1);
			for (m = 1, bs = 0; m && !(m & bm); ++bs, m <<= 1);

			for (i = 0; i < attr.visual->map_entries; ++i)
				pixels[i] =
					(rm & (i << rs)) | (gm & (i << gs)) |
						(bm & (i << bs));
			channels = 3;
		    }
		else
		    {
			for (i = 0; i < attr.visual->map_entries; ++i)
				pixels[i] = i;
			channels = 1;
		    }
		XeQueryColors(display, attr.colormap, attr.visual->map_entries,
			      pixels, r, g, b);
		raw = XeCreateRawImage(channels);
		raw->width = image->width;
		raw->height = image->height;
		raw->samples_per_pixel = channels;
		raw->bits_per_component = 8;
		raw->bytes_per_line = channels * raw->width;
		raw->data = data = (unsigned char *)malloc
			(raw->height * raw->bytes_per_line);
		if (!data)
			goto convert_failed;
		raw->alloc_data = True;
		raw->color_space = XeImageSpace_RGB;
		raw->bits_per_sample = 8; /* Kludge!! --msa */
		if (channels == 3)
		    {
			raw->image_class = XeImageClass_FULLCOLOR;
			rm = (rm >> rs);
			gm = (gm >> gs);
			bm = (bm >> bs);

			for (y = 0, p = data; y < image->height; ++y)
				for (x = 0; x < image->width; ++x)
				    {
					int ir, ig, ib;
					pixel = XGetPixel(image, x, y);
					ir = (pixel >> rs) & rm;
					ig = (pixel >> gs) & gm;
					ib = (pixel >> bs) & bm;
					*p++ = r[ir];
					*p++ = g[ig];
					*p++ = b[ib];
				    }
		    }
		else
		    {
			unsigned char *red, *green, *blue;

			raw->image_class = XeImageClass_PALETTE;
			raw->color_map = (unsigned char *)
				malloc((1 << raw->bits_per_sample) * 3);
			if (!raw->color_map)
				goto convert_failed;
			raw->alloc_map = True;
			for (y = 0, p = data; y < image->height; ++y)
				for (x = 0; x < image->width; ++x)
					*p++ = XGetPixel(image, x, y);
			red = raw->color_map;
			green = red + (1 << raw->bits_per_sample);
			blue = green + (1 << raw->bits_per_sample);
			for (i = 0; i < attr.visual->map_entries; ++i)
			    {
				red[i] = r[i];
				green[i] = g[i];
				blue[i] = b[i];
			    }
		    }
		for (i = 0, p = data; i < channels; ++i)
		    {
			raw->channel[i].addr = p + i;
			raw->channel[i].line = raw->bytes_per_line;
			raw->channel[i].inc = channels;
			raw->channel[i].w = raw->width;
			raw->channel[i].h = raw->height;
		    }
		XtFree(colors);
	    }
	return raw;

    convert_failed:
	if (data)
		free((void *)data);
	if (raw)
		XeDestroyRawImage(raw);
	return NULL;
    }
/*
** ***************
** Default methods
** ***************
*/
void _XeDefaultPrintCancel(w, clone, pp, client_data)
Widget w, clone;
XePrintProcess pp;
XtPointer client_data;
    {
    }

/*
** _XeDefaultPrint
*/
XtPointer _XeDefaultPrint(w, clone, x, y, pp)
Widget w, clone;
int x, y;
XePrintProcess pp;
    {
	XImage *image;

	image = XeGetXImage(w, 0, 0, w->core.width, w->core.height);
	if (image)
	    {
		XeRawImage *raw;
		XeImage *xe_image;
		XeImageChannel data[4];
		int i;
		XeImaging imaging;
		Display *display = XtDisplay(w);
		Window window = XtWindow(w);

		raw = XeRawImageFromXImage(display, window, image, False);
		imaging = pp->printer->imaging;
		if (imaging.max_colors == 0)
			imaging.max_colors = 65535;
		xe_image = _XeCreateImage
			(XtDisplay(clone), XtWindow(clone),
			 raw, &imaging, raw->width, raw->height);
		XDestroyImage(image);
		data[0].addr = (unsigned char *)xe_image->image->data;
		data[0].w = xe_image->image->width;
		data[0].h = xe_image->image->height;
		data[0].line = xe_image->image->bytes_per_line;
		data[0].inc = (xe_image->image->bits_per_pixel + 7) / 8;
		for (i = 1; i <= raw->num_channels && i < XtNumber(data); i++)
			data[i] = raw->channel[i-1];
		_XeBuildImage(xe_image, i, data);
		XPutImage(XtDisplay(clone), XtWindow(clone),
			  pp->gc,
			  xe_image->image, 0, 0, x, y,
			  w->core.width, w->core.height);
		_XeDestroyImage(xe_image);
		XeDestroyRawImage(raw);
	    }
	return NULL;
    }

