/*
** FrameSupport.c
**	A collection of definitions and miscellaneous utilities that
**	can be shared between XeFrame and XeText(Ed).
**
** Copyright 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>

extern int atoi( /* No header for this in SYSV */
#if NeedFunctionPrototypes
		char *
#endif
		);

#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <ctype.h>
#include <string.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xew/BasicP.h>
#include <X11/Xew/FrameP.h>
#include "FrameSupport.h"
#include "Color.h"


#undef MAX
#undef MIN
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define MIN(a,b) ((a) < (b) ? (a) : (b))

static CvtEnumInfo cvtBorderType[] =
    {
	{"invisible",	XeBorderType_INVISIBLE},
	{"solid",	XeBorderType_SOLID},
	{"dashed",	XeBorderType_DASHED},
	{"dot",		XeBorderType_DOT},
	{"dash-dot",	XeBorderType_DASH_DOT},
	{"dash-dot-dot",XeBorderType_DASH_DOT_DOT},
	{"shadow",	XeBorderType_SHADOW},
	{NULL,		sizeof(XeBorderType)},
    };

static CvtEnumInfo cvtFramePositionType[] =
    {
	{"normal",	XeFramePositionType_NORMAL},
	{"reverse",	XeFramePositionType_REVERSE},
	{"fixed",	XeFramePositionType_FIXED},
	{"inline",	XeFramePositionType_INLINE},
	{"start",	XeFramePositionType_START},
	{"end",		XeFramePositionType_END},
	{NULL,		sizeof(XeFramePositionType)},
    };

static CvtEnumInfo cvtFrameAlignment[] =
    {
	{"right",	XeFrameAlignment_RIGHT,},
	{"left",	XeFrameAlignment_LEFT,},
	{"center",	XeFrameAlignment_CENTER,},
	{"top",		XeFrameAlignment_TOP,},
	{"bottom",	XeFrameAlignment_BOTTOM,},
	{"above",	XeFrameAlignment_ABOVE,},
	{"around",	XeFrameAlignment_AROUND,},
	{"below",	XeFrameAlignment_BELOW,},
	{NULL,		sizeof(XeFrameAlignment)},
    };

static XeFramePosition init_FramePosition;
static XeFrameDimension init_FrameDimension;
static XeFrameBorder init_FrameBorder;

#define	CvtDone(type, value)				\
    do {						\
	if (toVal->addr != NULL)			\
	    {						\
		if (toVal->size < sizeof(type))		\
		    {					\
			toVal->size = sizeof(type);	\
			return False;			\
		    }					\
		*(type*)(toVal->addr) = (value);	\
	    }						\
	else						\
	    {						\
		static type static_val;			\
		static_val = (value);			\
		toVal->addr = (XPointer)&static_val;	\
	    }						\
	toVal->size = sizeof(type);			\
	return True;					\
    } while (0)


/*
** ExtractString
**	Extract a string terminated either by ',', ';' or NUL.
**
**	Ignore leading and trailing spaces. Returns a pointer to the
**	terminating
**	character.
*/
static char *ExtractString(s, str, len)
char *s, *str;
int len;
    {
	int i;
	
	if (len && s)
	    {
		while (isspace(*s))
			++s;
		for (i = 0 ;*s != ',' && *s != ';' && *s; ++s)
			if (i < len)
				str[i++] = *s;
		while (i > 0 && isspace(str[i-1]))
			i--;
		str[i < len ? i : len - 1] = 0;
	    }
	return s;
    }

/*
** StringToEnum
**	Find out the enumeration value corresponding the string.
**
** *NOTE*
**	The input string is clobbered by this function!!!
*/
static int StringToEnum(s, list, none)
char *s;		/* String to be searched */
CvtEnumInfo *list;	/* Enum definition list */
int none;		/* Value returned, if none matched */
    {
	XmuCopyISOLatin1Lowered(s, s);
	for (;list->name != NULL; list++)
		if (strcmp(s, list->name) == 0)
			return list->value;
	return none;
    }

/*
** ExtractNumberList
**	Extract a list of numbers from a comma separated number list.
**
**	At most 'len' numbers are extracted. If the list contains empty
**	element (like ",,"), the corresponding entry in the 'num' array
**	*not* changed (remember to initialize num array!).
**
** *NOTE*
**	Parsing is very loose, a string like "1234 foobar" will result
**	value 1234 and " foobar" gets silently ignored.
*/
static char *ExtractNumberList(s, num, len, cnt)
char *s;
int *num, len, *cnt;
    {
	int i = 0;
	char str[12];

	for (; s && *s && i < len; ++s)
	    {
		s = ExtractString(s, str, XtNumber(str));
		if (*str)	/* If non-empty */
			num[i] = atoi(str);
		i++;
		if (*s != ',')
			break;
	    }
	*cnt = i;
	return s;
    }

/*
** XeCvtStringToFrameDimension
**	Convert a string to XeFrameDimension structure
**
** <dimension> ::=
**	<number> |
**	["max" | "rule-a" | "rule-b"] ["," <number> ["," <number> ]]
**
**	Very loose parsing without error checks. Incorrect syntax
**	will most likely get you the default value silently.
*/
Boolean XeCvtStringToFrameDimension
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeFrameDimension dimension;
	int num[2], cnt;
	char str[20];
	char *s = (char *)fromVal->addr;
	
	dimension = init_FrameDimension;
	if (s != NULL)
	    {
		num[0] = num[1] = 0;
		s = ExtractString(s, str, XtNumber(str));
		if (*s == ',')
		    {
			s = ExtractNumberList(s + 1, num, XtNumber(num), &cnt);
			dimension.min = num[0];
			dimension.max = num[1];
		    }
		XmuCopyISOLatin1Lowered(str, str);
		if (strcmp(str, "max") == 0)
			dimension.value = XeFrameDimension_MAX;
		else if (strcmp(str, "rule-a") == 0)
			dimension.value = XeFrameDimension_RULE_A;
		else if (strcmp(str, "rule-b") == 0)
			dimension.value = XeFrameDimension_RULE_B;
		else if (str[0] == '+' || str[0] == '-' ||
			 (str[0] >= '0' && str[0] <= '9'))
			dimension.value = atoi(str);
	    }
	CvtDone(XeFrameDimension, dimension);
    }

/*
** XeCvtStringToFramePosition
**	Convert a string to XeFramePosition structure
**
** <position> ::=
**	"normal" | "reverse" | <number> ["," <number>]
**
**	Very loose parsing without error checks. Incorrect syntax
**	will most likely get you the default value silently.
*/
Boolean XeCvtStringToFramePosition
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeFramePosition position;
	int num[2], cnt;
	char str[20];
	char *s = (char *)fromVal->addr;
	
	position = init_FramePosition;
	if (s != NULL)
	    {
		num[0] = num[1] = 0;
		if (*s == '+' || *s == '-' || (*s >= '0' && *s <='9'))
		    {
			position.type = XeFramePositionType_FIXED;
			num[0] = num[1] = 0;
			s = ExtractNumberList(s, num, XtNumber(num), &cnt);
			position.x = num[0];
			position.y = num[1];
		    }
		else
		    {
			s = ExtractString(s, str, XtNumber(str));
			position.type = (XeFramePositionType)
				StringToEnum(str, cvtFramePositionType,
					     (int)XeFramePositionType_NORMAL);
		    }
	    }
	CvtDone(XeFramePosition, position);
    }

/*
** XeCvtStringToFrameBorder
**	Convert a string to XeFrameBorder structure
**
** <border>	::= <linetype> ["," <number> ["," <number> ["," <color> ]]]]
** <linetype>	::= "invisible" | "solid" | "dashed" | "dot" | "dash-dot" |
**			"dash-dot-dot" | "shadow"
*/
Boolean XeCvtStringToFrameBorder
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeFrameBorder border;
	int num[2], cnt;
	char str[30];
	char *s = (char *)fromVal->addr;
	Screen *screen;
	Colormap colormap;
	Status status;
	XColor color;
	Pixel bg, border_pixel;

	if (*num_args != 4)
	    {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"wrongParameters","XeCvtStringToFrameBorder",
				"XewError",
	      "String to FrameBorder needs screen, colormap and pixels.",
				(String *)NULL, (Cardinal *)NULL);
		return False;
	    }
	screen = *((Screen **)args[0].addr);
	colormap = *((Colormap *)args[1].addr);
	bg = *((Pixel *)args[2].addr);
	border_pixel = *((Pixel *)args[3].addr);
	border = init_FrameBorder;
	num[0] = border.width;
	num[1] = border.space;
	if (s == NULL)
		s = "";
	while (isspace(*s))
		++s;
	s = ExtractString(s, str, XtNumber(str));
	border.type =
		StringToEnum(str, cvtBorderType, XeBorderType_INVISIBLE);
	if (*s == ',')
		s = ExtractNumberList(s + 1, num, XtNumber(num), &cnt);
	border.width = num[0];
	border.space = num[1];
	if (*s == ',')
		++s;
	s = ExtractString(s, str, XtNumber(str));
	if (*str)
	    {
		status = XParseColor
			(DisplayOfScreen(screen),colormap,(char *)str,&color);
		if (status == 0)
		    {
			String params = str;
			Cardinal num_params = 1;

			XtAppWarningMsg
				(XtDisplayToApplicationContext(dpy),
				 "badColor","XeCvtStringToFrameBorder",
				 "XewWarning",
				 "Could not parse color '%s'.",
				 &params, &num_params);
		    }
		else
		    {
			border.color[0] = XToSample(color.red);
			border.color[1] = XToSample(color.green);
			border.color[2] = XToSample(color.blue);
			border.color[3] = 1;
		    }
	    }
	else
	    {
		if (border.type == XeBorderType_SHADOW)
			color.pixel = bg;
		else
			color.pixel = border_pixel;
		XeQueryColors(DisplayOfScreen(screen), colormap,
			      1, &color.pixel,
			      &border.color[0],
			      &border.color[1],
			      &border.color[2]);
		border.color[3] = 1;
	    }
	CvtDone(XeFrameBorder, border);
    }

/*
** _XeFrameConverters
**	Setup Resource converters for the Frame
*/
void _XeFrameConverters()
    {
	static XtConvertArgRec frameBorderArg[] =
	    {
		{
		    XtWidgetBaseOffset,
		    (XtPointer)XtOffsetOf(WidgetRec, core.screen),
		    sizeof(Screen *),
		},
		{
		    XtWidgetBaseOffset,
		    (XtPointer)XtOffsetOf(WidgetRec, core.colormap),
		    sizeof(Colormap),
		},
		{
		    XtWidgetBaseOffset,
		    (XtPointer)XtOffsetOf(WidgetRec,core.background_pixel),
		    sizeof(Pixel),
		},
		{
		    XtWidgetBaseOffset,
		    (XtPointer)XtOffsetOf(WidgetRec, core.border_pixel),
		    sizeof(Pixel),
		},
	    };
	static XtConvertArgRec frameAlignmentArg[] =
	    {
		{
		    XtAddress,
		    (XtPointer) &cvtFrameAlignment[0],
		    sizeof(XtPointer),
		}
	    };

	static int already_done;

	if (already_done)
		return;
	already_done = True;
	XtSetTypeConverter(XtRString, XeRFrameAlignment,
			   XeCvtStringToEnum,
			   frameAlignmentArg, XtNumber(frameAlignmentArg),
			   XtCacheNone, (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XeRFramePosition,
			   XeCvtStringToFramePosition,
			   (XtConvertArgRec *)NULL, 0,
			   XtCacheNone, (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XeRFrameDimension,
			   XeCvtStringToFrameDimension,
			   (XtConvertArgRec *)NULL, 0,
			   XtCacheNone, (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XeRFrameBorder,
			   XeCvtStringToFrameBorder,
			   frameBorderArg, XtNumber(frameBorderArg),
			   XtCacheAll, (XtDestructor)NULL);
    }


/*
** SetupGC
**	Modify frame.gc to match the BorderLine specification.
*/
#if NeedFunctionPrototypes
static void SetupGC(Widget, GC, XeFrameBorderP *, XeFrameBorderP *);
#endif

static void SetupGC(w, gc, current, line)
Widget w;
GC gc;
XeFrameBorderP *current;
XeFrameBorderP *line;
    {
	static unsigned char pattern_DASHED[] = {4,1};
	static unsigned char pattern_DOT[] = {1,1};
	static unsigned char pattern_DASH_DOT[] = {4,1,1,1};
	static unsigned char pattern_DASH_DOT_DOT[] = {4,1,1,1,1,1};

	unsigned char *pattern;
	unsigned char dash_list[sizeof(pattern_DASH_DOT_DOT)];
	int n;

	switch (line->value.type)
	    {
	    case XeBorderType_INVISIBLE:
		return;		/* If INVISIBLE, don't change anything!!! */
	    case XeBorderType_DASHED:
		pattern = pattern_DASHED;
		n = sizeof(pattern_DASHED);
		break;
	    case XeBorderType_DOT:
		pattern = pattern_DOT;
		n = sizeof(pattern_DOT);
		break;
	    case XeBorderType_DASH_DOT:
		pattern = pattern_DASH_DOT;
		n = sizeof(pattern_DASH_DOT);
		break;
	    case XeBorderType_DASH_DOT_DOT:
		pattern = pattern_DASH_DOT_DOT;
		n = sizeof(pattern_DASH_DOT_DOT);
		break;
	    default:
		pattern = NULL;
		n = 0;
		break;
	    }
	if (pattern && n > 0 &&
	    (line->value.type != current->value.type ||
	     line->value.width != current->value.width))
	    {
		int i, v, m = line->value.width != 0 ? line->value.width : 1;

		if (m < 0)
			m = -m; /* Probably an error... --msa */
		for (i = 0; i < n; i++)
		    {
			v = pattern[i] * m;
			dash_list[i] = v > 255 ? 255 : v;
		    }
		XSetDashes(XtDisplay(w), gc, dash_list[0] / 2,
			   (char *)&dash_list[0], n);
	    }
	if (line->value.type != current->value.type ||
	    line->value.width != current->value.width)
		XSetLineAttributes
			(XtDisplay(w), gc,
			 line->value.type == XeBorderType_SHADOW ? 0 :
			 line->value.width,
			 pattern ? LineOnOffDash : LineSolid,
			 CapButt, JoinMiter);
	if (line->pixel != current->pixel)
		XSetForeground(XtDisplay(w), gc,
			       line->pixel == FRAME_NO_PIXEL ?
			       w->core.border_pixel : line->pixel);
	*current = *line;
    }

#if NeedFunctionPrototypes
static void FreeBorderColor(Widget, XeFrameBorderP *);
#endif
static void FreeBorderColor(w, b)
Widget w;
XeFrameBorderP *b;
    {
	if (b->pixel != FRAME_NO_PIXEL)
	    {
		XeFreeColors(XtDisplay(w), w->core.colormap, 1, &b->pixel);
		b->pixel = FRAME_NO_PIXEL;
	    }
    }

/*
** ComputeTopShadowRGB
**	(This code is modified from Xaw3d library)
*/
static void ComputeTopShadowRGB (w, color, shadow_contrast)
Widget w;
XColor *color;
int shadow_contrast;
    {
	double contrast;
	
	if (color->red == color->green && color->green == color->blue &&
	    (color->red == 65535 || color->red == 0))
	    {
		contrast = (100 - shadow_contrast) / 100.0;
		color->red   = contrast * 65535.0;
		color->green = contrast * 65535.0;
		color->blue  = contrast * 65535.0;
	    } 
	else
	    {
		contrast = 1.0 + shadow_contrast / 100.0;
		color->red  = MIN(65535,(int)(contrast*(double)color->red));
		color->green= MIN(65535,(int)(contrast*(double)color->green));
		color->blue = MIN(65535,(int)(contrast*(double)color->blue));
	    }
    }

static void ComputeBottomShadowRGB (w, color, shadow_contrast)
Widget w;
XColor *color;
int shadow_contrast;
    {
	double contrast;

	if (color->red == color->green && color->green == color->blue &&
	    (color->red == 65535 || color->red == 0))
	    {
		contrast = shadow_contrast / 100.0;
		color->red   = contrast * 65535.0;
		color->green = contrast * 65535.0;
		color->blue  = contrast * 65535.0;
	    }
	else
	    {
		contrast = (100 - shadow_contrast) / 100.0;
		color->red  = MIN(65535,(int)(contrast*(double)color->red));
		color->green= MIN(65535,(int)(contrast*(double)color->green));
		color->blue = MIN(65535,(int)(contrast*(double)color->blue));
	    }
    }

#if NeedFunctionPrototypes
static void AllocBorderColor(Widget, XeFrameBorderP *, int, int *);
#endif
static void AllocBorderColor(w, border, edge, shadow_contrast)
Widget w;
XeFrameBorderP *border;
int edge, *shadow_contrast;
    {
	XColor color;
	XeSample r, g, b;

	color.red = border->value.color[0] * 257;
	color.green = border->value.color[1] * 257;
	color.blue = border->value.color[2] * 257;
	if (border->value.type == XeBorderType_SHADOW)
	    {
		if (edge == FRAME_TRAILING || edge == FRAME_LEFT)
			ComputeTopShadowRGB
				(w, &color, shadow_contrast[FRAME_TOP]);
		else
			ComputeBottomShadowRGB
				(w, &color, shadow_contrast[FRAME_BOTTOM]);
	    }
	else if (border->value.type == XeBorderType_INVISIBLE)
	    {
		border->pixel = FRAME_NO_PIXEL;
		return;
	    }
	r = XToSample(color.red);
	g = XToSample(color.green);
	b = XToSample(color.blue);
	(void)XeAllocColors(XtDisplay(w), XtWindow(w), w->core.colormap,
			    XeColormapUse_SHARED,
			    1, 1, &r, &g, &b, &border->pixel);
    }

#if NeedFunctionPrototypes
Boolean _XeChangeBorderColor
	(Widget, XeFrameBorderP *,XeFrameBorderP *, int, int *, int *);
#endif
Boolean _XeChangeBorderColor(w, old_b, set_b, edge, old_c, set_c)
Widget w;
XeFrameBorderP *old_b, *set_b;
int edge;
int *old_c, *set_c;
    {
	old_b += edge;
	set_b += edge;
	if (old_b->value.type == set_b->value.type &&
	    (set_b->value.type != XeBorderType_SHADOW ||
	     (old_c[FRAME_TOP] == set_c[FRAME_TOP] &&
	      old_c[FRAME_BOTTOM] == set_c[FRAME_BOTTOM])) &&
	    old_b->value.color[0] == set_b->value.color[0] &&
	    old_b->value.color[1] == set_b->value.color[1] &&
	    old_b->value.color[2] == set_b->value.color[2])
	    {
		set_b->pixel = old_b->pixel;
		return False;	/* Color not changed */
	    }
	FreeBorderColor(w, old_b);
	AllocBorderColor(w, set_b, edge, set_c);
	return True;
    }

/*
** _XeFrameInitialize
**	initialize XeFrameContraints record
*/
void _XeFrameInitialize(request, init, args, num_args)
Widget request, init;
ArgList args;
Cardinal *num_args;
    {
	XeFrameConstraints fc = (XeFrameConstraints)init->core.constraints;
	Widget w = XtParent(init);
	int i;

	if (fc)
		
		if (XtIsSubclass(init, xeFrameWidgetClass))
			/*
			** If the child is XeFrame, the constraint borders
			** are not used (the same values are set into the
			** XeFrame resources of the child and it will draw
			** own borders). Just initialize record.
			*/
			for (i = 0; i < XtNumber(fc->frame.border); ++i)
				fc->frame.border[i].pixel = FRAME_NO_PIXEL;
		else
			for (i = 0; i < XtNumber(fc->frame.border); ++i)
				AllocBorderColor(w, &fc->frame.border[i], i,
						 fc->frame.shadow_contrast);
    }
/*
** _XeFrameDestroy
**	destroy frame constraint record content.
*/
void _XeFrameDestroy(child)
Widget child;
    {
	Widget w = XtParent(child);
	XeFrameConstraints fc = (XeFrameConstraints)child->core.constraints;
	int i;

	if (fc)
	    {
		for (i = 0; i < XtNumber(fc->frame.border); ++i)
			FreeBorderColor(w, &fc->frame.border[i]);
	    }
    }

/*
** _XeDrawBorderRectangle
**
**	Uses XDrawLines only when dash patterns are requested. XDrawline
**	is unprecise... should draw the lines as rectangles with
**	stipple pattern to implement dots and dashes when needed --msa
**
**	Could also do better joining of corners for differing edges...
*/
void _XeDrawBorderRectangle
	(widget,gc,line,x,y,w,h,left,right,top,bottom,invert)
Widget widget;
GC gc;
XeFrameBorderP *line;
int x, y, w, h;
XeFrameBorderP *left, *right, *top, *bottom;
int invert;	/* Invert top/bottom shadow, if non-zero */
    {
	XeBorderType same_type;
	Window window = XtWindow(widget);
	Display *display = XtDisplay(widget);

	if (top->value.type == right->value.type &&
	    top->value.type == left->value.type &&
	    top->value.type == bottom->value.type)
		same_type = top->value.type;
	else
		/* 'INVISIBLE' to marks the case where border types differ */
		same_type = XeBorderType_INVISIBLE;
	
	if (same_type == XeBorderType_SHADOW)
	    {
		XPoint pt[6];

		SetupGC(widget, gc, line, invert ? bottom : top);
		pt[0].x = x;
		pt[0].y = y + h;
		pt[1].x = x;
		pt[1].y = y;
		pt[2].x = x + w;
		pt[2].y = y;
		pt[3].x = pt[2].x - right->value.width;
		pt[3].y = y + top->value.width;
		pt[4].x = x + left->value.width;
		pt[4].y = pt[3].y;
		pt[5].x = pt[4].x;
		pt[5].y = pt[0].y - bottom->value.width;
		XFillPolygon(display, window, gc, pt, XtNumber(pt),
			     Complex, CoordModeOrigin);
		SetupGC(widget, gc, line, invert ? top : bottom);
		pt[0].x = x;
		pt[0].y = y + h;
		pt[1].x = x + w;
		pt[1].y = pt[0].y;
		pt[2].x = pt[2].x;
		pt[2].y = y;
		pt[3].x = pt[2].x - right->value.width;
		pt[3].y = y + top->value.width;
		pt[4].x = pt[3].x;
		pt[4].y = pt[0].y - bottom->value.width;
		pt[5].x = x + left->value.width;
		pt[5].y = pt[4].y;
		XFillPolygon(display, window, gc, pt, XtNumber(pt),
			     Complex, CoordModeOrigin);
		return;
	    }
	/*
	** Make a special case, if all edges have the same parameters
	*/
	if (same_type &&
	    top->value.width == right->value.width &&
	    top->pixel == right->pixel &&

	    top->value.width == bottom->value.width &&
	    top->pixel == bottom->pixel &&

	    top->value.width == left->value.width &&
	    top->pixel == left->pixel)
	    {
		SetupGC(widget, gc, line, top);
		XDrawRectangle(display, window, gc, x, y, w, h);
		return;
	    }
	/*
	** Attempt to avoid bad joins at corners by drawing the lines
	** clockwise, each corner has at least one starting line, if
	** all edges have borders (sort of hack... does not work always )
	*/
	if (top->value.type)
	    {
		SetupGC(widget, gc, line, top);
		if (top->value.type == XeBorderType_SOLID)
			XFillRectangle(display, window, gc,
				       x, y, w, top->value.width);
		else
			XDrawLine(display, window, gc,
				  x,	y + top->value.width / 2,
				  x + w,y + top->value.width / 2);
	    }
	if (right->value.type)
	    {
		SetupGC(widget, gc, line, right);
		if (right->value.type == XeBorderType_SOLID)
			XFillRectangle(display, window, gc,
				       x + w - right->value.width, y,
				       right->value.width, h);

		else
			XDrawLine(display, window, gc,
				  x + w - (right->value.width+1)/2, y,
				  x + w - (right->value.width+1)/2, y + h);
	    }
	if (bottom->value.type)
	    {
		SetupGC(widget, gc, line, bottom);
		if (bottom->value.type == XeBorderType_SOLID)
			XFillRectangle(display, window, gc,
				       x, y + h - bottom->value.width,
				       w, bottom->value.width);
		else
			XDrawLine(display, window, gc,
				  x + w,y + h - (bottom->value.width + 1) / 2,
				  x,	y + h - (bottom->value.width + 1) / 2);
	    }
	if (left->value.type)
	    {
		SetupGC(widget, gc, line, left);
		if (left->value.type == XeBorderType_SOLID)
			XFillRectangle(display, window, gc,
				       x, y, left->value.width, h);
		else
			XDrawLine(XtDisplay(widget), XtWindow(widget), gc,
				  x + left->value.width / 2, y + h,
				  x + left->value.width / 2, y);
	    }
    }

/*
** _XeDrawOwnBorder
**	Draw Own Frame border inside the main window.
**
** *NOTE*
**	This function implicitly assumes that the widget is Xew widget
**	(e.g. the widget is a subclass of XeBasic!!!!)
*/
void _XeDrawOwnBorder(widget, gc, line, left, right, top, bottom, invert)
Widget widget;
GC gc;
XeFrameBorderP *line;
XeFrameBorderP *left, *right, *top, *bottom;
int invert;
     {
	int x, y, w, h;

	w = widget->core.width - left->value.space - right->value.space;
	h = widget->core.height - top->value.space - bottom->value.space;
	x = ((XeBasicWidget)widget)->basic.x_translation + left->value.space;
	y = ((XeBasicWidget)widget)->basic.y_translation + top->value.space;
	_XeDrawBorderRectangle
		(widget,gc,line,x,y,w,h, left,right,top,bottom,invert);
    }

/*
** _XeDrawChildBorder
**	Draw Child Frame border around the child widget.
**
** *NOTE*
**	This function implicitly assumes that the parent is Xew widget
**	(e.g. parent is a subclass of XeBasic!!!!)
*/
void _XeDrawChildBorder(widget, gc, line, child, left, right,top,bottom,invert)
Widget widget;
GC gc;
XeFrameBorderP *line;
Widget child;
XeFrameBorderP *left, *right, *top, *bottom;
int invert;
    {
	int x, y, w, h;
	int bw2 = child->core.border_width * 2;
	
	w = bw2 + child->core.width + left->value.space + left->value.width
		+ right->value.space + right->value.width;
	h = bw2 + child->core.height + top->value.space + top->value.width
		+ bottom->value.space + bottom->value.width;
	x = ((XeBasicWidget)widget)->basic.x_translation + child->core.x
		- left->value.space - left->value.width;
	y = ((XeBasicWidget)widget)->basic.y_translation + child->core.y
		- top->value.space - top->value.width;
	_XeDrawBorderRectangle
		(widget,gc,line,x,y,w,h, left,right,top,bottom,invert);
    }


static Boolean CheckBorder(old, set)
XeFrameBorderP *old, *set;
    {
	int i;

	for (i = 0; i < 4; ++i)
		if (old[i].value.width != set[i].value.width ||
		    old[i].value.space != set[i].value.space)
			return True;
	return False;
    }

static Boolean CheckSeparation(old, set)
int *old, *set;
    {
	int i;

	for (i = 0; i < 4; ++i)
		if (old[i] != set[i])
			return True;
	return False;
    }


static Boolean CheckOffset(old, set)
int *old, *set;
    {
	int i;
	for (i = 0; i < 4; ++i)
		if (old[i] != set[i])
			return True;
	return False;
    }

static Boolean CheckDimension(old, set)
XeFrameDimension *old, *set;
    {
	return	old[0].value != set[0].value ||
		old[0].min != set[0].min ||
		old[0].max != set[0].max ||
		old[1].value != set[1].value ||
		old[1].min != set[1].min ||
		old[1].max != set[1].max;
    }

static Boolean CheckPosition(old, set)
XeFramePosition *old, *set;
    {
	return	old->type != set->type ||
		(old->type == XeFramePositionType_FIXED &&
		 (old->x != set->x || old->y != set->y));
    }

/*
** _XeFrameSetValues
**	handle the basics of the constraints resource setting for
**	constraint setvalues method. Return a bitmask value
**
**	XeFrameSetValues_FRAMING
**		is set, if only the framing attributes (border) were
**		changed in a way that does not require new layout.
**	XeFrameSetValues_GEOMETRY
**		is set, if the geometry has been changed. This requires
**		a new layout computation from the parent.
*/
int _XeFrameSetValues(old, request, set, args, num_args)
Widget old, request, set;
ArgList args;
Cardinal *num_args;
    {
	Widget w = XtParent(set);
	XeFrameConstraints set_c = (XeFrameConstraints)set->core.constraints;
	XeFrameConstraints old_c = (XeFrameConstraints)old->core.constraints;
	int i, result = 0;

	/*
	** If the widget is a subclass of XeFrame, then it will take care of
	** it's own borders (border resources are only effective here for
	** non-Frame widgets.
	*/
	if (!XtIsSubclass(set, xeFrameWidgetClass))
	    {
		for (i = 0; i < XtNumber(set_c->frame.border); ++i)
			if (_XeChangeBorderColor
			    (w,old_c->frame.border,set_c->frame.border,i,
			     old_c->frame.shadow_contrast,
			     set_c->frame.shadow_contrast))
				result |= XeFrameSetValues_FRAMING;
		if (old_c->frame.invert_shadow != set_c->frame.invert_shadow)
			result |= XeFrameSetValues_FRAMING;
		if (CheckBorder(old_c->frame.border, set_c->frame.border))
			result |= XeFrameSetValues_GEOMETRY;
	    }
	if (CheckOffset(old_c->frame.offset, set_c->frame.offset) ||
	    CheckSeparation(old_c->frame.separation,set_c->frame.separation) ||
	    CheckDimension(old_c->frame.dimension, set_c->frame.dimension) ||
	    old_c->frame.alignment != set_c->frame.alignment ||
	    CheckPosition(&old_c->frame.position, &set_c->frame.position))
		result |= XeFrameSetValues_GEOMETRY;
	return result;
    }

/*
** _XeFrameCreateGC
**	create a graphics context for Frame border drawing and allocate
**	border colors.
*/
GC _XeFrameCreateGC(w, line, border, n, shadow_contrast)
Widget w;			/* Widget */
XeFrameBorderP *line;		/* Initialized to match GC */
XeFrameBorderP *border;		/* Borders */
int n;				/* Number of borders */
int *shadow_contrast;		/* TOP/BOTTOM constrast */
    {
	GC gc;
	int i;

	gc = XCreateGC(XtDisplay(w), RootWindowOfScreen(XtScreen(w)), 0, 0);
	line->value = init_FrameBorder;
	XSetLineAttributes(XtDisplay(w), gc,1,LineSolid,CapButt,JoinMiter);
	XSetForeground(XtDisplay(w), gc, w->core.border_pixel);
	XSetFunction(XtDisplay(w), gc, GXcopy);
	line->value.width = 1;
	line->value.type = XeBorderType_SOLID;
	for (i = 0; i < n; ++i)
		AllocBorderColor(w, &border[i], i, shadow_contrast);
	line->pixel = FRAME_NO_PIXEL;
	return gc;
    }

/*
** _XeFrameFreeGC
**	free the graphics context for Frame border drawing and release
**	border colors.
*/
void _XeFrameFreeGC(w, gc, border, n)
Widget w;
GC gc;
XeFrameBorderP *border;
int n;
    {
	int i;

	for (i = 0; i < n; ++i)
		FreeBorderColor(w, &border[i]);
	XFreeGC(XtDisplay(w), gc);
    }

#define BORDER(f,i) \
	((f)->frame.border[i].value.space + \
	 (f)->frame.border[i].value.width)

/*
** _XeFrameExtents
**	find out the current extents of the child widget assuming the
**	widget window has dimensions (width, height).
*/
void _XeFrameExtents(w, width, height, e)
Widget w;
int width, height;
XeFrameExtents *e;
    {
	static XeFrameExtents init_extents;
	XeFrameConstraints fc;

	*e = init_extents;
	fc = (XeFrameConstraints)w->core.constraints;
	if (!fc)
		return;
	e->window.width = width + 2 * w->core.border_width;
	e->window.height = height + 2 * w->core.border_width;
	e->border = e->window;
	e->border.width += BORDER(fc,FRAME_LEFT)+BORDER(fc,FRAME_RIGHT);
	e->border.height += BORDER(fc,FRAME_TRAILING)+BORDER(fc,FRAME_LEADING);
	e->width = e->border.width + fc->frame.separation[FRAME_LEFT] +
		fc->frame.separation[FRAME_RIGHT];
	e->height = e->border.height + fc->frame.separation[FRAME_TRAILING] +
		fc->frame.separation[FRAME_LEADING];
	e->border.x = fc->frame.separation[FRAME_LEFT];
	e->border.y = fc->frame.separation[FRAME_TRAILING];
	e->window.x = e->border.x + BORDER(fc,FRAME_LEFT);
	e->window.y = e->border.y + BORDER(fc,FRAME_TRAILING);
    }

static int LimitDimension(d, limit)
int d;
XeFrameDimension *limit;
    {
	if (limit->value > 0)
		return limit->value;
	if (limit->min > 0 && d < limit->min)
		d = limit->min;
	if (limit->max > 0 && d > limit->max)
		d = limit->max;
	return d;
    }

/*
** _XeFrameQueryExtents
**	find out the current dimensions of the child widget taking into
**	account the constraint resources.
**
** *NOTE*
**	The input width and height parameters are only to be used when
**	the corresponding constraint dimension requests 'max'.
*/
void _XeFrameQueryExtents(w, width, height, e)
Widget w;
int width, height;
XeFrameExtents *e;
    {
	static XeFrameExtents init_extents;
	XeFrameConstraints fc;
	XtWidgetGeometry intended, preferred;
	int value;

	*e = init_extents;
	if (!w || !XtIsManaged(w))
		return;
	fc = (XeFrameConstraints)w->core.constraints;
	if (!fc)
		return;
	if (width > 0)
	    {
		width -= BORDER(fc, FRAME_LEFT) + BORDER(fc,FRAME_RIGHT)
			+ fc->frame.separation[FRAME_LEFT]
			+ fc->frame.separation[FRAME_RIGHT];
		width = LimitDimension(width, &fc->frame.dimension[0]);
	    }
	if (height > 0)
	    {
		height -= BORDER(fc,FRAME_TRAILING) + BORDER(fc,FRAME_LEADING)
			+ fc->frame.separation[FRAME_TRAILING]
			+ fc->frame.separation[FRAME_LEADING];
		height = LimitDimension(height, &fc->frame.dimension[1]);
	    }
	intended.request_mode = 0;
	if ((value = fc->frame.dimension[0].value) > 0 ||
	    (fc->frame.dimension[0].value == XeFrameDimension_MAX &&
	     (value = width) > 0))
	    {
		intended.request_mode |= CWWidth;
		intended.width = width = value;
	    }
	else
		intended.width = width = 0;
	if ((value = fc->frame.dimension[1].value) > 0 ||
	    (fc->frame.dimension[1].value == XeFrameDimension_MAX &&
	     (value = height) > 0))
	    {
		intended.request_mode |= CWHeight;
		intended.height = height = value;
	    }
	else
		intended.height = height = 0;
	preferred = intended;
	if (((CWWidth|CWHeight) & intended.request_mode) != (CWWidth|CWHeight))
	    {
		/*
		** Width or height is still open, ask the widget for
		** it's preferred size on that dimension
		*/
		XtQueryGeometry(w, &intended, &preferred);
		intended.request_mode = CWWidth | CWHeight;
		if (width == 0)
			intended.width = LimitDimension
				(preferred.width, &fc->frame.dimension[0]);
		else
			intended.width = width;
		if (height == 0)
			intended.height = LimitDimension
				(preferred.height, &fc->frame.dimension[1]);
		else
			intended.height = height;
		/*
		** Re-query dimensions, if min/max constraints changed
		** the preferred values.
		*/
		if ((!width && intended.width != preferred.width) ||
		    (!height && intended.height != preferred.height))
		    {
			XtQueryGeometry(w, &intended, &preferred);
			if (width == 0)
				intended.width = LimitDimension
					(preferred.width,
					 &fc->frame.dimension[0]);
			if (height == 0)
				intended.height = LimitDimension
					(preferred.height,
					 &fc->frame.dimension[1]);
		    }
	    }
	/*
	** ...just an quick and dirty solution to the occasional "Widget
	** has zero width and/or height" error...
	*/
	if (intended.width == 0)
		intended.width = 1;
	if (intended.height == 0)
		intended.height = 1;
	_XeFrameExtents(w, intended.width, intended.height, e);
    }




