/*
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#include <X11/Xlib.h>
#include "Color.h"

#define NOPIX (~(unsigned long)0)

/*
** XeColorRec
*/
typedef struct XeColorRec
    {
	/*
	** The actual color values being used by the server for this pixel
	*/
	unsigned short red, green, blue;
	/*
	** Absolute value gives the count this pixel has been allocated by
	** this application. If the count is negative, the pixel value has
	** been used without allocating it (this means that the pixel is
	** allocated by some other application and we are just "borrowing"
	** the pixel without any guarantees (the actual pixel color may
	** not be what this application assumes)
	*/
	short allocated;
    } XeColorRec;

typedef struct XeColormap
    {
	Colormap colormap;	/* Colormap resource Id */
	Visual *visual;		/* Visual associated with the colormap */
	struct XeColormapDataBase *base; /* Back link to database */
	int private;		/* Non-Zero if private colormap */
	int allocated;		/* Allocated colors */
	int rs, gs, bs;		/* TrueColor/DirectColor field shifts */
	int num_colors;		/* Number of colors in map */
	XeColorRec colors[1];	/* <struct hack> */
    } XeColormap;

typedef struct XeColormapDataBase
    {
	struct XeColormapDataBase *next; /* Link to the next display */
	Display *display;	/* of the colormap */
	int count;		/* actual number of colormaps */
	XeColormap **colormaps;	/* array of colormap descriptors */
    } XeColormapDataBase;

/*
** FindVisual
**	return Visual associated with a window
*/
static Visual *FindVisual(display, window)
Display *display;
Window window;
    {
	XWindowAttributes attribs;

#if USING_XPRINTER
	/*
	** Currently only Xprinter library is known to have a display
	** structure without any screens. If a display has no screens
	** DefaultScreen will not work, and it is doubtful that the
	** XGetWindowAttributes would produce any better results either.
	** Thus, a kludge: if display has no screens, return a static
	** dummy visual pointer.
	*/
	if (ScreenCount(display) == 0)
	    {
		static Visual v;

		v.class = PseudoColor;
		v.bits_per_rgb = 8;
		v.map_entries = 256;
		return &v;
	    }
#endif
	if (!window)
		return DefaultVisual(display, DefaultScreen(display));
	XGetWindowAttributes(display, window, &attribs);
	return attribs.visual;
    }

/*
** FindColormap
**	locate internal colormap information (or allocate a new, if
**	such does not yet exist for the given colormap
*/
static XeColormap *FindColormap(display, window, colormap)
Display *display;
Window window;
Colormap colormap;
    {
	static XeColormapDataBase *color_data = NULL;

	XeColormapDataBase *cdb;
	XeColormap *cm;
	int i;
	XColor *cells;
	Colormap cmap = colormap;
	Visual *visual = NULL;

	if (!cmap)
	    {
		visual = FindVisual(display, window);
		cmap = XCreateColormap(display, window, visual, AllocNone);
	    }
	for (cdb = color_data; ; cdb = cdb->next)
		if (cdb == NULL)
		    {
			cdb = (XeColormapDataBase *)XtMalloc
				(sizeof(XeColormapDataBase));
			cdb->next = color_data;
			cdb->display = display;
			cdb->count = 1;
			cdb->colormaps =
				(XeColormap **)XtMalloc(sizeof(XeColormap *));
			cdb->colormaps[0] = NULL;
			color_data = cdb;
			break;
		    }
		else if (cdb->display == display)
		    {
			for (i = 0; i < cdb->count; ++i)
				if ((cm = cdb->colormaps[i]) != NULL &&
				    cm->colormap == cmap)
					return cm;
			break;
		    }
	/*
	** The colormap does not have an entry yet, make it
	*/
	for (i = 0; ; ++i)
	    {
		if (i == cdb->count)
		    {
			cdb->count += 1;
			cdb->colormaps = (XeColormap **)
				XtRealloc((char *)cdb->colormaps,
					  cdb->count * sizeof(XeColormap *));
			break;
		    }
		else if (cdb->colormaps[i] == NULL)
			break;
	    }
	if (visual == NULL)
		visual = FindVisual(display, window);
	cm = cdb->colormaps[i] = (XeColormap *)XtMalloc
		(XtOffsetOf(XeColormap, colors[0]) +
		 visual->map_entries * sizeof(XeColorRec));
	cm->colormap = cmap;
	cm->visual = visual;
	cm->base = cdb;
	cm->private = (cmap != colormap);
	cm->allocated = 0;
	cm->num_colors = visual->map_entries;
	cells = (XColor *)XtMalloc(cm->num_colors * sizeof(XColor));
	if (visual->class == DirectColor || visual->class == TrueColor)
	    {
		unsigned long rm, gm, bm, m;
		
		rm = visual->red_mask;
		gm = visual->green_mask;
		bm = visual->blue_mask;

		for (m = 1, cm->rs = 0; m && !(m & rm); ++cm->rs, m <<= 1);
		for (m = 1, cm->gs = 0; m && !(m & gm); ++cm->gs, m <<= 1);
		for (m = 1, cm->bs = 0; m && !(m & bm); ++cm->bs, m <<= 1);

		for (i = 0; i < visual->map_entries; ++i)
			cells[i].pixel = (rm & (i << cm->rs)) |
				(gm & (i << cm->gs)) |
				(bm & (i << cm->bs));
	    }
	else
		for (i = 0; i < cm->num_colors; ++i)
			cells[i].pixel = (unsigned long)i;

	XQueryColors(display, cmap, cells, cm->num_colors);
	for (i = 0; i < cm->num_colors; ++i)
	    {
		cm->colors[i].red   = cells[i].red;
		cm->colors[i].green = cells[i].green;
		cm->colors[i].blue  = cells[i].blue;
		cm->colors[i].allocated = 0;
	    }
	XtFree((char *)cells);
	return cm;
    }

/*
** RefreshColors
**	requeries unallocated color values from the server. The initial
**	values queried when the colormap structure was created are
**	out of date for such colors.
*/
static void RefreshColors(display, cm)
Display *display;
XeColormap *cm;
    {
	XColor *cells;
	int i, j;

	if (cm->visual->class == TrueColor ||
	    cm->visual->class == StaticColor ||
	    cm->visual->class == StaticGray)
		return;		/* Colors don't change in these visuals! */

	cells = (XColor *)XtMalloc(cm->num_colors * sizeof(XColor));
	if (cm->visual->class == DirectColor)
	    {
		unsigned long rm, gm, bm, m;
		
		rm = cm->visual->red_mask;
		gm = cm->visual->green_mask;
		bm = cm->visual->blue_mask;

		for (m = 1, cm->rs = 0; m && !(m & rm); ++cm->rs, m <<= 1);
		for (m = 1, cm->gs = 0; m && !(m & gm); ++cm->gs, m <<= 1);
		for (m = 1, cm->bs = 0; m && !(m & bm); ++cm->bs, m <<= 1);

		for (i = j = 0; i < cm->num_colors; ++i)
			if (cm->colors[i].allocated <= 0)
				cells[j++].pixel =
					(rm & (i << cm->rs)) |
					(gm & (i << cm->gs)) |
					(bm & (i << cm->bs));
	    }
	else
		/*
		** This works for PseudoColor and GrayScale
		*/
		for (i = j = 0; i < cm->num_colors; ++i)
			if (cm->colors[i].allocated <= 0)
				cells[j++].pixel = (unsigned long)i;
	if (j > 0)
	    {
		XQueryColors(display, cm->colormap, cells, j);
		for (i = j = 0; i < cm->num_colors; i++)
			if (cm->colors[i].allocated <= 0)
			    {
				/*
				  For future study: if allocated < 0 and
				  color is changed, it means that some
				  color used by this application does
				  not match the criteria it was originally
				  chosen... could do something about it?
				  -- msa
				  */

				cm->colors[i].red   = cells[j].red;
				cm->colors[i].green = cells[j].green;
				cm->colors[i].blue  = cells[j].blue;
				j++;
			    }
	    }
	XtFree((char *)cells);
    }

/*
** XeFreeColors
**	release colors allocated by XeAllocColors.
*/
void XeFreeColors(display, colormap, nc, pixels)
Display *display;
Colormap colormap;
int nc;
Pixel *pixels;
    {
	XeColormap *cm = FindColormap(display, None, colormap);
	unsigned long free_colors[256];
	int i, to_free = 0;

	if (cm->visual->class == TrueColor)
		return;

	for (i = 0; i < nc; i++)
	    {
		if (pixels[i] >= cm->num_colors)
			continue;
		if (cm->colors[pixels[i]].allocated < 0)
			cm->colors[pixels[i]].allocated += 1;
		else if (--cm->colors[pixels[i]].allocated == 0)
		    {
			if (to_free == XtNumber(free_colors))
			    {
				XFreeColors(display, cm->colormap,
					    free_colors, to_free, 0L);
				to_free = 0;
			    }
			free_colors[to_free++] = pixels[i];
		    }
		cm->allocated -= 1;
	    }
	if (cm->allocated == 0 && cm->private)
	    {
		for (i = 0; i < cm->base->count; i++)
			if (cm->base->colormaps[i] == cm)
			    {
				cm->base->colormaps[i] = NULL;
				XFreeColormap(display, cm->colormap);
				XtFree((char *)cm);
			    }
	    }
	else if (to_free > 0)
		XFreeColors(display, cm->colormap, free_colors, to_free, 0L);
    }


static int highbit(ul)
unsigned long ul;
    {
	/* returns position of highest set bit in 'ul' as an integer (0-31),
		   or -1 if none */
	
		int i;
	for (i=31; ((ul&0x80000000) == 0) && i>=0;  i--, ul<<=1);
	return i;
    }

#define FLIP(p) do {if ((p) < 0) p = - (p); } while (0)
 

static void InvalidPixelValue(cm, pixel)
XeColormap *cm;
Pixel pixel;
    {
	fprintf(stderr,	"X server returned invalid pixel %ld >= %d!\n",
		(long)pixel, cm->num_colors);
    }

/*
** XeAllocColors
**	allocate best possible colors. This function never fails,
**	all requested colors are always "allocated" (though, at worst
**	the returned colors may not have much common with the requested
**	colors).
*/
Colormap XeAllocColors
	(display, window, colormap, mode, max_alloc, nc, r, g, b, pixels)
Display *display;
Window window;
Colormap colormap;
XeColormapUse mode;
int max_alloc, nc;
XeSample *r, *g, *b;
Pixel *pixels;
    {
	XeColormap *cm = FindColormap(display, window, colormap);
	int num_done = 0, max_quota = 0;
	Colormap cmap = cm->colormap;
	int i, j;
	char *done;

	if (cm->visual->class == TrueColor)
	    {
		unsigned long rm = cm->visual->red_mask;
		unsigned long gm = cm->visual->green_mask;
		unsigned long bm = cm->visual->blue_mask;
		int rs, gs, bs;
		
		/*
		** Compute the amount of shifting that is required to position
		** the XeSample value into the final output pixel (to align the
		** most significant bits).
		*/
		rs = (XeSample_BITS-1) - highbit(rm);
		gs = (XeSample_BITS-1) - highbit(gm);
		bs = (XeSample_BITS-1) - highbit(bm);
		for (i = 0; i < nc; i++)
			pixels[i] =
				((rs > 0 ? r[i] >> rs : r[i] << -rs) & rm) |
				((gs > 0 ? g[i] >> gs : g[i] << -gs) & gm) |
				((bs > 0 ? b[i] >> bs : b[i] << -bs) & bm);
		rm = ~(rs > 0 ? (rm << rs) : (rm >> -rs)) & XeSample_MAX;
		gm = ~(gs > 0 ? (gm << gs) : (gm >> -gs)) & XeSample_MAX;
		bm = ~(bs > 0 ? (bm << bs) : (bm >> -bs)) & XeSample_MAX;
		for (i = 0; i < nc; i++)
		    {
			r[i] |= rm;
			g[i] |= gm;
			b[i] |= bm;
		    }
		return cmap;
	    }

	/*
	** 'done' array keeps track of unique pixels allocated
	*/
	done = (char *)XtCalloc(1, cm->num_colors);

	for (i = 0; i < nc; ++i)
		pixels[i] = NOPIX;
	/*
	** PASS 1: Locate all exact matching colors that have already been
	** allocated to the current application.
	*/
	for (i = 0; i < nc; ++i)
	    {
		unsigned short red = SampleToX(r[i]);
		unsigned short green = SampleToX(g[i]);
		unsigned short blue = SampleToX(b[i]);

		for (j = 0; j < cm->num_colors; ++j)
			if (cm->colors[j].allocated > 0 &&
			    cm->colors[j].red == red &&
			    cm->colors[j].green == green &&
			    cm->colors[j].blue == blue)
			    {
				pixels[i] = j;
				done[j] = True;
				max_quota += (i < max_alloc);
				cm->colors[j].allocated += 1;
				cm->allocated += 1;
				if (++num_done == nc)
					goto done_colors;
			    }
	    }
	/*
	** PASS 2: Try allocating still missing colors by XAllocColors
	*/
	for (i = 0; i < nc && max_quota < max_alloc; ++i)
	    {
		XColor color;

		if (pixels[i] != NOPIX)
			continue;
		color.red   = SampleToX(r[i]);
		color.green = SampleToX(g[i]);
		color.blue  = SampleToX(b[i]);
		if (XAllocColor(display, cm->colormap, &color))
		    {
			int pix = color.pixel;

			if (pix < 0 || pix >= cm->num_colors)
			    {
				/*
				** ..this really should not happen. The
				** server is broken if we get here...!
				*/
				InvalidPixelValue(cm, pix);
				break;
			    }
			if (!done[pix])
			    {
				done[pix] = True;
				max_quota += 1;
			    }
			pixels[i] = pix;
			/*
			** Server sanity check.. should probably be on
			** only when debugging... -- msa
			*/
			if (cm->colors[pix].allocated > 0 &&
			    (cm->colors[pix].red != color.red ||
			     cm->colors[pix].green != color.green ||
			     cm->colors[pix].blue != color.blue))
			    {
				fprintf(stderr,
		  "Server changed allocated color (%x,%x,%x) -> (%x,%x,%x)!\n",
					(int)cm->colors[pix].red,
					(int)cm->colors[pix].green,
					(int)cm->colors[pix].blue,
					(int)color.red,
					(int)color.green,
					(int)color.blue);
			    }
			cm->colors[pix].red = color.red;
			cm->colors[pix].green = color.green;
			cm->colors[pix].blue = color.blue;
			FLIP(cm->colors[pix].allocated);
			cm->colors[pix].allocated += 1;
			cm->allocated += 1;
			if (++num_done == nc)
				goto done_colors;
		    }
		else if (mode == XeColormapUse_OPTIONAL && cmap == colormap)
		    {
			/*
			** Release already allocated colors
			*/
			for (i = 0; i < nc; i++)
				if (pixels[i] != NOPIX)
				    {
					XeFreeColors
						(display, colormap,1,pixels+i);
					pixels[i] = NOPIX;
				    }
			XtFree((char *)done);
			cm = FindColormap(display, window, None);
			cmap = cm->colormap;
			done = (char *)XtCalloc(1, cm->num_colors);
			i = -1;
			continue;
		    }
		else
			/*
			** either we don't care about perfect color,
			** or we do care, have allocated our own
			** colormap, and have STILL run out of colors
			** (possible, even on an 8 bit display), just
			** mark pixel as unallocated.  We'll deal with
			** it later
			*/
			break;
	    }
	/*
	** Going to check colors not currently allocated to this application.
	** Need to refresh the cm information with the current state of such
	** colors, as they may have been changed since the last refresh...
	** (At best this is still approximation, the colors may change again
	** before the end of passes below).
	*/
	RefreshColors(display, cm);
	/*
	** PASS 3: Allocated closest colors
	*/
	for (i = 0; i < nc; ++i)
	    {
		int d, mdist, close;
		XColor color;

		if (pixels[i] != NOPIX)
			continue;
		mdist = 100000;
		close = -1;
		color.red   = SampleToX(r[i]);
		color.green = SampleToX(g[i]);
		color.blue  = SampleToX(b[i]);
		for (j = 0; j < cm->num_colors; j++)
		    {
			d = abs(color.red - cm->colors[j].red) +
			    abs(color.green - cm->colors[j].green) +
			    abs(color.blue - cm->colors[j].blue);
			if (d < mdist)
			    {
				mdist = d;
				close = j;
			    }
		    }
		color.red = cm->colors[close].red;
		color.green = cm->colors[close].green;
		color.blue = cm->colors[close].blue;
		if (cm->colors[close].allocated <= 0 &&
		    XAllocColor(display, cm->colormap, &color))
		    {
			close = color.pixel;
			if (close < 0 || close >= cm->num_colors)
			    {
				/*
				** ..this really should not happen. The
				** server is broken if we get here...!
				*/
				InvalidPixelValue(cm, close);
				continue;
			    }
			cm->colors[close].red = color.red;
			cm->colors[close].green = color.green;
			cm->colors[close].blue = color.blue;
		    }
		else
			continue;
		pixels[i] = close;
		done[close] = True;
		FLIP(cm->colors[close].allocated);
		cm->colors[close].allocated += 1;
		cm->allocated += 1;
		if (++num_done == nc)
			goto done_colors;
	    }
	/*
	** PASS 4: map all remaining colors to the closest existing colors
	*/
	for (i = 0; i < nc; i++)
	    {
		int d, mdist, close;
		unsigned short ri,gi,bi;

		if (pixels[i] != NOPIX)
			continue;

		mdist = 100000;   close = -1;
		ri = SampleToX(r[i]);
		gi = SampleToX(g[i]);
		bi = SampleToX(b[i]);
		if (mode != XeColormapUse_SHAREDOWN)
		    {   /* search the entire X colormap */
			for (j = 0; j < cm->num_colors; j++)
			    {
				d = abs(ri - cm->colors[j].red) +
				    abs(gi - cm->colors[j].green) +
				    abs(bi - cm->colors[j].blue);
				if (d < mdist)
				    {
					mdist = d;
					close = j;
				    }
			    }
		    }
		else
		    { 
			/* only search the alloc'd colors */
			for (j = 0; j < cm->num_colors; j++)
			    {
				if (cm->colors[j].allocated <= 0)
					continue;
				d = abs(ri - cm->colors[j].red) +
				    abs(gi - cm->colors[j].green) +
				    abs(bi - cm->colors[j].blue);
				if (d < mdist)
				    { 
					mdist = d;
					close = j;
				    }
			    }
		    }
		if (close < 0)
		    {
			/* No colors alloced??? */
		    }
		else
		    {
			pixels[i] = close;
			done[close] = True;
			if (cm->colors[close].allocated > 0)
				cm->colors[close].allocated += 1;
			else
				cm->colors[close].allocated -= 1;
			cm->allocated += 1;
			if (++num_done == nc)
				goto done_colors;
		    }
	    }
    done_colors:
	XtFree((char *)done);
	for (i = 0; i < nc; i++)
	    {
		r[i] = XToSample(cm->colors[pixels[i]].red);
		g[i] = XToSample(cm->colors[pixels[i]].green);
		b[i] = XToSample(cm->colors[pixels[i]].blue);
	    }
	return cmap;
    }

/*
** XeQueryColors
**	return RGB components of the specified pixel values
*/
void XeQueryColors(display, colormap, nc, pixels, r, g, b)
Display *display;
Colormap colormap;
int nc;
Pixel *pixels;
XeSample *r, *g, *b;
    {
	XeColormap *cm = FindColormap(display, None, colormap);
	int i;

	if (cm->visual->class == TrueColor || cm->visual->class == DirectColor)
	    {
		unsigned long rm, gm, bm;

		rm = cm->visual->red_mask >> cm->rs;
		gm = cm->visual->green_mask >> cm->gs;
		bm = cm->visual->blue_mask >> cm->bs;

		for (i = 0; i < nc; ++i)
		    {
			int ir, ig, ib;

			ir = (pixels[i] >> cm->rs) & rm;
			ig = (pixels[i] >> cm->gs) & gm;
			ib = (pixels[i] >> cm->bs) & bm;
			if (ir < cm->num_colors)
				r[i] = XToSample(cm->colors[ir].red);
			if (ig < cm->num_colors)
				g[i] = XToSample(cm->colors[ig].green);
			if (ib < cm->num_colors)
				b[i] = XToSample(cm->colors[ib].blue);
		    }
	    }
	else
	    {
		for (i = 0; i < nc; ++i)
			if (pixels[i] < cm->num_colors)
			    {
				r[i] = XToSample(cm->colors[pixels[i]].red);
				g[i] = XToSample(cm->colors[pixels[i]].green);
				b[i] = XToSample(cm->colors[pixels[i]].blue);
			    }
	    }
    }


