/*
** afm.c
**	Adobe Font Metric File (AFM) utilities
**
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xproto.h>
#include <X11/Xlib.h>
#include <sys/stat.h>
#include <math.h>
#include <ctype.h>

#include "printer.h"
#include "ps_types.h"
#include "afm.h"

typedef struct CharData
    {
	int code;
	char *name;
	int wx;
	int b[4];
    } CharData;

static FontData init_FontData;
static CharData init_CharData;
static XFontStruct init_XFontStruct;

/*
** fonts
**	An array of FontData pointers. This list defines
**	the current font set availble to the clients.
**
** fonts_size
**	is the current size of the fonts array.
**
** fonts_num
**	actual number of fonts in the array
*/
static FontData **fonts;
static int fonts_num;
static int fonts_size;
/*
** aliases
**	An array of (alias, XLFD) pairs.
**
** aliases_size
**	is the current size of aliases, expressed in *PAIRS*
**
** aliases_num
**	actual number of pairs in the array
*/
static char **aliases;
static int aliases_num;
static int aliases_size;
/*
** font_path
**	current font path, a NULL terminated list of strings (array and
**	list malloc'ed)
*/
static char **font_path;

typedef struct SectionDef *SectionDefPtr;
typedef struct KeyDef *KeyDefPtr;

typedef void (*CharHandler)(CharData *, char **, char *);
typedef void (*KeyHandler)
(FontData *, SectionDefPtr, KeyDefPtr, char *, char *);
typedef void (*SectionHandler)
(FontData *, SectionDefPtr, char *, char *);

typedef struct CharDef
    {
	char *name;
	CharHandler handler;
    } CharDef;

typedef struct KeyDef
    {
	char *name;
	KeyHandler handler;
    } KeyDef;

typedef struct SectionDef
    {
	char *name;
	SectionHandler handler;
	KeyDef *keys;
    } SectionDef;

/*
** Mapping from StandardAdobeEncoding into ISO 8859-1 encoding
*/
typedef struct CharConvert
    {
	char *name;
	int code;
    } CharConvert;

static CharConvert adobe_8859_1[] =
    {
/* 32 */
	{"space",	32},
	{"exclam",	33},
	{"quotedbl",	34},
	{"numbersign",	35},
	{"dollar",	36},
	{"percent",	37},
	{"ampersand",	38},
	{"quoteright",	39},
	{"parenleft",	40},
	{"parenright",	41},
	{"asterisk",	42},
	{"plus",	43},
	{"comma",	44},
	{"hyphen",	45},
	{"period",	46},
	{"slash",	47},

/* 48 */

	{"zero",	48},
	{"one",		49},
	{"two",		50},
	{"three",	51},
	{"four",	52},
	{"five",	53},
	{"six",		54},
	{"seven",	55},
	{"eight",	56},
	{"nine",	57},
	{"colon",	58},
	{"semicolon",	59},
	{"less",	60},
	{"equal",	61},
	{"greater",	62},
	{"question",	63},

/* 64 */

	{"at",		64},
	{"A",		65},
	{"B",		66},
	{"C",		67},
	{"D",		68},
	{"E",		69},
	{"F",		70},
	{"G",		71},
	{"H",		72},
	{"I",		73},
	{"J",		74},
	{"K",		75},
	{"L",		76},
	{"M",		77},
	{"N",		78},
	{"O",		79},

/* 80 */

	{"P",		80},
	{"Q",		81},
	{"R",		82},
	{"S",		83},
	{"T",		84},
	{"U",		85},
	{"V",		86},
	{"W",		87},
	{"X",		88},
	{"Y",		89},
	{"Z",		90},
	{"bracketleft",	91},
	{"backslash",	92},
	{"bracketright",93},
	{"asciicircum",	94},
	{"underscore",	95},

/* 96 */

	{"quoteleft",	96},
	{"a",		97},
	{"b",		98},
	{"c",		99},
	{"d",		100},
	{"e",		101},
	{"f",		102},
	{"g",		103},
	{"h",		104},
	{"i",		105},
	{"j",		106},
	{"k",		107},
	{"l",		108},
	{"m",		109},
	{"n",		110},
	{"o",		111},

/* 112 */

	{"p",		112},
	{"q",		113},
	{"r",		114},
	{"s",		115},
	{"t",		116},
	{"u",		117},
	{"v",		118},
	{"w",		119},
	{"x",		120},
	{"y",		121},
	{"z",		122},
	{"braceleft",	123},
	{"bar",		124},
	{"braceright",	125},
	{"asciitilde",	126},
	{NULL,	127},

/* 128 */

	{NULL,	128},
	{NULL,	129},
	{NULL,	130},
	{NULL,	131},
	{NULL,	132},
	{NULL,	133},
	{NULL,	134},
	{NULL,	135},
	{NULL,	136},
	{NULL,	137},
	{NULL,	138},
	{NULL,	139},
	{NULL,	140},
	{NULL,	141},
	{NULL,	142},
	{NULL,	143},

/* 144 */

	{NULL,	144},
	{NULL,	145},
	{NULL,	146},
	{NULL,	147},
	{NULL,	148},
	{NULL,	149},
	{NULL,	150},
	{NULL,	151},
	{NULL,	152},
	{NULL,	153},
	{NULL,	154},
	{NULL,	155},
	{NULL,	156},
	{NULL,	157},
	{NULL,	158},
	{NULL,	159},

/* 160 */

	{NULL,	160},
	{"exclamdown",	161},
	{"cent",	162},
	{"sterling",	163},
	{"fraction",	-1},
	{"yen",		165},
	{"florin",	-1},
	{"section",	167},

/* 168 */

	{"currency",	164},
	{"quotesingle",	-1},
	{"quotedblleft",-1},
	{"guillemotleft",0xAB},
	{"guilsinglleft",-1},
	{"guilsinglright",-1},
	{"fi",		-1},
	{"fl",		-1},

/* 176 */

	{NULL,		-1},
	{"endash",	-1},
	{"dagger",	-1},
	{"daggerdbl",	-1},
	{"periodcentered",0xB7},
	{NULL,		-1},
	{"paragraph",	0xB6},
	{"bullet",	-1},

/* 184 */

	{"quotesinglbase",-1},
	{"quotedblbase",-1},
	{"quotedblright",-1},
	{"guillemotright",0xBB},
	{"ellipsis",	-1},
	{"perthousand",	-1},
	{NULL,		-1},
	{"questiondown",0xBF},

/* 192 */

	{"Oacute",	0xD3},
	{"grave",	-1},
	{"acute",	0xB4},
	{"circumflex",	-1},
	{"tilde",	-1},
	{"macron",	0xAF},
	{"breve",	-1},
	{"dotaccent",	-1},

/* 200 */

	{"dieresis",	0xA8},
	{"Adieresis",	0xC4},
	{"ring",	-1},
	{"cedilla",	0xB8},
	{NULL,		-1},
	{"hungarumlaut",-1},
	{"ogonek",	-1},
	{"caron",	-1},

/* 208 */

	{"emdash",	-1},
	{"oacute",	0xF3},
	{"Ograve",	0xD2},
	{NULL,		-1},
	{NULL,		-1},
	{"scaron",	-1},
	{NULL,		-1},
	{NULL,		-1},

/* 216 */

	{NULL,		-1},
	{"adieresis",	0xE4},
	{NULL,		-1},
	{"Odieresis",	0xD6},
	{"odieresis",	0xF6},
	{"Otilde",	0xD5},
	{NULL,		-1},
	{NULL,		-1},

/* 224 */

	{"Aacute",	0xC1},
	{"AE",		0xC6},
	{"Aring",	0xC5},
	{"ordfeminine",	0xAA},
	{"Zcaron",	-1},
	{NULL,		-1},
	{NULL,		-1},
	{"Eacute",	0xC9},

/* 232 */

	{"Lslash",	-1},
	{"Oslash",	0xD8},
	{"OE",		-1},
	{"ordmasculine",0xBA},
	{NULL,		-1},
	{NULL,		-1},
	{NULL,		-1},
	{NULL,		-1},

/* 240 */

	{"aacute",	0xE1},
	{"ae",		0xE6},
	{"aring",	0xE5},
	{NULL,		-1},
	{"zcaron",	-1},
	{"dotlessi",	-1},
	{NULL,		-1},
	{"eacute",	0xE9},

/* 248 */

	{"lslash",	-1},
	{"oslash",	0xF8},
	{"oe",		-1},
	{"germandbls",	0xDF},
	{NULL,		-1},
	{NULL,		-1},
	{NULL,		-1},
	{NULL,		-1},

/* 256 */

	{"Acircumflex",	0xC2},
	{"Agrave",	0xC0},
	{"Atilde",	0xC3},
	{"Ccedilla",	0xC7},
	{"Ecircumflex",	0xCA},
	{"Edieresis",	0xCB},
	{"Egrave",	0xC8},
	{"Eth",		0xD0},
	{"Iacute",	0xCD},
	{"Icircumflex",	0xCE},
	{"Idieresis",	0xCF},
	{"Igrave",	0xCC},
	{"Ntilde",	0xD1},
	{"Ocircumflex",	0xD4},
	{"Scaron",	-1},
	{"Thorn",	0xDE},
	{"Uacute",	0xDA},
	{"Ucircumflex",	0xDB},
	{"Udieresis",	0xDC},
	{"Ugrave",	0xD9},
	{"Yacute",	0xDD},
	{"Ydieresis",	-1},
	{"acircumflex",	0xE2},
	{"agrave",	0xE0},
	{"atilde",	0xE3},
	{"brokenbar",	0xA6},
	{"ccedilla",	0xE7},
	{"copyright",	0xA9},
	{"degree",	0xB0},
	{"divide",	0xF7},
	{"ecircumflex",	0xEA},
	{"edieresis",	0xEB},
	{"egrave",	0xE8},
	{"eth",		0xF0},
	{"iacute",	0xED},
	{"icircumflex",	0xEE},
	{"idieresis",	0xEF},
	{"igrave",	0xEC},
	{"logicalnot",	0xAC},
	{"minus",	0xAD},
	{"mu",		0xB5},
	{"multiply",	0xD7},
	{"ntilde",	0xF1},
	{"ocircumflex",	0xF4},
	{"ograve",	0xF2},
	{"onehalf",	0xBD},
	{"onequarter",	0xBC},
	{"onesuperior",	0xB9},
	{"otilde",	0xF5},
	{"plusminus",	0xB1},
	{"registered",	0xAE},
	{"thorn",	0xFE},
	{"threequarters",0xBE},
	{"threesuperior",0xB3},
	{"trademark",	-1},
	{"twosuperior",	0xB2},
	{"uacute",	0xFA},
	{"ucircumflex",	0xFB},
	{"udieresis",	0xFC},
	{"ugrave",	0xF9},
	{"yacute",	0xFD},
	{"ydieresis",	0xFF},
    };

#define ADOBE_8859_1_SIZE (sizeof(adobe_8859_1) / sizeof(adobe_8859_1[0]))

#define KEY_PARAMS	FontData *, SectionDef *, KeyDef *, char *, char *
#define SECTION_PARAMS	FontData *, SectionDef *, char *, char *
#define CHAR_PARAMS	CharData *, char **, char *

static void handle_FontName	(KEY_PARAMS);
static void handle_Weight	(KEY_PARAMS);
static void handle_FamilyName	(KEY_PARAMS);
static void handle_FontBBox	(KEY_PARAMS);
static void handle_EncodingScheme(KEY_PARAMS);
static void handle_Characters	(KEY_PARAMS);
static void handle_Ascender	(KEY_PARAMS);
static void handle_Descender	(KEY_PARAMS);
static void handle_ItalicAngle	(KEY_PARAMS);
static void handle_CharWidth	(KEY_PARAMS);
static void handle_IsFixedPitch	(KEY_PARAMS);

static void handle_C		(KEY_PARAMS);
static void handle_CH		(KEY_PARAMS);

static void handle_char_WX	(CHAR_PARAMS);
static void handle_char_N	(CHAR_PARAMS);
static void handle_char_B	(CHAR_PARAMS);

static CharDef char_metrics_attributes[] =
    {
	{"WX",	handle_char_WX,	},
	{"N",	handle_char_N,	},
	{"B",	handle_char_B,	},
	{NULL},
    };

static KeyDef font_metrics_keys[] = 
    {
	    {"FontName",	handle_FontName,},
	    {"Weight",		handle_Weight,},
	    {"FamilyName",	handle_FamilyName,},
	    {"FontBBox",	handle_FontBBox,},
	    {"EncodingScheme",	handle_EncodingScheme,},
	    {"Characters",	handle_Characters,},
	    {"Ascender",	handle_Ascender,},
	    {"Descender",	handle_Descender,},
	    {"ItalicAngle",	handle_ItalicAngle},
	    {"CharWidth",	handle_CharWidth,},
	    {"IsFixedPitch",	handle_IsFixedPitch},
	    {NULL},
    };

static KeyDef char_metrics_keys[] =
    {
	    {"C",		handle_C,},
	    {"CH",		handle_CH,},
	    {NULL},
    };

static void handle_FontMetrics	(SECTION_PARAMS);
static void handle_CharMetrics	(SECTION_PARAMS);

static SectionDef section_table[] =
    {
	    {"FontMetrics",	handle_FontMetrics,	font_metrics_keys},
	    {"CharMetrics",	handle_CharMetrics,	char_metrics_keys},
	    {NULL},
    };

/*
** ReadFile
**	reads all of the file content into a memory buffer
**
**	Return buffer address or NULL, and number of bytes read.
**	(buffer is allocated one byte longer and NUL terminated).
*/
static char *ReadFile(char *name, long *length)
    {
	struct stat info;
	char *buffer;
	FILE *fp;

	if (!name || *name == 0)
		return NULL;
	if (stat(name, &info))
		return NULL;	/* Non-Accessible file */
	if ((info.st_mode & S_IFMT) == S_IFDIR)
		return NULL;	/* A directory, not a file */
	if ((fp = fopen(name, "rb")) == NULL)
		return NULL;
	buffer = (char *)malloc(info.st_size + 1);
	if (!buffer)
	    {
		fclose(fp);
		return NULL;	/* Out of memory! */
	    }
	*length = fread(buffer, 1, info.st_size, fp);
	fclose(fp);
	if (*length < 0)
	    {
		free(buffer);
		return NULL;
	    }
	buffer[*length] = 0;
	return buffer;
    }

/*
** BoundingBox
**	translate AFM bounding box (llx, lly, urx, ury) into XCharStruct
**	information
*/
static void BoundingBox(XCharStruct *ch, int *bbox)
    {
	ch->ascent = bbox[3];		/*   ury */
	ch->descent = - bbox[1];	/* - lly */
	ch->lbearing = bbox[0];		/*   llx */
	ch->rbearing = bbox[2];		/*   urx */
    }

/*
** Parsing Utility routines
*/
static int IsToken(char *s, char *e, char *token)
    {
	char *test;
	while (s < e && isspace(*s))
		++s;
	test = s;
	while (s < e && !isspace(*s))
		++s;
	*s = 0;
	return strcmp(test, token) == 0;
    }

static char *Token(char **sptr, char *e)
    {
	char *token, *s = *sptr;

	while (s < e && isspace(*s))
		++s;
	token = s;
	while (s < e && !isspace(*s))
		++s;
	*s = 0;
	if (s < e)
		++s;
	*sptr = s;
	return token;
    }

/*
** Individual Character metrics handlers
*/
static void handle_char_WX(CharData *c, char **s, char *e)
    {
	c->wx = atoi(*s);
    }

static void handle_char_N(CharData *c, char **s, char *e)
    {
	c->name = Token(s, e);
    }

static void handle_char_B(CharData *c, char **s, char *e)
    {
	int i;

	for (i = 0; *s < e && i < 4; ++i)
		c->b[i] = atoi(Token(s, e));
    }

/*
*/
static void handle_FontName
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->font_name = s;
	*e = 0;
    }

static void handle_Weight
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->weight = s;
	*e = 0;
    }

static void handle_FamilyName
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->family_name = s;
	*e = 0;
    }

static void handle_FontBBox
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	int bbox[4], i;

	for (i = 0; s < e && i < 4; ++i)
		bbox[i] = atoi(Token(&s, e));
	BoundingBox(&f->font_info->max_bounds, bbox);

    }

static void handle_EncodingScheme
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->encoding_scheme = s;
	/*
	** As a quick solution, if encoding is 'AdobeStandardEncoding'
	** show it to clients as 'iso88591-1' encoding.
	*/
	f->iso_8859_1 = strcmp(s, "AdobeStandardEncoding") == 0;
    }

static void handle_Characters
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->characters = atoi(s);
    }

static void handle_Ascender
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
    }

static void handle_Descender
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
    }

static void handle_ItalicAngle
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->italic_angle = atof(s);
    }

static void handle_CharWidth
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {

    }

static void handle_IsFixedPitch
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	f->is_fixed_pitch = IsToken(s, e, "true");
    }

static void ParseCharacter(FontData *f, int code, char *s, char *e)
    {
	char *attribute;
	CharDef *chd;
	CharData data;

	data = init_CharData;
	data.code = code;
	while (s < e)
	    {
		s = strchr(s, ';');
		if (!s)
			break;
		++s;
		while (s < e && isspace(*s))
			++s;
		attribute = s;
		while (s < e && !isspace(*s))
			++s;
		*s++ = 0;
		for (chd = char_metrics_attributes; chd->name; chd++)
			if (strcmp(chd->name, attribute) == 0)
				break;
		if (chd->handler)
			(*chd->handler)(&data, &s, e);
	    }
	if (f->iso_8859_1)
	    {
		if (code < 0)
		    {
			CharConvert *cc;

			for (cc = adobe_8859_1;;++cc)
			    {
				if (cc == &adobe_8859_1[ADOBE_8859_1_SIZE])
					break;
				else if (cc->name &&
					 strcmp(data.name, cc->name) == 0)
				    {
					code = cc->code;
					break;
				    }
			    }
		    }
		else if (code >= 32 && code - 32 < ADOBE_8859_1_SIZE)
			code = adobe_8859_1[code-32].code;
		else
			code = -1;
	    }
	if (code >= f->font_info->min_char_or_byte2 &&
	    code <= f->font_info->max_char_or_byte2)
	    {
		XCharStruct *ch;

		code -= f->font_info->min_char_or_byte2;
		ch = f->font_info->per_char + code;
		ch->width = data.wx;
		BoundingBox(ch, data.b);
	    }
    }

static void handle_C
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
	ParseCharacter(f, atoi(s), s, e);
    }

static void handle_CH
	(FontData *f, SectionDef *section, KeyDef *key, char *s, char *e)
    {
#if 0
	ParseCharacter(f, hextoi(s), s, e);
#endif
    }


static void handle_FontMetrics
	(FontData *f, SectionDef *section, char *s, char *e)
    {
    }
	
static void handle_CharMetrics
	(FontData *f, SectionDef *section, char *s, char *e)
    {
	if (f->iso_8859_1)
	    {
		f->font_info->min_char_or_byte2 = 0x20;
		f->font_info->max_char_or_byte2 = 0xFF;
		f->font_info->default_char = 0x20;
	    }
	else
	    {
		f->font_info->min_char_or_byte2 = 0x00;
		f->font_info->max_char_or_byte2 = 0xFF;
		f->font_info->default_char = 0x20;
	    }
	f->font_info->per_char = (XCharStruct *)calloc
		(f->font_info->max_char_or_byte2 -
		 f->font_info->min_char_or_byte2 + 1,
		 sizeof(XCharStruct));
    }

static char *ParseSection(FontData *f, char *s, char *e)
    {
	char *le;
	char *section_name, *key_name;
	SectionDef *section;
	KeyDef *key;

	section_name = s;
	le = strchr(s, '\n');
	if (!le)
		return e;
	while (s < le && !isspace(*s))
		++s;
	*s++ = 0;
	for (section = section_table; section->name; section++)
		if (strcmp(section->name, section_name) == 0)
			break;
	if (section->handler)
		(*section->handler)(f, section, s, le);
	s = le + 1;
	while (s < e)
	    {
		if ((le = strchr(s, '\n')) == NULL)
			break;
		if (le - s >= 5 && memcmp(s, "Start", 5) == 0)
			s = ParseSection(f, s + 5, e);
		else if (le - s >= 3 && memcmp(s, "End", 3) == 0)
			/* End of section, just assume it's correct! */
			return le + 1;
		else
		    {
			*le = 0;
			if (section->keys)
			    {
				key_name = s;
				while (s < le && !isspace(*s))
					++s;
				*s++ = 0;
				for (key = section->keys; key->name; ++key)
					if (strcmp(key->name, key_name) == 0)
						break;
				if (key->handler)
					(*key->handler)(f,section,key,s,le);
			    }
			s = le + 1;
		    }
	    }
	return e;
    }

static FontData *ParseAFM(char *afm_file)
    {
	char *buffer;
	long length;
	char *s, *e;
	FontData *f = NULL;

	buffer = ReadFile(afm_file, &length);
	if (!buffer)
		return NULL;	/* Out of memory or bad path! */
	/*
	** Locate Section "StartFontMetrics"
	*/
	s = buffer;
	e = &buffer[length];
	while (s < e)
	    {
		if (e - s >= 16 && memcmp(s, "StartFontMetrics", 16) == 0)
		    {
			int spacing;

			f = (FontData *)malloc(sizeof(*f));
			if (!f)
				break;
			*f = init_FontData;
			f->references = 1;
			f->scale = 1000;
			f->font_info = (XFontStruct *)
				malloc(sizeof(*f->font_info));
			if (!f->font_info)
			    {
				free((char *)f);
				f = NULL;
				break;
			    }
			*f->font_info = init_XFontStruct;
			/*
			** Add PIXEL_SIZE property
			*/
			f->font_info->n_properties = 1;
			f->font_info->properties =
				(XFontProp *)malloc(sizeof(XFontProp));
			f->font_info->properties[0].name =
				internAtom("PIXEL_SIZE");
			f->font_info->properties[0].card32 = f->scale;
			(void)ParseSection(f, s + 5, e);
			/*
			** Compute font ascent/descent. X font ascent and
			** descent are used for line spacing and are supposed
			** to include some white space that AFM FontBBox
			** does not include (max_bounds). The real spacing
			** is f->scale units, distribute extra white space
			** evenly to font ascent and descent...
			*/
			spacing = f->scale
				- f->font_info->max_bounds.ascent
				- f->font_info->max_bounds.descent;
			f->font_info->ascent =
				f->font_info->max_bounds.ascent + spacing / 2;
			spacing -= spacing / 2;
			f->font_info->descent =
				f->font_info->max_bounds.descent + spacing;
			/*
			** Things possibly still needed?
			** - compute min_bounds/max_bounds from the
			** char information?
			** - set all_chars_exist properly?
			** - default_char?
			** - direction?
			*/
			break;
		    }
		/*
		** just skip the line (anything else than empty is actually
		** an error in format, but let it pass..)
		*/
		if ((s = strchr(s, '\n')) == NULL)
			break;
		s += 1;
	    }
	if (f)
	    {
		if (f->font_name)
			f->font_name = strdup(f->font_name);
		if (f->family_name)
			f->family_name = strdup(f->family_name);
		if (f->weight)
			f->weight = strdup(f->weight);
		if (f->encoding_scheme)
			f->encoding_scheme = strdup(f->encoding_scheme);
	    }
	free(buffer);
	return f;
    }

typedef enum
    {
	XLFD_FOUNDRY,
	XLFD_FAMILY_NAME,
	XLFD_WEIGHT_NAME,
	XLFD_SLANT,
	XLFD_SETWIDTH_NAME,
	XLFD_ADD_STYLE_NAME,
	XLFD_PIXEL_SIZE,
	XLFD_POINT_SIZE,
	XLFD_RESOLUTION_X,
	XLFD_RESOLUTION_Y,
	XLFD_SPACING,
	XLFD_AVERAGE_WIDTH,
	XLFD_ENCODING,		/* CHARSET_REGISTRY, CHARSET_ENCODING */
	XLFD_FIELD_MAX
    } XLFD_FIELD;

static char *BuildXLFD(FontData *f)
    {
	int i, n;
	char *field[(int)XLFD_FIELD_MAX];
	int length[(int)XLFD_FIELD_MAX];

	char *xlfd, *s;
	
	field[(int)XLFD_FOUNDRY] = "adobe";
	field[(int)XLFD_FAMILY_NAME] = f->family_name;
	field[(int)XLFD_WEIGHT_NAME] = f->weight;
	/*
	** A quick hack! Change 'Roman' into 'medium' (Xew uses weight name
	** for font matching... should probably fix Xew instead, but not very
	** easy as AFM does not seem to contain anything "absolute" that one
	** could compute the WEIGHT property from... -- msa
	*/
	if (strcmp("Roman", f->weight) == 0)
		field[(int)XLFD_WEIGHT_NAME] = "medium";
	/* */

	if (fabs(f->italic_angle) < 0.1)
		field[(int)XLFD_SLANT] = "r";
	else if (f->italic_angle > 0)
		field[(int)XLFD_SLANT] = "ri";
	else
		field[(int)XLFD_SLANT] = "i";
	field[(int)XLFD_SETWIDTH_NAME] = "normal";
	field[(int)XLFD_ADD_STYLE_NAME] = "";
	field[(int)XLFD_PIXEL_SIZE] = "0";	/* all fonts scalable! */
	field[(int)XLFD_POINT_SIZE] = "0";	/* all fonts scalable! */
	field[(int)XLFD_RESOLUTION_X] = "0";	/* all fonts scalable! */
	field[(int)XLFD_RESOLUTION_Y] = "0";	/* all fonts scalable! */
	if (f->is_fixed_pitch)
		field[(int)XLFD_SPACING] = "c";
	else
		field[(int)XLFD_SPACING] = "p";
	field[(int)XLFD_AVERAGE_WIDTH] = "0";	/* all fonts scalable! */
	if (f->iso_8859_1)
		field[(int)XLFD_ENCODING] = "iso8859-1";
	else
		field[(int)XLFD_ENCODING] = "adobe-fontspecific";

	/*
	** Contstruct XLFD from the fields
	*/
	for (i = n = 0; i < XLFD_FIELD_MAX; ++i)
		n += (length[i] = strlen(field[i]));
	xlfd = (char *)malloc(n + XLFD_FIELD_MAX + 1);
	if (xlfd)
	    {
		for (i = 0, s = xlfd; i < XLFD_FIELD_MAX; ++i)
		    {
			char *p;

			*s++ = '-';

			for (p = field[i]; *p; ++p)
				*s++ = isupper(*p) ? tolower(*p) : *p;
		    }
		*s = 0;
	    }
	return xlfd;
    }
/*
** ParseMapLine
*/
static char *ParseMapLine(char *s, char **first, char **second)
    {
	char *le = strchr(s, '\n');

	if (!le)
		return NULL;
	*le = 0;
	while (isspace(*s))
		++s;
	*first = s;
	while (s < le && !isspace(*s))
		++s;
	if (s < le)
		*s++ = 0;
	while (s < le && isspace(*s))
		++s;
	*second = s;
	return le + 1;
    }
/*
** BuildFileName
*/
static void BuildFileName(char *path, char *name, char **buffer, int *size)
    {
	int path_N = strlen(path);
	int name_N = strlen(name);
	int N;

	N = path_N + name_N + 2;

	if (N < 200)
		N = 200;
	if (*buffer == NULL)
	    {
		*size = N;
		*buffer = (char *)malloc(N);
	    }
	else if (*size < N)
	    {
		*size = N;
		*buffer = (char *)realloc(*buffer, N);
	    }
	if (!*buffer)
		return;
	if (*name == '/')
		strcpy(*buffer, name);
	else
	    {
		strcpy(*buffer, path);
		while (path_N > 0 && path[path_N-1] == '/')
			--path_N;
		(*buffer)[path_N++] = '/';
		strcpy(*buffer + path_N, name);
	    }
    }

/*
** ReadFontDirectory
**	load (AFM) fonts from one directory
*/
static void ReadFontDirectory(char *path)
    {
	char *fonts_dir_name = "fonts.dir";
	char *fonts_alias_name = "fonts.alias";
	char *name = NULL;
	int size = 0;
	char *buffer;
	long length;
	
	/*
	** Load in fonts defined by the 'fonts.dir' file
	*/
	BuildFileName(path, fonts_dir_name, &name, &size);
	buffer = ReadFile(name, &length);
	if (buffer)
	    {
		char *s = buffer;
		char *font_file;
		char *xlfd_name;
		
		while ((s = ParseMapLine(s, &font_file, &xlfd_name)) != NULL)
		    {
			FontData *f;

			BuildFileName(path, font_file, &name, &size);
			f = ParseAFM(name);
			if (f)
			    {
				if (*xlfd_name)
					f->xlfd = strdup(xlfd_name);
				else
					f->xlfd = BuildXLFD(f);
				if (fonts_num == fonts_size)
				    {
					fonts_size += 500;
					fonts = (FontData **)realloc
						((char *)fonts,
						 fonts_size * sizeof(*fonts));
				    }
				fonts[fonts_num++] = f;
			    }
		    }
		free(buffer);
	    }
	else
	    {
		fprintf(stderr, "No 'fonts.dir' in %s\n", path);
	    }
	/*
	** Add aliases defined by 'fonts.alias' file
	*/
	BuildFileName(path, fonts_alias_name, &name, &size);
	buffer = ReadFile(name, &length);
	if (buffer)
	    {
		char *s = buffer;
		char *alias;
		char *xlfd_name;

		while ((s = ParseMapLine(s, &alias, &xlfd_name)) != NULL)
			if (*alias && *xlfd_name)
			    {
				if (aliases_num == aliases_size)
				    {
					aliases_size += 500;
					aliases = (char **)realloc
						((char *)aliases,
						 2 * aliases_size *
						 sizeof(*aliases));
				    }
				aliases[2*aliases_num] = strdup(alias);
				aliases[2*aliases_num+1] = strdup(xlfd_name);
				aliases_num += 1;
			    }
		free(buffer);
	    }
	if (name)
		free(name);
    }

/*
** ReferenceFont
**	add a reference to the FontData
*/
FontData *ReferenceFont(FontData *f)
    {
	if (f)
		f->references += 1;
	return f;
    }

/*
** FreeFont
**	decrease the reference count of the FontData and if it
**	reaches ZERO, release the resources
*/
void FreeFont(FontData *f)
    {
	XFontStruct *fs;

	if (!f || --f->references > 0)
		return;
	if (f->font_name)
		free(f->font_name);
	if (f->family_name)
		free(f->family_name);
	if (f->weight)
		free(f->weight);
	if (f->encoding_scheme)
		free(f->encoding_scheme);
	if (f->xlfd)
		free(f->xlfd);
	/*
	** Release XFontStruct resources
	*/
	if ((fs = f->font_info) != NULL)
	    {
		if (fs->per_char)
			free((char *)fs->per_char);
		if (fs->properties)
			free((char *)fs->properties);
		free((char *)fs);
	    }
	free((char *)f);
    }

/*
** LoadFonts
*/
static char *default_font_path[] =
    {
	"./fonts",
	NULL,
    };

void LoadFonts(char **paths)
    {
	int i;

	if (paths == NULL || *paths == NULL)
		paths = default_font_path;

	/*
	** Release old font_path
	*/
	if (font_path)
	    {
		for (i = 0; font_path[i]; ++i)
			free(font_path[i]);
		free((char *)font_path);
	    }
	/*
	** Build a new font_path array (for possible GetFontPath call)
	*/
	for (i = 0; paths[i]; ++i);	/* counts paths into i */
	font_path = (char **)malloc((i + 1) * sizeof(*font_path));
	for (i = 0; paths[i]; ++i)
		font_path[i] = strdup(paths[i]);
	font_path[i] = NULL;
	/*
	** Reset or initialize font information
	*/
	if (fonts == NULL)
	    {
		fonts_size = 1000;
		fonts = (FontData **)malloc(fonts_size * sizeof(*fonts));
	    }		
	else for (i = 0; i < fonts_num; ++i)
	    {
		FreeFont(fonts[i]);
		fonts[i] = NULL;
	    }
	fonts_num = 0;
	/*
	** Reset of initialize aliases information
	*/
	if (aliases == NULL)
	    {
		aliases_size = 1000;
		aliases = (char **)malloc(aliases_size * 2 * sizeof(*aliases));
	    }
	else for (i = 0; i < aliases_num; i++)
	    {
		if (aliases[2*i])
			free(aliases[2*i]);
		if (aliases[2*i+1])
			free(aliases[2*i+1]);
	    }
	aliases_num = 0;
	while (*paths)
		ReadFontDirectory(*paths++);
    }

FontData **GetFontsList(int *num)
    {
	*num = fonts_num;
	return fonts;
    }

char **GetFontAliases(int *num)
    {
	*num = aliases_num;
	return aliases;
    }

char **GetFontPath(void)
    {
	return font_path;
    }

int GetFontSize(char *xlfd)
    {
	int i;
	/*
	** Assume the name is XLFD and extract the pixel size from it
	*/
	for (i = 0; i < 7; ++i)
	    {
		xlfd = strchr(xlfd, '-');
		if (!xlfd)
			return 0;
		xlfd += 1;
	    }
	if (isdigit(*xlfd))
		return atoi(xlfd);
	return 0;
    }


/*
**  Compare if a given string (name) matches the given
**  mask (which can contain wild cards: '*' - match any
**  number of chars, '?' - match any single character.
**
**	return	0, if match
**		1, if no match
**
** Iterative matching function, rather than recursive.
** Written by Douglas A Lewis (dalewis@acsu.buffalo.edu)
*/
static int matches(ma, na)
char *ma, *na;
    {
	int	wild = 0;
	unsigned char *m = (unsigned char *)ma, *n = (unsigned char *)na;
	unsigned char *mask = (unsigned char *)ma;
	
	while (1)
	    {
		if (!*m)
		    {
	  		if (!*n)
				return 0;
			while (m > mask && *--m == '?')
				;
			if (*m == '*')
				return 0;
			if (wild) 
			    {
				m = (unsigned char *)ma;
				n = (unsigned char *)++na;
			    }
			else
				return 1;
		    }
		else if (!*n)
		    {
			while(*m == '*')
				m++;
			return (*m != 0);
		    }
		if (*m == '*')
		    {
			while (*m == '*')
				m++;
			wild = 1;
			ma = (char *)m;
			na = (char *)n;
		    }
	        if (*m != *n && *m != '?')
		    {
			if (wild)
			    {
				m = (unsigned char *)ma;
				n = (unsigned char *)++na;
			    }
			else
				return 1;
		    }
		else
		    {
			if (*m)
				m++;
			if (*n)
				n++;
		    }
	    }
    }


int FontMatch(char *pattern, int n, char *name)
    {
	char *tmp, *p;
	int i, j, replace;
	/*
	** Because all *real* fonts are scalable, replace all size
	** related fields with '0', if the pattern is in XLFD format.
	** The fields affected are
	**
	** 	PIXEL_SIZE (7)
	** 	POINT_SIZE (8)
	** 	RESOLUTION_X (9)
	** 	RESOLUTION_Y (10)
	** 	AVERAGE_WIDTH (12)
	*/
	tmp = (char *)malloc(n + 6); /* Will add at most 5 0's */
	
	for (replace = 0, p = tmp, i = j = 0; j < n; ++j)
	    {
		if (pattern[j] == '-')
		    {
			i += 1;
			*p++ = pattern[j];
			if (!isdigit(pattern[j+1]))
				replace = 0; /* ..ugh.. ugly hack..
					      ..if field does not start with
					      digit, copy it as is */
			else if (i==7 || i==8  || i==9 || i==10 || i==12)
			    {
				*p++ = '0';
				replace = 1;
			    }
			else
				replace = 0;
		    }
		else if (!replace)
			*p++ = pattern[j];
	    }
	*p = 0;
	i = matches(tmp, name) == 0;
	free(tmp);
	return i;
    }



char **GetEncoding_ISO8859_1()
    {
	static char *encoding[257]; /* ..one extra for NULL terminator! */

	if (!encoding[0])
	    {
		/* initialize on first call */
		int i;
		
		/* first make everything not defined by default */

		for (i = 0; i < 256; ++i)
			encoding[i] = ".notdef";

		/* ...then substitute names for iso 8859-1 characters */

		for (i = 0; i < ADOBE_8859_1_SIZE; ++i)
			if (adobe_8859_1[i].name &&
			    adobe_8859_1[i].code >= 0 &&
			    adobe_8859_1[i].code < 256)
				encoding[adobe_8859_1[i].code] =
					adobe_8859_1[i].name;
	    }
	return encoding;
    }

#if AFM_MAIN
int main(int argc, char *argv[])
    {
	FontData *f;
	int i, M, N;

	LoadFonts(argv + 1);
	printf("Total fonts loaded: %d\n", fonts_num);
	for (i = 0; i < fonts_num; ++i)
	    {
		if ((f = fonts[i]) != NULL)
		    {
			if (f->font_name)
				printf("Font Name: %s\n", f->font_name);
			if (f->family_name)
				printf("Family Name: %s\n", f->family_name);
			if (f->weight)
				printf("Weight: %s\n", f->weight);
			if (f->encoding_scheme)
				printf("Encoding Scheme: %s\n",
				       f->encoding_scheme);
			if (f->iso_8859_1)
				printf("ISO 8859-1\n");
			else
				printf("Font Specific");
			if (f->xlfd)
				printf("XLFD: %s\n", f->xlfd);
		    }
	    }
    }
#endif
