/*
** ui_tools.h
**	ui_tools module is a collection of Athena / Xew based utilies
**	which are used in building the User Interfaces for the
**	demo/test programs of Xew.
*/
#ifndef UI_TOOLS_H
#define UI_TOOLS_H

typedef enum
    {
	UI_EndOp,	/* End of table marker */
	UI_ToggleOp,	/* Toggle element  */
	UI_RadioOp,	/* Toggle element, link with previous Toggle  */
	UI_PaneOp,	/* A new pane */
	UI_InputOp,	/* Input data into destination value*/
	UI_LoadOp,	/* Load value into resource from src */
	UI_CopyOp,	/* Copy default value to destination */
	UI_LabelOp	/* Label entry */
    } UI_ElementType;

typedef struct UI_ElementDef
    {
	UI_ElementType op;	/* "Operation Code" */
	char *name;		/* Visible name */
	XtArgVal data;		/* Data parameter, depends on "op" */
	Widget widget;		/* Created widget, depends on "op" */
    } UI_ElementDef;

/*
** UI_ActionDef
**	is structure describing an element in UI_PanelDef action table.
**
**	The end of table is indicated by a NULL name field!
*/
typedef struct UI_ActionDef
    {
	char *name;		/* Name label for the action button */
	XtCallbackProc callback;/* Action callback function */
	XtPointer client_data;	/* client_data for the action function */
    } UI_ActionDef;

/*
** The following macroes can be used in building the panel definitions
*/
#define UI_TOGGLE(label, data)	{UI_ToggleOp, (label), (XtArgVal)(data), }
#define UI_RADIO(label, data)	{UI_RadioOp, (label), (XtArgVal)(data), }
#define UI_PANE(label)		{UI_PaneOp, (label),}
#define UI_INPUT(label)		{UI_InputOp, (label), }
#define UI_LOAD(resource)	{UI_LoadOp, (resource), }
#define UI_LABEL(label)		{UI_LabelOp, (label), }
#define UI_END()		{UI_EndOp,}

typedef struct UI_PanelDef
    {
	char *name;			/* Panel name */
	UI_ElementDef *elements;	/* Element defitions */
	UI_ActionDef *actions;		/* Action definitions */
	char *heading;			/* Panel heading */
	Widget target;			/* Optional Target Widget */
	Widget label;			/* Panel Target name label */
	Widget shell;			/* Panel PopupShell */
    } UI_PanelDef;

/*
** UI_MenuItemDef
**	Describe single entry in the menu.
**
**	*** The end of MenuDef table is singnalled by NULL name field ***
*/
typedef struct UI_MenuItemDef
    {
	char *name;			/* Menu text label */
	XtCallbackProc callback;	/* Menu select callback */
	XtPointer client_data;		/* Menu select client_data */
	Widget entry;
    } UI_MenuItemDef;
/*
** UI_MenuDef
**	Describe menu
*/
typedef struct UI_MenuDef
    {
	char *name;
	UI_MenuItemDef *items;
	Widget button;
	Widget menu;
    } UI_MenuDef;

/*
** UI_Quit
**	Callback procedure to terminate application unconditionally.
*/
#if NeedFunctionPrototypes
void UI_Quit(Widget, XtPointer, XtPointer);
#else
void UI_Quit();
#endif
/*
** UI_Popdown
**	Callback procedure to pop down (not destroy) a popup window
**	associated with the widget causing the callback.
*/
#if NeedFunctionPrototypes
void UI_Popdown(Widget, XtPointer, XtPointer);
#else
void UI_Popdown();
#endif
/*
** UI_AcceptPanel
**	Callback procedure to accept and execute the current definitions
**	of the panel. The client_data must point to a UI_PanelDef structure.
**	In this structure, the field 'widget' identifies the widget to which
**	all possible resource settings will apply.
*/
#if NeedFunctionPrototypes
void UI_AcceptPanel(Widget, XtPointer, XtPointer);
#else
void UI_AcceptPanel();
#endif

/*
** UI_SetToggles
**	This functions goes through panel and attempts to match the
**	initial values of the toggle/radio groups with the actual
**	resource settings in effect.
**
**	THis function is a kludge, works only if toggle/radio and load
**	operations follow a specific pattern used by this application.
*/
#if NeedFunctionPrototypes
void UI_SetToggles(UI_PanelDef *);
#else
void UI_SetToggles();
#endif
/*
** UI_GetStrings
**	Extract the contents of the input fields in a panel into given
**	char * array and return the number of strings actually returned.
**
**	The space for the strings is allocated by XtMalloc and application
**	must release them by XtFree when no more needed.
*/
#if NeedFunctionPrototypes
int UI_GetStrings(UI_PanelDef *, char **, int);
#else
int UI_GetStrings();
#endif
/*
** UI_PopupPanel
**	Callback procedure to popup a panel as described by UI_PanelDef
**	structure. The pointer to UI_PanelDef structure must be supplied
**	as client_data.
*/
#if NeedFunctionPrototypes
void UI_PopupPanel(Widget, XtPointer, XtPointer);
#else
void UI_PopupPanel();
#endif

/*
** UI_WarningMessage
**	This function can be used to process error and warning messages.
**	It will popup a panel window (as defined) containing the error
**	message. If the panel already exists, the same will be re-used
**	with the new message.
*/
#if NeedFunctionPrototypes
void UI_WarningMessage
	(Widget, UI_PanelDef *,
	 String, String, String, String, String *, Cardinal *);
#else
void UI_WarningMessage();
#endif

#if NeedFunctionPrototypes
void UI_PopupCentered(XEvent *, Widget, XtGrabKind);
#else
void UI_PopupCentered();
#endif
/*
** UI_CreateMenu
**	Create a menu under the specified parent.
*/
#if NeedFunctionPrototypes
void UI_CreateMenu(Widget, UI_MenuDef *);
#else
void UI_CreateMenu();
#endif

/*
** UI_Initialize
**	Currently this is only needed to register some type converters
*/
#if NeedFunctionPrototypes
void UI_Initialize(XtAppContext);
#else
void UI_Initialize();
#endif
#endif

