#include <stdio.h>
#include "message.h"
#ifndef False
#define False 0
#endif
#ifndef True
#define True 1
#endif


/* 
Function NewMessage() 
Params: none.
Actions:
	allocates memory for a struct _Message, 
  	allocates memory for a struct _MsgBody field,
	sets the struct _MsgBody field to be the 'root' struct _MsgBody, 
	initialises the rest of the fields of the struct _MsgBody field, 
	sets all other fields to default (=NULL) values,
Returns: a pointer to the allocated struct _Message or (Message)NULL on error.
*/
Message NewMessage(void)
{
	Message result;

	if(!(result=(Message)malloc(MessageSize))) {
		fprintf(stderr, "Oops! Out of memory in NewMessage()\n");
		return((Message)NULL);
	}
	
	/* malloc() a MsgBody... */
	if(!(result->body=(MsgBody)malloc(MsgBodySize))) {
			fprintf(stderr, "Ooops! Out of memory in NewMessage()\n");
			free(result);
			return((Message)NULL);
	}

	/* this field has to be set here, not in InitMsgBody */
	result->body->parent=(MsgBody)NULL;		/* i.e. it's the root MsgBody */

	/* initialise the InitMsgBody... */
	if(!(InitMsgBody(result->body))) {
			fprintf(stderr, "Ooops! Out of memory in NewMessage()\n");
			free(result->body);
			free(result);
			return((Message)NULL);
	}

	/* set up the defaults... */
	result->name=(char *)NULL;
	result->folder=(char *)NULL;
	result->state=Msg_No_State;
	result->tag=(MsgTag)NULL;

	return(result);
}




/*
Function InitMsgBody() 
Params: a pointer to an allocated struct _MsgBody with a valid "parent" field already set.
Actions:
	allocates memory for a struct _MsgHeader field,
	initialises the struct _MsgHeader field,
	sets up default values for some members of the structure,
		[Note: these default values can depend on the parent MsgBody(if any)]
	sets up default values (NULL) for the rest of the members of the structure.
Returns: True if successful, False otherwise.
*/
int InitMsgBody(MsgBody body)
{
	/* allocate the struct _MsgHeader */
	if(!(body->header=(MsgHeader)malloc(MsgHeaderSize))) 
		return(False);

	/* initialise the struct _MsgHeader */
	InitMsgHeader(body->header);
	/* setup defaults-- dependent on parent */

	/* if body->parent is NULL, then this is the root MsgBody */
	if(body->parent&&(body->parent->subtype==ST_digest)) {
		body->type=CT_message;
		body->transfer_type=CT_message;
		body->subtype=ST_rfc822;
		body->transfer_subtype=ST_rfc822;
	} else {
		body->type=CT_text;
		body->transfer_type=CT_text;
		body->subtype=ST_plain;
		body->transfer_subtype=ST_plain;
	}

	/* setup defaults-- independent of parent */
	body->encoding=ENC_sevenBit;
	body->charset=CS_us_ascii;

	/* setup 'NULL' defaults */
	body->next=(MsgBody)NULL;
	body->child=(MsgBody)NULL;
	body->id=(BodyPartID)NULL;
	body->description=(char *)NULL;
	body->params=(char **)NULL;
	body->content=(BodyContent)NULL;

	return(True);
}



/*
Function InitMsgHeader() 
Params: a pointer to an allocated struct _MsgHeader.
Actions:
	sets up default values (NULL) for each member of the structure.
Returns: nothing 
*/
void InitMsgHeader(MsgHeader header)
{
	header->date=(char *)NULL;
	header->id=(MsgID)NULL;
	header->from=(Originator)NULL;
	header->sender=(Originator)NULL;
	header->to=(Recipient *)NULL;
	header->subject=(char *)NULL;
	header->comments=(char *)NULL;
	header->keywords=(char *)NULL;
	header->cc=(Recipient *)NULL;
	header->bcc=(Recipient *)NULL;
	header->in_reply_to=(MsgID)NULL;
	header->obsoletes=(MsgID *)NULL;
	header->related_to=(MsgID *)NULL;
	header->reply_to=(Recipient *)NULL;
	header->expires_at=(struct tm *)NULL;
	header->reply_by=(struct tm *)NULL;
	header->importance=IMP_normal;
	header->sensitivity=SENS_private;
	header->unknownfields=(char **)NULL;

	return;
}


/*********************************************************************
	Start of the "Free" functions... deallocate memory recursively
*********************************************************************/

#define Free(A)	if(A) free(A)

void FreeMessage(Message msg)
{
	if(msg) {
		Free(msg->name);
		Free(msg->folder);
		Free(msg->tag);
		FreeMsgBody(msg->body);
		free(msg);
	}
	return;
}

void FreeMsgBody(MsgBody body)
{
	if(body) {

		/* first go down to bottom of tree... */
		if(body->child)
			FreeMsgBody(body->child);

		/* ...then go across to end of list */
		if(body->next)
			FreeMsgBody(body->next);

		FreeMsgHeader(body->header);
		FreeBodyPartID(body->id);
		Free(body->description);
		FreeStrList(body->params);
		FreeBodyContent(body->content);

		free(body);
	}
	return;
}

void FreeMsgHeader(MsgHeader header)
{
	if(header) {
		Free(header->date);
		FreeMsgID(header->id);
		FreeORName(header->from);
		FreeORName(header->sender);
		FreeORNameList(header->to);
		Free(header->subject);
		Free(header->comments);
		Free(header->keywords);
		FreeORNameList(header->cc);
		FreeORNameList(header->bcc);
		FreeMsgID(header->in_reply_to);
		FreeMsgIDList(header->obsoletes);
		FreeMsgIDList(header->related_to);
		FreeORNameList(header->reply_to);
		Free(header->expires_at);
		Free(header->reply_by);
		FreeStrList(header->unknownfields);
		free(header);
	}
	return;
}

void FreeBodyContent(BodyContent content)
{
	if(content) {
		Free(content->filename);
		free(content);
	}
	return;
}	

void FreeORNameList(ORName *list)
{
	FreeStrList(list);

	return;
}

void FreeORName(ORName name)
{
	Free(name);

	return;
}

void FreeMsgIDList(MsgID *list)
{
	FreeStrList(list);
	
	return;
}

void FreeMsgID(MsgID id)
{
	Free(id);

	return;
}

void FreeBodyPartID(BodyPartID id)
{
	Free(id);

	return;
}

void FreeStrList(char **list)
{
	char **ptr= list;

	if(list) {
		do Free(*ptr); while(*++ptr);
		free(list);
	}

	return;
}
