#ifndef Mfm_h
#define Mfm_h
/*
** Mfm.h - Multimedia File Management
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

/*
** ERRORS
**
**	MfmError enumeration describes error returns specific to the
**	Mfm. These values are all negative, positive non-zero returns
**	are system errors and refer to the errno.
*/
typedef enum
    {
	MfmError_SUCCESS = 0,	/* No Error */
	MfmError_ERROR = -1,	/* Unspecified generic error */
	MfmError_MEMORY = -2,	/* Memory allocation error */
	MfmError_NOPATH = -3,	/* No name/path given */
	MfmError_LONGNAME = -4	/* Message name/path is too long */
    } MfmError;

/*
**	Get the content type associated with the name (string).
**
**	Returns a pointer to a NUL terminated string that corresponds
**	the content type associated with the specified name. If no
**	content type could be determined either from .MFM or by the
**	global rules, a NULL is returned. The name parameter does not
**	have to be an existing file. The returned value is valid until
**	next call to any Mfm function.
**
** SPECIFICATION
**
**	Multimedia File Management is very simple library that maintains
**	"name" <-> "type" relation. "name" is usually assumed to be a
**	file name, but it does not need to be currently existing file.
**	Each directory can contain local association file (.MFM) for the
**	files that would not get resolved by the global rules file or
**	which would get resolved wrong.
**
**	1. If the .MFM exists on the same directory level as the file
**	name in question, then search for the type from that file.
**
**	2. Search the type from global configuration file (global rules).
**	This file has the same format as .MFM.
**
**	3. If neither of the above steps gave any matches, the file name
**	is untyped (a NULL is returned).
**
**	The general philopsophy is that global rules files are maintained
**	by system administrator, containing all well known rules for
**	guessing the filetype from the name (for example "*.jpg" ->
**	"image/jpeg", "*.mpg" -> "video/mpeg", etc.).
**
** GLOBAL RULES
**
**	When any of the MFM functions is used the first time within
**	program, MFM attempts to load the global rules file:
**
** 1) If environment variable MFM_RULES exists,
**    then the value of that variable is used as the filename of
**    the global rules. Note that existence of this environment
**    variable completely overrides the 'global_files'.
**
** 2) If environment variable is not defined, then the first file
**    in global_files ("/etc/mfm", "/usr/local/etc/mfm") that exists,
**    will be used.
**
** 3) If no global rules is found or some error occurs when reading
**    the selected file, then no global rules are used.
*/
char *MfmGetContentType
	(char *name	/* Name string for which the type is retrieved */
	 );
/*
**	Set/Modify/Remove the content type string associated with the name.
**	If type is non-NULL, then this will replace/set the type definition
**	of the specified file. If the type is NULL, then any existing local
**	type definition for the name is removed. The search for name is done
**	without wildcard processing. This function will unconditionally
**	modify the local .MFM (see MfmSetContentType for conditional change).
**
**	Returns zero on success, and non-zero (see MfmError) otherwise.
*/
int MfmPutContentType
	(char *name,	/* Name string to be modified */
	 char *type	/* Type string or NULL */
	 );
/*
**	Set/Modify the content type string associated with the name only
**	if global rules doen't define the content type for this name. If
**	the global rules don't match the name, then this is the same as
**	MfmPutContentType. If global rules do apply to this type, the
**	function will remove possible local override from .MFM.
*/
int MfmSetContentType
	(char *name,	/* Name string to be modified */
	 char *type	/* Type string */
	 );

typedef struct MfmDirectoryRec *MfmDirectory;

typedef int (*MfmFilter)(char *, void *);

MfmDirectory MfmOpenDirectory(char *, MfmFilter, void *);
char *MfmNextFile(MfmDirectory);
void MfmCloseDirectory(MfmDirectory);

#endif
