/*
** Mam.c
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

#include "common.h"
#include "Mfm.h"
#include "Mam.h"
#include "MamCommon.h"
#include "MamCoding.h"

#undef tables_h		/* Force re-include */
#define ENUM_TABLE_GENERATION
#include "tables.h"

int MamMakeMessage(char *name)
    {
	FILE *fp;
	int result;

	if ((result = setjmp(mam_error_exit)) != 0)
		return result;
	if ((fp = MamMakeHeader(name)) == NULL)
		return MamError_NOFILE;
	fclose(fp);
	return 0;
    }
/*
** MamGetRawContent
**	Return raw undecoded content of the body part
*/
char *MamGetRawContent(Message msg, MsgBody body, long *length)
    {
	char *name;
	long tmp, offset = 0;

	*length = 0;
	if (msg == NULL || body == NULL || body->content == NULL)
		return NULL;
	if ((name = body->content->filename) == NULL ||
	    body->content->length != -1)
	    {
		if (MamHeaderInfo(msg->name,&name,&offset,&tmp,(time_t *)NULL))
			return NULL;	/* No File to read!! */
	    }
	else if ((name = MamContentFile(name, msg)) == NULL)
		return NULL;
	if ((tmp = body->content->length) < 0)
	    {
		struct stat info;
		if (stat(name, &info))
			return NULL;
		tmp = info.st_size;
	    }
	return MamReadSection
		(name, offset + body->content->start_offset, tmp, length);
    }

/*
** MamGetContent
*/
char *MamGetContent(Message msg, MsgBody body, long *length)
    {
	unsigned char *buffer;

	buffer = (unsigned char *)MamGetRawContent(msg, body, length);
	if (buffer)
	    {
		if (body->encoding == ENC_base64)
			*length = Mam_from64(buffer, buffer + *length);
		else if (body->encoding == ENC_quotedPrintable)
			*length = Mam_fromqp(buffer, buffer + *length);
	    }
	return (char *)buffer;
    }

static int MamCopyMove(char *old, char *new, char *cmd)
    {
	struct stat info;
	int new_is_dir;
	char buf[MAM_PATH_SIZE*2];

	if (old == NULL || new == NULL)
		return MamError_NOPATH;
	if (stat(old, &info))
		return errno ? errno : MamError_ERRNO;
	if (stat(new, &info) == 0)
	    {
		/*
		** The new file exists, move/copy will fail if this is
		** of type eurobridge/message or if it is not a directory.
		*/
		char *type;
		
		if ((info.st_mode & S_IFMT) != S_IFDIR)
			return MamError_EXISTS;
		type = MfmGetContentType(new);
		if (type && strcmp(type, string_EBFileType[EB_message]) == 0)
			return MamError_IS_MSG;
		new_is_dir = 1;
	    }
	else
		new_is_dir = 0;
	if (sizeof(buf) <= strlen(cmd) + strlen(old) + strlen(new))
		return MamError_LONGNAME;
	sprintf(buf, cmd, old, new);
	if (system(buf))
		return MamError_ERRNO;
	strcpy(buf, new);
	if (new_is_dir)
	    {
		if ((new = strrchr(old, '/')) == NULL)
		    {
			strcat(buf, "/");
			new = old;
		    }
		strcat(buf, new);
	    }			
	MfmSetContentType(buf, MfmGetContentType(old));
	return MamError_SUCCES;
    }

int MamMoveMessage(char *old, char *new)
    {
	int result;

	result = MamCopyMove(old, new, "mv %s %s");
	if (result == 0)
		MfmPutContentType(old, (char *)NULL);
	return result;
    }

int MamCopyMessage(char *old, char *new)
    {
	return MamCopyMove(old, new, "cp -r %s %s");
    }


int MamMoveFolder(char *old, char *new)
    {
	int result;
	
	result = MamCopyMove(old, new, "mv %s %s");
	if (result == 0)
		MfmPutContentType(old, (char *)NULL);
	return result;
    }


int MamDeleteMessage(char *name)
    {
	char *type = MfmGetContentType(name);
	char buf[MAM_PATH_SIZE];

	if (type == NULL || strcmp(type, string_EBFileType[EB_message]) != 0)
		return MamError_NOT_MSG;
	if (sizeof(buf) <= 7+strlen(name))
		return MamError_LONGNAME;
	sprintf(buf, "rm -rf %s", name);
	if (system(buf))
		return MamError_ERRNO;
	MfmPutContentType(name, (char *)NULL);
	return MamError_SUCCES;
    }

/*
** MamContentFile
*/
char *MamContentFile(char *name, Message msg)
    {
	static char tmp[MAM_PATH_SIZE];
	int room = sizeof(tmp);
	char *user, *home;

	tmp[0] = '\0';
	if (name == NULL)
		return NULL;
	else if (*name == '~')
	    {
		/* User relative addressing */
		if (*++name == '\0' || *name == '/')
		    {
			if ((home = getenv("HOME")) == NULL)
				return NULL;
		    }
		else
		    {
			struct passwd *pw;

			user = strchr(name, '/');
			if (user == NULL)
			    {
				user = name;
				name = "";
			    }
			else
			    {
				if (name-user > sizeof(tmp))
					return NULL;
				memcpy(tmp, name, name-user);
				tmp[name-user] = 0;
				name = user;
				user = tmp;
			    }
			if ((pw = getpwnam(user)) == NULL)
				return NULL;
			home = pw->pw_dir;
		    }
		room -= strlen(home) + 1;
		if (room <= 0)
			return NULL;
		strcpy(tmp, home);
		strcat(tmp, "/");
	    }
	else if (*name != '/' && msg != NULL)
	    {
		/* Message relative addressing */
		long off, len;
		char *hdr;

		if (MamHeaderInfo(msg->name, &hdr, &off, &len, (time_t *)NULL))
			return NULL;
		if ((home = strrchr(hdr, '/')) != NULL)
		    {
			len = (home - hdr) + 1;
			room -= len;
			if (room <= 0)
				return NULL;
			memcpy(tmp, hdr, len);
			tmp[len] = '\0';
		    }
	    }
	room -= strlen(name);
	if (room <= 0)
		return NULL;
	strcat(tmp, name);
	return tmp;
    }

typedef struct MamFolderRec
    {
	MamFilter filter;
	void *data;
	MfmDirectory dir;	/* NON-NULL, if scanning directory */
	FILE *fp;		/* Folder file (if not scanning directory) */
	char *end;		/* Message *body* end id string */
	char *begin;		/* Message *body* begin id string (searched
				** after 'end' id string has been located)
				*/
	char *s;		/* Start of buffer segment */
	int count;		/* Number of bytes in the buffer segment */
	long offset;		/* File position of the buffer segment */
	/*
	** A "special struct hack", folder field must be the last! The
	** the use of this dynamic area is different depending on whether
	** the folder is a file or directory.
	**
	** When the mail folder is a directory, the allocation folder[N]
	** here is the max allocation allowed for the name component of the
	** file name. This size is extended with the length of the path.
	** The path is stored into the first 'length' bytes of folder[N] and
	** the complete path for each file in directory is formed by appending
	** the individual filenames to the foldername.
	**
	** When the folder is a directory 'length' bytes from the beginning
	** of the folder[N] is reserved for a string representing the
	** offset and lenght of one mail item in the folder. The remaining
	** allocation is a fixed size (BUFFER_SIZE) area reserved for reading
	** the folder content while scanning it.
	*/
	char *buf;		/* Start of the buffer area in folder[N] */
	int size;		/* Available buffer space in folder[N] */
	char folder[256];
    } MamFolderRec;
/*
** The format of the offset/length string used in identifying a message
** within a file folder. Be sure the OFFSET_ID_LENGTH covers the longest
** possible string generated by the 'offset_format'!
*/
static char offset_format[] = " %.9ld %ld";
#define OFFSET_ID_LENGTH 30
#define BUFFER_SIZE 64000

static int SelectMessages(char *name, void *data)
    {
	char *type;
	MamFolder fdr = (MamFolder)data;

	strncpy(fdr->buf, name, fdr->size);
	fdr->buf[fdr->size-1] = 0;
	type = MfmGetContentType(fdr->folder);
	return type != NULL && strcmp(string_EBFileType[EB_message],type) == 0;
    }

static int AdvanceBuffer(MamFolder fdr, int amount, int request)
    {
	int n;
	char *p;

	for (;;)
	    {
		n = fdr->count > amount ? amount : fdr->count;
		fdr->count -= n;
		fdr->offset += n;
		fdr->s += n;
		amount -= n;
		if (amount == 0)
			break;
		fdr->s = fdr->buf;
		fdr->count = fread(fdr->s, 1, fdr->size, fdr->fp);
		if (fdr->count <= 0)
			return 0;
	    }
	if (fdr->count >= request)
		return 1;
	p = fdr->s;
	fdr->s = fdr->buf;
	if (p != fdr->s)
		memcpy(fdr->s, p, fdr->count);
	p = fdr->s + fdr->count;
	n = fread(p, 1, fdr->size - fdr->count, fdr->fp);
	if (n <= 0)
		return 0;
	fdr->count += n;
	return fdr->count >= request;
    }

static int LocateSkipString(MamFolder fdr, char *key)
    {
	int n = key ? strlen(key) : 0;
	char *p;

	if (n == 0)
		return 0;
	for (;;)
	    {
		while (!(p = memchr(fdr->s, *key, fdr->count)))
			if (!AdvanceBuffer(fdr, fdr->count, 1))
				return 0;
		if (!AdvanceBuffer(fdr, p - fdr->s, n))
			return 0;
		if (memcmp(fdr->s, key, n) == 0)
			return AdvanceBuffer(fdr, n, 0);
		if (!AdvanceBuffer(fdr, 1, 0))
			return 0;
	    }
    }

MamFolder MamOpenFolder(char *name, MamFilter filter, void *data)
    {
	struct stat info;
	MamFolder fdr;
	int isdir, size;
	int length = name ? strlen(name) : 0;

	if (length == 0)
	    {
		name = ".";
		length = 1;
	    }
	if (stat(name, &info))
		return NULL;	/* File or directory not accessible! */
	isdir = (info.st_mode & S_IFMT) == S_IFDIR;
	if (isdir)
		size = sizeof(*fdr) + length + 2;
	else
		size = sizeof(*fdr) + BUFFER_SIZE;
	fdr = (MamFolder)malloc(size);
	if (fdr == NULL)
		return NULL;	/* No memory to do the operation */
	fdr->filter = filter;
	fdr->data = data;
	fdr->fp = NULL;
	fdr->dir = NULL;
	if (isdir)
	    {
		/* Folder is a directory of files, pick up messages */

		strcpy(fdr->folder, name);
		if (fdr->folder[length-1] != '/')
			fdr->folder[length++] = '/';
		fdr->buf = &fdr->folder[length];
		fdr->size = sizeof(fdr->folder);
		fdr->dir = MfmOpenDirectory(name,SelectMessages,(void *)fdr);
		if (fdr->dir)
			return fdr;
	    }
	else if ((fdr->fp = fopen(name, "rb")) != NULL)
	    {
		/* Folder is an ordinary file */

		fdr->s = fdr->buf = &fdr->folder[OFFSET_ID_LENGTH];
		fdr->size = sizeof(fdr->folder)-OFFSET_ID_LENGTH+BUFFER_SIZE;
		fdr->count = 0;
		fdr->offset = 0;
		if (AdvanceBuffer(fdr, 0, 5) && !memcmp(fdr->s, "BABYL", 5))
		    {
			/* Emacs RMAIL folder format */

			fdr->end = "\n\037";
			fdr->begin = "\n*** EOOH ***\n";
			if (LocateSkipString(fdr, fdr->end))
				return fdr;
		    }
		else
		    {
			/* Unix(?) folder format */

			fdr->end = "\nFrom ";
			fdr->begin = "\n";
			return fdr;
		    }
		fclose(fdr->fp);
	    }
	/* Failed in opening the folder */
	free((void *)fdr);
	return NULL;
    }

char *MamNextMessage(MamFolder fdr)
    {
	char *name;

	if (fdr == NULL)
		return NULL;
	
	if (fdr->dir)
	    {
		while ((name = MfmNextFile(fdr->dir)) != NULL)
			if (fdr->filter == NULL ||
			    (*fdr->filter)(name, fdr->data))
				return name;
	    }
	else if (fdr->fp)
	    {
		long offset, length;

		if (!LocateSkipString(fdr, fdr->begin))
			return NULL;
		offset = fdr->offset;
		/* Locate end of the message (or end of file) */
		if (LocateSkipString(fdr, fdr->end))
			length = - strlen(fdr->end);
		else
		    {
			AdvanceBuffer(fdr, fdr->count, 0); /* Goto EOF */
			length = 0;
		    }
		length += fdr->offset - offset;
		name = fdr->folder;
		sprintf(name, offset_format, offset, length);
		if (fdr->filter == NULL || (*fdr->filter)(name, fdr->data))
			return name;
	    }
	return NULL;
    }

void MamCloseFolder(MamFolder fdr)
    {
	if (fdr)
	    {
		if (fdr->dir)
			MfmCloseDirectory(fdr->dir);
		if (fdr->fp)
			fclose(fdr->fp);
		free(fdr);
	    }
    }

int MamCreateFolder(char *name)
    {
	if (name == NULL || *name == 0)
		return MamError_NOPATH;
	if (mkdir(name, MAM_DEFAULT_DIRMODE))
		return errno;
	MfmSetContentType(name, string_EBFileType[EB_folder]);
	return 0;
    }

int MamDeleteFolder(char *name)
    {
	if (name == NULL || *name == 0)
		return MamError_NOPATH;
	if (rmdir(name))
		return errno;
	MfmPutContentType(name, (char *)NULL);
	return 0;
    }

int MamSetLocation(char *id, char *loc)
    {
	return -1;
    }

char *MamGetLocation(char *id)
    {
	return NULL;
    }
