/*
 * $Id: pixmap.h,v 1.3 1999/08/25 17:15:27 falk Exp $
 *
 * Utilities to read a Pixmap.  Requires libxpm.
 *
 * Includes Xt resources converter.
 */

#ifndef	PIXMAP_H
#define	PIXMAP_H

/* Overview:
 *
 *  This header file describes some functions which can be used to read
 *  a full-color pixmap file in .xpm format.  It uses the libxpm library,
 *  but has the advantage of path searching and caching.  That is, when
 *  a pixmap is requested by name, and it has already been loaded (and
 *  the display, screen, colormap and depth all match), the previously-
 *  loaded pixmap is returned.
 *
 *  In adition, these routine save the dimensions and depth of the pixmap
 *  and return this information in a PixmapInfo structure.
 *
 *  Also, a PixmapInfo structure may be obtained for pixmaps which were
 *  created by the application, using GetPixmapInfo().
 *
 *  PixmapInfo structures contain a reference count which is incremented
 *  every time GetPixmap() or GetPixmapInfo() is called, and decremented
 *  every time ReleasePixmap() is called.  When the reference count reaches
 *  zero, the PixmapInfo structure *and* the pixmap itself are freed.
 *  This applies even if the pixmap was originally created elsewhere.
 *
 *  Finally, an Xt resources converter is supplied.  Register it with
 *  RegisterCvtStringtoPixmap().  Information about the converted
 *  pixmap is available by calling GetPixmapInfo().  Note that the
 *  PixmapInfo structure should be freed explicitly by the widget
 *  using ReleasePixmap().  This will also free the pixmap when the
 *  reference count reaches zero.
 *
 *  cvtStringToPixmap() searches for the pixmap in the paths supplied with
 *  both the pixDir and bitmapFilePath resources.  (The first for XawXpm
 *  compatibility, the second for Xmu compatibility.)
 *
 *
 *
 * Functions:
 *
 *  GetPixmap() accepts as inputs display, colormap, window, depth,
 *  search path and pixmap name.  It returns a pixmap info structure (below).
 *  This pointer remains valid until the pixmap is released.
 *  Pixmaps are shared among windows whenever possible (display, colormap,
 *  depth and name match).  If the pixmap is found in the cache, this
 *  function is quite fast, otherwise, the pixmap is created on the
 *  server.
 *
 *  GetPixmapInfo() accepts a display and existing pixmap as inputs, and
 *  returns a pixmap info structure.  If this pixmap was allocated by
 *  GetPixmap() or cvtStringToPixmap() or previously matched by
 *  GetPixmapInfo(), this function is quite fast.  Otherwise, a
 *  round trip to the server is involved.
 *
 *  ReleasePixmap() releases the specified pixmap.  Resources are freed
 *  after the last release.
 *
 *  RegisterCvtStringToPixmap() registers a resource converter to convert
 *  String => Pixmap.
 */

typedef	struct pm_info {
	  unsigned int	width, height, depth ;	/* dimensions of pixmap */
	  Pixmap	pixmap ;		/* the pixmap */
	  Pixmap	mask ;			/* bitmap mask */
	  Display	*dpy ;
	  Colormap	cmap ;
	  char		*name ;
	  struct pm_info *next ;
	  int		refcnt ;
	} PixmapInfo ;


#ifdef	__STDC__

extern	PixmapInfo	*GetPixmap(Display *, Colormap, Window,
				unsigned int depth, char *path, char *name);
extern	PixmapInfo	*GetPixmapInfo(Display *, Pixmap) ;
extern	PixmapInfo	*RealizePixmap(Widget, PixmapInfo *) ;
extern	void		ReleasePixmap(PixmapInfo *) ;
extern	void		RegisterCvtStringToPixmap(void) ;

#else

extern	PixmapInfo	*GetPixmap() ;		/* name => PixmapInfo */
extern	PixmapInfo	*GetPixmapInfo()  ;	/* Pixmap => PixmapInfo */
extern	PixmapInfo	*RealizePixmap()  ;
extern	void		ReleasePixmap()  ;
extern	void		RegisterCvtStringtoPixmap()  ;

#endif


#ifndef	XtNpixDir
#define	XtNpixDir	"pixDir"
#define	XtCPixDir	"PixDir"
#endif

#ifndef	XtNbitmapFilePath
#define	XtNbitmapFilePath	"bitmapFilePath"
#define	XtCBitmapFilePath	"BitmapFilePath"
#endif


#endif
