/*
# X-BASED ABACUS
#
#  Abacus.c
#
###
#
#  Copyright (c) 1994 - 95	David Albert Bagley, bagleyd@source.asset.com
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Abacus */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "AbacusP.h"

static void InitializeAbacus();
static void ExposeAbacus();
static void ResizeAbacus();
static void DestroyAbacus();
static Boolean SetValuesAbacus();
static void quit_abacus();
static void select_abacus();
static void clear_abacus();
static void move_abacus();
static void check_beads();
static void reset_beads();
static void clear_all_beads();
static void move_beads();
static void move_beads_up();
static void move_beads_down();
static void draw_frame();
static void draw_all_beads();
static void draw_bead();
static void add_bead();
static void sub_bead();
static int counter_empty();
static void set_counter();

static char defaultTranslationsAbacus[] =
  "<KeyPress>q: quit()\n\
   Ctrl<KeyPress>C: quit()\n\
   <Btn1Down>: select()\n\
   <Btn1Motion>: select()\n\
   <KeyPress>c: clear()\n\
   <Btn3Down>: clear()";

static XtActionsRec actionsListAbacus[] =
{
  {"quit", (XtActionProc) quit_abacus},
  {"select", (XtActionProc) select_abacus},
  {"clear", (XtActionProc) clear_abacus}
};

static XtResource resourcesAbacus[] =
{
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
   XtOffset(AbacusWidget, core.width), XtRString, "400"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
   XtOffset(AbacusWidget, core.height), XtRString, "200"},
  {XtNbars, XtCBars, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.bars), XtRString, "12"},
  {XtNbase, XtCBase, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.base), XtRString, "10"},
  {XtNspaces, XtCSpaces, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.spaces), XtRString, "2"},
  {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
   XtOffset(AbacusWidget, abacus.foreground), XtRString, XtDefaultForeground},
  {XtNbeadColor, XtCForeground, XtRPixel, sizeof(Pixel),
   XtOffset(AbacusWidget, abacus.bead), XtRString, XtDefaultForeground},
  {XtNtopNumber, XtCTopNumber, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.levels[TOP].number), XtRString, "2"},
  {XtNbottomNumber, XtCBottomNumber, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.levels[BOTTOM].number), XtRString, "5"},
  {XtNtopFactor, XtCTopFactor, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.levels[TOP].factor), XtRString, "5"},
  {XtNbottomFactor, XtCBottomFactor, XtRInt, sizeof(int),
   XtOffset(AbacusWidget, abacus.levels[BOTTOM].factor), XtRString, "1"},
  {XtNtopOrient, XtCTopOrient, XtRBoolean, sizeof(Boolean),
   XtOffset(AbacusWidget, abacus.levels[TOP].orientation), XtRString, "FALSE"},
  {XtNbottomOrient, XtCBottomOrient, XtRBoolean, sizeof(Boolean),
   XtOffset(AbacusWidget, abacus.levels[BOTTOM].orientation), XtRString, "FALSE"},
  {XtNselectCallback, XtCCallback, XtRCallback, sizeof(caddr_t),
   XtOffset(AbacusWidget, abacus.select), XtRCallback, NULL}
};

AbacusClassRec abacusClassRec =
{
  {
    (WidgetClass) &widgetClassRec,	/* superclass */
    "Abacus",				/* class name */
    sizeof(AbacusRec),			/* widget size */
    NULL,				/* class initialize */
    NULL,				/* class part initialize */
    FALSE,				/* class inited */
    InitializeAbacus,			/* initialize */
    NULL,				/* initialize hook */
    XtInheritRealize,			/* realize */
    actionsListAbacus,			/* actions */
    XtNumber(actionsListAbacus),	/* num actions */
    resourcesAbacus,			/* resources */
    XtNumber(resourcesAbacus),		/* num resources */
    NULLQUARK,				/* xrm class */
    TRUE,				/* compress motion */
    TRUE,				/* compress exposure */
    TRUE,				/* compress enterleave */
    TRUE,				/* visible interest */
    DestroyAbacus,			/* destroy */
    ResizeAbacus,			/* resize */
    ExposeAbacus,			/* expose */
    SetValuesAbacus,			/* set values */
    NULL,				/* set values hook */
    XtInheritSetValuesAlmost,		/* set values almost */
    NULL,				/* get values hook */
    NULL,				/* accept focus */
    XtVersion,				/* version */
    NULL,				/* callback private */
    defaultTranslationsAbacus,		/* tm table */
    NULL,				/* query geometry */
    NULL,				/* display accelerator */
    NULL				/* extension */
  },
  {
    0					/* ignore */
  }
};

WidgetClass abacusWidgetClass = (WidgetClass) &abacusClassRec;

static void InitializeAbacus(request, new)
Widget request, new;
{
  AbacusWidget w = (AbacusWidget) new;
  XGCValues values;
  XtGCMask valueMask;

  check_beads(w);
  reset_beads(w);

  valueMask = GCForeground | GCBackground;
  values.background = w->core.background_pixel;

  values.foreground = w->abacus.foreground;
  w->abacus.abacusGC = XtGetGC(new, valueMask, &values);

  values.foreground = w->abacus.bead;
  w->abacus.beadGC = XtGetGC(new, valueMask, &values);

  values.foreground = w->core.background_pixel;
  w->abacus.eraseGC = XtGetGC(new, valueMask, &values);

  ResizeAbacus(w);
}

static void DestroyAbacus(old)
Widget old;
{
  AbacusWidget w = (AbacusWidget) old;

  XtReleaseGC(old, w->abacus.beadGC);
  XtReleaseGC(old, w->abacus.abacusGC);
  XtReleaseGC(old, w->abacus.eraseGC);
  XtRemoveCallbacks(old, XtNselectCallback, w->abacus.select);
}

static void ResizeAbacus(w)
AbacusWidget w;
{
  int height;

  w->abacus.delta.x = 8;
  w->abacus.delta.y = 2;
  w->abacus.wid = MAX(((int) w->core.width - w->abacus.delta.x) /
    w->abacus.bars, 0);
  w->abacus.ht = MAX(((int) w->core.height - 2 * w->abacus.delta.y - 2) /
    (w->abacus.levels[TOP].room + w->abacus.levels[BOTTOM].room), 0);
  w->abacus.width = w->abacus.wid * w->abacus.bars +
    w->abacus.delta.x + 2;
  w->abacus.levels[TOP].height = w->abacus.ht * 
    w->abacus.levels[TOP].room + w->abacus.delta.y + 2;
  w->abacus.levels[BOTTOM].height = w->abacus.ht * 
    w->abacus.levels[BOTTOM].room + w->abacus.delta.y + 2;
  height = w->abacus.levels[TOP].height + w->abacus.levels[BOTTOM].height;
  w->abacus.offset.x = ((int) w->core.width - w->abacus.width + 2) / 2;
  w->abacus.offset.y = ((int) w->core.height - height + 2) / 2;
  w->abacus.bead_width = MAX(w->abacus.wid - w->abacus.delta.x, 0);
  w->abacus.bead_height = MAX(w->abacus.ht - w->abacus.delta.y , 0);
}

static void ExposeAbacus(new, event, region)
Widget new;
XEvent *event;
Region region; /* Not used */
{
  AbacusWidget w = (AbacusWidget) new;

  if (w->core.visible) {
    draw_frame(w, w->abacus.abacusGC);
    draw_all_beads(w, w->abacus.beadGC);
  }
}

static Boolean SetValuesAbacus(current, request, new)
Widget current, request, new;
{
  AbacusWidget c = (AbacusWidget) current, w = (AbacusWidget) new;
  XGCValues values;
  XtGCMask valueMask;
  Boolean redraw = FALSE;
  Boolean redraw_beads = FALSE;

  check_beads(w);
  if (w->abacus.bead != c->abacus.bead ||
      w->core.background_pixel != c->core.background_pixel) {
    valueMask = GCForeground | GCBackground;
    values.foreground = w->abacus.bead;
    values.background = w->core.background_pixel;
    XtReleaseGC(new, w->abacus.beadGC);
    w->abacus.beadGC = XtGetGC(new, valueMask, &values);

    values.foreground = w->core.background_pixel;
    values.background = w->abacus.bead;
    XtReleaseGC(new, w->abacus.eraseGC);
    w->abacus.eraseGC = XtGetGC(new, valueMask, &values);
    redraw_beads = TRUE;
  }
  if (w->abacus.bars != c->abacus.bars ||
      w->abacus.levels[BOTTOM].orientation !=
        c->abacus.levels[BOTTOM].orientation ||
      w->abacus.levels[TOP].orientation !=
        c->abacus.levels[TOP].orientation ||
      w->abacus.levels[BOTTOM].number !=
        c->abacus.levels[BOTTOM].number ||
      w->abacus.levels[TOP].number != c->abacus.levels[TOP].number ||
      w->abacus.levels[BOTTOM].factor !=
        c->abacus.levels[BOTTOM].factor ||
      w->abacus.levels[TOP].factor != c->abacus.levels[TOP].factor ||
      w->abacus.spaces != c->abacus.spaces ||
      w->abacus.base != c->abacus.base) {
    reset_beads(w);
    redraw = TRUE;
  }
  if (w->abacus.wid != c->abacus.wid || w->abacus.ht != c->abacus.ht)
    redraw_beads = TRUE;
  if (redraw_beads && !redraw && XtIsRealized(new) && new->core.visible)
  {
    draw_frame(c, c->abacus.eraseGC);
    draw_all_beads(c, c->abacus.eraseGC);
    draw_frame(w, w->abacus.abacusGC);
    draw_all_beads(w, w->abacus.beadGC);
  }
  return (redraw);
}

static void quit_abacus(w, event, args, n_args)
AbacusWidget w;
XEvent *event;
char *args[];
int n_args;
{
  XtCloseDisplay(XtDisplay(w));
  exit(0);
}

static void select_abacus(w, event, args, n_args)
AbacusWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_abacus(w, event->xbutton.x - w->abacus.offset.x,
                 event->xbutton.y - w->abacus.offset.y);
}

static void clear_abacus(w, event, args, n_args)
AbacusWidget w;
XEvent *event;
char *args[];
int n_args;
{
  if (counter_empty(w))
    return;
  /* Should check if one really wants to destroy calculations. */
  clear_all_beads(w);
}

static void move_abacus(w, x, y)
AbacusWidget w;
int x, y;
{
  int d, i, j;

  if (y > w->abacus.levels[TOP].height) {
    y = y - w->abacus.levels[TOP].height;
    d = BOTTOM;
  } else
    d = TOP;
  i = w->abacus.bars - 1 - (x - w->abacus.delta.x / 2) / w->abacus.wid;
  j = (y - w->abacus.delta.y / 2) / w->abacus.ht + 1;
  if (i < 0)
    i = 0;
  else if (i >= w->abacus.bars)
    i = w->abacus.bars - 1;
  if (j < 1)
    j = 1;
  else if (j > w->abacus.levels[d].room)
    j = w->abacus.levels[d].room;
  move_beads(w, d, i, j);
}

static void check_beads(w)
AbacusWidget w;
{
  char buf[121];

  if (w->abacus.base != 10) {
    XtWarning("Base in this version must be 10");
    w->abacus.base = 10;
  }
  if (w->abacus.bars < 1 || w->abacus.bars > MAXBARS) {
    (void) sprintf(buf, "Number of Bars out of bounds, use 1..%d", MAXBARS);
    XtWarning(buf);
    w->abacus.bars = 12;
  }
  if (w->abacus.levels[TOP].number < 1 ||
      w->abacus.levels[TOP].number > MAXBEADS) {
    (void) sprintf(buf,
             "Number of Top Beads out of bounds, use 1..%d", MAXBEADS);
    XtWarning(buf);
    w->abacus.levels[TOP].number = 2;
  }
  if (w->abacus.levels[BOTTOM].number < 1 ||
      w->abacus.levels[BOTTOM].number > MAXBEADS) {
    (void) sprintf(buf,
             "Number of Bottom Beads out of bounds, use 1..%d", MAXBEADS);
    XtWarning(buf);
    w->abacus.levels[BOTTOM].number = 5;
  }
  if (w->abacus.spaces < 1 || w->abacus.spaces > MAXBEADS) {
     (void) sprintf(buf, "Number of Spaces out of bounds, use 1..%d", MAXBEADS);
    XtWarning(buf);
    w->abacus.spaces = 2;
  }
  if (w->abacus.levels[TOP].factor < 1 ||
      w->abacus.levels[TOP].factor > w->abacus.base) {
    (void) sprintf(buf, "Factor of Top Beads out of bounds, use 1..%d",
       w->abacus.base);
    XtWarning(buf);
    w->abacus.levels[TOP].factor = 5;
  }
  if (w->abacus.levels[BOTTOM].factor < 1 ||
      w->abacus.levels[BOTTOM].factor > w->abacus.base) {
    (void) sprintf(buf, "Factor of Bottom Beads out of bounds, use 1..%d",
       w->abacus.base);
    XtWarning(buf);
    w->abacus.levels[BOTTOM].factor = 1;
  }
}

static void reset_beads(w)
AbacusWidget w;
{
  int i, d;

  w->abacus.levels[TOP].room =
    w->abacus.levels[TOP].number + w->abacus.spaces;
  w->abacus.levels[BOTTOM].room =
    w->abacus.levels[BOTTOM].number + w->abacus.spaces;
  for (i = 0; i < w->abacus.bars; i++)
    for (d = BOTTOM; d <= TOP; d++)
      w->abacus.levels[d].position[i] =
        (w->abacus.levels[d].orientation) ? w->abacus.levels[d].number : 0;
  for (i = 0; i < MAXDIGITS; i++)
    w->abacus.digits[i] = '0';
}

static void clear_all_beads(w)
AbacusWidget w;
{
  int d, i;

  for (i = 0; i < w->abacus.bars; i++) {
    for (d = DOWN; d <= UP; d++) {
      if (w->abacus.levels[d].orientation)
        move_beads_up(w, d, i, w->abacus.levels[d].room);
      else /* w->abacus.levels[d].orientation == DOWN */
        move_beads_down(w, d, i, 1);
    }
  }
}

static void move_beads(w, d, i, j)
AbacusWidget w;
int d, i, j;
{
  if (j <= w->abacus.levels[d].position[i])
    move_beads_down(w, d, i, j);
  else
    move_beads_up(w, d, i, j);
}

static void move_beads_up(w, d, i, j)
AbacusWidget w;
int d, i, j;
{
  int k;

  if (j > w->abacus.levels[d].position[i] + w->abacus.spaces) {
    for (k = w->abacus.levels[d].position[i] + w->abacus.spaces + 1;
         k <= j; k++) {
      draw_bead(w, w->abacus.eraseGC, d, i, k);
      draw_bead(w, w->abacus.beadGC, d, i, k - w->abacus.spaces);
    }
    if (w->abacus.levels[d].orientation)
      sub_bead(w, w->abacus.levels[d].factor *
        (j - w->abacus.spaces - w->abacus.levels[d].position[i]), i);
    else /* w->abacus.levels[d].orientation == DOWN */
      add_bead(w, w->abacus.levels[d].factor *
        (j - w->abacus.spaces - w->abacus.levels[d].position[i]), i);
    set_counter(w);
    w->abacus.levels[d].position[i] = j - w->abacus.spaces;
  }
}

static void move_beads_down(w, d, i, j)
AbacusWidget w;
int d, i, j;
{
  int k;

  if (j <= w->abacus.levels[d].position[i]) {
    for (k = w->abacus.levels[d].position[i]; k >= j; k--) {
      draw_bead(w, w->abacus.eraseGC, d, i, k);
      draw_bead(w, w->abacus.beadGC, d, i, k + w->abacus.spaces);
    }
    if (w->abacus.levels[d].orientation)
      add_bead(w, w->abacus.levels[d].factor *
        (w->abacus.levels[d].position[i] - j + 1), i);
    else /* w->abacus.levels[d].orientation == DOWN */
      sub_bead(w, w->abacus.levels[d].factor *
        (w->abacus.levels[d].position[i] - j + 1), i);
    set_counter(w);
    w->abacus.levels[d].position[i] = j - 1;
  }
}

static void draw_frame(w, gc)
AbacusWidget w;
GC gc;
{
  int d, i, dx, dy, x, y, y_offset;

  x = w->abacus.bars * w->abacus.wid + w->abacus.delta.x - 1;
  for (d = UP; d >= DOWN; d--) {
    dx = w->abacus.bead_width / 2 + w->abacus.delta.x + w->abacus.offset.x;
    y_offset = (d == UP) ? 0 : w->abacus.levels[TOP].height;
    y = w->abacus.levels[d].room * w->abacus.ht + w->abacus.delta.y - 1;
    dy = w->abacus.delta.y / 2 + y_offset + w->abacus.offset.y;
    XFillRectangle(XtDisplay(w), XtWindow(w), gc,
      w->abacus.offset.x, y_offset + w->abacus.offset.y, x, 1);
    XFillRectangle(XtDisplay(w), XtWindow(w), gc,
      w->abacus.offset.x, y + y_offset + w->abacus.offset.y, x, 1);
    XFillRectangle(XtDisplay(w), XtWindow(w), gc,
      w->abacus.offset.x, y_offset + w->abacus.offset.y, 1, y);
    XFillRectangle(XtDisplay(w), XtWindow(w), gc,
      x + w->abacus.offset.x, y_offset + w->abacus.offset.y, 1, y + 1);
    for (i = 0; i < w->abacus.bars ; i++) {
      XFillRectangle(XtDisplay(w), XtWindow(w), gc, dx, dy, 1, y - 1);
      dx += w->abacus.wid;
    }
  }
}

static void draw_all_beads(w, gc)
AbacusWidget w;
GC gc;
{
  int d, i, j;

  for (i = 0; i < w->abacus.bars; i++) {
    for (d = DOWN; d <= UP; d++) {
      for (j = 1; j <= w->abacus.levels[d].position[i]; j++)
        draw_bead(w, gc, d, i, j);
      for (j = w->abacus.spaces + w->abacus.levels[d].position[i] + 1;
           j <= w->abacus.levels[d].room; j++)
        draw_bead(w, gc, d, i, j);
    }
  }
  set_counter(w);
}

static void XFillCircle(display, window, gc, diameter, ctr_x, ctr_y)
Display *display;
Window window;
GC gc;
int diameter, ctr_x, ctr_y;
{
  XFillArc(display, window, gc, ctr_x - diameter / 2, ctr_y - diameter / 2,
    diameter, diameter, 0, CIRCLE);
}

static void draw_bead(w, gc, d, i, j)
AbacusWidget w;
GC gc;
int d, i, j;
{
  int dx, dy, y_offset;

  y_offset = (d == UP) ? 0 : w->abacus.levels[TOP].height;
  dx = (w->abacus.bars - i - 1) * w->abacus.wid + w->abacus.delta.x +
    w->abacus.offset.x;
  dy = (j - 1) * w->abacus.ht + w->abacus.delta.y + y_offset +
    w->abacus.offset.y;
  if (gc == w->abacus.beadGC) {
    if (w->abacus.bead_width > w->abacus.bead_height) {
      XFillCircle(XtDisplay(w), XtWindow(w), gc, w->abacus.bead_height - 1,
        dx + w->abacus.bead_width / 2 - 
        (w->abacus.bead_width - w->abacus.bead_height) / 2,
        dy + (w->abacus.bead_height - 1) / 2);
      XFillCircle(XtDisplay(w), XtWindow(w), gc, w->abacus.bead_height - 1,
        dx + w->abacus.bead_width / 2  + 
        (w->abacus.bead_width - w->abacus.bead_height) / 2,
        dy + (w->abacus.bead_height - 1) / 2);
      XFillRectangle(XtDisplay(w), XtWindow(w), gc,
        dx + w->abacus.bead_width / 2 - 
        (w->abacus.bead_width - w->abacus.bead_height) / 2, dy,
        w->abacus.bead_width - w->abacus.bead_height, w->abacus.bead_height);
    } else if (w->abacus.bead_width < w->abacus.bead_height) {
      XFillCircle(XtDisplay(w), XtWindow(w), gc, w->abacus.bead_width - 1,
        dx + (w->abacus.bead_width - 1) / 2,
        dy + w->abacus.bead_height / 2 - 
        (w->abacus.bead_height - w->abacus.bead_width) / 2);
      XFillCircle(XtDisplay(w), XtWindow(w), gc, w->abacus.bead_width - 1,
        dx + (w->abacus.bead_width - 1) / 2,
        dy + w->abacus.bead_height / 2 +
        (w->abacus.bead_height - w->abacus.bead_width) / 2);
      XFillRectangle(XtDisplay(w), XtWindow(w), gc,
        dx, dy + w->abacus.bead_height / 2 - 
        (w->abacus.bead_height - w->abacus.bead_width) / 2,
        w->abacus.bead_width, w->abacus.bead_height - w->abacus.bead_width);
    } else
      XFillCircle(XtDisplay(w), XtWindow(w), gc, w->abacus.bead_height - 1,
        dx + (w->abacus.bead_width - 1) / 2,
        dy + (w->abacus.bead_height - 1) / 2);
  } else {
    XFillRectangle(XtDisplay(w), XtWindow(w), gc,
      dx, dy, w->abacus.bead_width + 1, w->abacus.bead_height + 1);
    XFillRectangle(XtDisplay(w), XtWindow(w), w->abacus.abacusGC,
      dx + w->abacus.bead_width / 2, dy, 1, w->abacus.bead_height + 2);
  }
}

static void add_bead(w, d, p)
AbacusWidget w;
int d, p;
{
  int position = MAXDIGITS - 1 - p;
  int digit = w->abacus.digits[position] - '0' + d;


  w->abacus.digits[position] = (digit % w->abacus.base) + '0';
  if (digit >= w->abacus.base)
    add_bead(w, digit / w->abacus.base, p + 1);
}

static void sub_bead(w, d, p)
AbacusWidget w;
int d, p;
{
  int position = MAXDIGITS - 1 - p;
  int digit = w->abacus.digits[position] - '0' - d;

  w->abacus.digits[position] =
    ((digit + w->abacus.base) % w->abacus.base) + '0';
  if (digit < 0)
    sub_bead(w, 1 + (-1 - digit) / w->abacus.base, p + 1);
}

static int counter_empty(w)
AbacusWidget w;
{
  int n = 0;

  while (n != MAXDIGITS - 1 && w->abacus.digits[n] == '0')
    n++;
  return (n == MAXDIGITS - 1 && w->abacus.digits[n] == '0');
}

static void set_counter(w)
AbacusWidget w;
{
  abacusCallbackStruct cb;
  char buf[MAXDIGITS+3];
  int i, n = 0;

  while (n != MAXDIGITS - 1 && w->abacus.digits[n] == '0')
    n++;
  for (i = n; i < MAXDIGITS; i++)
    buf[i - n] = w->abacus.digits[i];
  buf[MAXDIGITS - n] = '\0';
  (void) strcpy(cb.buffer, buf);
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

#ifdef OLDANDINTHEWAY
/* Not used since this restricts the size of abacus to long int. */
static long int power(x, n) /* raise x to the nth power n >= 0 */
int x, n;
{
  int i;
  long int p = 1;

  for (i = 1; i <= n; ++i)
    p = p * x;
  return p;
}

/* This routine will do a XFillArc for a circle */
DiskXI(display, window, gc, diameter, ctr_x, ctr_y)
Display *display;
Window window;
GC gc;
int diameter, ctr_x, ctr_y;
{
  int x, y, p, d;

  x = 0;
  y = diameter / 2;
  p = diameter % 2;
  d = 1 - 2 * y + p;
  while (x < y) {
    DiskPointsXI(display, window, gc, ctr_x, ctr_y, x, y, p);
    if (d < 0)
      d = d + (4 * x) + 6;
    else {
      d = d + (4 * (x - y)) + 10;
      y--;
    }
    x++;
  }
  if (x == y)
    DiskPointsXI(display, window, gc, ctr_x, ctr_y, x, y, p);
} /* DiskXI */

DiskPointsXI(display, window, gc, ctr_x, ctr_y, x, y, p)
Display *display;
Window window;
GC gc;
int ctr_x, ctr_y, x, y, p;
{
  XFillRectangle(display, window, gc,
    ctr_x - x, ctr_y - y, 2 * x + p + 1, 1);
  XFillRectangle(display, window, gc,
    ctr_x - x, ctr_y + y + p, 2 * x + p + 1, 1);
  XFillRectangle(display, window, gc,
    ctr_x - y, ctr_y - x, 2 * y + p + 1, 1);
  XFillRectangle(display, window, gc,
    ctr_x - y, ctr_y + x + p, 2 * y + p + 1, 1);
  /*XDrawLine(display, window, gc, ctr_x - x, ctr_y - y, ctr_x + x + p, ctr_y - y);
  XDrawLine(display, window, gc,
     ctr_x - x, ctr_y + y + p, ctr_x + x + p, ctr_y + y + p);
  XDrawLine(display, window, gc, ctr_x - y, ctr_y - x, ctr_x + y + p, ctr_y - x);
  XDrawLine(display, window, gc,
     ctr_x - y, ctr_y + x + p, ctr_x + y + p, ctr_y + x + p);*/
} /* DiskPointsXI */
#endif  
