/* 3d.h
 *
 * Data structures for use in the 3d world
 */
#ifndef threed_h
#define threed_h

struct point {				/* a point in 3d */
	float x,y,z;
	};

#define FACECOUNT 8
typedef enum{f_face,f_plane,f_line,f_wireframe,f_glass,f_sphere,
	     f_point,f_shaded} facestyle;
/* f_face	normal face, do backface removal
 * f_plane	a plane, don't do backface removal
 * f_line	lines from 1st point to last
 * f_wireframe	like line, but connect last point to 1st
 * f_glass	like plane, but transparent like cellophane
 * f_circle	a sphere with it's centre at point 0 and surface at point 1
 * f_point	a single dot
 * f_shaded	a partially transparent surface
 */

#define TYPECOUNT 17
typedef enum{t_tank,t_fish,t_bird,t_shrapnel,t_static,t_bullet,
	     t_hover,t_scenery,t_fixedwing,t_thing,t_missile,
	     t_tree,t_seedpod,t_gunsite,t_mine,t_weapon,t_munitions
	    }vehicle_type;
/* t_tank	moves along the ground, shooting things
 * t_fish	flys through the air, in flocks
 * t_bird	flys around on it's own
 * t_shrapnel	a blast fragment
 * t_static	a house or tree or something
 * t_bullet	some kind of projectile
 * t_hover	a hovering aircraft such as a helicopter
 * t_scenery	a road or river. Immune from collision
 * t_fixedwing	a winged aircraft, prone to stalling
 * t_thing	a peice of scenery that moves
 * t_missile	a homing missile
 * t_tree	a tree, suitable for farming
 * t_seedpod	a seedpod from an exploded tree
 * t_gunsite	a non-moving automatic cannon
 * t_mine	a non-moving, explodes when hit
 * t_weapon	change weapon prize
 * t_munitions	ammunition prize
 */

#define WEAPONCOUNT 10
typedef enum{w_none,w_tracer,w_shell,w_bullet,w_bomb,w_flame,
	     w_missile,w_chicken,w_plasmaballs,w_torpedo} weapon_type;
/* w_none	  unarmed
 * w_tracer	  high velocity, low power ammuntion
 * w_shell	  high explosive tank ammuntion
 * w_bullet	  fast, relatively harmless ammuntion
 * w_bomb	  powerful, free fall explosives
 * w_flame	  a slow-firing wide area flame thrower
 * w_missile	  a homing missile
 * w_chicken	  armour-piercing anti-vehicular chicken
 * w_plasmaballs  a chain of plasma fireballs
 * w_torpedo	  high explosive homing missile
 */

#define OWNERCOUNT 4
typedef enum{o_player,o_game,o_network,o_none} owner_type;
/* o_player	controlled by the player on this client
 * o_game	controlled by this client
 * o_network	controlled by the server (probably another client)
 * o_none	vehicle is controlled by no-one
 */

struct star {
	float bearing;			/* compass bearing to star (rads) */
	float elevation;		/* angle above horizon (rads) */
	int intensity;			/* brightness, 0 - 31 */
	int radius;			/* size of the star */
	};
/* one star in the sky */

struct oscillate {
	struct point *pt;		/* point to oscillate */
	int pnum;			/* number of point to oscillate */
	struct point init;		/* intial point position */
	struct point ovec;		/* oscillation range */
	float rate;			/* oscillation speed */
	int axes;			/* axes to effect */
	float phase;			/* offset of oscillation */
	double (*func)(double);		/* function to apply to point */
	};
/* oscillation info for a point */
#define OSC_X_AXIS 1
#define OSC_Y_AXIS 2
#define OSC_Z_AXIS 4

struct polygon {			/* a polygon in 3d */
	int pcount;
	int *vertices;
	unsigned long colour;
	facestyle type;
	float radius;			/* for spheres only */
	};
/* a polygon is a list of vertices, with some colour */

struct object {				/* a 3d object */
	struct object *next;		/* next object in the list */
	int pcount;			/* number of points and faces */
	int fcount;
	struct point *points;		/* world co-ord points */
	struct point *cpoints;		/* view co-ord points */
	XPoint *ppoints;		/* perspective divided points */
	struct polygon *faces;		/* array of faces */
	struct point pos;		/* world co-ords posi of object.
					 * Also point to rotate about */
	struct point cent;		/* centre of object for depth sort */
	float dist;			/* distance (to furthest point) */
	float cdist;			/* distance (to closest point) */
	struct vehicle *parent;
	bool clockwise;
	double angle_vel;		/* object rotation rate */
	double angle;
	struct oscillate *oscli;	/* array of points to oscillate */
	int ocount;
	bool turret;			/* part of the turret? */
	int mx;				/* if this is a map square, these */
	int my;				/* are the co-ords of this piece */
	bool cvalid;			/* are this object's cpoints valid? */
	bool mightsave;			/* can this object's motionless help
					 * save some time later? */
	};
/* an object is a convex 3d shape, with a list of points and faces */

struct vehicle {			/* a vehicle in the game */
	struct vehicle *next;
	struct object **parts;
	int partcount;
	vehicle_type type;		/* see above */
	weapon_type weapon;		/* see above */
	owner_type owner;
	bool flying;
	char *code;			/* unique vehicle id */
	char *name;			/* short description */
	float velocity;			/* forward speed */
	float climb;			/* upward speed */
	double angle;			/* forward direction (rad) */
	double angle_vel;		/* rotation speed (rad/s) */
	double turret_ang;		/* angle of turret above horizontal */
	int hp;				/* hit points */
	struct point bmin;		/* bounding box min&max corners */
	struct point bmax;
	double xspin_vel;		/* spin rates around the axes */
	double yspin_vel;
	double zspin_vel;
	Bool spinrandom;		/* randomise spin on creation?  */
	struct vehicle *firer;		/* only appropriate for bullets */
	float range;			/* distance left to travel */
	bool alive;			/* is this vehicle alive? */
	int vid;			/* unique vehicle id */
	int target;			/* vid of target */
	float reload;			/* time till weapon is reloaded */
	struct {			/* maximums for this vehicle */
		double velocity;
		double angle_vel;
		double altitude;
		double turret_ang;
		int treeheight;
		} max;
	int lock;			/* vehicle locked onto */
	int transfer;			/* vehicle just transferred to */
	int missile;			/* vid of last bullet/missile */
	int ammo;			/* rounds of ammunition left */
	int res;			/* resource units collected */
	int vm9vid;			/* vehicle viewed in vmode 9 */
	int buildervid;			/* the vehicle who built this one */
	char *seed;			/* vehicle seedpods form into */
	int pnum;			/* who controls this vehicle */
	int stcount;			/* number of states in brain */
	struct state *states;		/* array of states */
	int currentstate;		/* state currently in */
	double lasthit;			/* time since last hit by something */
	double lastvid;			/* vid of last target looked at */
	double ftimer;			/* current frame timer */
	Bool bumped;			/* just bumped into something */
	};
/* a vehicle is a group of objects, such as a tank, house or missile */

struct view {
	struct point vrp;		/* view ref point */
	struct point n;			/* view axis */
	struct point v;
	struct point u;
	float d;			/* eye distance */
	float vmat[3][3];
	struct {			/* values of u,v & n last time */
		struct point u,v,n,vrp;
		int vmode;		/* vmode on last call */
		} last;
	int change;			/* change bits from worldtoview */
	};
/* the view contains all details needed for 3-d to 2-d conversion */

#define ch_u 1				/* changes defs for worldtoview */
#define ch_v 2
#define ch_n 4
#define ch_vrp 8
#define ch_uvn (ch_u | ch_v | ch_n)	/* u,v or n changed */
#define ch_all (ch_u | ch_v | ch_n | ch_vrp)

struct map {
	float **gr;			/* ground array */
	float **ht;			/* polygon heights */
	int map_w;			/* size */
	int map_h;
	float size;			/* size of a ground square */
	float scale;			/* max height */
	struct point lookout;		/* where vmode 2 looks from */
	int rfade,gfade,bfade;		/* colours world fades to */
	int tcol;			/* colour of the terrain */
	int gcol;			/* colour of the ground */
	int skycol;			/* colour of the sky */
	Bool ground;			/* is there any ground? */
	struct star *stars;		/* array of stars */
	int scount;			/* number of stars */
	};
/* contains all information about the game map */

#endif

