/* --------------------------------- udpcli.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* This client talks to the fly8src serber and can be used to shut it down.
 * You can request 'stats' from the driver too.
 * At the moment it receives no replies but it is ready to do so.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
/* #include <netinet/in.h>  */
#include "in.h"
#include <netdb.h>
#include <fcntl.h>


/* These are from fly.h.
*/
#define LADDRESS	6

/* These internal header definitions are from udp.c.
*/
#define PHTO		0
#define PHFROM		(PHTO+LADDRESS)
#define PHLEN		(PHFROM+LADDRESS)
#define PHDATA		(PHLEN+2)
#define SOCK_FLY8	SOCK_DGRAM
#define IPPORT_FLY8	0xf8f8

/* This is common to fly8srv.c.
*/
static char	admin[LADDRESS] = {0, 0, 0, 0, 0x12, 0x34};

static char	usage[] = "usage: %s server\n";
static char	*pname = 0;

int
main (int argc, char *argv[])
{
	int			fd;
	struct sockaddr_in	svr, cli;
	char			*protoname;
	struct protoent		*proto;
	char			msg[512];	/* message buffer */
	long			nin;		/* incommind messages counter */
	int			n;		/* incomming message length */
	unsigned long		srv_addr;	/* server address */
	struct hostent		*hostptr;
	int			len;

	pname = argv[0];

	if (argc < 2) {
		printf (usage, pname);
		exit (1);
	}

	protoname = "udp";
	if ((proto = getprotobyname (protoname)) == NULL) {
		printf ("%s: getprotobyname(%s) failed: %s\n", 
			pname, protoname, strerror (errno));
		exit (1);
	}

	if ((fd = socket (AF_INET, SOCK_FLY8, proto->p_proto)) < 0) {
		printf ("%s: socket() failed: %s\n", pname, strerror (errno));
		exit (1);
	}

	n = fcntl (fd, F_GETFL);
	if (fcntl (fd, F_SETFL, n|FLY8_NONBLOCK) < 0) {
		printf ("%s: fcntl(socket) failed: %s\n", pname, strerror (errno));
		exit (1);
	}

/* Set up server (our output) address.
*/
	if ((hostptr = gethostbyname (argv[1])) == NULL) {
		printf ("%s: gethostbyname() failed: %s\n", 
			pname, strerror (errno));
		exit (1);
	}

	if (hostptr->h_addrtype != AF_INET) {
		printf ("%s: not AF_INET address\n", 
			pname);
		exit (1);
	}

	srv_addr = ((struct in_addr *)hostptr->h_addr_list[0])->s_addr;

	memset (&svr, 0, sizeof (svr));
	svr.sin_family      = AF_INET;
	svr.sin_addr.s_addr = srv_addr;
	svr.sin_port        = htons (IPPORT_FLY8);

	printf ("server address is %08lx:%04x\n", 
		ntohl (svr.sin_addr.s_addr), ntohs (svr.sin_port));
	fflush (stdout);

/* Set up client (our input) address.
*/
	memset (&cli, 0, sizeof (cli));
	cli.sin_family      = AF_INET;
	cli.sin_addr.s_addr = htonl (INADDR_ANY);
	cli.sin_port        = htons (0);

	if (bind (fd, (struct sockaddr *) &cli, sizeof (cli)) < 0) {
		printf ("%s: bind() failed: %s\n", pname, strerror (errno));
		exit (1);
	}

	printf ("my     address is %08lx:%04x\n", 
		ntohl (cli.sin_addr.s_addr), ntohs (cli.sin_port));
	fflush (stdout);

/* Set up stdin for non-blocking.
*/
	n = fcntl (STDIN_FILENO, F_GETFL);
	if (fcntl (STDIN_FILENO, F_SETFL, n|FLY8_NONBLOCK) < 0) {
		printf ("%s: fcntl(stdin) failed: %s\n", pname, strerror (errno));
		exit (1);
	}

	for (nin = 0;;) {

/* Accept replies from server.
*/
		len = sizeof (cli);
		n = recvfrom (fd, msg, sizeof (msg), 0,
			(struct sockaddr *)&cli, (int *)&len);
		if (n < 0) {
			if (EWOULDBLOCK != errno) {
				printf ("%s: recvfrom() failed: %s\n", 
					pname, strerror (errno));
				fflush (stdout);
				break;
			}
		} else {
			printf ("%s %3u bytes from server\n",
				pname, n);
			fflush (stdout);
		}

/* Get user input.
*/
		if (NULL == fgets (msg+PHDATA, sizeof(msg)-PHDATA, stdin)) {
			if (EWOULDBLOCK == errno)
				continue;
			break;
		}

/* Check for quit message. The FLY8_NONBLOCK seems to make it impossible to
 * get a good EOF on a ^D.
*/
		if (!strcmp (msg+PHDATA, "end\n") ||
		    !strcmp (msg+PHDATA, "exit\n") ||
		    !strcmp (msg+PHDATA, "quit\n"))
			break;

/* Identify message as admin type.
*/
		memset (msg, 0, PHDATA);
		memcpy (msg+PHFROM, admin, LADDRESS);

		n = PHDATA + strlen (msg+PHDATA) + 1;
		if (n != sendto (fd, msg, n, 0, (struct sockaddr *)&svr,
							sizeof (svr))) {
			if (EWOULDBLOCK == errno) {
				printf ("%s: WOULDBLOCK\n", 
					pname);
				fflush (stdout);
				continue;
			}
			printf ("%s: sendto() failed: %s\n", 
				pname, strerror (errno));
			fflush (stdout);
			break;
		}
		printf ("%s %ld: %3u bytes sent\n", 
			pname, ++nin, n);
		fflush (stdout);
	}

	close (fd);

	n = fcntl (STDIN_FILENO, F_GETFL);
	fcntl (STDIN_FILENO, F_SETFL, n&~FLY8_NONBLOCK);

	exit (0);
	return (0);
}
