/* --------------------------------- max2mac.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Read a text macro definition and build a macro file. See 'mac2max'.
*/

#include "fly.h"
#include "keyname.h"


/* This is needed since "fly.h" defines it.
*/
struct status	NEAR st = {0};

static Ulong lineno = 1L, column = 0L, errs = 0L;

static int
putshort (short i, FILE *f)
{
	if (errs)
		return (0);

	fputc (0x00ff&(i>>8), f);
	if (ferror (f)) {
		++errs;
		fprintf (stderr, "write error\n");
		return (1);
	}

	fputc (0x00ff&i, f);
	if (ferror (f)) {
		++errs;
		fprintf (stderr, "write error\n");
		return (1);
	}

	return (0);
}

static char tabCTRL[] = "@abcdefghijklmnopqrstuvwxyz[\\]^_";

static Ulong
keyval (Ulong c)
{
	if (c < 32)
		return (tabCTRL[c]| K_CTRL);
	return (c);
}

static Ulong
eatline (FILE *f)
{
	int	c;

	do {
		c = fgetc (f);
	} while (EOF != c && '\n' != c);
	++lineno;
	column = 0;
	return (EOF == c ? VK_EOF : (Ulong)c);
}

static Ushort buf_str[1024] = {0};

static int
getstring (FILE *f)
{
	int	c, n, escape;

	for (n = 0, escape = 0;;) {
		c = fgetc (f);
		if (EOF == c) {
			++errs;
			printf ("EOF inside string at %ld:%ld\n",
				lineno, column);
			return (-1);
		}
		if (escape) {
			switch (c) {
			default:
				goto raw;
			case 'a':
				c = 'g';
				break;
			case 'b':
				c = 'h';
				break;
			case 'e':
				c = '[';
				break;
			case 'f':
				c = 'l';
				break;
			case 'n':
				c = 'm';
				break;
			case 'r':
				c = 'j';
				break;
			case 't':
				c = 'i';
				break;
			case 'v':
				c = 'k';
				break;
			case '\n':
				continue;	/* traditional line break */
			}
			c |= K_CTRL;
raw:			escape = 0;
		} else if ('\\' == c) {
			escape = 1;
			continue;
		} else if ('"' == c)
			return (n);
		else if ('\n' == c)		/* allow line split */
			continue;
		if (n >= rangeof (buf_str)) {
			++errs;
			printf ("string too long at %ld:%ld\n",
				lineno, column);
			continue;
		}
		buf_str[n++] = (Ushort)c;
	}
	/* never reached */
}

static Ulong
getterm (FILE *f)
{
	char		keyname[32+1], *p;
	int		n, i, c;
	Ulong		l;

retry:
	for (n = 0;;) {
		c = fgetc (f);
		++column;
		if (EOF == c) {
			if (0 == n)
				return (VK_EOF);
			break;
		} else if ('\n' == c) {
			++lineno;
			column = 0;
			if (n)
				break;
		} else if (isspace (c)) {
			if (n)
				break;
		} else if (n < sizeof (keyname) - 1) {	/* truncate */
			if (1 == n && '#' == keyname[0]) {
				if ('#' == c)
					return (VK_COM);
				if ('"' == c)
					return (VK_STR);
			}
			keyname[n++] = (char)c;
		}
	}

/* Look word up
*/
	keyname[n] = '\0';
	for (i = 0; k_name[i].name; ++i) {
		if (!stricmp (k_name[i].name, keyname))
			return (k_name[i].value);
	}

	if (1 == n && isprint (keyname[0]))
		return ((Uchar)keyname[0]);

	if (n > 1 && '\\' == keyname[0]) {
		if ('0' == keyname[1]) {
			if (2 == n)
				return (keyval (0));
			l = strtol (keyname+2, &p, 8);
			if (l >= 256L) {
				printf ("\\0%lo too big at %ld:%ld\n",
					l, lineno, column);
				l &= 0x00ffL;
			}
			if ('\0' == *p)
				return (keyval (l));
		} else if ('x' == keyname[1]) {
			if (n > 2) {
				l = strtol (keyname+2, &p, 16);
				if (l >= 256L) {
					printf ("\\x%lx too big at %ld:%ld\n",
						l, lineno, column);
					l &= 0x00ffL;
				}
				if ('\0' == *p)
					return (keyval (l));
			}
		} else {
			l = strtol (keyname+1, &p, 10);
			if (l >= 256L) {
				printf ("\\%lu too big at %ld:%ld\n",
					l, lineno, column);
				l &= 0x00ffL;
			}
			if ('\0' == *p)
				return (keyval (l));
		}
	}

	++errs;
	printf ("bad token '%s' at %ld:%ld\n", keyname, lineno, column);
	if (EOF == c)
		return (VK_EOF);
	goto retry;
}

static Ulong
getkey (FILE *f)
{
	Ulong	c, key;

	for (key = 0L;;) {
		c = getterm (f);
		if (VK_EOF == c) {
			if (key) {
				++errs;
				printf ("unexpected end of file at %ld:%ld\n",
					lineno, column);
			}
			return (VK_EOF);
		} else if (VK_DEF == c) {
			if (key) {
				++errs;
				printf ("bad key preceding 'def' at %ld:%ld\n",
					lineno, column);
			}
			key = VK_DEF;
		} else if (VK_COM == c) {
			if (key) {
				++errs;
				printf ("bad key preceding '##' %ld:%ld\n",
					lineno, column);
			}
			key = 0L;
			c = eatline (f);
			if (VK_EOF == c)
				return (VK_EOF);
		} else if (VK_STR == c) {
			if (key) {
				++errs;
				printf ("bad key preceding '#\"' %ld:%ld\n",
					lineno, column);
			}
			return (VK_STR);
		} else {
			key |= c;
			if (c & K_RAW)
				return (key);
		}
	}
	/* never reached */
}

static Ushort definition[1024] = {0};

static void
buildmac (char *mname)
{
	FILE	*mac;
	int	i, n;
	Ulong	c;

	mac = fopen (mname, WBMODE);
	if (!mac) {
		++errs;
		printf ("open '%s' failed", mname);
		return;
	}

	for (;;) {
		c = getkey (stdin);
		if (VK_EOF == c)
			break;
		if (!(VK_DEF & c)) {
			++errs;
			printf ("'def' not found at %ld:%ld\n", lineno, column);
			continue;
		}
def:
		if (putshort ((short)c, mac))	/* macro name */
			break;
		for (n = 0;;) {
			c = getkey (stdin);
			if (VK_EOF == c || (VK_DEF & c))
				break;
			if (VK_STR == c) {
				i = getstring (stdin);
				if (i <= 0)
					continue;
				if (n+i > rangeof (definition)) {
					++errs;
					printf ("macro too long at %ld:%ld\n",
						lineno, column);
				} else {
					memcpy (definition+n, buf_str, 2*i);
					n += i;
				}
				continue;
			}
			if (n >= rangeof (definition)) {
				++errs;
				printf ("macro too long at %ld:%ld\n",
					lineno, column);
				continue;
			}
			definition[n++] = (Ushort)c;
		}
		if (0 == n) {
			++errs;
			printf ("empty macro at %ld:%ld\n", lineno, column);
		}
		if (putshort ((short)n, mac))	/* macro length */
			break;
		for (i = 0; i < n; ++i) {	/* macro body */
			if (putshort (definition[i], mac))
				goto ret;
		}
		if (VK_DEF & c)
			goto def;
		if (VK_EOF == c)
			break;
	}
ret:
	fclose (mac);
}

int
main (int argc, char *argv[])
{
	char	*mname;

	if (argc < 2 || !(mname = argv[1]))
		mname = "fly.mac";

	buildmac (mname);

	exit (0);
	return (0);
}
