/* --------------------------------- plane.h -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Common header for plane dynamics models.
*/

#ifndef FLY8_PLANE
#define FLY8_PLANE

#include "hud.h"

struct e_parms {
	struct e_parms	*next;
	char	name[10];

	long	weight;			/* empty weight [lb] */
	short	Ixx;			/* Ixx as I/Weight in kg-m2/kg */
	short	Iyy;			/* Iyy */
	short	Izz;			/* Izz */
	short	Izx;			/* Izx */

	short	liftoff_speed;		/* [kts] */
	short	max_lift;		/* [vm] */
	short	min_lift;		/* [vm] */
	ANGLE	pitch_rate;		/* at 300 knots full elev. [ang/s] */
	ANGLE	roll_rate;		/* at 300 knots full ail. [ang/s] */
	short	eyez;			/* pilot eye above cg [vm] */
	short	eyey;			/* pilot eye forwad of cg [vm] */
	ANGLE	gpitch;			/* pitch on ground [ang] */
	long	ceiling;		/* operational ceiling [ft] */
	short	stores[N_WE];		/* count */

	short	brake_mu;		/* friction with brakes applied */
	short	wheel_mu;		/* friction when freewheeling */

	short	mil_thrust;		/* [lbf] */
	short	mil_sfc;		/* lb_fuel/(lb_thrust*hour) */
	short	ab_thrust;		/* [lbf] */
	short	ab_sfc;			/* lb_fuel/(lb_thrust*hour) */
	Ushort	fuel_capacity;		/* internal fuel [lb] */
	ANGLE	Ea;			/* Engine rig ang vs. plane [ang] */
	ANGLE	Eb;			/* Engine offset ang vs. plane [ang] */
	short	Er;			/* Engine offset vs. cg [vm] */

	short	wing_area;		/* wings area [vm] */
	short	wing_span;		/* tip to tip [vm] */
	short	wing_cord;		/* mean aerodynamic cord [vm] */
	short	ACy;			/* wing ac forward of cg [vm] */
	short	ACz;			/* wing ac upward of cg [vm] */
	ANGLE	Aoffset;		/* Wing rigging angle vs. plane [ang] */
	ANGLE	Cl0;			/* alpha where Cl=0 [ang] */
	short	maxCl;			/* max Cl for wing foil */
	short	minCl;			/* min Cl for wing foil */
	short	FEff;			/* flaps effective alpha/flaps */
	short	FEffCl;			/* flaps Clmax rate/DegFlaps */
	short	lefEffCl;		/* Clmax/LE flaps deg */
	short	efficiency_factor;	/* Oswald's efficiency factor */
	short	Cm0w;			/* wing foil Cm at 0 alpha */

	short	tail_area;		/* [vm] */
	short	tail_span;		/* [vm] */
	short	TACy;			/* tail ac forward of cg [vm] */
	short	TACz;			/* tail ac upward of cg [vm] */
	ANGLE	Toffset;		/* Tail rigging angle vs. wing [ang] */
	short	Tvol;			/* tail volume */
	short	TmaxCl;			/* max Cl for tail foil */
	short	TminCl;			/* min Cl for tail foil */

	short	rudd_area;		/* [vm] */
	short	rudd_span;		/* [vm] */
	short	RACy;			/* rudder ac forward of cg [vm] */
	short	RACz;			/* rudder ac upward of cg [vm] */
	short	RmaxCl;			/* max Cl for rudder foil */

	short	Cdp0;			/* parasitic drag: profile */
	short	Cds;			/* parasitic drag: speed brakes */
	short	Cdg;			/* parasitic drag: gear */
	short	CdMK82;			/* parasitic drag: each MK82 */

	ANGLE	MaxFlaps;		/* max flaps [ang] */
	ANGLE	MaxLEFlaps;		/* max LE flaps [ang] */
	ANGLE	MaxSpoilers;		/* max spoilers [ang] */
	ANGLE	MaxElevators;		/* max elevators [ang] */
	ANGLE	MaxAilerons;		/* max ailerons [ang] */
	ANGLE	MaxRudder;		/* max rudder [ang] */

	ANGLE	AFamin;			/* AutoFlaps: min aoa to engage */
	short	AFrate;			/* AutoFlaps: flaps/aoa rate */
	ANGLE	AFmax;			/* AutoFlaps: max flaps authority [%] */

	ANGLE	ALEFamin;		/* AutoLEFlaps: min aoa to engage */
	short	ALEFrate;		/* AutoLEFlaps: leFlaps/aoa rate */

	short	AErate;			/* AutoElevators: speed factor */

	short	APrate;			/* AutoPedals: speed factor */

	short	Cydr;			/* rudder sideforce */
	short	Cybeta;			/* vx damping */

	short	Cm0;			/* total Cm at 0 alpha */
	short	Cmde;			/* elevators effectiveness */
	short	Cmq;			/* pitch damping */
	short	Cmalpha;		/* alpha (stabilizer) induced pitch */
	short	Cmalphadot;		/* alpha rate induced pitch */

	short	Clda;			/* aileron effectiveness */
	short	Clp;			/* roll damping */
	short	Clbeta;			/* dihedral effect */
	short	Cldr;			/* roll from rudder */

	short	Cndr;			/* rudder effectiveness */
	short	Cnr;			/* yaw damping */
	short	Cnbeta;			/* weathercock stability */
	short	Cnda;			/* ailerons induced yaw */
	short	Cnp;			/* roll induced yaw */

	short	hudtype;		/* HUD_F16 etc. */

	short	opt[10];		/* options */
#define MODEL_BASIC	0		/* opt[0] values */
#define MODEL_CLASSIC	1
#define MODEL_XPLANE	2
#define MODEL_YPLANE	3
#define MODEL_FPLANE	4

	PID	PIDthrottle;
	PID	PIDpitch;
	PID	PIDroll;
	struct gear {		/* gear tip position relative to cg */
		short	x;		/* + means right */
		short	y;		/* + means forward */
		short	z;		/* - means down */
		short	dgmax;		/* max strut deflection */
		short	dtp;		/* tyre deflection at P */
		short	P;		/* strut precharge force */
		short	Cv;		/* strut damping coeff. */
		ANGLE	emax;		/* Steering angle at full pedals. */
		short	ur;		/* rolling friction coeff. */
		short	ub;		/* full brake friction coeff. */
		short	us;		/* sideslip friction coeff. */
		short	rateup;		/* percent/sec raise rate */
		short	ratedn;		/* percent/sec lower rate */
	} gear[5];
};

#define EEP(p)	EE(p)->parms
#define EP	EEP(p)

#ifndef EX
#define EX	EE(p)
#endif

/* autop.c */
extern void	FAR SetKillCorrection (OBJECT *p, OBJECT *target, VECT R,
	int *tti);
extern void	FAR dynamics_auto (OBJECT *p);

/* engine.c */
extern void	FAR f16engine (OBJECT *p, short sos);

/* gear.c */
extern void	FAR LandGear (OBJECT *p, VECT F, VECT MM);
extern void	FAR LandGearDamp (OBJECT *p);

/* oplane.c */
#define DDshow(p,n,t,v)	if (CC==(p)) fDDshow (n, t, v)
#define CCshow(p,n,t,v)	if (CC==(p)&&(st.flags&SF_DEBUG)) fCCshow (n, t, v)
#define CFshow(p,t,v)	if (CC==(p)&&(st.flags&SF_DEBUG)) fCFshow (t, v)
#define CAshow(p,t,v)	if (CC==(p)&&(st.flags&SF_DEBUG)) fCAshow (t, v)
#define CVshow(p,t,v)	if (CC==(p)&&(st.flags&SF_DEBUG)) fCVshow (t, v)

extern void	FASTCALL FAR fDDshow (int frac, char *title, long value);
extern void	FASTCALL FAR fCCshow (int frac, char *title, long value);
extern void	FASTCALL FAR fCFshow (char *title, int value);
extern void	FASTCALL FAR fCAshow (char *title, int value);
extern void	FASTCALL FAR fCVshow (char *title, int value);

extern void	FAR CCnote (OBJECT *p, char *note);
extern void	FAR CCland (OBJECT *p);
extern void	FAR CCfly  (OBJECT *p);
extern int	FAR check_land (OBJECT *p);
extern int	FAR check_takeoff (OBJECT *p);
extern void	FAR supply (OBJECT *p, int mode);
extern void	FAR dampen (short *old, int new, int factor);
extern int	FAR on_runway (OBJECT *p);
extern void	FAR plane_smoke (OBJECT *p);
extern int	FAR dynamics_input (OBJECT *p);

/* om61.c */
#define	BULLETV		(1000*VONE)
#define	BULLETSCATTER	(D90/180)
extern int	FAR BulletSpeed (OBJECT *p, VECT V);

/* omk82.c */
extern int	FAR BombSpeed (OBJECT *p, VECT V);
extern int	FAR BombIP (LVECT R, VECT V, long tz, LVECT IP);

/* prm.c */
extern void 	(FAR* FAR flight_models[])(OBJECT *p, int action);
extern struct e_parms * FAR parms_get (char *pname);
extern void	FAR parms_free (void);

/* sixdof.c */
extern void	FAR SixDOF (OBJECT *p, VECT F, VECT MM, MAT I);

#endif
