/* --------------------------------- loop.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* This is the main 'busy-loop' of the simulation.
*/

#include "fly.h"
#include "hud.h"


static int	NEAR FASTCALL world_dynamics (void);
static void	NEAR render_bg (OBJECT *pov, int type, int frame,
			int hudinfront);
static void	NEAR render_fg (OBJECT *pov, int type, int frame,
			int hudinfront);
static void	NEAR FASTCALL render_picture (void);
static void	NEAR FASTCALL show_picture (void);

#ifndef INLINE
INLINED extern int FAR FASTCALL
add_line (int x, int y, int t)
{
	register BUFLINE	*p;
	register BUFFER		*b;

	if ((T(b = st.buf[TAIL]) && (p = b->p) <= b->end) ||
			(T(b = new_buffer ()) && T(p = b->p))) {
		p->x = (short)x;
		p->y = (short)y;
		p->t = (short)t;
		b->p++;
		return (0);
	}
	return (1);
}
#endif /*INLINE*/

extern void FAR
active_loop (Ulong t)
{
	st.present += t;
	st.ObjectTimeout = st.present + TO_OBJECT;
	st.PlayerTimeout = st.present + TO_PLAYER;

	if (t > 1000L)			/* ignore long pauses */
		st.interval = REFRESH;
	else
		st.interval = (int)t;

	st.dither = Frand () % 1000;

	Tm->Interval (TMR_START, 0L);

	if (!(st.flags & SF_PAUSED)) {
		if (world_dynamics ())
			die ();
		if (objects_dynamics (st.interval))
			die ();
		if (land_update (CV))
			die ();

		render_picture ();	/* build picture into display list */
	}

	show_picture ();		/* draw display list on screen */

/* Gather stats.
*/
	st.misc[0] = (int)Tm->Interval (TMR_STOP, 10000L);
	st.misc[2] = (int)st.stats[27];
	st.misc[3] = (int)st.stats[28];
	st.misc[4] = (int)st.stats[29];
	st.misc[5] = (int)st.stats[40];

	++st.stats[31];
	st.stats[32] += st.misc[0];
	st.stats[33] += st.misc[1];
	st.stats[34] += st.stats[27];
	st.stats[35] += st.stats[28];
	st.stats[36] += st.stats[40];
	st.stats[37] += st.stats[29];

	st.stats[27] = 0L;
	st.stats[28] = 0L;
	st.stats[40] = 0L;
	st.stats[29] = 0L;
}

static void NEAR FASTCALL
show_buf (BUFFER *b)
{
	register BUFLINE	*p;
	register int	op;
	BUFLINE		*end;
	void		(FAR* MoveTo) (int x1, int y1);
	void		(FAR* DrawTo) (int x2, int y2, Uint c);

	MoveTo = Gr->MoveTo;
	DrawTo = Gr->DrawTo;

	p = b->first;
	end = b->p;
	for (; p < end; ++p) {
		if ((op = p->t) >= 0)
			DrawTo (p->x, p->y, op);
		else if (op == T_MOVE)
			MoveTo (p->x, p->y);
		else if (op == T_MOR || op == T_MXOR || op == T_MSET)
			Gr->WriteMode (op);
		else if (op == T_ELLIPSE) {
			Gr->Ellipse (p->x, p->y, (p+1)->x, (p+1)->y, (p+1)->t);
			++p;
		} else if (op == T_NOP)
			;
		else
			break;		/* should not happen! */
	}
}

static void NEAR FASTCALL
show_buffer (BUFFER *b)
{
	Gr->WriteMode (T_MSET);			/* initial mode */

	for (; b; b = b->next) {
		show_buf (b);
		sys_poll ();
	}
}

static void NEAR FASTCALL
show_buffers (int which)
{
	int	i;

	show_buffer (st.bufs[which]);

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;
		show_buffer (st.hdd[i].bufs[which]);
	}
}

static void NEAR FASTCALL
erase_buf (BUFFER *b)
{
	register BUFLINE	*p;
	register int	op;
	BUFLINE		*end;
	Uint		bg;
	void		(FAR* MoveTo) (int x1, int y1);
	void		(FAR* DrawTo) (int x2, int y2, Uint c);

	MoveTo = Gr->MoveTo;
	DrawTo = Gr->DrawTo;

	bg = CS->BgColor;

	Gr->WriteMode (T_MSET);			/* initial mode */
	p = b->first;
	end = b->p;
	for (; p < end; ++p) {
		if ((op = p->t) >= 0)
			DrawTo (p->x, p->y, bg);
		else if (op == T_MOVE)
			MoveTo (p->x, p->y);
		else if (op == T_MOR || op == T_MXOR || op == T_MSET)
			;
		else if (op == T_ELLIPSE) {
			Gr->Ellipse (p->x, p->y, (p+1)->x, (p+1)->y, bg);
			++p;
		} else if (op == T_NOP)
			;
		else
			break;		/* should not happen! */
	}
}

static void NEAR FASTCALL
erase_buffer (BUFFER *b)
{
	for (; b; b = b->next) {
		erase_buf (b);
		sys_poll ();
	}
}

static void NEAR FASTCALL
erase_buffers (int which)
{
	int	i;

	if (Gr->Clear) {
		Gr->Clear (CS);
		sys_poll ();
		return;
	}

	erase_buffer (st.bufs[which]);

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;
		erase_buffer (st.hdd[i].bufs[which]);
	}
}

static void NEAR FASTCALL
free_buffer (BUFFER *b)
{
	BUFFER	*next;

	for (; b; b = next) {
		next = b->next;
		memory_free (b,
			sizeof (*b) + (b->size - 1) * sizeof (BUFLINE));
		--st.nbuffers;
	}
}

static void NEAR FASTCALL
free_buffers (int which)
{
	int	i;

	free_buffer (st.bufs[which]);
	st.bufs[which] = 0;

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;
		free_buffer (st.hdd[i].bufs[which]);
		st.hdd[i].bufs[which] = 0;
	}
}

extern void FAR
buffers_free (void)
{
	int	i;

	for (i = 0; i < NBUFS; ++i)
		free_buffers (i);

	free_buffer (st.buf[HEAD]);
	st.buf[HEAD] = st.buf[TAIL] = 0;
}

static long NEAR FASTCALL
buffer_size (BUFFER *b)
{
	long	l;

	for (l = 0; b; b = b->next)
		l += b->p - b->first;
	return (l);
}

static long NEAR FASTCALL
buffers_size (int which)
{
	int	i;
	long	l;

	l = buffer_size (st.bufs[which]);
	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;
		l += buffer_size (st.hdd[i].bufs[which]);
	}
	return (l);
}

static int NEAR FASTCALL
world_dynamics (void)
/*
 * Do the global housekeeping. This routine is called repeatedly from the
 * main program just before the objects simulation.
*/
{
	if (st.quiet == 2)				/* engine noise */
#if 0
		TnEngine[0] = iabs (CC->speed/4);
		TnEngine[0] = iabs (EE(CC)->thrust/4);
#endif
		TnEngine[0] = iabs (EE(CC)->power/8);
	else
		TnEngine[0] = 0;

	return (0);
}

extern void FAR
double_buffer (int mode)
{
	Ushort	flags;

	if (!((mode ^ st.flags1) & SF_DBUFFERING))
		return;
	if (CS->device->npages > 1) {
		flags = st.flags;
		st.flags |= SF_SIMULATING;
		Gr->CloseTextWindow (CT);
		reset_page (1);

		st.flags1 ^= SF_DBUFFERING;
		if (st.flags1 & SF_DBUFFERING) {

			Gr->SetVisual (1-st.which_buffer);
			Gr->SetVisual (st.which_buffer);
		} else {
			Gr->SetVisual (0);
			Gr->SetActive (0);
		}

		Gr->OpenTextWindow (CT);
		clear_text ();				/* clear */
		Gr->SetTextPos (2, 1);
		show_fixed (0);				/* show */

		st.flags = flags;
		MsgPrintf (50, "%s buffering",
			st.flags1 & SF_DBUFFERING ? "double" : "single");
	} else
		MsgPrintf (50, "no double buffering");
}

extern void FAR
reset_page (int empty)
{
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);
		if (Gr->Clear)
			Gr->Clear (CS);
		else {
			erase_buffers (0);
			erase_buffers (1);
		}
		Gr->SetActive (1-st.which_buffer);
	}
	if (Gr->Clear)
		Gr->Clear (CS);
	else {
		erase_buffers (0);
		erase_buffers (1);
	}
	if (empty) {
		free_buffers (0);
		free_buffers (1);
	}
	if (!Gr->Clear)
		show_fixed (1);
}

static void NEAR
draw_border (int orgx, int orgy, int sizex, int sizey, int c)
{
	Gr->MoveTo (orgx - sizex, orgy - sizey);
	Gr->DrawTo (orgx + sizex, orgy - sizey, c);
	Gr->DrawTo (orgx + sizex, orgy + sizey, c);
	Gr->DrawTo (orgx - sizex, orgy + sizey, c);
	Gr->DrawTo (orgx - sizex, orgy - sizey, c);
}

static void NEAR FASTCALL
show_border (VIEW *v, int c, int dx)
{
	int	orgx, orgy, sizex, sizey;

	orgx  = fmul (v->window->orgx, v->screen->sizex) + v->screen->minx;
	orgy  = fmul (v->window->orgy, v->screen->sizey) + v->screen->miny;
	sizex = fmul (v->window->maxx, v->screen->sizex);
	sizey = fmul (v->window->maxy, v->screen->sizey);

	draw_border (orgx, orgy, sizex, sizey, c);
	if (dx)
		draw_border (orgx+dx, orgy, sizex, sizey, c);
}

static void NEAR FASTCALL
show_borders (int c)
{
	int	dx, i;

	dx = (VIS_STEREOSCOPIC == st.stereo)
		? fmul (muldiv(CW->maxx, 2*st.gap+2, st.gap-1), CS->sizex)
		: 0;

	show_border (CVIEW, c, dx);

	for (i = 0; i < NHDD; ++i) {
		if (st.hdd[i].view.window)
			show_border (&st.hdd[i].view, c, 0);
	}
}

static void NEAR FASTCALL
show_fixed_page (int del)
{
	if (st.flags & SF_BLANKER)
		return;

	show_borders (del ? CS->BgColor : CS->BoColor);
}

extern void FAR
show_fixed (int del)
{
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);
		show_fixed_page (del);
		Gr->SetActive (1-st.which_buffer);
	}
	show_fixed_page (del);
}

static void NEAR
render_bg (OBJECT *pov, int type, int frame, int hudinfront)
{
	int	skycolor;

	Tm->Interval (TMR_START, 0L);
	skycolor = frame ? ((1 == frame) ? st.red : st.blue)
			 : st.skyblue;
	show_sky (CVIEW, pov, skycolor);

	if (frame && !hudinfront)
		show_main (CVIEW, pov, type, 1);
	st.stats[40] += Tm->Interval (TMR_STOP, 10000L);
}

static void NEAR
render_fg (OBJECT *pov, int type, int frame, int hudinfront)
{
	Tm->Interval (TMR_START, 0L);
	show_main (CVIEW, pov, type, (frame && !hudinfront) ? 2 : (1|2));
	show_inst (CVIEW, CV);
	st.stats[40] += Tm->Interval (TMR_STOP, 10000L);
}

extern BUFFER * FAR
new_buffer (void)
{
	BUFFER	*b;

	if (st.nbuffers >= st.maxbuffers)
		return (0);
	b = (BUFFER *) memory_alloc (sizeof (*b) +
					(BUFLEN - 1) * sizeof (BUFLINE));
	if (!b)
		return (0);

	++st.nbuffers;
	b->next = 0;
	b->size = BUFLEN;
	b->end = b->first + b->size - 1;
	b->p = b->first;
	if (st.buf[TAIL])
		st.buf[TAIL]->next = b;
	else
		st.buf[HEAD] = b;
	st.buf[TAIL] = b;

	return (b);
}

extern int FAR
add_5op (int t, int a, int b, int c, int d, int e)
{
	register BUFFER	*buf;

	if (T(buf = st.buf[TAIL]) && buf->p == buf->end)
		add_line (0, 0, T_NOP);		/* will force new buf */

	if (!add_line (a, b, t))
		add_line (c, d, e);
	return (0);
}

static void NEAR FASTCALL
show_picture (void)
{
	int	bufold, bufnew;

	if (st.flags & SF_CLEARED) {
		st.flags &= ~SF_CLEARED;
		show_fixed (0);
	}

	if ((st.flags & SF_PAUSED) && (VIS_ALTERNATING != st.stereo))
		goto ret;

	bufnew = st.which_buffer;
	st.which_buffer = (st.which_buffer+1)%NBUFS;
	bufold = st.which_buffer;

	Tm->Interval (TMR_START, 0L);
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);	/* access new page */
		show_buffers (bufnew);			/* draw new page */
		if (Gr->Clear)
			show_fixed (0);
		Tm->Interval (TMR_START, 0L);
		Gr->SetVisual (st.which_buffer);	/* show new page */
		if (VIS_ALTERNATING == st.stereo)
			Gr->Shutters (st.which_buffer);
		st.stats[29] = Tm->Interval (TMR_STOP, 10000L);
		Gr->SetActive (1-st.which_buffer);	/* access old page */
		erase_buffers (bufold);			/* erase old page */
		free_buffers (bufold);			/* erase old page */
	} else {
		erase_buffers (bufold);			/* erase old screen */
		free_buffers (bufold);			/* erase old page */
		if (VIS_ALTERNATING == st.stereo)
			Gr->Shutters (st.which_buffer);
		show_buffers (bufnew);			/* show new screen */
		if (Gr->Clear)
			show_fixed (0);
	}
	st.misc[1] = (int)(Tm->Interval (TMR_STOP, 10000L) - st.stats[29]);
ret:
	Gr->Flush ();
}

extern OBJECT * FAR
get_viewer (int type)
{
	OBJECT	*pov;

	for (pov = CO; pov; pov = pov->next) {
		if (pov->name == O_VIEWER && pov->misc[0] == type)
			break;
	}
	return (pov);
}

extern void FAR
save_viewport (OBJECT *p)
{
	if (!p->viewport) {
		if (!NEW (p->viewport))
			return;
	}
	memcpy (p->viewport, CP, sizeof (*p->viewport));
}

extern void FAR
get_viewport (OBJECT *p)
{
	if (p->viewport)
		memcpy (CP, p->viewport, sizeof (*CP));
	else {
		CP->x    = 0*FONE;		/* viewer at center */
		CP->y    = 0*FONE;
		CP->shift = 0;
		CP->eyex = 0;
		CP->eyey = 0;
		CP->eyez = 0;
		CP->rotx = 0;
		CP->roty = 0;
		CP->rotz = 0;
	}
}

/* Render the whole screen. Some data may already be in the display list
 * from spurious debug stuff.
*/

static void NEAR FASTCALL
render_picture (void)
{
	ANGLE	cross;				/* stereo cross-eyed angle */
	int	rev, shift, rotz, hud, scene, hudinfront;
	int	frame;				/* 0=mono 1=left 2=right */
	int	type;				/* 0=main other=HDD type */
	OBJECT	*pov;

	st.flags |= SF_MAIN;			/* main display */

	if (st.flags1 & SF_EXTVIEW) {		/* find viewer */
		type = st.extview;
		hud = (HDT_HUD == type);
		scene = scenery (type);
		if (scene || hud) {
			pov = get_viewer (hud ? HDT_FRONT : type);
			if (!pov) {
				pov = CV;
				type = 0;
			}
		} else
			pov = CV;
	} else {
		pov = CV;
		type = 0;
		scene = 1;
	}

	cross = 0;	/*ASIN (fdiv (st.paralax, st.focus));*/

	if (st.stereo) {
		rev = st.flags1 & SF_STEREOREV;
		if (!scene)
			hudinfront = 1;
		else
			hudinfront = st.flags1 & SF_HUDINFRONT;
		if (VIS_ALTERNATING == st.stereo && st.which_buffer)
			rev = !rev;
		if (rev) {
			shift = -st.paralax;	/* transverse: right eye */
			rotz  = -cross;
		} else {
			shift = st.paralax;	/* parallel:   left eye */
			rotz  = cross;
		}
		CP->shift -= shift;		/* left eye */
		CP->rotz  += rotz;
	} else {
		shift = rotz = 0;		/* avoid compiler warning */
		hudinfront = 0;
	}

	if (VIS_REDBLUE == st.stereo) {
		add_line (0, 0, T_MOR);
	} else {
		st.cfg = st.red;
		st.hfg = st.hudlow;
		st.hfgi = st.hudhigh;
		add_line (0, 0, T_MSET);
	}

/* First we show the left eye view. This is also used to show the single view
 * (mono or alternating stereo).
*/
	frame = VIS_REDBLUE == st.stereo ? 1 : 0;	/* mono/left frame */
	if (frame)
		st.cfg = st.hfg = st.hfgi = st.red;

	if (scene) {
		render_bg (pov, type, frame, hudinfront);
		objects_show (0, CVIEW, pov, frame, 0, 0);
	}

	if (st.stereo && hudinfront) {
		CP->shift += shift;			/* back to center */
		CP->rotz  -= rotz;
	}

	if (VIS_REDBLUE != st.stereo)
		render_fg (pov, type, frame, hudinfront);

	if (st.stereo && !hudinfront) {
		CP->shift += shift;			/* back to center */
		CP->rotz  -= rotz;
	}
		
/* Second we show the right eye view (for non-alternating stereo).
*/
	if (st.stereo && VIS_ALTERNATING != st.stereo) {
		if (VIS_STEREOSCOPIC == st.stereo)
			CW->orgx += muldiv (CW->maxx, 2*st.gap+2, st.gap-1);

		CP->shift += shift;			/* right eye */
		CP->rotz  -= rotz;

		frame = VIS_REDBLUE == st.stereo ? 2 : 0; /* right frame */
		if (frame)
			st.cfg = st.hfg = st.hfgi = st.blue;
		if (scene) {
			render_bg (pov, type, frame, hudinfront);
			objects_show (0, CVIEW, pov, frame, 0, 0);
		}

		if (hudinfront) {
			add_line (0, 0, T_MSET);
			CP->shift -= shift;		/* back to center */
			CP->rotz  += rotz;
			if (VIS_REDBLUE == st.stereo)
				st.cfg = st.hfg = st.hfgi = st.white;
		}

		render_fg (pov, type, frame, hudinfront);

		if (!hudinfront) {
			CP->shift -= shift;		/* back to center */
			CP->rotz  += rotz;
		}

		if (VIS_STEREOSCOPIC == st.stereo)
			CW->orgx -= muldiv(CW->maxx,2*st.gap+2,st.gap-1);
	}

	st.bufs[st.which_buffer] = st.buf[HEAD];
	st.buf[HEAD] = st.buf[TAIL] = 0;

	if (VIS_REDBLUE == st.stereo) {
		st.cfg = st.red;
		st.hfg = st.hudlow;
		st.hfgi = st.hudhigh;
	}

	st.flags &= ~SF_MAIN;				/* secondary displays */

	show_hdd ();

	st.stats[26] = buffers_size (st.which_buffer);
}
