/* warp_tcl.c: TCL interface to warp routines.
**
** Written and copyright (C) 1996 by Michael J. Gourlay
**
** PROVIDED AS IS.  NO WARRENTIES, EXPRESS OR IMPLIED.
*/

#include <stdio.h>
#include <stdlib.h>

#include <tcl.h>

#include "my_malloc.h"
#include "mesh.h"
#include "rgba_image.h"
#include "hash.h"
#include "mesh_tcl.h"
#include "image_tcl.h"

#include "warp_tcl.h"




/* warpCmd: TCL command to warp an image
** Creates an output RgbaImage, which the user is responsible for freeing.
** Creates (and frees) the tween mesh given the tween_param:
**   If tween_param is positive, then warp forward from src to tween.
**   If tween_param is negative, then warp backward from dst to tween.
**
** Returns tag to RgbaImage.
*/
int
warpCmd(ClientData client_data, Tcl_Interp *interp, int argc, char *argv[])
{
  char *mesh_src_tag, *mesh_dst_tag;
  char *rgba_image_tag;
  RgbaImageT *rgbaP;
  RgbaImageT *warped;
  MeshT *mesh_srcP, *mesh_dstP;
  float tween_param;
  MeshT tween;

  /* ---- Parse the TCL command arguments */
  if(argc != 5) {
    interp->result = "Usage: warp ?start_mesh? ?end_mesh? ?tween_param? ?img?";
    return TCL_ERROR;
  }

  mesh_src_tag = argv[1];
  mesh_dst_tag = argv[2];
  tween_param = atof(argv[3]);
  rgba_image_tag = argv[4];

  /* Look up the meshes and image */
  HASH_FIND(mesh_table, mesh_src_tag, mesh_srcP, MeshT, "warpCmd");
  HASH_FIND(mesh_table, mesh_dst_tag, mesh_dstP, MeshT, "warpCmd");
  HASH_FIND(img_table, rgba_image_tag, rgbaP, RgbaImageT, "warpCmd");

  /* Make sure the meshes are the same size */
  if(meshCheck(mesh_srcP, mesh_dstP)) {
    interp->result = "warp: meshes don't match";
    return TCL_ERROR;
  }

  /* Create the tween mesh */
  tween.nx = mesh_srcP->nx;
  tween.ny = mesh_srcP->ny;
  if(meshAlloc(&tween, "warpCmd")) return TCL_ERROR;
  meshInterp(mesh_srcP, mesh_dstP, tween_param, &tween);

  /* Allocate the output image */
  if((warped = MY_CALLOC(1, RgbaImageT, "warpCmd"))==NULL) return TCL_ERROR;
  rgbaImageInit(warped);
  warped->nrows = rgbaP->nrows;
  warped->ncols = rgbaP->ncols;
  if(rgbaImageAlloc(warped, "warpCmd")) return TCL_ERROR;
  createImageTag(warped, interp->result);

  /* Warp the image, one channel at a time */
  if(tween_param >= 0) {
    /* Warp forward from src to tween */
    warp_image(rgbaP->ri, warped->ri, rgbaP->ncols, rgbaP->nrows, mesh_srcP->x, mesh_srcP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->gi, warped->gi, rgbaP->ncols, rgbaP->nrows, mesh_srcP->x, mesh_srcP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->bi, warped->bi, rgbaP->ncols, rgbaP->nrows, mesh_srcP->x, mesh_srcP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->ai, warped->ai, rgbaP->ncols, rgbaP->nrows, mesh_srcP->x, mesh_srcP->y, tween.x, tween.y, tween.nx, tween.ny);
  } else {
    /* Warp backward from dst to tween */
    warp_image(rgbaP->ri, warped->ri, rgbaP->ncols, rgbaP->nrows, mesh_dstP->x, mesh_dstP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->gi, warped->gi, rgbaP->ncols, rgbaP->nrows, mesh_dstP->x, mesh_dstP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->bi, warped->bi, rgbaP->ncols, rgbaP->nrows, mesh_dstP->x, mesh_dstP->y, tween.x, tween.y, tween.nx, tween.ny);
    warp_image(rgbaP->ai, warped->ai, rgbaP->ncols, rgbaP->nrows, mesh_dstP->x, mesh_dstP->y, tween.x, tween.y, tween.nx, tween.ny);
  }

  /* Clean up and return */
  meshFree(&tween, "warpCmd");
  return TCL_OK;
}
