/* mesh_tcl.c: TCL interface to mesh routines.
**
** written and copyright (C) 1996 by Michael J. Gourlay
**
** PROVIDED AS IS.  NO WARRENTIES, EXPRESS OR IMPLIED.
*/

#include <stdio.h>
#include <stdlib.h>

#include <tcl.h>

#include "hash.h"
#include "my_malloc.h"
#include "mesh.h"

#include "mesh_tcl.h"




#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE (!FALSE)
#endif



/* mesh_table: hash table of meshes for mapping a mesh tag to a
** mesh data structure.
*/
Tcl_HashTable mesh_table;


/* mesh_table_initF: flag which indicates whether the hash table has
** been initialized yet.
*/
int mesh_table_initF = FALSE;




/* createMeshTag: create a unique "tag" (i.e. TCL name) for the mesh.
**
** meshP: (in) pointer to the mesh
** tag: (out) tag of the mesh
**
*/
static void
createMeshTag(const MeshT *meshP, Tcl_Interp *interp)
{
  createTag(meshP, interp->result, "mesh", &mesh_table_initF, &mesh_table);
  Tcl_CreateCommand(interp, interp->result, meshValuesCmd,
                    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
}




/* meshLoadCmd: TCL command to create and load a mesh
*/
int
meshLoadCmd(ClientData client_data, Tcl_Interp *interp, int argc, char *argv[])
{
  char *filename;
  MeshT *meshP;

  /* ---- Allocate mesh instance */
  if((meshP = MY_CALLOC(1, MeshT, "meshLoadCmd")) == NULL) {
    return TCL_ERROR;
  }


  /* ---- Parse the arguments to get the file name. */
  if(argc != 2) {
    my_free(meshP, "meshLoadCmd");
    interp->result = "Usage: meshLoad ?filename?";
    return TCL_ERROR;
  }

  filename = argv[1];

  /* ---- Load the image using meshLoad. */
  if(meshLoad(meshP, filename)) {
    my_free(meshP, "meshLoadCmd");
    sprintf(interp->result, "meshLoad: could not load file %s\n", filename);
    return TCL_ERROR;
  }

  /* ---- Create a new tag for this mesh. */
  /* ---- Create a new hash table entry. */
  /* ---- Register the mesh in the hash table. */
  createMeshTag(meshP, interp);

  return TCL_OK;
}




meshValuesCmd(ClientData client_data,Tcl_Interp *interp, int argc, char *argv[])
{
  char *mesh_tag;
  char *mesh_member;
  MeshT *meshP;
  int xi;
  int vi;
  char num_str[32];

  /* ---- Parse the arguments to get the file name. */
  if(argc < 2 || argc > 4) {
    interp->result = "Usage: meshTag meshMember ...";
    return TCL_ERROR;
  }
  mesh_tag = argv[0];
  mesh_member = argv[1];

  /* Look up the mesh */
  HASH_FIND(mesh_table, mesh_tag, meshP, MeshT, "meshValuesCmd");

  if(!strcmp(mesh_member, "col")) {

    if(argc > 2) {
      xi = atoi(argv[2]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    if(xi < 0 || xi > meshP->nx) {
      sprintf(interp->result, "meshValues: bad value %i for index\n", xi);
      return TCL_ERROR;
    }
    for(vi=0; vi < meshP->ny; vi++) {
      sprintf(num_str, " %.0lf %0.lf ", meshP->x[vi * meshP->nx + xi],
                                        meshP->y[vi * meshP->nx + xi]);
      Tcl_AppendResult(interp, num_str, NULL);
    }

  } else if (!strcmp(mesh_member, "row")) {

    if(argc > 2) {
      yi = atoi(argv[2]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    if(yi < 0 || yi > meshP->ny) {
      sprintf(interp->result, "meshValues: bad value %i for index\n", yi);
      return TCL_ERROR;
    }
    for(vi=0; vi < meshP->nx; vi++) {
      sprintf(num_str, "%.0lf %0.lf ", meshP->x[yi * meshP->nx + vi],
                                       meshP->y[yi * meshP->nx + vi]);
      Tcl_AppendResult(interp, num_str, NULL);
    }

  } else if (!strcmp(mesh_member, "pointx")) {
    int yi;

    if(argc > 2) {
      xi = atoi(argv[2]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    if(argc > 3) {
      yi = atoi(argv[3]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    sprintf(interp->result, "%.0lf", meshP->x[yi * meshP->nx + xi]);

  } else if (!strcmp(mesh_member, "pointy")) {
    int yi;

    if(argc > 2) {
      xi = atoi(argv[2]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    if(argc > 3) {
      yi = atoi(argv[3]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    sprintf(interp->result, "%.0lf", meshP->y[yi * meshP->nx + xi]);

  } else if (!strcmp(mesh_member, "nearest")) {

    if(argc > 2) {
      xi = atoi(argv[2]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

    if(argc > 3) {
      yi = atoi(argv[3]);
    } else {
      sprintf(interp->result, "%s %s: requires index\n", mesh_tag, mesh_member);
      return TCL_ERROR;
    }

args are x and y of pointer
    meshPointNearest(meshP, xi, yi, etc...
call meshPointNearest
returns index of nearest mesh point

this should drawn over by a white dot.
A new dot should be created, and its location should be the pointer.

  } else if (!strcmp(mesh_member, "nx")) {
    sprintf(interp->result, "%i", meshP->nx);

  } else if (!strcmp(mesh_member, "ny")) {
    sprintf(interp->result, "%i", meshP->ny);

  } else {
    sprintf(interp->result,
      "meshValues: bad member '%s'.  Must be one of: col, row, point, nx, ny\n",
      mesh_member);
    return TCL_ERROR;
  }
  return TCL_OK;
}




