/* mesh_cb.c : Callbacks for Digital Image Warp mesh related widgets
**
** Written and Copyright (C) 1994 by Michael J. Gourlay
**
** NO WARRANTEES, EXPRESS OR IMPLIED.
*/

#include <stdio.h>
#include <stdlib.h>

#include "diw_map.h"
#include "mjg_dialog.h"
#include "mesh.h"
#include "mesh_cb.h"




/* ---------------------------------------------------------------------- */

/* reset_mesh_cb: callback to reset a mesh
** Accesses global_diw_map[0] mesh information
** Changes global_diw_map.mesh_[xy][s|d]
** Fakes an expose event to all diw_map image regions to redraw meshes
*/
void
reset_mesh_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  int type = (int) client_data;

  switch(type) {
    case 1:
      meshReset(&global_diw_map[0].mesh_src,
        global_diw_map[0].width, global_diw_map[0].height);
      break;

    case 2:
      meshReset(&global_diw_map[0].mesh_dst,
        global_diw_map[0].width, global_diw_map[0].height);
      break;

    default:
      fprintf(stderr, "reset_mesh_cb: Bad Value: type: %i\n", type);
      return;
      break;
  }

  /* Now redraw the meshes */
  FakeAllExpose(0, NULL, NULL, NULL);
}




/* ---------------------------------------------------------------------- */

/* functionalize_mesh_cb: callback to functionalize a mesh
** Accesses global_diw_map[0] mesh information
** Changes global_diw_map.mesh_[xy][s|d]
** Fakes an expose event to all diw_map image regions to redraw meshes
*/
void
functionalize_mesh_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  int type = (int) client_data;
  int chg;

  switch(type) {
    case 1:
    chg=meshFunctionalize(&global_diw_map[0].mesh_src,
        global_diw_map[0].width, global_diw_map[0].height);
      break;

    case 2:
    chg=meshFunctionalize(&global_diw_map[0].mesh_dst,
        global_diw_map[0].width, global_diw_map[0].height);
      break;

    default:
      fprintf(stderr, "functionalize_mesh_cb: Bad Value: type: %i\n", type);
      return;
      break;
  }

  /* If the mesh changed, redraw the meshes */
  if(chg) FakeAllExpose(0, NULL, NULL, NULL);
}




/* ---------------------------------------------------------------------- */

/* load_mesh_cb: callback to load mesh from file
** Accesses global_diw_map[0] mesh information
** Changes global_diw_map.mesh_[xy][s|d]
** Frees memory for old mesh
** Allocates memory for new mesh
** Fakes an expose event to all diw_map image regions to redraw meshes
*/
void
load_mesh_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  dialog_apdx_t *daP = (dialog_apdx_t *)call_data;
  int type = (int) client_data;
  String fn;
  MeshT  mesh;
  int    load_mesh_rv;

  fn = XawDialogGetValueString(daP->dialog);
  if(fn==NULL) {
    return;
  } else {
    if(load_mesh_rv = meshLoad(&mesh, fn)) {
      fprintf(stderr, "meshLoad returned %i\n", load_mesh_rv);
      return;
    }

    switch(type) {
      case 1:
        meshFree(&global_diw_map[0].mesh_src, "load_mesh_cb");
        global_diw_map[0].mesh_src.x = mesh.x;
        global_diw_map[0].mesh_src.y = mesh.y;
        break;

      case 2:
        meshFree(&global_diw_map[0].mesh_dst, "load_mesh_cb");
        global_diw_map[0].mesh_dst.x = mesh.x;
        global_diw_map[0].mesh_dst.y = mesh.y;
        break;

      default:
        fprintf(stderr, "load_mesh_cb: Bad Value: type: %i\n", type);
        return;
        break;
    }

    /* Check to see if new mesh geometry matches old */
    if(  (global_diw_map[0].mesh_src.nx != mesh.nx)
       ||(global_diw_map[0].mesh_src.ny != mesh.ny))
    {
      /* Geometries of old meshes and newly loaded meshes did not match */
      /* Reset the alternate mesh */
      switch(type) {
          /* The loaded mesh is going into the src, so we'll force the
          ** dst mesh geometry to match.
          */
        case 1:
          global_diw_map[0].mesh_dst.nx = mesh.nx;
          global_diw_map[0].mesh_dst.ny = mesh.ny;
          meshFree(&global_diw_map[0].mesh_dst, "load_mesh_cb");
          meshAlloc(&global_diw_map[0].mesh_dst, "load_mesh_cb");
          meshReset(&global_diw_map[0].mesh_dst,
            global_diw_map[0].src_img.ncols, global_diw_map[0].src_img.nrows);
          break;

          /* The loaded mesh is going into the dst, so we'll force the
          ** src mesh geometry to match.
          */
        case 2:
          meshFree(&global_diw_map[0].mesh_src, "load_mesh_cb");
          meshAlloc(&global_diw_map[0].mesh_src, "load_mesh_cb");
          meshReset(&global_diw_map[0].mesh_src,
            global_diw_map[0].src_img.ncols, global_diw_map[0].src_img.nrows);
          break;

        default:
          fprintf(stderr, "load_mesh_cb: Bad Value: type: %i\n", type);
          return;
          break;
      }
    }

    /* Update the geometry parameters */
    global_diw_map[0].mesh_src.nx = global_diw_map[0].mesh_dst.nx = mesh.nx;
    global_diw_map[0].mesh_src.ny = global_diw_map[0].mesh_dst.ny = mesh.ny;
  }

  /* Propagate the mesh changes to all of the diw_maps */
  diw_map_of_widget(NULL);

  /* Redraw the meshes */
  FakeAllExpose(0, NULL, NULL, NULL);
}




/* ---------------------------------------------------------------------- */

/* save_mesh_cb: callback to save mesh to file
** Accesses global_diw_map[0] mesh information
*/
void
save_mesh_cb(Widget w, XtPointer client_data, XtPointer call_data)
{
  dialog_apdx_t *daP = (dialog_apdx_t *)call_data;
  int type = (int) client_data;
  String fn;
  int save_mesh_rv;

  fn = XawDialogGetValueString(daP->dialog);
  if(fn==NULL) {
    return;
  } else {
    switch(type) {
      case 1:
        if(save_mesh_rv = meshSave(&global_diw_map[0].mesh_src, fn)) {
          fprintf(stderr, "meshSave returned %i\n", save_mesh_rv);
          return;
        }
        break;

      case 2:
        if(save_mesh_rv = meshSave(&global_diw_map[0].mesh_dst, fn)) {
          fprintf(stderr, "save_mesh returned %i\n", save_mesh_rv);
          return;
        }
        break;

      default:
        fprintf(stderr, "save_mesh_cb: Bad Value: type: %i\n", type);
        return;
        break;
    }
  }
}
