/*
** Copyright 1999 by Todd Allen.  All Rights Reserved.  Permission to use,
** copy, modify, distribute, and sell this software and its documentation for
** any purpose is hereby granted without fee, provided that the above
** copyright notice appear in all copies and that both the copyright notice
** and this permission notice appear in supporting documentation.
**
** No representations are made about the suitability of this software for any
** purpose.  It is provided ``as is'' without express or implied warranty,
** including but not limited to the warranties of merchantability, fitness
** for a particular purpose, and noninfringement.  In no event shall Todd
** Allen be liable for any claim, damages, or other liability, whether in
** action of contract, tort, or otherwise, arising from, out of, or in
** connection with this software.
*/

#include "diags.h"
#include "types.h"
#include "timemath.h"
#include "parse.h"
#include "pending.h"
#include "handlers.h"
#include "lite-w-green.xpm"
#include "lite-w-yellow.xpm"
#include "lite-w-red.xpm"
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Shell.h>
#include <X11/XawXpm/Box.h>
#include <X11/XawXpm/BoxCommand.h>
#include <X11/XawXpm/CompositeThreeD.h>
#include <X11/XawXpm/TransparentLabel.h>
#include <X11/Xmu/Misc.h>
#include <X11/extensions/shape.h>
#include <signal.h>
#include <stdio.h>

static  void   quit  (Widget     w,
                      XEvent*    event,
                      String*    params,
                      Cardinal*  num_params);
static  void   usage (String     name);

#define APP_CLASS    "XPing"

#if defined(linux)
#define DEFAULT_COMMAND "/bin/ping -c 1 %s"
#elif defined(_AIX)
#define DEFAULT_COMMAND "/usr/sbin/ping -l 1 -c 1 %s"
#elif defined(_PowerMAXOS)
#define DEFAULT_COMMAND "/usr/sbin/ping %s"
#else
#define DEFAULT_COMMAND ""
#endif

static String fallback_resources[] =
{
   "XPing*background:                      gray75",
   "XPing*foreground:                      black",
   "XPing*Box.vSpace:                      0",
   "XPing*Box.hSpace:                      0",
   "XPing*Box.borderWidth:                 0",
   "XPing*BoxCommand.vSpace:               2",
   "XPing*BoxCommand.hSpace:               4",
   "XPing*BoxCommand.borderWidth:          0",
   "XPing*TransparentLabel.borderWidth:    0",
   "XPing*TransparentLabel.internalHeight: 0",
   "XPing*TransparentLabel.internalWidth:  0",
   "XPing*Core.borderWidth:                0",
   "XPing*font:                            *-helvetica-bold-r-*--12-*",
   NULL
};

static XrmOptionDescRec options[] =
{
   { "-machines",    ".machines",       XrmoptionSepArg, (XPointer)NULL },
   { "-down",        ".downBitmap",     XrmoptionSepArg, (XPointer)NULL },
   { "-wasdown",     ".wasDownBitmap",  XrmoptionSepArg, (XPointer)NULL },
   { "-up",          ".upBitmap",       XrmoptionSepArg, (XPointer)NULL },
   { "-command",     ".pingCommand",    XrmoptionSepArg, (XPointer)NULL },
   { "-button2",     ".button2Command", XrmoptionSepArg, (XPointer)NULL },
   { "-button3",     ".button3Command", XrmoptionSepArg, (XPointer)NULL },
   { "-irregular",   ".regular",        XrmoptionNoArg,  "false" },
   { "-delay",       ".delay",          XrmoptionSepArg, (XPointer)NULL },
   { "-retries",     ".numRetries",     XrmoptionSepArg, (XPointer)NULL },
   { "-left",        ".bitmapPosition", XrmoptionNoArg,  "left" },
   { "-history",     ".historyWarning", XrmoptionNoArg,  "true" },
   { "-autodepress", ".autoDepress",    XrmoptionNoArg,  "true" },
   { "-maxparallel", ".maxParallel",    XrmoptionSepArg, (XPointer)NULL },
};

#define offset(field) XtOffsetOf(XPingRec, field)

static XtResource resources[] =
{
   { "machines",          "Machines",          XtRString,    sizeof(String),
        offset(machine_str),                   XtRString,    "localhost"      },
   { "downBitmap",        "AlertBitmap",       XtRString,    sizeof(String), 
        offset(icons[Down].name),              XtRString,    NULL             },
   { "downBitmapMask",    "AlertBitmapMask",   XtRString,    sizeof(String), 
        offset(icons[Down].mask_name),         XtRString,    NULL             },
   { "wasDownBitmap",     "WarningBitmap",     XtRString,    sizeof(String), 
        offset(icons[WasDown].name),           XtRString,    NULL             },
   { "wasDownBitmapMask", "WarningBitmapMask", XtRString,    sizeof(String), 
        offset(icons[WasDown].mask_name),      XtRString,    NULL             },
   { "upBitmap",          "NormalBitmap",      XtRString,    sizeof(String), 
        offset(icons[Up].name),                XtRString,    NULL             },
   { "upBitmapMask",      "NormalBitmapMask",  XtRString,    sizeof(String), 
        offset(icons[Up].mask_name),           XtRString,    NULL             },
   { "pingCommand",       "PingCommand",       XtRString,    sizeof(String),
        offset(ping_command.str),              XtRString,    DEFAULT_COMMAND  },
   { "button2Command",    "ButtonCommand",     XtRString,    sizeof(String),
        offset(button2_command.str),           XtRString,    ""               },
   { "button3Command",    "ButtonCommand",     XtRString,    sizeof(String),
        offset(button3_command.str),           XtRString,    ""               },
   { "regular",           "Regular",           XtRBoolean,   sizeof(Boolean), 
        offset(regular),                       XtRImmediate, (XtPointer)True  },
   { "delay",             "Delay",             XtRInt,       sizeof(int), 
        offset(delay),                         XtRImmediate, (XtPointer)60000 },
   { "numRetries",        "NumRetries",        XtRInt,       sizeof(int),
        offset(num_retries),                   XtRImmediate, (XtPointer)0     },
   { "bitmapPosition",    "BitmapPosition",    XtRString,    sizeof(String),
        offset(bitmap_position),               XtRString,    "right"          },
   { "historyWarning",    "HistoryWarning",    XtRBoolean,   sizeof(Boolean),
        offset(history_warning),               XtRImmediate, (XtPointer)False },
   { "autoDepress",       "AutoDepress",       XtRBoolean,   sizeof(Boolean),
        offset(auto_depress),                  XtRImmediate, (XtPointer)False },
   { "maxParallel",       "MaxParallel",       XtRInt,       sizeof(int),
        offset(max_parallel),                  XtRImmediate, (XtPointer)1     },
};

static XtActionsRec actions[] = 
{
    { "quit",    quit },
    { "button2", handle_button2 },
    { "button3", handle_button3 },
};
 

static Atom wm_delete_window;


static
void
interpret_bitmap_position(XPingRec*  xping)
{
   if (strcmp(xping->bitmap_position, "left") == 0) {
      xping->bitmap_left  = True;
      xping->text_justify = XtJustifyLeft;
   } else if (strcmp(xping->bitmap_position, "right") == 0) {
      xping->bitmap_left  = False;
      xping->text_justify = XtJustifyRight;
   } else {
      warning(static_format("Cannot convert string \"%s\" to type"
                            " BitmapPosition", xping->bitmap_position));
      xping->bitmap_left  = False;
      xping->text_justify = XtJustifyRight;
   }
}

static
void
load_lights(XPingRec*  xping,
            Display*   display,
            Widget     appShell)
{
   Pixel  fg;
   Pixel  bg;
   int    status;

   XtVaGetValues(appShell,
                 XtNforeground, &fg,
                 XtNbackground, &bg,
                 NULL);

   status = XULoadPixmap(display, &xping->icons[Up], green_xpm,  
                         fg, bg, "Up light");
   status &= XULoadPixmap(display, &xping->icons[Down], red_xpm,    
                          fg, bg, "Down light");
   if (xping->history_warning) {
      status &= XULoadPixmap(display, &xping->icons[WasDown], yellow_xpm, 
                             fg, bg, "WasDown light");
   }

   if (!status) exit(1);
}

static
void
size_lights(XPingRec*  xping)
{
   State  state;

   xping->light_height = 0;
   xping->light_width  = 0;
   for (state = 0; state < xping->num_states; state++) {
      xping->light_height = Max(xping->light_height, 
                                xping->icons[state].attributes.height);
      xping->light_width  = Max(xping->light_width, 
                                xping->icons[state].attributes.width);
   }
}

static
Cardinal
button_children_position(Widget w)
{
   if (strcmp(XtName(w), "label") == 0) {
      return 0;
   } else if (strcmp(XtName(w), "light") == 0) {
      XtPointer  user_data;
      Machine*   machine;

      XtVaGetValues(XtParent(w),
                    XtNuserData, &user_data,
                    NULL);
      machine = (Machine*)user_data;

      if (machine->xping->bitmap_left) {
         return 0;
      } else {
         return 1;
      }
   } else {
      assert(False);
   }
}

/*
** Create the widgets (except the lights which require more information)
*/
static
void
create_buttons(XPingRec*  xping,
               Widget     appShell)
{
   Widget          box;
   XtTranslations  button_translations;
   int             i;

   {
      Arg       args[32];
      Cardinal  cnt      = 0;
      
      XtSetArg(args[cnt], XtNorientation, XtorientVertical); cnt++;

      box = XtCreateManagedWidget(/* name         => */ "box",
                                  /* widget_class => */ boxWidgetClass,
                                  /* parent       => */ appShell,
                                  /* args         => */ args,
                                  /* num_args     => */ cnt);
   }

   button_translations
      = XtParseTranslationTable("<Btn1Up>: notify()\n"
                                "<Btn2Down>: set()\n"
                                "<Btn2Up>: button2()\n"
                                "<Btn3Down>: set()\n"
                                "<Btn3Up>: button3()\n"
                                "<EnterWindow>: highlight()\n"
                                "<LeaveWindow>: unhighlight()\n");

   for (i = 0; i < xping->numrows; i++) {
      Widget  rowbox;
      int     j;

      {
         String    name     = static_format("box%d", i);
         Arg       args[32];
         Cardinal  cnt      = 0;

         XtSetArg(args[cnt], XtNorientation, XtorientHorizontal); cnt++;

         rowbox = XtCreateManagedWidget(/* name         => */ name,
                                        /* widget_class => */ boxWidgetClass,
                                        /* parent       => */ box,
                                        /* args         => */ args,
                                        /* num_args     => */ cnt);
      }

      for (j = 0; j < xping->numcolumns; j++) {
         Machine*  machine = &xping->machines[i*xping->numcolumns+j];
         String    name    = machine->name;

         machine->xping   = xping;
         machine->pending = NOT_PENDING;

         if (name != (String)NULL) {
            if (strcmp(name, "-") == 0) {
               machine->button 
                  = XtCreateManagedWidget
                     (/* name         => */ "",
                      /* widget_class => */ simpleWidgetClass,
                      /* parent       => */ rowbox,
                      /* args         => */ NULL,
                      /* num_args     => */ 0);
            } else {
               {
                  Arg       args[32];
                  Cardinal  cnt      = 0;

                  XtSetArg(args[cnt], XtNorientation, 
                                      XtorientHorizontal); cnt++;
                  XtSetArg(args[cnt], XtNuserData, (XtPointer)machine); cnt++;
                  XtSetArg(args[cnt], XtNinsertPosition, 
                                      button_children_position); cnt++;

                  machine->button 
                     = XtCreateManagedWidget
                        (/* name         => */ name,
                         /* widget_class => */ boxCommandWidgetClass,
                         /* parent       => */ rowbox,
                         /* args         => */ args,
                         /* num_args     => */ cnt);

                  XtOverrideTranslations(machine->button, 
                                         button_translations);

                  XtAddCallback(/* w             => */ machine->button,
                                /* callback_name => */ XtNcallback,
                                /* callback      => */ handle_button,
                                /* client_data   => */ machine);
               }

               {
                  Arg       args[32];
                  Cardinal  cnt      = 0;

                  XtSetArg(args[cnt], XtNlabel,   machine->name); cnt++;
                  XtSetArg(args[cnt], XtNjustify, xping->text_justify); cnt++;

                  machine->label
                     = XtCreateManagedWidget
                          (/* name         => */ "label",
                           /* widget_class => */ transparentLabelWidgetClass,
                           /* parent       => */ machine->button,
                           /* args         => */ args,
                           /* num_args     => */ cnt);
               }
            }
         }
      }
   }
}

static
void
pad_buttons(XPingRec*  xping)
{
   Dimension  regular_height = 0;
   Dimension  regular_width  = 0;

   {
      int  i;

      for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
         if (xping->machines[i].button != (Widget)NULL) {
            Dimension  height;
            Dimension  width;

            XtVaGetValues(xping->machines[i].label,
                          XtNheight, &height,
                          XtNwidth,  &width,
                          NULL);

            regular_height = Max(regular_height, height);
            regular_width  = Max(regular_width,  width);
         }
      }
   }

   /*
   ** Vertically center the material in whichever of the label or light
   ** widgets is shorter.  The lights already have a regular height.  The
   ** labels will be guaranteed to if -regular or xping.regular:True was
   ** used.
   */
   if (xping->regular) {
      if (xping->light_height-1 > regular_height) {
         Dimension  pad = (xping->light_height - regular_height)/2;
         int        i;

         for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
            if (xping->machines[i].button != (Widget)NULL) {
               XtVaSetValues(xping->machines[i].label,
                             XtNinternalHeight, pad,
                             NULL);
            }
         }

         regular_height = xping->light_height;
      } else if (regular_height > xping->light_height) {
         /*
         ** Centering the light is handled in pad_lights(), after they
         ** have been created.
         */
         xping->light_height = regular_height;
      }

      /*
      ** Also make the label widgets have regular sizes.  
      */
      {
         int  i;

         for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
            if (xping->machines[i].button != (Widget)NULL) {
               XtVaSetValues(xping->machines[i].label,
                             XtNheight, regular_height,
                             XtNwidth,  regular_width,
                             NULL);
            }
         }
      }
   } else {
      int  i;

      for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
         if (xping->machines[i].button != (Widget)NULL) {
            Dimension  label_height;
            Dimension  label_width;
            
            XtVaGetValues(xping->machines[i].label,
                          XtNheight, &label_height,
                          NULL);
            if (xping->light_height-1 > label_height) {
               Dimension  pad = (xping->light_height - label_height) / 2;

               XtVaSetValues(xping->machines[i].label,
                             XtNinternalHeight, pad,
                             NULL);
            } else if (label_height > xping->light_height) {
               xping->light_height = label_height;
            }
         }
      }
   }
}

/*
** Now that all the heights have been regularized, create the light 
** widgets.
*/
static
void
create_lights(XPingRec*  xping)
{
   int  i;

   for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
      if (xping->machines[i].button != (Widget)NULL) {
         Arg       args[32];
         Cardinal  cnt      = 0;

         XtSetArg(args[cnt], XtNheight, xping->light_height); cnt++;
         XtSetArg(args[cnt], XtNwidth,  xping->light_width); cnt++;

         xping->machines[i].light
            = XtCreateManagedWidget
                 (/* name         => */ "light",
                  /* widget_class => */ coreWidgetClass,
                  /* parent       => */ xping->machines[i].button,
                  /* args         => */ args,
                  /* num_args     => */ cnt);
      }
   }
}

static XtSignalId  sigchld_id;

static
void
notice_sigchld(int  x)
{
   XtNoticeSignal(sigchld_id);
}

/*
** Signal Handling integrated with Xt.  Signal will be handled as follows:
**    SIGCHLD -> notice_sigchld -> XtNoticeSignal -> handle_sigchld
*/
static
void
setup_signals(XPingRec*  xping)
{
   sigchld_id = XtAppAddSignal(/* app_context => */ xping->context,
                               /* proc        => */ handle_sigchld,
                               /* client_data => */ (XtPointer)xping);

   {
      struct sigaction  act;

      act.sa_handler = notice_sigchld;
      sigemptyset(&act.sa_mask);
      act.sa_flags = SA_NOCLDSTOP;
      sigaction(SIGCHLD, &act, NULL);
   }
}

static
void
pad_lights(XPingRec*  xping,
           Display*   display)
{
   State  state;

   for (state = 0; state < xping->num_states; state++) {
      Dimension  pad_height = (xping->light_height
                               - xping->icons[state].attributes.height) / 2;
      Dimension  pad_width  = (xping->light_width
                               - xping->icons[state].attributes.height) / 2;

      xping->padded_icons[state] = XUPadPixmap(display,
                                               xping->icons[state],
                                               pad_width, 
                                               pad_height);
   }
}

static
void
shape_lights(XPingRec*  xping,
             Widget     appShell)
{
   {
      int  shape_error_base;
      int  shape_event_base;

      xping->shape = XShapeQueryExtension(XtDisplay(appShell), 
                                          &shape_event_base, 
                                          &shape_error_base);
   }

   {
      int  i;

      for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
         if (xping->machines[i].button != (Widget)NULL) {
            if (xping->shape) {
               XShapeCombineMask
                  (/* dpy      => */ XtDisplay(xping->machines[i].light),
                   /* dest     => */ XtWindow(xping->machines[i].light),
                   /* destKind => */ ShapeBounding,
                   /* xOff     => */ 0,
                   /* yOff     => */ 0,
                   /* pixmap   => */ xping->padded_icons[Up].mask,
                   /* op       => */ ShapeSet);
            }

            XSetWindowBackgroundPixmap(XtDisplay(xping->machines[i].light), 
                                       XtWindow(xping->machines[i].light),
                                       xping->padded_icons[Up].pixmap);
         }
      }
   }
}

extern
int
main(int    argc,
     char*  argv[])
{
   Display*  display;
   Widget    appShell;
   XPingRec  xping;

   program=strrchr(argv[0], '/');
   if (program != (String)NULL) {
      program++;
   } else {
      program = argv[0];
   }

   /*
   ** Initialize the application context and open the display.
   */
   XtToolkitInitialize();

   xping.context = XtCreateApplicationContext();
   XtAppSetFallbackResources(xping.context, fallback_resources);

   display = XtOpenDisplay(/* app_context    => */ xping.context,
                           /* display_string => */ NULL,
                           /* app_name       => */ program,
                           /* app_class      => */ APP_CLASS,
                           /* options        => */ options,
                           /* num_options    => */ XtNumber(options),
                           /* argc           => */ &argc,
                           /* argv           => */ argv);
   if (display == (Display*)NULL) {
      fatal("cannot open display");
   }

   if (argc != 1) {
      usage(program);
      return 1;
   }

   /*
   ** Create the appShell with mappedWhenManaged:False to allow Shape 
   ** modifications between Realization and Mapping.
   */
   {
      Arg       args[32];
      Cardinal  cnt      = 0;

      XtSetArg(args[cnt], XtNmappedWhenManaged, False); cnt++;

      appShell 
         = XtAppCreateShell(/* app_name     => */ program, 
                            /* app_class    => */ APP_CLASS,
                            /* widget_class => */ applicationShellWidgetClass,
                            /* display      => */ display, 
                            /* args         => */ args, 
                            /* num_args     => */ cnt);
   }

   XtAppAddActions(/* app_context => */ xping.context,
                   /* actions     => */ actions,
                   /* num_actions => */ XtNumber(actions));
 
   /*
   ** This is a hack to make f.delete work in this single-window application.
   */
   {
      XtTranslations  shell_translations 
         = XtParseTranslationTable("<Message>WM_PROTOCOLS: quit()");
      XtOverrideTranslations(/* widget       => */ appShell,
                             /* translations => */ shell_translations);
   }

   /*
   ** Read and interpret the resources and options
   */
   XtGetApplicationResources(/* w             => */ appShell,
                             /* base          => */ (XtPointer)&xping,
                             /* resources     => */ resources,
                             /* num_resources => */ XtNumber(resources),
                             /* args          => */ NULL,
                             /* num_args      => */ 0);

   interpret_bitmap_position(&xping);

   parse_machines(xping.machine_str, 
                  &xping.machines, 
                  &xping.numrows, 
                  &xping.numcolumns,
                  &xping.nummachines);

   xping.pending      = (Heap)XtCalloc(xping.nummachines, sizeof(Machine*));
   xping.num_pending  = 0;
   xping.num_parallel = 0;

   parse_command(xping.ping_command.str,
                 "ping command",
                 &xping.ping_command.args,
                 &xping.ping_command.arg_num);

   xping.button2_command.present = strcmp(xping.button2_command.str, "") != 0;
   if (xping.button2_command.present) {
      parse_command(xping.button2_command.str,
                    "button2 command",
                    &xping.button2_command.args,
                    &xping.button2_command.arg_num);
   }

   xping.button3_command.present = strcmp(xping.button3_command.str, "") != 0;
   if (xping.button3_command.present) {
      parse_command(xping.button3_command.str,
                    "button3 command",
                    &xping.button3_command.args,
                    &xping.button3_command.arg_num);
   }

   if (xping.history_warning) {
      xping.num_states = NUM_STATES;
   } else {
      assert(WasDown == NUM_STATES-1);
      xping.num_states = NUM_STATES-1;
   }

   load_lights(&xping, display, appShell);

   /*
   ** Set up widget hierarchy and adjust geometries
   */
   size_lights(&xping);

   create_buttons(&xping, appShell);
   pad_buttons(&xping);
   create_lights(&xping);

   setup_signals(&xping);

   XtRealizeWidget(appShell);

   pad_lights(&xping, display);
   shape_lights(&xping, appShell);

   /*
   ** Continuation of the hack to make f.delete work.
   */
   wm_delete_window = XInternAtom(/* display        => */ XtDisplay(appShell),
                                  /* atom_name      => */ "WM_DELETE_WINDOW",
                                  /* only_if_exists => */ False);
   XSetWMProtocols(/* display   => */ XtDisplay(appShell),
                   /* w         => */ XtWindow(appShell),
                   /* protocols => */ &wm_delete_window,
                   /* counts    => */ 1);
 
   XtSetMappedWhenManaged(appShell, True);
   XtMapWidget(appShell);

   /*
   ** Do the initial pass over the systems
   */
   check_systems(&xping);

   /*
   ** Set up the timeout for the next delayed ping and then block.
   */
   xping.next_check = end_of_time;
   setup_timeout(&xping);

   XtAppMainLoop(xping.context);

   return 0;
}

static  
void   
quit(Widget     w,
     XEvent*    event,
     String*    params,
     Cardinal*  num_params)
{
   if (event->type == ClientMessage
       && event->xclient.data.l[0] != wm_delete_window) {
      XBell(XtDisplay(w), 0);
      return;
   }
 
   XCloseDisplay(XtDisplay(w));
   exit(0);
}

#define p(x) fprintf(stderr, (x))

static
void
usage(String  name)
{
    fprintf(stderr, "usage: %s [options ...]\n", name);
    p("\n");
    p("options include:\n");
    p("                      any standard X(1) option\n");
    p("   -display dpy       X server on which to display\n");
    p("   -geometry geom     size and/or location of window\n");
    p("   -machines list     system list: "
                             "vertical:     \"upper ... lower\"\n");
    p("                                   "
                             "horizontal:   \"{ left ... right }\"\n");
    p("                                   "
                             "2 dimensions: \"{ upperleft ... upperright }\n");
    p("                                   "
                             "               {     :             :      }\n");
    p("                                   "
                             "               { lowerleft ... lowerright }\"\n");
    p("\n");                 
    p("   -fn font           button text font\n");
    p("   -bg color          background color\n");
    p("   -fg color          foreground color\n");
    p("   -up path           bitmap for up state         (default: green"
                             " light)\n");
    p("   -down path         bitmap for down state       (default: red"
                             " light)\n");
    p("   -wasdown path      bitmap for \"was down\" state (default: yellow"
                             " light)\n");
    p("   -history           activates \"was down\" state, which indicates"
                             " that a\n");
    p("                      system has been down since its last manual"
                             " ping\n");
    p("   -left              place up/down bitmaps on left (default: right)\n");
    p("   -irregular         pack rows of buttons tightly (default: grid)\n");
    p("   -autodepress       indicates an automatic ping in progress by"
                             " depressing\n");
    p("                      appropriate button\n");
    p("\n");                 
    p("   -command string    ping command returning 0 for up and other for"
                             " down\n");
    p("   -button2 string    command executed when middle mouse button is"
                             " pressed\n");
    p("   -button3 string    command executed when right mouse button is"
                             " pressed\n");
    p("   -delay num         minimum number of milliseconds between"
                             " automatic pings\n");
    p("                      for each system (default: 60 seconds)\n");
    p("   -retries num       number of retries on a failed ping before a"
                             " system is\n");
    p("                      considered down (default: 0)\n");
    p("   -maxparallel num   maximum number of parallel automatic pings"
                             " (default: 1)\n");
    p("\n");
}
