/*
** Copyright 1999 by Todd Allen.  All Rights Reserved.  Permission to use,
** copy, modify, distribute, and sell this software and its documentation for
** any purpose is hereby granted without fee, provided that the above
** copyright notice appear in all copies and that both the copyright notice
** and this permission notice appear in supporting documentation.
**
** No representations are made about the suitability of this software for any
** purpose.  It is provided ``as is'' without express or implied warranty,
** including but not limited to the warranties of merchantability, fitness
** for a particular purpose, and noninfringement.  In no event shall Todd
** Allen be liable for any claim, damages, or other liability, whether in
** action of contract, tort, or otherwise, arising from, out of, or in
** connection with this software.
*/

#include "diags.h"
#include "types.h"
#include "timemath.h"
#include "parse.h"
#include "pending.h"
#include "handlers.h"
#include <X11/XawXpm/BoxCommand.h>
#include <X11/XawXpm/CompositeThreeD.h>
#include <X11/Xmu/Misc.h>
#include <X11/extensions/shape.h>
#include <unistd.h>
#include <sys/wait.h>
#include <errno.h>

#define ERRNO_BIAS   (10)

static  void      set_state               (XPingRec*       xping,
                                           Machine*        machine,
                                           State           state);
static  Machine*  pid_to_machine          (XPingRec*       xping,
                                           pid_t           pid);
static  void      check_system            (XPingRec*       xping,
                                           Machine*        machine,
                                           Boolean         important,
                                           Boolean         retry);
static  void      check_pending           (XPingRec*       xping);
static  void      handle_timeout          (XtPointer       closure,
                                           XtIntervalId*   id);

static
void
set_state (XPingRec*  xping,
           Machine*   machine,
           State      state)
{
   assert(state != WasDown);

   machine->last_info = gettime();

   if (xping->history_warning 
       && !machine->check_requested
       && state == Up 
       && machine->state != Up) {
      state = WasDown;
   }

   if (state != machine->state) {
      machine->state = state;

      if (xping->shape) {
         XShapeCombineMask
            (/* dpy      => */ XtDisplay(machine->light),
             /* dest     => */ XtWindow(machine->light),
             /* destKind => */ ShapeBounding,
             /* xOff     => */ 0,
             /* yOff     => */ 0,
             /* pixmap   => */ xping->padded_icons[state].mask,
             /* op       => */ ShapeSet);
      }
      XSetWindowBackgroundPixmap(XtDisplay(machine->light),
                                 XtWindow(machine->light),
                                 xping->padded_icons[state].pixmap);
      XClearWindow(XtDisplay(machine->light),
                   XtWindow(machine->light));

#ifdef DEBUG_STATE
      switch (state) {
      case Up:
         printf("setting %s to Up\n", machine->name);
         break;
      case WasDown:  
         printf("setting %s to WasDown\n", machine->name);
         break;
      case Down:
         printf("setting %s to Down\n", machine->name);
         break;
      default:
         printf("setting %s to %d\n", machine->name, state);
         break;
      }
#endif
   }

   if (machine->check_requested || xping->auto_depress) {
      XtCallActionProc(/* widget     => */ machine->button, 
                       /* action     => */ "unset", 
                       /* event      => */ (XEvent*)NULL,
                       /* params     => */ (String*)NULL,
                       /* num_params => */ 0);
   }

   machine->check_requested = False;
}

static
Machine*
pid_to_machine (XPingRec*  xping,
                pid_t      pid)
{
   int  i;

   for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
      if (xping->machines[i].pid == pid) {
         return &xping->machines[i];
      }
   }

   return (Machine*)NULL;
}

/*
** important : ignore the maximum parallelism
** retry     : this check already included in num_parallel; don't increment it
*/
static
void
check_system (XPingRec*  xping,
              Machine*   machine,
              Boolean    important,
              Boolean    retry)
{
   if (machine->pid == NOT_RUNNING) {
      if (!important && !retry && xping->num_parallel >= xping->max_parallel) {
         pending_push(xping, machine);
      } else {
         pid_t  child;

         if (!machine->check_requested && xping->auto_depress) {
            XtCallActionProc(/* widget     => */ machine->button, 
                             /* action     => */ "set", 
                             /* event      => */ (XEvent*)NULL,
                             /* params     => */ (String*)NULL,
                             /* num_params => */ 0);
         }

#ifdef DEBUG_EXEC
         printf("pinging %s\n", machine->name);
#endif

         assert(machine->pending == NOT_PENDING);

         child = fork();

         if (child == -1) {
            error(static_format("unable to fork(); errno = %d", errno));
         } else if (child == 0) {
#ifndef DEBUG_EXEC_OUTPUT
            close(STDIN_FILENO);
            close(STDOUT_FILENO);
            close(STDERR_FILENO);
#endif

            xping->ping_command.args[xping->ping_command.arg_num] 
               = machine->name;
            execvp(xping->ping_command.args[0],
                   xping->ping_command.args);

            /*
            ** execvp() failed!  But our file descriptors are already gone
            ** so we can't report it.  Instead, we convey this information
            ** along with the errno in the return status and let our 
            ** parent process report it in handle_sigchld().
            */
            exit(ERRNO_BIAS + errno);
         } else {
            machine->pid = child;
            if (!retry) {
               xping->num_parallel++;
            }
         }
      }
   }
}

extern
void
check_systems(XPingRec*  xping)
{
   struct timeval  now = gettime();
   int             i;

   for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
      if (xping->machines[i].button != (Widget)NULL
          && XtClass(xping->machines[i].button) == boxCommandWidgetClass) {
         if (difftimevalms(now, xping->machines[i].last_info) >= xping->delay) {
            check_system(xping, 
                         &xping->machines[i], 
                         /* important => */ False,
                         /* retry     => */ False);
         }
      }
   }
}

static
void
check_pending(XPingRec*  xping)
{
#ifdef DEBUG_PENDING
   printf("check_pending\n");
#endif

   while (xping->num_parallel < xping->max_parallel && xping->num_pending > 0) {
      Machine*  machine = pending_pop(xping);

      check_system(xping, 
                   machine, 
                   /* important => */ False,
                   /* retry     => */ False);
   }
}

extern
void
setup_timeout (XPingRec*  xping)
{
   struct timeval  next  = end_of_time;
   int             i;

   for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
      if (xping->machines[i].button != (Widget)NULL
          && XtClass(xping->machines[i].button) == boxCommandWidgetClass) {
         if (xping->machines[i].pid == NOT_RUNNING
             && xping->machines[i].pending == NOT_PENDING
             && difftimevalms(next, 
                              xping->machines[i].last_info) > xping->delay) {
            next = addmstotimeval(xping->machines[i].last_info, xping->delay);
         }
      }
   }

   if (!sametimeval(next, xping->next_check)) {
      if (!sametimeval(xping->next_check, end_of_time)) {
#ifdef DEBUG_TIMEOUT
         printf("stomping current timeout for %u ms\n", 
                difftimevalms(xping->next_check, gettime()));
#endif
         XtRemoveTimeOut(xping->next_timeout);
      }

      xping->next_check = next;

      if (!sametimeval(next, end_of_time)) {
         struct timeval  now = gettime();

         if (lesstimeval(next, now)) {
            xping->next_timeout
               = XtAppAddTimeOut(/* app_context => */ xping->context,
                                 /* interval    => */ 0,
                                 /* proc        => */ handle_timeout,
                                 /* client_data => */ (XtPointer)xping);
#ifdef DEBUG_TIMEOUT
            printf("setting up timeout for 0 ms (%d)\n", 
                   difftimevalms(next, now));
#endif
         } else {
            unsigned long  interval = difftimevalms(next, now);

            xping->next_timeout
               = XtAppAddTimeOut(/* app_context => */ xping->context,
                                 /* interval    => */ interval,
                                 /* proc        => */ handle_timeout,
                                 /* client_data => */ (XtPointer)xping);
#ifdef DEBUG_TIMEOUT
            printf("setting up timeout for %u ms\n", interval);
#endif
         }
#ifdef DEBUG_TIMEOUT
      } else {
         printf("no timeout\n");
#endif
      }
#ifdef DEBUG_TIMEOUT
   } else {
      printf("keeping current timeout for %u ms\n",
             difftimevalms(xping->next_check, gettime()));
#endif
   }
}

static
void   
handle_timeout (XtPointer      closure,
                XtIntervalId*  id)
{
   XPingRec*  xping = (XPingRec*)closure;

#ifdef DEBUG_TIMEOUT
   printf("timeout reached\n");
#endif

   xping->next_check = end_of_time;

   check_systems(xping);

   setup_timeout(xping);
}

extern
void   
handle_sigchld (XtPointer    closure,
                XtSignalId*  id)
{
   XPingRec*  xping = (XPingRec*)closure;

#ifdef DEBUG_EXEC
   printf("SIGCHLD received\n");
#endif

   for (;;) {
      pid_t  child;
      int    status;

      child = waitpid(-1, &status, WNOHANG);

      if (child == -1) {
         if (errno == ECHILD) break;

         error(static_format("waitpid() failed; errno = %d", errno));
         break;
      } else if (child == 0) {
         break;
      } else {
         Machine*  machine = pid_to_machine(xping, child);

         /*
         ** The machine may be NULL if the SIGCHLD was for a command executed
         ** by button2Command or button3Command.  If so, we only deal with it
         ** enough to de-zombify the child.
         */
         if (machine != (Machine*)NULL) {
            machine->pid = NOT_RUNNING;

            if (WIFEXITED(status)) {
               int  stat = WEXITSTATUS(status);

#ifdef DEBUG_EXEC
               printf("ping of %s returned status %d\n", machine->name, stat);
#endif

               if (stat == 0) {
                  xping->num_parallel--;
                  machine->retries = 0;
                  set_state(xping, machine, Up);
               } else if (stat <= ERRNO_BIAS) {
                  if (machine->retries < xping->num_retries) {
                     machine->retries++;
#ifdef DEBUG_RETRY
                     printf("retry #%d for %s\n", 
                            machine->retries, machine->name);
#endif
                     check_system(xping, 
                                  machine, 
                                  /* important => */ False,
                                  /* retry     => */ True);
                  } else {
                     xping->num_parallel--;
                     set_state(xping, machine, Down);
                  }
               } else {
                  xping->num_parallel--;
                  error(static_format("execve() failed for ping of %s;"
                                      " errno = %d",
                                      machine->name,
                                      stat - ERRNO_BIAS));
               }
            } else if (WIFSIGNALED(status)) {
               xping->num_parallel--;
               error(static_format("ping (of %s) was terminated with signal %d",
                                   machine->name,
                                   WTERMSIG(status)));
            } else {
               xping->num_parallel--;
               error(static_format("ping (of %s) terminated in a strange way", 
                                   machine->name));
            }
         }
      }
   }

   check_pending(xping);

   setup_timeout(xping);
}

extern
void
handle_button (Widget     w,
               XtPointer  closure,
               XtPointer  call_data)
{
   Machine*   machine = (Machine*)closure;
   XPingRec*  xping   = machine->xping;

   pending_remove(xping, machine);

   machine->check_requested = True;
   check_system(xping, 
                machine, 
                /* important => */ True,
                /* retry     => */ False);
}

static
void
execute_command (Machine*  machine,
                 Cmd*      command)
{
   if (command->present) {
      pid_t  child;

#ifdef DEBUG_EXEC
      printf("button2 %s\n", machine->name);
#endif

      child = fork();

      if (child == -1) {
         error(static_format("unable to fork(); errno = %d", errno));
      } else if (child == 0) {
         setpgid(0, 0);

         command->args[command->arg_num] = machine->name;
         execvp(command->args[0],
                command->args);

         /*
         ** execvp() failed!  But our file descriptors are already gone
         ** so we can't report it.  Instead, we convey this information
         ** along with the errno in the return status and let our 
         ** parent process report it in handle_sigchld().
         */
         exit(ERRNO_BIAS + errno);
      } else {
         /*
         ** Just ignore the child forevermore.
         */
      }
   }
}
                 
extern  
void      
handle_button2 (Widget          w,
                XEvent*         event,
                String*         params,
                Cardinal*       num_params)
{
   XtPointer  user_data;
   Machine*   machine;
   XPingRec*  xping;


   XtVaGetValues(w,
                 XtNuserData, &user_data,
                 NULL);
   machine = (Machine*)user_data;
   xping = machine->xping;

   execute_command(machine, &xping->button2_command);

   XtCallActionProc(/* widget     => */ w,
                    /* action     => */ "unset", 
                    /* event      => */ (XEvent*)NULL,
                    /* params     => */ (String*)NULL,
                    /* num_params => */ 0);
}

extern  
void      
handle_button3 (Widget          w,
                XEvent*         event,
                String*         params,
                Cardinal*       num_params)
{
   XtPointer  user_data;
   Machine*   machine;
   XPingRec*  xping;


   XtVaGetValues(w,
                 XtNuserData, &user_data,
                 NULL);
   machine = (Machine*)user_data;
   xping = machine->xping;

   execute_command(machine, &xping->button3_command);

   XtCallActionProc(/* widget     => */ w,
                    /* action     => */ "unset", 
                    /* event      => */ (XEvent*)NULL,
                    /* params     => */ (String*)NULL,
                    /* num_params => */ 0);
}
