/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "layer.h"
#include "typ_msg.h"
#include "lay_msg.h"
#include "lay_arp.h"
#include "lay_icmp.h" 
#include "lay_tcp.h" 
#include "lay_arpethip.h" 
#include "lay_ip.h" 
#include "lay_tracert.h" 
#include "lay_data.h" 
#include "lay_udp.h" 
#include "lay_ether.h" 
#include "lay_pcap.h" 
#include "lay_gre.h" 
#include "lay_rip.h"
#include "lay_ripinfo.h"
#include "lay_ipopt.h"
#include "lay_inaddr.h"
#include "lay_tcpopt.h"
#include "lay_dns.h"
#include "lay_dnsqname.h"
#include "lay_dnsqtype.h"
#include "lay_dnsrttl.h"
#include "lay_dnsrdata.h"
#include "lay_time.h"
#include "lay_inaddrtime.h"
#include "lay_tftp.h"
#include "lay_dataedit.h"
#include "typ_lay.h"
#include "lay_tftpfname.h"
#include "lay_tftpmode.h"
#include "lay_tftpbn.h"
#include "lay_tftperror.h"
#include "lay_ieee802.h"
#include "lay_null.h"
#include "lay_rpc.h"
#include "lay_rpccall.h"
#include "lay_pad.h"

t_dict			*lay_dict = NULL;

/* adds a t_msg_proc to the layer dictionary.
   Performs a call to MSG_CLASS to check the class name which must be "lay".
   Returns 0 if OK. Might return -ERR_BADCLASS. */
t_status		lay_add(mp)
t_msg_proc		mp;		/* Lay msg_proc */
{
  char			class[STR_BUFSIZ];
  char			name[STR_BUFSIZ];
  t_status		status;
  t_bridled_str		classbs;
  t_bridled_str		namebs;

  assert(lay_dict);
  classbs.str = class;
  classbs.max_len = sizeof (class);
  class[0] = 0;
  if ((status = lay_msg(mp,
			MSG_CLASS,
			NULL,
			&classbs)) < 0)
    return (status);
  if (strcmp(class,"lay"))
    return (-ERR_BADCLASS);
  namebs.str = name;
  namebs.max_len = sizeof (name);
  name[0] = 0;
  if ((status = lay_msg(mp,
			LAY_NAME,
			NULL,
			&namebs)) < 0)
    return (status);
  if ((status = dict_add(lay_dict,name,mp)) < 0)
    return (status);
#ifdef DEBUG
  if (LAYER_VERB(VERB_LAYER_INIT))
    fprintf(stderr,"added %s to lay dict\n",name);
#endif
  return (0);
}

/* initializes the layer dictionary with all the classes provided by liblayer.
   Dictionary is small (HASH_SMALL_BASE).
   Returns 0 if OK. Might return various errors */
t_status		lay_dict_init(VOID_DECL)
{
  t_status		status;
  
  if ((lay_dict = LAYER_DICT_NEW(HASH_SMALL_BASE,
				 VEC_BASE,
				 &status)) == NULL)
    return (status);
  if (((status = lay_add(lay_arp_msg)) < 0) ||
      ((status = lay_add(lay_arpethip_msg)) < 0) ||
      ((status = lay_add(lay_data_msg)) < 0) ||
      ((status = lay_add(lay_dataebcdic_msg)) < 0) ||
      ((status = lay_add(lay_ether_msg)) < 0) ||
      ((status = lay_add(lay_icmp_msg)) < 0) ||
      ((status = lay_add(lay_icmpmbz_msg)) < 0) ||
      ((status = lay_add(lay_icmpmask_msg)) < 0) ||
      ((status = lay_add(lay_icmpts_msg)) < 0) ||
      ((status = lay_add(lay_icmpredirect_msg)) < 0) ||
      ((status = lay_add(lay_ip_msg)) < 0) || 
      ((status = lay_add(lay_slip_msg)) < 0) ||
      ((status = lay_add(lay_null_msg)) < 0) ||
      ((status = lay_add(lay_ppp_msg)) < 0) ||
      ((status = lay_add(lay_fddi_msg)) < 0) ||
      ((status = lay_add(lay_llc_msg)) < 0) ||
      ((status = lay_add(lay_snap_msg)) < 0) ||
      ((status = lay_add(lay_atm_rfc1483_msg)) < 0) ||
      ((status = lay_add(lay_tcp_msg)) < 0) ||
      ((status = lay_add(lay_udp_msg)) < 0) ||
      ((status = lay_add(lay_rip_msg)) < 0) ||
      ((status = lay_add(lay_ripinfo_msg)) < 0) ||
      ((status = lay_add(lay_ipopt_msg)) < 0) ||
      ((status = lay_add(lay_inaddr_msg)) < 0) ||
      ((status = lay_add(lay_tcpopt_msg)) < 0) ||
      ((status = lay_add(lay_dns_msg)) < 0) ||
      ((status = lay_add(lay_dnsqname_msg)) < 0) ||
      ((status = lay_add(lay_dnsqtype_msg)) < 0) ||
      ((status = lay_add(lay_dnsrttl_msg)) < 0) ||
      ((status = lay_add(lay_dnsrdata_msg)) < 0) ||
      ((status = lay_add(lay_time_msg)) < 0) ||
      ((status = lay_add(lay_inaddrtime_msg)) < 0) ||
      ((status = lay_add(lay_tftp_msg)) < 0) ||
      ((status = lay_add(lay_dataedit_msg)) < 0) ||
      ((status = lay_add(lay_tftpfname_msg)) < 0) ||
      ((status = lay_add(lay_tftpmode_msg)) < 0) ||
      ((status = lay_add(lay_tftpbn_msg)) < 0) ||
      ((status = lay_add(lay_tftperror_msg)) < 0) ||
      ((status = lay_add(lay_rpc_msg)) < 0) ||
      ((status = lay_add(lay_rpccall_msg)) < 0) ||
      ((status = lay_add(lay_pad_msg)) < 0) ||
      ((status = lay_add(lay_tracert_msg)) < 0))
    return (status);
  return (0);
}

/* destroys the layer dictionary */
VOID_FUNC		lay_dict_destroy(VOID_DECL)
{
  assert(lay_dict);
  dict_delete(lay_dict);
}

/* converts-and-catenates a layer to a bridled string.
   Performs a simple LAY_NAME (with optional_id set to NULL).
   Returns 0 if OK. */
t_status		lay_to_str(mp,str,max_len)
t_msg_proc		mp;		/* Msg_proc */
char			*str;		/* Destination string */
int			max_len;	/* Dest. string length */	
{
  t_status		status;
  t_bridled_str		namebs;

  namebs.str = str;
  namebs.max_len = max_len;
  return (lay_msg(mp,
		  LAY_NAME,
		  NULL,
		  &namebs));
}

/* converts a string to a layer.
   Note that the layer dictionary must be initialized. 
   Returns the msg_proc or NULL if it has failed */
t_msg_proc		lay_from_str(str,status)
char			*str;		/* E.g "ip" */
t_status		*status;	/* Filled if NULL */
{
  t_hash_elt		*he;

  assert(lay_dict);
  if (he = dict_get(lay_dict,str))
    return ((t_msg_proc)(he->value));
  else
    {
      *status = -ERR_NOENT;
      return (NULL);
    }
}

t_status		lay_get_choices_walk(he,vec_str)
t_hash_elt		*he;
t_vec			*vec_str;
{
  return (vec_str_add(vec_str,he->key));
}

/* gets all the layers names in a vector of strings.
   Note that layer dictionary must be initialized.
   Returns 0 if OK. Might return various errors */
t_status		lay_get_choices(vec_str)
t_vec			*vec_str;		/* Vector of strings */
{
  return (dict_walk(lay_dict,
		    (t_dict_walk_proc)lay_get_choices_walk,
		    vec_str));
}

/* ASCII-sorted version of lay_get_choices(3).
   Note that layer dictionary must be initialized.
   Returns 0 if OK. Might return various errors */
t_status		lay_get_choices_sorted(vec_str)
t_vec			*vec_str;		/* Vector of strings */
{
  return (dict_walk_sorted(lay_dict,
			   (t_dict_walk_proc)lay_get_choices_walk,
			   vec_str));
}

#ifdef DEBUG
/* shows a layer name.
   This is a debug function. */
VOID_FUNC		lay_show(mp)
t_msg_proc		mp;
{
  char			buf[STR_BUFSIZ];
  t_status		status;

  buf[0] = 0;
  if ((status = lay_to_str(mp,
			   buf,
			   sizeof (buf))) < 0)
    {
      err_print(-status,"lay_to_str");
      return ;
    }
  fprintf(stderr,"%s\n",buf);
}		
#endif

/* is a t_msg_proc.
   Manages "lay" message procs. It is not intended to use in a network 
   stream but it is used by configuration files. */
t_status		typ_lay_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("lay");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_msg_proc	lay;

	if (ed->b.len < sizeof (lay))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&lay,sizeof (t_msg_proc));
	return (lay_to_str(lay,
			   bs->str,
			   bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_msg_proc	lay;
	t_status	status;	

	if (ed->b.len < sizeof (lay))
	  return (-ERR_TRUNC);
	if ((lay = lay_from_str(str,&status)) == NULL)
	  return (status);
	FBCOPY(&lay,ed->b.buf,sizeof (t_msg_proc));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	
